/**
 * Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Auxiliary diagnostic functions to allow printing of cyclic values and a rough
 * estimation of the amount of memory consumed by a given object.
 */
(function(global)
{
	// Wrapper around JSON.stringify that supports cyclic values
	function cycleSafeStringify(object, space)
	{
		var seen = {};
		var numSeen = 0;
		var jsonString = JSON.stringify(object, function(key, val)
		{
			if (typeof val == "object")
			{
				if (seen.hasOwnProperty(val))
					return "__cyclicValue_" + seen[val] + "__";
				seen[val] = ++numSeen;
			}
			return val;
		}, space);

		// The "seen" set can generate a lot of garbage. This function is
		// intented to be used mainly in memory profiling situations, so we force
		// a GC to avoid unwanted spikes in the memory charts.
		seen = null;
		spidermonkey.forceGC();
		return jsonString;
	}

	var _sizesTable =
	{
		boolean: 4,
		char: 2,
		number: 8,
		pointer: 8,
		undefined: 0
	};

	function setSizesTable(newTable)
	{
		for (var key in newTable)
		{
			if (newTable.hasOwnProperty(key))
			{
				if (_sizesTable.hasOwnProperty(key))
				{
					_sizesTable[key] = newTable[key];
				}
				else
				{
					throw "setSizesTable(): Unexpected property '" + key + "'";
				}
			}
		}
	}

	function _sizeofValue(value)
	{
		switch (typeof value)
		{
			case 'boolean':
				return _sizesTable.boolean;
			case 'string':
				return value.length * _sizesTable.char;
			case 'number':
				return _sizesTable.number;
			case 'object':
			case 'function':
				// Size of a reference/pointer
				return _sizesTable.pointer;
			case 'undefined':
				return _sizesTable.undefined;
			default:
				console.warn("Unexpected value of type", type);
				return 0;
		}
	}

	function _accumulateSizeIn(
			value, sizeAccumulatorFunction, seen, ignoreSet, childCallback)
	{
		switch(typeof value)
		{
			case 'object':
			case 'function':
				if ((ignoreSet && ignoreSet.has(value)) || seen.has(value))
				{
					// If it's a cyclic value, adds the cost of a reference
					// (size of a pointer)
					sizeAccumulatorFunction(_sizeofValue(value));
				}
				else
				{
					// If it's a new object, computes its size
					// and adds it to the list of objects seen
					seen.add(value);

					var isArray = (value instanceof Array);
					for (var key in value)
					{
						if (value.hasOwnProperty(key))
						{
							childCallback(key, value[key]);
							if (!isArray)
							{
								// If it's a non-array object, compute the
								// size of its keys
								sizeAccumulatorFunction(_sizeofValue(key));
							}
						}
					}
				}
				break;
			default:
				sizeAccumulatorFunction(_sizeofValue(value));
				break;
		}
	}

	// Even though this function does not return the *exact* size of an object,
	// it provides a rough estimation for debug purposes
	function approximateSizeof(object, ignoreSet)
	{
		var seen = new Set();
		var stack = [object];
		var bytes = 0;

		var sizeAccumulator = function(size)
		{
			bytes += size;
		};

		var childCallback = function(key, child)
		{
			stack.push(child);
		};

		while (stack.length)
		{
			var value = stack.pop();

			_accumulateSizeIn(
					value, sizeAccumulator, seen, ignoreSet, childCallback);
		}

		// The "seen" set can generate a lot of garbage. This function is
		// intented to be used mainly in memory profiling situations, so we force
		// a GC to avoid unwanted spikes in the memory charts.
		seen = null;
		spidermonkey.forceGC();
		return bytes;
	}

	function _hierarchicalSizeof(object, seen, ignoreSet)
	{
		var node =
		{
			size: 0,
			children: {}
		};

		var sizeAccumulator = function(size)
		{
			node.size += size;
		};

		var childCallback = function(key, child)
		{
			var childTrace = _hierarchicalSizeof(child, seen, ignoreSet);

			node.size += childTrace.size;
			node.children[key] = childTrace;
		};

		_accumulateSizeIn(object, sizeAccumulator, seen, ignoreSet, childCallback);

		return node;
	}

	function hierarchicalApproximateSizeof(object, ignoreSet)
	{
		var seen = new Set();

		var root = _hierarchicalSizeof(object, seen, ignoreSet);

		// The "seen" set can generate a lot of garbage. This function is
		// intented to be used mainly in memory profiling situations, so we force
		// a GC to avoid unwanted spikes in the memory charts.
		seen = null;
		spidermonkey.forceGC();
		return root;
	}

	function sizeToString(size)
	{
		var kb = 1024;
		var mb = 1024*1024;
		if (size >= mb)
		{
			return (size/mb).toFixed(2) + " MB";
		}
		else if (size >= kb)
		{
			return (size/kb).toFixed(2) + " KB";
		}
		else
		{
			return size + " Bytes";
		}
	}

	function _criticalPath(trace, path)
	{
		if (!trace)
		{
			return;
		}

		var maxSize = -1;
		var maxKey = "";
		for(var key in trace.children)
		{
			if (trace.children.hasOwnProperty(key))
			{
				if (trace.children[key].size > maxSize)
				{
					maxSize = trace.children[key].size;
					maxKey = key;
				}
			}
		}

		if (maxKey !== "")
		{
			path.push(
			{
				size: trace.children[maxKey].size,
				biggestChild: maxKey
			});
			_criticalPath(trace.children[maxKey], path);
		}
	}

	function criticalPath(object)
	{
		var trace = hierarchicalApproximateSizeof(object);
		var path = [];
		_criticalPath(trace, path);
		return path;
	}

	// Expose
	global.diagnostic =
	{
		cycleSafeStringify: cycleSafeStringify,
		approximateSizeof: approximateSizeof,
		hierarchicalApproximateSizeof: hierarchicalApproximateSizeof,
		sizeToString: sizeToString,
		criticalPath: criticalPath,
		setSizesTable: setSizesTable
	};
})(this);