/**
 * Allows command line arguments to be passed through to the JS context.
 *
 * Usage from the command line:
 *
 *    ignition --js-args="--myString=foo --myFloat=3.142 --myBool=true --myFlag"
 *
 * Each of these args will be exposed via commandLine.args.myVariableName, e.g.
 *
 *    commandLine.args.myString;   // "foo"
 *    commandLine.args.myFloat;    // 3.142
 *    commandLine.args.myBool;     // true
 *    commandLine.args.myFlag;     // true
 *
 * The types are intelligently converted into corresponding JS types based on
 * inferring what the intended type is meant to be (things that look like booleans
 * get converted to real booleans, things that look like numbers get converted to
 * real numbers etc). If you pass an arg with no value (like myFlag above), it'll
 * get converted to a boolean with a value of true.
 */
(function(global)
{
	var args = {};
	var rawArgs = {};

	function parseArgs(argsString)
	{
		for (var i = 0; i < argsString.length; i++)
		{
			if (argsString[i] === '-')
			{
				i = readArg(argsString, i);
			}
		}
	}

	function addRawArg(argName, argValue)
	{
		if (argValue === '')
		{
			rawArgs[argName] = true;
		}
		else
		{
			rawArgs[argName] = interpretStringValue(argValue);
		}
	}

	function readArg(argsString, i)
	{
		// Remove the - or -- from the start of the string
		var nameStartIndex = advanceUntil(argsString, /[^-]/, i);

		// Read the arg name by reading until the next = or space character
		var nameEndIndex = advanceUntil(argsString, /[= ]/, nameStartIndex);
		var name = argsString.substring(nameStartIndex, nameEndIndex);

		// If the name is followed by an = sign, read the value
		if (argsString[nameEndIndex] === '=')
		{
			// Read until the next space character or the end of the string
			var valueStartIndex = nameEndIndex + 1;
			var valueEndIndex = advanceUntil(argsString, / /, valueStartIndex);
			var value = argsString.substring(valueStartIndex, valueEndIndex);

			args[name] = interpretStringValue(value);

			return valueEndIndex;
		}
		// Otherwise, interpret it as a flag arg
		else
		{
			args[name] = true;

			return nameEndIndex;
		}
	}

	function advanceUntil(string, pattern, startIndex)
	{
		for (var i = startIndex; i < string.length; i++)
		{
			if (pattern.test(string[i]))
			{
				return i;
			}
		}

		return i;
	}

	// Analyses whether the supplied string appears to be a boolean or a number,
	// and if so returns it as one of those types. Otherwise, leaves it as a string.
	function interpretStringValue(value)
	{
		if (value === 'true')
			return true;
		else if (value === 'false')
			return false;
		else if (value === parseFloat(value).toString())
			return parseFloat(value);
		else
			return value;
	}

	// Expose
	global.commandLine =
	{
		args: args,
		rawArgs: rawArgs,
		parseArgs: parseArgs,
		addRawArg: addRawArg
	};
})(this);