(function(global)
{
	var idCounter = 0;
	var timeoutsById = {};
	var intervalsById = {};

	function generateId()
	{
		return idCounter++;
	}

	function buildTimerInfo(id, func, delay, args)
	{
		var info =
		{
			id: id,
			func: func,
			delay: delay,
			args: Array.prototype.slice.call(args, 2),
			prevTime: Date.now()
		};

		return info;
	}

	function process(container, afterCall)
	{
		for (var id in container) if (container.hasOwnProperty(id))
		{
			var info = container[id];

			if (Date.now() - info.prevTime >= info.delay)
			{
				try
				{
					info.func.apply(global, info.args);
				}
				catch(e)
				{
					console.error("Exception leaked from setTimeout/setInterval callback:",
						e.message, ", stack:", e.stack);
				}
				afterCall(info);
			}
		}
	}

	function clearAllTimeouts()
	{
		for (var prop in timeoutsById)
		{
			if (timeoutsById.hasOwnProperty(prop))
			{
				delete timeoutsById[prop];
			}
		}

		for (var prop in intervalsById)
		{
			if (intervalsById.hasOwnProperty(prop))
			{
				delete intervalsById[prop];
			}
		}
	}

	function processAllTimeouts()
	{
		process(timeoutsById, deleteTimeout);
		process(intervalsById, updatePrevCallTime);
	}

	function bindTimeoutTick()
	{
		if (!timing.tick.has(processAllTimeouts, global))
		{
			timing.tick.add(
				processAllTimeouts,
				global,
				0,
				signals.SignalOwner.FRAMEWORK);
		}
	}

	function deleteTimeout(info)
	{
		delete timeoutsById[info.id];
	}

	function updatePrevCallTime(info)
	{
		info.prevTime = Date.now();
	}

	bindTimeoutTick();

	/**
	 * Creates a timeout to call the provided function after a delay.
	 *
	 * Note that the maximum resolution of the timer is bound by the frequency
	 * with which Ignition's timing.tick signal is being fired - if the maximum
	 * frequency of the signal is for it to fire every 50 ms then no timers will
	 * be fired more quickly than that.
	 *
	 * @see https://developer.mozilla.org/en-US/docs/Web/API/Window.setTimeout
	 * @param {Function} func
	 * @param {number} delay
	 * @returns {number}
	 */
	function setTimeout(func, delay)
	{
		var id = generateId();

		timeoutsById[id] = buildTimerInfo(id, func, delay, arguments);

		return id;
	}

	/**
	 * Clears the timeout associated with the provided ID.
	 *
	 * @see https://developer.mozilla.org/en-US/docs/Web/API/Window.clearTimeout
	 * @param {number} id
	 */
	function clearTimeout(id)
	{
		delete timeoutsById[id];
	}

	/**
	 * Creates an interval to call the provided function periodically.
	 *
	 * Note that the maximum resolution of the interval is bound by the frequency
	 * with which Ignition's timing.tick signal is being fired - if the maximum
	 * frequency of the signal is for it to fire every 50 ms then no intervals
	 * will be fired more quickly than that.
	 *
	 * @see https://developer.mozilla.org/en-US/docs/Web/API/Window.setInterval
	 * @param {Function} func
	 * @param {number} delay
	 * @returns {number}
	 */
	function setInterval(func, delay)
	{
		var id = generateId();

		intervalsById[id] = buildTimerInfo(id, func, delay, arguments);

		return id;
	}

	/**
	 * Clears the interval associated with the provided ID.
	 *
	 * @see https://developer.mozilla.org/en-US/docs/Web/API/Window.clearInterval
	 * @param {number} id
	 */
	function clearInterval(id)
	{
		delete intervalsById[id];
	}

	// Expose
	global.setTimeout = setTimeout;
	global.clearTimeout = clearTimeout;
	global.setInterval = setInterval;
	global.clearInterval = clearInterval;
	global.clearAllTimeouts = clearAllTimeouts;
})(this);