/* Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

#ifndef PLUGIN_INSPECTOR_EXTERNAL_INSPECTOR_CHANNELS_INSPECTOR_CHANNEL_CONNECTION_H_
#define PLUGIN_INSPECTOR_EXTERNAL_INSPECTOR_CHANNELS_INSPECTOR_CHANNEL_CONNECTION_H_

#include "inspector/InspectorConfig.h"
#include "inspector/channels/ChannelDefs.h"
#include "inspector/channels/messaging/Message.h"
#include "inspector/channels/messaging/responses/MessageResponse.h"
#include "inspector/channels/IInspectorChannelReceiver.h"
#include "inspector/channels/IInspectorChannelNotifier.h"

#include <crypto/HashedString.h>
#include <DataStructures.h>

namespace ignition
{
namespace inspector
{

class IInspectorChannelSender;

typedef core::unordered_map<NotificationTypeHash, bool> NotificationTypesToEnabledDisabledFlags;
typedef core::unordered_map<CommandTypeHash, bool> CommandTypesToEnabledDisabledFlags;
typedef core::unordered_map<CommandTypeHash, crypto::HashedString> CommandTypeHashesToStrings;

class PLUGININSPECTOR_API InspectorChannelConnection
{
public:
	InspectorChannelConnection(
			const ChannelName& name,
			IInspectorChannelSender* sender,
			IInspectorChannelReceiver* receiver,
			IInspectorChannelNotifier* notifier);
	virtual ~InspectorChannelConnection() {};

	virtual void notify(
			const NotificationType& notification);
	virtual void notify(
			const NotificationType& notification,
			const IInspectorSerializable& info);
	virtual void respond(
			Message::MessageId originalMessageId,
			Message::InspectorClientId requesterId,
			CommandTypeHash command,
			const MessageResponse& response);

	void enableNotificationType(const NotificationType& type);
	void disableNotificationType(const NotificationType& type);

	virtual MessageResponse execute(
			CommandTypeHash commandHash,
			Message::MessageId originalMessageId,
			Message::InspectorClientId requesterId);
	virtual MessageResponse execute(
			CommandTypeHash commandHash,
			const MessageInfo& info,
			Message::MessageId originalMessageId,
			Message::InspectorClientId requesterId);

	void enableCommandType(const CommandType& type);
	void disableCommandType(const CommandType& type);

	const ChannelName& getName() const;
	IInspectorChannelSender* getSender() const;
	IInspectorChannelReceiver* getReceiver() const;
	IInspectorChannelNotifier* getNotifier() const;

protected:
	bool _notificationTypeIsEnabled(const NotificationType& type);
	bool _commandTypeIsEnabled(CommandTypeHash type);
	void _mapCommandTypeHashToString(const CommandType& type);

	ChannelName _name;
	IInspectorChannelSender* _sender;
	IInspectorChannelReceiver* _receiver;
	IInspectorChannelNotifier* _notifier;

	NotificationTypesToEnabledDisabledFlags _notificationTypes;
	CommandTypesToEnabledDisabledFlags _commandTypes;
	CommandTypeHashesToStrings _commandTypeStrings;
};

} // namespace inspector
} // namespace ignition

#endif // PLUGIN_INSPECTOR_EXTERNAL_INSPECTOR_CHANNELS_INSPECTOR_CHANNEL_CONNECTION_H_
