-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local ffi = require('ffi')

local FocusSignalsManager = class(function(self, focusManager, isViewsSupportEnabled)
	self:setFocusManager(focusManager)
	self._isViewsSupportEnabled = isViewsSupportEnabled
end)

function FocusSignalsManager:setFocusManager(focusManager)
	self._focusManager = focusManager
end

function FocusSignalsManager:dispatchAllSignals(name, node, ...)
	local success = success
	success = self:dispatchGlobalSignal(name, node, ...) and success
	if self._isViewsSupportEnabled then
		success = self:dispatchViewHandleSignal(name, node, ...) and success
	end
	return success
end

function FocusSignalsManager:dispatchGlobalSignal(name, node, ...)
	if type(name) ~= "string" then
		log.warn("FocusSignalsManager: signal name must be a string. " ..
				"Value passed in is of type '" .. type(name) .. "'.")
	elseif self._focusManager == nil then
		log.warn("FocusSignalsManager: can't dispatch a global signal, " ..
				"FocusManager was not provided.")
	else
		local args = {...}
		if #args == 0 then
			self._focusManager:dispatchJsSignal(name, node)
		elseif #args == 2 then
			self._focusManager:dispatchMouseJsSignal(
					name, node, args[1], args[2])
		elseif #args == 4 then
			self._focusManager:dispatchMouseMoveJsSignal(
					name, node, args[1], args[2], args[3], args[4])
		else
			log.warn("FocusSignalsManager: can't dispatch a JS signal with " ..
					#args .. "arguments. Expecting 0, 2 or 4.")
			return false
		end

		return true
	end

	return false
end

function FocusSignalsManager:dispatchViewHandleSignal(signalName, node, ...)
	local signal = self:_getOrCreateSignal(signalName, node)
	if signal then
		local signalParams = ReflectableValueVectorUserData.new()

		self:_appendSignalParameter(signalParams, node:getId())
		self:_appendSignalParameter(signalParams, ffi.string(node:getName()))
		self:_appendRetainedObjectId(signalParams, node)
		for i, arg in ipairs({...}) do
			self:_appendSignalParameter(signalParams, arg)
		end

		signal:dispatch(signalParams)
		return true
	end

	return false
end

function FocusSignalsManager:dispatchInternalViewHandleSignal(signalName, node)
	local signal = self:_getOrCreateSignal(signalName, node)
	if signal then
		signal:dispatch(node)
		return true
	end

	return false
end

function FocusSignalsManager:_appendSignalParameter(signalParams, parameter)
	local data = ReflectableValueUserData.new()

	if type(parameter) == "number" then
		if parameter == math.floor(parameter) then
			data:setInt(parameter)
		else
			data:setFloat(parameter)
		end
	elseif type(parameter) == "string" then
		data:setString(parameter)
	elseif type(parameter) ~= "nil" then
		log.error("Can't append signal parameter of type '" ..
				type(parameter) .. "'")
		return false
	end

	signalParams:push(data)
	return true
end

function FocusSignalsManager:_appendRetainedObjectId(signalParams, node)
	local data = ReflectableValueUserData.new()

	local retainedObjectId = self:_getRetainedObjectId(node)
	if retainedObjectId then
		data:setId(retainedObjectId)
	end

	signalParams:push(data)
end

function FocusSignalsManager:_getRetainedObjectId(node)
	if context ~= nil then
		local view = context:getNodesRegistry():getViewElement(node:getId())
		if not view then
			return nil
		end

		-- The 'ViewElement' in the registry may actually be a Directive
		-- instead of a view, so we need to find the view that's retaining
		-- the model (if it exists...)
		while view and not view.getRetainedObjectId do
			view = view:getParent()
		end

		if view and view.getRetainedObjectId then
			local retainedObjectId = view:getRetainedObjectId()
			if not retainedObjectId then
				log.info("FocusSignalsManager: will dispatch signal " ..
						"for node " .. node:getId() .. " (view.typeName='" ..
						view.typeName .. "') without a retained object")
			end

			return retainedObjectId
		end
	end

	return nil
end

function FocusSignalsManager:_getOrCreateSignal(signalName, node)
	if type(signalName) ~= "string" then
		log.warn("FocusSignalsManager: signal name must be a string. " ..
				"Value passed in is of type '" .. type(signalName) .. "'.")
	elseif node and (context ~= nil) then
		local viewElement =
				context:getNodesRegistry():getViewElement(node:getId())

		if viewElement then
			local signalTarget = viewElement
			if viewElement:instanceOf(IDirective) then
				signalTarget = viewElement:getParent()
			end

			--[[
				It's possible that a node could be added to the scene graph, but not
				the view graph, which could cause the following code to fail
				(viewElement:getParent() will return nil). A solution to this would
				be to walk up the scene graph, and use the scene graph's node id
				as the parameter to get the viewElement from the node registry.
			]]
			local signal = signalTarget:getSignal(signalName)
			if not signal then
				signal = signalTarget:addSignal(signalName)
			end

			return signal
		end
	end

	return nil
end

return FocusSignalsManager
