-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local ffi = require('ffi')
local bit = require('bit')
local TreeUtils = plugins.focus:require('utils.TreeUtils')

local NodeDataOffsets =
{
	NODE_COUNT = 1,
	SOURCE_DATA = 9,
	TARGET_IDENTIFERS = 2,
	TARGET_DATA = 13
}

local CoordinateSpace =
{
	COORDINATE_SPACE_LOCAL = 0,
	COORDINATE_SPACE_WORLD = 1,
	COORDINATE_SPACE_SCREEN = 2
}

local FocusCalculator = class(function(self, isViewsSupportEnabled)
	self._isViewsSupportEnabled = isViewsSupportEnabled
	self._nodeDistanceCache = {}
	self._nodeAngleCache = {}
	self._coordinateSpace = CoordinateSpace.COORDINATE_SPACE_SCREEN
	self._focusableNodes = {}
	self._focusManager = nil
end)

function FocusCalculator:setFocusManager(focusManager)
	self._focusManager = focusManager
end

function FocusCalculator:setFocusableNodeSet(focusableNodes)
	self._focusableNodes = focusableNodes
end

function FocusCalculator:getFocusableNodeSet()
	return self._focusableNodes
end

function FocusCalculator:useLocalCoordinateSpace()
	self._coordinateSpace = CoordinateSpace.COORDINATE_SPACE_LOCAL
end

function FocusCalculator:useWorldCoordinateSpace()
	self._coordinateSpace = CoordinateSpace.COORDINATE_SPACE_WORLD
end

function FocusCalculator:useScreenCoordinateSpace()
	self._coordinateSpace = CoordinateSpace.COORDINATE_SPACE_SCREEN
end

function FocusCalculator:findNextNode(sourceNode, direction)
	self:_flushCaches()

	if not sourceNode then
		return nil
	end

	local bestCandidate = nil

	-- Get a list of all nodes which lie in the direction we're trying
	-- to pass focus from the source node
	local allCandidates = self:_findAllNodesInDirectionFromSourceNode(
		sourceNode, direction)

	-- If there's no candidate, focus doesn't change
	if #allCandidates == 0 then
		return sourceNode
	end

	-- Sort the list of candidates based on several criteria, to establish
	-- which candidate is the most suitable
	bestCandidates = self:_findBestCandidatesToPassFocusTo(
		sourceNode, allCandidates, direction)

	for i, bestCandidate in ipairs(bestCandidates) do

		-- In some situations, a node's parent might want to control which of its
		-- children receives focus whenever any of its children is chosen as the
		-- next focus target. This method gives it a chance to seize control.
		if self._isViewsSupportEnabled then
			bestCandidate = self:_checkWhetherParentWantsToHandleFocusAssignment(
				bestCandidate, sourceNode, direction)
		end

		-- if focus hasn't changed, don't need to check if direction matches
		if bestCandidate == sourceNode then
			return bestCandidate
		end

		-- Catch edge cases where focus is passed to candiates with oppose the
		-- focus direction
		bestCandidate = self:_checkDirectionMatchesBestCandidate(
			bestCandidate, sourceNode, direction)

		if bestCandidate then
			return bestCandidate
		end
	end

	return sourceNode
end

function FocusCalculator:findNodeToClick(coords)
	local validFlags = 
	{
		scene.InteractivityFlag.FOCUS_ON_MOUSE_CLICK,
		scene.InteractivityFlag.SELECT_ON_MOUSE_CLICK
	}

	return self:_selectNodeAtPoint(coords, validFlags)
end

function FocusCalculator:findNodeToHover(coords)
	return self:_selectNodeAtPoint(
			coords, {scene.InteractivityFlag.MOUSE_HOVERABLE})
end

function FocusCalculator:_findAllNodesInDirectionFromSourceNode(
	sourceNode, direction)

	local normalisedDirection = {}
	if direction == 'up' then
		normalisedDirection = {0,1,0}
	elseif direction == 'down' then
		normalisedDirection = {0,-1,0}
	elseif direction == 'left' then
		normalisedDirection = {-1,0,0}
	elseif direction == 'right' then
		normalisedDirection = {1,0,0}
	end

	local nodeData = self._focusManager:getSuitableFocusableNodes(
		sourceNode, normalisedDirection, self._coordinateSpace)

	local dataOffset = 0
	local numNodes = nodeData[dataOffset]
	dataOffset = dataOffset + NodeDataOffsets.NODE_COUNT

	-- Cache the currently focused node
	self:_cacheNodeProperties(sourceNode:getId(),
		nodeData, dataOffset)
	dataOffset = dataOffset + NodeDataOffsets.SOURCE_DATA

	local matchedNodes = {}
	local longestDistance = 0

	-- Lets copy the matched nodes in, and build the cached centres and bounds.
	-- We also need to check that the node is in the correct view layer, as we
	-- don't perform this check in native code.
	for i = 1, numNodes do
		local offset = dataOffset

		local nodeIdx = nodeData[offset] + 1 -- Lua indexes start from 1!
		local node    = self._focusableNodes[nodeIdx]
		local nodeId  = nodeData[offset + 1]

		if self:_shouldFocusOnKeyPress(node) and
				self:_nodesAreInTheSameFocusLayer(node, sourceNode) then
			self:_cacheNodeProperties(nodeId, nodeData, offset + NodeDataOffsets.TARGET_IDENTIFERS)

			local distance = self._nodeDistanceCache[nodeId]
			if distance > longestDistance then
				longestDistance = distance
			end

			if node then
				table.insert(matchedNodes, node)
			end
		end

		dataOffset = dataOffset + NodeDataOffsets.TARGET_DATA
	end

	-- Now normalize the distances
	local multiplier = 1 / longestDistance
	for k, v in pairs(self._nodeDistanceCache) do
		self._nodeDistanceCache[k] = v * multiplier
	end

	return matchedNodes
end

function FocusCalculator:_shouldFocusOnKeyPress(node)
	return node:hasInteractivityFlag(scene.InteractivityFlag.FOCUS_ON_KEY_PRESS)
end

function FocusCalculator:_cacheNodeProperties(
	nodeId, nodeData, offset)

	-- Skip bounds and centre data (9 floats).
	-- See FocusManager.cpp for reference of array layout.
	self._nodeDistanceCache[nodeId] = nodeData[offset + 9]
	self._nodeAngleCache[nodeId]    = nodeData[offset + 10]
end

function FocusCalculator:_findBestCandidatesToPassFocusTo(
	sourceNode, targetNodes, direction)
	if #targetNodes == 0 then
		return nil
	end

	local directionScores = self:_getPositionScoresForNodes(
		sourceNode, targetNodes, direction)

	local pathToRoot = self:_getPathToRootTable(sourceNode, targetNodes)

	local distancesFromCommonAncestor =
			self:_getDistancesFromCommonAncestor(sourceNode, targetNodes, pathToRoot)

	-- The policy is: try to keep the focused node within the same subtree
	--
	--     A        e.g. suppose the current focused node is "D" and the
	--    / \            candidates are "E" and "F". We'll give priority to
	--   B   C           "E" as it's in the same subtree. This can be verified
	--  / \   |          using the distance from the source node to the common
	-- D   E  F          ancestor with the new node.
	--
	-- In case both the candidate nodes are in the same subtree, the priority is
	-- given by the "physical" distance between the currently focused node and
	-- the new one.
	table.sort (targetNodes, function(a, b)
		local distanceSourceToAncestorA = distancesFromCommonAncestor[a:getId()]
		local distanceSourceToAncestorB = distancesFromCommonAncestor[b:getId()]

		if distanceSourceToAncestorA == distanceSourceToAncestorB then
			local aScore = directionScores[a:getId()]
			local bScore = directionScores[b:getId()]

			return bScore > aScore
		else
			return distanceSourceToAncestorA < distanceSourceToAncestorB
		end
	end)

	return targetNodes
end

function FocusCalculator:_getPositionScoresForNodes(
	originNode, subjectNodes, direction)

	local scores = {}

	for i = 1, #subjectNodes do
		local node = subjectNodes[i]
		local nodeId = node:getId()

		-- Here we perform 1 minus the node angle similarity, because this value
		-- is derived from the directional normal dotted with the normalized node
		-- vector. This means that 0 = perpendicular to the direction while 1 means
		-- parallel to the direction. As our scoring is based on lower=better, we
		-- need to invert this value.
		local anglularValue = 1 - self._nodeAngleCache[nodeId]

		-- Square the angular value, so that lesser angles have less of an impact
		-- on the overall score than larger angles.
		anglularValue = anglularValue * anglularValue

		scores[nodeId] = self._nodeDistanceCache[nodeId] + anglularValue
	end

	return scores
end

function FocusCalculator:_getPathToRootTable(originNode, subjectNodes)
	local pathToRoot = {}

	pathToRoot[originNode:getId()] = TreeUtils.getPathToRoot(originNode)
	for i = 1, #subjectNodes do
		local node = subjectNodes[i]
		pathToRoot[node:getId()] = TreeUtils.getPathToRoot(node)
	end

	return pathToRoot
end

function FocusCalculator:_getDistancesFromCommonAncestor(sourceNode, targetNodes, pathsToRoot)
	local distancesFromCommonAncestor = {}

	for i = 1, #targetNodes do
		local targetNode = targetNodes[i]
		local targetNodeId = targetNode:getId()
		local _1, distance = TreeUtils.getPathIntersection(
				pathsToRoot[sourceNode:getId()],
				pathsToRoot[targetNodeId])

		distancesFromCommonAncestor[targetNodeId] = distance
	end

	return distancesFromCommonAncestor
end

function FocusCalculator:_nodesAreInTheSameFocusLayer(targetNode, sourceNode)
	if self._isViewsSupportEnabled then
		local sourceView = self:_getViewContainingNode(sourceNode)
		local targetView = self:_getViewContainingNode(targetNode)

		if (not sourceView) or (not targetView) then
			-- Special case where there's no view associated to the node
			-- (e.g. the app dynamically adds the node to the scene)
			-- The focus algorithm should still work in this case
			return true
		else
			return (sourceView:getFocusLayer() == targetView:getFocusLayer())
		end
	else
		-- If views support is disabled then there's no concept of focus layers
		return true
	end
end

function FocusCalculator:_checkWhetherParentWantsToHandleFocusAssignment(
	bestCandidate, sourceNode, direction)

	if (context ~= nil) and (bestCandidate ~= nil) then
		local oldNodeLua = nil
		if sourceNode then
			oldNodeLua = context:getNodesRegistry():getViewElement(
					sourceNode:getId())
		end

		local newNodeLua =
				context:getNodesRegistry():getViewElement(bestCandidate:getId())

		local intercepted = false
		if oldNodeLua and oldNodeLua:hasParent() then
			local chosenByParent = oldNodeLua:getParent():interceptFocusLoss(
					sourceNode, bestCandidate, direction)
			intercepted = (chosenByParent ~= bestCandidate)
			bestCandidate = chosenByParent
		end

		if newNodeLua and newNodeLua:hasParent() and not intercepted then
			bestCandidate = newNodeLua:getParent():interceptFocusGain(
					sourceNode, bestCandidate, direction)
		end
	end

	return bestCandidate
end

function FocusCalculator:_checkDirectionMatchesBestCandidate(
	bestCandidate, sourceNode, direction)

	local candidate = bestCandidate

	if candidate then
		local sourcePosition = self:_getNodePosition(sourceNode)
		local candidatePosition = self:_getNodePosition(bestCandidate)

		if direction == 'up' then
			candidate = (candidatePosition.y > sourcePosition.y)
				and bestCandidate or nil

		elseif direction == 'down' then
			candidate = (candidatePosition.y < sourcePosition.y)
				and bestCandidate or nil

		elseif direction == 'left' then
			candidate = (candidatePosition.x < sourcePosition.x)
				and bestCandidate or nil

		elseif direction == 'right' then
			candidate = (candidatePosition.x > sourcePosition.x)
				and bestCandidate or nil

		else
			error('Invalid direction')
		end

	end

	return candidate
end

function FocusCalculator:_selectNodeAtPoint(coords, interactivityFlags)
	local selectedNode = nil
	local selectedNodePosition = nil
	local selectedNodeDrawIndex = nil
	local warnOnSize0 = self:_shouldWarnOnSize0(interactivityFlags)

	local function updateSelectedNode(newSelectedNode)
		selectedNode = newSelectedNode
		selectedNodePosition = self:_getNodePosition(selectedNode)
		selectedNodeDrawIndex = self:_getNodeDrawIndex(selectedNode)
	end

	for i, candidate in ipairs(self._focusableNodes) do
		for j, flag in ipairs(interactivityFlags) do
			if candidate:hasInteractivityFlag(flag) and
					self:_areCoordsWithinBounds(coords, candidate, warnOnSize0) then
				local candidatePosition = self:_getNodePosition(candidate)
				local candidateDrawIndex = self:_getNodeDrawIndex(candidate)

				if selectedNode ~= nil then
					if selectedNodePosition.z < candidatePosition.z then
						updateSelectedNode(candidate)
					elseif (selectedNodePosition.z == candidatePosition.z) and
							(selectedNodeDrawIndex < candidateDrawIndex) then
						updateSelectedNode(candidate)
					end
				else
					updateSelectedNode(candidate)
				end
			end
		end
	end

	return selectedNode
end

function FocusCalculator:_shouldWarnOnSize0(interactivityFlags)
	local InteractivityFlag = scene.InteractivityFlag
	local warnOnSize0 = false

	for i, flag in ipairs(interactivityFlags) do
		local shouldWarn = (flag == InteractivityFlag.FOCUS_ON_MOUSE_CLICK) or
				(flag == InteractivityFlag.SELECT_ON_MOUSE_CLICK)
		warnOnSize0 = warnOnSize0 or shouldWarn
	end

	return warnOnSize0
end

function FocusCalculator:_areCoordsWithinBounds(coords, node, verbose)
	local nodeBounds = self:_getNodeBounds(node)
	local sizeX = nodeBounds.right - nodeBounds.left
	local sizeY = nodeBounds.top - nodeBounds.bottom

	if verbose and ((sizeX == 0) or (sizeY == 0)) then
		log.warn("FocusCalculator: mouse interaction candidate should " ..
				"not have size 0. Node: [id=" .. node:getId() .. ", name='" ..
				ffi.string(node:getName()) .. "', partition='" ..
				ffi.string(node:getPartition()) .. "', width=" ..
				sizeX .. ", height=" .. sizeY .. "]")
	end

	return (coords[1] > nodeBounds.left) and (coords[1] < nodeBounds.right) and
			(coords[2] < nodeBounds.top) and (coords[2] > nodeBounds.bottom)
end

function FocusCalculator:_getNodePosition(node)
	return (self._coordinateSpace == CoordinateSpace.COORDINATE_SPACE_WORLD) and
			node:getWorldPosition() or node:getPosition()
end

function FocusCalculator:_getNodeDrawIndex(node)
	return (self._coordinateSpace == CoordinateSpace.COORDINATE_SPACE_WORLD) and
			node:getHierarchicalDrawIndex() or node:getDrawIndex()
end

function FocusCalculator:_getNodeBounds(node)
	if self._coordinateSpace == CoordinateSpace.COORDINATE_SPACE_WORLD then
		return node:getWorldBounds()
	elseif self._coordinateSpace == CoordinateSpace.COORDINATE_SPACE_LOCAL then
		return node:getBounds()
	else
		return node:getScreenBounds()
	end
end

function FocusCalculator:_flushCaches()
	self._nodeDistanceCache = {}
	self._nodeAngleCache = {}
end

function FocusCalculator:_getViewContainingNode(node)
	-- The nodes registry may be mapping the node to a directive rather
	-- than a view, so we need to go up in the ViewGraph until we find
	-- the view that contains this node
	local view = context:getNodesRegistry():getViewElement(node:getId())
	while view and (view:getType() == IElement.TYPE_DIRECTIVE) and view:hasParent() do
		view = view:getParent()
	end

	return view
end

return FocusCalculator
