-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.
ffi = require('ffi')
return function(namespace)
	assert(type(namespace) == 'string', 'Namespace must be a string')
	assert(_G[namespace] == nil, 'Global focus namespace must not already exist')

	local focus = {}
	_G[namespace] = focus

	-- Set the plugin alias so we may convieniently load scripts using:
	-- plugins.focus:require(...)
	plugins.setPluginAlias('focus', 'com.amazon.ignition.framework.focus.logic');

	local FocusTagsManager = plugins.focus:require('FocusTagsManager')
	local FocusSignalsManager = plugins.focus:require('FocusSignalsManager')
	local FocusForwarder = plugins.focus:require('FocusForwarder')
	local FocusCalculator = plugins.focus:require('FocusCalculator')

	local _isViewsSupportEnabled = plugins.getPluginAlias('views') ~= nil

	local _focusManager = nil
	local _focusCalculator = FocusCalculator.new(_isViewsSupportEnabled)
	local _signalsManager = FocusSignalsManager.new(nil, _isViewsSupportEnabled)
	local _tagsManager = FocusTagsManager.new()
	local _focusForwarder = FocusForwarder.new()

	local _directions = { 'up', 'down', 'left', 'right' }

	local _currentlyFocusedNodePtr = nil
	local _currentlyHoveredNodePtr = nil
	local _currentlyClickedNodePtr = nil

	local function _getSceneNodePtrForNode(node)
		local sceneNodePtr = nil
		if node then
			sceneNodePtr = SceneNodePtr.fromSceneNode(node)
		end

		return sceneNodePtr
	end

	local function _getSceneNodeForPtr(sceneNodePtr)
		local sceneNode = nil
		if sceneNodePtr ~= nil then
			sceneNode = sceneNodePtr:getSceneNode()
		end

		return sceneNode
	end

	local function _shouldSelectOnMouseClick(node)
		return node:hasInteractivityFlag(
				scene.InteractivityFlag.SELECT_ON_MOUSE_CLICK)
	end

	function focus.calculateUsingLocalCoords()
		_focusCalculator:useLocalCoordinateSpace()
	end

	function focus.calculateUsingWorldCoords()
		_focusCalculator:useWorldCoordinateSpace()
	end

	function focus.calculateUsingScreenCoords()
		_focusCalculator:useScreenCoordinateSpace()
	end

	function focus.updateFocusableNodeList(forceUpdate)
		_focusManager:updateFocusableNodeList(forceUpdate);
	end

	function focus.setFocusManager(focusManager)
		_focusManager = focusManager
		_focusCalculator:setFocusManager(_focusManager)
		_signalsManager:setFocusManager(_focusManager)
	end

	function focus.getCurrentlyFocusedNode()
		return _currentlyFocusedNodePtr
	end

	function focus.getCurrentlyHoveredNode()
		return _currentlyHoveredNodePtr
	end

	function focus.getCurrentlyClickedNode()
		return _currentlyClickedNodePtr
	end

	function focus.setFocusedNode(newFocusedNodePtr, params, supressSignals)
		local newFocusedNode = _getSceneNodeForPtr(newFocusedNodePtr)
		local currentlyFocusedNode = _getSceneNodeForPtr(_currentlyFocusedNodePtr)

		params = params or {}

		-- Focus forwarding is a views-related concept, so if views support is 
		-- disabled then we don't do any forwarding.
		if _isViewsSupportEnabled 
		and newFocusedNode 
		and not newFocusedNode:getFocusable() then
			local forwardedNode = _focusForwarder:getTarget(newFocusedNode)
			if forwardedNode then
				newFocusedNode = forwardedNode
				newFocusedNodePtr = _getSceneNodePtrForNode(newFocusedNode)
			end
		end

		_tagsManager:updateTags(
				currentlyFocusedNode, newFocusedNode,
				"focused", params.updateAncestorTags)

		if (currentlyFocusedNode ~= newFocusedNode and not supressSignals) then
			-- Dispatches the "global" event saying that the focus has changed.
			-- Works even if the newFocusedNode is null.
			-- TODO This should probably also be receiving the
			--      'currentlyFocusedNode' as a second parameter, and we
			--      could also dispatch a similar ViewHandleSignal and
			--      deprecate the distinct 'focusLost' and 'focusGained' ones
			_signalsManager:dispatchGlobalSignal(
					"focusChanged", newFocusedNode)

			if _isViewsSupportEnabled then
				_signalsManager:dispatchViewHandleSignal(
						"focusLost", currentlyFocusedNode);
				_signalsManager:dispatchViewHandleSignal(
						"focusGained", newFocusedNode);

				_signalsManager:dispatchInternalViewHandleSignal(
						"internalFocusLost", currentlyFocusedNode)
				_signalsManager:dispatchInternalViewHandleSignal(
						"internalFocusGained", newFocusedNode)
			end
		end

		_currentlyFocusedNodePtr = newFocusedNodePtr
	end

	function focus.setFocusableNodeSet(focusableNodes)
		_focusCalculator:setFocusableNodeSet(focusableNodes)
	end

	function focus.findNodeToFocus(direction)
		return _focusCalculator:findNextNode(
				_getSceneNodeForPtr(_currentlyFocusedNodePtr), direction)
	end

	function focus.findNodeToClick(posX, posY)
		return _focusCalculator:findNodeToClick({posX, posY, 0})
	end

	function focus.findNodeToHover(posX, posY)
		return _focusCalculator:findNodeToHover({posX, posY, 0})
	end

	function focus.selectNode(node, ...)
		if (node ~= nil) then
			_signalsManager:dispatchAllSignals(
					"nodeSelected", node:getSceneNode(), ...)
		end
	end

	local _keyDirections = {}
	function focus.addInputHooks()
		_keyDirections =
		{
			[input.keyCodes.KEY_UP]    = _directions[1],
			[input.keyCodes.KEY_DOWN]  = _directions[2],
			[input.keyCodes.KEY_LEFT]  = _directions[3],
			[input.keyCodes.KEY_RIGHT] = _directions[4]
		}

		input.keyPressSignal:add(focus._handleKeyPress)
		input.mouseUpSignal:add(focus._handleMouseUp)
		input.mouseDownSignal:add(focus._handleMouseDown)
		input.mouseMoveSignal:add(focus._handleMouseMove)
		input.mouseScrollSignal:add(focus._handleMouseScroll)
	end

	function focus._handleKeyPress(keyCode, keyModifier)
		focus.calculateUsingWorldCoords()

		if (input.isArrowKey(keyCode)) then
			local newNode = focus.findNodeToFocus(_keyDirections[keyCode])

			if newNode then
				focus.setFocusedNode(SceneNodePtr.fromSceneNode(newNode))
			end
		elseif keyCode == input.keyCodes.KEY_ENTER then
			focus.selectNode(_currentlyFocusedNodePtr)
		end
	end

	function focus._handleMouseDown(posX, posY, button)
		focus.calculateUsingScreenCoords()
		local clickedNode = focus.findNodeToClick(posX, posY)
		if clickedNode then
				local clickedNodePtr = SceneNodePtr.fromSceneNode(clickedNode)
			_currentlyClickedNodePtr = clickedNodePtr

		end
	end

	function focus._handleMouseUp(posX, posY, button)
		focus.calculateUsingScreenCoords()
		local clickedNode = focus.findNodeToClick(posX, posY)
		if clickedNode then
			local previouslyClickedNode = 
					_getSceneNodeForPtr(_currentlyClickedNodePtr)
		
			if (previouslyClickedNode == clickedNode) then
				local previouslyFocusedNode =
						_getSceneNodeForPtr(_currentlyFocusedNodePtr)

				if clickedNode:hasInteractivityFlag(
						scene.InteractivityFlag.FOCUS_ON_MOUSE_CLICK) then
					focus.setFocusedNode(SceneNodePtr.fromSceneNode(clickedNode))
				end

				if clickedNode:hasInteractivityFlag(
						scene.InteractivityFlag.SELECT_ON_MOUSE_CLICK) or
						(previouslyFocusedNode == clickedNode) then
					focus.selectNode(_currentlyClickedNodePtr, posX, posY)
				end
			end
		end
		_currentlyClickedNodePtr = nil
	end


	function focus._handleMouseScroll(dirX, dirY)
		-- TODO: Support mouse scrolling
	end

	function focus._handleMouseMove(posX, posY)
		focus.calculateUsingScreenCoords()

		local hoveredNode = focus.findNodeToHover(posX, posY)
		local currentlyHoveredNode =
				_getSceneNodeForPtr(_currentlyHoveredNodePtr)

		local function mouseOutCurrentlyHoveredNode()
			_tagsManager:updateTags(
					currentlyHoveredNode, nil, "hovered", true)

			_signalsManager:dispatchAllSignals(
					"mouseOut", currentlyHoveredNode, posX, posY)

			_currentlyHoveredNodePtr = nil
		end

		if hoveredNode then
			if currentlyHoveredNode then
				if (currentlyHoveredNode:getId() == hoveredNode:getId()) then
					_signalsManager:dispatchAllSignals("mouseMove",
							hoveredNode, posX, posY)
					return
				else
					mouseOutCurrentlyHoveredNode()
				end
			end

			local hoveredNodePtr = SceneNodePtr.fromSceneNode(hoveredNode)
			local newHoveredNode = _getSceneNodeForPtr(hoveredNodePtr)

			_tagsManager:updateTags(
					currentlyHoveredNode, newHoveredNode, "hovered", true)

			_signalsManager:dispatchAllSignals(
					"mouseOver", hoveredNode, posX, posY)

			_currentlyHoveredNodePtr = hoveredNodePtr

			if hoveredNode:hasInteractivityFlag(
					scene.InteractivityFlag.FOCUS_ON_MOUSE_HOVER) then
				focus.setFocusedNode(hoveredNodePtr)
			end
		elseif currentlyHoveredNode then
				mouseOutCurrentlyHoveredNode()
		end
	end

	if input then
		focus.addInputHooks()
	else
		log.warn("Input namespace is not defined. Will not add focus callbacks.");
	end

	if signalBus then
		signalBus:getSignal("requestFocus"):add(function(node, params)
			focus.setFocusedNode(_getSceneNodePtrForNode(node), params)
		end)
	end

	return focus
end
