/* Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

#ifndef PLUGIN_DEVICEINFO_DEVICEINFO_PROPERTIES_ABSTRACT_DEVICE_PROPERTIES_H_
#define PLUGIN_DEVICEINFO_DEVICEINFO_PROPERTIES_ABSTRACT_DEVICE_PROPERTIES_H_

#include "deviceinfo/DeviceInfoConfig.h"
#include "deviceinfo/properties/DeviceMode.h"

#include "meta/ReflectableValue.h"

#include "IgnitionStdint.h"
#include "DataStructures.h"
#include "SmartPointers.h"

#include <string>

namespace ignition
{
namespace deviceinfo
{

class PLUGINDEVICEINFO_API AbstractDeviceProperties
{
public:
	static const uint32_t MAX_RESOLUTION_X = 1920;
	static const uint32_t MAX_RESOLUTION_Y = 1080;

	virtual ~AbstractDeviceProperties() {}

	virtual std::string getManufacturer() = 0;
	virtual std::string getFirmwareVersion() = 0;
	virtual std::string getChipset() = 0;
	virtual std::string getModelName() = 0;
	
	//! @return the version number for the package of the platform agnostic 
	//! Ignition Framework and platform specific native code and other 
	//! shared libraries used to construct the native Ignition application.
	//! Typically this will be the Application Version shown in an App Store.
	virtual std::string getPlatformPackageVersion() = 0;
	
	virtual DeviceMode getDeviceMode() = 0;

	virtual bool supportsUHD() = 0;
	virtual bool supportsHEVC() = 0;
	virtual bool supportsAudioCodecSwitching() = 0;
	virtual bool supportsHDR() = 0;
	virtual bool supportsDolbyVisionHDR() = 0;
	virtual bool supportsOLED() = 0;

	//! Some platforms have their own "video buffer" and don't require using
	//! Ignition's VideoNodes for displaying full-screen video (in some cases,
	//! having an extra video node is not even allowed). This flag returns true
	//! if a platform has it's own equivalent of a VideoNode, meaning we don't
	//! have to create an Ignition one for video display.
	virtual bool hasPlatformVideoNode() = 0;
	virtual bool hasExternalOutput() = 0;
	virtual bool supportsHDCP() = 0;
	virtual std::string getHDCPFullVersion() = 0;
	virtual uint32_t getHDCPMajorVersion() = 0;
	virtual uint32_t getHDCPMinorVersion() = 0;

	//! The "Panel" dimensions are the raw screen dimensions retrieved from
	//! the device, normally equivalent to the maximum video output supported.
	virtual int getPanelWidth() = 0;
	virtual int getPanelHeight() = 0;

	//! The "Display" dimensions are the dimensions we use throughout Ignition
	//! for UI rendering, and window creation. It can normally be thought as
	//! the "Panel" dimensions "capped" to <MAX_RESOLUTION_X, MAX_RESOLUTION_Y>
	virtual int getDisplayWidth();
	virtual int getDisplayHeight();

	//! @return the language of the underlying OS running Ignition. The value
	//! returned is in the BCP 47 language tag format, e.g. en-US, fr-CA, etc
	virtual std::string getDeviceLanguage() = 0;

	//! Get the value for the property key. The available keys can be retrieved
	//! from getPlatformPropertyKeys. If a key is not available on the platform,
	//! default ReflectableValue (type is TYPE_NONE) will be returned.
	virtual const core::ReflectableValue getPlatformProperty(
			const std::string& key) const;

	//! @note If you change this type you must implement JS binding support.
	typedef core::vector<std::string> PlatformPropertyKeys;

	//! Get the list of property keys which are supported on the platform.
	//! The keys can be used as argument of getPlatformProperty.
	//! @return List of platform property keys.
	virtual const PlatformPropertyKeys getPlatformPropertyKeys() const;
};

typedef core::memory::SharedPtr<AbstractDeviceProperties> DevicePropertiesPtr;

} // namespace deviceinfo
} // namespace ignition

#endif // PLUGIN_DEVICEINFO_DEVICEINFO_PROPERTIES_ABSTRACT_DEVICE_PROPERTIES_H_
