local mod = {}

mod.useConfig = true;

local NEWTON_ITERATIONS = 5
local EPSILON = 0.001
local MIN_DIVISOR = 0.000001
local NOT_FOUND = "NOT_FOUND"

-- Calculate the polynomial coefficient A
local function getA(p1, p2)
	return 1.0 - 3.0 * p2 + 3.0 * p1;
end

-- Calculate the polynomial coefficient B
local function getB(p1, p2)
	return 3.0 * p2 - 6.0 * p1;
end

-- Calculate the polynomial coefficient C
local function getC(p1, p2)
	return 3.0 * p1
end

-- Cubic bezier calculation in one dimension using coefficients
local function cubicBezierCoefficient(a, b, c, t)
	return ((a * t + b) * t + c) * t;
end

-- Cubic bezier calculation in one dimention using parametric points
local function cubicBezier(p1, p2, t)
	return cubicBezierCoefficient(getA(p1, p2), getB(p1, p2), getC(p1, p2), t)
end

-- Cubic bezier derivative calculation in one dimention using coefficients
local function cubicBezierDerivativeCoefficient(a, b, c, t)
	return 3.0 * a * t * t + 2.0 * b * t + c
end

-- Newton's method to approximate T given the Bezier value, using coefficients
local function getTForProgressUsingNewton(a, b, c, progress)
	local t = progress
	local currentProgress
	local derivative

	for i = 0, NEWTON_ITERATIONS do
		currentProgress = cubicBezierCoefficient(a, b, c, t) - progress
		if math.abs(currentProgress) < EPSILON then
			return t
		end

		derivative = cubicBezierDerivativeCoefficient(a, b, c, t)
		if math.abs(derivative) < MIN_DIVISOR then
			break
		end

		t = t - currentProgress / derivative
	end

	return NOT_FOUND
end

-- Bisection method to approximate T given the Bezier value, using coefficients
local function getTForProgressUsingBisection(a, b, c, progress)
	local t = progress
	local tLeft = 0.0
	local tRight = 1.0
	local currentProgress

	while tLeft < tRight do
		currentProgress = cubicBezierCoefficient(a, b, c, t)
		if math.abs(currentProgress - progress) < EPSILON then
			break
		end
		if progress > currentProgress then
			tLeft = t
		else
			tRight = t
		end
		t = (tRight - tLeft) / 2 + tLeft
	end

	return t
end

-- Calculate parametric value T that Bezier value came from, using parametric points
local function getTForProgress(p1, p2, progress)
	local a = getA(p1, p2)
	local b = getB(p1, p2)
	local c = getC(p1, p2)

	local t = getTForProgressUsingNewton(a, b, c, progress)

	if t == NOT_FOUND then
		t = getTForProgressUsingBisection(a, b, c, progress)
	end

	return t
end

-- Ensure values in [0.0, 1.0]
local function getInRange(value)
	return math.min(math.max(value, 0.0), 1.0)
end

function mod.interpolate(from, change, progress, index, valueLength, fnConfig)
	if progress > 1.0 then
		return from + change
	elseif progress < 0.0 then
		return from
	end

	local p1x = getInRange(fnConfig[index])
	local p1y = getInRange(fnConfig[index + valueLength])
	local p2x = getInRange(fnConfig[index + (2 * valueLength)])
	local p2y = getInRange(fnConfig[index + (3 * valueLength)])

	local t = getTForProgress(p1x, p2x, progress)
	local y = cubicBezier(p1y, p2y, t)
	local position = from + (change * y)

	return position
end

return mod
