--[[
-- Builds the global namespace for the animation extension. Here we encapsulate
-- a single animator instance as a singleton on the global namespace.
-- We do this so that the animation api can appear as a singleton.
--
-- The methods as exposed by this builder may be accessed as follows:
--  animation.queue(...) -- Assuming namespace == 'animation'
--
-- This function also returns a table containing protected management methods
-- for the extension.
--
--]]
local compat = require('pl.compat') -- table.pack() support in LuaJIT

return function (namespace, animatorFfiClassName, interpolatorFfiClassName)
	assert(type(namespace) == 'string', 'namespace must be a string')
	assert(type(animatorFfiClassName) == 'string',
			'animatorFfiClassName must be a string')
	assert(type(interpolatorFfiClassName) == 'string',
			'interpolatorFfiClassName must be a string')
	assert(_G[namespace] == nil, 'global animation api must not be set already')

	_G[namespace] = {}
	local api = _G[namespace]

	local animatorFfiClass = _G[animatorFfiClassName]
	local interpolatorFfiClass = _G[interpolatorFfiClassName]

	assert(animatorFfiClass ~= nil, 'animatorFfiClass must not be nil. ' ..
			'Unable to find ffiClass ' .. animatorFfiClassName)
	assert(interpolatorFfiClass ~= nil, 'interpolatorFfiClass must not be nil. ' ..
			'Unable to find ffiClass ' .. interpolatorFfiClassName)

	-- Internal helper function for building api methods.
	-- These methods prepend the animator user data as the self method when
	-- invoking the identically named method on the FFI class.
	local function buildEncapsulation(fnName, subject, ffiClass)
		api[fnName] = function (...)
			assert(api[subject] ~= nil, namespace ..".".. subject .. ' must be set')
			return ffiClass[fnName](api[subject], ...)
		end
	end

	local function legacyQueue(args)
		log.warn(namespace .. ".queue(node, property, from, to, " ..
				"duration, easingParams, easing, ...) is deprecated! " ..
				"Instead, use the new: " .. namespace ..
				".queue(node, property, type, {params})")

		local easing = api._interpolator:getFunctionByName(args[7])
		args[7] = easing

		-- It's annoyingto have to manually unwrap this, but Lua5.1 (and
		-- thus LuaJIT) gets lost when calling 'unpack' in lists that
		-- contain 'nil', such as this one (possibly)
		return api._animator:queueFixedDuration(
				args[1], args[2], args[3], args[4],
				args[5], args[6], args[7], args[8])
	end

	local function queueFixedDuration(args)
		local easing = api._interpolator:getFunctionByName(args[3].easing)
		return api._animator:queueFixedDuration(args[1], args[2],
				args[3].from, args[3].to, args[3].duration,
				args[3].easingParams, easing, args[3].repetitions)
	end

	local function queueContinuous(args)
		return api._animator:queueContinuous(
				args[1], args[2], args[3].initialSpeed,
				args[3].maxSpeed, args[3].accelerationDuration)
	end

	-- Encapsulate the various methods we intend to expose on the binding,
	api["type"] =
	{
		FIXED_DURATION = 0,
		CONTINUOUS = 1
	}

	api["queue"] = function(...)
		assert(api._animator ~= nil, 'animator must be set')
		assert(api._interpolator ~= nil, 'interpolator must be set')

		local args = table.pack(...)
		-- We have to use the 'n' field for retrieving this list size, since
		-- the '#' operator doesn't work correctly when we have 'nil'
		-- values in the middle of the list
		if args.n >= 7 and args.n <= 8 then
			return legacyQueue(args)
		elseif args.n == 4 then
			local operationType = table.remove(args, 3)
			if operationType == api.type.FIXED_DURATION then
				return queueFixedDuration(args)
			elseif operationType == api.type.CONTINUOUS then
				return queueContinuous(args)
			else
				log.error("Argument #3 of '" .. namespace .. "." ..
						"queue' should be a valid '" .. namespace .. ".type'")
			end
		else
			log.error("'" .. namespace .. ".queue' expects 4 arguments, " ..
					"received " .. args.n)
		end
	end

	buildEncapsulation('clear', "_animator", animatorFfiClass)
	buildEncapsulation('length', "_animator", animatorFfiClass)
	buildEncapsulation('hasFunction', "_interpolator", interpolatorFfiClass)
	buildEncapsulation('loadFunction', "_interpolator", interpolatorFfiClass)
	buildEncapsulation('unloadFunction', "_interpolator", interpolatorFfiClass)

	--[[
	-- Method to allow the animator instance to be set from the
	-- extension (called of the builder)
	--
	--]]
	local function setAnimator(animator)
		api._animator = animator
	end

	local function setInterpolator(interpolator)
		api._interpolator = interpolator
	end

	local function _moveSignals()
		assert(_G['_animation_complete_signal'] ~= nil,
			'temporary complete signal has to be set beforehand')
		assert(_G['_animation_repeating_signal'] ~= nil,
			'temporary repeating signal has to be set beforehand')
		-- move the temporary global signals into the local namespace
		api.complete = _animation_complete_signal
		api.repeating = _animation_repeating_signal
		-- nil-ify the temporary global signals
		_animation_complete_signal = nil
		_animation_repeating_signal = nil
	end

	return
	{
		setAnimator = setAnimator,
		setInterpolator = setInterpolator,
		_moveSignals = _moveSignals
	}
end
