-- Path from which Lua functions are loaded
local functionSearchPath = ""

-- The master list of all loader interpolation functions
local functionTable = {}

-- Sets the lua function loading path
function setFunctionSearchPath(searchPath)
	functionSearchPath = searchPath
end

-- Gets the lua function loading path
function getFunctionSearchPath()
  return functionSearchPath;
end

-- Check to see if a given function is available
function isFunctionAvailable(fnName)
	-- Check to see if the module (function) has already been loaded into memory
	if package.loaded[fnName] then
		return true
	else
		-- Otherwise, build a searcher to che for existance
		for _, searcher in ipairs(package.searchers or package.loaders) do
			local searchFn = searcher(fnName)
			if type(searchFn) == 'function' then
				package.preload[fnName] = searchFn
				return true
			end
		end
		return false
	end
end


-- Attempts to load an interpolation module to the table.
-- If this fails for any reason, false is returned
function safeLoad(fnName)
	local module

	-- Wrap the require function
    local function requiref(fnName)
        module = lazyRequire(fnName)
    end

    result = pcall(requiref,fnName)

    if (result) then
        -- Check for an interpolation function
        if (type(module.interpolate) ~= "function") then
        	log.error("Interpolator.safeLoad: Interpolation module " .. fnName .. " is missing the require 'interpolate' function")
        	return false
        end

        -- Register the function
        functionTable[fnName] = module
        return true
    else
    	-- Loaded with an error
    	log.error("Interpolator.safeLoad: Interpolation module " .. fnName .. " threw an error on load")
    	-- TODO We should look at giving a more descriptive error for developers
    	return false
    end
end


-- Attempts to load a named interpolation function
function loadFunction (fnName)
	local result = false;

	-- Be very restrictive in where we load modules from
	local sysPath = package.path
	package.path = functionSearchPath

	-- Check to see if it 'may' be possible to load the function
	if (isFunctionAvailable(fnName)) then
		result = safeLoad(fnName)
	else
		log.error("Interpolator.loadFunction: Unable to locate interpolation function " .. fnName)
	end

	-- Reset System package path
	package.path = sysPath

	-- Print some errors for debugging
	if (not result) then
		log.error("Interpolator.loadFunction:Loading failed for interpolation function " .. fnName)
	end

	return result
end


-- Attempts to unload a named interpolation function
function unloadFunction (fnName)
	-- Check if the names function exists
	if (functionTable[fnName] == nil) then
		return false
	else
		functionTable[fnName] = nil
		package.loaded[fnName] = nil
		package.preload[fnName] = nil
		_G[fnName] = nil
		return true;
	end
end


function interpolate (from, to, progress, fnName, fnParams)
	local valueLength = #from
	local fnModule = functionTable[fnName]

	-- Check that the interpolation module exists
	if (fnModule == nil) then
		log.error("Interpolator.interpolate: No interpolator by the name " .. fnName)
		return false
	end

	local result = {}
	for index = 1, valueLength do
		local change = to[index] - from[index]
		local resultComponent = fnModule.interpolate(
				from[index], change, progress, index, valueLength, fnParams)
		table.insert(result, resultComponent)
	end

	return result
end
