/**
 * Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 */

(function()
{
	var RESTART_ON_ERROR_TIMEOUT_MS = 30 * 1000;

	var fetchFileModule = require('./fetchFile.js');
	var createFetchFileWithRetries = fetchFileModule.createFetchFileWithRetries;

	var restartModule = require('./restart.js');
	var initRestarter = restartModule.initRestarter;

	var fetchIndexJsModule = require('./fetchIndexJs.js');
	var fetchIndexJsWithRetries = fetchIndexJsModule.fetchIndexJsWithRetries;

	var RequestManager = require('./requestManager.js');

	// Naming it this way so it's not confused with the global one.
	var _restartInterpreter = restartModule.restartInterpreter;

	var ConnectivityErrorMessage = require('./connectivityErrorMessage.js');
	var SplashScreenManager = require('./splashScreenManager.js');

	var networkUtilsModule = require('./networkUtils.js');
	var getFilenameFromUrl = networkUtilsModule.getFilenameFromUrl;

	initRestarter();

	var connectivityService = network.connectivity.service.get();

	function loadPlayer()
	{
		if (global.player && !player.hawaii.isLoaded())
		{
			log.info('Reloading Ruby player');
			player.hawaii.reload();
		}
	}

	function unloadPlayer()
	{
		if (global.player && player.hawaii.isLoaded())
		{
			log.info('Unloading ruby player');
			player.hawaii.unload();
		}
	}

	var BOOTSTRAP_STATES = {
		LAUNCH_BLAST: 0,
		INIT: 1,
		FETCH_INTERPRETER: 2,
		BACKGROUND: 3,
		DISCONNECTED: 4,
		CHECK_CONNECTION: 5,
		FETCH_INDEX_JS: 7
	};

	var stateReverseLookup = {};

	Object.keys(BOOTSTRAP_STATES).forEach(function (name)
	{
		var number = BOOTSTRAP_STATES[name];
		stateReverseLookup[number] = name;
	});

	function prettyPrintState(stateNumber)
	{
		if (stateNumber.length)
		{
			return '[' + stateNumber.map(prettyPrintState).join(',') + ']';
		}
		return '"' + stateReverseLookup[stateNumber] + ' (' + stateNumber + ')"';
	}

	var BUCKET_NAME = 'com.amazon.ignition.app.blast';
	var SECURE_BUCKET_NAME = 'com.amazon.ignition.app.blast.secure';

	function StateManager(options)
	{
		var self = this;
		this.options = options;
		this.state = BOOTSTRAP_STATES.INIT;

		this.connectivityProblemMessage = new ConnectivityErrorMessage(
				this.options.errorMessageImages);
		this.splashScreenManager = new SplashScreenManager();

		this.fetchInProgress = null;

		this._restartInterpreter = _restartInterpreter;

		this.transitions = [
			{
				name: 'start',
				from: [BOOTSTRAP_STATES.INIT,
					BOOTSTRAP_STATES.DISCONNECTED
				],
				to: BOOTSTRAP_STATES.CHECK_CONNECTION
			},

			{
				name: 'fetchIndexJs',
				from: [
					BOOTSTRAP_STATES.CHECK_CONNECTION,
					BOOTSTRAP_STATES.FETCH_INDEX_JS,
					BOOTSTRAP_STATES.FETCH_INTERPRETER
				],
				to: BOOTSTRAP_STATES.FETCH_INDEX_JS
			},

			{name: 'fetch',
				from: BOOTSTRAP_STATES.FETCH_INDEX_JS,
				to: BOOTSTRAP_STATES.FETCH_INTERPRETER
			},

			{
				name: 'launchBlast',
				from: [
					BOOTSTRAP_STATES.FETCH_INDEX_JS,
					BOOTSTRAP_STATES.FETCH_INTERPRETER
				],
				to: BOOTSTRAP_STATES.LAUNCH_BLAST
			},

			{
				name: 'background',
				from: [BOOTSTRAP_STATES.CHECK_CONNECTION,
					BOOTSTRAP_STATES.FETCH_INTERPRETER,
					BOOTSTRAP_STATES.FETCH_INDEX_JS
				],

				to: BOOTSTRAP_STATES.BACKGROUND
			},
			{
				name: 'foreground',
				from: BOOTSTRAP_STATES.BACKGROUND,
				to: BOOTSTRAP_STATES.CHECK_CONNECTION
			},

			{
				name: 'waitForConnection',
				from: [BOOTSTRAP_STATES.CHECK_CONNECTION,
					BOOTSTRAP_STATES.FETCH_INTERPRETER,
					BOOTSTRAP_STATES.FETCH_INDEX_JS
				],
				to: BOOTSTRAP_STATES.DISCONNECTED
			},

			{
				name: 'restart',
				from: [BOOTSTRAP_STATES.INIT,
					BOOTSTRAP_STATES.CHECK_CONNECTION,
					BOOTSTRAP_STATES.FETCH_INDEX_JS,
					BOOTSTRAP_STATES.FETCH_INTERPRETER,
					BOOTSTRAP_STATES.LAUNCH_BLAST],
				to: BOOTSTRAP_STATES.INIT
			}
		];

		this.transitions.forEach(function(transition)
		{
			self[transition.name] = function()
			{
				var message;
				log.info('from: ' + prettyPrintState(transition.from )+ ' to: ' +
						prettyPrintState(transition.to) + ' name: ' + transition.name);
				if (transition.from.length)
				{
					if (transition.from.indexOf(self.state) === -1)
					{
						message = 'Invalid transition "' + transition.name +
								'": Current state is: "' +
								prettyPrintState(self.state) + ' but required: ' +
								prettyPrintState(transition.from);
						log.errorEvent('loader', 'InvalidTransition', message);
						throw new Error(message);
					}
				}
				else if (self.state !== transition.from)
				{
					message = 'Invalid transition "' + transition.name +'": Current state is: ' +
							prettyPrintState(self.state) +
						' but required: ' + prettyPrintState(transition.from);
					log.errorEvent('loader', 'InvalidTransition', message);
					throw new Error(message);
				}

				if (self.onLeave && self.onLeave[self.state])
				{
					self.onLeave[self.state].apply(self);
				}
				// XXX If it make sense, implement transition callbacks:
				// self.on && self.on[transition.name] & self.on[transition.name].apply(self);
				self.state = transition.to;
				if (self.onEnter && self.onEnter[transition.to])
				{
					self.onEnter[transition.to].apply(self);
				}
			};
		});


		// Exposing restartInterpreter so it can be triggered by blast
		global.restartInterpreter = this.restart;
	}

	StateManager.prototype.showConnectivityProblemMessage = function()
	{
		this.connectivityProblemMessage.show();
		if (!this.enterStateDisconnectedTime)
		{
			this.enterStateDisconnectedTime = metrics.getNowTimestamp();
		}
	};

	StateManager.prototype.hideConnectivityProblemMessage = function()
	{
		if (this.enterStateDisconnectedTime)
		{
			global.appLoadMetrics.msWaitingForConnection +=
					metrics.getNowTimestamp() - this.enterStateDisconnectedTime;
			this.enterStateDisconnectedTime = null;
		}
		this.connectivityProblemMessage.hide();
	};

	function onFirstFail()
	{
		this.splashScreenManager.hide(function ()
		{
			this.showConnectivityProblemMessage();
		}.bind(this));

		// Unload ruby player
		unloadPlayer();
	}

	// non-public methods used for testing
	StateManager.prototype._getStates = function()
	{
		return BOOTSTRAP_STATES;
	};

	StateManager.prototype.onEnter = {};
	StateManager.prototype.onLeave = {};

	// This case would be when we go back to the INIT state after a restart.
	StateManager.prototype.onEnter[BOOTSTRAP_STATES.INIT] = function()
	{
		this.splashScreenManager.hide();
		this.hideConnectivityProblemMessage();

		global.appLoadMetrics.appRestartTime = metrics.getNowTimestamp();
		global.appLoadMetrics.prevAppStartTime = global.appLoadMetrics.appStartTime;

		this._restartInterpreter(function()
		{
			this.start();
			this.splashScreenManager.show();
			global.restartInterpreter = this.restart;
		}.bind(this));
	};

	// State machine entry point.
	StateManager.prototype.onLeave[BOOTSTRAP_STATES.INIT] = function()
	{
		global.appLoadMetrics.msWaitingForConnection = 0;
		lifecycle.enterBackground.addOnce(this.background, this);
	};

	StateManager.prototype.onEnter[BOOTSTRAP_STATES.FETCH_INDEX_JS] = function ()
	{
		this.fetchIndexJsRequest = fetchIndexJsWithRetries(this.options,
		function (err, indexJsContents)
		{
			if (err)
			{
				if (err.name === 'ConnectivityStateError')
				{
					this.waitForConnection();
					return;
				}

				log.info('Scheduling retry for ' + (RESTART_ON_ERROR_TIMEOUT_MS / 1000) +
						' seconds.');
				this.restartOnErrorTimer = setTimeout(function () {
					log.info('retrying...');
					this.fetchIndexJs();
				}.bind(this), RESTART_ON_ERROR_TIMEOUT_MS);
				return;
			}

			eval(indexJsContents);

			this.blastConfig = BlastIgnition.getBlastConfig();
			this.resources = this.blastConfig.files;

			// TODO Do a proper check if a file is local and the other is remote.
			if (this.resources[0].location === 'local')
			{
				this.launchBlast();
				return;
			}

			this.fetch();
		}.bind(this),
		onFirstFail.bind(this));
	};

	StateManager.prototype.onLeave[BOOTSTRAP_STATES.FETCH_INDEX_JS] = function ()
	{
		if (this.fetchIndexJsRequest)
		{
			this.fetchIndexJsRequest.stop();
			this.fetchIndexJsRequest = null;
		}
		clearTimeout(this.restartOnErrorTimer);
		this.restartOnErrorTimer = null;
	};

	StateManager.prototype.onEnter[BOOTSTRAP_STATES.LAUNCH_BLAST] = function()
	{
		if(this.options.clearStorage)
		{
			storage.getBucket(BUCKET_NAME, storage.SQLITE).eraseAll();
			storage.getBucket(SECURE_BUCKET_NAME, storage.SECURE).eraseAll();
		}

		// Make sure player is loaded to load the app.
		loadPlayer();

		lifecycle.enterBackground.remove(this.background, this);
		lifecycle.enterForeground.remove(this.foreground, this);

		var resourceTypeHandler = {
			'application/javascript': function(path)
			{
				var metricFilename = path.replace(/^.*[\\\/]/, '');
				global.appLoadMetrics.includes["startTime_" + metricFilename] = metrics.getNowTimestamp();

				include(path, true);

				global.appLoadMetrics.includes["endTime_" + metricFilename] = metrics.getNowTimestamp();
			},
			'application/lua': function(path)
			{
				style.loadStylesheet(path);
			}
		};

		var error = false;
		this.resources.forEach(function(resource)
		{
			var handler = resourceTypeHandler[resource.type];

			if (!handler)
			{
				log.warnEvent('loader', 'UnknownFileTypeError', 'Skipping file "' +
					resource.filepath + '" since there is not resourceTypeHandler associated' +
					' for type "' + resource.type + '".');
				return;
			}

			try
			{
				handler(resource.filepath);
			} catch (e)
			{
				log.errorEvent('loader', 'InvalidFileError', 'Error evaluating "' +
						resource.filepath + '". Message: ' + e.message + ': ' + e.stack);
				error = true;
			}

		});

		if (error)
		{
			this.restart();
			return;
		}

		var blastConfig = JSON.parse(JSON.stringify(this.blastConfig));

		blastConfig.files = this.resources.map(function (resource)
		{
			resource.url = resource.filepath;
			resource.location = 'local';
			return resource;
		});

		log.info('before blastConfig done');
		this.blastConfig.done(blastConfig);
		log.info('after blastConfig done');

		this.hideConnectivityProblemMessage();
		this.splashScreenManager.show();
		log.info('Time spent waiting for connection: ' +
				global.appLoadMetrics.msWaitingForConnection + ' ms');
	};

	StateManager.prototype.onLeave[BOOTSTRAP_STATES.LAUNCH_BLAST] = function()
	{
		lifecycle.enterBackground.remove(this.background, this);
		lifecycle.enterForeground.remove(this.foreground, this);
		this.splashScreenManager.hide();
	};

	StateManager.prototype.onEnter[BOOTSTRAP_STATES.BACKGROUND] = function()
	{
		log.info('Entering background mode');

		unloadPlayer();
		lifecycle.notifyLifecycleTransitionFinished();

		lifecycle.enterForeground.addOnce(this.foreground, this);
	};
	StateManager.prototype.onLeave[BOOTSTRAP_STATES.BACKGROUND] = function()
	{
		log.info('Entering foreground mode');

		lifecycle.notifyLifecycleTransitionFinished();

		this.splashScreenManager.show();
		lifecycle.enterBackground.addOnce(this.background, this);
	};

	StateManager.prototype.onEnter[BOOTSTRAP_STATES.DISCONNECTED] = function()
	{
		var self = this;

		function onConnectivityChange()
		{
			connectivityService.stateChange.remove(onConnectivityChange, this);
			self.start();
		}

		connectivityService.stateChange.addOnce(onConnectivityChange, this);
	};

	StateManager.prototype.onLeave[BOOTSTRAP_STATES.DISCONNECTED] = function()
	{
		// Nothing to do here but defined for completeness
	};

	StateManager.prototype.onEnter[BOOTSTRAP_STATES.CHECK_CONNECTION] = function()
	{
		var self = this;
		if (connectivityService.getState() === network.connectivity.state.CONNECTED)
		{
			self.fetchIndexJs();
			return;
		}
		self.waitForConnection();
	};

	StateManager.prototype.onLeave[BOOTSTRAP_STATES.CHECK_CONNECTION] = function()
	{
		// Nothing to do here but defined for completeness
	};


	function onRequestsDone(err, res)
	{
		if (err)
		{
			if (err.name === 'ConnectivityStateError')
			{
				this.waitForConnection();
				return;
			}

			this.fetchIndexJs();
			return;
		}

		this.resources = this.resources.map(function (resource)
		{
			resource.filepath = res[resource.url];
			return resource;
		});

		this.launchBlast();
	}

	StateManager.prototype.onLeave[BOOTSTRAP_STATES.FETCH_INTERPRETER] = function()
	{
		this.fetchInProgress.stop();
	};

	StateManager.prototype.onEnter[BOOTSTRAP_STATES.FETCH_INTERPRETER] = function()
	{
		this.fetchInProgress = new RequestManager(createFetchFileWithRetries,
				onRequestsDone.bind(this), onFirstFail.bind(this));

		this.resources = this.resources.map(function (resource)
		{
			resource.filepath = getFilenameFromUrl(resource.url);
			return resource;
		});

		this.resources.forEach(function(resource)
		{
			this.fetchInProgress.addRequest(resource.url, resource.filepath);
		}.bind(this));
		this.fetchInProgress.start();

	};

	module.exports = function(options)
	{
		if (!options)
		{
			log.errorEvent('loader', 'OptionsException', 'Error: Invalid options.');
			return;
		}

		if (!options.errorMessageImages)
		{
			log.errorEvent('loader', 'OptionsException',
				'"errorMessageImages" must be set.');
			return;
		}

		return new StateManager(options);
	};

}());
