/**
 * Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 */

(function () {

	function purgeGlobals()
	{
		// Getting rid of all globals that were created since bootstrapping
		var allGlobals = Object.getOwnPropertyNames(global);
		var disposableGlobals = allGlobals.filter(function(n)
		{
			return global.originalGlobals.indexOf(n) === -1;
		});

		disposableGlobals.forEach(function(g)
		{
			delete global[g];
		});
	}

	function purgeSceneGraph()
	{
		var rootNode = scene.getRootNode();
		var nChildren = rootNode.getNumberOfChildren();
		for(var i = (nChildren - 1); i >= 0; --i)
		{
			var child = rootNode.getChildAtIndex(i);
			if (!(child instanceof scene.CameraNodeProxy))
			{
				rootNode.removeChild(child);
			}
		}
	}

	function shutdownPlayer(callback)
	{
		if (!global.player || !player.hawaii.isLoaded())
		{
			log.info('shutdownPlayer: Ignoring since player is not loaded.');
			callback();
			return;
		}
		// Must wait for the "shutdown_complete" message. It will be sent whether
		// the player is running or not.
		player.hawaii.registerMessageListener(function(jsonMessage){
			var message;
			try {
				message = JSON.parse(jsonMessage);
			} catch (err) {
				log.errorEvent('restart', 'PlayerShutdownException',
					 'Failed to parse Ruby message "' + jsonMessage + '"');
			}

			if (message && message.cmd === "shutdown_complete")
			{
				// Clear the "tick" interval
				clearInterval(tickIntervalId);

				// Let's try to get some memory back!
				player.hawaii.sendMessage(
					JSON.stringify({
						cmd: "device_garbage_collection_request",
						params:
						{
						}
					})
				);

				player.hawaii.registerMessageListener(function(){});

				if (typeof callback === "function")
				{
					callback();
				}
			}

			// If the player state changed then it was being used.
			// We probably don't want this to happen so logging.
			else if(message && message.cmd === "player_state_change")
			{
				log.errorEvent('restart', 'PlayerShutdownException',
					'Blast was restarted during playback');
			}
		});

		// Tick messages is what triggers responses so must keep sending them
		var tickIntervalId = setInterval(function()
		{
			player.hawaii.sendMessage("tick");
		}, 100);

		// Stops the player in case it is running. Do nothing in case it isn't
		player.hawaii.sendMessage(
			JSON.stringify({
				cmd: "player_control",
				params:
				{
					call_function: "stop"
				}
			})
		);

		// Shuts down the player, will *ALWAYS* trigger a "shutdown_complete" response
		player.hawaii.sendMessage(
			JSON.stringify({
				cmd: "player_control",
				params:
				{
					call_function: "shutdown"
				}
			})
		);

	}

	function restartInterpreter(done)
	{
		log.info("Restarting blast interpreter");
		var delay = 100;

		setTimeout(function()
		{
			purgeGlobals();

			// Destroying the scene graph
			purgeSceneGraph();

			// Clear all stylesheets
			style.clearLoadedFiles();

			// Disabling timeouts
			clearAllTimeouts();

			// Remove all native signals listeners
			signalHelpers.removeListenersByOwnerGlobally(signals.SignalOwner.APPLICATION);

			// Must ensure player will be shutdown, this operation is async
			shutdownPlayer(function()
			{
				// GC all the pending objects, must be done after the scene is
				// flushed to get rid of all scene nodes references in JS
				scene.afterNextFlush(function()
				{
					spidermonkey.forceGC();
					log.info("Blast interpreter stopped");

					// Only after another scene flush will the scene nodes be deleted
					scene.afterNextFlush(function()
					{
						log.info("Blast interpreter restarting");
						initRestarter();
						done();
					});
				});
			});
		}, delay);
	}

	function initRestarter()
	{
		global.originalGlobals = Object.getOwnPropertyNames(global);

		// TODO (djanbell): implement restart input sequence in a platform-generic way
		// This sequence is only valid for samsung
		var restartKeySequence =
		[
			117, //GREEN
			38, //UP
			40, //DOWN
			37, //LEFT
			39, //RIGHT
			117 //GREEN
		];

		var currentInputSequence = [];

		input.keyPress.add(function(code, mod)
		{
			var sequenceTimeoutId;

			if (code === restartKeySequence[currentInputSequence.length])
			{
				currentInputSequence.push(code);
				if (currentInputSequence.length === restartKeySequence.length)
				{
					clearTimeout(sequenceTimeoutId);
					currentInputSequence = [];
					global.restartInterpreter();
				}
			}
			else
			{
				clearTimeout(sequenceTimeoutId);
				currentInputSequence = [];
			}

			// Only allowing 5 seconds to input the full sequence
			if (currentInputSequence.length === 1)
			{
				sequenceTimeoutId = setTimeout(function()
				{
					currentInputSequence = [];
				}, 5000);
			}
		});

	}

	module.exports = {
		initRestarter: initRestarter,
		restartInterpreter: restartInterpreter
	};

}());
