/**
 * Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 */

(function()
{
	function RequestManager(requestFactory, onceCompleteListener, onceFailed)
	{
		this._state = RequestManager.STATES.READY;
		this._requests = [];
		this._requestCount = 0;
		this._onceCompleteListener = onceCompleteListener;
		this._onceFailed = onceFailed;
		this._requestFactory = requestFactory;
		this._results = {
			err: null,
			res: {}
		};
	}

	RequestManager.STATES = {
		'READY': 0,
		'RUNNING': 1,
		'DONE': 2,
		'ERROR': 3
	};

	RequestManager.prototype._onRequestComplete = function(index, err, res)
	{

		if (this._state === RequestManager.STATES.DONE)
		{
			return;
		}

		var request = this._requests[index];
		this._requestCount--;

		if (this._state === RequestManager.STATES.ERROR)
		{
			if (this._requestCount === 0)
			{
				this._state = RequestManager.STATES.DONE;
				return;
			}

			return;
		}

		if (err)
		{
			this._results.res = null;
			this._results.err = err;
			this._state = RequestManager.STATES.ERROR;

			this._onceCompleteListener(this._results.err, this._results.res);
			return;
		}

		this._results.res[request.url] = res;

		if (this._requestCount === 0)
		{
			this._onceCompleteListener(this._results.err, this._results.res);
		}
	};

	RequestManager.prototype.addRequest = function(url, filepath)
	{
		if (this._state !== RequestManager.STATES.READY)
		{
			throw new Error('Requests already started.');
		}
		this._requestCount++;
		var req = this._requestFactory({url: url, localFilename: filepath, maxRetries: 3},
				this._onRequestComplete.bind(this, this._requests.length), this._onceFailed);
		this._requests.push(req);
	};

	RequestManager.prototype.start = function()
	{
		if (this._state !== RequestManager.STATES.READY)
		{
			throw new Error('Requests already started.');
		}

		if (this._requestCount.length === 0)
		{
			log.warnEvent('requestManager', 'EmptyRequestList', 'List of entries to fetch' +
					' is empty.');
		}

		this._requests.forEach(function(request)
		{
			request.start();
		});

		this._state = RequestManager.STATES.RUNNING;

	};

	RequestManager.prototype.stop = function()
	{
		if (this._state !== RequestManager.STATES.RUNNING)
		{
			return;
		}

		this._requests.forEach(function(request)
		{
			log.info('Stop ' + request.url);
			request.stop();
		});

		this._requests.splice(0, this._requests.length);
		this._requestCount = 0;
	};

	module.exports = RequestManager;
}());