/**
 * Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 */

(function ()
{
	var networkUtilsModule = require('./networkUtils.js');
	var appendQueryStringArgs = networkUtilsModule.appendQueryStringArgs;
	var ConnectivityStateError = networkUtilsModule.ConnectivityStateError;

	var connectivityService = network.connectivity.service.get();

	// Jitter to add to retries
	var DEFAULT_JITTER_MS = 400;

	// Amount of times we will retry loading content. The limit would be ~27 minutes.
	var MAX_RETRIES_FOR_EXPONENTIAL_BACKOFF = 11;

	// Milliseconds between each retry to connect attempt.
	var TIME_BETWEEN_RETRIES_MS = 800;

	function RetriesExceededError(message)
	{
		this.name = 'RetriesExceededError';
		this.message = (message || '');
	}

	RetriesExceededError.prototype = Object.create(Error.prototype);

	function request(options, cb)
	{
		var url = options.url;
		var downloadedFilename = options.localFilename;
		var localPath;

		var metricFilename = downloadedFilename && downloadedFilename.replace(/^.*[\\\/]/, '');

		var request = network.http.request.create();
		var response = request.getResponse();

		if (downloadedFilename)
		{

			log.info('Including ' + url + '...');
			localPath = plugins.resolvePath(
					'user://com.amazon.ignition.app.stark/web-cache/' + downloadedFilename);

			response.setResponseConsumer(network.http.data.createFileConsumer(localPath));
		}
		else
		{
			response.setResponseConsumer(network.http.data.createBufferConsumer());
		}

		// Append required blast querystring parameters
		url = appendQueryStringArgs(url);
		request.setUrl(url);

		if (downloadedFilename)
		{
			global.appLoadMetrics.requests['startTime_' + metricFilename] = metrics.getNowTimestamp();
		}

		request.complete.addOnce(function()
		{
			var status = String(response.getStatus());

			if (status[0] === '2')
			{
				log.info('Loading ' + url + ' completed with ' + status + ' status code.');

				if (localPath)
				{
					global.appLoadMetrics.requests['endTime_' + metricFilename] = metrics.getNowTimestamp();

					return cb(null, localPath);
				}
				else
				{
					return cb(null, response.getResponseConsumer().asString());
				}
			}

			if (connectivityService.getState() !== network.connectivity.state.CONNECTED)
			{
				cb(new ConnectivityStateError());
				return;
			}

			log.errorEvent('requestWithRetries', 'DownloadException', 'Download of ' + url +
					' failed with HTTP status code: ' + status);

			return cb(new Error('Download of ' + url +
					' failed with HTTP status code: ' + status));

		});
		request.send();

		return request;
	}


	function FetchWithRetryManager(options)
	{
		this.retryCount = 0;
		this.timer = null;

		options = options || {};

		this.url = options.url;
		this.localFilename = options.localFilename;
		this.cb = options.cb;
		this.firstFailCb = options.firstFailCb;
		this.hasFailedBefore = false;
		this.maxRetries = options.maxRetries;
	}

	FetchWithRetryManager.prototype._computeNextRetry = function ()
	{
		var retryCountForBackoff = Math.min(MAX_RETRIES_FOR_EXPONENTIAL_BACKOFF,
				this.retryCount);
		var nextRetry = TIME_BETWEEN_RETRIES_MS * (1 << (retryCountForBackoff - 1)) +
				(Math.random() * (retryCountForBackoff * DEFAULT_JITTER_MS)) | 0;
		return nextRetry;
	};

	FetchWithRetryManager.prototype._onRemoteIncludeDone = function(err, res)
	{
		if (err)
		{

			if (connectivityService.getState() !== network.connectivity.state.CONNECTED)
			{
				this.cb(new ConnectivityStateError());
				return;
			}

			// TODO Refactor me to be a request counter.
			if (!this.hasFailedBefore && this.firstFailCb)
			{
				this.firstFailCb(err);
				this.hasFailedBefore = true;
			}

			this.retryCount++;
			if (this.maxRetries !== null && this.maxRetries !== undefined)
			{
				log.info('Retry ' + this.retryCount + ' out of ' + this.maxRetries);
			}
			if (this.retryCount >= this.maxRetries)
			{
				this.timer = null;
				return this.cb(new RetriesExceededError('Maximum amount of retries exceeded: ' +
						this.maxRetries));
			}
			var nextRetry = this._computeNextRetry();
			log.warnEvent('start', 'RetryDownloadFailed', 'File fetch failed. Retrying' +
					' in ' + nextRetry + 'ms. ');
			this.request = null;
			this.timer = setTimeout(function()
			{
				this._fetch();
			}.bind(this), nextRetry);
			return;
		}

		this.cb(null, res);
		this.timer = null;
	};

	FetchWithRetryManager.prototype._fetch = function()
	{
		this.request = request({url: this.url, localFilename: this.localFilename},
				this._onRemoteIncludeDone.bind(this));
	};

	FetchWithRetryManager.prototype.start = function()
	{
		this._fetch();
	};

	FetchWithRetryManager.prototype.stop = function()
	{
		clearTimeout(this.timer);
		this.timer = null;
		if (this.request)
		{
			this.request.complete.removeAll();
			this.request = null;
		}
	};

	function requestWithRetries(options, cb)
	{
		var fetchWithRetryManager = new FetchWithRetryManager({
			url: options.url,
			localFilename: options.localFilename,
			firstFailCb: options.firstFailCb,
			maxRetries: options.maxRetries,
			cb: cb
		});
		fetchWithRetryManager.start();
		return fetchWithRetryManager;
	}

	function createFetchFileWithRetries(options, cb, firstFailCb)
	{
		var fetchWithRetryManager = new FetchWithRetryManager({
			url: options.url,
			localFilename: options.localFilename,
			maxRetries: options.maxRetries,
			cb: cb,
			firstFailCb: firstFailCb
		});

		return fetchWithRetryManager;
	}

	function createFetchFileOnce(options, cb)
	{
		function SimpleRequest()
		{
			this.request = null;
			this.url = options.url;
		}

		SimpleRequest.prototype.start = function ()
		{
			this.request = request(options, cb);
		};

		SimpleRequest.prototype.stop = function ()
		{
			this.request.complete.removeAll();
		};

		return new SimpleRequest();
	}


	module.exports = {
		requestWithRetries: requestWithRetries,
		request: request,
		createFetchFileWithRetries: createFetchFileWithRetries,
		createFetchFileOnce: createFetchFileOnce
	};

}());
