/**
 * Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 */

(function()
{

	// If a given region has more than one supported language, this is the amount of
	// milliseconds the connectivity error message is going to be displayed before toggling
	// to another language.
	// Since the message is ~80 words long and a slow reading is around 2 words per second
	// then 40 seconds should be enough to read the whole message.
	var INTERVAL_BETWEEN_LOCALE_TOGGLE_MS = 40 * 1000;

	var DEFAULT_ERROR_MESSAGE =
		'static://com.amazon.ignition.app.stark/images/clientError_us.jpg';

	function ConnectivityErrorMessage(messagesToShow)
	{
		this.messagesToShow = messagesToShow;
		this.timer = null;
		this.imageNode = null;
		this.showing = false;
	}

	ConnectivityErrorMessage.prototype.show = function()
	{
		if (this.showing)
		{
			return;
		}

		log.info('Showing connectivity error message');

		var height = renderer.getHeight();
		var width = renderer.getWidth();

		var ratio = height / 1080;
		var cameraNode = scene.getCameraNode();
		var position = cameraNode.getOriginalPosition();

		position[0] = 0.0;
		position[1] = 0.0;

		var cameraZ = position[2] * ratio;
		position[2] = cameraZ;

		cameraNode.setPosition(position);

		var rootNode = scene.getRootNode();
		rootNode.setPosition([0, 0, 0]);

		this.imageNode = new scene.ImageNodeProxy();

		this.imageNode.setSourceChangeMode(
			scene.SourceChangeMode.KEEP_PREVIOUS_UNTIL_NEW_TEXTURE_IS_READY
		);

		this.imageNode.setWidth(width);


		if (this.messagesToShow.length === 0)
		{
			this.imageNode.setSource(DEFAULT_ERROR_MESSAGE);

		}
		else if (this.messagesToShow.length === 1)
		{
			this.imageNode.setSource(this.messagesToShow[0].path);
		}
		else
		{
			this.index = 0;
			this.imageNode.setSource(this.messagesToShow[this.index].path);
			this.timer = setInterval(function()
			{
				this.pendingAnimation = animation.queue(this.imageNode, 'alpha', 1, 0, 200,
					'outQuad', true);

				this.pendingAnimation.complete.addOnce(this._onFadeOutComplete, this);

			}.bind(this), INTERVAL_BETWEEN_LOCALE_TOGGLE_MS);
		}

		this.showing = true;

		rootNode.addChild(this.imageNode);
	};

	ConnectivityErrorMessage.prototype._onFadeOutComplete = function()
	{
		this.index = (this.index + 1) % this.messagesToShow.length;
		this.imageNode.setSource(this.messagesToShow[this.index].path);
		this.imageNode.loaded.addOnce(this._onImageLoaded, this);

	};

	ConnectivityErrorMessage.prototype._onImageLoaded = function()
	{
		animation.queue(this.imageNode, 'alpha', 0, 1, 200, 'outQuad', true);
	};

	ConnectivityErrorMessage.prototype.hide = function()
	{
		if (!this.showing)
		{
			return;
		}

		var rootNode = scene.getRootNode();

		log.info('Hiding connectivity error message');

		clearInterval(this.timer);
		if (this.pendingAnimation && this.pendingAnimation.complete &&
		this.pendingAnimation.complete.remove)
		{
			this.pendingAnimation.complete.remove(this._onFadeOutComplete, this);
		}
		if (this.imageNode && this.imageNode.loaded &&
			this.imageNode.loaded.remove)
		{
			this.imageNode.loaded.remove(this._onImageLoaded, this);
		}
		this.timer = null;

		rootNode.removeChild(this.imageNode);
		scene.getCameraNode().setPosition(scene.getCameraNode().getOriginalPosition());
		this.imageNode = null;

		this.showing = false;
	};

	module.exports = ConnectivityErrorMessage;
}());
