/* jshint moz: true, esnext: true */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

/* Copyright (c) 2014 - 2015 Panasonic Corporation */

/* This Source Code Form is "Incompatible With Secondary Licenses",
 * as defined by the Mozilla Public License, v. 2.0. */

"use strict";

// Don't modify this, instead set services.push.debug.
let gDebuggingEnabled = false;

function debug(s) {
  if (gDebuggingEnabled)
    dump("-*- PanaPushService.jsm: " + s + "\n");
}

// function debug(aMsg) {
//   Cc["@mozilla.org/consoleservice;1"]
//     .getService(Ci.nsIConsoleService)
//     .logStringMessage("--*-- PanaPushService.js : " + aMsg);
// }

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;
const Cr = Components.results;

Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/Timer.jsm");
Cu.import("resource://gre/modules/Preferences.jsm");
Cu.import("resource://gre/modules/Promise.jsm");

var threadManager = Cc["@mozilla.org/thread-manager;1"].getService(Ci.nsIThreadManager);

this.EXPORTED_SYMBOLS = ["PushService"];

const prefs = new Preferences("services.push.");
// Set debug first so that all debugging actually works.
gDebuggingEnabled = prefs.get("debug");

const kCHILD_PROCESS_MESSAGES = ["Push:Register", "Push:Unregister",
                                 "Push:Registrations"];

// unixsocket name
const panaSunPath = '/tmp/ua_b2g_socket';
//const panaSunPath = '/tmp/b2g_pana_push_sock'; //for unit test
// unixsocket states
const STATE_SHUT_DOWN = 0;
const STATE_WAITING_FOR_READY = 1;
const STATE_READY = 2;


/**
 * The implementation of the SimplePush system. This runs in the B2G parent
 * process and is started on boot.
 * Persistent storage (flash) and server connection will manage panasonicTV system.
 */
this.PushService = {
  observe: function observe(aSubject, aTopic, aData) {
    var self = this;
    switch (aTopic) {
      /*
       * We need to call uninit() on shutdown to clean up things that modules aren't very good
       * at automatically cleaning up, so we don't get shutdown leaks on browser shutdown.
       */
      case "xpcom-shutdown":
        this.uninit();
        break;
      case "nsPref:changed":
        if (aData == "services.push.debug") {
          gDebuggingEnabled = prefs.get("debug");
        }
        break;
      case "timer-callback": //for ready. (network connection)
        debug("timer-callback");
        if (aSubject == this._requestTimeoutTimer) {
          if (this._requestQueue.length == 0) {
            this._requestTimeoutTimer.cancel();
	    debug("timer cancel in observe()");
          } else {
            // All disposal if top request timed out.
            let duration = Date.now() - this._requestQueue[0][2];
	    if (duration > this._requestTimeout){
	      debug ("All disposal of _requestQueue by timeout");
	      this._requestQueue.forEach(function(req, index) {
		let requestID = req[1].requestID;
		self._pendingRequests[requestID]
                  .deferred.reject({requestID: requestID, status: 0, error: "TimeoutError"});
		delete self._pendingRequests[requestID];
	      });
	      this._requestQueue = [];
	    }
	  }
	}
        break;
      case "webapps-clear-data":
        debug("webapps-clear-data");

        let data = aSubject.QueryInterface(Ci.mozIApplicationClearPrivateDataParams);
        if (!data) {
          debug("webapps-clear-data: Failed to get information about application");
          return;
        }

        // Only remove push registrations for apps.
        if (data.browserOnly) {
          return;
        }

        let appsService = Cc["@mozilla.org/AppsService;1"]
                            .getService(Ci.nsIAppsService);
        let manifestURL = appsService.getManifestURLByLocalId(data.appId);
        if (!manifestURL) {
          debug("webapps-clear-data: No manifest URL found for " + data.appId);
          return;
        }
          if (this._us){
	    let data = {
	      'message' : 'webappClearData',
	      'manifestURL': manifestURL
	    };
	    this._us.send(data);
	  }
        break;
    }
  },

  // keeps requests buffered if the server disconnects or is not connected
  _requestQueue: [],
  _pendingRequests: {},
  _currentState: STATE_SHUT_DOWN,
  _requestTimeout: 0,
  _requestTimeoutTimer: null,
  _retryFailCount: 0,

  init: function() {
    let self = this;
    debug("init()");
    if (!prefs.get("enabled"))
        return null;

    let ppmm = Cc["@mozilla.org/parentprocessmessagemanager;1"]
                 .getService(Ci.nsIMessageBroadcaster);

    kCHILD_PROCESS_MESSAGES.forEach(function addMessage(msgName) {
        ppmm.addMessageListener(msgName, this);
    }.bind(this));

    this._requestTimeout = prefs.get("requestTimeout");
    debug("requestTimeout:" +  this._requestTimeout);

    Services.obs.addObserver(this, "xpcom-shutdown", false);
    Services.obs.addObserver(this, "webapps-clear-data", false);

    // Debugging
    prefs.observe("debug", this);

    let us = this._us = new UnixSocket();
    us.init();
    us.onopen = function (e) {
      debug("called onopen");
      self._currentState = STATE_WAITING_FOR_READY;
    };
    us.onclose = function (e) {
      //when the daemon is dead.
      debug("called onclose");

      //For disposal in the socket out.
      for (let requestID in self._pendingRequests) {
	let tmp = self._pendingRequests[requestID];
	if(tmp.inSocket){
	  tmp.deferred.reject({requestID: requestID, status: 0, error: "Network Error"});
	  delete self._pendingRequests[requestID];
	}
      }
      self._currentState = STATE_SHUT_DOWN;
      //setTimeout(us.init.bind(us), 3000);      //for recovery.
    };
    us.ondata = this._ondata.bind(this);
    us.onsend = function (e) {
      debug("called onsend" + e.data);
      if(self._pendingRequests[e.data]) {
	self._pendingRequests[e.data].inSocket = true;
      }
    };
    this._started = true;
  },

  uninit: function() {
    if (!this._started)
      return;

    debug("uninit()");

    prefs.ignore("debug", this);
    Services.obs.removeObserver(this, "webapps-clear-data", false);
    Services.obs.removeObserver(this, "xpcom-shutdown", false);

    if (this._requestTimeoutTimer) {
      this._requestTimeoutTimer.cancel();
      debug("timer cancel in unint()");
    }

    if(this._us){
      this._us.onclose = null;
      this._us.close();      
    }
    debug("shutdown complete!");
  },

  _ondata: function(event) {
    debug("called ondata" + event.data);

    let reply = JSON.parse(event.data);

    // A whitelist of protocol handlers. Add to these if new messages are added
    // in the protocol.
    let handlers = ["Ready", "Register", "Registrations", "Unregister", "Notification", "Networkdown"];

    // Build up the handler name to call from messageType.
    // e.g. messageType == "register" -> _handleRegisterReply.
    let handlerName = reply.message[0].toUpperCase() +
                      reply.message.slice(1).toLowerCase();

    if (handlers.indexOf(handlerName) == -1) {
      debug("No whitelisted handler " + handlerName + ". message: " +
            reply.message);
      return;
    }

    let handler = "_handle" + handlerName + "Reply";
    if (typeof this[handler] !== "function") {
      debug("Handler whitelisted but not implemented! " + handler);
      return;
    }
    this[handler](reply);
  },

  // It has been called from this._ondata. (Call function name is created in the string)
  _handleReadyReply: function(reply) {
    debug("handleReadyReply()");
    if (this._currentState != STATE_WAITING_FOR_READY) {
      debug("Unexpected state " + this._currentState +
            "(expected STATE_WAITING_FOR_READY)");
      return;
    }

    function finishHandshake() {
      this._currentState = STATE_READY;
      this._processNextRequestInQueue();
    }

    finishHandshake.bind(this)();
  },

  _handleCommonReply: function(reply) {
    if (typeof reply.requestID !== "string" ||
        typeof this._pendingRequests[reply.requestID] !== "object")
      return;

    let tmp = this._pendingRequests[reply.requestID];
    delete this._pendingRequests[reply.requestID];
    if ((this._requestQueue.length == 0) &&
        this._requestTimeoutTimer){
      this._requestTimeoutTimer.cancel();
      debug("timer cancel in _handleCommonReply()");
    }
    if (reply.hasOwnProperty('error')){
      tmp.deferred.reject(reply);
    } else {
      tmp.deferred.resolve(reply);      
    }
  },

  // It has been called from this._ondata. (Call function name is created in the string)
  _handleRegisterReply: function(reply) {
    debug("handleRegisterReply()");
    this._handleCommonReply(reply);
  },

  // It has been called from this._ondata. (Call function name is created in the string)
  _handleRegistrationsReply: function(reply) {
    debug("handleRegistrationsReply()");
    let registrations = [];
    reply.records.forEach(function(pushRecord) {
      registrations.push({
          __exposedProps__: { pushEndpoint: 'r', version: 'r' },
          pushEndpoint: pushRecord.pushEndpoint,
          version: pushRecord.version
      });
    });
    let msg = {
      registrations: registrations,
      requestID: reply.requestID};
    if (reply.hasOwnProperty('error')){
      msg.error = reply.error;
    }
    this._handleCommonReply(msg);
  },

  // It has been called from this._ondata. (Call function name is created in the string)
  _handleUnregisterReply: function(reply) {
    debug("handleUnregisterReply()");
    this._handleCommonReply(reply);
  },


  // It has been called from this._ondata. (Call function name is created in the string)
  _handleNotificationReply: function(reply) {
    debug("handleNotificationReply()");
    debug("Update: " + reply.channelID + ": " + reply.version);
    
    if (reply.version === undefined) {
      debug("version does not exist");
      return;
    }
    
    let version = reply.version;
    
    if (typeof version === "string") {
      version = parseInt(version, 10);
    }
    
    if (typeof version === "number" && version >= 0) {
      // FIXME(nsm): this relies on app update notification being infallible!
      // eventually fix this
      this._notifyApp(reply);
      this._sendAckNotification(reply.channelID, version);
    }
  },

  // It has been called from this._ondata. (Call function name is created in the string)
  _handleNetworkdownReply: function(reply){
    debug("_handleNetworkdownReply()");
    this._currentState = STATE_SHUT_DOWN;
  },

  _sendAckNotification: function(channelID, version) {
    debug("_sendAckNotification()" + version);
    if (typeof channelID !== "string") {
      debug("Invalid update literal at index " + i);
    }

    if (this._us){
      this._us.send({
	message: 'ackNotification',
	channelID: channelID,
	version: version
      });
    }
  },

  _sendRequest: function(action, data, isWaitReady) {
    debug("sendRequest() " + action);

    if (typeof data.requestID !== "string") {
      debug("Received non-string requestID");
      return Promise.reject("Received non-string requestID");
    }

    let deferred = Promise.defer();
    this._pendingRequests[data.requestID] = { deferred: deferred,
					      inSocket: false     //For disposal in the socket out.
					    };    
    if (isWaitReady) {
      this._send(action, data);
    }
    else
    {
      if (this._us){
	data.message = action;
	this._us.send(data);
      }
    }
    return deferred.promise;
  },

  _send: function(action, data) {
    debug("_send()");
    //Request deletion timeout only when register.(Timeout assumed the network unconnected.)
    if (this._requestQueue.length == 0) {
      // start the timer since we now have at least one request
      debug("start timer ");
      if (!this._requestTimeoutTimer)
        this._requestTimeoutTimer = Cc["@mozilla.org/timer;1"]
                                      .createInstance(Ci.nsITimer);
       this._requestTimeoutTimer.init(this,
                                      this._requestTimeout,
                                      Ci.nsITimer.TYPE_REPEATING_SLACK);
     }
    this._requestQueue.push([action, data, Date.now()]); 
    debug("Queued " + action);
    this._processNextRequestInQueue();
  },

  _processNextRequestInQueue: function() {
    debug("_processNextRequestInQueue()");

    if (this._requestQueue.length == 0) {
      debug("Request queue empty");
      return;
    }

    if (this._currentState != STATE_READY) {
      debug("state is not STATE_READY!!");
      return;
    }

    let [action, data, ctime] = this._requestQueue.shift();
    data.message = action;

     if (this._us){
       this._us.send(data);
     }

    // Process the next one as soon as possible.
    setTimeout(this._processNextRequestInQueue.bind(this), 0);
  },

  _notifyApp: function(aPushRecord) {
    if (!aPushRecord || !aPushRecord.pageURL || !aPushRecord.manifestURL) {
      debug("notifyApp() something is undefined.  Dropping notification");
      return;
    }

    debug("notifyApp() " + aPushRecord.pageURL +
          "  " + aPushRecord.manifestURL);
    let pageURI = Services.io.newURI(aPushRecord.pageURL, null, null);
    let manifestURI = Services.io.newURI(aPushRecord.manifestURL, null, null);
    let message = {
      pushEndpoint: aPushRecord.pushEndpoint,
      version: aPushRecord.version
    };
    let messenger = Cc["@mozilla.org/system-message-internal;1"]
                      .getService(Ci.nsISystemMessagesInternal);
    messenger.sendMessage('push', message, pageURI, manifestURI);
  },

  receiveMessage: function(aMessage) {
    debug("receiveMessage(): " + aMessage.name);

    if (kCHILD_PROCESS_MESSAGES.indexOf(aMessage.name) == -1) {
      debug("Invalid message from child " + aMessage.name);
      return;
    }

    let mm = aMessage.target.QueryInterface(Ci.nsIMessageSender);
    let json = aMessage.data;
    this[aMessage.name.slice("Push:".length).toLowerCase()](json, mm);
  },

  /**
   * Called on message from the child process. aPageRecord is an object sent by
   * navigator.push, identifying the sending page and other fields.
   */
  register: function(aPageRecord, aMessageManager) {
    debug("register()");
    var msg = {
      'pageURL': aPageRecord.pageURL,
      'manifestURL': aPageRecord.manifestURL,
      'requestID' : aPageRecord.requestID
    };

    this._sendRequest("register", msg, true)
      .then(
	function(reply) {
	  let message = {"message": reply.message, "requestID": reply.requestID, "pushEndpoint": reply.pushEndpoint};
          aMessageManager.sendAsyncMessage("PushService:Register:OK", message);
        },
        function(message) {
	  debug("register() fail() error " + message.error);
          aMessageManager.sendAsyncMessage("PushService:Register:KO", message);
	});
  },


  unregister: function(aPageRecord, aMessageManager) {
    debug("unregister()");
    var msg = {
      'pageURL': aPageRecord.pageURL,
      'manifestURL': aPageRecord.manifestURL,
      'requestID' : aPageRecord.requestID,
      'pushEndpoint' : aPageRecord.pushEndpoint
    };

    //In panasonicTV, because persistent storage is a DTV side,
    //need (because of an error in the DTV side) deferred.
    this._sendRequest("unregister", msg, false)
      .then(
	function(message) {
	  aMessageManager.sendAsyncMessage("PushService:Unregister:OK", message);
	},
	function(message) {
	  debug("unregister() fail() error " + message.error);
	  aMessageManager.sendAsyncMessage("PushService:Unregister:KO", message);
	});
  },

  /**
   * Called on message from the child process
   */
  registrations: function(aPageRecord, aMessageManager) {
    debug("registrations()");
    var msg = {
      'manifestURL': aPageRecord.manifestURL,
      'requestID' : aPageRecord.requestID
    };
    
    if (aPageRecord.manifestURL) {
      this._sendRequest("registrations", msg, false)
      .then(
	function(message) {
	  aMessageManager.sendAsyncMessage("PushService:Registrations:OK", message);
	},
	function(message) {
	  debug("registrations() fail() error " + message.error);
	  aMessageManager.sendAsyncMessage("PushService:Registrations:KO", message);
	});
    }
  },
}


const CC = Components.Constructor;
const ScriptableInputStream = CC("@mozilla.org/scriptableinputstream;1",
                                 "nsIScriptableInputStream",
                                 "init");
const socketTransportService = Cc["@mozilla.org/network/socket-transport-service;1"]
                               .getService(Ci.nsISocketTransportService);
const InputStreamPump = CC(
        "@mozilla.org/network/input-stream-pump;1", "nsIInputStreamPump", "init"),
      BinaryInputStream = CC(
        "@mozilla.org/binaryinputstream;1", "nsIBinaryInputStream", "setInputStream"),
      BinaryOutputStream = CC(
        "@mozilla.org/binaryoutputstream;1", "nsIBinaryOutputStream", "setOutputStream");

let do_make_sockfile = function(sunPath) {
  let file = Components.classes["@mozilla.org/file/local;1"]
                       .createInstance(Components.interfaces.nsILocalFile);
  file.initWithPath(sunPath);
  debug('do_make_sockfile: ' + file.exists());
  // if (file.exists()) {
  //   file.remove(false);
  // }
  return file;
};

const kOPEN = 'open';
const kCLOSING = 'closing';
const kCLOSED = 'closed';

function UnixSocketEvent(type, sock, data) {
  this._type = type;
  this._target = sock;
  this._data = data;
}

UnixSocketEvent.prototype = {
  __exposedProps__: {
    type: 'r',
    target: 'r',
    data: 'r'
  },
  get type() {
    return this._type;
  },
  get target() {
    return this._target;
  },
  get data() {
    return this._data;
  }
}

this.UnixSocket = function UnixSocket() {
  this._readyState = kCLOSED;  
  this._onopen = null;
  this._onsend = null;
  this._ondata = null;
  this._onclose = null;
  this._unixDomainClientSocket =null;
}

this.UnixSocket.prototype = {
  __exposedProps__: {
    send: 'r',
    onopen: 'rw',
    onsend: 'rw',
    ondata: 'rw',
    onclose: 'rw'
  },
  // Internal
  // Raw socket streams
  _transport: null,
  _socketInputStream: null,
  _socketOutputStream: null,

  // Input stream machinery
  _inputStreamPump: null,
  _inputStreamBinary: null,

  // Output stream machinery
  _binaryOutput: null,

  // Input string
  _state: 0,
  _lengthBit: [0, 0, 0, 0],
  _length: 0,
  _str: null,
  _sendQueue: [],
  // Public accessors.
  get onopen() {
    return this._onopen;
  },
  set onopen(f) {
    this._onopen = f;
  },
  get onsend() {
    return this._onsend;
  },
  set onsend(f) {
    this._onsend = f;
  },
  get ondata() {
    return this._ondata;
  },
  set ondata(f) {
    this._ondata = f;
  },
  get onclose() {
    return this._onclose;
  },
  set onclose(f) {
    this._onclose = f;
  },

  _maybeReportErrorAndCloseIfOpen: function(status) {
    // If we're closed, we've already reported the error or just don't need to
    // report the error.
    debug("_maybeReportErrorAndCloseIfOpen");
    if (this._readyState === kCLOSED)
      return;
    this._readyState = kCLOSED;
    setTimeout(this.init.bind(this), 3000);       //for recovery.
    this.callListener("close");
  },

  _initStream: function us_initStream(client) {
    let clientOutput = client.openOutputStream(0, 0, 0);
    let clientInput = this._clientInput = client.openInputStream(0, 0, 0);
    clientInput.asyncWait(this, 0, 0, threadManager.currentThread);
    //let clientScriptableInput = new ScriptableInputStream(clientInput);

    this._inputStreamBinary = new BinaryInputStream(clientInput);
    this._state = 0;
    this._binaryOutput = new BinaryOutputStream(clientOutput);
  },

  callListener: function us_callListener(type, data) {
    if (!this["on" + type])
      return;

    this["on" + type].call(null, new UnixSocketEvent(type, this, data || ""));
  },

  init: function (){
    let sunPath = panaSunPath;
    debug(sunPath);
    // Connect a client socket to the listening socket.
    let socketName = do_make_sockfile(sunPath);
    //const allPermissions = parseInt("777", 8);
    if(!socketName.exists()){
      setTimeout(this.init.bind(this), 3000);       //for recovery.
      return;
    }
    let client = this._unixDomainClientSocket = socketTransportService.createUnixDomainTransport(socketName);
    client.setEventSink(this, threadManager.currentThread);
    this._initStream(client);
  },

  close :function(){
    debug("unixDomaiclientSocket close" + Cr.NS_OK);
    if(this._unixDomainClientSocket){
      this._unixDomainClientSocket.close(Cr.NS_OK);
    }
  },

  // nsITransportEventSink (Triggered by transport.setEventSink)
  onTransportStatus: function ts_onTransportStatus(
    transport, status, progress, max) {
    debug("status: " + status);
    if (status === Ci.nsISocketTransport.STATUS_CONNECTED_TO) {
      this._readyState = kOPEN;
      this.sendDequeue();
      this.callListener("open");
      let pump =  new InputStreamPump(this._clientInput, -1, -1, 0, 0, false);
      pump.asyncRead(this, null);
    }
  },

  // nsIAsyncInputStream (Triggered by _socketInputStream.asyncWait)
  // Only used for detecting connection refused
  onInputStreamReady: function(aStream) {
    debug("onInputStreamReady" );
    try {
      aStream.available();
    } catch (e) {
      this._maybeReportErrorAndCloseIfOpen(0x804B000C);
    }
  },

  sendDequeue: function (){
    if (this._sendQueue.length == 0){
      return;
    }
    let obj = this._sendQueue.shift();
    let data = JSON.stringify(obj);
    if (typeof(data) !== 'string') {
      debug('send: not string');
      return;
    }

    debug('send data=' + data);

    this._binaryOutput.writeUtf8Z(data);
    this.callListener("send", obj.requestID);
    setTimeout(this.sendDequeue.bind(this), 0);
  },

  send: function pus_send(data) {
    this._sendQueue.push (data);
    if (this._readyState === kOPEN){
      this.sendDequeue();
    }
    return true;
  },

  // nsIRequestObserver (Triggered by _inputStreamPump.asyncRead)
  onStartRequest: function PPTS_onStartRequest(request, context) {
    debug('onStartRequest');
  },

  // nsIRequestObserver (Triggered by _inputStreamPump.asyncRead)
  onStopRequest: function PPTS_onStopRequest(request, context, status) {
    debug('onStopRequest');
    this._maybeReportErrorAndCloseIfOpen(status);
  },
  // nsIRequestObserver (Triggered by _inputStreamPump.asyncRead)
  onDataAvailable: function PPTS_onDataAvailable(request, context, inputStream, offset, count){
    debug('onDataAvailable count=' + count);
    var streamLen = count;
    var size;
    while (streamLen > 0) {
      switch (this._state) {
      case 0: // wait for 1st bit of 32-bit length field
      case 1: // wait for 2nd bit of 32-bit length field
      case 2: // wait for 3rd bit of 32-bit length field
      case 3: // wait for 4th bit of 32-bit length field
        this._lengthBit[this._state] = this._inputStreamBinary.readBytes(1).charCodeAt(0);
        streamLen--;
        if (this._state == 3) {
          this._length =
            (this._lengthBit[0] << 24) |
            (this._lengthBit[1] << 16) |
            (this._lengthBit[2] << 8) |
            this._lengthBit[3];
          debug('onDataAvailable _length=' + this._length);
          this._str = '';
          if (this._length <= 0) {
            if (this._length < 0) {
	      debug('nagative length');
            }
            this._state = 0;
            break;
          }
        }
        this._state++;
        break;
      case 4: // wait for string
        size = Math.min(streamLen, this._length - this._str.length);
        this._str += this._inputStreamBinary.readBytes(size);
        streamLen -= size;
        if (this._str.length >= this._length) {
          debug('onDataAvailable str end');
          // The streaming data is UTF-8, but JavaScript string is UTF-16.
          // So, we have to decode from UTF-8 to UTF-16.
          var str_utf16 = decodeURIComponent(escape(this._str));
          this.callListener("data", str_utf16);

          this._state = 0;
        }
        break;
      default:
        debug('invalid state');
        break;
      }
    }
      
  }
}

