/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

const DEBUG = false;
function debug(s) {
  if (DEBUG) dump("-*- TVStoreManager: " + s + "\n");
}

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;

Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/NetUtil.jsm");
Cu.import("resource://gre/modules/FileUtils.jsm");


function TVStoreManager() {
 this.wrappedJSObject = this;
  this._callbacks = null;
}

TVStoreManager.prototype = {
  _wrap: function _wrap(obj) {
    return Cu.cloneInto(obj, this._window);
  },

  init: function(aWindow) {
    debug('init ()');
    this._window = aWindow;
    this._tvstoreImpl = Cc["@mozilla.org/dom/tvstore;1"].createInstance(Ci.nsITVStore);
    this._tvstoreImpl.setTVStoreChangedCallback(this.tvstoreDataChanged.bind(this));
  },

  get: function get(aKey) {
    debug('get()');
    let data = this._tvstoreImpl.get(aKey);
    return this._wrap(data);
  },

  set: function set(aData) {
    debug('set()');
    let keys = Object.getOwnPropertyNames(aData);
    let ret;

    if (keys.length === 0) {
      debug("No keys to change");
      return false;
    }

    for (let i = 0; i < keys.length; i++) {
      let key = keys[i];
      let value = aData[key];
      let setdata = {"key": key, "value": value};
      debug('key: ' + key + ', value:' + value);
      ret = this._tvstoreImpl.set(setdata);
    }
    
    return ret;
  },

  addObserver: function addObserver(aName, aCallback) {
    debug("addObserver " + aName);
    if (!this._callbacks) {
      this._callbacks = {};
    }
    if (!this._callbacks[aName]) {
      this._callbacks[aName] = [aCallback];
      debug('enable notification -  ' + aName);
      this._tvstoreImpl.enableTVStoreChangeNotification(aName);
    } else {
      this._callbacks[aName].push(aCallback);
    }
  },

  removeObserver: function removeObserver(aName, aCallback) {
    debug("removeObserver " + aName);
    if (this._callbacks && this._callbacks[aName]) {
      let index = this._callbacks[aName].indexOf(aCallback);
      if (index != -1) {
        this._callbacks[aName].splice(index, 1);
        if (this._callbacks[aName].length == 0) {
          delete this._callbacks[aName];
	  debug('disable notification -  ' + aName);
	  this._tvstoreImpl.disableTVStoreChangeNotification(aName);
        }
      } else {
        debug("Callback not found for: " + aName);
      }
    } else {
      debug("No observers stored for " + aName);
    }
  },

  tvstoreDataChanged: function tvstoreDataChanged(key, value) {
    debug ('data changed key: ' + key + ', value: ' + value);

    if (this._callbacks && this._callbacks[key]) {
      if (DEBUG) debug("observe callback called! " + key + " " + this._callbacks[key].length);
      let callbacks = this._callbacks[key].concat();
      callbacks.forEach(function(cb) {
        cb(this._wrap({tvstoreKey: key, tvstoreValue: value}));
      }.bind(this));
    } else {
      debug("no observers stored!");
    }
  },


  classID: Components.ID("{ad164b59-d65e-425d-8e64-a8019a674532}"),
  contractID: "@mozilla.org/tvstoremanager2;1",
  QueryInterface: XPCOMUtils.generateQI([
    Ci.nsISupports,
    Ci.nsIDOMGlobalPropertyInitializer
  ])
};

this.NSGetFactory = XPCOMUtils.generateNSGetFactory([TVStoreManager])
