/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

/* Copyright (c) 2014 - 2015 Panasonic Corporation */

/* This Source Code Form is "Incompatible With Secondary Licenses",
 * as defined by the Mozilla Public License, v. 2.0. */

"use strict";

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;
const Cr = Components.results;
const CC = Components.Constructor;

Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/Services.jsm");

const InputStreamPump = CC(
        "@mozilla.org/network/input-stream-pump;1", "nsIInputStreamPump", "init"),
      BinaryInputStream = CC(
        "@mozilla.org/binaryinputstream;1", "nsIBinaryInputStream", "setInputStream"),
      BinaryOutputStream = CC(
        "@mozilla.org/binaryoutputstream;1", "nsIBinaryOutputStream", "setOutputStream");
const PanaUnixServerSocket = CC(
        "@mozilla.org/pana-unix-server-socket;1", "nsIPanaUnixServerSocketInternal", "init");

const kOPEN = 'open';
const kCLOSING = 'closing';
const kCLOSED = 'closed';

XPCOMUtils.defineLazyModuleGetter(this, 'setTimeout', // jshint ignore:line
  'resource://gre/modules/Timer.jsm');

/*
 * Debug logging function
 */

let debug = false;
function LOG(msg) {
  if (debug)
    dump("PanaUnixSocket: " + msg + "\n");
}

/*
 * nsIPanaUnixSocketEvent object
 */

function PanaUnixSocketEvent(type, sock, data) {
  this._type = type;
  this._target = sock;
  this._data = data;
}

PanaUnixSocketEvent.prototype = {
  __exposedProps__: {
    type: 'r',
    target: 'r',
    data: 'r'
  },
  get type() {
    return this._type;
  },
  get target() {
    return this._target;
  },
  get data() {
    return this._data;
  }
}

/*
 * nsIDOMPanaUnixSocket object
 */

function PanaUnixSocket() {
  this._readyState = kCLOSED;

  this._ondata = null;
  this._onclose = null;

  this.useWin = null;
}

PanaUnixSocket.prototype = {
  __exposedProps__: {
    send: 'r',
    listen: 'r',
    ondata: 'rw',
    onclose: 'rw'
  },
  // Internal
  _hasPrivileges: null,

  // Raw socket streams
  _transport: null,
  _socketInputStream: null,
  _socketOutputStream: null,

  // Input stream machinery
  _inputStreamPump: null,
  _inputStreamBinary: null,

  // Output stream machinery
  _binaryOutput: null,

  // Input string
  _state: 0,
  _lengthBit: [0, 0, 0, 0],
  _length: 0,
  _str: null,

  // Public accessors.
  get ondata() {
    if (this._hasPrivileges !== true && this._hasPrivileges !== null) {
      LOG("PanaUnixSocket does not have permission in this context.");
      return undefined;
    }
    return this._ondata;
  },
  set ondata(f) {
    if (this._hasPrivileges !== true && this._hasPrivileges !== null) {
      LOG("PanaUnixSocket does not have permission in this context.");
      return;
    }
    this._ondata = f;
  },
  get onclose() {
    if (this._hasPrivileges !== true && this._hasPrivileges !== null) {
      LOG("PanaUnixSocket does not have permission in this context.");
      return undefined;
    }
    return this._onclose;
  },
  set onclose(f) {
    if (this._hasPrivileges !== true && this._hasPrivileges !== null) {
      LOG("PanaUnixSocket does not have permission in this context.");
      return;
    }
    this._onclose = f;
  },

  _initStream: function pus_initStream(binaryType) {
    this._socketInputStream = this._transport.openInputStream(0, 0, 0);
    this._socketOutputStream = this._transport.openOutputStream(
      Ci.nsITransport.OPEN_BLOCKING, 0, 0);

    // If the other side is not listening, we will
    // get an onInputStreamReady callback where available
    // raises to indicate the connection was refused.
    this._socketInputStream.asyncWait(
      this, this._socketInputStream.WAIT_CLOSURE_ONLY, 0, Services.tm.currentThread);

    this._inputStreamBinary = new BinaryInputStream(this._socketInputStream);
    this._state = 0;

    this._binaryOutput = new BinaryOutputStream(this._socketOutputStream);
  },

  callListener: function pus_callListener(type, data) {
    if (!this["on" + type])
      return;

    this["on" + type].call(null, new PanaUnixSocketEvent(type, this, data || ""));
  },

  /* nsIPanaUnixSocketInternal methods */
  createAcceptedParent: function pus_createAcceptedParent(transport, binaryType) {
    let that = new PanaUnixSocket();
    that._transport = transport;
    that._initStream(binaryType);

    // ReadyState is kOpen since accepted transport stream has already been connected
    that._readyState = kOPEN;
    that._inputStreamPump = new InputStreamPump(that._socketInputStream, -1, -1, 0, 0, false);
    that._inputStreamPump.asyncRead(that, null);

    return that;
  },

  /* end nsIPanaUnixSocketInternal methods */

  init: function pus_init(aWindow) {
    let principal = aWindow.document.nodePrincipal;
    let perm = Services.perms.testExactPermissionFromPrincipal(principal, "pana-ext");

    this._hasPrivileges = perm == Ci.nsIPermissionManager.ALLOW_ACTION;

    let util = aWindow.QueryInterface(
      Ci.nsIInterfaceRequestor
    ).getInterface(Ci.nsIDOMWindowUtils);

    this.useWin = XPCNativeWrapper.unwrap(aWindow);
    this.innerWindowID = util.currentInnerWindowID;
    LOG("window init: " + this.innerWindowID);
  },

  forceReleaseFD: function pus_forceReleaseFD() {
    // Force GC to release socket fd.
    setTimeout(function pus_gc_timer() {
      LOG('forceGC');
      Cu.forceGC();
    }, 5000);
  },

  observe: function(aSubject, aTopic, aData) {
    if (aTopic == "inner-window-destroyed") {
      let wId = aSubject.QueryInterface(Ci.nsISupportsPRUint64).data;
      if (wId == this.innerWindowID) {
        LOG("inner-window-destroyed: " + this.innerWindowID);

        // This window is now dead, so we want to clear the callbacks
        // so that we don't get a "can't access dead object" when the
        // underlying stream goes to tell us that we are closed
        this.ondata = null;
        this.onclose = null;

        this.useWin = null;
      }
    }
  },

  // nsIDOMPanaUnixSocket
  listen: function pus_listen(sunPath) {
    LOG('listen sunPath=' + sunPath);
    // in the testing case, init won't be called and
    // hasPrivileges will be null. We want to proceed to test.
    if (this._hasPrivileges !== true && this._hasPrivileges !== null) {
      throw new Error("PanaUnixSocket does not have permission in this context.\n");
    }

    let that = new PanaUnixServerSocket(this.useWin || this);

    that.listen(sunPath);
    return that;
  },

  send: function pus_send(data, byteOffset, byteLength) {
    if (this._hasPrivileges !== true && this._hasPrivileges !== null) {
      throw new Error("PanaUnixSocket does not have permission in this context.\n");
    }

    LOG('send data=' + data);
    if (typeof(data) !== 'string') {
      LOG('send: not string');
      return false;
    }
    try {
      this._binaryOutput.writeUtf8Z(data);
    } catch (e) {
      LOG("PanaUnixSocket send error");
      return false;
    }
    return true;
  },

  _maybeReportErrorAndCloseIfOpen: function(status) {
    this.forceReleaseFD();
    // If we're closed, we've already reported the error or just don't need to
    // report the error.
    if (this._readyState === kCLOSED)
      return;
    this._readyState = kCLOSED;

    this.callListener("close");
  },

  // nsIAsyncInputStream (Triggered by _socketInputStream.asyncWait)
  // Only used for detecting connection refused
  onInputStreamReady: function pus_onInputStreamReady(input) {
    LOG('onInputStreamReady');
    try {
      input.available();
    } catch (e) {
      // NS_ERROR_CONNECTION_REFUSED
      this._maybeReportErrorAndCloseIfOpen(0x804B000C);
    }
  },

  // nsIRequestObserver (Triggered by _inputStreamPump.asyncRead)
  onStartRequest: function pus_onStartRequest(request, context) {
    LOG('onStartRequest');
  },

  // nsIRequestObserver (Triggered by _inputStreamPump.asyncRead)
  onStopRequest: function pus_onStopRequest(request, context, status) {
    LOG('onStopRequest');
    // We call this even if there is no error.
    this._maybeReportErrorAndCloseIfOpen(status);
  },

  // nsIStreamListener (Triggered by _inputStreamPump.asyncRead)
  onDataAvailable: function pus_onDataAvailable(request, context, inputStream, offset, count) {
    LOG('onDataAvailable count=' + count);
    var streamLen = count;
    var size;
    while (streamLen > 0) {
      switch (this._state) {
      case 0: // wait for 1st bit of 32-bit length field
      case 1: // wait for 2nd bit of 32-bit length field
      case 2: // wait for 3rd bit of 32-bit length field
      case 3: // wait for 4th bit of 32-bit length field
        this._lengthBit[this._state] = this._inputStreamBinary.readBytes(1).charCodeAt(0);
        streamLen--;
        if (this._state == 3) {
          this._length =
            (this._lengthBit[0] << 24) |
            (this._lengthBit[1] << 16) |
            (this._lengthBit[2] << 8) |
            this._lengthBit[3];
          LOG('onDataAvailable _length=' + this._length);
          this._str = '';
          if (this._length <= 0) {
            if (this._length < 0) {
              LOG('nagative length');
            }
            this._state = 0;
            break;
          }
        }
        this._state++;
        break;
      case 4: // wait for string
        size = Math.min(streamLen, this._length - this._str.length);
        this._str += this._inputStreamBinary.readBytes(size);
        streamLen -= size;
        if (this._str.length >= this._length) {
          LOG('onDataAvailable str end');
          // The streaming data is UTF-8, but JavaScript string is UTF-16.
          // So, we have to decode from UTF-8 to UTF-16.
          var str_utf16 = decodeURIComponent(escape(this._str));
          this.callListener("data", str_utf16);
          this._state = 0;
        }
        break;
      default:
        LOG('invalid state');
        break;
      }
    }
  },

  classID: Components.ID("{ce6ad718-125a-11e4-b4fc-ac220bcc9310}"),

  classInfo: XPCOMUtils.generateCI({
    classID: Components.ID("{ce6ad718-125a-11e4-b4fc-ac220bcc9310}"),
    contractID: "@mozilla.org/pana-unix-socket;1",
    classDescription: "Client PanaUnix Socket",
    interfaces: [
      Ci.nsIDOMPanaUnixSocket,
    ],
    flags: Ci.nsIClassInfo.DOM_OBJECT,
  }),

  QueryInterface: XPCOMUtils.generateQI([
    Ci.nsIDOMPanaUnixSocket,
    Ci.nsIPanaUnixSocketInternal,
    Ci.nsIDOMGlobalPropertyInitializer,
    Ci.nsIObserver,
    Ci.nsISupportsWeakReference
  ])
}

this.NSGetFactory = XPCOMUtils.generateNSGetFactory([PanaUnixSocket]);
