/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp, NTRACE, ASSERT, nrdp_platform */
nrdp = {
    classname: "NrdpBridge",
    _init: false,
    _isError: false,
    _classes: { },
    _backchannels: [],
    _MAX_STACK_SIZE: 16384, // When we log the stack trace of an exception
    get isReady() { return this._init && !this._isError; },
    get debug() { return this._syncData.debug; },
    get options() { return this._syncData.options; },
    get uiQueryString() { return this.system.uiQueryString; },
    set uiQueryString(query) { nrdp.system.uiQueryString = query; },
    get bootURL() { return this.system.bootURL; },
    get started() { return this._syncData.started; },
    get trustStoreHash() { return this._syncData.trustStoreHash; },
    get capabilities() { return this._syncData.capabilities; },
    get READY() { return "READY"; },
    get COMPLETE() { return "COMPLETE"; },
    get NETWORK_ERROR() { return "NETWORK_ERROR"; },
    get ACTION_ID() { return "ACTION_ID"; },
    get ERROR() { return "ERROR"; },
    get hasSuspendHandler() { return this._syncData.hasSuspendHandler; },
    set hasSuspendHandler(v) { this._syncData.hasSuspendHandler=v; nrdp._setProperty(null, "hasSuspendHandler", v); },
    _mapProperty: function _mapProperty(object, property) {
        var classname = object;
        if (typeof classname !== "string") {
            classname = classname.classname;
            if(!classname)
                return undefined;
        }
        var classinfo = this._classes[classname];
        if(!classinfo)
            return undefined;
        if (typeof property === "string")
            return classinfo.properties.byName[property];
        return classinfo.properties.byIndex[property];
    },
    _mapMethod: function _mapMethod(object, method) {
        var classname = object;
        if (typeof classname !== "string") {
            classname = classname.classname;
            if(!classname) {
                nrdp.log.error("Could not find object classname " + object, "NRDP_SCRIPT");
                return undefined;
            }
        }
        var classinfo = this._classes[classname];
        if(!classinfo)
            return undefined;
        if (typeof method === "string")
            return classinfo.methods.byName[method];
        return classinfo.methods.byIndex[method];
    },
    _tryBackchannel: function _tryBackchannel(chan) {
        if (chan.init()) {
            ;
            nrdp._backchannel = chan;
            return true;
        } else {
            return false;
        }
    },
    setupBackchannel: function setupBackchannel() {
        var i;
        if (nrdp._backchannel) {
            return;
        }
        ;
        for (i = 0; i < nrdp._backchannels.length; i++) {
            if (typeof nrdp._backchannels[i] !== "object")
                continue;
            if (nrdp._tryBackchannel(nrdp._backchannels[i]))
                return;
        }
    },
    shutdownBackchannel: function shutdownBackchannel() {
        if (!nrdp._backchannel) {
            return;
        }
        ;
        this._sendSyncdEvent(function() { this._callEventListeners(this, { type: "shutdown" } ); }, this);
        nrdp.storage.flush();
        if(nrdp._backchannel.shutdown)
            nrdp._backchannel.shutdown();
    },
    _sendSyncdEvent: function _sendSyncdEvent(fn, that, event) { //for gibbon to hook
        fn.call(that, event);
    },
    hookInit: function hookInit(fn) {
        nrdp._hookInitFunction = fn;
    },
    _sendInitEvent: function _sendInitEvent() {
        var sendInit = this._sendSyncdEvent.bind(this, function() { this._callEventListeners(nrdp, {type: "init", status: this.READY}, true); }, this);
        if (this._hookInitFunction) {
            this._hookInitFunction(sendInit);
            delete this._hookInitFunction;
        } else {
            sendInit();
        }
    },
    init: function init() {
        if (!nrdp._init) {
            nrdp._init = true;
            ;
            nrdp.setupBackchannel();
        }
        nrdp._sendInitEvent();
    },
    now: function now() {
        if (nrdp._backchannel)
            return nrdp._invoke(null, "now");
        return Date.now();
    },
    objects: function objects(cb) {
        nrdp._fn("objects", {}, cb);
    },
    locks: function locks(cb, flags) {
        nrdp._fn("locks", {flags: flags ? flags : 0}, cb);
    },
    exit: function exit(code) {
        nrdp._invoke(null, "exit", {code: code});
    },
    ping: function ping(cb) {
        var sentTime = nrdp.mono();
        nrdp._fn("ping", {}, function(receivedTime) {
            var eventTime = nrdp.mono();
            if (cb)
                cb(sentTime, receivedTime, eventTime);
            else
                nrdp.log.warn('Bridge ping : JS -> bridge = ' + (receivedTime - sentTime) +
                              ' : bridge -> JS = ' + (eventTime - receivedTime) +
                              ' : roundtrip = ' + (eventTime - sentTime));
        });
    },
    assert: function assert(a, message) {
        if (!a) {
            message = 'JS ASSERTION FAILED' + (message ? ' : ' + message : "");
            var s = nrdp.stacktrace();
            nrdp.log.fatal(message + (s ? '\n' + s : "" ), "NRDP_SCRIPT");
            nrdp._invoke(null, "assert");
        }
    },
    stacktrace: function stacktrace() {
        try { throw new Error(); }
        catch(e) { return e.stack; }
    },
    mono: function mono() {
        return nrdp._backchannel ? nrdp._backchannel.mono() : undefined;
    },
    pmono: function pmono() {
        return nrdp._backchannel ? nrdp._backchannel.pmono() : undefined;
    },
    drmTime: function drmTime() {
        return nrdp.drmsystem.getDrmTime();
    },
    atob: function atob(s, returnTypedArray) {
        if (nrdp._backchannel && nrdp._backchannel.atob)
            return nrdp._backchannel.atob(s, returnTypedArray);
        return undefined;
    },
    btoa: function btoa(s, urlSafe, returnTypedArray) {
        if (nrdp._backchannel && nrdp._backchannel.btoa)
            return nrdp._backchannel.btoa(s, urlSafe, returnTypedArray);
        return undefined;
    },
    atoutf8: function atoutf8(s) {
        if (nrdp._backchannel && nrdp._backchannel.atoutf8)
            return nrdp._backchannel.atoutf8(s);
        return undefined;
    },
    utf8toa: function utf8toa(s) {
        if (nrdp._backchannel && nrdp._backchannel.utf8toa)
            return nrdp._backchannel.utf8toa(s);
        return undefined;
    },
    compress: function compress(data, type, binary) {
        if (nrdp._backchannel && nrdp._backchannel.compress)
            return nrdp._backchannel.compress(data, type, binary);
        return undefined;
    },
    uncompress: function compress(data, type, returnTypedArray) {
        if (nrdp._backchannel && nrdp._backchannel.compress)
            return nrdp._backchannel.uncompress(data, type, returnTypedArray);
        return undefined;
    },
    random: function random(data) {
        if (nrdp._backchannel && nrdp._backchannel.random)
            return nrdp._backchannel.random(data);
        return undefined;
    },
    gctag: function gctag(name) {
        if (nrdp._backchannel && nrdp._backchannel.gctag)
            return nrdp._backchannel.gctag(name);
        return {};
    },
    getConfigList: function getConfigList() {
        this._callEventListeners(this, {type:'config', list: this._syncData.configList});
    },
    get config() { return this._syncData.config; },
    setConfigData: function setConfigData(file, data) {
        nrdp.config[file] = data;
        nrdp._invoke(null, "setConfigData", {name: file, data: data});
        this._callEventListeners(this, {type:'configChanged', name: file, data: data});
    },
    _path: "nrdp",
    addEventListener: function addEventListener(evt, listener) {
        if(evt == "suspendChanged")
            this.hasSuspendHandler = true;
        nrdp._addEventListener(this, evt, listener);
    },
    removeEventListener: function removeEventListener(evt, listener) {
        return nrdp._removeEventListener(this, evt, listener);
    },
    _findObject: function _findObject(name) {
        if (name instanceof Object)
            return name;
        var result = nrdp;
        if (name != "nrdp") {
            var bits = name.split('.');
            for(var i = 1; result && i < bits.length; ++i) { // skip beginning "nrdp"
                result = result[bits[i]];
            }
        }
        return result;
    },
    _setProperty: function _setProperty(subobj, prop, val) {
        if (!nrdp._backchannel) {
            nrdp.log.info("unhandled _setProperty " + subobj + " " + prop, "NRDP_SCRIPT");
        }
        var objName = subobj ? "nrdp." + subobj : "nrdp";
        var obj = nrdp._findObject(objName);
        if (!obj) {
            nrdp.log.error("could not find object " + objName + " to set property " + prop + " on", "NRDP_SCRIPT");
            return;
        }
        if (obj._setProperty_current && obj._setProperty_current[prop]) {
            if (!obj._setProperty_pending)
                obj._setProperty_pending = {};
            obj._setProperty_pending[prop] = {
                object: subobj,
                property: prop,
                value: val
            };
            return;
        }
        if(nrdp._backchannel.setProperty(subobj, prop, val)) {
            if (!obj._setProperty_current)
                obj._setProperty_current = {};
            obj._setProperty_current[prop] = true;
        }
    },
    _invoke: function _invoke(subobj, method, args) {
        if (!nrdp._backchannel) {
            nrdp.log.info("unhandled _invoke " + subobj + " " + method, "NRDP_SCRIPT");
        }
        return nrdp._backchannel.invoke(subobj, method, args);
    },
    _addEventListener: function _addEventListener(object, eventType, listener) {
        if (!listener)
            return false;
        var listeners, path;
        if (typeof object === "string") {
            if (!nrdp._oldStyleListeners)
                nrdp._oldStyleListeners = {};
            listeners = nrdp._oldStyleListeners;
            eventType = object + "." + eventType;
            path = eventType;
        } else {
            if (!object._eventListeners)
                object._eventListeners = {};
            listeners = object._eventListeners;
            path = object._path + "." + eventType;
        }
        ;
        if (!listeners[eventType])
            listeners[eventType] = [];
        listeners[eventType].push(listener);
        return true;
    },
    _removeEventListener: function _removeEventListener(object, eventType, listener) {
        if (!listener)
            return false;
        var myListeners, parent, path;
        if (typeof object === "string") {
            eventType = object + "." + eventType;
            parent = nrdp._oldStyleListeners;
            path = eventType;
        } else if (typeof object === "object") {
            parent = object._eventListeners;
            path = object._path + "." + eventType;
        }
        if (!parent)
        {
            ;
            return false;
        }
        myListeners = parent[eventType];
        if (!myListeners)
        {
            ;
            return false;
        }
        var index = myListeners.indexOf(listener);
        if (index < 0)
        {
            ;
            return false;
        }
        if (index >= 0) {
            if (myListeners.length == 1)
                delete parent[eventType];
            else
                myListeners.splice(index, 1);
        }
        var len = parent[eventType] ? parent[eventType].length : 0;
        ;
        return true;
    },
    _hasEventListener: function _hasEventListener(object, eventType) {
        if (typeof object === "string") {
            eventType = object + "." + eventType;
            return (nrdp._oldStyleListeners && nrdp._oldStyleListeners[eventType] && nrdp._oldStyleListeners[eventType].length);
        } else {
            return (object._eventListeners &&
                    object._eventListeners[eventType] &&
                    object._eventListeners[eventType].length);
        }
    },
    _callEventListeners: function _callEventListeners(object, event, remove) {
        var myListeners;
        var path;
        if (typeof object === "string") {
            var eventType = object + "." + event.type;
            if (!nrdp._oldStyleListeners || !nrdp._oldStyleListeners[eventType])
                return;
            if (remove) {
                myListeners = nrdp._oldStyleListeners[eventType];
                delete nrdp._oldStyleListeners[eventType];
            } else {
                myListeners = nrdp._oldStyleListeners[eventType].slice(0);
            }
            path = object;
        } else {
            if (!object._eventListeners || !object._eventListeners[event.type])
                return;
            if (remove) {
                myListeners = object._eventListeners[event.type];
                delete object._eventListeners[event.type];
            } else {
                myListeners = object._eventListeners[event.type].slice(0);
            }
            path = object._path;
        }
        for (var i = 0; i < myListeners.length; i++) {
            var listener = myListeners[i];
            if (listener) {
                ;
                listener(event);
            }
        }
    },
    _urlEncode: function _urlEncode(obj) {
        var str = "";
        for (var p in obj) {
            if (str) str += "&";
            str += p + "=" + encodeURIComponent(obj[p]);
        }
        return str;
    },
    describeEvent: function describeEvent(event) { return nrdp._describeEvent(event); },
    _describeEvent: function _describeEvent(event) {
        if(!event || !event.type)
            return undefined;
        var result;
        var path;
        if (event.object instanceof Object) {
            path = event.object._path;
        } else {
            path = event.object;
        }
        if(event.type == "Event") {
            result = path + "::" + event.name;
        } else if(event.type == "PropertyUpdate") {
            result = path + "::" + event.type + JSON.stringify(event.properties);
        } else if(event.type == "SetProperty") {
            result = path + "::" + event.type + "(" + event.property + ")";
        } else {
            result = path + "::" + event.type;
        }
        return result;
    },
    _gotEvent: function _gotEvent(event) {
        var start, evt;
        if (nrdp._syncData && nrdp.debug && nrdp._init) {
            start = nrdp.mono();
            if (start - event.time > 50) {
                var log = "JS event waited " + (start - event.time) + "ms: " + nrdp.describeEvent(event);
                if (event.timerPreemption)
                    log += " timerPreemption: " + event.timerPreemption;
                nrdp.log.debug(log);
            }
        }
        try {
            if (event.type == "Event") {
                this._gotEventEvent(event);
            } else if (event.type == "ClassSync") {
                this._gotClassSyncEvent(event);
            } else if (event.type == "PropertyUpdate") {
                this._gotPropertyUpdateEvent(event);
            } else if (event.type == "SetProperty") {
                this._gotSetPropertyEvent(event);
            } else if (event.type == "Method") {
                this._gotMethodEvent(event);
            } else if (event.type == "EventSourceError") {
                nrdp._isError = true;
                nrdp.log.info("EventSource went away, sending fatalerror", "NRDP_SCRIPT");
                evt = {
                    type: "fatalerror"
                };
                this._callEventListeners(this, evt);
            } else {
                nrdp.log.error("unhandled eventsource type " + event.type, "NRDP_SCRIPT");
            }
        } catch (e) {
            var tags = {},
                length;
            if (typeof e !== "string") {
                for (var n in e) {
                    if (e.hasOwnProperty(n)) {
                        tags[n] = e[n];
                    }
                }
            }
            // chrome defines this
            if (!tags.stack && e.stack) {
                tags.stack = e.stack;
            }
            if (typeof tags.stack == 'string') {
                length = tags.stack.length;
                if (length && length > nrdp._MAX_STACK_SIZE) {
                    tags.stack = tags.stack.substr(0, nrdp._MAX_STACK_SIZE/2) +
                        '\n...[stack truncated, it was ' + length + ' characters]...\n' +
                        tags.stack.substr(-nrdp._MAX_STACK_SIZE/2);
                }
            }
            nrdp.log.error("JAVASCRIPT EXCEPTION: " + e.toString(), "NRDP_SCRIPT", undefined, tags);
            evt = {
                type: "exception",
                exception: e
            };
            this._callEventListeners(this, evt);
            ;
        }
        if (start !== undefined) {
            var end = nrdp.mono();
            if (nrdp._syncData && (end - start > nrdp.options.js_delay)) {
                nrdp.log.warn("handling JS event took " + (end - start) + "ms: " + nrdp.describeEvent(event), "NRDP_SCRIPT");
            }
        }
    },
    _gotEventEvent: function _gotEventEvent(event) {
        if (!event.object) {
            if (event.name == "factoryReset") {
                var evt = {
                    type: event.name
                };
                this._callEventListeners(this, evt);
            }
        } else {
            var obj = this._findObject(event.object);
            if (!obj || !obj._handleEvent || !obj._handleEvent(event))
                nrdp.log.warn("unhandled event " + event.object + " " + event.name, "NRDP_SCRIPT");
        }
    },
    _gotClassSyncEvent: function _gotClassSyncEvent(event) {
        var classinfo = { methods: { byIndex: [], byName: {} }, properties: { byIndex: [], byName: {} } };
        for(var method in event.value.methods) {
            classinfo.methods.byIndex.push(event.value.methods[method]);
            classinfo.methods.byName[event.value.methods[method]] = parseInt(method);
        }
        for(var property in event.value.properties) {
            classinfo.properties.byIndex.push(event.value.properties[property]);
            classinfo.properties.byName[event.value.properties[property]] = parseInt(property);
        }
        this._classes[event.name] = classinfo;
    },
    _gotPropertyUpdateEvent: function _gotPropertyUpdateEvent(event) {
        var obj = this._findObject(event.object);
        if (!obj) {
            nrdp.log.error("Could not find object " + event.object + " for sync data", "NRDP_SCRIPT");
            return;
        }
        var fn;
        if (obj._updateProperty)
            fn = obj._updateProperty;
        else if (!obj._syncData)
            obj._syncData = {};
        for (var property in event.properties) {
            if (typeof property !== "string") {
                var propertyName = this._mapProperty(obj, property);
                if(!propertyName) {
                    nrdp.log.error("Could not map PropertyUpdate: " + event.object + " " + property, "NRDP_SCRIPT");
                    continue;
                }
                property = propertyName;
            }
            if (typeof event.properties[property] !== "function") {
                if (fn)
                    fn.call(obj, property, event.properties[property]);
                else
                    obj._syncData[property] = event.properties[property];
            }
        }
    },
    _gotSetPropertyEvent: function _gotSetPropertyEvent(event) {
        var obj = this._findObject(event.object);
        if (obj) {
            var property = event.property;
            if (typeof property !== "string") {
                var propertyName = this._mapProperty(obj, property);
                if(!propertyName) {
                    nrdp.log.error("Could not map SetProperty: " + event.object + " " + property, "NRDP_SCRIPT");
                    return;
                }
                property = propertyName;
            }
            if(obj._setProperty_current)
                delete obj._setProperty_current[property];
            if (obj._setProperty_pending && obj._setProperty_pending[property]) {
                var newset = obj._setProperty_pending[property];
                delete obj._setProperty_pending[property];
                nrdp._setProperty(newset.object, newset.property, newset.value);
            }
        }
    },
    _gotMethodEvent: function _gotMethodEvent(event) {
        // this only happens for an invalid argument error
        var method = event.method;
        if (typeof method !== "string") {
            var methodName = this._mapMethod(event.object, method);
            if(!methodName) {
                nrdp.log.error("Could not map Method: " + event.object + " " + method, "NRDP_SCRIPT");
                return;
            }
            method = methodName;
        }
        var evt = {
            type: "invalidargument",
            object: event.object,
            method: method,
            argument: event.returnValue
        };
        this._callEventListeners(this, evt);
    },
    _nextIdx: 1,
    _cbs: {},
    _fn: function _fn(name, args, cb) {
        if (!args) args = {};
        args.id = this._nextIdx++;
        this._cbs[args.id] = cb;
        nrdp._invoke(null, name, args);
    },
    parseXml: function parseXml(xml, cb) {
        if (cb) {
            nrdp._fn("parseXML", {xml: xml}, cb);
        } else if (nrdp._backchannel && nrdp._backchannel.parseXML) {
            return nrdp._backchannel.parseXML(xml);
        }
        return undefined;
    },
    parseJSON: function parseJSON(json, cb) {
        if (cb) {
            nrdp._fn("parseJSON", {json: json}, cb);
        } else if (nrdp._backchannel && nrdp._backchannel.parseJSON) {
            return nrdp._backchannel.parseJSON(json);
        }
        return undefined;
    },
    setTrustStore: function setTrustStore(trustStore) {
        nrdp._invoke(null, "setTrustStore", {trustStore: trustStore});
    },
    setServerTime: function setServerTime(time) {
        nrdp._invoke(null, "setServerTime", {time: time});
    },
    setTestDriverIpAddress: function setTestDriverIpAddress(address) {
        nrdp._invoke(null, "setTestDriverIpAddress", {address: address});
    },
    get suspended() { return this._syncData.suspended; },
    requestSuspend: function requestSuspend(reason) {
        nrdp._invoke(null, "requestSuspend", {reason: reason});
    },
    suspendComplete: function suspendComplete() {
        nrdp._fn("suspendComplete");
    },
    _fixXml: function _fixXml(obj) {
        var children = obj["$children"];
        var len, child;
        if (!children || !children.length)
            return;
        len = children.length;
        for (var i = 0; i < len; i++) {
            child = children[i];
            child["$parent"] = obj;
            child["$sibling"] = children[i + 1];
            var name = child["$name"];
            if (name !== undefined && obj[name] === undefined)
                obj[name] = child;
            this._fixXml(child);
        }
    },
    console: {
        _log: function(func, args) {
            var msg = "";
            var offset = 0;
            while(offset < args.length) {
                var tmp = args[offset++];
                if(offset != 1)
                    msg += " ";
                if(tmp && tmp instanceof Object) {
                    try {
                        if(tmp instanceof Array)
                            msg += JSON.stringify(tmp);
                        else
                            msg += "Object " + JSON.stringify(tmp);
                    } catch(e) {
                        msg += tmp;
                    }
                } else {
                    msg += tmp;
                }
            }
            func(msg);
        },
        error: function() { this._log(nrdp.log.error, arguments); },
        debug: function() { this._log(nrdp.log.debug, arguments); },
        log: function() { this._log(nrdp.log.debug, arguments); },
        warn: function() { this._log(nrdp.log.warn, arguments); },
        info: function() { this._log(nrdp.log.info, arguments); }
    },
    _setConfigurationOverrides: function (xml) {
        return nrdp._invoke(null, "setConfigurationOverrides", { xml: xml }).success;
    },
    // If these return something that is not undefined, that value is passed to the callback, if any.
    _eventBrokers:
    {
        'commandReceived':
            function(event) {
                this._callEventListeners(this, {type:'command', parameters: event.data});
            },
        'parsedXML':
            function(event) {
                if (event.data.success)
                    this._fixXml(event.data.object);
                return {data: event.data};
            },
        'suspendChanged':
            function(event) {
                this._callEventListeners(this, {type:'suspendChanged',
                    data: event.data,
                    time: event.time
                });
            },
        'requestedResume':
            function(event) {
                return {data: event.data};
            },
        'parsedJSON':
            function(event) {
                return {data: event.data.object};
            },
        'pong':
            function(event) {
                return {data: event.data.received};
            },
        'objects':
            function(event) {
                return {data: event.data.objects};
            },
        'locks':
           function(event) {
                return {data: event.data.locks};
            }
    },
    _dumpEvent: function() {
        for (var i=0; i<arguments.length; ++i) {
            var json;
            try {
                json = JSON.stringify(arguments[i], null, 4);
            } catch (err) {
                json = arguments[i];
            }
            nrdp.log.error(json);
        }
    },
    _dumpKeys: function() {
        for (var i=0; i<arguments.length; ++i) {
            nrdp.log.error("object: " + i);
            for (var key in arguments[i]) {
                nrdp.log.error(key);
            }
        }
    },
    _handleEvent: function _handleEvent(event) {
        var broker = this._eventBrokers[event.name],
            result,
            id,
            cb;
        if (!broker)
            return false;
        result = broker.call(this, event);
        if (result) {
            id = event.data.id;
            cb = this._cbs[id];
            try {
                if (typeof cb === 'function')
                    cb(result.data);
            } finally {
                delete this._cbs[id];
            }
        }
        return true;
    },
    _suspend: function _suspend(mode) {
        nrdp.gibbon._runConsole("/suspend " + (mode || ""));
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp */
nrdp.audio = {
    classname: "AudioBridge",
    get codecs() { return this._syncData.codecs; },
    get urls() { return this._syncData.urls; },
    get capability() { return this._syncData.capability; },
    load: function load(URL, cb) {
        nrdp.audio._fn("load", { URL: URL }, cb);
    },
    unload: function unload(URL, cb) {
        nrdp.audio._fn("unload", { URL: URL }, cb);
    },
    unloadAll: function unloadAll(cb) {
        nrdp.audio._fn("unloadAll", undefined, cb);
    },
    play: function play(URL, volume, cb, fadeInMs, fadeOutMs, fadeInEase, fadeOutEase) {
        nrdp.audio._fn("play", { URL: URL, volume: volume, fadeInMs: fadeInMs || 0, fadeOutMs: fadeOutMs || 0, fadeInEase: fadeInEase || 0, fadeOutEase: fadeOutEase || 0 }, cb);
    },
    stop: function stop(URL, cb, fadeOutMs, fadeOutEase) {
        nrdp.audio._fn("stop", { URL: URL, fadeOutMs: fadeOutMs || 0, fadeOutEase: fadeOutEase || 0 }, cb);
    },
    getVolume: function getVolume(cb) {
        nrdp.audio._fn("getVolume", null, cb);
    },
    setVolume: function setVolume(volume, transitionMs, ease) {
        nrdp.audio._fn("setVolume", { volume: volume, transitionMs: transitionMs || 0, ease: ease || 0 });
    },
    get EASE_LINEAR() { return 0; },
    get EASE_IN_CUBIC() { return 1; },
    get EASE_OUT_CUBIC() { return 2; },
    _nextIdx: 1,
    _cbs: {},
    _fn: function _fn(name, args, cb) {
        if (!args) args = {};
        args.id = this._nextIdx++;
        if (cb)
            this._cbs[args.id] = cb;
        nrdp._invoke("audio", name, args);
    },
    _handleEvent: function _handleEvent(event) {
        if (event.data && event.data.id) {
            if (typeof this._cbs[event.data.id] == "function") {
                if (event.data.hasOwnProperty("success")) {
                    var args = [event.data.success, event.data.size];
                    if (event.data.hasOwnProperty("URL"))
                        args.unshift(event.data.URL);
                    this._cbs[event.data.id].apply(undefined, args);
                } else {
                    this._cbs[event.data.id](event.data.data);
                }
                delete this._cbs[event.data.id];
            }
        } else {
            return false;
        }
        return true;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp */
nrdp.device = {
    classname: "DeviceBridge",
    _path: "device",
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get VOLUME_NONE() { return 0; },
    get VOLUME_SPEAKER() { return 1; },
    get VOLUME_STREAM() { return 2; },
    // Drm types
    get PLAY_READY() { return 0; },
    get SESSION_DH1() { return 1; },
    get WIDEVINE_CEF() { return 2; },
    get drmType() { return this._syncData.capability.drmType; },
    get currentViewMode() { return this._syncData.currentViewMode; },
    get availableViewModes() { return this._syncData.availableViewModes; },
    get softwareVersion() { return this._syncData.softwareVersion; },
    get certificationVersion() { return this._syncData.certificationVersion; },
    get deviceModel() { return this._syncData.deviceModel; },
    get ESNPrefix() { return this._syncData.ESNPrefix; },
    get SDKVersion() { return this._syncData.SDKVersion; },
    get ESN() { return this._syncData.ESN; },
    get language() { return this._syncData.language; },
    get friendlyName() { return this._syncData.friendlyName; },
    get startupTags() { return this._syncData.startupTags; },
    get volumeControlType() { return this._syncData.volumeControlType; },
    get volumeStep() { return this._syncData.volumeStep; },
    get mute() { return this._syncData.mute; },
    set mute(m) { nrdp._setProperty("device", "mute", m); },
    get capability() { return this._syncData.capability; },
    get videoOutput() { return this._syncData.videoOutput; },
    get supportedVideoOutput() { return this._syncData.supportedVideoOutput; },
    get activeVideoOutput() { return this._syncData.activeVideoOutput; },
    get dnslist() { return this._syncData.dnslist; },
    get iflist() { return this._syncData.iflist; },
    get ipversions() { return this._syncData.ipversions; },
    get UIVersion() { return this._syncData.UIVersion; },
    set UIVersion(version) { nrdp._setProperty("device", "UIVersion", version); },
    addLibrary: function addLibrary(name, version, component) {
        nrdp._invoke("device", "addLibrary", { name: name, version: version, component: component });
    },
    setUIVersion: function setUIVersion(version) { nrdp._setProperty("device", "UIVersion", version); },
    get UILanguages() {
        return this._syncData.UILanguages;
    },
    set UILanguages(langs) {
        langs = Array.isArray(langs) ? langs : [];
        var val = langs.join(",");
        nrdp._setProperty("device", "UILanguages", val);
    },
    isScreensaverOn: function isScreensaverOn() { return this._syncData.screensaverOn; },
    factoryReset: function factoryReset(cb) {
        nrdp.storage._clearAll();
        nrdp.device._fn("factoryReset", null, cb);
    },
    setViewMode: function setViewMode(viewMode) {
        nrdp._invoke("device", "setViewMode", {viewMode : viewMode});
    },
    getVolume: function getVolume(cb) {
        nrdp.device._fn("getVolume", null, cb);
    },
    setVolume : function(volume) {
        nrdp._invoke("device", "setVolume",
                      {"targetVolume": volume});
    },
    setRegistered: function setRegistered(value) {
        nrdp._setProperty("device", "registered", !!value);
    },
    get registered() { return this._syncData.registered; },
    getSignatures: function getSignatures(cb) {
        nrdp.device._fn("getSignatures", null, cb);
    },
    getUptime: function getUptime(cb) {
        nrdp.device._fn("getUptime", null, cb);
    },
    getSystemValue: function getSystemValue(key, cb) {
        nrdp.device._fn("getSystemValue", {key : key}, cb);
    },
    getDisplaySize: function getDisplaySize(cb) {
        nrdp.device._fn("getDisplaySize", null, cb);
    },
    /*
     * dual video
     */
    // codec profiles
    get CODEC_NOT_AVAILABLE() { return 0;},
    get AVC_MPL30() {return 1;},
    get AVC_MPL31() {return 2;},
    get AVC_MPL40() {return 3;},
    get AVC_SHPL30() {return 4;},
    get AVC_SHPL31() {return 5;},
    get AVC_SHPL40() {return 6;},
    get HEVC_MAIN10_L20() {return 7;},
    get HEVC_MAIN10_L21() {return 8;},
    get HEVC_MAIN10_L30() {return 9;},
    get HEVC_MAIN10_L31() {return 10;},
    get HEVC_MAIN10_L40() {return 11;},
    get HEVC_MAIN10_L41() {return 12;},
    get HEVC_MAIN10_L50() {return 13;},
    get HEVC_MAIN10_L51() {return 14;},
    // zorder
    get ZORDER_NOT_AVAILABLE() {return 0;},
    get ZORDER_ANY() {return 1;},
    get ZORDER_TOP_ONLY() {return 2;},
    get ZORDER_BOTTOM_ONLY() {return 3;},
    get maxVideoPipelines(){
        return this._syncData.maxVideoPipelines;
    },
    get getVideoPipelineCapabilities(){
        return this._syncData.videoPipelineCapabilities;
    },
    getRemainingVideoPipelineCapability:
    function getRemainingVideoPipelineCapability(videoPipelineCapabilites, cb) {
        nrdp.device._fn("getRemainingVideoPipelineCapability",
                        {videoPipelineCapabilites:videoPipelineCapabilites},
                        cb);
    },
    disableDolbyVisionELComposing: function disableDolbyVisionELComposing(disable, cb) {
        nrdp.device._fn("disableDolbyVisionELComposing", {disable:disable}, cb);
    },
    _nextIdx: 1,
    _cbs: {},
    _fn: function _fn(name, args, cb) {
        if (!args) args = {};
        args.idx = this._nextIdx++;
        if (cb)
            this._cbs[args.idx] = cb;
        nrdp._invoke("device", name, args);
    },
    _handleEvent: function _handleEvent(event) {
        if (event.data && event.data.idx) {
            if (typeof this._cbs[event.data.idx] == "function") {
                this._cbs[event.data.idx](event.data.data);
                delete this._cbs[event.data.idx];
            }
        } else {
            return false;
        }
        return true;
    },
    _syncData: {},
    _updateProperty: function _updateProperty(property, value) {
        var evt;
        if (nrdp.isReady) {
            if (property == "screensaverOn") {
                evt = {
                    type: "screensaverchange"
                };
            } else if (property == "capability") {
                evt = {
                    type: "capabilitychange",
                    old: this.capability
                };
            } else if (property == "videoOutput") {
                evt = {
                    type: "videooutputchange",
                    old: this.videoOutput
                };
            } else if (property == "language") {
                evt = {
                    type: "languagechange",
                    old: this.language
                };
            } else if (property == "currentViewMode") {
                evt = {
                    type: "viewmodechange"
                };
            } else if (property == "iflist" ) {
                evt = {
                    type: "networkchange"
                };
            } else if (property.match(/^volume/) || property == "mute") {
                evt = {
                    type: "volumechange",
                    oldvolume: this.volume,
                    oldmute: this.mute
                };
            }
        }
        this._syncData[property] = value;
        if (evt) {
            nrdp._callEventListeners(this, evt);
        }
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp */
nrdp.drmsystem = {
    // properties
    _path : "drmsystem",
    get INVALID_LICENSE() { return 0;},
    get LIMITED_DURATION_LICENSE() { return 1;},
    get STANDARD_LICENSE() { return 2;},
    get drmType() { return this._syncData.drmType; },
    get supportSecureStop() { return this._syncData.supportSecureStop; },
    get supportStorageDeletion() { return this._syncData.supportStorageDeletion; },
    get isOpen() { return this._syncData.isOpen; },
    /*
     * drm session state - defined at IDrmSession.h
     * enum SessionState {
     *     LicenseAcquisitionState = 0,
     *     InactiveDecryptionState,
     *     ActiveDecryptionState,
     *     InvalidState }
     */
    get DRMSESSION_LICENSE_ACQUISITION_STATE() { return 0;},
    get DRMSESSION_INACTIVE_DECRYPTION_STATE() { return 1;},
    get DRMSESSION_ACTIVE_DECRYPTION_STATE() {return 2;},
    get DRMSESSION_INVALID_STATE() { return 3;},
    /*
     * drm system open/close
     */
    openDrmSystem: function(drmType, cb)
    {
        nrdp.drmsystem._fn("openDrmSystem", {drmType:drmType}, cb);
    },
    closeDrmSystem: function(drmType, cb)
    {
        nrdp.drmsystem._fn("closeDrmSystem", {drmType:drmType}, cb);
    },
    /*
     * secure stop JS APIs
     */
    getSecureStopIds: function(cb)
    {
        nrdp.drmsystem._fn("getSecureStopIds", {}, cb);
    },
    getSecureStop: function(secureStopId, cb)
    {
        var args = {'secureStopId': secureStopId};
        nrdp.drmsystem._fn("getSecureStop", args, cb);
    },
    commitSecureStop: function(secureStopId, serverResponse)
    {
        var args = {'secureStopId': secureStopId, 'serverResponse': serverResponse};
        nrdp.drmsystem._fn("commitSecureStop", args);
    },
    resetSecureStops: function()
    {
        nrdp.drmsystem._fn("resetSecureStops");
    },
    useSecureStop: function(enable)
    {
        var args = {'enable': enable};
        nrdp.drmsystem._fn("enableSecureStop", args);
    },
    isSecureStopEnabled: function isSecureStopEnabled()
    {
        return nrdp._invoke("drmsystem", "isSecureStopEnabled");
    },
    /*
     * delete drm license store / key store JS APIs
     */
    deleteDrmStore: function(cb)
    {
      nrdp.drmsystem._fn("deleteDrmStore", {}, cb);
    },
    deleteKeyStore: function(cb)
    {
      nrdp.drmsystem._fn("deleteKeyStore", {}, cb);
    },
    getDrmStoreHash: function(cb)
    {
        nrdp.drmsystem._fn("getDrmStoreHash", {}, cb);
    },
    getKeyStoreHash: function(cb)
    {
        nrdp.drmsystem._fn("getKeyStoreHash", {}, cb);
    },
    getDrmTime: function getDrmTime()
    {
        return nrdp._invoke("drmsystem", "getDrmTime");
    },
    /*
     * License handling JS APIs
     */
    generateChallenge: function(drmType, contentId, licenseType, drmHeader, cb)
    {
        var args = {
            'drmType':drmType,
            'contentId':contentId,
            'licenseType':licenseType,
            'drmHeader':drmHeader
        };
        nrdp.drmsystem._fn("generateChallenge", args, cb);
    },
    provideLicense: function(sessionId, license, cb)
    {
        var args = {
            'sessionId':sessionId,
            'license':license
        };
        nrdp.drmsystem._fn("provideLicense", args, cb);
    },
    getDrmSessionIds: function(cb)
    {
        nrdp.drmsystem._fn("getDrmSessionIds", {}, cb);
    },
    deleteDrmSession: function(sessionId, cb)
    {
        var args = {
            'sessionId':sessionId
        };
        nrdp.drmsystem._fn("deleteDrmSession", args, cb);
    },
    flushDrmSessions: function(cb)
    {
        nrdp.drmsystem._fn("flushDrmSessions", {}, cb);
    },
    getLdlSessionsLimit: function(cb)
    {
        nrdp.drmsystem._fn("getLdlSessionsLimit", {}, cb);
    },
    _fn: function(name, params, cb)
    {
        if (!params) params = {};
        params.idx = this._nextIdx++;
        if (cb)
            this._cbs[params.idx] = cb;
        nrdp._invoke("drmsystem", name, params);
        return params.idx;
    },
    _handleEvent: function(event)
    {
        var cb, idx;
        if (event.name != "result"){
            return false;
        }
        idx = event.data.idx;
        if (typeof this._cbs[idx] == "function")
        {
            cb = this._cbs[idx];
            delete event.data.idx;
            cb(event.data);
        }
        delete this._cbs[idx];
        return true;
    },
    _nextIdx: 1,
    _cbs: {}
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp */
nrdp.instrumentation = {
    classname: "InstrumentationBridge",
    _path: "instrumentation",
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    setParams: function setParams(enabled, events) {
        nrdp._invoke("instrumentation", "setParams", { enabled:enabled, events: events });
    },
    get ON() { return 0; },
    get SWITCHED() { return 1; },
    get TEST() { return 2; },
    get DEBUG() { return 3; },
    get WATCHDOG() { return 4; },
    get verbose() { return this._syncData.verbose; },
    generateEvent: function generateEvent(category, name, value) {
        nrdp._invoke("instrumentation", "event", { category:category, name: name, value:value });
    },
    startInterval: function startInterval(category, name, value, reset){
        nrdp._invoke("instrumentation", "intervalStart", { category: category, name: name, value: value, reset: reset });
    },
    incIntervalCounter: function incIntervalCounter(category, name, counter, increment) {
        nrdp._invoke("instrumentation", "intervalCount",
                     { category: category, name: name, counter: counter, increment: increment});
    },
    tagInterval: function tagInterval(category, name, value) {
        nrdp._invoke("instrumentation", "intervalTag", { category: category, name: name, value: value });
    },
    endInterval: function endInterval(category, name, value) {
        nrdp._invoke("instrumentation", "intervalEnd", { category: category, name: name, value: value});
    },
    cancelInterval: function cancelInterval(category, name) {
        nrdp._invoke("instrumentation","intervalCancel", { category: category, name: name });
    },
    stashOn: function stashOn() {
        nrdp._invoke("instrumentation", "stash", { on: true });
    },
    stashOff: function stashOff() {
        nrdp._invoke("instrumentation", "stash", { on: false });
    },
    popStash: function popStash(cb)
    {
        var idx = this._nextIdx++;
        this._cbs[idx] = cb;
        nrdp._invoke("instrumentation", "popStash", { idx: idx });
    },
    _flush: function _flush(cb) {
        var idx = this._nextIdx++;
        this._cbs[idx] = cb;
        nrdp._invoke("instrumentation", "flush", { idx: idx });
    },
    _handleEvent: function _handleEvent(event) {
        if (event.data && event.data.idx) {
            if (this._cbs[event.data.idx] instanceof Function) {
                this._cbs[event.data.idx](event.data.events);
            }
            delete this._cbs[event.data.idx];
        } else if (event.name == "verboseChanged") {
            nrdp._callEventListeners(this, { type: event.name });
        } else {
            return false;
        }
        return true;
    },
    _nextIdx: 1,
    _cbs: {}
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp, nrdp_platform */
nrdp.log = {
    classname: "LogBridge",
    _path: "log",
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get UIContext() { return this._syncData.UIContext; },
    set UIContext(s) { nrdp._setProperty("log","UIContext",s); },
    get appid() { return this._syncData.appid; },
    get areas() { return this._syncData.areas; },
    get aseCodes() { return this._syncData.aseCodes; },
    get errorCodes() { return this._syncData.errorCodes; },
    get haveTracing() { return this._syncData.haveTracing; },
    get level() { return this._syncData.level; },
    set level(e) { nrdp._setProperty("log", "level", e); },
    get levels() { return this._syncData.levels; },
    get sessionid() { return this._syncData.sessionid; },
    get traceAreas() { return this._syncData.traceAreas; },
    set traceAreas(e) { nrdp._setProperty("log", "traceAreas", e); },
    //get xid() { return this._syncData.xid; },
    flush: function flush() {
        nrdp._invoke("log", "flush");
    },
    _traceAreaMap: null,
    _log: function _log(level, args) {
        var msg = args[0],
            area = args[1],
            type = args[2],
            tags = args[3],
            critical = args[4],
            sendtoAppboot = args[5]?args[5]:false,
            logLevel;
        if (!nrdp._isError && this._syncData.levels) {
            if (level == "trace" && !nrdp.debug && !this.haveTracing)
                return;
            logLevel = this.levels[level];
            // compat with cadmium, could get an exception object passed in here
            if (typeof args[1] === "object" && args[1] !== null && /^\w*Error$/.test(args[1].name)) {
                area = undefined;
                if (!tags)
                    tags = {};
                tags["exception"] = args[1].message || ("" + args[1]);
                tags["stack"] = args[1].stack;
            }
            nrdp._invoke("log", "log", {
                logLevel: logLevel,
                msg: msg,
                traceArea: area,
                type: type,
                tags: tags,
                critical: critical,
                sendtoappboot: sendtoAppboot
            });
            if (!critical && ( type || (nrdp.log.level > nrdp.log.levels.trace && logLevel >= nrdp.log.level ))) {
                nrdp._callEventListeners(this, {
                    'type': 'logMsgsReady',
                    'data': [{
                        'monotime': nrdp.mono(),
                        'area': area,
                        'level': logLevel,
                        'msg': msg,
                        'tags': tags,
                        'logtype': type,
                        'threadname': "UI_THREAD",
                        'threadid': "1",
                        'critical': false
                    }]
                });
            }
        } else {
            if (args.length != 1)
                msg = args[1] + ": " + args[0];
            this.console(msg);
        }
    },
    milestone: function milestone(msg) { nrdp._invoke("log", "milestone", { msg: msg } ); },
    console: function console(msg) {
        if (nrdp._backchannel && nrdp._backchannel.console)
            nrdp._backchannel.console(msg);
        else if (typeof nrdp_platform !== "undefined" && nrdp_platform.console) //nrdp
            nrdp_platform.console(msg);
        else if (typeof window !== "undefined" && window.console.log) //browser
            window.console.log(msg);
    },
    debug: function debug() { nrdp.log._log("debug", arguments); },
    info: function info() { nrdp.log._log("info", arguments); },
    warn: function warn() { nrdp.log._log("warn", arguments); },
    error: function error() { nrdp.log._log("error", arguments); },
    fatal: function fatal() { nrdp.log._log("fatal", arguments); },
    trace: function trace() { nrdp.log._log("trace", arguments); },
    resetAppID: function resetAppID(newId) {
        nrdp.log.error("Resetting Appid:" + nrdp.log.appid );
        nrdp._invoke("log", "resetAppID", {appid: newId});
    },
    createArea: function createArea(area, groups) {
        nrdp._invoke("log", "createArea", {area: area, groups: groups});
        if (this._traceAreaMap) {
            this._traceAreaMap[area] = area;
        }
    },
    resetSessionID: function resetSessionID() { return nrdp._invoke("log", "resetSessionID"); },
    getLogMessages: function getLogMessages() { nrdp._invoke("log", "getLogMessages"); },
    getCriticalMessages: function getCriticalMessages() { nrdp._invoke("log", "getCriticalMessages"); },
    deleteCriticalMessages: function deleteCriticalMessages(criticalMessages) { nrdp._invoke("log", "deleteCriticalMessages", {criticalMessages: criticalMessages}); },
    _syncData: { },
    _updateProperty: function _updateProperty(property, value) {
        var evt;
        if (nrdp.isReady) {
            if (property == "sessionid") {
                evt = {
                    type: "sessionIDChanged",
                    data: value
                };
            }
        }
        this._syncData[property] = value;
        if (evt) {
            nrdp._callEventListeners(this, evt);
        }
    },
    _handleEvent: function _handleEvent(event) {
        var evt;
        if ( event.name == "logMsgsReady") {
            evt = { data : event.data };
            evt.type = event.name;
            nrdp._callEventListeners(this, evt);
        } else if ( event.name == "criticalMsgsReady") {
            evt = { data : event.data };
            evt.type = event.name;
            nrdp._callEventListeners(this, evt);
        } else if (event.name == "logflush") {
            evt = { name: "logflush", type: "logflush"};
            nrdp._callEventListeners(this, evt);
        } else if (event.name == "appIdResetComplete") {
            evt = { type: "appIdResetComplete", data : event.data};
            nrdp._callEventListeners(this, evt);
        } else {
            return false;
        }
        return true;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp */
nrdp.mediarequest = {
    _path: "mediarequest",
    addEventListener: function addEventListener(evt, listener)
    {
        nrdp._addEventListener(this, evt, listener);
    },
    removeEventListener: function removeEventListener(evt, listener)
    {
        nrdp._removeEventListener(this, evt, listener);
    },
    // MediaType constants: matches AseCommonDataType.h
    get MEDIA_AUDIO() { return 0; },
    get MEDIA_VIDEO() { return 1; },
    get MEDIA_TEXT() { return 2; },
    get MEDIA_HEADERS() { return 3; },
    // ResponseTypes for how a MediaRequest returns its data: matches MediaRequestBridge.h
    get RESPONSE_DATABUFFER() { return 0; },
    get RESPONSE_STREAM() { return 1; },
    // mediaSource
    // @return "sourceid" id of created mediaSource
    createMediaSource: function(cb, player)
    {
        return this._fn("mediaSourceCreate", {"player": player}, cb);
    },
    destroyMediaSource: function(sourceid, cb)
    {
        return this._fn("mediaSourceDestroy", {"sourceid": sourceid}, cb);
    },
    // mediaBufferPool
    createMediaBufferPool: function(cb)
    {
        return this._fnAsync("mediaBufferPoolCreate", {}, cb);
    },
    // sourceBuffer
    // @param[in] sourceid, the mediaSourceId to create buffers for
    // @param[in] mediatypes, array of mediatypes to create
    // @return "bufferids" array of bufferid's created
    createSourceBuffers: function(sourceid, mediatypes, cb)
    {
        return this._fn("sourceBuffersCreate",
                        {
                            "sourceid": sourceid,
                            "mediatypes": mediatypes
                        }, cb);
    },
    // NOT NEEDED
    attachDrmHeaderToSourceBuffer: function(){},
    // DEPRECATED
    attachHeaderToSourceBuffer: function(header, streamId, bufferId, contentId,
                                         contentProfile, bitrate, frameRateValue,
                                         frameRateScale, cb, sourceId)
    {
        return this.attachDataToSourceBuffer( header, sourceId, bufferId, cb );
    },
    attachDataToSourceBuffer: function( data, sourceId, bufferId, cb )
    {
        return this._fn("sourceBufferAttachData",
            {
                "sourceid": sourceId,
                "bufferid": bufferId,
                "data": data
            }, cb);
    },
    attachRequestToSourceBuffer: function(requestid, bufferid, cb, sourceId)
    {
        return this._fn("sourceBufferAttachRequest",
            {
                "sourceid": sourceId,
                "bufferid": bufferid,
                "requestid": requestid
            }, cb);
    },
    sourceBufferSetTimestampOffset: function(sourceId, bufferId, timeOffset, timescale, cb)
    {
      return this._fn("sourceBufferSetTimestampOffset",
                      {
                          "sourceid": sourceId,
                          "bufferid": bufferId,
                          "timeoffset": timeOffset,
                          "timescale": timescale
                      }, cb );
    },
    // downloadTrack's
    // @param[in] mediatypes, array of mediatypes to add
    // @return "trackids" array of trackid's added
    addDownloadTracks: function(configs, cb)
    {
        return this._fnAsync("downloadTracksCreate", {"configs": configs}, cb);
    },
    // @param[in] trackid, the trackid to pause
    pauseDownloadTrack: function(trackid, cb)
    {
        return this._fnAsync("downloadTrackPause", {"trackid": trackid}, cb);
    },
    // @param[in] trackid, the trackid to resume
    resumeDownloadTrack: function(trackid, cb)
    {
        return this._fnAsync("downloadTrackResume", {"trackid": trackid}, cb);
    },
    // @param[in] trackid, the trackid to reconfigure
    // @param[in] config, the config object
    reconfigureDownloadTrack: function(trackid, config, cb)
    {
        return this._fnAsync("downloadTrackReconfigure", {"trackid": trackid, "config": config}, cb);
    },
    // @param[in] trackid, the trackid to remove
    removeDownloadTracks: function(trackids, cb)
    {
        return this._fnAsync("downloadTracksDestroy", {"trackids": trackids}, cb);
    },
    // MediaRequest's
    createRequest: function(requestId, trackId, url, byteStart, byteEnd, responseType,
                            ptsStart, ptsEnd, cb)
    {
        return this._fn("requestCreate",
            {
                "requestid": requestId,
                "trackid" : trackId,
                "url" : url,
                "start" : byteStart,
                "end" : byteEnd,
                "responsetype" : responseType,
                "ptsStart" : ptsStart,
                "ptsEnd" : ptsEnd
            }, cb);
    },
    abortRequest: function(requestid, cb)
    {
        return this._fn("requestAbort", {"requestid" : requestid}, cb);
    },
    removeRequest: function(requestid, cb)
    {
        return this._fn("requestRemove", {"requestid": requestid}, cb);
    },
    swapRequestURL: function(requestid, url, cb)
    {
        return this._fn("requestSwapURL",
                        {
                            "requestid" : requestid,
                            "url" : url
                        },
                        cb);
    },
/*
    getRequestData: function(requestid, cb)
    {
        if ( this._requestData[ requestid ] ) {
            cb( { "success" : true, "data" : this._requestData[ requestid ] } );
            delete this._requestData[ requestid ];
        } else {
            cb( { "success" : false, "error" : -3 } );
        }

        return 0;
    },
*/
    endOfStream: function(bufferId, cb, sourceId)
    {
        return this._fn("endOfStream",
                        {
                            "sourceid" : sourceId,
                            "bufferid" : bufferId
                        },
                        cb);
    },
    parseSegmentIndex: function( data, anchor )
    {
        return this._fn("parseSegmentIndex", { "data" : data, "anchor" : anchor } );
    },
    // private
    _nextIdx: 1,
    _cbs: {},
    // XXX Temporarily, we will keep the request data here when it is returned in oncomplete
    //     events for backwards compatibility with JS-ASE in the 9/23 push
    //     This can be removed after 10/7, where this is handled properly in NRDJS
    //_requestData: { },
    _fn: function(name, params, cb)
    {
        if (!params) params = {};
        var ret = nrdp._invoke("mediarequest", name, params);
        if ( cb ) {
            cb( ret );
            return undefined;
        }
        return ret;
    },
    _fnAsync: function(name, params, cb)
    {
        if (!params) params = {};
        params.idx = this._nextIdx++;
        this._cbs[params.idx] = cb;
        nrdp._invoke("mediarequest", name, params);
        return params.idx;
    },
    _handleEvent: function(event)
    {
        var cb, idx;
        // handle invoke callbacks
        if (event.name == "result")
        {
            idx = event.data.idx;
            if (typeof this._cbs[idx] == 'function')
            {
                cb = this._cbs[idx];
                delete event.data.idx;
                cb(event.data);
            }
            delete this._cbs[idx];
            return true;
        }
        //if (event.name == "oncomplete") {
        //    this._requestData[ event.data["requestId"] ] = event.data["data"];
        //}
        // handle progress events
        if ((event.name == "onloadstart") ||
            (event.name == "onfirstbyte") ||
            (event.name == "onprogress") ||
            (event.name == "oncomplete") ||
            (event.name == "onerror") ||
            (event.name == "downloadpaused") ||
            (event.name == "downloadresumed") ||
            (event.name == "pipelinedetection") ||
            (event.name == "networkfailing") ||
            (event.name == "TransportReporter" ))
        {
            nrdp._callEventListeners(this, event.data);
            return true;
        }
        // unhandled
        return false;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp */
function MediaSourcePlayer(fullpath, shortpath)
{
    this._path = fullpath;
    this._shortpath = shortpath;
};
MediaSourcePlayer.constructor = MediaSourcePlayer;
Object.defineProperties(MediaSourcePlayer.prototype, {
    // native object
    classname: { value: "MediaSourcePlayerBridge" },
    path: { get: function() { return this._shortpath; } },
    // IAdaptiveStreamingPlayer state
    OPENING: { value: 0 },
    PLAYING: { value: 1 },
    PAUSED: { value: 2 },
    STOPPED: { value: 3 },
    CLOSED: { value: 4 },
    // MediaType
    MEDIA_UNKNOWN: { value: -1 },
    MEDIA_AUDIO: { value: 0 },
    MEDIA_VIDEO: { value: 1 },
    MEDIA_TEXT: { value: 2 },
    // synced properties
    bufferPoolSize: { get: function() { return this._syncData.bufferPoolSize; } },
    state: { get: function() { return this._syncData.state; } }
});
//////////
// Events
//////////
MediaSourcePlayer.prototype.addEventListener = function addEventListener(evt, listener)
{
    nrdp._addEventListener(this, evt, listener);
};
MediaSourcePlayer.prototype.removeEventListener = function removeEventListener(
    evt, listener)
{
    nrdp._removeEventListener(this, evt, listener);
};
MediaSourcePlayer.prototype._handleEvent = function _handleEvent(event)
{
    for (var prop in event.data._propups)
    {
        this._syncData[prop] = event.data._propups[prop];
    }
    if ((event.name == "MediaSourcePlayer") || (event.name == "PlaybackReporter"))
    {
        event.data.time = event.time;
        nrdp._callEventListeners(this, event.data);
        return true;
    }
    else
    {
        return false;
    }
};
//////////
// player control
//////////
MediaSourcePlayer.prototype.open = function open(args)
{
    nrdp._invoke(this._path, "open", args);
};
MediaSourcePlayer.prototype.play = function play(pts)
{
    var args = {};
    if (pts != undefined && pts != null)
        args.pts = pts;
    nrdp._invoke(this._path, "play", args);
};
MediaSourcePlayer.prototype.stop = function stop()
{
    nrdp._invoke(this._path, "stop");
};
MediaSourcePlayer.prototype.pause = function pause()
{
    nrdp._invoke(this._path, "pause");
};
MediaSourcePlayer.prototype.unpause = function unpause()
{
    nrdp._invoke(this._path, "unpause");
};
MediaSourcePlayer.prototype.skip = function skip(ms)
{
    nrdp._invoke(this._path, "skip", {pts:ms});
};
MediaSourcePlayer.prototype.swim = function swim(pts, absolute, fuzz, allowRebuffer)
{
    var args = { pts: pts };
    if (typeof absolute == "boolean")
        args.absolute = absolute;
    if (typeof fuzz == "number" && !isNaN(fuzz))
        args.fuzz = fuzz;
    if (typeof allowRebuffer == "boolean")
        args.allowRebuffer = allowRebuffer;
    nrdp._invoke(this._path, "swim", args);
};
MediaSourcePlayer.prototype.close = function close()
{
    nrdp._invoke(this._path, "close");
};
//////////
// display control
//////////
MediaSourcePlayer.prototype.setVideoWindow = function setVideoWindow(
    x, y, width, height, transitionDuration, zOrder)
{
    var args = {
        x: x,
        y: y,
        width: width,
        height: height,
        transitionDuration: transitionDuration,
        zOrder : zOrder
    };
    nrdp._invoke(this._path, "setVideoWindow", args);
};
MediaSourcePlayer.prototype.bringVideoToFront = function bringVideoToFront()
{
    // no operation
};
MediaSourcePlayer.prototype.sendVideoToBack = function sendVideoToBack()
{
    // no operation
};
//////////
// audio control
//////////
MediaSourcePlayer.prototype.disableAudio = function()
{
    // disable audio so that audio track switching can happen
    nrdp._invoke(this._path, "disableAudio");
};
MediaSourcePlayer.prototype.enableAudio = function()
{
    // audio data for new track is buffered enough, and notify player to enable audio
    nrdp._invoke(this._path, "enableAudio");
};
MediaSourcePlayer.prototype.getVolume = function(cb)
{
    // getVolume is returned in an event type "volume"
    nrdp._invoke(this._path, "getVolume");
};
MediaSourcePlayer.prototype.setVolume = function(volume, transition, ease)
{
    nrdp._invoke(this._path, "setVolume", {
        "targetVolume": volume,
        "transitionDuration": transition,
        "ease": ease
    });
};
//////////
// player stats
//////////
MediaSourcePlayer.prototype.getBufferRange = function getBufferRange(cb)
{
    // returns in event type "bufferrange"
    nrdp._invoke(this._path, "getBufferRange");
};
MediaSourcePlayer.prototype.obtainPlaybackStat = function obtainPlaybackStat()
{
    nrdp._invoke(this._path, "obtainPlaybackStat");
};
//////////
// factory
//////////
nrdp.mediasourceplayerfactory = {
    classname: "MediaSourcePlayerFactoryBridge",
    _path: "mediasourceplayerfactory",
    createPlayer: function createPlayer(index, cb)
    {
        var name = "mediasourceplayer" + index;
        // check to see if a player already exists here
        if (nrdp[this._path][name])
        {
            // if so return that one
            var result = {
                "success": true,
                "path": name,
                "player": nrdp[this._path][name]
            };
            cb(result);
            return nrdp[this._path][name];
        }
        // create the JS player bridge
        var fullpath = this._path + "." + name;
        var player = new MediaSourcePlayer(fullpath, name);
        // store the player into nrdp
        nrdp[this._path][name] = player;
        // create the native player bridge
        var args = {
            "name": name
        };
        this._fn(
            "createPlayer", args,
            function(result)
            {
                if (result["success"])
                {
                    // add player to the result
                    result["path"] = name;
                    result["player"] = player;
                }
                // continue to original cb
                if (typeof cb == 'function')
                    cb(result);
            });
        // return the JS player
        return player;
    },
    destroyPlayer: function destroyPlayer(index, cb)
    {
        var name = "mediasourceplayer" + index;
        if (nrdp[this._path][name])
        {
            // remove the js object
            delete nrdp[this._path][name];
            // destroy the native object
            return this._fn("destroyPlayer", {"name": name}, cb);
        }
        return undefined;
    },
    // private
    _nextIdx: 0,
    _cbs: {},
    _fn: function(name, params, cb)
    {
        // add index to params
        if (!params) params = {};
        params.idx = this._nextIdx++;
        // store cb away
        this._cbs[params.idx] = cb;
        // invoke native method
        nrdp._invoke(this._path, name, params);
        return params.idx;
    },
    _handleEvent: function(event)
    {
        var cb, idx;
        // handle invoke callbacks
        if (event.name == "result")
        {
            idx = event.data.idx;
            if (typeof this._cbs[idx] == 'function')
            {
                cb = this._cbs[idx];
                delete event.data.idx;
                cb(event.data);
            }
            delete this._cbs[idx];
            return true;
        }
        // unhandled
        return false;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp */
nrdp.storage = {
    classname: "StorageBridge",
    _path: "storage",
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get NO_DEVICE_ACCOUNT() { return "NDAKADN"; },
    get size() { return this._syncData.data ? JSON.stringify(this._syncData.data).length : 0; },
    get flushCount() { return this._syncData.flushCount; },
    get secureStoreSize() { return this._syncData.secureStoreSize; },
    get transientData() { return this._syncData.transientData; },
    set transientData(d) {
        this._syncData.transientData = d;
        // no need to _setProperty here, it will be done as part of flush() on page exit
    },
    length: function length(dak) {
        var data = nrdp.storage._getData(dak);
        var count = 0;
        for (var key in data) {
            if (data[key] != undefined)
                count++;
        }
        return count;
    },
    key: function key(dak, n) {
        var data = nrdp.storage._getData(dak);
        for (var k in data) {
            if (data[k] == undefined)
                continue;
            if (n == 0)
                return k;
            n--;
        }
        return undefined;
    },
    _normalizeKey: function _normalizeKey(key) {
        if (key === undefined)
            return "undefined";
        else if (key === null)
            return "null";
        else
            return key.toString();
    },
    getItem: function getItem(dak, key) {
        var stor = nrdp.storage;
        var data = stor._getData(dak);
        return data[stor._normalizeKey(key)];
    },
    setItem: function setItem(dak, key, value) {
        var stor = nrdp.storage;
        var data = stor._getData(dak);
        key = stor._normalizeKey(key);
        data[key] = value;
        nrdp._invoke("storage", "setItem", {dak: dak, key: key, value: value});
    },
    removeItem: function removeItem(dak, key) {
        var stor = nrdp.storage;
        var data = stor._getData(dak);
        key = stor._normalizeKey(key);
        if (data.hasOwnProperty(key))
            delete data[key];
        nrdp._invoke("storage", "removeItem", {dak: dak, key: key});
    },
    clear: function clear(dak) {
        if (nrdp.storage._syncData.data)
            delete nrdp.storage._syncData.data[dak];
        nrdp._invoke("storage", "clear", {dak: dak});
        nrdp.storage.disk.clear(dak);
    },
    _clearAll: function _clearAll() {
        nrdp.storage._syncData = {
            data: {}
        };
        nrdp._invoke("storage", "clearAll");
        nrdp.storage.disk.clearAll();
    },
    get disk() {
        if (!this._diskContexts["ui-cache"]) {
            return nrdp.storage.createDiskStoreContext({ context: "ui-cache",
                                                         size: nrdp.options.ui_cache_capacity,
                                                         encrypted: true,
                                                         signature: true });
        }
        return nrdp.storage._diskContexts["ui-cache"];
    },
    _diskContextPrototype: {
        size: undefined,
        valid: undefined,
        signature: undefined,
        encrypted: undefined,
        getSize: function getSize(cb) {
            if (!this.valid)
                throw "getSize called on destroyed context";
            nrdp.storage._fn("diskStoreGetSize", { context: this.context }, cb);
        },
        create: function create(dak, key, value, cb) {
            if (!this.valid)
                throw "create called on destroyed context";
            var obj = {
                context: this.context,
                dak: dak,
                key: key,
                value: value
            };
            nrdp.storage._fn("diskStoreCreate", obj, cb);
        },
        clear: function clear(dak, cb) {
            if (!this.valid)
                throw "clear called on destroyed context";
            var obj = {
                context: this.context,
                dak: dak
            };
            nrdp.storage._fn("diskStoreClear", obj, cb);
        },
        clearAll: function clear(cb) {
            if (!this.valid)
                throw "clearAll called on destroyed context";
            var obj = {
                context: this.context
            };
            nrdp.storage._fn("diskStoreClearAll", obj, cb);
        },
        append: function append(dak, key, value, cb) {
            if (!this.valid)
                throw "append called on destroyed context";
            var obj = {
                context: this.context,
                dak: dak,
                key: key,
                value: value
            };
            nrdp.storage._fn("diskStoreAppend", obj, cb);
        },
        remove: function remote(dak, key, cb) {
            if (!this.valid)
                throw "remove called on destroyed context";
            var obj = {
                context: this.context,
                dak: dak,
                key: key
            };
            nrdp.storage._fn("diskStoreRemove", obj, cb);
        },
        read: function read(dak, key, begin, end, cb) {
            if (!this.valid)
                throw "read called on destroyed context";
            var obj = {
                context: this.context,
                dak: dak,
                key: key,
                begin: begin || 0,
                end: end || -1
            };
            nrdp.storage._fn("diskStoreRead", obj, cb);
        },
        query: function query(dak, prefix, cb, validate) {
            if (!this.valid)
                throw "query called on destroyed context";
            var obj = { context: this.context, dak: dak, prefix: prefix, validate: validate };
            nrdp.storage._fn("diskStoreQuery", obj, function(data) { if (cb) cb(data.keys); });
        },
        validate: function(cb) {
            if (!this.valid)
                throw "query called on destroyed context";
            var obj = { context: this.context, validate: true };
            nrdp.storage._fn("diskStoreQuery", obj, function(data) { if (cb) cb(data.keys); });
        },
        info: function info(cb) {
            if (!this.valid)
                throw "info called on destroyed context";
            nrdp.storage._fn("diskStoreInfo", { context: this.context }, function(data) { cb(data.info); });
        },
        corrupt: function corrupt(dak, key, mode, cb) {
            if (!this.valid)
                throw "corrupt called on destroyed context";
            var obj = {
                context: this.context,
                mode: mode,
                dak: dak,
                key: key
            };
            nrdp.storage._fn("diskStoreCorrupt", obj, cb);
        }
    },
    _diskContext: function _diskContext(data) {
        this.context = data.context;
        this.encrypted = data.encrypted || false;
        this.signature = data.signature || false;
        this.size = data.size;
        this.valid = true;
    },
    createDiskStoreContext: function(object, cb) {
        if (!object.context || this._diskContexts.hasOwnProperty(object.context)) {
            var err = "Invalid context: " + object.context;
            nrdp.log.error(err);
            if (cb)
                cb({success: false, error: err});
            return undefined;
        }
        var result = nrdp._invoke("storage", "diskStoreCreateContext", object);
        if (result.success) {
            var ret = new nrdp.storage._diskContext(object);
            this._diskContexts[object.context] = ret;
            if (cb)
                cb({success: true});
            return ret;
        }
        if (cb)
            cb({success: false, error: result.error});
        return undefined;
    },
    destroyDiskStoreContext: function(ctx, cb) {
        var context;
        if (ctx instanceof Object) {
            context = ctx;
        } else {
            context = this._diskContexts[ctx];
        }
        if (!context || !context.valid) {
            if (cb)
                cb({success: false, error: "destroyDiskStoreContext called on destroyed context"});
            return false;
        }
        context.valid = false;
        delete this._diskContexts[context.context];
        var result = nrdp._invoke("storage", "diskStoreDestroyContext", { context: context.context });
        if (cb)
            cb(result);
        return result && result.success;
    },
    get diskStoreContexts() { return this._diskContexts; },
    _getData: function _getData(dak) {
        if (!this._syncData)
            this._syncData = { data: {} };
        else if (!this._syncData.data)
            this._syncData.data = {};
        if (!this._syncData.data[dak])
            this._syncData.data[dak] = {};
        return this._syncData.data[dak];
    },
    setPersistentData: function setPersistentData(dak, key, data, deflate, cb) {
        nrdp.storage._fn("setPersistentData",
                         {dak:dak, key:key, data:data, deflate:deflate}, cb);
    },
    unsetPersistentData: function unsetPersistentData(dak, key, cb) {
        nrdp.storage._fn("unsetPersistentData",
                         {dak:dak, key:key}, cb);
    },
    getPersistentData: function getPersistentData(dak, key, inflate, cb) {
        nrdp.storage._fn("getPersistentData",
                         {dak:dak, key:key, inflate:inflate}, cb);
    },
    getFlushCount: function getFlushCount() {
        nrdp._invoke("storage", "getFlushCount");
    },
    flush: function flush() {
        nrdp._setProperty("storage", "transientData", nrdp.storage._syncData.transientData);
        nrdp._invoke("storage", "flush");
    },
    getUsedSecureStoreSize: function getUsedSecureStoreSize() {
        return nrdp._invoke("storage", "getUsedSecureStoreSize");
    },
    _nextIdx: 1,
    _cbs: {},
    _syncData: {},
    _diskContexts: {},
    _fn: function _fn(name, args, cb) {
        if (!args)
            args = {};
        args.idx = this._nextIdx++;
        if (cb)
            this._cbs[args.idx] = cb;
        nrdp._invoke("storage", name, args);
    },
    _handleEvent: function _handleEvent(event) {
        if (event.data && event.data.idx) {
            if (typeof this._cbs[event.data.idx] == "function") {
                this._cbs[event.data.idx](event.data);
            }
            delete this._cbs[event.data.idx];
        } else if (event.name == "overbudget") {
            event.data.type = event.name;
            nrdp._callEventListeners(this, event.data);
        } else if (event.name == "flushCount") {
            event.data.type = event.name;
            nrdp._callEventListeners(this, event.data);
        } else {
            return false;
        }
        return true;
    },
    _updateProperty: function _updateProperty(property, value) {
        if (property === "diskStoreInfo") {
            var old = this._diskContexts;
            this._diskContexts = {};
            for (var v in value) {
                if (old.hasOwnProperty(v)) {
                    this._diskContexts[v] = old[v];
                    delete old[v];
                } else {
                    var vals = value[v];
                    vals.context = v;
                    this._diskContexts[v] = new nrdp.storage._diskContext(vals);
                }
            }
            for (var i in old) {
                old[i].valid = false;
            }
        } else {
            this._syncData[property] = value;
        }
    }
};
nrdp.storage._diskContext.prototype = nrdp.storage._diskContextPrototype;
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp */
nrdp.texttospeech = {
    classname: "TextToSpeechBridge",
    _path: "texttospeech",
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get languages() { return this._syncData.languages; },
    get language() { return this._syncData.language; },
    set language(v) { nrdp._setProperty("texttospeech", "language", v); },
    get voices() { return this._syncData.voices; },
    get voice() { return this._syncData.voice; },
    get rate() { return this._syncData.rate; },
    get pitch() { return this._syncData.pitch; },
    get capabilities() { return this._syncData.capabilities; },
    VERBOSITY_LOW: 0, VERBOSITY_HIGH: 1,
    get verbosity() { return this._syncData.verbosity; },
    say: function say(text, cb) {
        if (typeof text === "string")
            nrdp.texttospeech._fn("say", { text: text }, { cb: cb, time: nrdp.mono() });
        else if (cb)
            cb("said", text);
    },
    stopAndFlush: function stopAndFlush(cb) {
        nrdp.texttospeech._fn("stopAndFlush", {}, cb);
    },
    silence: function silence(ms, cb) {
        if (ms < 0) {
            if (cb)
                cb("said");
        } else {
            nrdp.texttospeech._fn("silence", { ms: ms }, cb);
        }
    },
    _nextIdx: 1,
    _cbs: {},
    _fn: function _fn(name, args, cb) {
        if (!args) args = {};
        args.id = this._nextIdx++;
        if (cb)
            this._cbs[args.id] = cb;
        nrdp._invoke("texttospeech", name, args);
    },
    _handleEvent: function _handleEvent(event) {
        if (event.data && event.data.id) {
            var cb = this._cbs[event.data.id];
            if (typeof cb === "function") {
                cb(event.name, event.data.data);
                if (event.name !== "saying")
                    delete this._cbs[event.data.id];
            } else if (typeof cb === "object" && cb.hasOwnProperty("time")) {
                if (event.name !== "saying")
                    delete this._cbs[event.data.id];
                if (typeof cb.cb === "function") {
                    if (event.data.data === undefined)
                        event.data.data = {};
                    event.data.data.elapsed = nrdp.mono() - cb.time;
                    cb.cb(event.name, event.data.data);
                }
            }
        } else {
            return false;
        }
        return true;
    },
    _syncData: {},
    _updateProperty: function _updateProperty(property, value) {
        var evt;
        if (nrdp.isReady) {
            if (property == "capabilities") {
                evt = {
                    type: "capabilities",
                    old: this.capabilities
                };
            } else if (property == "language") {
                evt = {
                    type: "language",
                    old: this.language
                };
            } else if (property == "voice") {
                evt = {
                    type: "voice",
                    old: this.voice
                };
            } else if (property == "rate") {
                evt = {
                    type: "rate",
                    old: this.rate
                };
            } else if (property == "pitch") {
                evt = {
                    type: "pitch",
                    old: this.pitch
                };
            } else if (property == "verbosity") {
                evt = {
                    type: "verbosity",
      old: this.verbosity
  };
     }
        }
        this._syncData[property] = value;
        if (evt) {
            nrdp._callEventListeners(this, evt);
        }
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp */
nrdp.webcrypto = {
    _nextIdx: 1,
    _cbs: {},
    // properties
    get authenticationType() { return this._syncData.authenticationType; },
    _fn: function _fn(name, params, cb)
    {
        if (!params) params = {};
        params.idx = this._nextIdx++;
        this._cbs[params.idx] = cb;
        nrdp._invoke("webcrypto", name, params);
        return params.idx;
    },
    _handleEvent: function _handleEvent(event)
    {
        var cb, idx;
        if (event.name != "result")
        {
            return false;
        }
        idx = event.data.idx;
        if (typeof this._cbs[idx] == "function")
        {
            cb = this._cbs[idx];
            delete event.data.idx;
            cb(event.data);
        }
        delete this._cbs[idx];
        return true;
    },
    // keyformat constants: matches IWebCrypto::KeyFormat
    get RAW() { return 0; },
    get PKCS8() { return 1; },
    get SPKI() { return 2; },
    get JWK() { return 3; },
    // keyusage contants: matches IWebCrypto::KeyUsage
    get ENCRYPT() { return 0x01; },
    get DECRYPT() { return 0x02; },
    get SIGN() { return 0x04; },
    get VERIFY() { return 0x08; },
    get DERIVE() { return 0x10; },
    get WRAP() { return 0x20; },
    get UNWRAP() { return 0x40; },
    // keytype constants: matches IWebCrypto::KeyType
    get SECRET() { return 0; },
    get PUBLIC() { return 1; },
    get PRIVATE() { return 2; },
    // keywrap method: matches IWebCrypto::JweEncMethod
    get A128GCM() { return 0; },
    get A256GCM() { return 1; },
    setBinary: function setBinary(binary, cb)
    {
        return nrdp.webcrypto._fn("setBinary", {binary: binary}, cb);
    },
    // Algorithm objects
    // Valid algorithm names are: HMAC, AES-CBC, AES-GCM, AES-CTR, RSAES-PKCS1-v1_5,
    // RSASSA-PKCS1-v1_5, RSA-OAEP
    // Valid hash names are:
    // SHA-1, SHA-224, SHA-256, SHA-384, SHA-512
    // var hashparams = {
    //       hash: {
    //          name: string // hashname
    //       }
    // };
    // var rsakeyparams = {
    //       modulusLength: integer,
    //       publicExponent: string
    // };
    // var symkeyparams = {
    //       length: integer // key length in bits
    // };
    // var algorithm = {
    //       name: string, // algorithm name
    //       params: hashparams || rsakeyparams || symkeyparams
    // };
    // Key information
    // When a key is returned in a result, it will have the following properties:
    //  handle (integer)
    //  type (KeyType constants above)
    //  extractable (bool)
    //  usage (array of 0 or more KeyUsage constants above)
    //  algorithm (an algorithm object as described above)
    // import a key
    // @param data: The raw data of the key. This can either be Base64 encoded or a
    //              a DataBuffer.
    // @param keyformat: A keyformat constant (RAW, PKCS8, SPKI, JWK)
    // @param algorithm: An object describing the key algorithm
    // @param extractable: Whether or not the raw keying material may be extracted by app
    // @param keyusage: An array of constants indicating how the key may be used
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "key" property with key information
    importKey: function importKey(data, keyformat, algorithm, extractable, keyusage,
                                  cb)
    {
        return nrdp.webcrypto._fn(
            "importKey", {data: data, keyformat: keyformat, algorithm: algorithm,
                          extractable: extractable, usage: keyusage}, cb);
    },
    // export a key
    // @param key: The keyhandle to export
    // @param keyformat: The keyformat to export the key to
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a property "data" with the raw key data
    exportKey: function exportKey(key, keyformat, cb)
    {
        return nrdp.webcrypto._fn("exportKey", {key: key, keyformat: keyformat}, cb);
    },
    // get key info
    // @param key: The keyhandle to get info on
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "key" property with key information.
    getKeyInfo: function getKeyInfo(key, cb)
    {
        return nrdp.webcrypto._fn("getKeyInfo", {key: key}, cb);
    },
    // Compute message digest
    // @param data: The raw data to hash. Either Base64 encoded or a DataBuffer
    // @param hashalgorithm: The algorithm for hashing
    //                       (SHA-1, SHA-224, SHA-256, SHA-384, SHA-512)
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the hash
    digest: function digest(data, hashalgorithm, cb)
    {
        return nrdp.webcrypto._fn(
            "digest", {data: data, algorithm: hashalgorithm}, cb);
    },
    // AES-128-CBC encrypt data
    // @param key: The keyhandle (returned from importKey) for key to encrypt with
    // @param init: Initialization vector for encryption. Either Base64 or a DataBuffer
    // @param data: The raw data to encrypt. Either Base64 encoded or a DataBuffer
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will haven a "data" property containing the encrypted
    //   buffer (Base64 encoded)
    aesEncrypt: function aesEncrypt(key, init, data, cb)
    {
        return nrdp.webcrypto._fn(
            "aesCrypt", {key: key, init: init, data: data, operation: 0}, cb);
    },
    // AES-128-CBC decrypt data
    // @param key: The keyhandle (returned from importKey) for key to decrypt with
    // @param init: Initialization vector for encryption. Either Base64 or a DataBuffer
    // @param data: The raw data to decrypt. Either Base64 encoded or a DataBuffer
    // @param cb: Continuation function(event):
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the decrypted
    //   buffer (Base64 encoded)
    aesDecrypt: function aesDecrypt(key, init, data, cb)
    {
        return nrdp.webcrypto._fn(
            "aesCrypt", {key: key, init: init, data: data, operation: 1}, cb);
    },
    // AES unwrap - RFC 3394
    // @param data: The wrapped data
    // @param wrappedAlgorithm: The algorithm for the wrapped key
    // @param wrappingKey: The handle to the key used to unwrap
    // @param usage: The usage for the unwrapped key
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "key" property with key information
    aesUnwrap: function aesUnwrap(data, wrappedAlgorithm, wrappingKey, usage, cb)
    {
        return nrdp.webcrypto._fn(
            "aesUnwrap", {data: data, algorithm: wrappedAlgorithm, key: wrappingKey,
                          usage: usage}, cb);
    },
    // Generate a symmetric key
    // This method generates a single random key and places it in the key store.
    // @param algorithm In. The algorithm used to generate the key.
    // @param extractable In. Whether the key should be marked as extractable
    // @param keyusage: An array of constants indicating how the key may be used
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "key" property with key information.
    symKeyGen: function symKeyGen(algorithm, extractable, keyusage, cb)
    {
        return nrdp.webcrypto._fn(
            "symKeyGen", {algorithm: algorithm, extractable: extractable,
                          usage: keyusage}, cb);
    },
    // Generate a RSA public/private key pair
    // @param algorithm: The algorithm object describing how the key will be generated
    // @param extractable: Whether or not the raw keying material may be extracted by app
    // @param keyusage: An array of constants indicating how the key may be used
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have "publickey" and "privatekey" properties each
    //   containing key information.
    rsaKeyGen: function rsaKeyGen(algorithm, extractable, keyusage, cb)
    {
        return nrdp.webcrypto._fn(
            "rsaKeyGen", {algorithm: algorithm, extractable: extractable,
                          usage: keyusage}, cb);
    },
    // RSAES-PKCS1-v1_5 encrypt data
    // @param key: The keyhandle (returned from rsaKeyGen) for key to encrypt with
    // @param data: The raw data to encrypt. Either Base64 encoded or a DataBuffer
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the encrypted
    //   buffer (Base64 encoded)
    rsaEncrypt: function rsaEncrypt(key, data, cb)
    {
        return nrdp.webcrypto._fn(
            "rsaCrypt", {key: key, data: data, operation: 0}, cb);
    },
    // RSAES-PKCS1-v1_5 decrypt data
    // @param key: The keyhandle (returned from rsaKeyGen) for key to decrypt with
    // @param data: The raw data to decrypt. Either Base64 encoded or a DataBuffer
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the decrypted
    //   buffer (Base64 encoded)
    rsaDecrypt: function rsaDecrypt(key, data, cb)
    {
        return nrdp.webcrypto._fn(
            "rsaCrypt", {key: key, data: data, operation: 1}, cb);
    },
    // RSA sign a block of data
    // @param key: The keyhandle (returned from rsaKeyGen) for key to sign with
    // @param data: The raw data to sign. Either Base64 encoded or a DataBuffer
    // @param hashalgorithm: The algorithm for hashing
    //                       (SHA-1, SHA-224, SHA-256, SHA-384, SHA-512)
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the rsa
    //   signature (Base64 encoded)
    rsaSign: function rsaSign(key, data, hashalgorithm, cb)
    {
        return nrdp.webcrypto._fn(
            "rsaSign", {key: key, data: data, algorithm: hashalgorithm}, cb);
    },
    // RSA verify a block of data
    // @param key: The keyhandle (returned from rsaKeyGen) for key to verify with
    // @param data: The raw data to compute the signature of. Either Base64 encoded or a
    //              DataBuffer
    // @param hashalgorithm: The algorithm for hashing
    //                       (SHA-1, SHA-224, SHA-256, SHA-384, SHA-512)
    // @param signature: The signature to verify against
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "verified" property which will be true if the
    //   computed signature matches the passed in signature
    rsaVerify: function rsaVerify(key, data, hashalgorithm, signature, cb)
    {
        return nrdp.webcrypto._fn(
            "rsaVerify", {key: key, data: data, algorithm: hashalgorithm,
                          signature: signature}, cb);
    },
    // Generate a Diffie-Hellman public/private key pair
    // This method computes DH public/private key pair
    // @param algorithm In. The full details about the key gen algorithm, including
    //     the prime and generator values
    // @param extractable In. Whether or not the raw key material may be exported
    // @param keyUsage In. The allowed usages of the keys
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have "publickey" and "privatekey" properties each
    //   containing key information.
    dhKeyGen: function dhKeyGen(algorithm, extractable, keyusage, cb)
    {
        return nrdp.webcrypto._fn(
            "dhKeyGen", {algorithm: algorithm, extractable: extractable,
                         usage: keyusage}, cb);
    },
    // Derive a shared private key
    // This method computes a shared private key using a baseKey produced by
    // dhKeyGen() plus the public key from the remote peer who has previously
    // obtained the public baseKey.
    // @param basekey In. The handle of the key that started the DH exchange,
    //   produced by a call to dhKeyGen
    // @param peerkey In. The raw public key received from the remote peer, base64-encoded
    // @param algorithm In. The full details about the algorithm to be associated with
    //   the derived key
    // @param extractable In. Whether or not the raw key material of the derived
    //   key may be exported
    // @param keyUsage In. The allowed usages of the derived key
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "key" property with key information.
    dhDerive: function dhDerive(basekey, peerkey, algorithm, extractable, keyusage, cb)
    {
        return nrdp.webcrypto._fn(
            "dhDerive", {basekey: basekey, peerkey: peerkey, algorithm: algorithm,
                         extractable: extractable, usage: keyusage}, cb);
    },
    nflxDhDerive: function nflxDhDerive(basekey, peerKeyData, derivationKey, cb)
    {
        return nrdp.webcrypto._fn(
            "nflxDhDerive", {basekey: basekey, peerkey: peerKeyData, derivationKey: derivationKey}, cb);
    },
    // Compute the HMAC signature of the data
    // @param key: The keyhandle for key to HMAC with
    // @param data: The raw data to HMAC. Either Base64 encoded or a DataBuffer
    // @param hashalgorithm: The algorithm for hashing
    //                       (SHA-1, SHA-224, SHA-256, SHA-384, SHA-512)
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the
    //   HMAC (Base64 encoded)
    hmac: function hmac(key, data, hashalgorithm, cb)
    {
        return nrdp.webcrypto._fn(
            "hmac", {key: key, data: data, algorithm: hashalgorithm}, cb);
    },
    // Verify an HMAC signature
    // @param key: The keyhandle for key to HMAC with
    // @param data: The raw data to HMAC. Either Base64 encoded or a DataBuffer
    // @param hashalgorithm: The algorithm for hashing
    //                       (SHA-1, SHA-224, SHA-256, SHA-384, SHA-512)
    // @param signature: The signature to verify against
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "verified" property which will be true if the
    //   computed signature matches the passed in signature
    hmacVerify: function hmacVerify(key, data, hashalgorithm, signature, cb)
    {
        return nrdp.webcrypto._fn(
            "hmacVerify", {key: key, data: data, algorithm: hashalgorithm,
                           signature: signature}, cb);
    },
    // Unwrap a JWE-wrapped key
    // @param data: The base64-encoded wrapped key formatted to the JWE spec
    // @param key: The keyhandle for key to decrypt with
    // @param algorithm In. In case the unwrapped JDK does not have the 'alg'
    //                      field inside it, use this value, otherwise ignore
    // @param extractable: Whether or not the raw keying material may be extracted by app
    // @param keyusage: An array of constants indicating how the key may be used
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "key" property containing key information
    unwrapJwe: function unwrapJwe(data, key, algorithm, extractable, keyusage, cb)
    {
        return nrdp.webcrypto._fn(
            "unwrapJwe", {data: data, key: key, algorithm: algorithm,
                          extractable: extractable, usage: keyusage}, cb);
    },
    //* JWE-wrap an existing key
    // This method wraps an existing key in the keystore according to the rules
    // in draft-ietf-jose-json-web-encryption-08, using an existing wrapping key
    // also in the keystore. The result is a base-64 encoded JWE Compact
    // Serialization described above.
    // @param key In. The handle of the key to use in during the key wrap process.
    // @param keytowrap In. The handle of the key to be wrapped
    // @param algorithm In. The encryption algorithm to be applied to the
    //    Content Master Key (CMK) during the wrapping process. This must be
    //    consistent with the algorithm associated with wrappingKeyHandle.
    //    Only RSA-OAEP and AES-KW algorithms are supported.
    // @param method In. The keywrap method to wrap with.
    //    For AES-KW, only A128GCM is supported,
    //    while RSA-OAEP supports A128GCM and A256GCM.
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the wrapped key
    wrapJwe: function wrapJwe(key, keytowrap, algorithm, method, cb)
    {
        return nrdp.webcrypto._fn(
            "wrapJwe", {key: key, keytowrap: keytowrap, algorithm: algorithm,
                        meth: method}, cb);
    },
    // ECC sign a block of data
    // @param key: The keyhandle for key to sign with
    // @param data: The raw data to sign. Either Base64 encoded or a DataBuffer
    // @param hashalgorithm: The algorithm for hashing
    //                       (SHA-1, SHA-224, SHA-256, SHA-384, SHA-512)
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, result will have a "data" property containing the ECC
    //   signature (Base64 encoded)
    eccSign: function eccSign(key, data, hashalgorithm, cb)
    {
        return nrdp.webcrypto._fn(
            "eccSign", {key: key, data: data, algorithm: hashalgorithm}, cb);
    },
    // Get a key by name
    // @param name. In. string
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    //   On success, and if the key was found, result will have a "key" property
    //   containing the keyInfo (with the name) for the key. If the key was not
    //   found, success will still be true, but key will be undefined.
    getKeyByName: function getKeysByName(name, cb)
    {
        return nrdp.webcrypto._fn(
            "getKeyByName", {name: name}, cb);
    },
    // Given an array of keys, they are persisted so that they will be
    // restored later with the same handles.
    // @param keys: In. array of key objects
    // @param cb: Continuation function(event)
    //   Event named "result" will have a property "success" indicating success.
    persistKeys: function persistKeys(keys, cb)
    {
        return nrdp.webcrypto._fn(
            "persistKeys", {keys: keys}, cb);
    },
    deleteKey: function deleteKey(key, cb)
    {
        return nrdp.webcrypto._fn(
            "deleteKey", {key: key}, cb);
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp */
nrdp.websocket = {
    _nextWebSocket: 0,
    _websockets: {},
    _nextIdx: 1,
    _cbs: [],
    _fn: function _fn(name, params, cb) {
        if (!params) params = {};
        params.idx = this._nextIdx++;
        this._cbs[params.idx] = cb;
        nrdp._invoke("websocket", name, params);
        return params.idx;
    },
    _handleEvent: function _handleEvent(event) {
        var that;
        if (event.name === "result") {
            if(!event.data || !event.data.idx) {
                return false;
            }
            if (typeof this._cbs[event.data.idx] === "function") {
                this._cbs[event.data.idx](event.data);
            }
            delete this._cbs[event.data.idx];
        } else if (event.name == "open") {
            that = this._websockets[event.data.websocket];
            if (!that) return false;
            that._protocol = event.data.protocol;
            that._url = event.data.url;
            that._readyState = that.OPEN;
            if(typeof that.onopen === "function") {
                ;
                that.onopen();
            }
        } else if (event.name == "message") {
            that = this._websockets[event.data.websocket];
            if (!that) return false;
            if(that.readyState === that.OPEN) {
                if(typeof that.onmessage === "function") {
                    ;
                    that.onmessage(event.data);
                }
            }
        } else if (event.name == "error") {
            that = this._websockets[event.data.websocket];
            if (!that) return false;
            that._readyState = that.CLOSING;
            if(typeof that.onerror === "function") {
                ;
                that.onerror();
            }
        } else if (event.name == "close") {
            that = this._websockets[event.data.websocket];
            if (!that)
                return false;
            that._readyState = that.CLOSED;
            delete this._websockets[that._websocket];
            if (that.onclose instanceof Function) {
                ;
                that.onclose();
            }
            that._websocket = null;
        } else {
            return false;
        }
        return true;
    },
    _onShutdown: function _shutdown() {
        for (var i in nrdp.websocket._websockets) {
            nrdp.websocket._websockets[i].close();
        }
    },
    create: function create(url, protocols, cb) {
        if (!this._nextWebSocket)
            nrdp.addEventListener("shutdown", nrdp.websocket._onShutdown);
        var websocket = ++this._nextWebSocket;
        this._fn("create",
                 { websocket: websocket,
                   url: url,
                   protocols: protocols },
                 cb);
        return websocket;
    },
    send: function send(websocket, data, cb) {
        this._fn("send",
                 { websocket: websocket,
                   data: data },
                 cb);
    },
    close: function close(websocket, cb) {
        this._fn("close",
                 { websocket: websocket },
                 cb);
    },
    test: function test(url, cb) {
        this._fn("test",
                 { url: url },
                 cb);
    }
};
nrdp.WebSocket = function(url, protocols) {
    var that = this;
    that._websocket = nrdp.websocket.create(url, protocols ? protocols : "", function(result) {
        if(result.success) {
            that._readyState = that.CONNECTING;
        }
    });
    nrdp.websocket._websockets[that._websocket] = that;
};
nrdp.WebSocket.prototype = (function() {
    var proto = {
        _websocket: null,
        _readyState: 3,
        _protocol: "",
        _url: "",
        get protocol() { return this._protocol; },
        get url() { return this._url; },
        get URL() { return this._url; },
        get readyState() { return this._readyState; },
        get CONNECTING() { return 0; },
        get OPEN() { return 1; },
        get CLOSING() { return 2; },
        get CLOSED() { return 3; },
        onclose: null,
        onerror: null,
        onmessage: null,
        onopen: null,
        send: function send(message) {
            if(this.readyState === this.OPEN) {
                nrdp.websocket.send(this._websocket, message);
            } else {
                ;
            }
        },
        close: function close() {
            if(this._websocket) {
                nrdp.websocket.close(this._websocket);
            }
        }
    };
    return proto;
})();
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp */
nrdp.tile = {
    classname: "TileBridge",
    _path: "tile",
    get version() { return this._syncData.version; },
    get capabilities() { return nrdp.capabilities.tile; },
    get enabled() { return nrdp.options.tilesEnabled; },
    get errorCodes() { return this._syncData.errorCodes; },
    setSplash: function setSplash(splashes, cb) {
        function sendError(error, errorCode) {
            if (cb) {
                cb({ success: false, error: error, errorCode: errorCode });
            }
        }
        if (!splashes)
            splashes = [];
        var splashFields = [
            { name: "width", type: "number" },
            { name: "height", type: "number" },
            { name: "data", type: function(t) { return (t instanceof ArrayBuffer || t instanceof Uint8Array || typeof t === 'string'); } }
        ];
        for (var s=0; s<splashes.length; ++s) {
            var splash = splashes[s];
            for (var f=0; f<splashFields.length; ++f) {
                var prop = splashFields[f].name;
                if (!splash.hasOwnProperty(prop)) {
                    if (!splashFields[f].optional) {
                        sendError("Splash is missing required field: " + prop, this.errorCodes.RequiredFieldMissing);
                        return;
                    }
                    continue;
                }
                if (splashFields[f].type instanceof Function ? (!splashFields[f].type(splash[prop])) : typeof splash[prop] != splashFields[f].type) {
                    sendError("Splash property: " + prop + " has wrong type", this.errorCodes.InvalidPropertyType);
                    return;
                }
                if (splashFields[f].maxSize && splash[prop].length > splashFields[f].maxSize) {
                    splash[prop] = splash[prop].substr(0, splashFields[f].maxSize);
                } else if (typeof splash[prop] == 'number' && splash[prop] < 0) {
                    sendError("Invalid integral negative property " + prop, this.errorCodes.InvalidPropertyValue);
                    return;
                }
            }
        }
        var id = this._nextIdx++;
        this._cbs[id] = cb;
        nrdp._invoke(this._path, "setSplash", { id: id, splashes: splashes },
                     function(result) {
                         if (!result.success) {
                             sendError(result.error, result.errorCode);
                         } else if (cb) {
                             cb({success: true, errorCode: this.errorCodes.Success});
                         }
                     });
    },
    setTiles: function setTiles(tiles, cb) {
        function sendError(error, errorCode) {
            if (cb) {
                cb({ success: false, error: error, errorCode: errorCode });
            }
        }
        if (tiles) {
            if (!(tiles instanceof Object)) {
                sendError("Invalid tile type", this.errorCodes.InvalidTileType);
                return;
            } else if (!(tiles.groups instanceof Array) || !tiles.groups.length) {
                sendError("Invalid tile object, no groups", this.errorCodes.NoTileGroups);
                return;
            }
            if (tiles.groups.length > this.capabilities.maxNumGroups) {
                sendError("Too many groups", this.errorCodes.GroupMaximumExceeded);
                return;
            }
            var tileFields = [
                { name: "url", type: "string", maxSize: 1024 },
                { name: "width", type: "number" },
                { name: "height", type: "number" },
                { name: "expiry", type: "number" },
                { name: "title", type: "string", maxSize: 50, optional: true },
                { name: "description", type: "string", maxSize: 150, optional: true },
                { name: "shortText", type: "string", maxSize: 150, optional: true },
                { name: "deepLink", type: "string", maxSize: 4096, optional: true }
            ];
            for (var g=0; g<tiles.groups.length; ++g) {
                var group = tiles.groups[g];
                if (!(group.tiles instanceof Array) || !group.tiles.length) {
                    sendError("Invalid tile group, no tiles in group", this.errorCodes.EmptyTileGroup);
                    return;
                }
                if (group.tiles.length > this.capabilities.maxTilesPerGroup) {
                    sendError("Too many tiles", this.errorCodes.TileMaximumExceeded);
                    return;
                }
                for (var t=0; t<group.tiles.length; ++t) {
                    var tile = group.tiles[t];
                    for (var f=0; f<tileFields.length; ++f) {
                        var prop = tileFields[f].name;
                        if (!tile.hasOwnProperty(prop)) {
                            if (!tileFields[f].optional) {
                                sendError("Tile is missing required field: " + prop, this.errorCodes.RequiredFieldMissing);
                                return;
                            }
                            continue;
                        }
                        if (typeof tile[prop] != tileFields[f].type) {
                            sendError("Tile property: " + prop + " has wrong type", this.errorCodes.InvalidPropertyType);
                            return;
                        }
                        if (tileFields[f].maxSize && tile[prop].length > tileFields[f].maxSize) {
                            tile[prop] = tile[prop].substr(0, tileFields[f].maxSize);
                        } else if (typeof tile[prop] == 'number' && tile[prop] < 0) {
                            sendError("Invalid integral negative property " + prop, this.errorCodes.InvalidPropertyValue);
                            return;
                        }
                    }
                }
            }
        }
        var id = this._nextIdx++;
        this._cbs[id] = cb;
        nrdp._invoke(this._path, "setTiles", { id: id, tiles: tiles },
                     function(result) {
                         if (!result.success) {
                             sendError(result.error, result.errorCodes);
                         } else if (cb) {
                             cb({success: true, errorCode: this.errorCodes.Success});
                         }
                     });
    },
    validateTiles: function validateTiles(cb) {
        var id = this._nextIdx++;
        this._cbs[id] = cb;
        nrdp._invoke(this._path, "validateTiles", { id: id });
    },
    validateSplash: function validateSplash(cb) {
        var id = this._nextIdx++;
        this._cbs[id] = cb;
        nrdp._invoke(this._path, "validateSplash", { id: id });
    },
    _handleEvent: function _handleEvent(event) {
        var cb = this._cbs[event.data.id];
        delete this._cbs[event.data.id];
        if (cb instanceof Function) {
            cb(event.data);
        }
        return true;
    },
    _syncData: {},
    _nextIdx: 1,
    _cbs: {}
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp */
nrdp.player = {
    classname: "PlayerBridge",
    _path: "player",
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get allowNrdpVideoSwitch() { return this._syncData.allowNrdpVideoSwitch; },
    get bufferPoolSize() { return this._syncData.bufferPoolSize; },
    get currentAudioTrack() { return this._syncData.currentAudioTrack; },
    set currentAudioTrack(trackid) { nrdp._setProperty("player", "currentAudioTrack", trackid); },
    get displayAspectRatio() { return this._syncData.displayAspectRatio; },
    get duration() { return this._syncData.duration; },
    get gopSize() { return this._syncData.gopSize; },
    get maxStreamingBuffer() { return this._syncData.maxStreamingBuffer; },
    get networkProfile() { return this._syncData.networkProfile; },
    set networkProfile(t) { nrdp._setProperty("player", "networkProfile", t); },
    get state() { return this._syncData.state; },
    // IAdaptiveStreamingPlayer state
    get OPENING() { return 0; },
    get PLAYING() { return 1; },
    get PAUSED() { return 2; },
    get STOPPED() { return 3; },
    get CLOSED() { return 4; },
    // NetworkProfile
    get WIRED() { return 0; },
    get WIFI() { return 1; },
    get MOBILE() { return 2; },
    // MediaType
    get MEDIA_UNKNOWN() { return -1; },
    get MEDIA_AUDIO() { return 0; },
    get MEDIA_VIDEO() { return 1; },
    get MEDIA_TEXT() { return 2; },
    // AudioTrackType
    get UNKNOWN_AUDIO() { return -1; },
    get PRIMARY_AUDIO() { return 0; },
    get COMMENTARY_AUDIO() { return 1; },
    get ASSISTIVE_AUDIO() { return 2; },
    addManifest: function addManifest(movieId, manifest) {
        nrdp._invoke("player", "addManifest", {movieId: movieId, manifest: manifest});
    },
    bringVideoToFront: function bringVideoToFront() { nrdp._invoke("player", "bringVideoToFront"); },
    cacheFlush: function cacheFlush(args) {
        nrdp._invoke("player", "cacheFlush", args);
    },
    cacheList: function cacheList(args) {
        nrdp._invoke("player", "cacheList", args);
    },
    cachePrepare: function cachePrepare(drmOnly, manifest, videoType, audioTrack) {
        var args = {
            'drmOnly': drmOnly,
            'manifest': manifest,
            'videoType': videoType,
            'audioTrack': audioTrack
        };
        nrdp._invoke("player", "cachePrepare", args);
    },
    cacheSetSize: function cacheSetSize(maxItems) {
        var args = {'maxItems' : maxItems};
        nrdp._invoke("player", "cacheSetSize", args);
    },
    close: function close() { nrdp._invoke("player", "close"); },
    externalIpAddressChange: function externalIpAddressChange(ipaddr) {
        nrdp._invoke("player", "externalIpAddressChange", {ipaddr: ipaddr});
    },
    getBufferRange: function getBufferRange() { nrdp._invoke("player", "getBufferRange"); },
    obtainPlaybackStat: function obtainPlaybackStat() { nrdp._invoke("player", "obtainPlaybackStat"); },
    obtainStreamingStat: function obtainStreamingStat() { nrdp._invoke("player", "obtainStreamingStat"); },
    open: function open(args) {
        nrdp._invoke("player", "open", args);
    },
    pause: function pause() { nrdp._invoke("player", "pause"); },
    play: function play(pts) {
        var args = {};
        if (pts != undefined && pts != null)
            args.pts = pts;
        nrdp._invoke("player", "play", args);
    },
    provideLicense: function provideLicense(license) { nrdp._invoke("player", "provideLicense", {license: license}); },
    sendVideoToBack: function sendVideoToBack() { nrdp._invoke("player", "sendVideoToBack"); },
    setStreamingBuffer: function setStreamingBuffer(powerSaving, maxBufferLen, minBufferLen) {
        nrdp._invoke("player", "setStreamingBuffer",
                     {powerSaving: powerSaving,
                      maxBufferLen: maxBufferLen,
                      minBufferLen: minBufferLen});
    },
    setVideoBitrateRanges: function setVideoBitrateRanges(ranges) {
        nrdp._invoke("player", "setVideoBitrateRanges", {ranges: ranges});
    },
    setVideoWindow: function setVideoWindow(x, y, width, height, transitionDuration) {
        var args = {
            x: x,
            y: y,
            width: width,
            height: height,
            transitionDuration: transitionDuration
        };
        nrdp._invoke("player", "setVideoWindow", args);
    },
    skip: function skip(ms) { nrdp._invoke("player", "skip", {pts:ms}); },
    stop: function stop() { nrdp._invoke("player", "stop"); },
    swim: function swim(pts, absolute, fuzz, allowRebuffer) {
        var args = { pts: pts };
        if (typeof absolute == "boolean")
            args.absolute = absolute;
        if (typeof fuzz == "number" && !isNaN(fuzz))
            args.fuzz = fuzz;
        if (typeof allowRebuffer == "boolean")
            args.allowRebuffer = allowRebuffer;
        nrdp._invoke("player", "swim", args);
    },
    unpause: function unpause() { nrdp._invoke("player", "unpause"); },
    // player volume
    // getVolume is returned in an event type "volume"
    getVolume : function(cb) {
        nrdp._invoke("player", "getVolume");
    },
    setVolume : function(volume, transition, ease) {
        nrdp._invoke("player", "setVolume",
                     {"targetVolume": volume, "transitionDuration": transition, "ease": ease});
    },
    // audio track switches
    startAudioTrackSwitch: function() {
        nrdp._invoke("player", "startAudioTrackSwitch", {});
    },
    completeAudioTrackSwitch: function() {
        nrdp._invoke("player", "completeAudioTrackSwitch", {});
    },
    notifyBufferingComplete: function notifyBufferingComplete(pts)
    {
        nrdp._invoke("player", "notifyBufferingComplete", {"pts": pts});
    },
    _handleEvent: function _handleEvent(event) {
        for (var prop in event.data._propups) {
            this._syncData[prop] = event.data._propups[prop];
        }
        if (event.name == "IASPlayer" || event.name == "PlaybackReporter") {
            event.data.time = event.time;
            nrdp._callEventListeners(this, event.data);
            return true;
        } else {
            return false;
        }
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp*/
nrdp.mdx = {
    _path: "mdx",
    addEventListener: function(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    _encodedList: [ { list: ['body', 'url', 'USN', 'friendlyName', 'serviceType', 'location'],
                      decodeFn: decodeURIComponent },
                    { list: [ 'responseHeaders' ],
                      decodeFn: function(obj) {
                          var decodedObj = {},
                          member;
                          for(member in obj) {
                              decodedObj[member] = decodeURIComponent(obj[member]);
                          }
                          return decodedObj;
                      }
                    }
                  ],
    get NOT_INITIALIZED() { return 0; },
    get INITIALIZED() { return 1; },
    get NOT_ADVERTISING() { return 0; },
    get ADVERTISING() { return 1; },
    get NOT_DISCOVERING() { return 0; },
    get DISCOVERING() { return 1; },
    get interfaceName() { return this._syncData ? this._syncData.interfaceName : undefined; },
    get localIPAddress() { return this._syncData ? this._syncData.localIPAddress : undefined; },
    get nativeVersion() { return this._syncData ? this._syncData.nativeVersion : undefined; },
    get state() { return this._syncData ? this._syncData.state : undefined; },
    MdxConfigure: function(advertisingPeriod,
                           advertisingTTL,
                           advertisingPort,
                           listeningPort,
                           numSsdpReplies,
                           msgLimit) {
        nrdp.mdx._fn("MdxConfigure",
                     {advertisingPeriod: advertisingPeriod,
                      advertisingTTL: advertisingTTL,
                      advertisingPort: advertisingPort,
                      listeningPort: listeningPort,
                      numSsdpReplies: numSsdpReplies,
                      msgLimit: msgLimit});
    },
    MdxInit: function(host, port, serviceType, uuid, asyncHttpRequests, webSocketServer) {
        nrdp.mdx._fn("MdxInit",
                     {host: encodeURIComponent(host),
                      port: port,
                      serviceType: encodeURIComponent(serviceType),
                      uuid: encodeURIComponent(uuid),
                      asyncHttpRequests: asyncHttpRequests,
                      webSocketServer: webSocketServer});
    },
    MdxDeinit: function() {
        nrdp.mdx._fn("MdxDeinit");
    },
    AddInterfaceName: function(name) {
        nrdp.mdx._fn("AddInterfaceName", {name: encodeURIComponent(name)});
    },
    InterfaceChanged: function(newInterface, connected, ipaddress, ssid) {
        nrdp.mdx._fn("InterfaceChanged", {newInterface: encodeURIComponent(newInterface), connected: connected, ipaddress: ipaddress, ssid:ssid});
    },
    SearchForMdxDevices: function(serviceType, headerPatterns, mx, numSsdpSearches) {
        nrdp.mdx._fn("SearchForMdxDevices",
                     {serviceType: encodeURIComponent(serviceType),
                      headerPatterns: nrdp.mdx._encodeArray(headerPatterns),
                      mx: mx,
                      numSsdpSearches: numSsdpSearches});
    },
    StopMdxDiscovery: function(cb) {
        nrdp.mdx._fn("StopMdxDiscovery", undefined, cb);
    },
    RevealTargetPresence: function(cb) {
        nrdp.mdx._fn("RevealTargetPresence", undefined, cb);
    },
    SetDeviceReplyHeaders: function(deviceReplyHeaders, cb) {
        nrdp.mdx._fn("SetDeviceReplyHeaders",
                     {deviceReplyHeaders: nrdp.mdx._encodeArray(deviceReplyHeaders)},
                    cb);
    },
    HideTargetPresence: function(cb) {
        nrdp.mdx._fn("HideTargetPresence", undefined, cb);
    },
    StartMdxAdvertising: function(cb) {
        nrdp.mdx._fn("StartMdxAdvertising", undefined, cb);
    },
    StopMdxAdvertising: function() {
        nrdp.mdx._fn("StopMdxAdvertising");
    },
    SendMdxHttpRequest: function(url, requestType, xid, curlTimeout, requestHeader, requestBody) {
        nrdp.mdx._fn("SendMdxHttpRequest",
                     {url: encodeURIComponent(url),
                      requestType: requestType,
                      xid: xid,
                      curltimeout: curlTimeout,
                      requestHeader: encodeURIComponent(requestHeader),
                      requestBody: encodeURIComponent(requestBody)});
    },
    SendSessionMessage: function(url, requestType, xid, curlTimeout, context, requestHeader, requestBody, message, plaintext) {
        nrdp.mdx._fn("SendSessionMessage",
                     {url: encodeURIComponent(url),
                      requestType: requestType,
                      xid: xid,
                      curltimeout: curlTimeout,
                      context: encodeURIComponent(context),
                      requestHeader: encodeURIComponent(requestHeader),
                      requestBody: encodeURIComponent(requestBody),
                      message: encodeURIComponent(message),
                      plaintext: encodeURIComponent(plaintext)});
    },
    SendWebSocketMessage: function(host, xid, body) {
        nrdp.mdx._fn("SendWebSocketMessage",
                     {host: encodeURIComponent(host),
                      xid : xid,
                      body: encodeURIComponent(body)});
    },
    ProcessSessionMessage: function(context, xid, message, messageHmac, ciphertext, cb) {
        nrdp.mdx._fn("ProcessSessionMessage",
                     { context: encodeURIComponent(context),
                       xid: xid,
                       message: encodeURIComponent(message),
                       messageHmac: encodeURIComponent(messageHmac),
                       ciphertext: encodeURIComponent(ciphertext)
                     },
                     cb);
    },
    DialGetDeviceInfo: function(url, USN, serviceType, timeout) {
        nrdp.mdx._fn("DialGetDeviceInfo",
                     {url: encodeURIComponent(url),
                      USN: encodeURIComponent(USN),
                      serviceType: encodeURIComponent(serviceType),
                      timeout: timeout});
    },
    beginContext: function(sharedSecret, context, cb) {
        // Note: the parameter "context" is only included in order to make the
        // function signature for nrdp.mdx.beginContext identical to the function
        // signature for nrdp.ntba.beginCustomContext, it is unused (it is unused
        // in the ntba version as well, but we don't bother to even pass it here)
        nrdp.mdx._fn("beginContext",
                     {sharedSecret: sharedSecret},
                     cb);
    },
    decrypt: function(context, data, cb) {
        nrdp.mdx._fn("decrypt",
                     { context: context,
                       data: data },
                     cb);
    },
    encrypt: function(context, data, cb) {
        nrdp.mdx._fn("encrypt",
                     { context: context,
                       data: data },
                     cb);
    },
    endContext: function(context, cb) {
        nrdp.mdx._fn("endContext",
                     {context: context},
                     cb);
    },
    hmac: function(context, data, cb) {
        nrdp.mdx._fn("hmac",
                     { context: context,
                       data: data },
                     cb);
    },
    _nextIdx: 1,
    _cbs: [],
    _fn: function(name, params, cb) {
        if (!this._syncData)
            return undefined;
        if (typeof cb === "function") {
            if (!params) params = {};
            params.idx = this._nextIdx++;
            this._cbs[params.idx] = cb;
        }
        nrdp._invoke("mdx", name, params);
        return params ? params.idx : undefined;
    },
    _isEncodedString: function(eventkey) {
        for ( var index in nrdp.mdx._encodedStringList) {
            if (eventkey == nrdp.mdx._encodedStringList[index])
                return true;
        }
        return false;
    },
    _decodeEventData: function(data, key) {
        var list, encodedList, item, compareKey, value;
        // set here, so if not found in any lists, the value is returned as-is
        value = data[key];
        if(typeof value != null) {
            for (list in nrdp.mdx._encodedList) {
                if(nrdp.mdx._encodedList.hasOwnProperty(list)) {
                    encodedList = nrdp.mdx._encodedList[list];
                    for (item in encodedList.list) {
                        if(encodedList.list.hasOwnProperty(item)) {
                            compareKey = encodedList.list[item];
                            if (compareKey === key) {
                                return encodedList.decodeFn(value);
                            }
                        }
                    }
                }
            }
        }
        return value;
    },
    _handleEvent: function(event) {
        var mydata = {};
        for (var key in event.data) {
            mydata[key] = nrdp.mdx._decodeEventData(event.data, key);
        }
        if (event.name === "returnValue") {
            if(!event.data || !event.data.idx) {
                return false;
            }
            if (typeof this._cbs[event.data.idx] === "function") {
                var cb = this._cbs[event.data.idx];
                delete this._cbs[event.data.idx];
                cb(mydata.data, (mydata.data ? mydata.data.idx : undefined));
            }
            return true;
        } else {
            nrdp._callEventListeners(this, mydata);
            return true;
        }
    },
    _encodeArray: function(arr) {
        var encodedArr = [],
        i;
        for(i in arr) {
            if(arr.hasOwnProperty(i)) {
                encodedArr.push(encodeURIComponent(arr[i]));
            }
        }
        return encodedArr;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp, nrdpPartner, nrdp_platform, NTRACE */
nrdp.system = {
    get bootURL() { return this._syncData.bootUrl; },
    get bootURLTime() { return this._syncData.bootUrlTime; },
    get uiQueryString() { return this._syncData.uiQueryString; },
    set uiQueryString(query) { nrdp._setProperty("system", "uiQueryString", query); },
    _handleEvent: function(event) {
        if (event.name == "keypress") {
            var keyCode = -1;
            for (var key in nrdpPartner.Keys) {
                if (nrdpPartner.Keys[key] == event.data) {
                    keyCode = key;
                    break;
                }
            }
            if (keyCode == -1) {
                nrdp.log.warn("Unknown key: " + event.data);
                return false;
            } else {
                ;
                if (window.nrdp_platform != null && window.nrdp_platform.sendKey != null) {
                    nrdp_platform.sendKey(keyCode);
                } else {
                    var downEvent = document.createEvent("Event");
                    downEvent.initEvent("keydown", true, true);
                    downEvent.which = keyCode;
                    document.dispatchEvent(downEvent);
                    var upEvent = document.createEvent("Event");
                    upEvent.initEvent("keyup", true, true);
                    upEvent.which = keyCode;
                    document.dispatchEvent(upEvent);
                }
            }
            return true;
        } else {
            return false;
        }
    },
    _syncData: {}
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
nrdp.network = {
    _path: "network",
    addEventListener: function(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    _urlIndex:0,
    _dnsIndex:0,
    _urlList: [],
    _datas: {},
    _errorStack: [],
    _googleDNS: "8.8.8.8",
    _netflixDomain : "netflix.com",
    _isDiagnoseRunning: false,
    _abortDiagnosis: false,
    _abortDiagnosisCb:0,
    CONNTIMEOUT: 6000,
    DATATIMEOUT: 4000,
    CONNECTIONSTATE : {
        CONNECTED : 0,
        DISCONNECTED : 1,
        UNKNOWN : 2
    },
    get ERRORGROUP() { return this._syncData.ERRORGROUP; },
    get ERRORCODE() { return this._syncData.ERRORCODE; },
    RECOMMENDEDACTION : {
        RETRY : 0,
        RESTART : 1,
        REACTIVATE : 2,
        MODIFY_DNS : 3,
        REDIRECT_SETTINGS : 4,
        REINSTALL : 5,
        RUNDIAGTOOL : 6,
        LOG_ERROR : 7
    },
    DNSERRORS : {
        SUCCESS : 0,
        NODATA : 1,
        EFORMERR : 2,
        SERVFAIL : 3,
        NOTFOUND : 4,
        NOTIMP : 5,
        REFUSED : 6,
        BADQUERY : 7,
        BADNAME : 8,
        BADFAMILY : 9,
        BADRESP : 10,
        CONNREFUSED : 11,
        TIMEOUT : 12,
        EOF : 13,
        EFILE : 14,
        NOMEM : 15,
        NODNSSERVER : 16,
        OTHER : 17
    },
    _sendCompletedEvent: function(ra) {
        this._urlIndex = 0;
        this._dnsIndex = 0;
        this._urlList = [];
        var endEvent = {
            type: "diagnosisResult",
            resultArray: this._errorStack,
            recommendedAction: ra
        };
        this._errorStack = [];
        this._isDiagnoseRunning = false;
        nrdp._callEventListeners(this, endEvent);
    },
    _sendAbortCb: function() {
        this._urlIndex = 0;
        this._dnsIndex = 0;
        this._urlList = [];
        this._errorStack = [];
        this._isDiagnoseRunning = false;
        this._abortDiagnosis = false;
        if (nrdp.network._abortDiagnosisCb) {
            nrdp.network._abortDiagnosisCb(true);
        }
    },
    _getDomain: function(url) {
        var domain = url.match(/:\/\/(www[0-9]?\.)?(.[^/:]+)/);
        if (domain) {
            return domain[2];
        }
        return "";
    },
    _dnsCallback: function (data, dns) {
        if (this._abortDiagnosis) {
            this._sendAbortCb();
            return;
       }
        var event = {
            URL: data.url,
            errorgroup: this.ERRORGROUP.DNS_CHECK,
            errorcode: (data.result > this.DNSERRORS.NOMEM) ?
                 this.DNSERRORS.OTHER :
                data.result,
            // recommendedAction: data.result,
            dnsip: dns,
            type: "diagnosisStatus",
            success: !data.result
        };
        nrdp._callEventListeners(this, event);
        this._errorStack.push(event);
        if ((dns == this._googleDNS) && (nrdp.device.dnslist.length < this._dnsIndex)) {
            var myurl = nrdp.options.appboot_url;
            this._dnsIndex = 0;
            this._continueDiag(myurl);
            return;
        }
        if (nrdp.device.dnslist.length > this._dnsIndex) {
            var nextdns = nrdp.device.dnslist[this._dnsIndex++];
            this.checkDNS(nextdns, event.URL, function(evt) {
                nrdp.network._dnsCallback(evt, nextdns);
            });
        } else {
            // now check google DNS
            this._dnsIndex++;
            this.checkDNS(this._googleDNS, event.URL, function(evt) {
                nrdp.network._dnsCallback(evt, nrdp.network._googleDNS);
            });
        }
    },
    _continueDiag: function(url) {
       if (this._abortDiagnosis) {
            this._sendAbortCb();
            return;
       }
        if (this._urlList.length > this._urlIndex) {
            this.getOperation(this._urlList[this._urlIndex++],
                              this.CONNTIMEOUT,
                              this.DATATIMEOUT,
                              function(evt) {
                                  nrdp.network._getCallback(evt);
                              });
        } else {
            // completed. send complete event
            var reco = this.RECOMMENDEDACTION.RETRY;
            for (var index in this._errorStack) {
                if ( (this._errorStack[index].errorcode == this.ERRORCODE.DNS_ERROR) && (reco == this.RECOMMENDEDACTION.RETRY) ){
                    reco = this.RECOMMENDEDACTION.REDIRECT_SETTINGS;
                }else if ( (this._errorStack[index].errorcode == this.ERRORCODE.CERT_STATUS_PEW_REVOKED) && (reco == this.RECOMMENDEDACTION.RETRY) ){
                    reco = this.RECOMMENDEDACTION.REACTIVATE;
                }
            }
            this._sendCompletedEvent(reco);
        }
    },
    _nflxUrlCallback: function(data, url) {
        if (!this._isDiagnoseRunning) {
            return;
        }
        if (this._abortDiagnosis) {
            this._sendAbortCb();
            return;
        }
        var event = {
            URL: data.url,
            errorgroup: data.errorgroup,
            errorcode: data.errorcode,
            nativeerrorcode: data.nativeerrorcode,
            // recommendedAction: data.result,
            type: "diagnosisStatus",
            success: !data.result
        };
        if (event.errorcode == this.ERRORCODE.DNS_ERROR) {
            //do dnscheck;
            if (nrdp.device.dnslist.length <= 0) {
                event.errorcode = this.ERRORCODE.NO_DNS_SERVER;
                nrdp._callEventListeners(this, event);
                this._errorStack.push(event);
                this._sendCompletedEvent(this.RECOMMENDEDACTION.REDIRECT_SETTINGS);
            } else {
                nrdp._callEventListeners(this, event);
                this._errorStack.push(event);
                var dns = nrdp.device.dnslist[this._dnsIndex++];
                // XXX any reason not to check using all DNS servers at same time?
                if (nrdp.network._getDomain(event.URL) != "") {
                    this.checkDNS(dns,
                                  nrdp.network._getDomain(event.URL),
                                  function(evt) {
                                      nrdp.network._dnsCallback(evt, dns);
                                  });
                } else {
                    this._continueDiag(url);
                }
            }
            return;
        }
        nrdp._callEventListeners(this, event);
        this._errorStack.push(event);
        this._continueDiag(url);
    },
    _getCallback: function(data) {
        if (!this._isDiagnoseRunning) {
            return;
        }
        if (this._abortDiagnosis) {
            this._sendAbortCb();
            return;
        }
        var event = {
            URL: data.url,
            errorgroup: data.errorgroup,
            errorcode: data.errorcode,
            nativeerrorcode: data.nativeerrorcode,
            // recommendedAction: data.result;
            type: "diagnosisStatus",
            success: !data.result
        };
        // XXX take same action if !event.success as _nflxUrlCallback?
        // recommendation to REACTIVATE if PEW revoked and recommendation to REDIRECT  for DNS error
        nrdp._callEventListeners(this, event);
        this._errorStack.push(event);
        if (this._urlList.length > this._urlIndex) {
            this.getOperation(this._urlList[this._urlIndex++],
                              this.CONNTIMEOUT,
                              this.DATATIMEOUT,
                              function(evt) {
                                  nrdp.network._getCallback(evt);
                              });
        } else {
            var reco = this.RECOMMENDEDACTION.RETRY;
            for (var index in this._errorStack) {
                if ( (this._errorStack[index].errorcode == this.ERRORCODE.DNS_ERROR) && (reco == this.RECOMMENDEDACTION.RETRY) ){
                    reco = this.RECOMMENDEDACTION.REDIRECT_SETTINGS;
                }else if ( (this._errorStack[index].errorcode == this.ERRORCODE.CERT_STATUS_PEW_REVOKED) && (reco == this.RECOMMENDEDACTION.RETRY) ){
                    reco = this.RECOMMENDEDACTION.REACTIVATE;
                }
            }
            this._sendCompletedEvent(reco);
        }
    },
    _isValidIpAvailable: function() {
        var ifList = nrdp.device.iflist;
        var found = false;
        for (var index in ifList) {
            if (ifList[index].ipAddress) {
                if ( (ifList[index].ipAddress.indexOf("127", 0) != 0) &&
                   (ifList[index].ipAddress != "0.0.0.0") &&
                   (ifList[index].ipAddress != "") &&
                   (ifList[index].ipAddress.indexOf("169.254", 0) != 0) ) {
                    found = true;
                    ;
                    break;
                }
            } else if (ifList[index].ipv6Addresses && ifList[index].ipv6Addresses.length > 0){
                    found = true;
                    ;
                    break;
            }
        }
        return found;
    },
    isValidIpAvailable: function() {
        return this._isValidIpAvailable();
    },
    diagnose: function(urls) {
        nrdp.network._datas = {};
        if (!nrdp.network._isValidIpAvailable()) {
            var event = {
                URL: "",
                errorgroup: nrdp.network.ERRORGROUP.NO_IP_ADDRESS,
                errorcode: nrdp.network.ERRORCODE.UNKNOWN_ERROR,
                nativeerrorcode: -1, // duplicated from NativeerrorCodes.h
                type: "diagnosisStatus",
                success: false
            };
            nrdp.log.debug("diagnosis status event " + JSON.stringify(event) );
            nrdp._callEventListeners(nrdp.network, event);
            nrdp.network._errorStack.push(event);
            nrdp.network._sendCompletedEvent(nrdp.network.RECOMMENDEDACTION.REDIRECT_SETTINGS);
            return;
        }
        if (!nrdp.network._abortDiagnosis) {
            nrdp.network._isDiagnoseRunning = true;
            // XXX need to handle multiple calls to diagnose in parallel?
            nrdp.network._urlList = urls;
            // XXX any reason not to check all urls at the same time?
            nrdp.network.getOperation(nrdp.options.appboot_url,
                                      nrdp.network.CONNTIMEOUT,
                                      nrdp.network.DATATIMEOUT,
                                      function(evt) {
                                          nrdp.network._nflxUrlCallback(evt, nrdp.options.appboot_url);
                                      });
        } else {
            nrdp.network._abortDiagnosis = false;
            nrdp.network._abortDiagnosisCb(true);
        }
    },
    abortDiagnosis: function(abortCb) {
        if (nrdp.network._isDiagnoseRunning) {
            nrdp.network._abortDiagnosis = true;
            nrdp.network._abortDiagnosisCb = abortCb;
        }
    },
    getOperation: function(url, connectiontimeout, datatimeout, callback) {
        // XXX what if there's two get operations for the same url at the same time?
        nrdp.network._datas[url] = {cb: callback, results: {}};
        nrdp._invoke("network", "get",
                     {url: url, connectiontimeout: connectiontimeout, datatimeout: datatimeout});
    },
    checkDNS: function(dnsip, url, callback) {
        // XXX what if there's two checks for the same url at the same time?
        nrdp.network._datas[url] = {cb: callback, results: {}};
        nrdp._invoke("network", "checkdns", {dnsip: dnsip, url: url});
    },
    useFallbackDnsOnly: function(fallback) {
        nrdp._invoke("network", "useFallbackDnsOnly", {fallback: fallback});
    },
    _handleEvent: function(event) {
        if (event.name != "INetwork") {
            return false;
        }
        if ( (event.data.type !== "dnsresult") && (event.data.type !== "getresult") ){
            return false;
        }
        var url = event.data.url;
        var data = this._datas[url];
        if (data) {
            data.cb(event.data);
        }
        return true;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp, nrdp_platform, nrdpPartner*/
nrdp.gibbon = {
    _callback: 1,
    _callbacks: {},
    _nextWidgetId: 1,
    _nextEffectId: 1,
    _nextImageId: 1,
    _nextChildId: 1,
    _nextAnimId: 1,
    _nextDebugEvent: 0,
    _debugKeyEvents: 0,
    _debugAnimations: 0,
    _sync_Widget: 1,
    _sync_Effect: 2,
    _sync_Image: 3,
    _sync_Text: 4,
    INT_MAX: 0x7fffffff,
    init: function init(cb, url) {
        function makeArray(elt) {
            return Array.isArray(elt) ? elt : [elt];
        }
        var urlList = url ? makeArray(url) : [];
        function load_urls() {
            if (urlList.length == 0) {
                if (cb) cb();
            } else {
                var u = urlList.shift();
                nrdp.gibbon.loadScript({ url: u }, load_urls);
            }
        }
        if (nrdp.isReady) {
            load_urls();
        } else {
            nrdp.addEventListener("init", load_urls);
            nrdp.init();
        }
    },
    _proxyObject: function _proxyObject(object, property) {
        var proxy = {};
        //proxy._gcTag = nrdp_platform.gctag("Gibbon::ProxyObject::" + property);
        function createProxyValue(valueName, value) {
            //nrdp.log.error(" + Creating: " + valueName + " = " + JSON.stringify(value));
            Object.defineProperty(proxy, valueName, { enumerable: true, get: function() { return value; },
                                                      set: function(v) {
                                                          //nrdp.log.error("Setting: " + object.classname + "(" + object._path + ")::" + property + "::" + valueName + " = " + JSON.stringify(value));
                                                          value = v; object[property] = proxy;
                                                      } });
        }
        var defaults = object._defaults[property];
        var pulled = object._syncData.hasOwnProperty(property) ? object._syncData[property] : defaults;
        if(!(pulled instanceof Object))
            return pulled;
        var values = undefined;
        if(defaults)
            values = Object.keys(defaults);
        else
            values = Object.keys(pulled);
        if(values) {
            //nrdp.log.error("Proxy : " + object.classname + "(" + object._path + ")::" + property + " = " + JSON.stringify(values));
            for(var i = 0; i < values.length; ++i) {
                var valueName = values[i];
                var value = pulled[valueName];
                if(value && valueName.toLowerCase().indexOf("color") != -1)
                    value = { r: value.r, g: value.g, b: value.b, a: (value.a === undefined ? 255 : value.a) };
                createProxyValue(valueName, value);
            }
        }
        return proxy;
    },
    // *** DEPRECATED
    get CURL_INIT_ERROR() { return -1; },
    // Debugger
    get DEBUGGER_ATTRIBUTE_MODIFIED() { return 1; },
    get DEBUGGER_SUBTREE_MODIFIED() { return 2; },
    get DEBUGGER_NODE_REMOVED() { return 3; },
    fonts: {
        get entries() { return nrdp.gibbon._syncData.font_entries; },
        set entries(f) { nrdp._setProperty("gibbon", "fonts", f); },
        get scripts() { return nrdp.gibbon._syncData.font_scripts; },
        get storeCapacity() { return nrdp.gibbon._syncData.fontStoreCapacity; },
        attributes: {
            get NONE() { return 0x00000000; },
            get EMBOLDEN() { return 0x00000001; },
            get SLANT() { return 0x00000002; },
            get MONOSPACE() { return 0x00000004; },
            get SYNTHESIZE() { return 0x00000008; },
            get HINTING() { return 0x00000100; },
            get AUTOHINTER() { return 0x00000200; },
            get HINTNORMAL() { return 0x00000400; },
            get HINTLIGHT() { return 0x00000800; },
            get HINTMONO() { return 0x00001000; },
            get HINTLCD() { return 0x00002000; },
            get HINTLCDV() { return 0x00004000; }
        },
        weight: {
            get NORMAL() { return 0; },
            get BOLD() { return 1; }
        },
        style: {
            get NORMAL() { return 0; },
            get ITALIC() { return 1; }
        },
        info: {
            get NORMAL() { return 0; },
            get CACHE() { return 1; },
            get LRU() { return 2; },
            types: {
                get Url() { return 1; },
                get Buffer() { return 2; },
                get Mmap() { return 3; }
            },
            get: function get(mode, cb) {
                var id = nrdp.gibbon._setValue(cb);
                if (mode === undefined)
                   mode = nrdp.gibbon.fonts.info.NORMAL;
                nrdp._invoke("gibbon", "fontManagerInfo", {mode: mode, id: id});
            }
        },
        _initFont: function _initFont(info, fileName) {
            if (!nrdp.gibbon._syncData.downloadableFonts)
                nrdp.gibbon._syncData.downloadableFonts = {};
            var key = nrdp.gibbon.fonts._makeKey(info.params);
            nrdp.gibbon._syncData.downloadableFonts[key] = info;
            var priority = info.params && info.params instanceof Object && info.params.priority;
            if (priority === undefined)
                priority = 100;
            nrdp._invoke("gibbon", "loadFont", { key: key, priority: priority, font: info.params, fileName: fileName });
        },
        init: function init() {
            if (nrdp.gibbon.fonts._inited)
                return;
            nrdp.gibbon.fonts._inited = true;
            var store = nrdp.gibbon.fonts._store;
            if (!store)
                return;
            store.query(nrdp.storage.NO_DEVICE_ACCOUNT, "", function(q) {
                var keys = Object.keys(q);
                keys.forEach(function(k) {
                    // verify that we have a _info for every non-info and the other way around
                    var hasOpposite = function(key, obj, info) {
                        if (info)
                            return obj.hasOwnProperty(key.substr(0, key.length - 5));
                        return obj.hasOwnProperty(key + "_info");
                    };
                    if (k.indexOf("_info", k.length - 5) !== -1) {
                        if (!hasOpposite(k, q, true)) {
                            store.remove(nrdp.storage.NO_DEVICE_ACCOUNT, k);
                            return;
                        }
                        store.read(nrdp.storage.NO_DEVICE_ACCOUNT, k, 0, -1, function(v) {
                            if (v.success) {
                                try {
                                    var data = nrdp.utf8toa(v.value);
                                    var json = JSON.parse(data);
                                } catch (e) {
                                    nrdp.log.error("Unable to JSON parse font data from " + k);
                                    return;
                                }
                                // now query the ttf filename
                                var url = k.substr(0, k.length - 5);
                                store.query(nrdp.storage.NO_DEVICE_ACCOUNT, url, function(f) {
                                    if (f.hasOwnProperty(url)) {
                                        nrdp.gibbon.fonts._initFont(json, f[url].fileName);
                                    } else {
                                        nrdp.log.error("Unable to query font store for font " + url);
                                    }
                                }, true /* validate */);
                            } else {
                                nrdp.log.error("Unable to read info for " + k);
                            }
                        });
                    } else {
                        if (!hasOpposite(k, q, false)) {
                            store.remove(nrdp.storage.NO_DEVICE_ACCOUNT, k);
                        }
                    }
                });
            });
        },
        get _store() {
            if (!nrdp.gibbon.fonts._storeContext || !nrdp.gibbon.fonts._storeContext.valid) {
                if (!nrdp.gibbon.fonts.storeCapacity || nrdp.gibbon.fonts.storeCapacity < 0)
                    return undefined;
                nrdp.gibbon.fonts._storeContext = (nrdp.storage.diskStoreContexts["font-store"] ||
                                                   nrdp.storage.createDiskStoreContext({ context: "font-store",
                                                                                         size: nrdp.gibbon.fonts.storeCapacity,
                                                                                         encrypted: false,
                                                                                         signature: true }));
            }
            return nrdp.gibbon.fonts._storeContext;
        },
        _makeKey: function _makeKey(req) {
            if (!req instanceof Object)
                return undefined;
            if (!req.hasOwnProperty("family"))
                return undefined;
            if (!req.hasOwnProperty("style"))
                return undefined;
            if (!req.hasOwnProperty("weight"))
                return undefined;
            return req.family + "-" + req.style + "-" + req.weight;
    },
        _storeContext: undefined,
        _inited: false
    },
    get email() {
        return this._syncData.email;
    },
    diskCache: {
        get capacity() {
            if(nrdp.gibbon._syncData.diskCacheConfiguration)
                return nrdp.gibbon._syncData.diskCacheConfiguration.capacity;
            return 0;
        },
        get maxPending() {
            if (nrdp.gibbon._syncData.diskCacheConfiguration)
                return nrdp.gibbon._syncData.diskCacheConfiguration.maxPending;
            return 0;
        },
        get writeLimit() {
            if(nrdp.gibbon._syncData.diskCacheConfiguration)
                return nrdp.gibbon._syncData.diskCacheConfiguration.writeLimit;
            return 0;
        },
        set writeLimit(limit) {
            nrdp._invoke("gibbon", "diskCacheSetWriteLimit", { limit: limit });
        },
        get catalogTimer() {
            if(nrdp.gibbon._syncData.diskCacheConfiguration)
                return nrdp.gibbon._syncData.diskCacheConfiguration.catalogTimer;
            return 0;
        },
        get writeSpeed() { // bytes/sec
            if(nrdp.gibbon._syncData.diskCacheConfiguration)
                return nrdp.gibbon._syncData.diskCacheConfiguration.writeSpeed;
            return 0;
        },
        insert: function insert(obj, cb) {
            obj.id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "diskCacheInsert", obj);
        },
        clear: function clear(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "clearDiskCache", {id: id});
        },
        clearStats: function clear(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "clearDiskCacheStats", {id: id});
        },
        flush: function flush(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "flushDiskCache", {id: id});
        },
        reinit: function reinit(spec, cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "reinitDiskCache", {spec: spec, id: id});
        },
        info: function info(args, cb) {
            if (!args)
                args = {};
            args.id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "diskCacheInfo", args);
        },
        remove: function(cacheKey, cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "diskCacheRemove", {cacheKey: cacheKey, id: id});
        },
        purgeExpired: function(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "diskCachePurgeExpired", {id: id});
        },
        _dump: function _dump(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "dumpDiskCache", {id: id});
        }
    },
    surfaceCache: {
        get capacity() {
            if (nrdp.gibbon._syncData.surfaceCacheCapacity)
                return nrdp.gibbon._syncData.surfaceCacheCapacity.browse;
            return undefined;
        },
        set capacity(cap) {
            nrdp._invoke("gibbon", "surfaceCacheSetBrowseCapacity", { capacity: cap });
        },
        get playbackCapacity() {
            if (nrdp.gibbon._syncData.surfaceCacheCapacity)
                return nrdp.gibbon._syncData.surfaceCacheCapacity.playback;
            return undefined;
        },
        set playbackCapacity(cap) {
            nrdp._invoke("gibbon", "surfaceCacheSetPlaybackCapacity", { capacity: cap });
        },
        info: function(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "surfaceCacheInfo", {id: id});
        }
    },
    resourceManager: {
        get capacity() {
            return nrdp.gibbon._syncData.resourceManagerCapacity;
        },
        set capacity(cap) {
            nrdp._invoke("gibbon", "resourceManagerSetCapacity", { capacity: cap });
        },
        info: function(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "resourceManagerInfo", {id: id});
        },
        disable: function(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "resourceManagerDisable", {id: id});
        },
        remove: function(cacheKey, cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "resourceManagerRemove", {id: id, cacheKey: cacheKey});
        },
        clear: function(cb) {
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "resourceManagerClear", {id: id});
        }
    },
    get debugFlags() {
        return this._syncData.debugFlags;
    },
    get password() {
        return this._syncData.password;
    },
    get imageFormats() {
        return this._syncData.imageFormats;
    },
    get defaultLocale() {
        return this._syncData.defaultLocale;
    },
    localeInfo: function localeInfo(l) {
        return nrdp._invoke("gibbon", "localeInfo", { locale: l });
    },
    set defaultLocale(l) {
        nrdp._setProperty("gibbon", "defaultLocale", l);
    },
    get defaultDirection() {
        return this._syncData.defaultDirection;
    },
    set defaultDirection(d) {
        nrdp._setProperty("gibbon", "defaultDirection", d);
    },
    get garbageCollectTimeout() {
        return this._syncData.garbageCollectTimeout;
    },
    set garbageCollectTimeout(t) {
        nrdp._setProperty("gibbon", "garbageCollectTimeout", t);
    },
    get prefetchUrls() {
        return this._syncData.prefetchUrls;
    },
    get prefetchKey() {
        return this._syncData.prefetchKey;
    },
    get effectivePrefetchUrls() {
        return this._syncData.effectivePrefetchUrls;
    },
    set prefetchUrls(urls) {
        if (urls && !(urls instanceof Object)) {
            nrdp.log.error("Invalid prefetch urls");
            return;
        }
        this._syncData.prefetchUrls = urls;
        nrdp._setProperty("gibbon", "prefetchUrls", urls);
    },
    _path: "gibbon",
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get effects() { return this.capabilities.effects; },
    get capabilities() { return nrdp.capabilities.gibbon; },
    setStat: function setStat(name, value) { nrdp._invoke("gibbon", "setStat", {name: name, value: value}); },
    get cookie() { return (this._syncData) ? this._syncData.cookie: ""; },
    set cookie(c) { nrdp._invoke("gibbon", "setCookie", {cookie: c}); },
    setCookie: function setCookie(cookie, callback) {
        var id = nrdp.gibbon._setValue(callback);
        nrdp._invoke("gibbon", "setCookie", {cookie: cookie, id: id});
    },
    get cookieFilters() {
        return nrdp.gibbon._syncData.cookieFilters;
    },
    set cookieFilters(filters) {
        if (!filters)
            filters = [];
        if (!(filters instanceof Array)) {
            nrdp.log.error("Invalid cookie filters");
            return;
        }
        nrdp.gibbon._syncData.cookieFilters = filters;
        nrdp._setProperty("gibbon", "cookieFilters", filters);
    },
    _createAnimation: function _createAnimation(property, start, end, duration, ease, append, flags) {
        if(append === undefined)
            append = false;
        if(start instanceof Object && start.relative)
            start.relative = start.relative._id;
        var value = { value: end,
                      animate: { id: nrdp.gibbon._nextAnimId++, start: start, end: end,
                                 duration: duration, ease: ease, append: append, flags: flags } };
        return value;
    },
    _loadSplash: function _loadSplash(request) {
        if(!(/^https?:/).test(nrdp.options.splash_screen) || nrdp.gibbon._splashUrl)
            return;
        var defaultSplash = nrdp.options.splash_screen;
        if(request && request.url && request.url.length) {
            var transientData = nrdp.storage.transientData || {};
            transientData.splash = request;
            nrdp.storage.transientData = transientData;
        } else {
            if(nrdp.storage.transientData)
                request = nrdp.storage.transientData.splash;
            if(!request || !request.url || !request.url.length)
                request = { url: defaultSplash };
        }
        nrdp.gibbon._splashUrl = request.url;
        nrdp.gibbon.loadScript(request, function(response) {
            if (response.exception || response.statusCode != 200) {
                nrdp.log.info("falling back to splash screen " + defaultSplash);
                nrdp.gibbon._splashUrl = defaultSplash;
                nrdp.gibbon.loadScript({ 'url': defaultSplash });
            } else {
                nrdp.gibbon.sync();
            }
        });
    },
    get location() {
        var result = this._baseUrl;
        if(!result) {
            // there is a race condition in c++ when the location is the
            // result of a redirect; the updated location may not come
            // until after nrdp isReady. but nrdp_platform can always be
            // right.
            if(typeof nrdp_platform !== "undefined" && typeof nrdp_platform.location === "function")
                result = nrdp_platform.location();
            else
                result = this._syncData.location;
            if(typeof result == 'object')
                result = result.url;
        }
        return result;
    },
    set location(l) {
        var args = l;
        this._baseUrl = undefined;
        if(typeof args == 'string')
            args = {url: args};
        args.url = this._resolveUrl(args.url);
        nrdp._setProperty("gibbon", "location", args);
    },
    get _locationCount() {
        var result;
        if(typeof nrdp_platform !== "undefined" && typeof nrdp_platform.location === "function")
            result = nrdp_platform.location();
        else
            result = this._syncData.location;
        if(typeof result == 'object')
            return result.count;
        return 0;
    },
    get queryParams() {
        if(!this._syncData['queryParams'])
            this._syncData['queryParams'] = nrdp.gibbon._parseQueryParams(this.location);
        return this._syncData['queryParams'];
    },
    loaded: function loaded() {
        nrdp._invoke("gibbon", "uiLoaded");
    },
    startTask: function startTask(cb, priority) {
        var id = nrdp.gibbon._setValue({ callback: cb });
        nrdp._invoke("gibbon", "startTask", { id: id, priority: priority });
        if(nrdp.gibbon.debugFlags.debugScriptEvents)
            nrdp.log.error("StartTask " + id);
        return id;
    },
    stopTask: function stopTask(id) {
        if(nrdp.gibbon._getValue(id) !== undefined) {
            if(nrdp.gibbon.debugFlags.debugScriptEvents)
                nrdp.log.error("StopTask " + id);
            nrdp.gibbon._deleteValue(id);
            nrdp._invoke("gibbon", "stopTask", { id: id });
        }
    },
    setTimeout: function setTimeout(cb, interval, singleShot) {
        if(singleShot === undefined) // default singleShot to true
            singleShot = true;
        var id = nrdp.gibbon._setValue({ callback: cb, singleShot: singleShot, interval: interval });
        nrdp._invoke("gibbon", "startTimer", { id: id, singleShot: singleShot, interval: interval });
        if (nrdp.gibbon.debugFlags.logZeroTimeouts && (interval === 0))
        {
            nrdp.log.info("Created[" + id + "]: " + singleShot + "::" + interval,
                          "TIMER");
            var st = nrdp.stacktrace();
            var a = st.split('\n');
            a.forEach(function(m) { nrdp.log.info(m, "TIMER"); });
        }
        if(nrdp.gibbon.debugFlags.debugScriptEvents)
            nrdp.log.error("StartTimer " + id + "@" + interval + ":" + singleShot);
        return id;
    },
    clearTimeout: function clearTimeout(id) {
        if(nrdp.gibbon._getValue(id) !== undefined) {
            if(nrdp.gibbon.debugFlags.debugScriptEvents)
                nrdp.log.error("StopTimer " + id);
            nrdp.gibbon._deleteValue(id);
            nrdp._invoke("gibbon", "stopTimer", { id: id });
        }
    },
    garbageCollect: function garbageCollect(callback, type) {
        var id = callback ? nrdp.gibbon._setValue(callback) : 0;
        nrdp._invoke("gibbon", "garbageCollect", { id: id, type: type });
    },
    eval: function eval(script, file) {
        return nrdp._invoke("gibbon", "eval", { script: script, file: file });
    },
    _load: function _load(request, callback) {
        request.id = nrdp.gibbon._setValue(callback);
        request.url = this._resolveUrl(request.url);
        nrdp._invoke("gibbon", "startRequest", request);
        if(nrdp.gibbon.debugFlags.debugScriptEvents)
            nrdp.log.error("StartRequest " + request.id + "@" + request.url);
        return request.id;
    },
    load: function load(request, callback) {
        if(typeof request !== "object") {
            if(typeof callback === "function")
                callback({});
            return -1;
        }
        request.eval = false;
        if(request.async === undefined)
            request.async = true;
        return nrdp.gibbon._load(request, callback);
    },
    loadScript: function loadScript(request, callback) {
        if(typeof request !== "object") {
            if(typeof callback === "function")
                callback({});
            return -1;
        }
        request.eval = true;
        if(request.async === undefined)
            request.async = false;
        return nrdp.gibbon._load(request, callback);
    },
    stopLoad: function stopLoad(id) {
        nrdp.gibbon._deleteValue(id);
        if(nrdp.gibbon.debugFlags.debugScriptEvents)
            nrdp.log.error("StopRequest " + id);
        nrdp._invoke("gibbon", "stopRequest", {id: id });
        return id;
    },
    addInjectJS: function addInjectJS(js, options) {
        nrdp._invoke("gibbon", "addInjectJS", { js: js, options: options });
    },
    removeInjectJS: function addInjectJS(url) {
        nrdp._invoke("gibbon", "removeInjectJS", { url: url });
    },
    setAtlasSize: function setAtlasSize(group, width, height) {
        nrdp._invoke("gibbon", "setAtlasSize", { group: group, width: width, height: height });
    },
    setGlyphAtlasSize: function setGlyphAtlasSize(width, height) {
        nrdp._invoke("gibbon", "setGlyphAtlasSize", { width: width, height: height });
    },
    sync: function sync() {
        nrdp._invoke("gibbon", "sync");
    },
    beginPendingSync: function beginPendingSync() {
        nrdp._invoke("gibbon", "beginPendingSync");
    },
    endPendingSync: function endPendingSync() {
        nrdp._invoke("gibbon", "endPendingSync");
    },
    _parseQueryParams: function _parseQueryParams(url)
    {
        var result = {};
        if(url) {
            var question = url.indexOf('?');
            if (question > 0) {
                url.substr(question+1).split('&').forEach(
                    function(e) {
                        if (e) {
                            var opt = e.split('=');
                            result[opt[0]] = (opt.length > 1 ? decodeURIComponent(opt[1]) : true);
                        }
                    });
            }
        }
        return result;
    },
    _compareColor: function _compareColor(color1, color2)
    {
        if(color1 instanceof Object && color2 instanceof Object)
            return (color1.r == color2.r && color1.g == color2.g &&
                    color1.b == color2.b && color1.a == color2.a);
        return color1 == color2;
    },
    _encodeColor: function _encodeColor(color)
    {
        if(color instanceof Object) {
            var result = 0;
            if(color.r)
                result |= (color.r & 0xFF) << 16;
            if(color.g)
                result |= (color.g & 0xFF) << 8;
            if(color.b)
                result |= (color.b & 0xFF) << 0;
            if(color.a === undefined)
                result |= 0xFF << 24;
            else if(color.a)
                result |= (color.a & 0xFF) << 24;
            return result;
        }
        return color;
    },
    _needPendingSync: function _needPendingSync() {
        return nrdp.gibbon._pendingSync;
    },
    _addDOMBreakpoint: function _addDOMBreakpoint(widgetId, type) {
        if (nrdp.gibbon._breaks === undefined)
            nrdp.gibbon._breaks = {};
        if (nrdp.gibbon._breaks[widgetId] == undefined)
            nrdp.gibbon._breaks[widgetId] = {};
        nrdp.gibbon._breaks[widgetId][type] = function() {
            nrdp.log.error("calling debugger!");
            debugger;
        };
    },
    _removeDOMBreakpoint: function _removeDOMBreakpoint(widgetId, type) {
        var hasAnyProperties = function(obj) {
            for (var p in obj) {
                if (obj[p] !== undefined)
                    return true;
            }
            return false;
        };
        delete nrdp.gibbon._breaks[widgetId][type];
        if (!hasAnyProperties(nrdp.gibbon._breaks[widgetId]))
            delete nrdp.gibbon._breaks[widgetId];
        if (!hasAnyProperties(nrdp.gibbon._breaks))
            nrdp.gibbon._breaks = undefined;
    },
    _debugObject: function _debugObject(obj, objname, num) {
        function getDescription(o) {
            if(typeof o === "object") {
                if(o instanceof Array)
                    return "Array[" + o.length + "]";
                else if(o instanceof Date)
                    return "Date";
                else if(o instanceof RegExp)
                    return "RegExp";
                else
                    return "Object";
            } else {
                return o + "";
            }
        }
        if(typeof obj === "object") {
            var arr = false;
            if(obj instanceof Array) {
                arr = true;
            }
            var list = [];
            var tp, stp, isobj, name;
            for (var i in obj) {
                tp = typeof obj[i];
                stp = undefined;
                if(tp === "object") {
                    if(obj[i] instanceof Array) {
                        stp = "array";
                    } else if(obj[i] instanceof Date)
                        stp = "date";
                    else if(obj[i] instanceof RegExp)
                        stp = "regexp";
                    isobj = true;
                } else {
                    isobj = false;
                }
                if(arr)
                    name = objname + "[" + i + "]";
                else
                    name = objname + "." + i;
                list.push({ name: i, value: { objectId: (isobj ? (name + ":" + num) : undefined),
                                              type: tp, subtype: stp,
                                              description: getDescription(obj[i])
                                            }
                          });
            }
            return list;
        } else {
            //return [{ value: { type: typeof obj, description: obj + "" }}];
        }
        return undefined;
    },
    get widgets() {
        var result = {};
        function recurse(widget) {
            if(widget) {
                result[widget._name] = widget;
                for(var i = 0; i < widget.children.length; ++i)
                    recurse(widget.children[i]);
            }
        }
        recurse(nrdp.gibbon.scene.root);
        recurse(nrdp.gibbon.scene.overlay);
        return result;
    },
    findWidget: function findWidget(id) {
        var widget;
        if(nrdp.gibbon.scene.root) {
            widget = nrdp.gibbon.scene.root.findWidget(id);
            if(widget)
                return widget;
        }
        if(nrdp.gibbon.scene.overlay) {
            widget = nrdp.gibbon.scene.overlay.findWidget(id);
            if(widget)
                return widget;
        }
        return undefined;
    },
    makeWidget: function makeWidget(obj) {
        var widgetid = nrdp.gibbon._nextWidgetId++;
        if(widgetid > nrdp.gibbon.INT_MAX)
            widgetid = 1;
        var widget = new nrdp.gibbon.Widget(widgetid);
        if(obj) {
            for(var property in obj) {
                if(property == "children") {
                    for(var childobj in obj.children) {
                        var childwidget = nrdp.gibbon.makeWidget(obj.children[childobj]);
                        childwidget.parent = widget;
                    }
                } else if(property == "animations") {
                    for(var aproperty in obj.animations)
                        widget.animate(obj.animations[aproperty].property, obj.animations[aproperty].duration,
                                       obj.animations[aproperty].ease);
                } else if(property == "effects") {
                    for(var eproperty in obj.effects)
                        widget.addEffect(obj.effects[eproperty].type, obj.effects[eproperty].params);
                } else if(property == "images") {
                    for(var iproperty in obj.images)
                        widget.addImage(obj.images[iproperty].background, obj.images[iproperty]);
                } else if(property == "parent" || property == "clone") {
                    widget[property] = nrdp.gibbon.findWidget(obj[property]);
                } else if(property == "image" || property == "backgroundImage") {
                    widget.addImage(property == "backgroundImage", obj[property]);
                } else {
                    widget[property] = obj[property];
                }
            }
        }
        return widget;
     },
    dump: function dump(cb) { nrdp.gibbon.scene.dump(cb); },
    _hookProperty : function _hookProperty(obj, property, hook) {
        var _hook_old = Object.getOwnPropertyDescriptor(obj, property);
        var _hook_new = Object.create(_hook_old);
        _hook_new.set = function(_value) {
            hook.call(this, _value, _hook_old.set);
        };
        Object.defineProperty(obj, property, _hook_new);
    },
    startFpsTimer: function startFpsTimer() { nrdp._invoke("gibbon", "startFpsTimer"); },
    stopFpsTimer: function stopFpsTimer() { nrdp._invoke("gibbon", "stopFpsTimer"); },
    getRenderedFps: function getRenderedFps(callback) {
        var id = nrdp.gibbon._setValue(callback);
        nrdp._invoke("gibbon", "getRenderedFps", {id: id});
    },
    getHeapSize: function getHeapSize(callback) {
        if (nrdp.ps3 && nrdp.ps3.getMemoryStats) {
            // grab the PS3 info
            nrdp.ps3.getMemoryStats(function(ps3data) {
                var id = nrdp.gibbon._setValue(function(data) {
                    data["systemallocated"] = ps3data.heap.mmap.used + ps3data.heap.used;
                    data["systemused"] = ps3data.heap.mmap.total + ps3data.heap.total;
                    callback(data);
                });
                nrdp._invoke("gibbon", "getHeapSize", {id: id});
            });
        } else {
            var id = nrdp.gibbon._setValue(callback);
            nrdp._invoke("gibbon", "getHeapSize", {id: id});
        }
    },
    moveMouse: function moveMouse(x, y) {
        nrdp._invoke("gibbon", "moveMouse", {x: x, y: y});
    },
    widgetsAt: function widgetsAt(x, y, cb) {
        var id = nrdp.gibbon._setValue(cb);
        nrdp._invoke("gibbon", "widgetsAt", {x: x, y: y, id: id});
    },
    reloadFonts: function reloadFonts() {
        nrdp._invoke("gibbon", "reloadFonts");
    },
    _fontTTFCallback: function _fontTTFCallback(key, data) {
        if (!nrdp.gibbon._syncData.downloadableFonts.hasOwnProperty(key))
            return;
        var info = nrdp.gibbon._syncData.downloadableFonts[key];
        var params = info.params;
        if (!params.hasOwnProperty("attributes"))
            params.attributes = nrdp.gibbon.fonts.attributes.HINTING | nrdp.gibbon.fonts.attributes.HINTNORMAL;
        if (data.errorcode !== 0 || (data.statusCode < 200 || data.statusCode >= 300) || !data.data.byteLength) {
            if (info.callback)
                info.callback({ success: false, error: "Error in download of TTF" });
            delete nrdp.gibbon._syncData.downloadableFonts[key];
            return;
        }
        var persist = (params && params instanceof Object && params.persist == true);
        var priority = params && params instanceof Object && params.priority;
        if (priority === undefined)
            priority = 100;
        if (persist) {
            var store = nrdp.gibbon.fonts._store;
            if (!store) {
                if (info.callback)
                    info.callback({ success: false, error: "Font store could not be created" });
                delete nrdp.gibbon._syncData.downloadableFonts[key];
                return;
            }
            store.create(nrdp.storage.NO_DEVICE_ACCOUNT, key, data.data, function(o) {
                if (o.success) {
                    // get the file name and mmap it in
                    store.query(nrdp.storage.NO_DEVICE_ACCOUNT, key, function(q) {
                        if (q.hasOwnProperty(key)) {
                            var fn = q[key].fileName;
                            var cb = info.callback;
                            delete info["callback"];
                            var id = nrdp.gibbon._setValue(cb);
                            nrdp._invoke("gibbon", "loadFont", { id: id, priority: priority, key: key, font: params, fileName: fn });
                        } else {
                            if (info.callback)
                                info.callback({ success: false, error: "Unable to query font from font store" });
                            delete nrdp.gibbon._syncData.downloadableFonts[key];
                        }
                    }, true /* validate */);
                    // also persist the info structure so we can read it in on startup next time
                    store.create(nrdp.storage.NO_DEVICE_ACCOUNT, key + "_info", JSON.stringify(info));
                } else {
                    if (info.callback)
                        info.callback({ success: false, error: "Unable to store font in font store" });
                    delete nrdp.gibbon._syncData.downloadableFonts[key];
                }
            });
        } else {
            var cb = info.callback;
            delete info["callback"];
            var id = nrdp.gibbon._setValue(cb);
            nrdp._invoke("gibbon", "loadFont", { id: id, priority: priority, key: key, font: params, data: data.data });
        }
    },
    addFont: function addFont(params, callback) {
        if (!(params instanceof Object)) {
            if (callback) {
                callback({ success: false, error: "Params is not an object" });
            }
            return;
        }
        if (!nrdp.gibbon._syncData.downloadableFonts)
            nrdp.gibbon._syncData.downloadableFonts = {};
        if (!params.hasOwnProperty("request") || !(params.request instanceof Object)) {
            if (callback) {
                callback({ success: false, error: "Request missing" });
            }
            return;
        }
        var key = nrdp.gibbon.fonts._makeKey(params);
        if (key === undefined) {
            // bad
            if (callback) {
                callback({ success: false, error: "Unable to make font key" });
            }
            return;
        }
        if (nrdp.gibbon._syncData.downloadableFonts.hasOwnProperty(key)) {
            // already have the font
            if (callback)
                callback({ success: false, error: "Font exists" });
            return;
        }
        params.request.format = "arraybuffer";
        params._key = key;
        nrdp.gibbon._syncData.downloadableFonts[key] = { callback: callback, params: params };
        nrdp.gibbon.load(params.request, function(data) { nrdp.gibbon._fontTTFCallback(key, data); });
    },
    removeFont: function removeFont(params, cb) {
        if (!nrdp.gibbon._syncData.downloadableFonts) {
            if (cb)
                cb({success: false, error: "No downloadable fonts"});
            return;
        }
        var key = nrdp.gibbon.fonts._makeKey(params);
        if (key === undefined) {
            // bad
            if (cb) {
                cb({ success: false, error: "Unable to make font key" });
                return;
            }
        }
        if (!nrdp.gibbon._syncData.downloadableFonts.hasOwnProperty(key)) {
            if (cb)
                cb({success: false, error: "No such font: " + key});
            return;
        }
        var data = nrdp.gibbon._syncData.downloadableFonts[key];
        delete nrdp.gibbon._syncData.downloadableFonts[key];
        var store = nrdp.gibbon.fonts._store;
        store.remove(nrdp.storage.NO_DEVICE_ACCOUNT, key);
        store.remove(nrdp.storage.NO_DEVICE_ACCOUNT, key + "_info");
        var id = nrdp.gibbon._setValue(cb);
        nrdp._invoke("gibbon", "removeFont", {family: params.family, weight: params.weight, style: params.style, id: id});
    },
    cpuProfileStart: function cpuProfileStart(profileId) {
        nrdp._invoke("gibbon", "cpuProfileStart", {profileId: profileId});
    },
    cpuProfileStop: function cpuProfileStop(profileId, callback) {
        var id = nrdp.gibbon._setValue(callback);
        nrdp._invoke("gibbon", "cpuProfileStop", {id: id, profileId: profileId});
    },
    _runConsole: function _runConsole(cmd) {
        nrdp._invoke("gibbon", "runConsole", {command: cmd});
    },
    get _baseUrl() {
        if(this._baseUrlVar)
            return this._baseUrlVar.url;
        return undefined;
    },
    set _baseUrl(u) {
        if(u) {
            var dir = u;
            var q = dir.indexOf("?");
            if(q != -1)
                dir = dir.substr(0, q);
            var s = dir.lastIndexOf("/");
            if(s != -1)
                dir = dir.substr(0, s + 1);
            this._baseUrlVar = { dir: dir, url: u };
        } else {
            this._baseUrlVar = undefined;
        }
    },
    _resolveUrl: function _resolveUrl(url) {
        if(this._baseUrlVar && typeof url == "string" && url.indexOf("://") == -1 && url.indexOf("data:") != 0) {
            var baseUrl = this._baseUrlVar.dir;
            if(url[0] == "/") {
                var s = baseUrl.indexOf("/", 8);
                if(s != -1)
                    baseUrl = baseUrl.substr(0, s);
            }
            return baseUrl + url;
        }
        return url;
    },
    _syncData: { debugFlags: { } },
    _updateProperty: function _updateProperty(property, value) {
        if(property == "networkErrorCodes" || property == "networkErrorGroups") {
            for(var prop in value)
                nrdp.gibbon[prop] = value[prop];
        } else {
            var evt;
            if(property == "location") {
                this._syncData['queryParams'] = undefined;
            } else if(property == "fonts") {
                evt = {
                    type: "fontschange"
                };
                nrdp.gibbon._syncData.font_scripts = value.scripts;
                delete value["scripts"];
                property = "font_entries";
            } else if(nrdp.isReady && property == "cookie" && nrdp.gibbon.cookie != value) {
                evt = {
                    type: "cookiechange",
                    old: nrdp.gibbon.cookie
                };
            } else if (property == "diskCacheConfiguration" && value && value.writeSpeed != this.diskCache.writeSpeed) {
                evt = {
                    type: "writeSpeedMeasured",
                    writeSpeed: value.writeSpeed
                };
            }
            this._syncData[property] = value;
            if(evt)
                nrdp._callEventListeners(this, evt);
        }
    },
    sendKey: function sendKey(key, userData) {
        var lookupKey = function(name) {
            if (nrdp.nrdpPartner && nrdp.nrdpPartner.Keys) {
                for (var c in nrdp.nrdpPartner.Keys) {
                    if (nrdp.nrdpPartner.Keys[c] == name)
                        return parseInt(c);
                }
            }
            return undefined;
        };
        var evt = { type: "key", data: { type: key.type, code: lookupKey(key.name), uiEvent: key.name, userData: userData } };
        nrdp._callEventListeners(this, evt);
    },
    _handleEvent: function _handleEvent(event) {
        var handled = true;
        var cb;
        if(event.name == "gibbonEvent") {
            var type = event.data.type;
            if(type == "click" || type == "press" || type == "release") {
                var evt = { type: "key", time: event.time , data: event.data };
                if(nrdp.gibbon._debugKeyEvents) {
                    var desc = (event.data.repeat) ? "repeat" : type;
                    {
                        var monoTime = nrdp.mono();
                        if(nrdp.gibbon._keyEventLastMonoTime === undefined)
                            nrdp.gibbon._keyEventLastMonoTime = monoTime;
                        desc += ": " + (monoTime-nrdp.gibbon._keyEventLastMonoTime);
                        nrdp.gibbon._keyEventLastMonoTime = monoTime;
                        if(nrdp.gibbon._keyEventLastTime === undefined)
                            nrdp.gibbon._keyEventLastTime = event.data.time;
                        desc += " (" + (event.time-nrdp.gibbon._keyEventLastTime) + ":" + (monoTime-event.time) + ")";
                        nrdp.gibbon._keyEventLastTime = event.time;
                    }
                    nrdp.log.error("Sending keyevent(" + nrdp.gibbon._debugKeyEvents++ + "): " + desc + " [" + JSON.stringify(event) + "]");
                }
                if(nrdpPartner && nrdpPartner.Keys && !evt.data.uiEvent)
                    evt.data.uiEvent = nrdpPartner.Keys[evt.data.code];
                if(type == "click") {
                    evt.data.type = "press";
                    nrdp._callEventListeners(this, evt);
                    if(!event.data.repeat)
                        evt.data.type = "release";
                    else
                        evt = 0;
                }
                if(evt) {
                    nrdp._callEventListeners(this, evt);
                }
            } else if(type == "focusChange") {
                nrdp._callEventListeners(this, { type: "focus", time: event.time, data: event.data });
            } else if(type == "mouseChange") {
                nrdp._callEventListeners(this, { type: "mouse", time: event.time, data: event.data });
            } else if(type == "mouseMove" || type == "mousePress" || type == "mouseRelease") {
               if(event.data.widgets) {
                   for(var i = 0; i < event.data.widgets.length; ++i)
                       event.data.widgets[i].widget = nrdp.gibbon.findWidget(event.data.widgets[i].id);
               }
                nrdp._callEventListeners(this, { type: "mouse", time: event.time, data: event.data });
           } else if(type == "touchMove" || type == "touchPress" || type == "touchRelease") {
               if(event.data.widgets) {
                   for(var j = 0; j < event.data.widgets.length; ++j)
                       event.data.widgets[j].widget = nrdp.gibbon.findWidget(event.data.widgets[j].id);
               }
               nrdp._callEventListeners(this, { type: "touch", time: event.time, data: event.data });
           } else {
               nrdp.log.error("unhandled gibbonEvent " + type);
           }
        } else if(event.name == "requestFinished") {
            cb = this._getValue(event.data.id);
            if(!event.data.refresh || event.data.state == "refresh")
                this._deleteValue(event.data.id);
            if(event.data.parsed && event.data.xml)
                nrdp._fixXml(event.data.xml);
            if(typeof cb == "function")
                cb(event.data);
        } else if(event.name == "task") {
            var task_id = event.data;
            var task = this._getValue(task_id);
            if(task !== undefined) {
                if(task.callback !== undefined)
                    task.callback();
                this._deleteValue(task_id);
            }
        } else if(event.name == "timer") {
            var timer_id = event.data;
            var timer = this._getValue(timer_id);
            if(timer !== undefined) {
                if(timer.callback !== undefined)
                    timer.callback();
                if(timer.singleShot)
                    this._deleteValue(timer_id);
            }
        } else if(event.name == "getRenderedFps"
                  || event.name == "diskCacheCleared"
                  || event.name == "diskCacheStatsCleared"
                  || event.name == "diskCacheFlushed"
                  || event.name == "diskCacheReinited"
                  || event.name == "diskCacheDumped"
                  || event.name == "diskCacheInfo"
                  || event.name == "diskCacheInsert"
                  || event.name == "diskCachePurgeExpired"
                  || event.name == "diskCacheRemove"
                  || event.name == "garbageCollected"
                  || event.name == "hash"
                  || event.name == "widgetsAt"
                  || event.name == "heapsize"
                  || event.name == "resourceManagerDisable"
                  || event.name == "resourceManagerInfo"
                  || event.name == "resourceManagerRemove"
                  || event.name == "resourceManagerClear"
                  || event.name == "setCookie"
                  || event.name == "cpuProfile"
                  || event.name == "fontLoaded"
                  || event.name == "fontRemoved"
                  || event.name == "fontManagerInfo"
                  || event.name == "surfaceCacheInfo") {
            cb = this._getValue(event.data.id);
            if(cb !== undefined) {
                if(event.name == "widgetsAt") {
                    if(event.data.widgets) {
                        for(var k = 0; k < event.data.widgets.length; ++k)
                            event.data.widgets[k].widget = nrdp.gibbon.findWidget(event.data.widgets[k].id);
                    }
                    cb(event.data);
                } else if(event.name == "getRenderFps") {
                    cb(event.data.fps);
                } else if(event.name == "heapsize"
                          || event.name == "cpuProfile") {
                    cb(event.data);
                } else if(event.name == "diskCacheDumped") {
                    cb(event.data.dump, event.data.stats);
                } else if(event.name == "diskCacheInfo"
                          || event.name == "fontLoaded"
                          || event.name == "fontRemoved") {
                    cb(event.data.data);
                } else if(event.name == "diskCacheInsert"
                          || event.name == "diskCacheRemove"
                          || event.name == "resourceManagerRemove") {
                    cb(event.data.success);
                } else if(event.name == "diskCachePurgeExpired") {
                    cb(event.data.count);
                } else if(event.name == "hash") {
                    cb(event.data.hash);
                } else if(event.name == "surfaceCacheInfo"
                          || event.name == "fontManagerInfo"
                          || event.name == "resourceManagerInfo") {
                    cb(event.data.info);
                } else {
                    cb();
                }
            }
            this._deleteValue(event.data.id);
        } else if (event.name == "capacityReached") {
            nrdp._callEventListeners(this, { type: "capacityReached", data: event.data });
        } else if (event.name == "codepointMissing") {
            nrdp._callEventListeners(this, { type: "codepointMissing", data: event.data });
        } else if (event.name == "diskCacheWriteLimiter") {
            nrdp._callEventListeners(this, { type: "diskCacheWriteLimiter", data: event.data });
        } else {
            handled = false;
        }
        return handled;
    },
    _setValue: function _setValue(value) {
        do {
            ++nrdp.gibbon._callback;
            if(nrdp.gibbon._callback >= nrdp.gibbon.INT_MAX)
                nrdp.gibbon._callback = 1;
        } while(nrdp.gibbon._callbacks[nrdp.gibbon._callback] !== undefined);
        nrdp.gibbon._callbacks[nrdp.gibbon._callback] = value;
        return nrdp.gibbon._callback;
    },
    _getValue: function _getValue(id) {
        return nrdp.gibbon._callbacks[id];
    },
    _deleteValue: function _deleteValue(id) {
        delete nrdp.gibbon._callbacks[id];
    },
    _invoke: function _invoke(object, method, args) {
        nrdp._invoke("gibbon", "invoke", {subObject:object, subMethod:method, subArgs:args});
    },
    addSocketizerConditions: function addSocketizerConditions(conditions) {
        nrdp._invoke("gibbon", "addSocketizerConditions", { conditions: conditions });
    },
    _hash: function (request, cb) {
        var req = request instanceof Object ? request : { url: request };
        req.id = nrdp.gibbon._setValue(cb);
        nrdp._invoke("gibbon", "hash", req);
    }
};
nrdp.addEventListener('init', function() {
    if(nrdp.gibbon.location != nrdp.system.bootURL) {
        nrdp.gibbon._loadSplash();
    }
    nrdp.gibbon.fonts.init();
});
nrdp.addEventListener('factoryReset', function() {
    if (nrdp.gibbon && nrdp.gibbon._syncData)
        nrdp.gibbon._syncData.downloadableFonts = {};
});
nrdp.describeEvent = function describeGibbonEvent(event) {
    if(event && event.type == "Event") {
        if(event.name == "timer" && event.object == "nrdp.gibbon")
            return "timer:" + event.data;
        else if(event.name == "requestFinished" && event.object == "nrdp.gibbon")
            return "requestFinished:" + event.data.id;
        else if(event.name == "gibbonEvent" && event.object == "nrdp.gibbon")
            return "gibbonEvent:" + JSON.stringify(event.data);
    }
    return nrdp._describeEvent(event);
},
nrdp._sendSyncdEvent = function(fn, that, event) {
    if(nrdp.gibbon.debugFlags.debugScriptEvents) {
        var monoTime = nrdp.mono();
        ++nrdp.gibbon._nextDebugEvent;
        nrdp.log.error("Sending event(" + nrdp.gibbon._nextDebugEvent + ") " + nrdp.describeEvent(event));
        fn.call(that, event);
        nrdp.log.error("~Sending event(" + nrdp.gibbon._nextDebugEvent + ") " + (nrdp.mono()-monoTime));
    } else {
        fn.call(that, event);
    }
};
if(typeof nrdpPartner === "undefined")
    nrdpPartner = {};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp */
nrdp.gibbon.scene = {
    classname: "ScreenBridge",
    _callbacks: {},
    _nextCallbackId: 1,
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get suspend() { return (this._suspendPending === undefined) ? this._syncData.suspend : this._suspendPending; },
    set suspend(_suspend) {
        if(_suspend == this.suspend)
            return;
        if(this._suspendPending === undefined) {
            this._syncData.suspend = _suspend;
            nrdp._setProperty("gibbon.scene", "suspend", _suspend);
        }
        this._suspendPending = _suspend;
    },
    get widget() { return this.root; },
    set widget(_widget) { this.root = _widget; },
    get root() {
        var wid = this._syncData.root;
        return (typeof wid === "object") ? wid : null;
    },
    set root(_widget) {
        if(_widget && this.root && _widget._id == this.root._id)
            return;
        var oldRoot = this.root;
        this._syncData.root = _widget;
        nrdp._setProperty("gibbon.scene", "root", _widget ? _widget._id : undefined);
        this._rootChanged();
    },
    get overlay() {
        var wid = this._syncData.overlay;
        return (typeof wid === "object") ? wid : null;
    },
    set overlay(_widget) {
        if(_widget && this.overlay && _widget._id == this.overlay._id)
            return;
        var oldOverlay = this.overlay;
        this._syncData.overlay = _widget;
        nrdp._setProperty("gibbon.scene", "overlay", _widget ? _widget._id : undefined);
    },
    get scale() { return this._syncData.scale; },
    get width() { return this._syncData.width; },
    get height() { return this._syncData.height; },
    dump: function dump(cb) {
        this.widget.dump(cb);
    },
    grab: function grab(cb) {
        var id = this._registerCallback(cb);
        nrdp._invoke("gibbon.scene", "grab", { id: id });
    },
    update: function update(widget) {
        nrdp._invoke("gibbon.scene", "update", { id: widget ? widget._id : this.widget._id });
    },
    _rootChanged: function _rootChanged() {
        if(nrdp._hasEventListener(this, "rootChanged"))
            nrdp._callEventListeners(this, { type: "rootChanged" });
    },
    _handleEvent: function _handleEvent(event) {
        if(event.name == "grab") {
            this._callCallback(event);
            return true;
        } else if(event.name == "suspendChanged") {
            this._syncData.suspend = event.data.suspended;
            var evt = { type: event.name, data: event.data };
            nrdp._callEventListeners(this, evt);
            if(this._suspendPending !== undefined) {
                var suspendPending = this._suspendPending;
                this._suspendPending = undefined;
                if(suspendPending !== event.data.suspended) {
                    this.suspend = suspendPending;
                }
            }
            return true;
        }
        return false;
    },
    _registerCallback: function _registerCallback(cb) {
        var id = this._nextCallbackId++;
        this._callbacks[id] = cb;
        return id;
    },
    _callCallback: function _callCallback(event) {
        if (event.data.id === undefined)
            return;
        var cb = this._callbacks[event.data.id];
        delete this._callbacks[event.data.id];
        if (cb)
            cb(event.data.data);
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp, nrdp_platform, GIBBON_PULL, GIBBON_PUSH, GibbonSyncObject */
nrdp.gibbon.Image = function(widget, background, id) {
    if(!widget._syncData.images)
        widget._syncData.images = {};
    this._object = new GibbonSyncObject(this, widget._id, nrdp.gibbon._sync_Image, id, background);
    this._id = id;
    this._widget = widget;
    this._background = background;
    this._name = nrdp.gibbon.Image._createName(this._id);
    this._path = widget._path + ".images." + this._name;
    if(!this._syncData)
        this._syncData = widget._syncData.images[this._name] = {};
    this._defaults = nrdp.gibbon.Image._default;
    this._properties = nrdp._classes[this.classname].properties.byName;
};
nrdp.gibbon.Image._createName = function(id) { return "image" + id; };
nrdp.gibbon.Image.LAZY_DECODE = 0x01;
nrdp.gibbon.Image.LAZY_DOWNLOAD = 0x02;
nrdp.gibbon.Image.ALIGN_NORMAL = 0x00;
nrdp.gibbon.Image.ALIGN_LEFT = nrdp.gibbon.Image.ALIGN_NORMAL;
nrdp.gibbon.Image.ALIGN_TOP = nrdp.gibbon.Image.ALIGN_NORMAL;
nrdp.gibbon.Image.ALIGN_CENTER = 0x01;
nrdp.gibbon.Image.ALIGN_RIGHT = 0x02;
nrdp.gibbon.Image.ALIGN_BOTTOM = nrdp.gibbon.Image.ALIGN_RIGHT;
nrdp.gibbon.Image.ALIGN_TILE = 0x04;
nrdp.gibbon.Image.ALIGN_STRETCH = 0x08;
nrdp.gibbon.Image.ALIGN_ASPECT = 0x18;
nrdp.gibbon.Image._default = {
    src: "",
    purge: true,
    player: undefined,
    sendImageLoaded: false,
    sourceRect: { x: 0, y: 0, width: undefined, height: undefined },
    verticalAlignment: nrdp.gibbon.Image.ALIGN_NORMAL,
    horizontalAlignment: nrdp.gibbon.Image.ALIGN_NORMAL
};
nrdp.gibbon.Image.prototype = {
    classname: "ImageBridge",
    constructor: nrdp.gibbon.Image,
    get sendImageLoaded() { return (this._syncData.hasOwnProperty("sendImageLoaded") ? this._syncData["sendImageLoaded"] : this._defaults["sendImageLoaded"]); },
    set sendImageLoaded(_sendImageLoaded) {
        if(_sendImageLoaded == this.sendImageLoaded)
            return;
        this._syncData["sendImageLoaded"] = _sendImageLoaded; this._object.setSyncProperty(this._properties["sendImageLoaded"], _sendImageLoaded);
    },
    addEventListener: function addEventListener(evt, listener) {
        if(nrdp._addEventListener(this, evt, listener)) {
            if(evt == "imageLoaded")
                this.sendImageLoaded = true;
        }
    },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    get height() { return (this._syncData.hasOwnProperty("height") ? this._syncData["height"] : this._defaults["height"]); },
    get width() { return (this._syncData.hasOwnProperty("width") ? this._syncData["width"] : this._defaults["width"]); },
    get visible() { return (this._syncData.hasOwnProperty("visible") ? this._syncData["visible"] : this._defaults["visible"]); },
    set visible(_visible) {
        if(_visible == this.visible)
            return;
        this._syncData["visible"] = _visible; this._object.setSyncProperty(this._properties["visible"], _visible);
    },
    get sourceRect() { return (this._syncData.hasOwnProperty("sourceRect") ? this._syncData["sourceRect"] : this._defaults["sourceRect"]); },
    set sourceRect(_sourceRect_in) {
        var _sourceRect = (this._syncData.hasOwnProperty("sourceRect") ? this._syncData["sourceRect"] : this._defaults["sourceRect"]);
        if(_sourceRect &&
           _sourceRect_in.x == _sourceRect.x &&
           _sourceRect_in.y == _sourceRect.y &&
           _sourceRect_in.width == _sourceRect.width &&
           _sourceRect_in.height == _sourceRect.height)
            return;
        this._syncData["sourceRect"] = _sourceRect_in; this._object.setSyncProperty(this._properties["sourceRect"], ([_sourceRect_in.x, _sourceRect_in.y, _sourceRect_in.width, _sourceRect_in.height]));
    },
    get purge() { return (this._syncData.hasOwnProperty("purge") ? this._syncData["purge"] : this._defaults["purge"]); },
    set purge(_purge) {
        if(_purge == this.purge)
            return;
        this._syncData["purge"] = _purge; this._object.setSyncProperty(this._properties["purge"], _purge);
    },
    _encodeAlignment: function _encodeAlignment(align) {
        var result;
        if(align instanceof Array) {
            result = 0;
            for(var i = 0; i < align.length; ++i)
                result |= this._encodeAlignment(align[i]);
        } else if(typeof align === "string") {
            result = 0;
            var aligns = align.split(" ");
            for(var j = 0; j < aligns.length; ++j) {
                var a = aligns[j];
                if(a === "normal")
                    result |= nrdp.gibbon.Image.ALIGN_NORMAL;
                else if(a === "left")
                    result |= nrdp.gibbon.Image.ALIGN_LEFT;
                else if(a === "top")
                    result |= nrdp.gibbon.Image.ALIGN_TOP;
                else if(a === "center")
                    result |= nrdp.gibbon.Image.ALIGN_CENTER;
                else if(a === "right")
                    result |= nrdp.gibbon.Image.ALIGN_RIGHT;
                else if(a === "bottom")
                    result |= nrdp.gibbon.Image.ALIGN_BOTTOM;
                else if(a === "tile")
                    result |= nrdp.gibbon.Image.ALIGN_TILE;
                else if(a === "stretch")
                    result |= nrdp.gibbon.Image.ALIGN_STRETCH;
                else if(a === "aspect")
                    result |= nrdp.gibbon.Image.ALIGN_ASPECT;
            }
        } else {
            result = align;
        }
        return result;
    },
    get halign() { return this.horizontalAlignment; },
    set halign(_halign) { this.horizontalAlignment = _halign; },
    get horizontalAlignment() { return (this._syncData.hasOwnProperty("horizontalAlignment") ? this._syncData["horizontalAlignment"] : this._defaults["horizontalAlignment"]); },
    set horizontalAlignment(_halign) {
        _halign = this._encodeAlignment(_halign);
        if(_halign == this.halign)
            return;
        this._syncData["horizontalAlignment"] = _halign; this._object.setSyncProperty(this._properties["horizontalAlignment"], _halign);
    },
    get valign() { return this.verticalAlignment; },
    set valign(_valign) { this.verticalAlignment = _valign; },
    get verticalAlignment() { return (this._syncData.hasOwnProperty("verticalAlignment") ? this._syncData["verticalAlignment"] : this._defaults["verticalAlignment"]); },
    set verticalAlignment(_valign) {
        _valign = this._encodeAlignment(_valign);
        if(_valign == this.valign)
            return;
        this._syncData["verticalAlignment"] = _valign; this._object.setSyncProperty(this._properties["verticalAlignment"], this._encodeAlignment(_valign));
    },
    reload: function reload() {
        var url = this.url;
        if(url instanceof Object) {
            if(!url.url || !url.url.length)
                return;
            url.lazy = false;
        } else {
            if(!url || !url.length)
                return;
        }
        this._syncData["src"] = url; this._object.setSyncProperty(this._properties["src"], url);
    },
    get group() { return (this._syncData.hasOwnProperty("group") ? this._syncData["group"] : this._defaults["group"]); },
    set group(_group) {
        if(_group == this.group)
            return;
        this._syncData["group"] = _group; this._object.setSyncProperty(this._properties["group"], _group);
    },
    get player() { return (this._syncData.hasOwnProperty("player") ? this._syncData["player"] : this._defaults["player"]); },
    get url() { return this.src; },
    set url(_url) { this.src = _url; },
    get src() { return (this._syncData.hasOwnProperty("src") ? this._syncData["src"] : this._defaults["src"]); },
    set src(_url) {
        var url = (this._syncData.hasOwnProperty("src") ? this._syncData["src"] : this._defaults["src"]);
        if(_url instanceof Object) {
            if(_url.url)
                _url.url = nrdp.gibbon._resolveUrl(_url.url);
            if(typeof _url.lazy == "boolean")
                _url.lazy = _url.lazy ? nrdp.gibbon.Image.LAZY_DOWNLOAD : undefined;
            // not comparing typed arrays by content, only by reference
            if(url instanceof Object
               && ((_url.url && _url.url == url.url) || (_url.data && _url.data == url.data))
               && _url.lazy == url.lazy
               && _url.async == url.async
               && _url.scale == url.scale
               && _url.required == url.required
               && _url.timeout == url.timeout
               && _url.highPriority == url.highPriority
               && _url.connectTimeout == url.connectTimeout
               && _url.stallTimeout == url.stallTimeout) {
                if(_url.headers instanceof Object && url.headers instanceof Object) {
                    var same = true;
                    var prop;
                    for(prop in _url.headers) {
                        if(!_url.headers.hasOwnProperty(prop))
                            continue;
                        if(url.headers[prop] != _url.headers[prop]) {
                            same = false;
                            break;
                        }
                        if(same) {
                            for(prop in url.headers) {
                                if(!url.headers.hasOwnProperty(prop))
                                    continue;
                                if(_url.headers[prop] != url.headers[prop]) {
                                    same = false;
                                    break;
                                }
                            }
                            if(same)
                                return;
                        }
                    }
                } else if(!_url.headers && !url.headers) {
                    return;
                }
            } else if(!url && (!_url.url || !_url.url.length) && (!_url.data || !_url.data.length)) {
                return;
            }
            var tmp = _url;
            _url = { };
            var properties = 0;
            if(tmp.fast === true) {
                ++properties;
                _url.fast = tmp.fast;
            }
            if(tmp.lazy !== undefined) {
                ++properties;
                _url.lazy = tmp.lazy;
            }
            if(tmp.async === false) {
                ++properties;
                _url.async = tmp.async;
            }
            if(tmp.scale !== undefined) {
                ++properties;
                _url.scale = tmp.scale;
            }
            if(tmp.required === false) {
                ++properties;
                _url.required = tmp.required;
            }
            if(tmp.highPriority === true) {
                ++properties;
                _url.highPriority = tmp.highPriority;
            }
            if(tmp.timeout) {
                ++properties;
                _url.timeout = tmp.timeout;
            }
            if(tmp.connectTimeout) {
                ++properties;
                _url.connectTimeout = tmp.connectTimeout;
            }
            if(tmp.stallTimeout) {
                ++properties;
                _url.stallTimeout = tmp.stallTimeout;
            }
            if(tmp.url) {
                ++properties;
                _url.url = tmp.url;
            }
            if(tmp.data) {
                ++properties;
                _url.data = tmp.data;
            }
            if(tmp.headers) {
                var headers = {};
                var hasHeaders = false;
                for(var header in tmp.headers) {
                    if(!tmp.headers.hasOwnProperty(header))
                        continue;
                    var val = tmp.headers[header];
                    if(typeof val === "string") {
                        hasHeaders = true;
                        headers[header] = val;
                    }
                }
                if(hasHeaders) {
                    ++properties;
                    _url.headers = headers;
                }
            }
            if((properties <= 1 || !_url.url) && !_url.data) {
                _url = _url.url ? _url.url : "";
                if(url == _url)
                    return;
            }
        } else {
            _url = nrdp.gibbon._resolveUrl(_url);
            if(_url == url)
                return;
        }
        this._syncData["src"] = _url; this._object.setSyncProperty(this._properties["src"], _url);
        // temporarily delete width/height; when the image comes back it will be populated again
        delete this._syncData.width;
        delete this._syncData.height;
    },
    _handleEvent: function _handleEvent(event) {
        var handled = true;
        if(event.name == "imageLoaded") {
            if(event.data.success) {
                if(nrdp.gibbon._breaks !== undefined)
                    this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
                if(event.data.state == "player")
                    this._syncData.player = nrdp.player;
                this._syncData.width = event.data.width;
                this._syncData.height = event.data.height;
            }
            var evt = { type: event.name, data: event.data };
            nrdp._callEventListeners(this, evt);
        } else {
            handled = false;
        }
        return handled;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp, GibbonSyncObject*/
nrdp.gibbon.Text = function(widget) {
    this._object = new GibbonSyncObject(this, widget._id, nrdp.gibbon._sync_Text);
    this._widget = widget;
    this._path = widget._path + ".text";
    if(!this._syncData)
        this._syncData = widget._syncData.text = {};
    this._defaults = nrdp.gibbon.Text._default;
    this._properties = nrdp._classes[this.classname].properties.byName;
};
nrdp.gibbon.Text._default = {
    size: 30,
    lineHeight: undefined,
    sendTextLoaded: false,
    sendRenderProperties: false,
    weight: "bold",
    variant: "normal",
    align: nrdp.gibbon.ALIGN_LEFT,
    color: { r: 0, g: 0, b: 0, a: 255 },
    backgroundColor: { r: 0, g: 0, b: 0, a: 0 },
    padding: 0,
    indent: 0,
    shadow: {
        offsetX: 0,
        offsetY: 0,
        color: { r: 0, g: 0, b: 0, a: 255 }
    },
    truncation: {
        position: "none",
        ellipsis: "…"
    },
    edgeEffect: {
        type: "none",
        width: 1,
        lightColor: { r: 0, g: 0, b: 0, a: 255 },
        darkColor: { r: 0, g: 0, b: 0, a: 255 },
        outlineColor: { r: 0, g: 0, b: 0, a: 255 }
    },
    cursor: {
        visible: false,
        style: "line",
        color: {r: 0, g: 0, b: 0, a: 255 },
        interval: 500,
        width: 1
    },
    typography: {
        kerning: true,
        tracking: 0,
        fontFeatures: ""
    },
    wrap: false,
    underline: false,
    strikethrough: false,
    richText: false,
    text: undefined,
    firstLine: 0,
    cursorPosition: undefined,
    renderFirstLine: 0
};
nrdp.gibbon.Text.prototype = {
    classname: "TextBridge",
    constructor: nrdp.gibbon.Text,
    toString: function toString() {
        var contents = (this._syncData.hasOwnProperty("contents") ? this._syncData["contents"] : this._defaults["contents"]);
        if(contents === undefined)
            return "";
        return contents;
    },
    addEventListener: function addEventListener(evt, listener) {
        if(nrdp._addEventListener(this, evt, listener)) {
            if(evt == "renderpropertychange")
                this.sendRenderProperties = true;
            else if(evt == "textLoaded")
                this.sendTextLoaded = true;
        }
    },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    _mergeValues: function _mergeValues(newObject, oldObject, values) {
        var mergeObject = {};
        for(var v in values) {
            var valueName = values[v];
            var newValue;
            if(newObject.hasOwnProperty(valueName)) {
                newValue = newObject[valueName];
                if(valueName === "color")
                    newValue = { r: newValue.r, g: newValue.g, b: newValue.b, a: (newValue.a === undefined ? 255 : newValue.a) };
                if(valueName === "backgroundColor")
                    newValue = { r: newValue.r, g: newValue.g, b: newValue.b, a: (newValue.a === undefined ? 255 : newValue.a) };
            } else {
                newValue = oldObject[valueName];
            }
            mergeObject[valueName] = newValue;
        }
        return mergeObject;
    },
    _sendRenderProperty: function _sendRenderProperty(property, renderProperty, value) {
        //nrdp.log.error("EVENT: " + evt.type + " - p=" + evt.property + ", rP=" + evt.renderProperty + ", v=" + evt.value);
        nrdp._callEventListeners(this, { type: "renderpropertychange",
                                         property: property,
                                         renderProperty: renderProperty,
                                         value: value });
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    _renderProperty: function _renderProperty(property) {
        if(!this.sendRenderProperties) {
            nrdp.log.error("RenderProperty(" + property + "): Used but TEXT(" + this._widget._id + ") has not set sendRenderProperties!");
            this.sendRenderProperties = true;
        }
        return (this._syncData.hasOwnProperty(property) ? this._syncData[property] : this._defaults[property]);
    },
    get sendRenderProperties() { return (this._syncData.hasOwnProperty("sendRenderProperties") ? this._syncData["sendRenderProperties"] : this._defaults["sendRenderProperties"]); },
    set sendRenderProperties(_sendRenderProperties) {
        if(_sendRenderProperties == this.sendRenderProperties)
            return;
        if(this._text)
            this._text.sendRenderProperties = _sendRenderProperties;
        this._syncData["sendRenderProperties"] = _sendRenderProperties; this._object.setSyncProperty(this._properties["sendRenderProperties"], _sendRenderProperties);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get sendTextLoaded() { return (this._syncData.hasOwnProperty("sendTextLoaded") ? this._syncData["sendTextLoaded"] : this._defaults["sendTextLoaded"]); },
    set sendTextLoaded(_sendTextLoaded) {
        if(_sendTextLoaded == this.sendTextLoaded)
            return;
        this._syncData["sendTextLoaded"] = _sendTextLoaded; this._object.setSyncProperty(this._properties["sendTextLoaded"], _sendTextLoaded);
    },
    get renderLines() { return this._renderProperty("renderLines"); },
    get renderTextLines() { return this.renderLines; }, //compat
    get renderFirstLine() { return this._renderProperty("renderFirstLine"); },
    get renderPadding() { return this._renderProperty("renderPadding"); },
    get renderTextFirstLine() { return this.renderFirstLine; }, //compat
    get renderCursorPosition() { return this._renderProperty("renderCursorPosition"); },
    get renderTextCursorPosition() { return this.renderCursorPosition; }, //compat
    get contents() { return (this._syncData.hasOwnProperty("contents") ? this._syncData["contents"] : this._defaults["contents"]); },
    set contents(_contents) {
        if(_contents === null || typeof _contents === "undefined")
            _contents = "";
        if(_contents == this.contents)
            return;
        this._syncData["contents"] = _contents; this._object.setSyncProperty(this._properties["contents"], _contents);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get firstLine() { return (this._syncData.hasOwnProperty("firstLine") ? this._syncData["firstLine"] : this._defaults["firstLine"]); },
    set firstLine(_firstLine) {
        if(_firstLine == this.firstLine)
            return;
        this._syncData["firstLine"] = _firstLine; this._object.setSyncProperty(this._properties["firstLine"], _firstLine);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get cursorPosition() { return (this._syncData.hasOwnProperty("cursorPosition") ? this._syncData["cursorPosition"] : this._defaults["cursorPosition"]); },
    set cursorPosition(_cursorPosition) {
        if(_cursorPosition == this.cursorPosition)
            return;
        this._syncData["cursorPosition"] = _cursorPosition; this._object.setSyncProperty(this._properties["cursorPosition"], _cursorPosition);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get family() { return (this._syncData.hasOwnProperty("family") ? this._syncData["family"] : this._defaults["family"]); },
    set family(_family) {
        if(_family == this.family)
            return;
        this._syncData["family"] = _family; this._object.setSyncProperty(this._properties["family"], _family);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get locale() { return (this._syncData.hasOwnProperty("locale") ? this._syncData["locale"] : this._defaults["locale"]); },
    set locale(_locale) {
        if(_locale == this.locale)
            return;
        this._syncData["locale"] = _locale; this._object.setSyncProperty(this._properties["locale"], _locale);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get direction() { return (this._syncData.hasOwnProperty("direction") ? this._syncData["direction"] : this._defaults["direction"]); },
    set direction(_direction) {
        if(_direction == this.direction)
            return;
        this._syncData["direction"] = _direction; this._object.setSyncProperty(this._properties["direction"], _direction);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get script() { return (this._syncData.hasOwnProperty("script") ? this._syncData["script"] : this._defaults["script"]); },
    set script(_script) {
        if(_script == this.script)
            return;
        this._syncData["script"] = _script; this._object.setSyncProperty(this._properties["script"], _script);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get size() { return (this._syncData.hasOwnProperty("size") ? this._syncData["size"] : this._defaults["size"]); },
    set size(_size) {
        if(typeof _size != "number")
            _size = this._defaults.size;
        if(_size == this.size && nrdp.gibbon.scene.scale == 1.0)
            return;
        this._syncData["size"] = _size; this._object.setSyncProperty(this._properties["size"], _size);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get padding() { return (this._syncData.hasOwnProperty("padding") ? this._syncData["padding"] : this._defaults["padding"]); },
    set padding(_padding) {
        if (_padding instanceof Array) {
            if (_padding.length > 2) {
                _padding = this._defaults.padding;
            } else {
                for (var i = 0; i < _padding.length; ++i) {
                    if (typeof _padding[i] != "number" && typeof _padding[i] != "undefined") {
                        _padding = this._defaults.padding;
                        break;
                    }
                }
            }
        } else if (typeof _padding != "number" && typeof _padding != "undefined") {
            if (typeof _padding != "object")
                _padding = this._defaults.padding;
        }
        if (_padding == this.padding)
            return;
        this._syncData["padding"] = _padding; this._object.setSyncProperty(this._properties["padding"], _padding);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get indent() { return (this._syncData.hasOwnProperty("indent") ? this._syncData["indent"] : this._defaults["indent"]); },
    set indent(_indent) {
        if(_indent == this.indent)
            return;
        this._syncData["indent"] = _indent; this._object.setSyncProperty(this._properties["indent"], _indent);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get lineHeight() { return (this._syncData.hasOwnProperty("lineHeight") ? this._syncData["lineHeight"] : this._defaults["lineHeight"]); },
    set lineHeight(_lineHeight) {
        if(_lineHeight == this.lineHeight)
            return;
        this._syncData["lineHeight"] = _lineHeight; this._object.setSyncProperty(this._properties["lineHeight"], _lineHeight);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get wrap() { return (this._syncData.hasOwnProperty("wrap") ? this._syncData["wrap"] : this._defaults["wrap"]); },
    set wrap(_wrap) {
        if(_wrap == this.wrap)
            return;
        this._syncData["wrap"] = _wrap; this._object.setSyncProperty(this._properties["wrap"], _wrap);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get richText() { return (this._syncData.hasOwnProperty("richText") ? this._syncData["richText"] : this._defaults["richText"]); },
    set richText(_richText) {
        if(_richText == this.richText)
            return;
        this._syncData["richText"] = _richText; this._object.setSyncProperty(this._properties["richText"], _richText);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get maxLines() { return (this._syncData.hasOwnProperty("maxLines") ? this._syncData["maxLines"] : this._defaults["maxLines"]); },
    set maxLines(_maxLines) {
        if(_maxLines == this.maxLines)
            return;
        this._syncData["maxLines"] = _maxLines; this._object.setSyncProperty(this._properties["maxLines"], _maxLines);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get underline() { return (this._syncData.hasOwnProperty("underline") ? this._syncData["underline"] : this._defaults["underline"]); },
    set underline(_underline) {
        if(_underline == this.underline)
            return;
        this._syncData["underline"] = _underline; this._object.setSyncProperty(this._properties["underline"], _underline);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get strikethrough() { return (this._syncData.hasOwnProperty("strikethrough") ? this._syncData["strikethrough"] : this._defaults["strikethrough"]); },
    set strikethrough(_strikethrough) {
        if(_strikethrough == this.strikethrough)
            return;
        this._syncData["strikethrough"] = _strikethrough; this._object.setSyncProperty(this._properties["strikethrough"], _strikethrough);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get bold() { return this.weight == "bold"; },
    set bold(_bold) { this.weight = _bold ? "bold" : "normal"; },
    get weight() { return (this._syncData.hasOwnProperty("weight") ? this._syncData["weight"] : this._defaults["weight"]); },
    set weight(_weight) {
        if(_weight == this.weight)
            return;
        this._syncData["weight"] = _weight; this._object.setSyncProperty(this._properties["weight"], _weight);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get align() { return (this._syncData.hasOwnProperty("align") ? this._syncData["align"] : this._defaults["align"]); },
    set align(_align) {
        _align = nrdp.gibbon.Widget._encodeAlignment(_align);
        if(_align == this.align)
            return;
        this._syncData["align"] = _align; this._object.setSyncProperty(this._properties["align"], _align);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get italic() { return this.variant == "italic"; },
    set italic(_italic) { this.variant = _italic ? "italic" : "normal"; },
    get variant() { return (this._syncData.hasOwnProperty("variant") ? this._syncData["variant"] : this._defaults["variant"]); },
    set variant(_variant) {
        if(_variant == this.variant)
            return;
        this._syncData["variant"] = _variant; this._object.setSyncProperty(this._properties["variant"], _variant);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get color() { return (this._syncData.hasOwnProperty("color") ? this._syncData["color"] : this._defaults["color"]); },
    set color(_color_in) {
        var _color = this.color;
        if(nrdp.gibbon._compareColor(_color_in, _color))
            return;
        var value = { r: _color_in.r, g: _color_in.g, b: _color_in.b, a: _color_in.a };
        this._syncData["color"] = value; this._object.setSyncProperty(this._properties["color"], nrdp.gibbon._encodeColor(_color_in));
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get backgroundColor() { return (this._syncData.hasOwnProperty("backgroundColor") ? this._syncData["backgroundColor"] : this._defaults["backgroundColor"]); },
    set backgroundColor(_color_in) {
        var _color = this.backgroundColor;
        if(nrdp.gibbon._compareColor(_color_in, _color))
            return;
        var value = { r: _color_in.r, g: _color_in.g, b: _color_in.b, a: _color_in.a };
        this._syncData["backgroundColor"] = value; this._object.setSyncProperty(this._properties["backgroundColor"], nrdp.gibbon._encodeColor(_color_in));
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get cursor() { return (this._syncData.hasOwnProperty("cursor") ? this._syncData["cursor"] : this._defaults["cursor"]); },
    set cursor(_cursor_in) {
        var _cursor = (this._syncData.hasOwnProperty("cursor") ? this._syncData["cursor"] : this._defaults["cursor"]);
        if(!_cursor_in && !_cursor)
            return;
        if(_cursor && _cursor_in &&
           _cursor.style == _cursor_in.style &&
           _cursor.width == _cursor_in.width &&
           _cursor.interval == _cursor_in.interval &&
           nrdp.gibbon._compareColor(_cursor.color, _cursor_in.color) &&
           _cursor.visible == _cursor_in.visible)
            return;
        var value = this._mergeValues(_cursor_in ? _cursor_in : this._defaults.cursor, _cursor,
                                      ["style", "width", "visible", "interval", "color"]);
        this._syncData["cursor"] = value; this._object.setSyncProperty(this._properties["cursor"], ([value.style, value.visible, value.interval, value.width, nrdp.gibbon._encodeColor(value.color)]));
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get edgeEffect() { return (this._syncData.hasOwnProperty("edgeEffect") ? this._syncData["edgeEffect"] : this._defaults["edgeEffect"]); },
    set edgeEffect(_edgeEffect_in) {
        var _edgeEffect = (this._syncData.hasOwnProperty("edgeEffect") ? this._syncData["edgeEffect"] : this._defaults["edgeEffect"]);
        if(!_edgeEffect_in && !_edgeEffect)
            return;
        var value = this._mergeValues(_edgeEffect_in ? _edgeEffect_in : this._defaults.edgeEffect, _edgeEffect,
                                      ["type", "width", "color1", "color2", "lightColor", "darkColor", "outlineColor"]);
        if(value.type == "raised" || value.type == "depressed") {
            if(_edgeEffect_in.lightColor instanceof Object) {
                value.lightColor = _edgeEffect_in.lightColor;
                value.color1 = {
                    r: _edgeEffect_in.lightColor.r,
                    g: _edgeEffect_in.lightColor.g,
                    b: _edgeEffect_in.lightColor.b,
                    a: _edgeEffect_in.lightColor.a
                };
            }
            if(_edgeEffect_in.darkColor instanceof Object) {
                value.darkColor = _edgeEffect_in.darkColor;
                value.color2 = {
                    r: _edgeEffect_in.darkColor.r,
                    g: _edgeEffect_in.darkColor.g,
                    b: _edgeEffect_in.darkColor.b,
                    a: _edgeEffect_in.darkColor.a
                };
            }
        } else if(value.type == "outline") {
            if(_edgeEffect_in.outlineColor instanceof Object) {
                value.outlineColor = _edgeEffect_in.outlineColor;
                value.color1 = {
                    r: _edgeEffect_in.outlineColor.r,
                    g: _edgeEffect_in.outlineColor.g,
                    b: _edgeEffect_in.outlineColor.b,
                    a: _edgeEffect_in.outlineColor.a
                };
            }
        }
        if(_edgeEffect && _edgeEffect.type == value.type && _edgeEffect.width == value.width &&
           nrdp.gibbon._compareColor(_edgeEffect.color1, value.color1) &&
           nrdp.gibbon._compareColor(_edgeEffect.color2, value.color2))
            return;
        this._syncData["edgeEffect"] = value; this._object.setSyncProperty(this._properties["edgeEffect"], ([value.type, value.width, nrdp.gibbon._encodeColor(value.color1), nrdp.gibbon._encodeColor(value.color2)]));
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get shadow() { return (this._syncData.hasOwnProperty("shadow") ? this._syncData["shadow"] : this._defaults["shadow"]); },
    set shadow(_shadow_in) {
        var _shadow = (this._syncData.hasOwnProperty("shadow") ? this._syncData["shadow"] : this._defaults["shadow"]);
        if(!_shadow_in && !_shadow)
            return;
        if(_shadow && _shadow_in && _shadow.color &&
           _shadow.offsetX == _shadow_in.offsetX &&
           _shadow.offsetY == _shadow_in.offsetY &&
           nrdp.gibbon._compareColor(_shadow.color, _shadow_in.color))
            return;
        var value = this._mergeValues(_shadow_in ? _shadow_in : this._defaults.shadow, _shadow,
                                      ["offsetX", "offsetY", "color"]);
        this._syncData["shadow"] = value; this._object.setSyncProperty(this._properties["shadow"], ([value.offsetX, value.offsetY, nrdp.gibbon._encodeColor(value.color)]));
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get truncation() { return (this._syncData.hasOwnProperty("truncation") ? this._syncData["truncation"] : this._defaults["truncation"]); },
    set truncation(_truncation_in) {
        var _truncation = (this._syncData.hasOwnProperty("truncation") ? this._syncData["truncation"] : this._defaults["truncation"]);
        if(!_truncation_in && !_truncation)
            return;
        if(_truncation && _truncation_in &&
           _truncation.position == _truncation_in.position && _truncation.ellipsis == _truncation_in.ellipsis)
            return;
        var value = this._mergeValues(_truncation_in ? _truncation_in : this._defaults.truncation, _truncation,
                                      ["position", "ellipsis"]);
        this._syncData["truncation"] = value; this._object.setSyncProperty(this._properties["truncation"], ([value.position, value.ellipsis]));
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get typography() { return (this._syncData.hasOwnProperty("typography") ? this._syncData["typography"] : this._defaults["typography"]); },
    set typography(_typography_in) {
        var _typography = (this._syncData.hasOwnProperty("typography") ? this._syncData["typography"] : this._defaults["typography"]);
        if(!_typography_in && !_typography)
            return;
        if(_typography && _typography_in &&
           _typography.kerning == _typography_in.kerning && _typography.tracking == _typography_in.tracking &&
           _typography.fontFeatures == _typography_in.fontFeatures)
            return;
        var value = this._mergeValues(_typography_in ? _typography_in : this._defaults.typography, _typography,
                                      ["kerning", "tracking", "fontFeatures"]);
        this._syncData["typography"] = value; this._object.setSyncProperty(this._properties["typography"], value);
        if (nrdp.gibbon._breaks !== undefined)
            this._widget._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    info: function info(cb) {
        var id = nrdp.gibbon._setValue(cb);
        nrdp.gibbon._invoke(this._path, "textInfo", {id: id});
    },
    _updateProperty: function _updateProperty(property, value) {
        //nrdp.log.error("Updated " + this._path + "::" + property + " " + nrdp._hasEventListener(this._widget, "renderpropertychange") + " " + nrdp._hasEventListener(this, "renderpropertychange"));
        this._syncData[property] = value;
        if(property.lastIndexOf("render", 0) == 0 &&
           (nrdp._hasEventListener(this._widget, "renderpropertychange") ||
            nrdp._hasEventListener(this, "renderpropertychange"))) {
            var renderProperty, widgetProperty;
            if(property == "renderCursorPosition") {
                renderProperty = "cursorPosition";
                widgetProperty = { property: "renderTextCursorPosition", renderProperty: "textCursorPosition" };
            } else if(property == "renderLines") {
                renderProperty = property;
                widgetProperty = { property: "renderTextLines", renderProperty: "renderTextLines" };
            } else if(property == "renderFirstLine") {
                renderProperty = "firstLine";
                widgetProperty = { property: "renderTextFirstLine", renderProperty: "textFirstLine" };
            } else if(property == "renderPadding") {
                renderProperty = property;
                widgetProperty = { property: "renderPadding", renderProperty: "renderPadding" };
            }
            if(renderProperty) {
                this._sendRenderProperty(property, renderProperty, value);
                if(widgetProperty && this._widget.sendRenderProperties)
                    this._widget._sendRenderProperty(widgetProperty.property, widgetProperty.renderProperty, value);
                return;
            } else {
                nrdp.log.error("Unhandled text.renderProperty: " + property);
            }
        }
    },
    _handleEvent: function _handleEvent(event) {
        var handled = true;
        if(event.name == "textLoaded") {
            var evt = { type: event.name, data: event.data };
            nrdp._callEventListeners(this, evt);
        } else if(event.name == "textInfo") {
            var cb = nrdp.gibbon._getValue(event.data.id);
            if (cb)
                cb(event.data.info);
        } else {
            handled = false;
        }
        return handled;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*global nrdp, nrdp_platform, GIBBON_PULL, GIBBON_PUSH, GIBBON_PUSH_APPEND, GibbonSyncObject */
nrdp.gibbon.Widget = function(id) {
    this._object = new GibbonSyncObject(this, id, nrdp.gibbon._sync_Widget);
    this._id = id;
    this._animations = { pending: {} };
    this._name = nrdp.gibbon.Widget._createName(this._id);
    this._path = "nrdp.gibbon.widgets." + this._name;
    this._children = [];
    this._callbacks = {};
    this._syncData = {};
    this._defaults = nrdp.gibbon.Widget._default;
    this._properties = nrdp._classes[this.classname].properties.byName;
};
nrdp.gibbon.Widget._createName = function(id) { return "widget" + id; };
nrdp.gibbon.Widget.ALIGN_LEFT = 0x01;
nrdp.gibbon.Widget.ALIGN_RIGHT = 0x02;
nrdp.gibbon.Widget.ALIGN_CENTER_HORIZONTAL = 0x04;
nrdp.gibbon.Widget.ALIGN_TOP = 0x10;
nrdp.gibbon.Widget.ALIGN_BOTTOM = 0x20;
nrdp.gibbon.Widget.ALIGN_CENTER_VERTICAL = 0x40;
nrdp.gibbon.Widget._encodeAlignment = function(align) {
    var result;
    if(align instanceof Array) {
        result = 0;
        for(var i = 0; i < align.length; ++i)
            result |= this._encodeAlignment(align[i]);
    } else if(typeof align === "string") {
        result = 0;
        var aligns = align.split(" ");
        for(var j = 0; j < aligns.length; ++j) {
            var a = aligns[j];
            if(a == "right")
                result |= nrdp.gibbon.Widget.ALIGN_RIGHT;
            else if(a == "left")
                result |= nrdp.gibbon.Widget.ALIGN_LEFT;
            else if(a == "top")
                result |= nrdp.gibbon.Widget.ALIGN_TOP;
            else if(a == "bottom")
                result |= nrdp.gibbon.Widget.ALIGN_BOTTOM;
            else if(a == "center" || a == "center-both")
                result |= nrdp.gibbon.Widget.ALIGN_CENTER_HORIZONTAL|nrdp.gibbon.Widget.ALIGN_CENTER_VERTICAL;
            else if(a == "center-horizontal")
                result |= nrdp.gibbon.Widget.ALIGN_CENTER_HORIZONTAL;
            else if(a == "center-vertical")
                result |= nrdp.gibbon.Widget.ALIGN_CENTER_VERTICAL;
        }
    } else {
        result = align;
    }
    return result;
};
nrdp.gibbon.Widget._default = {
    push_warnedPendingSync: false,
    effects: undefined,
    name: undefined,
    x: undefined,
    y: undefined,
    width: undefined,
    height: undefined,
    minWidth: undefined,
    minHeight: undefined,
    maxWidth: undefined,
    maxHeight: undefined,
    padding: { left: 0, top: 0, bottom: 0, right: 0, wrap: 0 },
    layout: undefined,
    parent: undefined,
    layoutStretch: 0,
    layoutSpacing: 0,
    opacity: 1.0,
    drawOrder: 0,
    scale: 1.0,
    clip: true,
    cache: undefined,
    opaque: false,
    erase: false,
    video: false,
    visible: true,
    loadImages: undefined,
    smoothScale: false,
    sendRenderProperties: false,
    sendAnimationFinished: false,
    backgroundColor: { r: 0, g: 0, b: 0, a: 0 },
    scrollX: 0,
    scrollY: 0,
    transformOriginX: 0,
    transformOriginY: 0
};
nrdp.gibbon.Widget.prototype = {
    classname: "WidgetBridge",
    constructor: nrdp.gibbon.Widget,
    _describe: function _describe() {
        var result;
        if(this.name)
            result = this.name;
        else
            result = "UNNAMED";
        result += "(" + this._id + ")";
        return result;
    },
    get effects() { return this._effects; },
    addEffect: function addEffect(type, params) {
        if(!this._effects)
            this._effects = {};
        var effectid = nrdp.gibbon._nextEffectId++;
        if(effectid > nrdp.gibbon.INT_MAX)
            effectid = 1;
        var effect = new nrdp.gibbon.Effect(this, type, effectid);
        effect.params = params ? params : {};
        this._effects[effect._name] = effect;
        return effect;
    },
    removeEffect: function removeEffect(effect) {
        var effectID = effect;
        if(effect instanceof Object)
            effectID = effect._id;
        for(var e in this._effects) {
            if(this._effects[e]._id == effectID) {
                nrdp.gibbon._invoke(this._path, "removeEffect", { effect: effectID });
                delete this._effects[e];
                break;
            }
        }
    },
    findWidget: function(id) {
        if(typeof id === "number") {
            if(this._id == id)
                return this;
        } else if(this.name == id) {
            return this;
        }
        for(var i = 0; i < this._children.length; ++i) {
            var widget = this._children[i].findWidget(id);
            if(widget)
                return widget;
        }
        return undefined;
    },
    addEventListener: function addEventListener(evt, listener) {
        if(nrdp._addEventListener(this, evt, listener)) {
            if(evt == "renderpropertychange") {
                this.sendRenderProperties = true;
            } else if(evt == "animationFinished") {
                this.sendAnimationFinished = true;
            } else if(evt == "imageLoaded" || evt == "backgroundImageLoaded") {
                var image = (evt == "imageLoaded") ? this.image : this.backgroundImage;
                if(!image._widgetImageLoadedHandler) {
                    image._widgetImageLoadedHandler = function(event) {
                        var loadEvent = { type: evt, data: event.data };
                        nrdp._callEventListeners(image._widget, loadEvent);
                    };
                    image.addEventListener("imageLoaded", image._widgetImageLoadedHandler);
                }
            }
        }
    },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    _sendRenderProperty: function _sendRenderProperty(property, renderProperty, value) {
        //nrdp.log.error("RENDEREVENT: " + " - p=" + property + ", rP=" + renderProperty + ", v=" + value);
        nrdp._callEventListeners(this, { type: "renderpropertychange",
                                         property: property,
                                         renderProperty: renderProperty,
                                         value: value });
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    _renderProperty: function _renderProperty(property) {
        if(!this.sendRenderProperties) {
            nrdp.log.error("RenderProperty(" + property + "): Used but WIDGET(" + this._id + ") has not set sendRenderProperties!");
            this.sendRenderProperties = true;
        }
        return (this._syncData.hasOwnProperty(property) ? this._syncData[property] : this._defaults[property]);
    },
    get sendRenderProperties() { return (this._syncData.hasOwnProperty("sendRenderProperties") ? this._syncData["sendRenderProperties"] : this._defaults["sendRenderProperties"]); },
    set sendRenderProperties(_sendRenderProperties) {
        if(_sendRenderProperties == this.sendRenderProperties)
            return;
        if(this._text)
            this._text.sendRenderProperties = _sendRenderProperties;
        this._syncData["sendRenderProperties"] = _sendRenderProperties; this._object.setSyncProperty(this._properties["sendRenderProperties"], _sendRenderProperties);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get sendAnimationFinished() { return (this._syncData.hasOwnProperty("sendAnimationFinished") ? this._syncData["sendAnimationFinished"] : this._defaults["sendAnimationFinished"]); },
    set sendAnimationFinished(_sendAnimationFinished) {
        if(_sendAnimationFinished == this.sendAnimationFinished)
            return;
        this._syncData["sendAnimationFinished"] = _sendAnimationFinished; this._object.setSyncProperty(this._properties["sendAnimationFinished"], _sendAnimationFinished);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get color() { return this.backgroundColor; },
    set color(_color) { this.backgroundColor = _color; },
    get backgroundColor() { return (this._syncData.hasOwnProperty("backgroundColor") ? this._syncData["backgroundColor"] : this._defaults["backgroundColor"]); },
    set backgroundColor(_color) {
        if(typeof _color === "string") {
            if(_color[0] == '#') {
                var r = parseInt(_color.substr(1, 2), 16); if(isNaN(r)) r = 0;
                var g = parseInt(_color.substr(3, 2), 16); if(isNaN(g)) g = 0;
                var b = parseInt(_color.substr(5, 2), 16); if(isNaN(b)) b = 0;
                var a = (_color.length == 9 ? parseInt(_color.substr(7, 2), 16) : 255); if(isNaN(a)) a = 255;
                _color = { r:r, g:g, b:b, a:a };
            } else {
                _color = undefined;
            }
        }
        if(_color === undefined)
            _color = { r:0, g:0, b:0, a:0 };
        else if(!(_color instanceof Object))
            _color = { r:0, g:0, b:0, a:255 };
        if(nrdp.gibbon._compareColor(_color, (this._syncData.hasOwnProperty("backgroundColor") ? this._syncData["backgroundColor"] : this._defaults["backgroundColor"])))
            return;
        var value = { r: _color.r, g: _color.g, b: _color.b, a: _color.a };
        this._syncData["backgroundColor"] = value; this._object.setSyncProperty(this._properties["backgroundColor"], nrdp.gibbon._encodeColor(value));
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get padding() { return (this._syncData.hasOwnProperty("padding") ? this._syncData["padding"] : this._defaults["padding"]); },
    set padding(_padding) {
        var encodedPadding = _padding;
        var padding = (this._syncData.hasOwnProperty("padding") ? this._syncData["padding"] : this._defaults["padding"]);
        if(_padding instanceof Object) {
            if(padding instanceof Object && _padding.top == padding.top && _padding.left == padding.left &&
               _padding.bottom == padding.bottom && _padding.right == padding.right && _padding.wrap == padding.wrap)
                return;
            _padding = { top: _padding.top, left: _padding.left, bottom: _padding.bottom, right: _padding.right, wrap: _padding.wrap };
            encodedPadding = [_padding.top, _padding.left, _padding.bottom, _padding.right, _padding.wrap];
        } else if(typeof _padding == "number") {
            if(_padding == padding)
                return;
        } else {
            return;
        }
        this._syncData["padding"] = _padding; this._object.setSyncProperty(this._properties["padding"], encodedPadding);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get name() { return (this._syncData.hasOwnProperty("name") ? this._syncData["name"] : this._defaults["name"]); },
    set name(_name) {
        if(typeof _name == "number")
            _name += "";
        if(_name && !_name.length)
            _name = undefined;
        if(_name == this.name)
            return;
        this._syncData["name"] = _name; this._object.setSyncProperty(this._properties["name"], _name);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get image() {
        if(this._image === undefined) {
            var widget = this;
            this._image = this.addForegroundImage();
        }
        return this._image;
    },
    get backgroundImage() {
        if(this._backgroundImage === undefined) {
            var widget = this;
            this._backgroundImage = this.addBackgroundImage();
        }
        return this._backgroundImage;
    },
    get images() { return this._images; },
    addForegroundImage: function addForegroundImage(obj) { return this.addImage(false, obj); },
    addBackgroundImage: function addBackgroundImage(obj) { return this.addImage(true, obj); },
    addImage: function addImage(background, obj) {
        if(!this._images)
            this._images = {};
        var imageid = nrdp.gibbon._nextImageId++;
        if(imageid > nrdp.gibbon.INT_MAX)
            imageid = 1;
        var image = new nrdp.gibbon.Image(this, background, imageid);
        if(obj) {
            for(var i in obj)
                image[i] = obj[i];
        }
        this._images[image._name] = image;
        if(background) {
            if(!this._backgroundImage)
                this._backgroundImage = image;
        } else if(!this._image) {
            this._image = image;
        }
        return image;
    },
    removeImage: function removeImage(image) {
        var imageID = image;
        if(image instanceof Object)
            imageID = image._id;
        if(image._background) {
            if(this._backgroundImage && this._backgroundImage._id == imageID)
                this._backgroundImage = undefined;
        } else if(this._image && this._image._id == imageID) {
            this._image = undefined;
        }
        for(var i in this._images) {
            if(this._images[i]._id == imageID) {
                nrdp.gibbon._invoke(this._path, "removeImage", { image: imageID });
                delete this._images[i];
                break;
            }
        }
    },
    get renderX() { return this._renderProperty("renderX"); },
    get renderY() { return this._renderProperty("renderY"); },
    get renderWidth() { return this._renderProperty("renderWidth"); },
    get renderHeight() { return this._renderProperty("renderHeight"); },
    get renderTextLines() { return this.text.renderLines; },
    get renderTextFirstLine() { return this.text.renderFirstLine; },
    get renderTextCursorPosition() { return this.text.renderCursorPosition; },
    get drawOrder() { return (this._syncData.hasOwnProperty("drawOrder") ? this._syncData["drawOrder"] : this._defaults["drawOrder"]); },
    set drawOrder(_drawOrder) {
        if(_drawOrder == this.drawOrder)
            return;
        this._syncData["drawOrder"] = _drawOrder; this._object.setSyncProperty(this._properties["drawOrder"], _drawOrder);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get minHeight() { return (this._syncData.hasOwnProperty("minHeight") ? this._syncData["minHeight"] : this._defaults["minHeight"]); },
    set minHeight(_minHeight) {
        if(_minHeight == this.minHeight)
            return;
        this._syncData["minHeight"] = _minHeight; this._object.setSyncProperty(this._properties["minHeight"], _minHeight);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get minWidth() { return (this._syncData.hasOwnProperty("minWidth") ? this._syncData["minWidth"] : this._defaults["minWidth"]); },
    set minWidth(_minWidth) {
        if(_minWidth == this.minWidth)
            return;
        this._syncData["minWidth"] = _minWidth; this._object.setSyncProperty(this._properties["minWidth"], _minWidth);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get maxHeight() { return (this._syncData.hasOwnProperty("maxHeight") ? this._syncData["maxHeight"] : this._defaults["maxHeight"]); },
    set maxHeight(_maxHeight) {
        if(_maxHeight == this.maxHeight)
            return;
        this._syncData["maxHeight"] = _maxHeight; this._object.setSyncProperty(this._properties["maxHeight"], _maxHeight);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get maxWidth() { return (this._syncData.hasOwnProperty("maxWidth") ? this._syncData["maxWidth"] : this._defaults["maxWidth"]); },
    set maxWidth(_maxWidth) {
        if(_maxWidth == this.maxWidth)
            return;
        this._syncData["maxWidth"] = _maxWidth; this._object.setSyncProperty(this._properties["maxWidth"], _maxWidth);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get loadImages() { return (this._syncData.hasOwnProperty("loadImages") ? this._syncData["loadImages"] : this._defaults["loadImages"]); },
    set loadImages(_loadImages) {
        if(_loadImages == this.loadImages)
            return;
        this._syncData["loadImages"] = _loadImages; this._object.setSyncProperty(this._properties["loadImages"], _loadImages);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get visible() { return (this._syncData.hasOwnProperty("visible") ? this._syncData["visible"] : this._defaults["visible"]); },
    set visible(_visible) {
        if(_visible == this.visible)
            return;
        this._syncData["visible"] = _visible; this._object.setSyncProperty(this._properties["visible"], _visible);
        if(nrdp.gibbon.scene.root && nrdp.gibbon.scene.root._id == this._id)
            nrdp.gibbon.scene._rootChanged();
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get smoothScale() { return (this._syncData.hasOwnProperty("smoothScale") ? this._syncData["smoothScale"] : this._defaults["smoothScale"]); },
    set smoothScale(_smoothScale) {
        if(_smoothScale == this.smoothScale)
            return;
        this._syncData["smoothScale"] = _smoothScale; this._object.setSyncProperty(this._properties["smoothScale"], _smoothScale);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get hidden() { return !this.visible; },
    set hidden(_hidden) { this.visible = !_hidden; },
    get opacity() {
        var value = (this._syncData.hasOwnProperty("opacity") ? this._syncData["opacity"] : this._defaults["opacity"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set opacity(_opacity) {
        if(isNaN(_opacity))
            _opacity = undefined;
        if(_opacity == this.opacity)
            return;
        this._setAnimatedValue("opacity", _opacity);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get clone() {
        var c = (this._syncData.hasOwnProperty("clone") ? this._syncData["clone"] : this._defaults["clone"]);
        if(c)
            return nrdp.gibbon.findWidget(c);
        return undefined;
    },
    set clone(_clone) {
        if(!_clone && !this.clone)
            return;
        if(_clone && this.clone && _clone._id == this.clone._id)
            return;
        var clone_id = _clone ? _clone._id : undefined;
        this._syncData["clone"] = clone_id; this._object.setSyncProperty(this._properties["clone"], clone_id);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    _addChild: function _addChild(child) {
        this._children.push(child);
    },
    _removeChild: function _removeChild(child) {
        for(var ch in this._children) {
            if(this._children[ch]._id == child._id) {
                this._children.splice(ch, 1);
                break;
            }
        }
    },
    get _isRoot() {
        if(nrdp.gibbon.scene.root && nrdp.gibbon.scene.root._id == this._id)
            return true;
        else if(nrdp.gibbon.scene.overlay && nrdp.gibbon.scene.overlay._id == this._id)
            return true;
        return false;
    },
    get children() { return this._children; },
    get parent() { return (this._syncData.hasOwnProperty("parent") ? this._syncData["parent"] : this._defaults["parent"]); },
    set parent(_parent) {
        if(typeof _parent === "number")
            _parent = nrdp.gibbon.findWidget(_parent);
        if(!_parent && !this.parent)
            return;
        if(_parent && this.parent && _parent._id == this.parent._id)
            return;
        if(this.parent)
            this.parent._removeChild(this);
        if(_parent) {
            this._syncData["parent"] = _parent; this._object.setSyncProperty(this._properties["parent"], ([ _parent._id, nrdp.gibbon._nextChildId++ ]));
            _parent._addChild(this);
        } else {
            this._syncData["parent"] = _parent; this._object.setSyncProperty(this._properties["parent"], _parent);
        }
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreakSubtree(_parent);
    },
    get scale() {
        var value = (this._syncData.hasOwnProperty("scale") ? this._syncData["scale"] : this._defaults["scale"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set scale(_scale) {
        if(_scale == this.scale)
            return;
        this._setAnimatedValue("scale", _scale);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get text() {
        if(!this._text) {
            this._text = new nrdp.gibbon.Text(this);
            this._text.sendRenderProperties = this.sendRenderProperties;
        }
        return this._text;
    },
    set text(_text) {
        if(_text && _text instanceof Object) {
            for(var property in _text) {
                if(_text[property] !== undefined)
                    this.text[property] = _text[property];
            }
        } else {
            this.text.contents = _text;
        }
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get clip() { return (this._syncData.hasOwnProperty("clip") ? this._syncData["clip"] : this._defaults["clip"]); },
    set clip(_clip) {
        if(_clip == this.clip)
            return;
        this._syncData["clip"] = _clip; this._object.setSyncProperty(this._properties["clip"], _clip);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get cache() { return (this._syncData.hasOwnProperty("cache") ? this._syncData["cache"] : this._defaults["cache"]); },
    set cache(_cache) {
        if(_cache == this.cache)
            return;
        this._syncData["cache"] = _cache; this._object.setSyncProperty(this._properties["cache"], _cache);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get textFirstLine() {
        if(!this._text)
            return nrdp.gibbon.Text._default.firstLine;
        return this._text.firstLine;
    },
    set textFirstLine(_textFirstLine) { this.text.firstLine = _textFirstLine; },
    get textCursorPosition() {
        if(!this._text)
            return nrdp.gibbon.Text._default.cursorPosition;
        return this._text.cursorPosition;
    },
    set textCursorPosition(_textCursorPosition) { this.text.cursorPosition = _textCursorPosition; },
    get textStyle() { return this.text; },
    set textStyle(_textStyle) {
        if(this.textStyle == _textStyle)
            return;
        if(_textStyle._widget !== undefined && _textStyle._widget !== this)
            nrdp.log.error("CopyTextStyle(" + this._id + "): From different WIDGET(" + _textStyle._widget._id + ")!");
        for(var property in _textStyle) {
            if(_textStyle[property] !== undefined)
                this.text[property] = _textStyle[property];
        }
    },
    get transformOriginX() { return (this._syncData.hasOwnProperty("transformOriginX") ? this._syncData["transformOriginX"] : this._defaults["transformOriginX"]); },
    set transformOriginX(_transformOriginX) {
        if(_transformOriginX == this.transformOriginX)
            return;
        this._syncData["transformOriginX"] = _transformOriginX; this._object.setSyncProperty(this._properties["transformOriginX"], _transformOriginX);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get transformOriginY() { return (this._syncData.hasOwnProperty("transformOriginY") ? this._syncData["transformOriginY"] : this._defaults["transformOriginY"]); },
    set transformOriginY(_transformOriginY) {
        if(_transformOriginY == this.transformOriginY)
            return;
        this._syncData["transformOriginY"] = _transformOriginY; this._object.setSyncProperty(this._properties["transformOriginY"], _transformOriginY);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get scrollX() {
        var value = (this._syncData.hasOwnProperty("scrollX") ? this._syncData["scrollX"] : this._defaults["scrollX"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set scrollX(_scrollX) {
        if(_scrollX == this.scrollX)
            return;
        this._setAnimatedValue("scrollX", _scrollX);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get scrollY() {
        var value = (this._syncData.hasOwnProperty("scrollY") ? this._syncData["scrollY"] : this._defaults["scrollY"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set scrollY(_scrollY) {
        if(_scrollY == this.scrollY)
            return;
        this._setAnimatedValue("scrollY", _scrollY);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get erase() { return (this._syncData.hasOwnProperty("erase") ? this._syncData["erase"] : this._defaults["erase"]); },
    set erase(_erase) {
        if(_erase == this.erase)
            return;
        this._syncData["erase"] = _erase; this._object.setSyncProperty(this._properties["erase"], _erase);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get video() { return (this._syncData.hasOwnProperty("video") ? this._syncData["video"] : this._defaults["video"]); },
    set video(_video) {
        if(_video == this.video)
            return;
        this._syncData["video"] = _video; this._object.setSyncProperty(this._properties["video"], _video);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get opaque() { return (this._syncData.hasOwnProperty("opaque") ? this._syncData["opaque"] : this._defaults["opaque"]); },
    set opaque(_opaque) {
        if(_opaque == this.opaque)
            return;
        this._syncData["opaque"] = _opaque; this._object.setSyncProperty(this._properties["opaque"], _opaque);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get layout() { return (this._syncData.hasOwnProperty("layout") ? this._syncData["layout"] : this._defaults["layout"]); },
    set layout(_layout) {
        var layout = this.layout;
        if(_layout instanceof Object) {
            _layout = { layout: _layout.layout, align: nrdp.gibbon.Widget._encodeAlignment(_layout.align) };
            if(layout instanceof Object && _layout.align == layout.align && _layout.layout == layout.layout)
                return;
        } else if(_layout == layout) {
            return;
        }
        this._syncData["layout"] = _layout; this._object.setSyncProperty(this._properties["layout"], _layout);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get layoutSpacing() { return (this._syncData.hasOwnProperty("layoutSpacing") ? this._syncData["layoutSpacing"] : this._defaults["layoutSpacing"]); },
    set layoutSpacing(_layoutSpacing) {
        if(_layoutSpacing == this.layoutSpacing)
            return;
        this._syncData["layoutSpacing"] = _layoutSpacing; this._object.setSyncProperty(this._properties["layoutSpacing"], _layoutSpacing);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get layoutStretch() { return (this._syncData.hasOwnProperty("layoutStretch") ? this._syncData["layoutStretch"] : this._defaults["layoutStretch"]); },
    set layoutStretch(_layoutStretch) {
        if(_layoutStretch == this.layoutStretch)
            return;
        this._syncData["layoutStretch"] = _layoutStretch; this._object.setSyncProperty(this._properties["layoutStretch"], _layoutStretch);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get rect() { return { x: this.x, y: this.y, width: this.width, height: this.height,
                          minWidth: this.minWidth, minHeight: this.minHeight, maxWidth: this.maxWidth, maxHeight: this.maxHeight }; },
    set rect(_rect_in) {
        var _rect;
        var animate;
        if(_rect_in.hasOwnProperty("animate")) {
            animate = _rect_in.animate;
        } else if(!this._canAnimate()) {
            animate = false;
        }
        var rect_props = { x: 1, y: 1, width: 1, height: 1, minWidth: 0, minHeight: 0, maxWidth : 0, maxHeight: 0 };
        for(var d in rect_props) {
            if(d in _rect_in && _rect_in[d] != this[d]) {
                var value = _rect_in[d];
                if(rect_props[d] && _rect_in[d] !== undefined) {
                    var valueAnimation = animate ? animate : this._findAnimation(d);
                    if(valueAnimation) {
                        this.startAnimation(d, undefined, value, valueAnimation.duration, valueAnimation.ease, false, valueAnimation.flags);
                        continue;
                    }
                }
                this._syncData[d] = value; this._object.setSyncProperty(this._properties[d], value);
            }
        }
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get x() {
        var value = (this._syncData.hasOwnProperty("x") ? this._syncData["x"] : this._defaults["x"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set x(_x) {
        if(isNaN(_x))
            _x = undefined;
        if(_x == this.x)
            return;
        this._setAnimatedValue("x", _x);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get y() {
        var value = (this._syncData.hasOwnProperty("y") ? this._syncData["y"] : this._defaults["y"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set y(_y) {
        if(isNaN(_y))
            _y = undefined;
        if(_y == this.y)
            return;
        this._setAnimatedValue("y", _y);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get width() {
        var value = (this._syncData.hasOwnProperty("width") ? this._syncData["width"] : this._defaults["width"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set width(_width) {
        if(isNaN(_width))
            _width = undefined;
        if(_width == this.width)
            return;
        this._setAnimatedValue("width", _width);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    get height() {
        var value = (this._syncData.hasOwnProperty("height") ? this._syncData["height"] : this._defaults["height"]);
        if(value && value instanceof Object)
            return value.value;
        return value;
    },
    set height(_height) {
        if(isNaN(_height))
            _height = undefined;
        if(_height == this.height)
            return;
        this._setAnimatedValue("height", _height);
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    dump: function dump(cb) {
        if(cb !== undefined) {
            var id = this._registerCallback(cb);
            nrdp.gibbon._invoke(this._path, "dump", { id: id });
        } else {
            nrdp.gibbon._invoke(this._path, "dump");
        }
    },
    _exec: function _exec(args) {
        nrdp.gibbon._invoke(this._path, "exec", args);
    },
    grab: function grab(cb) {
        var id = this._registerCallback(cb);
        nrdp.gibbon._invoke(this._path, "grab", { id: id });
    },
    setDisplayFlags: function setDisplayFlags(_flags) { //compat
        var opaque = false;
        var erase = false;
        var alignment = nrdp.gibbon.Image.ALIGN_NORMAL;
        var flags = _flags.split(" ");
        for (var f in flags) {
            var flag = flags[f];
            if(flag == "opaque")
                opaque = true;
            else if(flag == "erase")
                erase = true;
            else if(flag == "scale")
                alignment = nrdp.gibbon.Image.ALIGN_STRETCH;
            else if(flag == "tile")
                alignment = nrdp.gibbon.Image.ALIGN_TILE;
            else if(flag == "center")
                alignment = nrdp.gibbon.Image.ALIGN_CENTER;
        }
        this.erase = erase;
        this.opaque = opaque;
        this.image.halign = alignment;
        this.image.valign = alignment;
    },
    _setAnimatedValue: function _setAnimatedValue(property, value) {
        if(!(value instanceof Object) && value !== undefined) {
            var implicitAnimation = this._findAnimation(property);
            if(implicitAnimation && this._canAnimate())
                return this.startAnimation(property, undefined, value, implicitAnimation.duration, implicitAnimation.ease, false, implicitAnimation.flags);
        }
        if(nrdp.gibbon._debugAnimations)
            nrdp.log.error("SetAnimatedValue: " + this._id + " -> " + JSON.stringify(value));
        this._animations.pending[property] = undefined;
        this._syncData[property] = value; this._object.setSyncProperty(this._properties[property], value, true);
        return undefined;
    },
    _canAnimate: function _canAnimate() {
        if(!this.parent && !this._isRoot)
            return false;
        for(var p = this; p; p = p.parent) {
            if(!p.visible)
                return false;
        }
        return true;
    },
    _findAnimation: function _findAnimation(property) {
        var animations = this._animations;
        if(animations[property] && animations[property].duration)
            return animations[property];
        return false;
    },
    stopAnimation: function stopAnimation(property, end) {
        if(end === undefined)
            end = true;
        nrdp.gibbon._invoke(this._path, "stopAnimation", { property: property, end: end });
    },
    startAnimation: function startAnimation(property, start, end, duration, ease, append, flags) {
        if(!duration && !append) { //Nonsense, duration 0 is non-animated
            this._setAnimatedValue(property, { value: end });
            return undefined;
        }
        var value = nrdp.gibbon._createAnimation(property, start, end, duration, ease, append, flags);
        if(start === undefined)
            value.animate.previous = this[property];
        if(nrdp.gibbon._debugAnimations)
            nrdp.log.error('AnimationStarted:' + this._id + ':' + JSON.stringify(start) + ":" + value.animate.id + ":" + property + " == " + JSON.stringify(value));
        this._animations.pending[property] = value.animate.id;
        this._syncData[property] = value; this._object.setSyncProperty(this._properties[property], value, true);
        return value.animate.id;
    },
    animate: function animate(property, duration, ease, flags) {
        if(!this._animations[property]) {
            if(!duration)
                return;
        } else if(this._animations[property].duration == duration &&
                  this._animations[property].ease == ease) {
            return;
        }
        this._animations[property] = { duration: duration, ease: ease, flags: flags };
    },
    _updateProperty: function _updateProperty(property, value) {
        //nrdp.log.error("Updated " + this._path + "::" + property);
        this._syncData[property] = value;
        if(property.lastIndexOf("render", 0) == 0 && nrdp._hasEventListener(this, "renderpropertychange")) {
            var renderProperty;
            if(property == "renderX")
                renderProperty = "x";
            else if(property == "renderY")
                renderProperty = "y";
            else if(property == "renderWidth")
                renderProperty = "width";
            else if(property == "renderHeight")
                renderProperty = "height";
            if(renderProperty) {
                this._sendRenderProperty(property, renderProperty, value);
                return;
            }
            nrdp.log.error("Unhandled widget.renderProperty: " + property);
        }
        if (nrdp.gibbon._breaks !== undefined)
            this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
    },
    _maybeBreakSubtree: function _maybeBreakSubtree(obj) {
        if (obj === undefined)
            return;
        _maybeBreakSubtree(obj.parent);
        if (nrdp.gibbon._breaks[obj._id] && nrdp.gibbon._breaks[obj._id][nrdp.gibbon.DEBUGGER_SUBTREE_MODIFIED]) {
            nrdp.gibbon._breaks[obj._id][nrdp.gibbon.DEBUGGER_SUBTREE_MODIFIED]();
        }
    },
    _maybeBreak: function _maybeBreak(type) {
        if (nrdp.gibbon._breaks[this._id] && nrdp.gibbon._breaks[this._id][type]) {
            nrdp.gibbon._breaks[this._id][type]();
        }
        this._maybeBreakSubtree(this.parent);
    },
    _handleEvent: function _handleEvent(event) {
        var handled = true;
        if(event.name == "animationFinished") {
            event.data.property = nrdp._classes[this.classname].properties.byIndex[event.data.propertyID];
            var property = event.data.property;
            if(nrdp.gibbon._debugAnimations)
                nrdp.log.error((event.data.aborted ? "AnimationAborted:" : "AnimationFinished:") +
                               this._id + ':' + event.data.id + ":" + property + " == " + event.data.value +
                               (this._animations.pending[property] == event.data.id ? "*" : ""));
            if(event.data.id && this._animations.pending[property] == event.data.id) {
                if (nrdp.gibbon._breaks !== undefined)
                    this._maybeBreak(nrdp.gibbon.DEBUGGER_ATTRIBUTE_MODIFIED);
                this._syncData[property] = event.data.value;
            }
            var evt = { type: event.name, data: event.data };
            nrdp._callEventListeners(this, evt);
        } else if(event.name == "grab" || event.name == "dump") {
            this._callCallback(event);
        } else {
            handled = false;
        }
        return handled;
    },
    _registerCallback: function _registerCallback(cb) {
        var id = nrdp.gibbon._setValue(cb);
        return id;
    },
    _callCallback: function _callCallback(event) {
        if(event.data.id === undefined)
            return;
        var cb = nrdp.gibbon._getValue(event.data.id);
        nrdp.gibbon._deleteValue(event.data.id);
        if(cb)
            cb(event.data.data);
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/* global nrdp, GibbonSyncObject */
nrdp.gibbon.Effect = function(widget, type, id) {
    if(!widget._syncData.effects)
        widget._syncData.effects = {};
    this._object = new GibbonSyncObject(this, widget._id, nrdp.gibbon._sync_Effect, id, type);
    this._id = id;
    this._type = type;
    this._widget = widget;
    this._name = nrdp.gibbon.Effect._createName(this._id);
    this._path = widget._path + ".effects." + this._name;
    this._syncData = widget._syncData.effects[this._name];
    if(!this._syncData)
        this._syncData = widget._syncData.effects[this._name] = {};
    this._defaults = nrdp.gibbon.Effect._default;
    this._properties = nrdp._classes[this.classname].properties.byName;
};
nrdp.gibbon.Effect._createName = function(id) { return "effect" + id; };
nrdp.gibbon.Effect._default = {
    visible: true,
    accelerated: undefined,
    params: undefined
};
nrdp.gibbon.Effect.prototype = {
    classname: "EffectBridge",
    constructor: nrdp.gibbon.Effect,
    addEventListener: function addEventListener(evt, listener) { nrdp._addEventListener(this, evt, listener); },
    removeEventListener: function removeEventListener(evt, listener) { nrdp._removeEventListener(this, evt, listener); },
    stopAnimation: function stopAnimation(property, end) {
        if(end === undefined)
            end = true;
        nrdp.gibbon._invoke(this._path, "stopAnimation", { property: property, end: end });
    },
    startAnimation: function startAnimation(property, start, end, duration, ease, append, flags) {
        var value = nrdp.gibbon._createAnimation(property, start, end, duration, ease, append, flags);
        var params = (this._syncData.hasOwnProperty("params") ? this._syncData["params"] : this._defaults["params"]);
        if(start === undefined) {
            var previous = params[property];
            if(previous && previous instanceof Object)
                previous = previous.value;
            value.animate.previous = previous;
        }
        params[property] = value;
        var _encodedParams = {};
        _encodedParams[property] = value;
        this._syncData["params"] = params; this._object.setSyncProperty(this._properties["params"], _encodedParams, true);
        return value.animate.id;
    },
    get type() { return this._type; },
    get accelerated() { return (this._syncData.hasOwnProperty("accelerated") ? this._syncData["accelerated"] : this._defaults["accelerated"]); },
    get visible() { return (this._syncData.hasOwnProperty("visible") ? this._syncData["visible"] : this._defaults["visible"]); },
    set visible(_visible) {
        this._syncData["visible"] = _visible; this._object.setSyncProperty(this._properties["visible"], _visible);
    },
    get params() { return (this._syncData.hasOwnProperty("params") ? this._syncData["params"] : this._defaults["params"]); },
    set params(_params) {
        if((this.type == "mask" || this.type == "diff") && _params.image) //fixup
            _params.image.url = nrdp.gibbon._resolveUrl(_params.image.url);
        this._syncData["params"] = _params; this._object.setSyncProperty(this._properties["params"], _params, true);
    },
    _handleEvent: function _handleEvent(event) {
        var handled = true;
        if (event.name == "animationFinished") {
            this._syncData[event.data.property] = event.data.value;
            nrdp._callEventListeners(this, {type: event.name, data: event.data});
        } else if (event.name == "effectChanged") {
            nrdp._callEventListeners(this, {type: event.name, data: event.data});
        } else {
            handled = false;
        }
        return handled;
    }
};
/*
 * (c) 1997-2015 Netflix, Inc.  All content herein is protected by
 * U.S. copyright and other applicable intellectual property laws and
 * may not be copied without the express permission of Netflix, Inc.,
 * which reserves all rights.  Reuse of any of this content for any
 * purpose without the permission of Netflix, Inc. is strictly
 * prohibited.
 */
/*
 * all nrdp communication helpers are required to do four things:
 *
 * 1. add object to nrdp._backchannels for initialization
 * 2. provide nrdp._setProperty()
 * 3. provide nrdp._invoke()
 * 4. call nrdp._gotEvent() when an event comes in.
 */
/*global nrdp, nrdp_platform*/
(function() {
function _gotEvent(arg) {
    nrdp._sendSyncdEvent(nrdp._gotEvent, nrdp, arg);
}
function _setupBackchannel() {
    if (typeof nrdp_platform === "undefined" || typeof this.platform.jscBridgeEnabled === "undefined" || !this.platform.jscBridgeEnabled() )
        return false;
    return this.platform.jscBridgeInit(_gotEvent);
}
function _setProperty(subobj, prop, val) {
    var obj = subobj ? "nrdp." + subobj : "nrdp";
    return this.platform.jscBridgeSetProperty(obj, prop, val);
}
function _invoke(obj, method, args) {
    obj = obj ? "nrdp." + obj : "nrdp";
    return this.platform.jscBridgeInvoke(obj, method, args);
}
function _console(msg) {
    this.platform.log(msg);
}
function _atob(s, returnTypedArray) {
    return this.platform.atob(s, returnTypedArray);
}
function _btoa(s, urlSafe, returnTypedArray) {
    return this.platform.btoa(s, urlSafe, returnTypedArray);
}
function _parseJSON(json) {
    return this.platform.parseJSON(json);
}
function _mono() {
    return this.platform.mono();
}
function _pmono() {
    return this.platform.pmono();
}
function _parseXML(xml) {
    var obj = this.platform.parseXML(xml);
    if (obj) nrdp._fixXml(obj);
    return obj;
}
function _compress(data, type, binary) {
    return this.platform.compress(data, type, binary);
}
function _uncompress(data, type, returntypedarray) {
    return this.platform.uncompress(data, type, returntypedarray);
}
function _random(typed_array) {
    return this.platform.random(typed_array);
}
function _gctag(s) {
    return this.platform.gctag(s);
}
var backchannel = {
    name: "GibbonJavaScriptCore",
    isNative: true,
    init: _setupBackchannel,
    console: _console,
    setProperty: _setProperty,
    invoke: _invoke,
    atob: _atob,
    btoa: _btoa,
    mono: _mono,
    pmono: _pmono,
    atoutf8: function atoutf8(s) {
        return this.platform.atoutf8(s);
    },
    utf8toa: function utf8toa(s) {
        return this.platform.utf8toa(s);
    },
    parseJSON: _parseJSON,
    parseXML: _parseXML,
    compress: _compress,
    uncompress: _uncompress,
    random: _random,
    gctag: _gctag,
    // I know this looks weird, but running in Chrome has a problem otherwise
    platform: typeof nrdp_platform === "undefined" ? undefined : nrdp_platform
};
nrdp._backchannels.unshift(backchannel);
})();
