FREETIME_VERSION = { "tag": "GA4-2-c5a5b50", "buildNum": "dsat-2016021121:21-c5a5b50" };
/**
 * Bootstap the Application Monitoring subsystem for Panasonic.
 *
 * Panasonic closes and reopens the Freetime app as different apps are launched. This component attempts to maintain
 * a session using web storage. If the user enters an app within the timeout period then the same session ID is used
 * to initialise AM (in setConfig). If the session is stale then a new session ID is generated and startSession is
 * called.
 *
 * If sessionRefreshSecs is non-zero then a timer is set to push out the session expiry periodically.
 */
var freesat_am = freesat_am || {};

freesat_am.factory = function(appMonitor, appMonitorFailureUrl, appMonitorCustomData, sessionTimeoutSecs, sessionRefreshSecs, mode, recdataLocation, primaryRegion, secondaryRegion) {
    var initAM = function(am) {
        if (!sessionTimeoutSecs) {
             sessionTimeoutSecs = 2 * 60 * 60; // 2 hours default
        }
        if (!sessionRefreshSecs) {
            sessionRefreshSecs = 0; // disabled by default
        }

        var config = window.oipfObjectFactory.createConfigurationObject();
        var bootstrapConfig = {
            amVersion:'1.1',
            amConfig: {
                platformName:'Freesat',
                platformType:mode ? mode.toUpperCase() : '',
                optedIn: window.localStorage.getItem('marketingOptIn') === 'true',
                applicationName: 'freesat.tv',
                primaryRegion: primaryRegion,
                secondaryRegion: secondaryRegion,
                customData: appMonitorCustomData,
                deviceID: config.localSystem.deviceID,
                brand: config.localSystem.vendorName,
                model: config.localSystem.modelName,
                applicationVersion: config.localSystem.softwareVersion,
            }
        };

        // Fallback for old development TVs that sometimes don't have a deviceID
        if (!bootstrapConfig.amConfig.deviceID ||
                bootstrapConfig.amConfig.deviceID === '' ||
                bootstrapConfig.amConfig.deviceID == null ||
                bootstrapConfig.amConfig.deviceID == 'undefined') {
            bootstrapConfig.amConfig.deviceID = 'FS-ABC-01A-0000-2500';
        }

        // reuse sessionId if found and not stale...
        var doStartNewAppMonitoringSession = true;
        if (typeof window.localStorage !== 'undefined') {
            var currentTimestamp = Math.floor(new Date().getTime() / 1000);
            var newSessionId = bootstrapConfig.amConfig.deviceID + '-' + currentTimestamp;
            var sessionCookie = window.localStorage.getItem('AMF_SESSION_COOKIE');

            if (sessionCookie) {
                console.log('Freesat AMFx - existing AMF_SESSION_COOKIE is: ' + sessionCookie);

                try {
                    var cookieParts = sessionCookie.split(',');
                    var oldSessionId = cookieParts[0];
                    var lastActivityTimestamp = parseInt(cookieParts[1], 10);

                    console.log('Freesat AMFx - session cookie is ' + (currentTimestamp - lastActivityTimestamp) + ' seconds old');

                    if ((currentTimestamp - lastActivityTimestamp) >= sessionTimeoutSecs) {
                        console.log('Freesat AMFx - stale AMF_SESSION_COOKIE found (> ' + sessionTimeoutSecs + ' seconds old) so creating new session');
                        sessionCookie = newSessionId + ',' + currentTimestamp;
                    } else {
                        console.log('Freesat AMFx - fresh AMF_SESSION_COOKIE found (<' + sessionTimeoutSecs + ' seconds old) so reusing old session');
                        sessionCookie = oldSessionId + ',' + currentTimestamp;
                        doStartNewAppMonitoringSession = false;
                    }
                } catch (e) {
                    console.log('WARNING: Freesat AMFx - failed to extract AMF_SESSION_COOKIE "' + sessionCookie + '" so creating new session. Exception was: ' + e);
                    sessionCookie = newSessionId + ',' + currentTimestamp;
                }
            } else {
                console.log('Freesat AMFx - no AMF_SESSION_COOKIE found so creating new session');
                sessionCookie = newSessionId + ',' + currentTimestamp;
            }

            console.log('Freesat AMFx - setting AMF_SESSION_COOKIE to: ' + sessionCookie
                + ' (format is: sessionId,lastActivityTimestamp) - will expire in ' + sessionTimeoutSecs + 's');
            window.localStorage.setItem('AMF_SESSION_COOKIE', sessionCookie);

            // session ID is first part of the cookie in 'sessionId,lastActivityTimestamp' format
            bootstrapConfig.amConfig.sessionID = sessionCookie.split(',')[0];
        } else {
            console.log('WARNING: Freesat AMFx - no support for LocalStorage: cannot use persistent sessions');
        }

        console.log("Freesat AMFx - using following configuration:");
        console.log(JSON.stringify(bootstrapConfig, null, 4));

        am.setConfig(bootstrapConfig);

        if (doStartNewAppMonitoringSession) {
            console.log("Freesat AMFx - starting new session");
            am.startSession();
        }

        // freshen the token every minute whilst 1LG/3LG is loaded
        if (sessionRefreshSecs > 0) {
            window.setInterval(function() {
                var currentTimestamp = Math.floor(new Date().getTime() / 1000);
                var sessionCookie = window.localStorage.getItem('AMF_SESSION_COOKIE');
                if (sessionCookie) {
                    var cookieParts = sessionCookie.split(',');
                    var sessionId = cookieParts[0];
                    var lastActivityTimestamp = parseInt(cookieParts[1], 10);

                    sessionCookie = sessionId + ',' + currentTimestamp;

                    console.log('Freesat AMFx - freshening AMF_SESSION_COOKIE to: ' + sessionCookie
                         + ' (format is: sessionId,lastActivityTimestamp) - will expire in ' + sessionTimeoutSecs + 's');
                    window.localStorage.setItem('AMF_SESSION_COOKIE', sessionCookie);
                }
            }, sessionRefreshSecs * 1000);
        }
    };

    // this funciton instantiates and initialises AM, calling cb() when complete
    this.build = function(cb) {
        var am = freesat_am.application_monitor(
            appMonitorFailureUrl,
            recdataLocation,
            appMonitor,
            function() {
                try {
                    initAM(am);
                } catch (e) {
                    console.log('ERROR: failed to initialize AM: ' + e);
                }
                cb();
            }
        );
        return am;
    };
};

/**
* Copyright 2015 Freesat
* Application Measurement Framework.
* This code defines the public interface to the AMFx and loads and delegates to one fo the following implementations in order of priority:
* Preferred: First choice from where the client has selected it from receiver data.
* Fallback: The safe implementation in receiver data.
* Safe Internal: A do nothing implementation from within this file.
*/

/*
 * If the implementation throws an exception then dynamically load the safe internal implementation.
*/

 // Initialise freesat_am global name space.
var freesat_am = freesat_am || {};

// Safe internal implementation to fall back to if rec data version(s) are not available.
freesat_am.application_monitor_safe_internal_impl = function Freesat_am_safe_internal_impl(failureURL, receiverDataURL, frameworkPath) {
  this._uriParamLast = null;

  this.setURI = function(uriParam) {
    this._uriParamLast = uriParam;
  };

  this.getURI = function() {
    return this._uriParamLast;
  };

  this.setOptIn = function(optIn){
  };

  this.setConfig = function(configParam) {
  };

  this.startSession = function() {
  };

  this.endSession = function() {
  };

  this.screenView = function(uriParam, infoParam) {
    if (uriParam &&
        uriParam !== '') {
      this._uriParamLast = uriParam;
    }
  };

  this.screenEvent = function(uriParam, infoParam) {
    if (uriParam &&
        uriParam !== '') {
      this._uriParamLast = uriParam;
    }
  };

  this.techEvent = function(uriParam, infoParam) {
  };

  this.biEvent = function(uriParam, infoParam) {
  };

  this.amEvent = function(uriParam, infoParam) {
  };
};

freesat_am.application_monitor = function Freesat_am(failureURL, receiverDataURL, frameworkPath, callback) {
  var self = this;

  // Store AMFx initialisation parameters.
  this._failureURL      = failureURL;
  this._receiverDataURL = receiverDataURL;
  this._frameworkPath   = frameworkPath;
  this._callback = callback;

  // Construct receiver data URLs.
  this._preferredURL = receiverDataURL + frameworkPath;
  this._fallbackURL  = receiverDataURL + '/freetimeui/js/am/freesat_am_safe_impl.js';

  // Assign safe internal implementation by default.
  this._freesat_am_impl = new freesat_am.application_monitor_safe_internal_impl(failureURL, receiverDataURL, frameworkPath);

  this._exception = function(exception) {
    console.log('AMFx exception - ' + exception);

    console.log("Freesat AMFx - reverting to safe internal implementation.");

    this._postFailure('/v1-1/exception', exception.toString());
    this._freesat_am_impl = new freesat_am.application_monitor_safe_internal_impl(failureURL, receiverDataURL, frameworkPath);
  };

  // Public API - delegates implementation to the dynamically loaded AMFx.
  this.setURI = function(uriParam) {
    try {
      this._freesat_am_impl.setURI(uriParam);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.getURI = function() {
    try {
      return this._freesat_am_impl.getURI();
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.setOptIn = function(optIn) {
    try {
      this._freesat_am_impl.setOptIn(optIn);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.setConfig = function(configParam) {
    try {
      this._freesat_am_impl.setConfig(configParam);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.startSession = function() {
    try {
      this._freesat_am_impl.startSession();
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.endSession = function() {
    try {
      this._freesat_am_impl.endSession();
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.screenView = function(uriParam, infoParam) {
    try {
      this._freesat_am_impl.screenView(uriParam, infoParam);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.screenEvent = function(uriParam, infoParam) {
    try {
      this._freesat_am_impl.screenEvent(uriParam, infoParam);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.techEvent = function(uriParam, infoParam) {
    try {
      this._freesat_am_impl.techEvent(uriParam, infoParam);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.biEvent = function(uriParam, infoParam) {
    try {
      this._freesat_am_impl.biEvent(uriParam, infoParam);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this.amEvent = function(uriParam, infoParam) {
    try {
      this._freesat_am_impl.amEvent(uriParam, infoParam);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  // String prototype add-in.
  String.prototype.toAMStandardised = function() {
    var standardisedText = this.toLowerCase();

    // Make all initial letters capitals.
    standardisedText = standardisedText.replace(/\w\S*/g, function(txt) {
      return txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase();
    });

    // Remove punctuation.
    standardisedText = standardisedText.replace(/[^\w\s]|_/g, '');

    // Remove spaces.
    standardisedText = standardisedText.replace(/ /g,'');

    // Make first letter lowercase.
    standardisedText = standardisedText[0].toLowerCase() + standardisedText.slice(1);

    return standardisedText;
  };

  // Post AMFx load failures to remote Freesat server.
  this._postFailureRequest = function(path, message, onError) {
    if (this._failureURL && this._failureURL.valueOf() !== '') {
      var logEntry = {
        version: '1.1',
        deviceId: 'AMFx',
        timestamp: new Date(),
        api: 'exception',
        params: {uriParam: '/AMFx', infoParam: {message: message}}
      };

      console.log("Freesat AMFx - _postFailureRequest()." + message);

      var deviceID = 'unknown';
      if (freesat_am.support &&
        freesat_am.support.getDeviceID) {
        deviceID = freesat_am.support.getDeviceID();
      }

      var request = new XMLHttpRequest();
      request.open("POST", this._failureURL + path + '/' + encodeURIComponent(message), true);
      request.setRequestHeader("Content-type", "application/json");
      request.setRequestHeader("X-Freesat-FUSN", deviceID);
      request.send(JSON.stringify(logEntry, null, 4));
      request.onerror = onError;
    }
  };

  this._postFailure = function(path, message) {
    this._postFailureRequest(path, message, function() {
      console.log("Freesat AMFx - _postFailure() failure.");
      // Retry once after 10 minutes but before 30 minutes are up.
      var timeout = 10 * 60 * 1000 + (Math.random() * 20 * 60 * 1000);
      setTimeout(function() {
        self._postFailureRequest(path, 'Retry-' + message, function () {
          console.log("Freesat AMFx - _postFailure() retry failure.");
        }); }, timeout);
      });
  };

  // Synchronously load a script element to the head of the DOM.
  this._loadScript = function(id, src, onLoad, onError) {
    var existingScriptElem = document.getElementById(id);

    if (existingScriptElem) {
      document.head.removeChild(existingScriptElem);
      freesat_am.apiLog = undefined;
      freesat_am.support = undefined;
      freesat_am_config = undefined;
    }

    var scriptElem = document.createElement('script');
    scriptElem.type = 'text/javascript';
    scriptElem.src = src;
    scriptElem.async = false;
    scriptElem.id = id;

    scriptElem.onload = onLoad;
    scriptElem.onerror = onError;

    document.head.appendChild(scriptElem);
  };


  // Fallback - safe receiver data implementation.
  this._onFallbackLoad = function() {
    console.log("Freesat AMFx - Fallback load successful.");
    self._freesat_am_impl = new freesat_am.application_monitor_impl(self._failureURL, self._receiverDataURL, self._frameworkPath);
    self._callback();
  };

  this._onFallbackError = function() {
    console.log("Freesat AMFx - fallback failed, using safe internal implementation.");
    self._postFailure('/v1-1/AMFxFailedToLoad', 'Fallback AMFx failed to load. ' + self._fallbackURL);
    self._callback();
  };

  // Preferred - receiver data implementation.
  this._onPreferredLoad = function() {
    console.log("Freesat AMFx - Preferred load successful.");
    self._freesat_am_impl = new freesat_am.application_monitor_impl(self._failureURL, self._receiverDataURL, self._frameworkPath);
    self._callback();
  };

  this._onPreferredError = function() {
    console.log("Freesat AMFx - Preferred load failed, falling back to safe receiver data implementation. " + this._fallbackURL);
    try {
      self._postFailure('/v1-1/AMFxFailedToLoad', 'Preferred AMFx failed to load. ' + self._preferredURL);
      self._loadScript('FreesatAMFx', self._fallbackURL, self._onFallbackLoad, self._onFallbackError);
    }
    catch(exception) {
      self._exception(exception);
    }
  };

  this._init = function() {
    console.log("Freesat AMFx - Attempting preferred implementation from receiver data. " + this._preferredURL);
    try {
      this._loadScript('FreesatAMFx', this._preferredURL, this._onPreferredLoad, this._onPreferredError);
    }
    catch(exception) {
      this._exception(exception);
    }
  };

  this._init();

  return this;
};

/**
 * Freesat
 * @name freesat
 * @namespace freesat
 * @return {object} semi-global variables and public methods
 */
freesat = (function (){
    /** @private */
    var _currentChannel = 1;
    var _prevZone = 'video';
    var _zone = 'video';
    var _backFocusObject = null;
    var _prevSection = null;
    var startTime= 0;
    var debugTime = 0;
    var DAY = ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"],
        SHORT_MONTH = [ "Jan", "Feb", "Mar", "Apr", "May", "Jun",
        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" ];

    return {
        lockedKeys: false,
        debugVar: 'none',
        fullscreenWidth: 1280,
        fullscreenHeight: 720,
        defaultHttpTimeoutMs: 10000,
        ipServicesLoadFailed: false,
        //controllers
        master: null,
        header: null,  //refactor into controller based scope
        behind : null,
        video: null,
        front: null,
        overlay : null,
        popup: null,
        //dynamic settings
        actionData : null, //selected item
        context: 'Tvguide',
        animationMode : 'css', // js || css
        datamode : 'ip',
        backJourney: false,
        section : '',
        clearBackCookie: function(name){
            var d = new Date();
            var expires = "expires="+d.toGMTString();
            document.cookie = 'fsat'+name + "=; " + expires;
            freesat.console.log('=== ' + name + ' === deleted');
        },

        getBackCookieValue: function(name){
            var cname = 'fsat'+name + "=";
            var ca = document.cookie.split(';');
            for(var i=0; i<ca.length; i++){
                var c = ca[i].trim();
                freesat.console.log('c --- '+ c);
                if (c.indexOf(cname)===0) return c.substring(cname.length,c.length);
            }
            return "";
        },
        setBackCookieValue: function(name, value){
            var d = new Date();
            d.setTime(d.getTime()+(1*24*60*60*1000));
            var expires = "expires="+d.toGMTString();
            document.cookie = 'fsat' + name + "=" + String(value) + "; " + expires;
            freesat.console.log('=== ' + name + ' === '+ String(value));
        },

        testingip : null,
        testingHooks : null,

        get ipavailable(){
            if (this.ipServicesLoadFailed) {
                return false;
            }
            if (this.testingip) {
                return false;
            }
            return new freesat.models.NetworkInfo().isConnected();
        },
        set ipavailable(value){
            //you can not set ipavailable
        },
        screenmode: 'full',
        isAnimating: false,
        channelType: 'tv',
        //data model
        channelData: null,
        oipfElement: null,
        dataStore: null,
        dataLayer: null,
        dataInterfaceFactory: null, //fake singleton

        oipfEnabled: false,
        localConfig: null,
        pageTTLinterval: null,

        primeCurrentChannel: function(number){
            freesat.console.log('primeCurrentChannel:'+number);
            _currentChannel = number;
        },
        /**
         * Set the current channel
         * @param {number} number Current channel number
         * @memberOf freesat
         */
        setCurrentChannel: function(number){
            freesat.console.log('setCurrentChannel:'+number);
            if(freesat.video.channelChanger.channelChange(number) && freesat.datamode != 'fake'){
                _currentChannel = number;
                this.video.handleChannelChange();
                return true;
            }else{
                return false;
            }
        },
        console: function(){
            return {
                log : function(str){
                    try{
                    if(accedo.config.get('freesat_log')){
                         var dTime = new Date().getTime();
                         if(debugTime===0) {startTime = dTime;}
                         if(console)console.log(str + '  T = ' + (dTime - debugTime) + '  Total = ' + (dTime - startTime));

                        freesat.consoleStack = freesat.consoleStack || [];
                        freesat.consoleStack.push(str + '  T = ' + (dTime - debugTime) + '  Total = ' + (dTime - startTime));
                        if (freesat.consoleStack.length >= 64) {
                            freesat.consoleStack.shift();
                        }

                         debugTime = dTime;
                    }
                    }catch(e){
                        var dTime = new Date().getTime();
                        if(debugTime===0) {startTime = dTime;}
                        if(console)console.log(str + '  T = ' + (dTime - debugTime) + '  Total = ' + (dTime - startTime));
                        debugTime = dTime;
                    }
                }
            };
        }(),
        showConsole: function() {
            if (!freesat.consoleDiv) {
                var d = document.createElement('div');
                d.style.border = '3px solid #c00';
                d.style.border = '3px';
                d.style.background = 'rgba(255,0,0,0.9)';
                d.style.width = '1280px';
                d.style.height = '720px';
                d.style.left = '0px';
                d.style.top = '0px';
                d.style.position = 'absolute';
                d.style.zIndex = '201';
                d.style.color = 'white';
                d.style.fontSize = '8px';
                document.body.appendChild(d);
                freesat.consoleDiv = d;
            }

            var doLog = function() {
                var logStr = '';
                for (var i = 0; i < freesat.consoleStack.length; i++) {
                    logStr += freesat.consoleStack[i] + '<br/>'
                }
                freesat.consoleDiv.innerHTML = logStr;
            };

            if (!freesat.consoleTimer) {
                freesat.consoleTimer = window.setInterval(doLog, 1000);
            }
            doLog();
            return;
        },
        hideConsole: function() {
            if (freesat.consoleTimer) {
                window.clearInterval(freesat.consoleTimer);
                freesat.consoleTimer = null;
            }
            if (freesat.consoleDiv) {
                document.body.removeChild(freesat.consoleDiv);
                freesat.consoleDiv = null;
            }
        },
        isConsoleShowing: function() {
            if (freesat.consoleDiv) {
                return true;
            }
            return false;
        },
        getAppMonConfig: function(){
            var optIn = false;
            if (window.localStorage){
                optIn = (window.localStorage.getItem('usageDataOptIn') === "true");
            }
            return {
                optIn: optIn
            }
        },

        /**
         * Get the current channel
         * @memberOf freesat
         * @return {number} Current channel number
         */
        getCurrentChannel: function(){
            return _currentChannel;
        },

        getZone: function(){
            return _zone;
        },
        setZone: function(zone){
            _prevZone = _zone;
            _zone = zone;
        },
        getPrevSection: function () {
            return _prevSection;
        },
        setPrevSection: function(section){
            _prevSection = section;
        },
        setBackFocus: function(focusObject){
            _backFocusObject = focusObject;
        },
        moveBackFocus: function(){
            if(_backFocusObject) {
                _backFocusObject.setFocus();
                _backFocusObject.removeClass('disable');
            }
            _zone = _prevZone;
            if(freesat.actionData && freesat.actionData.timeZone === 'earlier') {
                _zone = 'earlier';
            }else{
                if (freesat.getZone() == 'now' || freesat.getZone() == 'later'){
                    freesat.oipfElement.setKeysetWithoutNumbers();
                }else{
                    freesat.oipfElement.setKeysetWithNumbers();
                }
                freesat.console.log('######## @@ freesat.moveBackFocus '+ freesat.getZone());
            }
        },
        /**
         *
         * @param {number} starttime in seconds
         * @returns {string} Today|Tomorrow|formatted date [shortDayName date shortMonthName]
         */
        formatDate: function(starttime) {
            var start = starttime * 1000,
                startDate = new Date(start),
                startDateTime = new Date(startDate.getFullYear(), startDate.getMonth(), startDate.getDate()).getTime(),
                currentTime = new Date().getTime(),
                currentDate = new Date(currentTime),
                currentDateTime = new Date(currentDate.getFullYear(), currentDate.getMonth(), currentDate.getDate()).getTime(),
                dayInMilliseconds = 24 * 60 * 60 * 1000,
                d;

            if (currentDateTime === startDateTime) {
                d = 'Today';
            } else if ((currentDateTime + dayInMilliseconds) === startDateTime) {
                d = 'Tomorrow';
            } else {
                d = DAY[startDate.getDay()] + ' ' + startDate.getDate() + ' ' + SHORT_MONTH[startDate.getMonth()];
            }
            return d;

        },
        /**
         *
         * @param {number} starttime in seconds
         * @param {number} days
         * @returns {string} Today|Tomorrow|formatted date [shortDayName date shortMonthName]
         */
        formatShortDate: function(starttime, days) {
            var timeInMilliseconds = starttime * 1000,
                startDate = new Date(timeInMilliseconds),
                d;

            startDate.setDate(startDate.getDate() + days);
            d = DAY[startDate.getDay()] + ' ' + startDate.getDate();

            return d;

        },

        getEasyTime: function(seconds) {
            var date = new Date(seconds * 1000).toISOString().substr(11,5); //

            return date;
        },

        loadRemainingSections: function () {
            if (freesat.allSectionsLoaded) return;
            trace("loadRemainingSections", true);
            var sectionsLoaded = 0;
            var callback = function () {
                trace(this.id, false);
                ++sectionsLoaded;
                if (sectionsLoaded === freesat.sectionScripts.length) {
                    trace("loadRemainingSections");
                    freesat.allSectionsLoaded = true;
                }
            };
            for (var i = 0; i < freesat.sectionScripts.length; i++) {
                var _script = document.createElement("script");
                _script.type = "text/javascript";
                _script.onload = callback;
                _script.id = freesat.sectionScripts[i];
                _script.src = freesat.sectionScripts[i];
                trace(freesat.sectionScripts[i], true);
                document.body.appendChild(_script);
            }
        }
    };
})();
window._XDK_configs = {
    "core.console.lv": 1,

    "ui.appsystem.resolution": {
        "width": 1280,
        "height": 720
    },
    "ui.appcontroller.enablehistory": false,
    "freesat_log" : true,
    "config_url" : "file:///transient/freetime/freetimed/ram/freetimeconf.xml",

    //Do not modify or remove this object, it is used by the build tool
    "build": {
        "version": "@build-version@",
        "date": "@build-date@",
        "git": {
            "hash": "@git-hash@",
            "branch": "@git-branch@"
        }
    }
};
/*jslint browser: true, devel: true, evil:true */

/**
 * @fileOverview Base functionality for XDK
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a
 */

/*global accedo: true */
/**
 * @ignore
 */
(function(window){
    freesat.console.log('XDK code start ');
    trace('LoadToXDKApp'); //end tag for loading to the Start of XDK
    trace('XDKstart', true); //start tag for XDK load
    document.onload = function(){
        var testcontainerElement = document.createElement('div');
        testcontainerElement.setAttribute('id','testEle');
        document.body.appendChild(testcontainerElement);
    }

    var _configFile = (function(){
        var i,metas=document.getElementsByTagName("meta"),length; 
        for(i=0,length = metas.length;i < length;i++){
            if(metas[i].name==="XDK.config" ) {
                return metas[i].content;
            }
        }
        return null;
    }()),
    
    _createTransport = function() {
        return new XMLHttpRequest();
    },
    
    
    /**
     * The private config object.
     * @ignore
     */
    _config = {
        /**
         * @private
         * @ignore
         */
        _data: (function(){
            
            
            var transport, configs,
            /**
             * @ignore
             */
            __injectPlatform = function(configs){
                if (window._XDK_platform) {
                    configs["device.target.platform"] = configs["device.target.platform"] || window._XDK_platform;
                    delete window._XDK_platform;
                }
            };
            
            // check if the it's config by the new method 
            if (window._XDK_configs) {
                configs = window._XDK_configs;
                delete window._XDK_configs;

                __injectPlatform(configs);
                return configs;
            }
            
            // fallback to the old method
            if (_configFile===null) {
                return {};
            }
            
            transport = _createTransport();
            
            transport.open('GET', _configFile, false);
            transport.send('');
            try {
                configs = eval("("+transport.responseText+");"); 
            }
            catch(e) {
                if (typeof console !== "undefined") {
                    console.warn("failed to load config file:url="+_configFile+"&content="+ transport.responseText+ "&error="+e);
                }
                //when there is no console,it won't pop up an alert because it may crash the apps on some platforms.
                configs = {};
            }

            __injectPlatform(configs);
            return configs;
        }()),
        /**
         * Get configuration value if exist, otherwise return default value
         * @name get
         * @function 
         * @memberof accedo.module:config
         * @param {Object} key configuration entry key
         * @param {Object} defaultValue default value to return if the configuration entry does not exist
         * @return {Object} retrieved configuration entry value
         * @static
         * @public
         */
        get: function(key, defaultValue) {
            if (typeof key==="undefined") {
                return this._data;
            }
            if (this._data.hasOwnProperty(key)) {
                return this._data[key];
            }
            return defaultValue;
        },
        /**
         * Set configuration value 
         * @name set
         * @function 
         * @memberof accedo.module:config
         * @param {Object} key configuration entry key
         * @param {Object} value configuration entry value to set
         * @static
         * @public
         */
        set: function(key, value) {
            if (accedo.Object.isUndefined(key) || accedo.Object.isUndefined(value)) {
                return;
            }
            this._data[key] = value;
        }
    },
    _loadingScripts = {},
    LOAD_ID_PREFIX = "__js__",
    LOAD_TIMEOUT = _config.get('core.scriptTimeout', 20000),
    _loadCount = 0,
    _nextLoadIndex = 0,
    DL_STARTED = 0,
    DL_FAILURE = 1,
    DL_SUCCESS = 2,
    DL_EXECUTED = 3,
    /**
     * @ignore
     */
    _onScriptReadyState = function (loadID) {
        var loadScriptObj = _loadingScripts[loadID],
        node = document.getElementById(loadID),
        head = document.getElementsByTagName('head')[0],
        child;


        accedo.console.debug("_onScriptReadyState: "+ loadScriptObj.scriptPath);

        if(!node.readyState || /loaded|complete/.test(node.readyState)) {
            /**
             * @ignore
             */
            accedo.console.log("script "+loadID+" loaded");
            if (loadScriptObj.timeoutHandle) {
                clearTimeout(loadScriptObj.timeoutHandle);
            }
            loadScriptObj.status = DL_EXECUTED;

            node.onload = node.onreadystatechange = null;

            _runNextScript();

            //Prevent memory leaks in Samsung devices using a left-hand operand
            setTimeout(function(){
                try{
                    node.parentNode.removeChild(node);
                }
                catch(e) {
                    accedo.console.warn("Exception removing head script node." + e);
                }
                child = null;
                node = null;
            },15);
        }

    },
    /**
     * @ignore
     */
    _onScriptDownloadTimeout = function(loadID) {
        _loadingScripts[loadID].status = DL_FAILURE;

        _runNextScript();
    },
    /**
     * @ignore
     */
    _runNextScript = function () {
        var loadID = LOAD_ID_PREFIX + _nextLoadIndex,
        loadObj = _loadingScripts[loadID],
        i,se;

        //no downloads pending to execute
        if (!loadObj) {
            return;
        }

        /**
         *@ignore
         */
        function execQueuedFunctions() {
            /*
             * Execute queued functions
             */
            if (loadObj.fnQueue) {

                for (i=0;i<loadObj.fnQueue.length; i++) {
                    loadObj.fnQueue[i]();
                }
            }
        }

        if(loadObj.status === DL_STARTED) {
            //still downloding, wait.
            return;
        }
        //Script download failure
        else if(loadObj.status === DL_FAILURE) {
            accedo.console.debug("script download failure: "+loadObj.scriptPath);
            if (loadObj.onFailure) {
                loadObj.onFailure();
            }
            execQueuedFunctions();
        }

        if (loadObj.status === DL_EXECUTED) {
            if (loadObj.onSuccess) {
                loadObj.onSuccess();
            }
            execQueuedFunctions();
        }

        _nextLoadIndex++;
        _runNextScript();

    },
    _loadingDefinition = {},

    /**
     * @ignore
     */
    _onDefinitionDefined = function(definition) {
        if (_loadingDefinition[definition]) {
            var i, defLoadingObj = _loadingDefinition[definition];

            if (defLoadingObj.dependeciesLoadingHandle) {
                clearTimeout(defLoadingObj.dependeciesLoadingHandle);
            }

            for (i=0;i<defLoadingObj.onSuccess.length;i++) {
                defLoadingObj.onSuccess[i](definition);
            }

            delete _loadingDefinition[definition];
        }
    },
    /**
     * @ignore
     */
    _onDefinitionDefineFailure = function(definition) {
        accedo.console.warn("Failed to load deinfinition: "+definition);

        var i, defLoadingObj = _loadingDefinition[definition];
        try{
            for (i=0;i<defLoadingObj.onFailure.length;i++) {
                defLoadingObj.onFailure[i](definition);
            }
        }catch (e){
            freesat.console.log('DEF err ' + definition);
        }


        delete _loadingDefinition[definition];
    },

    /**
     * Global unique ID
     * @name _guid
     * @private
     */
    _guid = 1;

    /**
     * @name accedo
     * @module
     */
    window.accedo = {
        /**
         * The current version of XDK in string
         * @name version
         * @public
         * @static
         * @memberof module:accedo
         */
        version: '1.7.1',
        
        /**
         * The build number of the current version
         * @name versionBuild
         * @static
         * @memberof module:accedo
         */
        buildNumber: "buildNumber: dsat-2015022616:35-cbc6f2d",
        repoTag: "Sprint-11-DSAT-Release-K-1.11.header",
        /***
         * Returns the reference of the root scope
         * 
         * @name rootScope
         * @public 
         * @static
         * @memberof module:accedo
         */
        rootScope:  (function() {
            return this; // || (1,eval)('this')
        }()), // Compatible with ES3, ES5 . To make even ES5-strict compatible, uncomment the ending of the above line (and remove ;)
    
        /**
         * Special object used to stop certain loop utility for accedo.Array
         * @name $break
         * @public
         * @static
         * @memberof module:accedo
         */
        $break: {},
        /**
         * Returns the rootPath. The path to the 'js/' folder
         * 
         * @name rootPath
         * @public
         * @static
         * @memberof module:accedo
         */
        rootPath: (function() {
            var scriptTag = document.getElementById('XDK'),fullPath;
            if(scriptTag){
                fullPath = scriptTag.getAttribute('src');
                return fullPath.substr(0, fullPath.lastIndexOf('accedo/'));
            }else{
                scriptTag = document.getElementById('MIN');
                fullPath = scriptTag.getAttribute('src');
                return fullPath.substr(0, fullPath.lastIndexOf('merge_minify'));
            }
        }()),
        
        /**
         * An empty function that does nothing
         * @name emptyFunction
         * @public
         * @static
         * @memberof module:accedo
         */
        emptyFunction: function() {},

        /**
         * Gets the next global unique ID
         * @name getGuid
         * @public
         * @static
         * @memberof module:accedo
         */
        getGuid: function() {
            return _guid++;
        },

        /**
         * @ignore
         */
        K: function(x) {
            return x;
        },
        /**
         * The accedo config object.
         * @name config
         * @memberof accedo
         * @public
         * @module
         */
        config: _config,
        /**
         * Console object for logging
         * @name console
         * @memberof accedo
         * @module
         */
        console: {
            // XDK CHANGE
            // LV_DEBUG: 0,
            // LV_INFO: 1,
            // LV_LOG:2,
            // LV_WARNING: 3,
            // LV_ERROR: 4,
            // level: _config.get('core.console.lv' ,2),
            /**
             * Send a log message
             * @name log
             * @memberof accedo.module:console
             * @function
             * @param {Mixed} string 1st log message
             * @param {Mixed} [...] nth log message
             */
            log: function(){
                // XDK CHANGE
                // if (this.level > accedo.console.LV_LOG) {
                //     return;
                // }
            
                // var args = Array.prototype.slice.call(arguments),
                // dConsole = accedo.getNamespaceReference('accedo.device.console');
                // if (dConsole){
                //     accedo.device.console.log.apply(accedo.device.console,args);
                // }
                // else if (typeof console !== "undefined") {
                //     freesat.console.log(args + '  T = ' + new Date().getTime());
                // }
            },
            /**
             * Send a info message
             * @name info
             * @memberof accedo.console
             * @function
             * @param {Mixed} string log message
             * @param {Mixed} [...] nth log message
             * 
             */
            info: function(){
                // XDK CHANGE
                // if (this.level > accedo.console.LV_INFO) {
                //     return;
                // }
            
                // var args = Array.prototype.slice.call(arguments),
                // dConsole = accedo.getNamespaceReference('accedo.device.console');
                // if (dConsole){
                //     accedo.device.console.info.apply(accedo.device.console,args);
                // }
                // else if (typeof console !== "undefined") {
                //     freesat.console.log(args + '  T = ' + new Date().getTime());
                // }
            },
            /**
             * Send an error message
             * @name error
             * @memberof accedo.module:console
             * @function
             * @param {Mixed} string log message
             * @param {Mixed} [...] nth log message
             * @public
             */
            error: function(){
                // XDK CHANGE
                // if (this.level > accedo.console.LV_ERROR) {
                //     return;
                // }
            
            
                // var args = Array.prototype.slice.call(arguments),
                // dConsole = accedo.getNamespaceReference('accedo.device.console');
                // if (dConsole){
                //     accedo.device.console.error.apply(accedo.device.console,args);
                // }else if (typeof console !== "undefined") {
                //     freesat.console.log(args + '  T = ' + new Date().getTime());
                // }
            },
            /**
             * Send a debug message
             * @name debug
             * @memberof accedo.module:console
             * @function
             * @param {Mixed} string log message
             * @param {Mixed} [...] nth log message
             * @public
             */
            debug: function(msg){
                // XDK CHANGE
                // if (this.level > accedo.console.LV_DEBUG) {
                //     return;
                // }
            
                // var args = Array.prototype.slice.call(arguments),
                // dConsole = accedo.getNamespaceReference('accedo.device.console');
                // if (dConsole){
                //     accedo.device.console.debug.apply(accedo.device.console,args);
                // }
                // else if (typeof console !== "undefined") {
                //     console.log(args.join(",") + '  T = ' + new Date().getTime());
                // }
            },
            /**
             * Send a warning message
             * @name warn
             * @memberof accedo.module:console
             * @function
             * @param {Mixed} string log message
             * @param {Mixed} [...] nth log message
             * @public
             */
            warn: function(msg){
                // XDK CHANGE
                // if (this.level > accedo.console.LV_WARN) {
                //     return;
                // }
            
                // var args = Array.prototype.slice.call(arguments),
                // dConsole = accedo.getNamespaceReference('accedo.device.console');
                // if (dConsole){
                //     accedo.device.console.warn.apply(accedo.device.console,args);
                // }
                // else if (typeof console !== "undefined") {
                //     console.log(args.join(",") + '  T = ' + new Date().getTime());
                // }
            }
        },
    
        /**
         * The detected platform
         * @name platfrom
         * @memberof module:accedo
         * @static
         * @public
         */
        platform: _config.get("device.target.platform") || /*deprecated*/(function() {
            var platform = "unknown", agent = navigator.userAgent.toLowerCase();
            if (typeof navigator !== "undefined" && typeof navigator.userAgent !== "undefined" ) {
                if (agent.indexOf("maple") !== -1) {
                    platform = "samsung";
                }
                else if (agent.indexOf("lg") !== -1) {
                    platform = "lg";
                }
                else if(agent.indexOf("sagem") !== -1){
                    platform = "sagem";
                }
                else if(agent.indexOf("nettv") !== -1){
                    platform = "philips";
                }
                //opera TV Store userAgent Opera/9.80/ (Linux i686;U; en) Presto/2.9.167 Version/11.5
                else if(agent.indexOf("opera") !== -1){
                    platform = "opera";  
                }
                else if(agent.indexOf("boxee") !== -1){
                    platform = "boxee";
                }
                else if(agent.indexOf("playstation") !== -1){
                    platform = "ps3";
                }
                else if(agent.indexOf("aquosbrowser") !== -1){
                    platform = "sharp";
                }
                else if ((agent.indexOf("windows nt") !== -1) ||
                    (agent.indexOf("linux") !== -1) || 
                    (agent.indexOf("macintel") !== -1) ||
                    (agent.indexOf("macintosh") !== -1)
                    ) {
                    platform = "workstation";
                }
            }
            return platform;
        }()),
    
        ScriptFragment: '<script[^>]*>([\\S\\s]*?)<\/script>',
        JSONFilter: /^\/\*-secure-([\s\S]*)\*\/\s*$/,
    
        /**
         *  Utility function to extract a path string into namespace & identifier 
         * 
         * @name extractPath
         * @param {String} path defining string
         * @return {Object} obj
         * @return {String} obj.namespace Namespace
         * @return {String} obj.identifier Identifier Name
         * @memberof module:accedo
         * @function
         * @example 
         * accedo.extractPath("accedo.ui.widget.Button"); //-> { namespace: "accedo.ui.widget", identifier: "Button" }
         * 
         */
        extractPath: function(path) {
            var namespace, identifier,
            dot = path.lastIndexOf('.');
            if (dot === -1) {
                namespace = '';
                identifier = path;
            }
            else {
                namespace = path.substr(0, dot);
                identifier =  path.substr(dot+1);
            }
      
            return {
                namespace: namespace,
                identifier: identifier
            };
        },
    
        /**
         * Gets the namespace reference in the given scope.
         * @param {String} namespace - String namespace identifier
         * @param {Object} [scope] - Optional scope. If none given - root scope used.
         * @return Reference or undefined
         * @name getNamespaceReference
         * @memberof module:accedo
         * @function
         */
        getNamespaceReference: function(namespace, scope) {
            scope = scope || accedo.rootScope;
            try{
                if(namespace.indexOf(':')>-1){
                    if(namespace.indexOf('file:///')>-1){
                        namespace = namespace.split('file:///')[1];
                    }else if(namespace.indexOf('http://')>-1){
                        namespace = namespace.split('http://')[1];
                    }
                    if(namespace.indexOf('src/')>-1){
                        namespace = namespace.split('src/')[1];
                    }
    //                namespace = namespace.toLowerCase();
                    while(namespace.indexOf('/') > -1){
                        namespace = namespace.replace('/','.');
                    }
                    if(namespace.substring(namespace.length-3, namespace.length) === '.js'){
                        namespace = namespace.substring(0, namespace.length-3);
                    }
                }
            }catch(e){
                return;
            }
            return (new Function("scope","try { return scope." + namespace + "; } catch(ex) { return undefined; }"))(scope);
        },
    
        /**
         * Checks if the namespace if available in the scope.
         * @param {String} namespace - String namespace identifier
         * @param {Object} [scope] - Optional scope. If none given - root scope used.
         * @return true or false
         * @name isNamespaceDefined
         * @memberof module:accedo
         * @function
         */
        isNamespaceDefined: function(namespace, scope) {
            return !!accedo.getNamespaceReference(namespace, scope);
        },

        /**
         * @ignore
         */
        _onScriptReady: function(loadId){
            _onScriptReadyState(loadId); // calls the private method
        },
    
        /**
         * Load a script dynamically. The execution order of the scripts loaded by this function shall be preserved.
         * 
         * @name loadScript
         * @public 
         * @memberof module:accedo
         * @function
         * @param {String} scriptPath
         * @param {Object} [options] - Optional
         * @param {Function} [options.onSuccess] Callback function on success 
         * @param {Function} [options.onFailure] Callback function on failure
         * @param {Boolean} [options.async] Weather to download asynchrounously. Default: true.
         * @example
         *  accedo.loadScript("js/lib/emile.js")
         * 
         */
        loadScript: function (scriptPath, options) {

            var loadID = LOAD_ID_PREFIX + _loadCount++,s,head,
            
            timeoutHandle = setTimeout(function(){
                if (_loadingScripts[loadID]) {
                    accedo.console.warn("Load script timeout: " + _loadingScripts[loadID].scriptPath);
                
                    _onScriptDownloadTimeout(loadID);
                }
            },LOAD_TIMEOUT),
            
            loadObj = {
                async : true,
                scriptPath: scriptPath,
                timeoutHandle: timeoutHandle,
                status : DL_STARTED
            };
        
            if (options) {
                if (typeof options.onSuccess !== "undefined") {
                    loadObj.onSuccess = options.onSuccess;
                }
                if (typeof options.onFailure !== "undefined") {
                    loadObj.onFailure = options.onFailure;
                }
                if (typeof options.async !== "undefined") {
                    loadObj.async = options.async;
                }
            }        
            
            //use document.write
            //opera retains script tags excution as insertion order
            if (!(accedo.Env && accedo.Env._domready) && navigator.userAgent.toLowerCase().indexOf("opera")===-1) {

                _loadingScripts[loadID] = loadObj;
                
                document.write("<script type='text/javascript' id='"+loadID+"' src='"+loadObj.scriptPath+"' onreadystatechange='accedo._onScriptReady(\""+ loadID+"\")' onload='accedo._onScriptReady(\""+ loadID+"\")' ><\/script>");
            }
            else {
        
                _loadingScripts[loadID] = loadObj;
                s = document.createElement('script');
                head = document.getElementsByTagName('head')[0];
                s.type = 'text/javascript';
                s.src = loadObj.scriptPath;
                s.async = loadObj.async;
                s.id = loadID;
                s.onreadystatechange = s.onload = function() {
                    _onScriptReadyState(loadID);
                };

                    // use body if available. more safe in IE
                (document.body || head).appendChild(s);
            }
        },
        /**
         * Run a function at the position of the current script loading queue
         *
         * @function
         * @name run
         * @param {Function} fn - Function
         * @memberof module:accedo
         * 
         */
        run: function (fn) {
            var lastLoadID = LOAD_ID_PREFIX + (_loadCount -1),
            loadObj = _loadingScripts[lastLoadID];
        
            //when there's no scriptd pending to execute
            if (!loadObj || loadObj.status === DL_EXECUTED) {
                if (fn) {
                    fn();
                }
                return;
            }
        
            if (accedo.Object.isUndefined(loadObj.fnQueue)) {
                loadObj.fnQueue = [];
            }
            loadObj.fnQueue.push(fn);
        },
    
        /**
         *  Create namespace if not exists. And define the identifier at the namespace 
         *  with the provided value. 
         *  
         *  @name create
         *  @param {String} namespace - namespace 
         *  @param {String} [identifier] - identifier
         *  @param {Mixed} [value] - value
         *  @param {Object} [options] - options container
         *  @param {Object} [options.cb] - callback to be called at the moment defined.
         *  (before definition dependencies callbacks).
         *  @param {Scope} [options.scope] - scope for the namespace to define at
         *  
         *  @memberof module:accedo
         *  @function
         *  @example
         *  accedo.create('accedo.ui.test','Object', {});
         *  accedo.ui.test.Object; // -> {}
         */
        create: function (namespace, identifier, value, options) {
            var scope  = (options && options.scope)?options.scope:accedo.rootScope,
            
            cb  = (options && options.cb)?options.cb:null,
            
            internalCallback = function() {
                var i, len, entities, root, def;

                //Prepare namespace
                if (namespace) {
                    entities = namespace.split(".");
                    root = scope;
                    len = entities.length;
                    for (i = 0; i < len; i++) {
                        if (typeof root[entities[i]] === "undefined") {
                            root[entities[i]] = {};
                        }
                        root = root[entities[i]];
                    }
                    if (identifier) {
                        root[identifier] = value;
                        if (cb) {
                            cb();
                        }
                        //def dependencies
                        def = [namespace,identifier].join('.');
                        _onDefinitionDefined(def);
                    }
                }
            };

            internalCallback();
        },
    
        /**
         * Defines a namespace module.
         * @param {String} namespace - Namespace identifier
         * @param {Array} dependencies - Array of string dependencies
         * @param {Function} constructor - Constructor function
         * @param {Object} [scope] - Optional scope to define in.
         * @name define
         * @memberOf module:accedo
         * @function
         */
        define: function(namespace, dependencies, constructor, scope) {
            scope = scope || accedo.rootScope;

            /**
             * @private
             * @function
             */
            var internalCallback = function() {
                var i, len, entities, root;

                //Prepare namespace
                if (namespace) {
                    entities = namespace.split(".");
                    root = scope;
                    len = entities.length;
                    for (i = 0; i < len; i++) {
                        if (typeof root[entities[i]] === "undefined") {
                            root[entities[i]] = {};
                        }
                        root = root[entities[i]];
                    }
                }

                try {
                    if (namespace) {
                        (new Function("constructor", namespace + " = constructor();"))(constructor);
                        _onDefinitionDefined(namespace);
                    } else {
                        constructor();
                        _onDefinitionDefined(namespace);
                    }
                } catch(ex) {
                    throw new Error('### Unable to prepare namespace "' + namespace + '" dependency (missing dependency?), ' + ex);
                }
            };

            if(dependencies) {
                accedo.loadDefinitions(dependencies, {
                    onSuccess : internalCallback,
                    onFailure : function(definition){
                        accedo.console.error('### Unable to prepare namespace "' + namespace + '", dependency "' + definition + '" failed!');
                    }
                });
            } else {
                internalCallback();
            }
        },
    
        /**
         * Remove the definition from memory
         * This is useful if you delete an object corresponding to a loaded script and, later on,
         * want them to be loaded again.
         * Note this doesn't unload the objects that were created by scripts! Do it yourself before or after
         * calling this function.
         * @param {String} definition - definition string
         * @name undefine
         * @memberof module:accedo
         * @function
         * @public
         * @static
         */
        undefine: function(definition) {
            var obj, ns, ref = accedo.getNamespaceReference(definition);
            if (!ref) {
                return;
            }
        
            obj = accedo.extractPath(definition);
        
            if (obj.namespace.length === 0) {
                delete accedo.rootScope[obj.identifier];
            } else {
                ns = accedo.getNamespaceReference(obj.namespace);
                delete ns[obj.identifier];
            }
        
        },
        
        /**
         * Load a definition dynamically
         *
         * @function
         * @name loadDefinition
         * @param {String} definition - definition string e.g. 'accedo.ui.Button'
         * @param {Object} [options]
         * @param {Function} [options.onSuccess] Callback function on success 
         * @param {Function} [options.onFailure] Callback function on failure
         * @memberof module:accedo
         * 
         */
        loadDefinition : function (definition, options) {
            var defLoadingObj, loadOptions, scriptPath;

            accedo.console.debug("loadDefinition: " + definition);
            if(accedo.isNamespaceDefined(definition)) {
                accedo.console.debug("definition already loaded: " + definition);
                if (options && options.onSuccess) {
                    options.onSuccess(definition);
                }
                return;
            }
            if (_loadingDefinition[definition]) {
                accedo.console.debug("definition loading: " + definition);
                if (options && options.onFailure) {
                    _loadingDefinition[definition].onFailure.push(options.onFailure);
                }
                if (options && options.onSuccess) {
                    _loadingDefinition[definition].onSuccess.push(options.onSuccess);
                }
                return;
            }
        
            defLoadingObj = {
                onSuccess: [],
                onFailure: []
            };
        
            if (options && options.onSuccess) {
                defLoadingObj.onSuccess.push(options.onSuccess);
            }
            if (options && options.onFailure) {
                defLoadingObj.onFailure.push(options.onFailure);
            }
        
            if (definition.indexOf('accedo')===0) {
                scriptPath = accedo.rootPath;
                scriptPath = this.subScriptPath(scriptPath, definition);
            }else if (definition.indexOf('file')===0) {
                scriptPath = definition ;;
            }else if (definition.indexOf('http')===0) {
                scriptPath = definition ;
            }else {
                scriptPath = "js/";
                scriptPath = this.subScriptPath(scriptPath, definition);
            }
            if(definition.indexOf(':')>-1){
                if(definition.indexOf('file:///')>-1){
                    definition = definition.split('file:///')[1];
                }else if(definition.indexOf('http://')>-1){
                    definition = definition.split('http://')[1];
                }
                if(definition.indexOf('src/')>-1){
                    definition = definition.split('src/')[1];
                }
//                definition = definition.toLowerCase();
                while(definition.indexOf('/') > -1){
                    definition = definition.replace('/','.');
                }
                if(definition.substring(definition.length-3, definition.length) === '.js'){
                    definition = definition.substring(0, definition.length-3);
                }
            }

            _loadingDefinition[definition] = defLoadingObj;

            loadOptions = {
                /**
                 *@ignore
                 */
                onSuccess: function() {
                    if (accedo.isNamespaceDefined(definition)) {
                        _onDefinitionDefined(definition);
                    }else{
                        //assuming there's still dependencies loading
                        defLoadingObj.dependeciesLoadingHandle = setTimeout(function(){
                            if (options && options.onFailure){
                                options.onFailure(definition);
                            }
                                accedo.console.error("Definition dependencies loading timeout: "+definition);
                        },LOAD_TIMEOUT);
                    }
                },
                /**
                 *@ignore
                 */
                onFailure: function() {
                    _onDefinitionDefineFailure(definition);
                }
            };


//            scriptPath += definition.replace(/\./g, '/').replace(/\/[A-Z]([A-Z0-9a-z]*)$/,function(m) {
//                return m.substr(0,1).concat((m.substr(1,1).toLowerCase()).concat(m.substr(2)));
//            }) + '.js';
            accedo.loadScript(scriptPath, loadOptions);
        },

        subScriptPath: function(scriptPath, definition){
            scriptPath += definition.replace(/\./g, '/').replace(/\/[A-Z]([A-Z0-9a-z]*)$/,function(m) {
                return m.substr(0,1).concat((m.substr(1,1).toLowerCase()).concat(m.substr(2)));
            }) + '.js';
            return scriptPath
        },

        /**
         * Load a list of definitions dynamically
         * @function
         * @name loadDefinitions
         * @param {Array} arr - array of definition strings
         * @param {Object} [options]
         * @param {Function} [options.onSuccess] Callback function on success 
         * @param {Function} [options.onFailure] Callback function on failure
         * @memberof module:accedo
         * 
         */
        loadDefinitions: function(arr, options) {
            var i, __defDefined, arrCopy;
            options = options || {};
        
            //Check if deps is already defined
            for (i=0; i< arr.length; i++) {
                if (accedo.isNamespaceDefined(arr[i])) {
                    accedo.console.debug(arr[i] + " removed");
                    arr.splice(i, 1);
                    i--;
                }
            }
        
            //load script no load dependencies
            if (arr.length===0) {
                if (options && options.onSuccess) {
                    options.onSuccess();
                }
                return;
            }
        
            //dependencies handling
            /**
             *@ignore
             */
            __defDefined = function (def) {
                var idx = arr.indexOf(def);
                if (idx===-1) {
                    accedo.console.error("Unexpected dependencies defining flow! definition="+def + '&arr='+arr.join(','));
                }
                arr.splice(idx,1);
                if (arr.length===0) {
                    if (options && options.onSuccess) {
                        options.onSuccess();
                    }
                }
            };
        
            //clone the array to prevent synchronous script loading (for lg)
            arrCopy = accedo.Array.clone(arr);
        
            for (i=0; i< arrCopy.length;i++) {
                accedo.loadDefinition(arrCopy[i],{
                    onSuccess: __defDefined,
                    onFailure: options.onFailure
                });
            }
        },
    
        /**
         * Provides a simple idiom for trying out blocks of code in sequence.
         * Such a sequence of attempts usually represents a downgrading approach to obtaining a given feature.
         * @return The first succeeded functions result
         * @name Try
         * @memberof module:accedo
         * @function
         */
        Try : function() {
            var returnValue, lambda, i = 0, length;

            for (length = arguments.length; i < length; i++) {
                lambda = arguments[i];
                try {
                    returnValue = lambda();
                    break;
                } catch (e) {
                }
            }

            return returnValue;
        }
    
    };
}(window));

accedo.console.log("======== Accedo XDK version "+accedo.version+" Build "+accedo.buildNumber+" ========");
//Utility Classes
accedo.loadDefinition('accedo.Object');
accedo.loadDefinition('accedo.Class');
accedo.loadDefinition('accedo.Mixin');
accedo.loadDefinition('accedo.mixin.EventDispatcher');
accedo.loadDefinition('accedo.String');
accedo.loadDefinition('accedo.Fn');
//accedo.loadDefinition('accedo.RegExp');
//accedo.loadDefinition('accedo.Date');
accedo.loadDefinition('accedo.Array');
accedo.loadDefinition('accedo.Hash');
accedo.loadDefinition('accedo.Dom');
accedo.loadDefinition('accedo.Element');
//accedo.loadDefinition('accedo.QueryString');
accedo.loadDefinition('accedo.device.Manager');
accedo.loadDefinition('accedo.VKey');
accedo.loadDefinition('accedo.Env');
accedo.loadDefinition('accedo.Ajax');
accedo.loadDefinition('accedo.AjaxRequest');

//Load the mvc system
accedo.loadDefinition('accedo.ui.AppSystem');
accedo.loadDefinition('accedo.ui.HistoryManager');
/*jslint browser: true, devel: true, strict: false */
/*global accedo: false */
/**
 * Object utility functions.
 * If configuration parameter "extendObject" is set, functions will extend naitive Object.
 * @name Object
 * @memberof accedo
 * @module
 * @author <a href="mailto:alex@accedobroadband.com">Alexej Kubarev</a>
 */
accedo.Object = (function() {

    /**
     * A reference to naitive toString function.
     * @private
     */
    var _toString = Object.prototype.toString,
    NULL_TYPE = 'Null',
    UNDEFINED_TYPE = 'Undefined',
    BOOLEAN_TYPE = 'Boolean',
    NUMBER_TYPE = 'Number',
    STRING_TYPE = 'String',
    OBJECT_TYPE = 'Object',
    FUNCTION_CLASS = '[object Function]',
    BOOLEAN_CLASS = '[object Boolean]',
    NUMBER_CLASS = '[object Number]',
    STRING_CLASS = '[object String]',
    ARRAY_CLASS = '[object Array]',
    DATE_CLASS = '[object Date]',
    OBJECT_CLASS = '[object Object]',
    HASH_CLASS = '[object Hash]',
    _type = function (o) {
        switch(o) {
            case null:
                return NULL_TYPE;
            case (void 0):
                return UNDEFINED_TYPE;
        }
        var type = typeof o;
        switch(type) {
            case 'boolean':
                return BOOLEAN_TYPE;
            case 'number':
                return NUMBER_TYPE;
            case 'string':
                return STRING_TYPE;
        }
        return OBJECT_TYPE;
    },
    _objUtils;
    
    _objUtils = {
        /**
         * Extends a dest with properties from src.
         * Possible deep copy/extending by setting deep = true.
         * @name extend
         * @memberof accedo.module:Object
         * @function
         * @param {Object} dest Destination object to be extended
         * @param {Object} src Source object with properties to extend with
         * @param {Boolean} [deep] Deep copy
         * @return {Object} extended object
         * @static
         * @public
         */
        extend: function(dest, src, deep) {
            var objIsArray, orig, copy, clone, att;
            for (att in src) {

                orig = dest[att];
                copy = src[att];

                // Prevent never-ending loop
                if ( dest === copy ) {
                    continue;
                }
                
                objIsArray = _objUtils.isArray(copy);
                
                if(deep === true && copy && (objIsArray || _objUtils.isPlainObject(copy))) {
                    if(objIsArray) {
                        objIsArray = false;
                        clone = orig && _objUtils.isArray(orig) ? orig : [];
                    } else {
                        clone = orig && _objUtils.isPlainObject(src) ? orig : {};
                    }

                    // Never move original objects, clone them
                    dest[att] = _objUtils.extend(clone, copy, true);
                } else if (!_objUtils.isUndefined(copy)) {
                    dest[att] = copy;
                }
            }
            return dest;
        },

        /**
         * Clones an object.
         * @name clone
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @param {Boolean} [deep] Deep copy
         * @return {Object} a copy of the object
         * @static
         * @public
         */
        clone: function(obj, deep) {
            return _objUtils.extend({}, obj, deep);
        },

        /**
         * Checks is object is undefined.
         * @name isUndefined
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is undefined, false otherwise
         * @static
         * @public
         */
        isUndefined: function(obj) {
            return typeof obj === "undefined";
        },

        /**
         * Checks if the object is a plain object, created using {} or new Object().
         * @name isPlainObject
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to test.
         * @return {Boolean} true if parameter is an object, false otherwise
         * @static
         * @public
         */
        isPlainObject: function(obj) {
            if(!obj || typeof obj !== "object" || obj.nodeType) {
                return false;
            }

            // Not own constructor property must be Object
            if ( obj.constructor &&
                !obj.hasOwnProperty("constructor") &&
                !obj.constructor.prototype.hasOwnProperty("isPrototypeOf") ) {
                return false;
            }

            //Loop through all properties to get access to the last one.
            var key;
            for ( key in obj ) {}

            //Own properties are iterated first, so it is enough to look at the last one.
            return (key === undefined || obj.hasOwnProperty(key));
        },

        /**
         * Checks if the object is an array.
         * @name isArray
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is an Array, false otherwise
         * @public
         * @static
         */
        isArray: function(obj) {
            //Information  on the best way to perform this type of check is taken from:
            // http://perfectionkills.com/instanceof-considered-harmful-or-how-to-write-a-robust-isarray/

            return _toString.call(obj) === ARRAY_CLASS;
        },

        /**
         * Checks if the object is Function.
         * @name isFunction
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a Function, false otherwise
         * @public
         * @static
         */
        isFunction: function(obj) {
            return _toString.call(obj) === FUNCTION_CLASS;
        },

        /**
         * Checks if the object is a string.
         * @name isString
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a String, false otherwise
         * @public
         * @static
         */
        isString: function(obj) {
            return _toString.call(obj) === STRING_CLASS;
        },
        /**
         * Checks if the object is a number.
         * @name isNumber
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a Number, false otherwise
         * @public
         * @static
         */
        isNumber: function(obj) {
            return _toString.call(obj) === NUMBER_CLASS;
        },
        /**
         * Checks if the object is a boolean.
         * @name isBoolean
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a Boolean, false otherwise
         * @public
         * @static
         */
        isBoolean: function(obj) {
            return _toString.call(obj) === BOOLEAN_CLASS;
        },
        /**
         * Checks if the object is a date.
         * @name isDate
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a Date, false otherwise
         * @public
         * @static
         */
        isDate: function(obj) {
            return _toString.call(obj) === DATE_CLASS;
        },

        /**
         * Checks if the object is a DataSource.
         * @name isDataSource
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a DataSource, false otherwise
         * @public
         * @static
         */
        isDataSource: function(obj) {
            return (typeof obj==="object") && (obj.toString() === '[object DataSource]');
        },

        /**
         * Checks if the object is a DOM element.
         * @name isDOMElement
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a DOM element, false otherwise
         * @public
         * @static
         */
        isDOMElement: function(obj) {
            //Browsers not supporting W3 DOM2 don't have HTMLElement and
            //an exception is thrown and we end up here. Testing some
            //properties that all elements have. (works on IE7)
            return (typeof obj === "object") &&
            (obj.nodeType === 1) && (typeof obj.style === "object") &&
            (typeof obj.ownerDocument === "object") &&
            (typeof obj.nodeName === "string");
        },

        /**
         * Checks if the object is an instance of accedo.Hash.
         * @name isHash
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is a Hash, false otherwise
         * @public
         * @static
         */
        isHash: function(obj) {
            try {
                return obj.toString() === HASH_CLASS;
            } catch (ex) {
                return false;
            }
        },
        
        /**
         * Tests whether an object is empty or not (an empty object is {})
         * @name isEmpty
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is empty, false otherwise
         * @public
         * @static
         */
        isEmpty: function(obj) {
            var prop;
            for(prop in obj) {
                if(obj.hasOwnProperty(prop)) {
                    return false;
                }
            }

            return true;
        },

        /**
         * Tests whether an object is null
         * @name isNull
         * @memberof accedo.module:Object
         * @function
         * @param {Object} obj Object to check
         * @return {Boolean} true if object is empty, false otherwise
         * @public
         * @static
         */
        isNull: function(obj){
            return obj === null;
        },
        
        /**
         * Returns a JSON string. (yet to test whether it works fine for all cases)
         * @author <a href="mailto:cheung.chunho@accedobroadband.com">Cheung Chun Ho</a>
         * @name toJSON
         * @memberof accedo.module:Object
         * @function
         * @param {Object} object Object to check
         * @return {String} a string of the object
         * @public
         * @static
         */
        toJSON:  function(object) {
            var type = typeof object,
            results = [],
            i = 0, 
            value, 
            property, 
            len;
            
            switch (type) {
                case 'undefined':
                case 'function':
                case 'unknown':
                    return;
                case 'boolean':
                    return object.toString();
                case 'string' :
                    return '"' + object.replace(/"/g,'\\"') + '"';
                case 'number' :
                    return object.toString();
            }

            if(this.isArray(object)) {
                results = [];
                len = object.length;

                for(; i < len; i++) {
                    value = accedo.Object.toJSON(object[i]);
                    if (!this.isUndefined(value)) {
                        results.push(value);   
                    }
                }
                return '[' + results.join(',') + ']';
            }
            if(object === null){
                return 'null';
            }
            if (accedo.Object.isFunction(object.toJSON)) {
                return object.toJSON();
            } 
            if (accedo.Object.isDOMElement(object)) {
                return;
            }

            results = [];
            for (property in object) {
                value = accedo.Object.toJSON(object[property]);
                if (!accedo.Object.isUndefined(value)) {
                    results.push('"' + property.toString() + '":' + value);
                }
            }

            return '{' + results.join(',') + '}';
        },
        
        /**
         * Returns an array of keys.
         * @name keys
         * @memberof accedo.module:Object
         * @function
         * @param {Object} object Object to check
         * @return {Array} an array contain all keys
         * @public
         * @static
         */
        keys: function (object) {
            if (_type(object) !== OBJECT_TYPE) {
                throw new TypeError();
            }
            var results = [],property ;
            for (property in object) {
                if (object.hasOwnProperty(property)) {
                    results.push(property);
                }
            }
            return results;
        }
    };
   
    return _objUtils;
}());
/*jslint browser: true, devel: true,newcap:false    */
/*global accedo: false */

/**
 * Class-based OOP implementation for javascript
 * @name Class
 * @memberof accedo
 * @module
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 */
accedo.Class = (function() {
    /**
     * @ignore
     */
    var IS_DONTENUM_BUGGY = true,p,_extendFn,_create,__create,_onFailure,addMethods,addStaticMethods,abstractFn;
    for (p in {
        toString: 1
    }) {
        if (p === 'toString') {
            IS_DONTENUM_BUGGY = false;
        }
    }

    
    /**
     * Internal utility function to extend only functions from source 
     * 
     * @function
     * @private
     * @ignore
     */
    _extendFn = function (destination, source) {
        var k;
        for (k in source) {
            if (accedo.Object.isFunction(source[k])){
                destination[k] = source[k];
            }
        }
    };
    
    
    
    /**
     * @private
     * @ignore
     */
    function Subclass() {}
    
    
    /**
     * accedo.Class.create creates a class and returns a constructor function for instances     
     * of the class. Calling the constructor function (typically as part of a new 
     * statement) will invoke the class's initialize method.
     * 
     * If a subclass overrides an instance method declared in a superclass, the 
     * subclass's method can still access the original method. To do so, declare
     * the subclass's method as normal, but insert $super as the first argument.
     * This makes $super available as a method for use within the function.
     * To extend a class after it has been defined, use accedo.Class#addMethods. 
     * 
     * @name create
     * @memberof accedo.module:Class
     * @function
     * @param {String} classDef - Definition string (optional) e.g. 'accedo.ui.Button'.
     * @param {String} [parentClassDef] - Parent class's definition string (if applicable) e.g. 'accedo.ui.Container'.
     * @param {Array} [dependencies] - Array of definition string. Will be loaded automatically before this class is defined.
     * @param {Object|Function} static - Static declaration of the class.  For function, it will be executed and 
     *  the return object will be used.
     * @param {Object|Function|String} [members] Member declarations of the class. For function, it will be executed and 
     *  the return object will be used. For string, it is treated as definition string, loaded if definition not ready.
     * @param {Object|Function|String} [...] More member declarations to add/overload to the class
     * @return {Class}
     * @static
     * 
     */
    /**
     * Stop NetBeans from complaining
     * @ignore
     */
    _create = function () {
        var classPath = null, classDef = null, parent = null, properties = Array.prototype.slice.call(arguments), 
        dependencies = null, statics = null, members = [], toLoad = [], i, parentDepList=[], length, ref,member;
        try{
            if (accedo.Object.isString(properties[0])) {
                classPath = properties.shift();
                classDef = accedo.extractPath(classPath);
            }
            else {
                accedo.console.error("Class definition input missing");
            }
        
            if (accedo.Object.isString(properties[0])) {
                parent = properties.shift();
                ref = accedo.getNamespaceReference(parent);
                if (!ref) {
                    toLoad.push(parent);
                    parentDepList.push(parent);
                }
                else {
                    parent = ref;
                }
            }
        
            //Check dependencies
            if (accedo.Object.isArray(properties[0])) {
                dependencies = properties.shift();
                // toLoad = toLoad.concat(dependencies);
                // parentDepList = parentDepList.concat(dependencies);
            }
        
            //Check statics declaration
            statics = properties.shift();
        
            if (!accedo.Object.isPlainObject(statics) && !accedo.Object.isFunction (statics)) {
                accedo.console.error('statics param definition must be an object');
            }
        
            for ( i= 0, length = properties.length; i < length; i++){
                if (accedo.Object.isString(properties[i])){
                    ref = accedo.getNamespaceReference(properties[i]);
                
                    if(!ref){
                        toLoad.push(properties[i]);
                    }
                    else {
                        properties[i] = ref;
                    }
                }
                members.push(properties[i]);
            }
        
            /**
             * @ignore
             */
            __create = function(){
                var i, parentDepRefs = [];
        
                if (parent && accedo.Object.isString(parent)){
                    parent = accedo.getNamespaceReference(parent);
                }
                /**
             *@ignore
             */
                function klass() {
                    if (accedo.Object.isUndefined(this.__initDefaults)){
                        accedo.console.log("This is a Class definition function, it shouldn't be called directly");
                        return;
                    }
                    this.__initDefaults.apply(this);
                    this.__initMixins.apply(this);
                    this.initialize.apply(this, arguments);
                    //to do post initialize after the class is initialized
                    if(this.postInitialize){
                        this.postInitialize.apply(this);
                    }
                }        
                
                // Prepare the parent and dependencies as arguments
                for (i=0;i<parentDepList.length;i++) {
                    parentDepRefs.push(accedo.getNamespaceReference(parentDepList[i]));
                }
                
                if (parent) {
                    _extendFn(klass, parent);
            
                    Subclass.prototype = parent.prototype;
                   
                    klass.prototype = new Subclass();
                    parent.__subclasses.push(klass);
            
                    // Copy ancestor's default values to instance
                    klass.__defaults = accedo.Object.clone(parent.__defaults);
                    klass.__mixinInitializers = Array.prototype.slice.call(parent.__mixinInitializers);
                }
                else {
                    accedo.Object.extend(klass, accedo.Class.Methods);
            
                    klass.__defaults = {};
                    klass.__mixinInitializers = [];
                }
        
                klass.__superclass = parent;
                klass.__subclasses = [];
                klass.__definition = classPath;
        
                //run the statics method to create the static declaration
                if (accedo.Object.isFunction(statics)) {
                    statics = statics.apply(null,parentDepRefs);
                } 
        
                klass.addStaticMethods(statics);
        
        
                //Add per object class functionalites here
                klass.addMethods({
                    getDefinition: function() {
                        return this._class;
                    },
                    __initDefaults: function() {
                        accedo.Object.extend(this,this.constructor.__defaults,true);
                    },
                    __initMixins: function() {
                        var i;
                        for (i=0;i<this.constructor.__mixinInitializers.length; i++) {
                            this.constructor.__mixinInitializers[i].apply(this);
                        }
                    },
                    _class: classPath 
           
                });
        
                
                for (i=0;i< members.length; i++) {
                    member = members[i];
                    if (accedo.Object.isFunction(member)) {
                        member = member.apply(null,parentDepRefs);
                    }
                    else if (accedo.Object.isString(member)) {
                        member = accedo.getNamespaceReference(member);
                        if (!member) {
                            accedo.console.error("Member declaration error: (class=" + classDef.namespace + '.' + classDef.identifier + "). "+member+ " is undefined");
                        }
                    }
                    klass.addMethods(member);
                }

                if (!klass.prototype.initialize) {
                    klass.prototype.initialize = accedo.emptyFunction;
                }

                klass.prototype.constructor = klass;
            
                if (accedo.Object.isFunction(klass.prototype.deinit)) {
                    klass.prototype.destructor = klass.prototype.deinit;
                }
                accedo.create(classDef.namespace, classDef.identifier, klass, {
                    cb: function(){
                        if (accedo.Object.isFunction(klass.main)) {
                            klass.main();
                        }
                    }
                });
            
                accedo.console.debug("Class created: "+ classPath);
            };

            _onFailure = function(){
                accedo.console.warn("Dependencies loading failed for class: "+ classPath + ", possible failures: ["+ toLoad.join(",") + "]");
            };
            
            accedo.console.debug(classPath + " pending on ["+ toLoad.join(",") + "]") ;   
        
            accedo.loadDefinitions(toLoad, {
                onSuccess: __create,
                onFailure: _onFailure
            });
            
        } catch(e){
            accedo.console.error('load class error: '+'classPath= '+classPath+', error= '+e.stack?e.stack:e);
        }
    };
    
    
    /**
    * Add/Overload class members
    * 
    * @name addMethods
    * @memberof accedo.module:Class
    * @function
    * @param {String} source Member definitions of the class
    * @return {Class}
    * @static
    */
   
    /**
    * Ignore for JSDoc
    * @ignore
    */
    addMethods = function (source) {
        var ancestor = this.__superclass && this.__superclass.prototype,i,length,property,value,method,wrapFunc,
        properties = accedo.Object.keys(source);

        if (IS_DONTENUM_BUGGY) {
            if (source.toString !== Object.prototype.toString){
                properties.push("toString");
            }
            if (source.valueOf !== Object.prototype.valueOf){
                properties.push("valueOf");
            }
        }
        wrapFunc = function(m) {
            return function() {
                return ancestor[m].apply(this, arguments);
            };
        };
        for (i = 0, length = properties.length; i < length; i++) {
            property = properties[i];
            value = source[property];
            
            /* 
             * Handle Mixin initializers
             */
            if (property === "initMixin") {
                this.__mixinInitializers.push(value);
                
            }else{
            
                if (ancestor && accedo.Object.isFunction(value) &&
                    accedo.Fn.argumentNames(value)[0] === "$super") {
                    method = value;
                    value = accedo.Fn.wrap(wrapFunc(property),method);
                    value.valueOf = accedo.Fn.bind(method.valueOf,method);
                    value.toString = accedo.Fn.bind(method.toString,method);
                }

                if (accedo.Object.isFunction(value)) {
                    this.prototype[property] = value;
                }
                else {
                    this.__defaults[property] = value;
                }
            }
            
        }

        return this;
    };
    
    /**
    * Add/Overload static  members
    * 
    * @name addStaticMethods
    * @memberof accedo.module:Class
    * @function
    * @param {String} source Static definitions of the class
    * @returns {Class}
    * @static
    */
   
    /**
    * Ignore for JSDoc
    * @ignore
    */
    addStaticMethods= function(source) {
        var ancestor = this.__superclass,i,length, property,value,method,wrapFunc,
        properties = accedo.Object.keys(source);

        if (IS_DONTENUM_BUGGY) {
            if (source.toString !== Object.prototype.toString){
                properties.push("toString");
            }
            if (source.valueOf !== Object.prototype.valueOf){
                properties.push("valueOf");
            }
        }
        wrapFunc = function(m) {
            return function() {
                return ancestor[m].apply(this, arguments);
            };
        };
        for (i = 0, length = properties.length; i < length; i++) {
            property = properties[i];
            value = source[property];
            
            if (ancestor && accedo.Object.isFunction(value) &&
                accedo.Fn.argumentNames(value)[0] === "$super") {
                method = value;
                value = accedo.Fn.wrap(wrapFunc(property),method);

                value.valueOf = accedo.Fn.bind(method.valueOf,method);
                value.toString = accedo.Fn.bind(method.toString,method);
            }
            
            this[property] = value;
        }

        return this;
    };
    
    
    
    /**
     * Reference function for absract function declaration
     * @name abstractFn
     * @memberof accedo.module:Class
     * @function
     * @public
     * @static
     */
    
    /**
     * Ignore for JSDoc
     * @ignore
     */
    abstractFn = function (){
        accedo.console.warn("Abstract function called! ");
    };
    
    return {
        _eventCallbacks: {},
        create: _create,
        Methods: {
            addMethods: addMethods,
            addStaticMethods: addStaticMethods
        },
        
        abstractFn: abstractFn
    };
}());
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Mixin Support
 * Standardizing mixin creation
 * @class Mixin Support
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @name Mixin
 * @memberof accedo
 * @interface
 */
accedo.Mixin = (function() {
    var _create = function() {
        var classDef = null, properties = Array.prototype.slice.call(arguments), mixin =null;
        
        if (accedo.Object.isString(properties[0])) {
            classDef = accedo.extractPath(properties.shift());
        }
        else {
            accedo.console.error("Class definition input missing");
        }
        
        if (!accedo.Object.isPlainObject(properties[0]))
        {
            accedo.console.error("Expecting object for mixin implementation");
        }
        
        mixin = properties[0];
        
        accedo.create(classDef.namespace, classDef.identifier, mixin);
        
        return mixin;
    };
    
    return {
        /**
         * Factory method for Mixin creation
         *
         * @name create
         * @memberof accedo.Mixin
         * @function
         * @param {String} [classDef] defining string (optional)
         * @param {Object} [members] Member definitions of the mixin
         * @return {Object}
         *
         */
        create: _create
    };
}());
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Mixin implementation of Observer Model
 * @constructor
 * @name EventDispatcher
 * @memberof accedo.mixin
 * @class Event dispatcher functionality
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 */
accedo.Mixin.create('accedo.mixin.EventDispatcher', {
    
    /**
     * A hashmap for event handling
     * @private
     * @field
     * 
     */
    _eventCallbacks: null,
    
    /**
     * A hashmap for event handling
     * @public
     * @protected
     * @memberof accedo.mixin.EventDispatcher#
     */
    initMixin: function() {
        this._eventCallbacks = new accedo.Hash({});
    },

    /**
     * Gets a list of handlers for given type
     * @name _getHandlersForType
     * @param {String} Type description
     * @return {Array} Handlers for given type
     * @function
     * @private
     * @memberof accedo.mixin.EventDispatcher#
     */
    _getHandlersForType: function(type) {
        var handlers = this._eventCallbacks.get(type);
        if(!handlers && !accedo.Object.isArray(handlers)) {
            handlers = [];
        }

        return handlers;
    },
    
    

    /**
     * Dispatches an event with additional data.
     * @name dispatchEvent
     * @param {accedo.ui.Evt | accedo.Env | String } type String type to dispatch
     * @param {Object} data Memo object.
     * @function
     * @public
     * @memberof accedo.mixin.EventDispatcher#
     * 
     */
    dispatchEvent: function(type, data) {
        if(this.onDispatchEvent && !this.onDispatchEvent(type, data)){
            accedo.console.log("Event blocked for type=" + type + " data=" + data);
            return;
        }

        var onces= [], obj, i, handlers = this._getHandlersForType(type);

                
        if(accedo.Object.isUndefined(data)){
            data = {};
        }
        
        for (i=0;i< handlers.length; i++) {
            obj = handlers[i];
            
            if(accedo.Object.isFunction(obj.handler)) {
                if (obj.context) {
                    obj.handler.call(obj.context,data);
                }
                else {
                    obj.handler(data);
                }
            }
            
            //Remove handlers that is listen for once only
            if (obj.once) {
                onces.push(obj);
            }
        }
        
        for (i=0;i<onces.length;i++) {
            this.removeEventListener(type,onces[i].handler, onces[i].context);
        }
    },
    
    /**
     * Registers an event listener for a certain type, and at the end it will call onListenerAdded function which passing the type name and do further action
     * @name addEventListener
     * @param {accedo.ui.Evt | accedo.Env | String } type Event type
     * @param {Function} handler Event listener function
     * @param {Object} [context] Event listener function's context
     * @param {Boolean} [once] Event listener will be called for only once
     * @returns {Object|false} obj - Handler for event removal or false if Event handler 
     * is added before
     * @returns {Function} obj.handler - reference of event listener function
     * @returns {Object} [obj.context] - reference of event listener function's context
     * @public
     * @function
     * @memberof accedo.mixin.EventDispatcher#
     */
    addEventListener: function(type, handler, context, once) {
        
        if (!once){
            once = false;
        }
        
        var i, handlers = this._eventCallbacks.get(type), obj;
        if(!handlers || !accedo.Object.isArray(handlers)) {
            handlers = [];
            this._eventCallbacks.set(type, handlers);
        }
        
        obj = {
            handler : handler,
            once : once
        };
        
        if (context){
            obj.context = context;
        }
        
        //Do not add handler if it is already registred
        for (i=0;i<handlers.length; i++) {
            if (handlers[i].handler === handler && handlers[i].context === context){
                return false;
            }
        }
        
        handlers.push(obj);
        
        if (this.onListenerAdded) {
            this.onListenerAdded(type);
        }
        
        return obj;
    },

    /**
     * Removes an event listener and at the end it will call onListenerRemoved function which passing the type name and do further action
     * @name removeEventListener
     * @param {accedo.ui.Evt | accedo.Env | String } type Event type
     * @param {Function} handler Event listener function
     * @function
     * @public
     * @memberof accedo.mixin.EventDispatcher#
     */
    removeEventListener: function(type, handler, context) {
        var i, handlers = this._eventCallbacks.get(type);
        if (handlers && handlers.length) {
            
            //Do not add handler if it is already registred
            for (i=0;i<handlers.length; i++) {
                if (handlers[i].handler === handler && handlers[i].context === context){
                    break;
                }
            }
            
            if(i < handlers.length) {
                handlers.splice(i, 1);
            }
        }
        
        if (this.onListenerRemoved) {
            this.onListenerRemoved(type);
        }
    },

    /**
     * Removes all event listeners for a type
     * @param {accedo.ui.Evt | accedo.Env | String } type Event type
     * @name removeAllListeners
     * @function
     * @public
     * @memberof accedo.mixin.EventDispatcher#
     */
    removeAllListeners: function(type) {
        if (!accedo.Object.isUndefined(type)){
            this._eventCallbacks.unset(type);
        }else{
            this._eventCallbacks = new accedo.Hash({});
        }
    }
});
/*jslint regexp: true, browser: true, devel: true, evil:true */
/*global accedo: false, window:false*/
/**
 * @desc Utility functions for working with String.
 * @name String
 * @memberof accedo
 * @static
 * @module
 * @author <a href="mailto:alex@accedobroadband.com">Alexej Kubarev</a>
 */
accedo.Class.create('accedo.String', {
    /**
     * Insert a string to a given position
     * @name insert
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with
     * @param {int} position Number of position to insert (Default last)
     * @param {String} phrase A phrase to inserted (Default '')
     * @return {String} Inserted string
     * @static
     */
    insert: function(str, position, phrase) {
        position = accedo.Object.isUndefined(position) ? str.length : position;
        phrase = accedo.Object.isUndefined(phrase) ? '' : phrase;
        
        return str.slice(0, position) + phrase + str.slice(position, str.length);
    },
    
    /**
     * Truncates a string to a given length, with optional truncation
     * @name truncate
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with
     * @param {int} length Number of characters to strip at (Default 30)
     * @param {String} truncation A truncation(ending) string. (Default '...')
     * @return {String} Truncated string
     * @static
     */
    truncate: function(str, length, truncation) {
        length = accedo.Object.isNumber(length) ? length : 30;
        truncation = accedo.Object.isUndefined(truncation) ? '...' : truncation;
        var resultLen = length - truncation.length;
        resultLen = resultLen < 0 ? 0 : resultLen;
        return str.length > length ? str.slice(0, resultLen) + truncation : String(str);
    },

    /**
     * Strips all leading and trailing whitespaces from a string.
     * @name strip
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with
     * @return {String} String with striped whitespaces
     * @static
     */
    strip: function(str) {
        return str.replace(/^\s+/, '').replace(/\s+$/, '');
    },

    /**
     * Strips all tags from the given string.
     * This only strips tags that are not namespace prefixed.
     * This does not remove tag content (i.e '<script>var a=2</script>' will be striped to 'var a=2').
     * Use accedo.utils.String.stripScripts() to remove script contents.
     * @name stripTags
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with
     * @return {String} A new string without tags
     * @static
     */
    stripTags: function(str) {
        return str.replace(/<\w+(\s+("[^"]*"|'[^']*'|[^>])+)?>|<\/\w+>/gi, '');
    },

    /**
     * Strips a string of things that look like an HTML script blocks.
     * @name stripScripts
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with
     * @return {String} A new string without script tags and their content
     * @static
     */
    stripScripts: function(str) {
        return str.replace(new RegExp('<script[^>]*>([\\S\\s]*?)<\/script>', 'img'), '');
    },

    /**
     * Converts HTML special characters to their entity equivalents.
     * @name escapeHTML
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with
     * @return {String} String with escaped HTML
     * @static
     */
    escapeHTML: function(str) {
        return str.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
    },

    /**
     * Converts the entity forms of special HTML characters to their normal form.
     * @name unescapeHTML
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with.
     * @return {String} String with unescaped HTML special characters
     * @static
     */
    unescapeHTML: function(str) {
        return str.replace(/&lt;/g, '<').replace(/&gt;/g, '>').replace(/&amp;/g, '&');
    },

    /**
     * Turns an regular expression in plain string format into an escaped format that can be comsumed by the Javascript RegExp object.
     * e.g. "\w" will be turned into "\\\w"
     * @name escapeRegex
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to perform regular expression escaping
     * @return {String} the escaped regular expression string
     * @public
     * @static
     */
    escapeRegex : function(str) {
        return String(str).replace(/([.*+?\^=!:${}()|\[\]\/\\])/g, '\\$1');
    },

    /**
     * Converts a string separated by dashes into a camelCase equivalent. For
     * instance, `'foo-bar'` would be converted to `'fooBar'`.
     * @name camelize
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with.
     * @return {String} Camelized string
     * @static
     */
    camelize: function(str) {
        return str.replace(/-+(.)?/g, function(match, chr) {
            return chr ? chr.toUpperCase() : '';
        });
    },


    /**
     * Converts a string from camelCase to decamelized equivalent. For
     * instance, `'fooBar'` would be converted to `'foo_bar'`.
     * @name decamelize
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with.
     * @param {String} separator String to separate the words.
     * @return {String} Decamelized string
     * @static
     */
    decamelize: function(str, separator) {
        separator = separator || '_';
        return str.replace(/([a-z])([A-Z])/g,'$1'+separator+'$2').toLowerCase();
    },

    /**
     * Checks if the string is empty or only contains whitespaces.
     * @name empty
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to work with
     * @return {Boolean} true if string is empty, false otherwise.
     * @static
     */
    empty: function(str) {
        return (/^\s*$/).test(str);
    },

    /**
     * Checks if the given string is a JSON object.
     * @name isJSON
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to check
     * @return {Boolean} true if given string is a valid JSON object, false otherwise
     * @static
     */
    isJSON: function(str) {
        var json;
        if (accedo.String.empty(str)) {
            return false;
        }
        try{
            json = this.parseJSON(str); // use left hand operand to prevent memory leak on old devices
            if(!json){
                return false;
            }
            return true;
        } catch (ex) {
            return false;
        }
        
        return false;
    },

    /**
     * Parses JSON data and return corresponding object or throws an error if the string is invalid.
     * @name parseJSON
     * @memberof accedo.module:String
     * @function
     * @param {String} str String to parse
     * @return Corresponding JSON object
     * @static
     */
    parseJSON: function(str) {
        try {
            return (window.JSON && window.JSON.parse) ? window.JSON.parse(str) : (new Function("return " + str))();
        } catch (ex) {
            accedo.console.error("unable to parse the JSON"+str);
        }
        return null;
    },

    /**
     * Checks if a string starts by the same characters than another specified string
     * @name startsWith
     * @memberof accedo.module:String
     * @function
     * @param {String} str String inside which we search for a starting substring
     * @param {String} startStr Starting substring to search for
     * @return {Boolean} true if str starts with startStr
     * @static
     */
    startsWith: function(str, startStr) {
        return str.substr(0, startStr.length) === startStr;
    },
    /**
     * encode the character like & to &amp;
     * @name encodeHTML
     * @memberof accedo.module:String
     * @function
     * @param {String} str String inside which we need to encode
     * @return {String} s String which is encoded.
     * @static
     */
    encodeHTML: function (str) {   
        var    s    =    "";   
        if    (str.length    ===    0) {
            return    "";
        }
        s    =    str.replace(/&/g,    "&amp;");   
        s    =    s.replace(/</g,        "&lt;");   
        s    =    s.replace(/>/g,        "&gt;");   
        s    =    s.replace(/ /g,        "&nbsp;");   
        s    =    s.replace(/\'/g,      "'");   
        s    =    s.replace(/\"/g,      "&quot;");   
        s    =    s.replace(/\n/g,      "<br>");   
        return    s;   
    },
    /**
     * decode the character like &amp; to &
     * @name decodeHTML
     * @memberof accedo.module:String
     * @function
     * @param {String} str String which we need to decode
     * @return {String} s String which is decoded
     * @static
     */
    decodeHTML: function (str) {
        var    s    =    "";   
        if    (str.length    ===    0){
            return    "";
        }
        s    =    str.replace(/&amp;/g,    "&");   
        s    =    s.replace(/&lt;/g,        "<");   
        s    =    s.replace(/&gt;/g,        ">");   
        s    =    s.replace(/&nbsp;/g,        " ");   
        s    =    s.replace(/'/g,      "\'");   
        s    =    s.replace(/&quot;/g,      "\"");   
        s    =    s.replace(/<br>/g,      "\n");   
        return    s;   
    },
    /**
     * Format a string of number (or just a number) into HH:MM:SS format
     * @name toHHMMSS
     * @memberof accedo.module:String
     * @function
     * @param {String | Integer} str String to work with
     * @return {String} Formatted string
     * @static
     */
    toHMMSS: function(str){
        var sec_numb = parseInt(str, 10),
        hours, minutes, seconds, time;
        if (!sec_numb) {
            return '0:00:00';
        }

        hours = Math.floor(sec_numb / 3600);
        minutes = Math.floor((sec_numb - (hours * 3600)) / 60);
        seconds = sec_numb - (hours * 3600) - (minutes * 60);

        if (minutes < 10) {
            minutes = "0"+minutes;
        }
        if (seconds < 10) {
            seconds = "0"+seconds;
        }
        time = hours + ':' + minutes + ':' + seconds;
        return time;
    }
});
/*jslint browser: true, devel: true, regexp: true */
/*global accedo: false,window:false */
/**
 * @desc Function Utility functions.
 * @name Fn
 * @memberof accedo
 * @module
 * @author <a href="mailto:alex@accedobroadband.com">Alexej Kubarev</a>
 */
accedo.Class.create('accedo.Fn', {
    /**
     * Updates an array (of arguments) with values arguments list (args).
     * This function modifies the "array".
     * @param array Array to update
     * @param args Arguments list
     * @return Updated array of arguments
     * @private
     * @ignore
     */
    _update : function(array, args) {
        var arrayLength = array.length, length = args.length;
        while (length--) {
            array[arrayLength + length] = args[length];
        }

        return array;
    },

    /**
     * Copies array and updates it with arguments.
     * Same as _update but does not modify the original array
     * @param array Array to update (Not modified)
     * @param args Arguments list
     * @return Updated array of arguments
     * @private
     * @ignore
     */
    _merge : function(array, args) {
        array = Array.prototype.slice.call(array, 0);
        return accedo.Fn._update(array, args);
    },
    

    /**
     * Bind a function (func) to a context.
     * If used directly on the function and extending function is allowed, first argument can be omited.
     * @name bind
     * @memberof accedo.module:Fn
     * @function
     * @param func Function to extend (if function extending allowed, this is to be omited)
     * @param context The context to bind to
     * @param {Object} third or extra arguments will be passed as an input parameter to the Function func binded.
     * @return Wraped function where execution is guaranteed in the given context
     * @public
     * @static
     */
    bind: function(func, context) {
        if (arguments.length < 2 && accedo.Object.isUndefined(arguments[0])) {
            return func;
        }

        var __method = func, args = Array.prototype.slice.call(arguments, 2);

        return function() {
            var a = accedo.Fn._merge(args, arguments);
            return __method.apply(context, a);
        };
    },

    /**
     * Return a list of argument names for a function.
     * @name argumentNames
     * @memberof accedo.module:Fn
     * @function
     * @param func Function to return a list of arguments for
     * @return An array of strings (argument names)
     * @public
     * @static
     */
    argumentNames: function(func) {
        var names = func.toString().match(/^[\s\(]*function[^(]*\(([^)]*)\)/)[1]
        .replace(/\/\/.*?[\r\n]|\/\*(?:.|[\r\n])*?\*\//g, '')
        .replace(/\s+/g, '').split(',');

        return names.length === 1 && !names[0] ? [] : names;
    },

    /**
     * Delays execution of function "func" for a time specified in seconds by "timeout".
     * 
     * @name delay
     * @memberof accedo.module:Fn
     * @function
     * @param func Function to delay execution of
     * @param timeout Delay time in seconds
     * @return Timer reference to cancel if needed (with clearTimeout())
     * @public
     * @static
     */
    delay: function(func, timeout) {
        var __method = func, args = Array.prototype.slice.call(arguments, 2);
        timeout = timeout * 1000;
        
        return window.setTimeout(function() {
            return __method.apply(__method, args);
        }, timeout);
    },

    /**
     * Defers execution of the function until the processor is ready.
     * This is essencialy the same as setting timeout to 1ms.
     * @name defer
     * @memberof accedo.module:Fn
     * @function
     * @param func Function to defer execution of
     * @return Time reference to cancel if needed (with clearTimeout())
     * @public
     * @static
     */
    defer: function(func) {
        var args = accedo.Fn._update([func, 0.01], Array.prototype.slice.call(arguments, 1)),f = accedo.Fn.delay;
        return f.apply(func, args);
    },

    /**
     * Returns a function "wrapped" around the original function.
     * This function lets you easily build on existing functions by specifying before and after behavior, 
     * transforming the return value, or even preventing the original function from being called.
     *
     *  The wraper function is called with this signature:
     *      function wrapper(callOriginal[, args...])
     *
     * "callOriginal" is a function that can be used to call the original (wrapped) function (or not, as appropriate).
     * It is not a direct reference to the original function, there's a layer of indirection in-between that sets up 
     * the proper context for it.
     *
     * @name wrap
     * @memberof accedo.module:Fn
     * @function
     * @param target Function to wrap
     * @param wrapper Function to wrap with
     * @return Wrapped function with correct references
     * @public
     * @static
     */
    wrap: function (target, wrapper) {
        return function() {
            var a = accedo.Fn._update([accedo.Fn.bind(target, this)], arguments);
            return wrapper.apply(this, a);
        };
    },

    /**
     * Wraps the function inside another function that, when called, pushes 'this' to the original function 
     * as the first argument (with any further arguments following it).
     * This method transforms the original function that has an explicit first argument to a function that
     * passes `this` (the current context) as an implicit first argument at call time.
     * It is useful when wanted to transform a function that takes an object to a method of that object or
     * its prototype, shortening its signature by one argument.
     * @name methodize
     * @memberof accedo.module:Fn
     * @function
     * @param target Function to transform.
     * @return wrapped function
     * @public
     * @static
     */
    methodize: function(target) {
        /**
         * @ignore
         */
        if(target._methodized) {
            return target._methodized;
        }
        /**
         * @ignore
         */
        var __method = target;
        return (
            /**
             * @ignore
             */
            target._methodized = function() {
                var a = accedo.Fn._update([this], arguments);
                return __method.apply(null, a);
            });
    }
});


/*jslint browser: true, devel: true */
/*global accedo: false */

/**
* Helper function for working with Arrays.
* If configuration parameter "extendArray" is set, functions will extend naitive Array.
* @name Array
* @module
* @memberof accedo
* @author <a href="mailto:alex@accedobroadband.com">Alexej Kubarev</a>
*/
accedo.Class.create('accedo.Array', {
    /**
     * Clears the array (makes it empty) and returns the array reference.
     * @name clear
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to clear
     * @return Array reference
     * @public
     * @static
     */
    clear: function(arr) {
        arr.length = 0;
        return arr;
    },

    /**
     * Returns a duplicate of the array, leaving the original array intact.
     * Performs deep copy if so required (slower).
     * @name clone
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to duplicate
     * @param {Boolean} deep Perform deep copy
     * @return {Array} a new copy of the array
     * @public
     * @static
     */
    clone: function(arr, deep) {
        if(deep === true) {
            var copy = [];
            accedo.Object.extend(copy, arr, true);
            return copy;
        }
        return Array.prototype.slice.call(arr, 0);
    },

    /**
     * Iterates array and for each item calls iterator function, with given context.
     * To break the loop, throw {accedo.$break}
     * @name each
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to iterate
     * @param {Function} iter Iterator function
     * @param {Object} context A context to bind iterator to
     * @public
     * @static
     */
    each: function(arr, iter, context) {
        var i = 0, len = arr.length;

        for (; i < len; i++) {
            try {
                iter.call(context, arr[i], i, arr);
            } catch (e) {
                if (e !== accedo.$break) {
                    throw e;
                }
                return;
            }
        }
    },

    /**
     * Tests whether all elements in the array pass the test implemented by the provided iterator functionfunction.
     * For compatibility with JS 1.6 this also passes while array to iterator as 3rd argument
     * @name every 
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to loop through
     * @param {Function} iter Iterator function
     * @param {Object} context A context to bind iterator to
     * @return {Boolean} true if every element passes the test, false otherwise
     * @public
     * @static
     */
    every: function(arr, iter, context) {
        iter = iter || function(x) {
            return x;
        };
        var result = true;
        accedo.Array.each(arr, function(value, index) {
            result = result && !! iter.call(context, value, index, arr);
            //Stop execution as we foudn first "false"
            if (!result) {
                throw accedo.$break;
            }
        });

        return result;
    },

    /**
     * Creates a new array with all elements that pass the test implemented by the provided iterator function.
     * For compatibility with JS 1.6 this also passes while array to iterator as 3rd argument
     * @name filter
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to search in
     * @param {Function} iter Iterator function to use for testing (returns true for pass, false for fail)
     * @param {Object} context Scope of the iterator
     * @return {Array} Elements that passed the test
     * @public
     * @static
     */
    filter: function(arr, iter, context) {
        var results = [];
        accedo.Array.each(arr, function(value, index) {
            if (iter.call(context, value, index, arr)) {
                results.push(value);
            }
        });

        return results;
    },

    /**
     * Returns the first element of the array.
     * @name first
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to fetch element of
     * @return first element of the array
     * @public
     * @static
     */
    first: function(arr) {
        return arr[0];
    },

    /**
     * Returns the first index at which a given element can be found in the array, or -1 if it is not present.
     * IndexOf implementation for JS < 1.6.
     * @name indexOf
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to search in
     * @param {Any} item Item to search for
     * @param {int} i Index to search from (optional)
     * @return {int} first index at which element is found int he array or -1 if not present
     * @public
     * @static
     */
    indexOf: function(arr, item, i) {
        //to use the default indexOf
        if(accedo.Object.isFunction(arr.indexOf)){
            return arr.indexOf(item,i);
        }
        
        if(!i){
            i = 0;
        }
        var len = arr.length;
        if (i < 0) {
            i = len + i;
        }

        for (; i < len; i++) {
            if (arr[i] === item) {
                return i;
            }
        }
        return -1;
    },

    /**
     * Returns the last element of the array.
     * @name last
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to fetch element of
     * @return last element of the array
     * @public
     * @static
     */
    last: function(arr) {
        return arr[arr.length - 1];
    },

    /**
     * Returns the position of the last occurrence of 'item' within the array or -1 if 'item' doesn't exist in the array.
     * @name lastIndexOf
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to search in
     * @param {Any} item Item to search for
     * @param {int} i Index to start searching from (optional)
     * @return {int} last index of the item in the array or -1 if item does not exist
     * @public
     * @static
     */
    lastIndexOf: function(arr, item, i) {
        i = isNaN(i) ? arr.length : (i < 0 ? arr.length + i : i) + 1;
        var n = accedo.Array.indexOf( arr.slice(0, i).reverse(), item);
        return (n < 0) ? n : i - n - 1;
    },

    /**
     * Creates a new array with the results of calling a provided iterator function on every element in this array.
     * @name map
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to work on
     * @param {Function} iter Iterator function to call on every object
     * @param {Object} context Scope to bind everything to
     * @return {Array} New array with modified values
     * @public
     * @static
     */
    map: function(arr, iter, context) {
        iter = iter || function(x){
            return x;
        };
        var results = [];

        accedo.Array.each(arr, function(value, index) {
            results.push(iter.call(context, value, index));
        });

        return results;
    },

    /**
     * Returns the length of array. This is the same as calling arr.length
     * @name size
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to get size of
     * @return {int} length of the array
     * @public
     * @static
     */
    size: function(arr) {
        return arr.length;
    },

    /**
     * Tests whether some element in the array passes the test implemented by the provided iterator function.
     * @name some
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to work on
     * @param {Function} iter Iterator function to test with
     * @param {Object} context Scope to bind everything to
     * @return {Boolean} true if at least one object passes iterator test, false otherwise
     * @public
     * @static
     */
    some: function(arr, iter, context) {
        iter = iter || function(x){
            return x;
        };
        var result = false;
        accedo.Array.each(arr, function(value, index) {
            result = iter.call(context, value, index);
            if (result) {
                throw accedo.$break;
            }
        });
        return result;
    },
       
    /**
     * Array Remove - By John Resig (MIT Licensed)
     * http://ejohn.org/blog/javascript-array-remove/
     * Removes all items of an Array from an index to another
     * /!\ Your array will be modified, not a copy of it
     * @name remove
     * @memberof accedo.module:Array
     * @function
     * @param {Array} arr Array to work on
     * @param {Integer} from Index from which we remove elements
     * @param {Integer} to (optional) Index up to which we remove elements
     * @returns {Integer} the resulting Array's length
     * @public
     * @static
     * @examples 
     * Remove the second item from the array
     *  array.remove(1);
     * Remove the second-to-last item from the array
     *  array.remove(-2);
     * Remove the second and third items from the array
     *  array.remove(1,2);
     * Remove the last and second-to-last items from the array
     *  array.remove(-2,-1);
     */
    remove : function(arr, from, to) {
        var rest = arr.slice((to || from) + 1 || arr.length);
        arr.length = from < 0 ? arr.length + from : from;
        return arr.push.apply(arr, rest);
    },

    /**
     * Array contains
     * check if the array contains certain element
     * @author <a href="mailto:cheung.chunho@accedobroadband.com">Cheung Chun Ho</a>
     * @name contains
     * @memberof accedo.module:Array
     * @function
     * @param {Array} array An array to check with
     * @param {Object} element An element to look for
     * @return { Boolean } state whether the array contains the element or not
     * @public
     * @static
     * @deprecated duplicate to indexOf function
     */
    contains : function(array, element){
        if(this.indexOf(array,element)>-1){
            return true;
        }
        return false;
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Hash Map object
 * @name Hash
 * @memberof accedo
 * @class Hash map support
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @constructor
 */
accedo.Class.create('accedo.Hash', {
    /**
     * Create a new accedo.Hash object
     * @name create
     * @memberof accedo.Hash
     * @function
     * @param {Object} object Object that to be converted into Hash Object
     * @return {accedo.Hash} A Hash object
     * @public
     * @static
     * 
     */
    create: function (object) {
        return new accedo.Hash(object);
    },
    /**
     * Returns JSON respresentation (in string) of the hash
     * @name toJSON
     * @memberof accedo.Hash
     * @function
     * @param {Object} hash Hash to be converted
     * @return {Object} Internal data structure in the form of JS Object
     * @public
     * @static
     * 
     */
    toJSON: function (hash) {
        return accedo.Object.toJSON(hash._data);
    },
    
    /**
     * Returns the debug-oriented string representation of the hash
     * @name inspect
     * @memberof accedo.Hash
     * @function
     * @param {Object} hash Hash object need to be inspected
     * @return {String} string representation of the hash
     * @public
     * @static
     * 
     */
    inspect: function(hash) {
        var result,keys = hash.keys(),values = hash.values(),i,length;
        result = "#<Hash:{";
        for (i = 0,length = keys.length ; i < length ; i++) {
            result = result + keys[i] + ":" + values[i] + ", ";
        }
        result = result.substr(0, result.length - 2) + "}>";
        return result;
    },
    
    /**
     * Returns a clone of input hash
     * @name clone
     * @memberof accedo.Hash
     * @function
     * @param {Object} hash Hash object need to be cloned
     * @return {accedo.Hash} cloned Hash object
     * @public
     * @static
     * 
     */
    clone: function(hash) {
        return new accedo.Hash(hash);
    }
},
{
    /**
     * The actual data storing object.
     * @name _data
     * @memberof accedo.Hash#
     * @private
     */
    _data : null,
    /**
     * @override
     * @protected
     * @memberof accedo.Hash#
     * @public
     */
    initialize: function (object) {
        object = object || {};
        this._data = accedo.Object.isHash(object) ? accedo.Hash.toObject(object) : accedo.Object.clone(object);
    },  
    /**
     * Sets a value and associates it to a key in the map.
     * @name set
     * @memberof accedo.Hash#
     * @function
     * @param {String} key Kkey to associate with
     * @param {Any} value Mixed data to associate with the key
     * @void
     * @public
     * @example
     * var h = accedo.Hash.create();
     * h.get("a"); // -> null
     * h.set("a", 1);
     * h.get("a"); // -> 1
     */
    set: function(key, value) {
        this._data[key] = value;
        return true;
    },
    /**
     * Gets a value associated with the key in the map.
     * @name get
     * @memberof accedo.Hash#
     * @function
     * @param {String} key Key to look for
     * @return associated value or undefined
     * @public 
     * @example
     * var h = accedo.Hash.create({a:1,b:2,c:3});
     * h.get("a"); // -> 1
     * h.get("c"); // -> 3
     * h.get("d"); // -> null
     */
    get: function(key) {
        if (this._data[key] !== Object.prototype[key]){
            return this._data[key];
        }
        return undefined;
    },
    /**
     * Unsets a value associated with the key, deletes the key and returns the value.
     * @name unset
     * @memberof accedo.Hash#
     * @function
     * @param {String} key Key to unset
     * @return value associated with the removed key
     * @public 
     * @example
     * var h = accedo.Hash.create({a:1});
     * h.get("a"); // -> 1
     * h.unset("a");
     * h.get("a"); // -> null
     */
    unset: function(key) {
        var value = this._data[key];
        delete this._data[key];
        return value;
    },
    /**
     * Unsets all values in the hash
     * @name clear
     * @memberof accedo.Hash#
     * @function
     * @public 
     */
    clear: function() {
        var i=0,keys = this.keys(), len = keys.length;
        for (; i<len; i++) {
            this.unset(keys[i]);
        }
    },
    /**
     * Returns an array of keys.
     * Note: Key order is depending on JS implementation
     * @name keys
     * @memberof accedo.Hash#
     * @function
     * @return {Array} Keys of this Hash
     * @public 
     * @example
     * var h = accedo.Hash.create({a:1, b:2, c:3});
     * h.keys(); // -> ["a", "b", "c"]
     */
    keys: function() {
        
        var k = [],key;
        for (key in this._data) {
            k.push(key);
        }
        return k;
    },    
    
    /**
     * Returns an array of values.
     * Note: Key order is depending on JS implementation
     * @name values
     * @memberof accedo.Hash#
     * @function
     * @return {Array} Values of this Hash
     * @public 
     * @example
     * var h = accedo.Hash.create({a:1, b:2, c:3});
     * h.values(); // -> [1, 2, 3]
     */
    values: function() {
        var v = [], key;
        for (key in this._data) {
            v.push(this._data[key]);
        }
        return v;
    },
    
    /**
     * Returns only the data structure without any function references.
     * @name toObject
     * @memberof accedo.Hash#
     * @function
     * @return {Object} Internal data structure in the form of JS Object
     * @public 
     * @example
     * var h = accedo.Hash.create({a:1});
     * h.set("b", 2);
     * h.toObject(); // -> {a:1, b:2}
     */
    toObject: function() {
        return accedo.Object.clone(this._data);
    },

    /**
     * Iterates hash and for each item calls iterator function, with given context.
     * To break the loop, throw {@link accedo.$break}
     * Same type of functionality as {@link accedo.utils.array#each}
     * @name each
     * @memberof accedo.Hash#
     * @function
     * @see accedo.utils.Array.each
     * @param {Function} iter Iterator function
     * @param context A context to bind iterator to
     * @public 
     * @void
     * @example
     * var h = accedo.Hash.create({a: 1, b:2, c:3});
     * h.each(function(pair){
     *    var key = pair.key // -> "a", "b", "c" (for first, second or third runs)  
     *    var value = pair.value // -> 1, 2, 3 (for first, second or third runs)
     * })
     */
    each: function(iter, context) {
        try {
            var key,value,pair;
            for (key in this._data) {
                value = this._data[key];
                pair = [key, value];
                pair.key = key;
                pair.value = value;

                iter.call(context, pair);
            }
        } catch (e) {
            if (e !== accedo.breaker) {
                throw e;
            }
        }
    }, 
    /**
     * Gets object type as string.
     * @name toString
     * @memberof accedo.Hash#
     * @function
     * @public
     * @return Type of object 
     */
    toString: function() {
        return "[object Hash]";
    },

    /**
     * Returns a URL-encoded string containing the hash's contents as query parameters.
     * @name toQueryString
     * @memberof accedo.Hash#
     * @function
     * @return {String} key-value pairs represented as a querystring
     * @public 
     * @example
     *  accedo.Hash.create({action: 'ship',
     *      order_id: 123,
     *      fees: ['f1', 'f2']
     *  }).toQueryString(); // -> "action=ship&order_id=123&fees=f1&fees=f2"
     *
     *  accedo.Hash.create({comment: '',
     *      'key with spaces': true,
     *      related_order: undefined,
     *      contents: null,
     *      'label': 'a demo'
     *  }).toQueryString(); // -> "comment=&key%20with%20spaces=true&related_order&contents=&label=a%20demo"
     *
     *  // an empty hash is an empty query string:
     *  accedo.Hash.create().toQueryString(); // -> ""
     */
    toQueryString: function() {
        var results = [], __toQueryPair;

        /**
         * Converts a key/value pair to a query pair in the form of "key=value"
         * @private
         */
        __toQueryPair = function(key, value) {
            if (accedo.Object.isUndefined(value)){
                return key;
            }
                
            return (key + '=' + encodeURIComponent( (value === null) ? "" : String(value)));
        };

            
        this.each(function(pair){
            var key = encodeURIComponent(pair.key), values = pair.value ,i,len,value,queryValues;
                
            if (values && typeof values === 'object') {
                if (accedo.Object.isArray(values)) {
                    i = 0;
                    len = values.length;
                    queryValues = [];
                    for (; i < len; i++) {
                        value = values[i];
                        queryValues.push(__toQueryPair(key, value));
                    }
                    results = results.concat(queryValues);
                }
            } else {
                results.push(__toQueryPair(key, values));
            }

        }, this);

        return results.join("&");
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @desc Utility functions for working DOM.
 * @name Dom
 * @memberof accedo
 * @module
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @deprecated Please just use accedo.Element instead
 */
accedo.Class.create('accedo.Dom',{ 
    /**
     * Gets the element by ID and returs a corresponding wrapper or null of element does not exist.
     * @name getById
     * @memberof accedo.module:Dom
     * @function
     * @param {String} id String ID of the element in the DOM
     * @return {accedo.Element.create} Element wrapper
     * @deprecated Please just use accedo.Element.getById() instead
     */
    getById: function(id) {
        var _dom = document.getElementById(id);
        if(_dom && accedo.Object.isDOMElement(_dom)) {
            return this.element(_dom);
        }

        return null;
    },
    /**
     * @memberof accedo.module:Dom
     * @function
     * @deprecated Please just use accedo.Element.create() instead
     */
    element: function (type, attributes, parent) {
        return new accedo.Element(type, attributes, parent);
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Convenience methods for DOM Elements, as a wrapper.
 * @name Element
 * @class Convenience methods for DOM Elements.
 * @author <a href="mailto:alex@accedo.tv">Andy Hui</a>
 * @param {String | DOM Element} type The node type, eg. "g" in SVG, "div" in HTML
 * @param {Object} [attributes] A map with the attributes and their values, eg
 *                             {x: 0, y: 15}
 * @param {accedo.Element} [parent] The parent node
 * @return {accedo.Element} the element node
 * @memberof accedo
 */
accedo.Class.create('accedo.Element', {

    /**
     * Whether to check if the class name container upper case letter before using the new DOM.classList API.
     * @name _checkClassCap
     * @memberof accedo.Element
     * @private
     */
    _checkClassCap: accedo.config.get('ui.element.checkClassNameCapital', true),

    /**
     * Creates a new DOM element wrapped in wrapper.
     * @name create
     * @memberof accedo.Element
     * @param {String} type HTML element tag type name
     * @param {Object} attributes HTML attributes to assign for this element
     * @param {accedo.Element} parent the parent element to append to
     * @return {accedo.Element.create} Element wrapper
     */
    create: function(type, attributes, parent){
        return new accedo.Element(type, attributes, parent);
    },

    /**
     * Gets the element by ID and returs a corresponding wrapper or null of element does not exist.
     * @name getById
     * @memberof accedo.Element
     * @param {String} id String ID of the element in the DOM
     * @return {accedo.Element.create} Element wrapper
     */
    getById: function(id){
        var _dom = document.getElementById(id);
        if(_dom && accedo.Object.isDOMElement(_dom)) {
            return this.create(_dom);
        }

        return null;
    },

    /**
     * Whether the platform supports "DOMElement.classList" property
     * @name _dom
     * @private
     * @memberof accedo.Element#
     * @static
     */
    _supportsClassList: (function(){
        // document.body is not available now, go for the one that must exist: a script element
        var scriptEle = document.getElementsByTagName("script")[0];
        return !!(scriptEle.classList && scriptEle.classList.add);
    }()),

    /**
     * Fix up an event so that can use W3C standard methods to cancel the event and stop event propagation.
     * Brought in a lot of ideas from Dean Edwards' addEvent()
     * @name _fixEvent
     * @returns {Event} the updated native event object
     * @private
     * @memberof accedo.Element#
     * @static
     */
    _fixEvent : function(evt) {
        // add W3C standard event methods
        if(!evt.preventDefault){
            evt.preventDefault = accedo.Element._preventDefault;
        }
        if(evt.stopPropagation){
            evt.stopPropagation = accedo.Element._stopPropagation;
        }
        return evt;
    },

    /**
     * Handler method for cancelling the event default action.
     * @name _preventDefault
     * @private
     * @memberof accedo.Element#
     * @static
     */
    _preventDefault : function() {
        this.returnValue = false;
    },

    /**
     * Handler method for stopping event propagation.
     * @name _stopPropagation
     * @private
     * @memberof accedo.Element#
     * @static
     */
    _stopPropagation : function() {
        this.cancelBubble = true;
    }
}, {
    /**
     * The ID of this element.
     * @name id
     * @protected
     * @memberof accedo.Element#
     */
    id: null,
    /**
     * The underlying HTML DOM element.
     * @name _dom
     * @private
     * @memberof accedo.Element#
     */
    _dom: null,
    /**
     * The container for event listeners.
     * @name _event
     * @private
     * @memberof accedo.Element#
     */
    _events: null,
    /*
     * @public
     * @protected
     * @override
     */
    initialize: function(type, attributes, parent) {
        if(accedo.Object.isDOMElement(type)) {
            this._dom = type;
        } else {
            this._dom = document.createElement(type);
        }

        //Parse and set attributes as requested
        if (accedo.Object.isPlainObject(attributes)) {
            this.setAttributes(attributes);
        }

        //Append child to parent if possible
        if (parent) {
            parent.appendChild(this._dom);
        }
    },
    /**
     * Returns the actual HTML Element.
     * @name getHTMLElement
     * @function
     * @public
     * @memberof accedo.Element#
     */
    getHTMLElement: function() {
        return this._dom;
    },

    /**
     * Gets parent element wrapped in DOM object.
     * @name getParent
     * @function
     * @public
     * @memberof accedo.Element#
     */
    getParent: function() {
        var parent = this._dom.parentNode;
        if(parent) {
            return new accedo.Element(parent);
        }

        return null;
    },

    /**
     * Remove element from the DOM tree.
     * @name remove
     * @return {accedo.Element.create} the Element itself
     * @function
     * @public
     * @memberof accedo.Element#
     */
    remove: function() {
        var parent = this._dom.parentNode, child;

        if(parent) {
            //Prevent memory leaks in Samsung devices using a left-hand operand
            child = parent.removeChild(this._dom);
        }

        child = null;
        parent = null;

        return this;
    },

    /**
     * Sets the element as depending on its parent (so it shows it, if its parent is not hidden itself)
     * @name show
     * @function
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    show: function() {
        this._dom.style.display  = '';
        return this;
    },

    /**
     * Hides element
     * @name hide
     * @function
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    hide: function() {
        this._dom.style.display = 'none';
        return this;
    },

    /**
     * Hides or shows the element (if its display is none, sets it to inherit, otherwise set it to none.
     * It means you'll lose the current value by toggling twice, if display was 'inline' or 'block' for instance)
     * @name toggle
     * @function
     * @public
     * @memberof accedo.Element#
     * @returns {boolean} false if the element is hidden after toggling, true otherwise
     */
    toggle: function() {
        if (this._dom.style.display === 'none') {
            this.show();
            return true;
        }
        this.hide();
        return false;
    },

    /**
     * Set's visibility of the element.
     * This is different from show/hide and toggle functions that work with display property.
     * This function updates the visibility CSS property instead.
     * As a result, no change in layout is done.
     * @name setVisible
     * @param {Boolean} visible
     * @return {accedo.Element.create} the Element itself
     * @function
     * @public
     * @memberof accedo.Element#
     */
    setVisible: function(visible) {
        this._dom.style.visibility = visible ? 'visible' : 'hidden';
        return this;
    },

    /**
     * Add event listener to the DOM Object.
     * Brought in a lot of ideas from Dean Edwards' addEvent()
     * @name addEventListener
     * @param {String} type Event type
     * @param {Function} listener Function to call back on event fire
     * @param {Boolean} useCapture Use capture?
     * @return {Element} The element itself, for chaining
     * @public
     * @memberof accedo.Element#
     */
    addEventListener: function(type, listener, useCapture) {
        var element = this._dom,
        events = this._events, listeners;

        listener.$$useCapture = !!useCapture;
        // assign each event handler a unique ID
        // so that they can be removed later to prevent memory leaks on some devices
        if (!listener.$$guid) {
            listener.$$guid = accedo.getGuid();
        }
        // create a hash table of event types for the element
        if (!events) {
            this._events = events = {};
        }
        // create a hash table of event handlers for each element/event pair
        listeners = events[type];
        if (!listeners) {
            listeners = events[type] = {};
            // store the existing event handler (if there is one)
            if (element["on" + type]) {
                listeners[0] = element["on" + type];
            }
        }
        // store the event handler in the hash table
        listeners[listener.$$guid] = listener;

        if (element.addEventListener) {
            element.addEventListener(type, listener, useCapture);
        } else { // use home-baked handler that calls all registered handlers
            element["on" + type] = accedo.Fn.bind(this._handleEvent, this);
        }
        return this;
    },

    /**
     * Handles an event in case native addEventListener is not supported.
     * Brought in a lot of ideas from Dean Edwards' addEvent()
     * @name _handleEvent
     * @returns {boolean} false if any of the handlers returned false
     * @private
     * @memberof accedo.Element#
     */
    _handleEvent : function(evt) {
        // grab the event object (IE uses a global event object)
        evt = evt || window.event;
        var returnValue = true,
        target = evt.target || evt.srcElement;
        // get a reference to the hash table of event handlers
        var handlers = this._events[evt.type];
        // execute each event handler
        for (var i in handlers) {
            target.$$handleEvent = handlers[i];
            if (target.$$handleEvent(evt) === false) {
                returnValue = false;
            }
        }
        return returnValue;
    },

    /**
     * Removes event listener from the DOM Object.
     * @name removeEventListener
     * @function
     * @param {String} type Event type, removes all event listeners if not provided
     * @param {Function} listener Function to call back on event fire, removes all event listeners of the specified type if not provided
     * @param {Boolean} useCapture Use capture?
     * @return {Element} The element itself, for chaining
     * @public
     * @memberof accedo.Element#
     */
    removeEventListener: function(type, listener, useCapture) {
        var element = this._dom,
        events = this._events,
        listeners, curType, curGuid, curListener;
        if (!type) { // remove all listeners
            if (!events) {
                return this;
            }
            for(curType in events){
                listeners = events[curType];
                for(curGuid in listeners){
                    curListener = listeners[curGuid];
                    if (element.removeEventListener) {
                        element.removeEventListener(curType, curListener, curListener.$$useCapture);
                    }
                    delete events[curType][curGuid];
                }
                delete events[curType];
            }
            this._events = null; // remove all listeners
            return this;
        } else if (!listener) { // remove all listeners of certain type
            if (!events) {
                return this;
            }
            listeners = events[type];
            for(curGuid in listeners){
                curListener = listeners[curGuid];
                if (element.removeEventListener) {
                    element.removeEventListener(type, curListener, curListener.$$useCapture);
                }
                delete events[type][curGuid];
            }
            delete events[type]; // remove whole type
            return this;
        }
        // just remove one specific listener
        if (element.removeEventListener) {
            element.removeEventListener(type, listener, useCapture);
        }
        if (events && events[type]) {
            delete events[type][listener.$$guid];
        }
        return this;
    },

    /**
     * Removes all event listeners from the DOM Object.
     * @name removeAllEventListener
     * @function
     * @return {Element} The element itself, for chaining
     * @public
     * @memberof accedo.Element#
     */
    removeAllEventListener: function() {
        if (!this._events) { // for better performance
            return this;
        }
        this.removeEventListener();
        return this;
    },

    /**
     * This function removes a child node from a parent node.
     * @name removeChild
     * @function
     * @param {DOM Element} child The child node
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    removeChild: function(child) {
        if(!accedo.Object.isDOMElement(child)) {
            child = child.getHTMLElement();
        }

        if (accedo.Object.isFunction(this._dom.removeChild)){
            this._dom.removeChild(child);
        }else{
            accedo.Array.each(
                Array.prototype.slice.call(this._dom.childNodes,0),
                function (_child) {
                    if (_child === child) {
                        //Prevent memory leaks in Samsung devices using a left-hand operand
                        var myChild = this._dom.removeChild(_child);
                        myChild = null;
                    }
                },this);
        }
        return this;
    },
    /**
     * This function removes all child nodes from a parent node.
     * @name removeAll
     * @function
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    removeAll: function() {

        var child = this._dom.childNodes[0];
        while(child) {
            this.removeChild(child);
            child = this._dom.childNodes[0];
        }
        return this;
    },

    /**
     * This function appends a child node to a parent node.
     * @name appendChild
     * @function
     * @param {DOM Element | accedo.Element} child The child node
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    appendChild: function(child) {
        if(accedo.Object.isDOMElement(child)) {
            this._dom.appendChild(child);
        } else {
            this._dom.appendChild(child.getHTMLElement());
        }
        return this;
    },

    /**
     * This function appends a new child node to a parent node, before one of its children.
     * @name insertBefore
     * @function
     * @param {DOM Element | accedo.Element} newChild The new child node
     * @param {DOM Element | accedo.Element} existingChild The existing child node
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    insertBefore: function(newChild, existingChild) {
        var existingC,newC;
        existingC = accedo.Object.isDOMElement(existingChild) ? existingChild : existingChild.getHTMLElement();
        newC = accedo.Object.isDOMElement(newChild) ? newChild : newChild.getHTMLElement();

        this._dom.insertBefore(newC, existingC);
        return this;
    },

    /**
     * This function appends a new child node to a parent node, after one of its children.
     * @name insertAfter
     * @function
     * @param {DOM Element | accedo.Element} newChild The new child node
     * @param {DOM Element | accedo.Element} existingChild The existing child node
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    insertAfter: function(newChild, existingChild) {
        var existingC,newC;
        existingC = accedo.Object.isDOMElement(existingChild) ? existingChild : existingChild.getHTMLElement();
        newC = accedo.Object.isDOMElement(newChild) ? newChild : newChild.getHTMLElement();

        if (this._dom.lastChild === existingC) {
            this._dom.appendChild(newC);
        }
        else {
            this._dom.insertBefore(newC,existingC.nextSibling);
        }
        return this;
    },
    
    /**
     * This function attach a new child node to a parent node, replacing one of its children.
     * @name replaceChild
     * @function
     * @param {DOM Element | accedo.Element} newChild The new child node
     * @param {DOM Element | accedo.Element} existingChild The existing child node
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    replaceChild: function(newChild, existingChild) {
        var existingC,newC,sibling;
        existingC = accedo.Object.isDOMElement(existingChild) ? existingChild : existingChild.getHTMLElement();
        newC = accedo.Object.isDOMElement(newChild) ? newChild : newChild.getHTMLElement();
        
        if (this._dom.lastChild === existingC) {
            this._dom.removeChild(existingC);
            this._dom.appendChild(newC);
        }
        else {
            sibling = existingC.nextSibling;
            this._dom.removeChild(existingC);
            this._dom.insertBefore(newC,sibling);
        }
        return this;
    },
    /**
     * This function appends a new child node to a parent node, before the first child.
     * @name prependChild
     * @function
     * @param {DOM Element | accedo.Element} newChild The new child node
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */

    prependChild: function(newChild){
        var newC = accedo.Object.isDOMElement(newChild) ? newChild : newChild.getHTMLElement();
        this._dom.insertBefore(newC, this._dom.childNodes[0]);
        return this;
    },

    /**
     * This function finds out whether this node is attached to a parent or not.
     * It reflects the accedo.ui.AbstractComponent's method with the same name.
     * @name isAttached
     * @function
     * @public
     * @memberof accedo.Element#
     * @returns {true|false}
     */
    isAttached: function() {
        return this.getHTMLElement().parentNode !== null;
    },

    /**
     * This function removes an attribute from an DOM element.
     * @name removeAttribute
     * @function
     * @param {String} attribute The attribute
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    removeAttribute: function(attribute) {
        if (this.hasAttribute(attribute)) {
            this._dom.removeAttribute(attribute);
        }
        return this;
    },

    /**
     * This function removes a style definition from an DOM element.
     * @name removeStyle
     * @function
     * @param {String} styleName The style name
     * @return {accedo.Element.create} the Element itself
     * @public
     * @memberof accedo.Element#
     */
    removeStyle: function(styleName) {
        if (this.hasStyle(styleName)) {
            delete this._dom.style[styleName];
        }
        return this;
    },

    /**
     * This function checks if an element has a certain attribute defined or not.
     * @name hasAttribute
     * @function
     * @param {String} attribute The attribute
     * @return true/false
     * @public
     * @memberof accedo.Element#
     */
    hasAttribute: function(attribute) {
        if (this._dom.hasAttribute) {
            return this._dom.hasAttribute(attribute);
        }
        return !accedo.Object.isUndefined(this._dom[attribute]);
    },

    /**
     * This function checks if an element has a certain style defined or not.
     * @name hasStyle
     * @function
     * @param {String} styleName The style name
     * @return true/false
     * @public
     * @memberof accedo.Element#
     */
    hasStyle: function(styleName) {
        return !accedo.Object.isUndefined(this._dom.style[styleName]);
    },

    /**
     * Gets the style text of the element
     * @name _getStyleText
     * @function
     * @return {String} the style text of the element
     * @memberof accedo.Element#
     * @private
     */
    _getStyleText : function(){
        if( !accedo.Object.isUndefined(this._dom.style.cssText) ) {
            return this._dom.style.cssText;
        } else {
            return this._dom.getAttribute('style');
        }
    },

    /**
     * Sets the style text of the element, will overwrites the original text
     * @name _setStyleText
     * @function
     * @param styleText {String} the style text to set into the element
     * @memberof accedo.Element#
     * @private
     */
    _setStyleText : function(styleText){
        if( !accedo.Object.isUndefined(this._dom.style.cssText) ) {
            this._dom.style.cssText = styleText;
        } else {
            this._dom.setAttribute('style',styleText);
        }
    },

    /**
     * This function sets style attributes on a give context.
     * @name setStyle
     * @function
     * @param {Object} styles A object containing CSS name/value pairs
     * @return {Element} The element itself, for chaining
     * @public
     * @memberof accedo.Element#
     */
    setStyle: function(styles) {
        var style, curStyleStr,
        curCSS = this._getStyleText() || '',
        newCSS = curCSS,
        isCurCSSEmpty = !curCSS,
        overrideArr = [], key;

        // set style text way does not work at all on Samsung 2010
        if(accedo.platform === 'samsung' && accedo.device.id.getFirmwareYear() === 2010){
            for (style in styles) {
                if (styles.hasOwnProperty(style)){
                    this._dom.style[style] = styles[style];
                }
            }
            return this;
        }

        for (style in styles) {
            if (styles.hasOwnProperty(style)){
                style = accedo.String.camelize(style);
                curStyleStr = accedo.String.decamelize(style, "-") + ":" + styles[style] + ";";
                if(isCurCSSEmpty){
                    newCSS += curStyleStr;
                } else if(!accedo.Object.isNull(this._dom.style[style]) &&
                    !accedo.Object.isUndefined(this._dom.style[style]) &&
                    this._dom.style[style] !== "" ) { // existing style
                    overrideArr.push(style);
                } else { // non-existing style
                    newCSS = curStyleStr + newCSS;
                }
            }
        }
        this._setStyleText(newCSS); // set new css in batch
        for (key in overrideArr) { // override old ones one by one
            style = overrideArr[key];
            this._dom.style[style] = styles[style];
        }
        return this;
    },

    /**
     * Gets a given style for the DOM element.
     * @name getStyle
     * @function
     * @memberof accedo.Element#
     * @param {String} style Style name to get.
     * @public
     * @return style value or null
     */
    getStyle: function(style) {
        style = style === 'float' ? 'cssFloat' : accedo.String.camelize(style);

        var value = null;
        if (document.defaultView && document.defaultView.getComputedStyle) {
            value = document.defaultView.getComputedStyle(this._dom, null)[style];
        } else if (this._dom.currentStyle) {
            value = this._dom.currentStyle[style];
        } else {
            value = this._dom.style[style];
        }

        /*if (!value || value == 'auto') {
                      var css = document.defaultView.getComputedStyle(this._dom, null);
                      value = css ? css[style] : null;
                    }*/
        if (style === 'opacity'){
            return value ? parseFloat(value) : 1.0;
        }
        return value === 'auto' ? null : value;
    },

    /**
     * Gets Cumulative offset for element.
     * Returned data is an object with x and y properties.
     * @name cumulativeOffset
     * @function
     * @public
     * @memberof accedo.Element#
     * @return {Object} OFsset object with x and y properties
     */
    cumulativeOffset: function() {
        var valueT = 0, valueL = 0,element = this._dom;
        if (element.parentNode) {
            do {
                valueT += element.offsetTop  || 0;
                valueL += element.offsetLeft || 0;
                element = element.offsetParent;
            } while (element);

        }
        return {
            x: valueL,
            y: valueT
        };
    },

    /**
     * This function adds one or more CSS classes to the given element.
     * @name addClass
     * @function
     * @param {String | Array} className Either an array of CSS classname or a single CSS classname
     * @return {Element} The element itself, for chaining
     * @public
     * @memberof accedo.Element#
     */
    addClass: function(className) {
        if (accedo.Object.isArray(className)) {
            accedo.Array.each( className, function(def) {
                this.addClass(def);
            }, this);
        } else if (accedo.Object.isString(className)) {
            //Remove extra space to avoid DOM exception
            className = className.replace(/\s+/g,' ').replace(/^\s/, '').replace(/\s$/, '');
            if (className.indexOf(" ") >= 0) { // className contains space and multiple classes, need to chop them up
                accedo.Array.each( className.split(" "), function(def) {
                    this.addClass(def);
                }, this);
            } else if (accedo.Element._supportsClassList && (!accedo.Element._checkClassCap || !/[A-Z]/g.exec(className))) { // platform supports classList feature, so just use it
                this._dom.classList.add(className);
            } else if (!this.hasClass(className)) { // do it the old fashioned way
                //Either add space or do not add space before appending className
                this._dom.className += (this._dom.className ? ' ' : '') + className;
            }
        }
        return this;
    },

    /**
     * Checks if the element has a CSS Class name assigned.
     * @name hasClass
     * @function
     * @public
     * @memberof accedo.Element#
     * @param {String} className CSS Class name to look for.
     * @return {Boolean} true if element has class name, false otherwise
     */
    hasClass: function(className) {
        if (accedo.Element._supportsClassList && (!accedo.Element._checkClassCap || !/[A-Z]/g.exec(className))) { // platform supports classList feature, so just use it
            return this._dom.classList.contains(className);
        }

        var elemClass = this._dom.className;
        return (elemClass.length > 0 && (elemClass === className || new RegExp("(^|\\s)" + className + "(\\s|$)").test(elemClass)));
    },

    /**
     * Removes a given CSS class name from the element.
     * @name removeClass
     * @function
     * @public
     * @memberof accedo.Element#
     * @param {String} className CSS Class name to remove from the element.
     * @return {Element} The element itself, for chaining
     */
    removeClass: function(className) {
        if (accedo.Element._supportsClassList && (!accedo.Element._checkClassCap || !/[A-Z]/g.exec(className))) { // platform supports classList feature, so just use it
            this._dom.classList.remove(className);
            return this;
        }

        var cname = this._dom.className.replace(new RegExp("(^|\\s+)" + className + "(\\s+|$)"), ' ');
        cname = accedo.String.strip(cname);
        this._dom.className = cname;
        return this;
    },

    /**
     * Replaces the current CSS class(es) with given the given className
     * @name setClass
     * @function
     * @param {String} className CSS Classname
     * @return {Element} The element itself, for chaining
     * @public
     * @memberof accedo.Element#
     */
    setClass: function(className){
        this._dom.className = accedo.String.strip(className);
        return this;
    },

    /**
     * This function sets the textual content of a DOM element.
     * @name setText
     * @function
     * @param {String} content The text content to set
     * @param {boolean} isPureText Set to true if the text content should not be translated as html (element.innerText will be used)
     * @return {Element} The element itself, for chaining
     * @public
     * @memberof accedo.Element#
     */
    setText: function(content, isPureText) {
        if(isPureText){
            //firefox is unable to set InnerText
            if(accedo.device && accedo.device.appengine && accedo.device.appengine.getLayoutEngine().name === "Gecko" && accedo.device.id.getDeviceType() === "Workstation"){
                this._dom.textContent = content;
            }else{
                this._dom.innerText = content;
            }
            return this;
        }
        this._dom.innerHTML = content;
        return this;
    },

    /**
     * This function returns the textual content of a DOM element.
     * @name getText
     * @function
     * @public
     * @memberof accedo.Element#
     */
    getText: function() {
        return this._dom.innerHTML;
    },

    /**
     * Alters and/or adds attributes in a node using the default namespace
     * @name setAttributes
     * @function
     * @public
     * @param element [node] the node to which attribues shall be applied
     * @param attributes [Object] map with the attributes and their values, eg
     *                             { x: 0, y: 15 }
     * @return {Element} The element itself, for chaining
     * @memberof accedo.Element#
     */
    setAttributes: function(attributes) {
        var attr;
        for (attr in attributes){
            if (attributes.hasOwnProperty(attr)){
                if (attr === 'style'){
                    this.setStyle(attributes[attr]);
                } else if (attr === 'textContent'){
                    this._dom.textContent = attributes[attr];
                } else {
                    this._dom.setAttribute(attr, attributes[attr]);
                }
            }
        }
        return this;
    },

    /**
     * Get the farthest-back ancestor of our node
     * @name _findUltimateAncestor
     * @return {DOM Element} the farthest-back ancestor
     * @function
     * @private
     * @memberof accedo.ui.Element#
     */
    _findUltimateAncestor: function() {
        var ancestor = this._dom;
        while(ancestor.parentNode) {
            ancestor = ancestor.parentNode;
        }
        return ancestor;
    },

    /**
     * If the farthest-back ancestor of our node has a "body" property (that node would be the document itself), we assume it is in the page's DOM tree.
     * @name isInDOMTree
     * @param node
     * @return {Boolean}
     * @function
     * @public
     * @memberof accedo.ui.Element#
     */
    isInDOMTree:function () {
        return this._findUltimateAncestor().body === document.body;
    },

    /**
     * This function signals this element will not be used any more, and should collect memory to prevent any memory leaks.
     * @name deinit
     * @function
     * @public
     * @memberof accedo.ui.Element#
     */
    deinit:function() {
        this.removeAllEventListener();
    }
});

/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @desc This class do not need to be instantiated. Please just use its static members.
 * @name VKey
 * @memberof accedo
 * @static
 * @class Contain definition for all supported key in XDK.
 * @author <a href="mailto:mike.leung@accedo.tv">Mike Leung</a>
 */
accedo.Class.create('accedo.VKey', {
        /**
        * @constant
        * @name KEY_0
        * @memberof accedo.VKey
        */
        KEY_0 : '0',
        /**
        * @constant
        * @name KEY_1
        * @memberof accedo.VKey
        */
        KEY_1 : '1',
        /**
        * @constant
        * @name KEY_2
        * @memberof accedo.VKey
        */
        KEY_2 : '2',
        /**
        * @constant
        * @name KEY_3
        * @memberof accedo.VKey
        */
        KEY_3 : '3',
        /**
        * @constant
        * @name KEY_4
        * @memberof accedo.VKey
        */
        KEY_4 : '4',
        /**
        * @constant
        * @name KEY_5
        * @memberof accedo.VKey
        */
        KEY_5 : '5',
        /**
        * @constant
        * @name KEY_6
        * @memberof accedo.VKey
        */
        KEY_6 : '6',
        /**
        * @constant
        * @name KEY_7
        * @memberof accedo.VKey
        */
        KEY_7 : '7',
        /**
        * @constant
        * @name KEY_8
        * @memberof accedo.VKey
        */
        KEY_8  : '8',
        /**
        * @constant
        * @name KEY_9
        * @memberof accedo.VKey
        */
        KEY_9  : '9',

        /**
        * @constant
        * @name KEY_ENTER
        * @memberof accedo.VKey
        */
        KEY_ENTER  : 'ok',
        /**
        * @constant
        * @name KEY_OK
        * @memberof accedo.VKey
        */
        KEY_OK  : 'ok', //for compatibility
        /**
        * @constant
        * @name KEY_UP
        * @memberof accedo.VKey
        */
        KEY_UP : 'up',
        /**
        * @constant
        * @name KEY_DOWN
        * @memberof accedo.VKey
        */
        KEY_DOWN : 'down',
        /**
        * @constant
        * @name KEY_LEFT
        * @memberof accedo.VKey
        */
        KEY_LEFT : 'left',
        /**
        * @constant
        * @name KEY_RIGHT
        * @memberof accedo.VKey
        */
        KEY_RIGHT : 'right',
        /**
        * @constant
        * @name KEY_EXIT
        * @memberof accedo.VKey
        */
        KEY_EXIT : 'exit',
        /**
        * @constant
        * @name KEY_HOME
        * @memberof accedo.VKey
        */
        KEY_HOME : 'home',
        /**
        * @constant
        * @name KEY_BACK
        * @memberof accedo.VKey
        */
        KEY_BACK : 'back',

        /**
        * @constant
        * @name KEY_RED
        * @memberof accedo.VKey
        */
        KEY_RED : 'red',
        /**
        * @constant
        * @name KEY_GREEN
        * @memberof accedo.VKey
        */
        KEY_GREEN : 'green',
        /**
        * @constant
        * @name KEY_YELLOW
        * @memberof accedo.VKey
        */
        KEY_YELLOW : 'yellow',
        /**
        * @constant
        * @name KEY_BLUE
        * @memberof accedo.VKey
        */
        KEY_BLUE : 'blue',

        /**
        * @constant
        * @name KEY_PAGE_UP
        * @memberof accedo.VKey
        */
        KEY_PAGE_UP : 'page_up',
        /**
        * @constant
        * @name KEY_PAGE_DOWN
        * @memberof accedo.VKey
        */
        KEY_PAGE_DOWN : 'page_down',
        /**
        * @constant
        * @name KEY_CH_UP
        * @memberof accedo.VKey
        */
        KEY_CH_UP     : 'ch_up',
        /**
        * @constant
        * @name KEY_CH_DOWN
        * @memberof accedo.VKey
        */
        KEY_CH_DOWN   : 'ch_down',

        /**
        * @constant
        * @name KEY_PLAY
        * @memberof accedo.VKey
        */
        KEY_PLAY : 'play',
		
        /**
        * @constant
        * @name KEY_REC
        * @memberof accedo.VKey
        */
        KEY_REC : 'rec',

		/**
        * @constant
        * @name KEY_PLAY_PAUSE
        * @memberof accedo.VKey
        */
	KEY_PLAY_PAUSE : 'playPause',
        /**
        * @constant
        * @name KEY_PAUSE
        * @memberof accedo.VKey
        */
        KEY_PAUSE : 'pause',
        /**
        * @constant
        * @name KEY_STOP
        * @memberof accedo.VKey
        */
        KEY_STOP : 'stop',
        /**
        * @constant
        * @name KEY_FF
        * @memberof accedo.VKey
        */
        KEY_FF : 'ff',
        /**
        * @constant
        * @name KEY_RW
        * @memberof accedo.VKey
        */
        KEY_RW : 'rw',
        /**
        * @constant
        * @name KEY_NEXT
        * @memberof accedo.VKey
        */
        KEY_NEXT : 'next',
        /**
        * @constant
        * @name KEY_PREV
        * @memberof accedo.VKey
        */
        KEY_PREV : 'prev',
        /**
        * @constant
        * @name KEY_MUTE
        * @memberof accedo.VKey
        */
        KEY_MUTE : 'mute',
        /**
        * @constant
        * @name KEY_INFO
        * @memberof accedo.VKey
        */
        KEY_INFO : 'info',
        /**
        * @constant
        * @name KEY_TOOLS
        * @memberof accedo.VKey
        */
        KEY_TOOLS : 'tools',
        /**
        * @constant
        * @name KEY_PRE_CH
        * @memberof accedo.VKey
        */
        KEY_PRE_CH : 'pre_ch',
        /**
        * @constant
        * @name KEY_SUBTITLE
        * @memberof accedo.VKey
        */
        KEY_SUBTITLE : 'subtitle',
        /**
        * @constant
        * @name KEY_TEXT
        * @memberof accedo.VKey
        */
        KEY_TEXT : 'text',
        /**
        * @constant
        * @name KEY_A
        * @memberof accedo.VKey
        */
        KEY_A : 'A',
        /**
        * @constant
        * @name KEY_B
        * @memberof accedo.VKey
        */
        KEY_B : 'B',
        /**
        * @constant
        * @name KEY_C
        * @memberof accedo.VKey
        */
        KEY_C : 'C',
        /**
        * @constant
        * @name KEY_D
        * @memberof accedo.VKey
        */
        KEY_D : 'D',
        /**
        * @constant
        * @name KEY_E
        * @memberof accedo.VKey
        */
        KEY_E : 'E',
        /**
        * @constant
        * @name KEY_F
        * @memberof accedo.VKey
        */
        KEY_F : 'F',
        /**
        * @constant
        * @name KEY_G
        * @memberof accedo.VKey
        */
        KEY_G : 'G',
        /**
        * @constant
        * @name KEY_H
        * @memberof accedo.VKey
        */
        KEY_H : 'H',
        /**
        * @constant
        * @name KEY_I
        * @memberof accedo.VKey
        */
        KEY_I : 'I',
        /**
        * @constant
        * @name KEY_J
        * @memberof accedo.VKey
        */
        KEY_J : 'J',
        /**
        * @constant
        * @name KEY_K
        * @memberof accedo.VKey
        */
        KEY_K : 'K',
        /**
        * @constant
        * @name KEY_L
        * @memberof accedo.VKey
        */
        KEY_L : 'L',
        /**
        * @constant
        * @name KEY_M
        * @memberof accedo.VKey
        */
        KEY_M : 'M',
        /**
        * @constant
        * @name KEY_N
        * @memberof accedo.VKey
        */
        KEY_N : 'N',
        /**
        * @constant
        * @name KEY_O
        * @memberof accedo.VKey
        */
        KEY_O : 'O',
        /**
        * @constant
        * @name KEY_P
        * @memberof accedo.VKey
        */
        KEY_P : 'P',
        /**
        * @constant
        * @name KEY_Q
        * @memberof accedo.VKey
        */
        KEY_Q : 'Q',
        /**
        * @constant
        * @name KEY_R
        * @memberof accedo.VKey
        */
        KEY_R : 'R',
        /**
        * @constant
        * @name KEY_S
        * @memberof accedo.VKey
        */
        KEY_S : 'S',
        /**
        * @constant
        * @name KEY_T
        * @memberof accedo.VKey
        */
        KEY_T : 'T',
        /**
        * @constant
        * @name KEY_U
        * @memberof accedo.VKey
        */
        KEY_U : 'U',
        /**
        * @constant
        * @name KEY_V
        * @memberof accedo.VKey
        */
        KEY_V : 'V',
        /**
        * @constant
        * @name KEY_W
        * @memberof accedo.VKey
        */
        KEY_W : 'W',
        /**
        * @constant
        * @name KEY_X
        * @memberof accedo.VKey
        */
        KEY_X : 'X',
        /**
        * @constant
        * @name KEY_Y
        * @memberof accedo.VKey
        */
        KEY_Y : 'Y',
        /**
        * @constant
        * @name KEY_Z
        * @memberof accedo.VKey
        */
        KEY_Z : 'Z',
        /**
        * @constant
        * @name KEY_A_LOWER
        * @memberof accedo.VKey
        */
        KEY_A_LOWER : 'a',
        /**
        * @constant
        * @name KEY_B_LOWER
        * @memberof accedo.VKey
        */
        KEY_B_LOWER : 'b',
        /**
        * @constant
        * @name KEY_C_LOWER
        * @memberof accedo.VKey
        */
        KEY_C_LOWER : 'c',
        /**
        * @constant
        * @name KEY_D_LOWER
        * @memberof accedo.VKey
        */
        KEY_D_LOWER : 'd',
        /**
        * @constant
        * @name KEY_E_LOWER
        * @memberof accedo.VKey
        */
        KEY_E_LOWER : 'e',
        /**
        * @constant
        * @name KEY_F_LOWER
        * @memberof accedo.VKey
        */
        KEY_F_LOWER : 'f',
        /**
        * @constant
        * @name KEY_G_LOWER
        * @memberof accedo.VKey
        */
        KEY_G_LOWER : 'g',
        /**
        * @constant
        * @name KEY_H_LOWER
        * @memberof accedo.VKey
        */
        KEY_H_LOWER : 'h',
        /**
        * @constant
        * @name KEY_I_LOWER
        * @memberof accedo.VKey
        */
        KEY_I_LOWER : 'i',
        /**
        * @constant
        * @name KEY_J_LOWER
        * @memberof accedo.VKey
        */
        KEY_J_LOWER : 'j',
        /**
        * @constant
        * @name KEY_K_LOWER
        * @memberof accedo.VKey
        */
        KEY_K_LOWER : 'k',
        /**
        * @constant
        * @name KEY_L_LOWER
        * @memberof accedo.VKey
        */
        KEY_L_LOWER : 'l',
        /**
        * @constant
        * @name KEY_M_LOWER
        * @memberof accedo.VKey
        */
        KEY_M_LOWER : 'm',
        /**
        * @constant
        * @name KEY_N_LOWER
        * @memberof accedo.VKey
        */
        KEY_N_LOWER : 'n',
        /**
        * @constant
        * @name KEY_O_LOWER
        * @memberof accedo.VKey
        */
        KEY_O_LOWER : 'o',
        /**
        * @constant
        * @name KEY_P_LOWER
        * @memberof accedo.VKey
        */
        KEY_P_LOWER : 'p',
        /**
        * @constant
        * @name KEY_Q_LOWER
        * @memberof accedo.VKey
        */
        KEY_Q_LOWER : 'q',
        /**
        * @constant
        * @name KEY_R_LOWER
        * @memberof accedo.VKey
        */
        KEY_R_LOWER : 'r',
        /**
        * @constant
        * @name KEY_S_LOWER
        * @memberof accedo.VKey
        */
        KEY_S_LOWER : 's',
        /**
        * @constant
        * @name KEY_T_LOWER
        * @memberof accedo.VKey
        */
        KEY_T_LOWER : 't',
        /**
        * @constant
        * @name KEY_U_LOWER
        * @memberof accedo.VKey
        */
        KEY_U_LOWER : 'u',
        /**
        * @constant
        * @name KEY_V_LOWER
        * @memberof accedo.VKey
        */
        KEY_V_LOWER : 'v',
        /**
        * @constant
        * @name KEY_W_LOWER
        * @memberof accedo.VKey
        */
        KEY_W_LOWER : 'w',
        /**
        * @constant
        * @name KEY_X_LOWER
        * @memberof accedo.VKey
        */
        KEY_X_LOWER : 'x',
        /**
        * @constant
        * @name KEY_Y_LOWER
        * @memberof accedo.VKey
        */
        KEY_Y_LOWER : 'y',
        /**
        * @constant
        * @name KEY_Z_LOWER
        * @memberof accedo.VKey
        */
        KEY_Z_LOWER : 'z',
        /**
        * @constant
        * @name KEY_SPACE
        * @memberof accedo.VKey
        */
        KEY_SPACE : ' ',
        /**
        * @constant
        * @name KEY_TILDE
        * @memberof accedo.VKey
        */
        KEY_TILDE : '~',
        /**
        * @constant
        * @name KEY_BACK_QUOTE
        * @memberof accedo.VKey
        */
        KEY_BACK_QUOTE : '`',
        /**
        * @constant
        * @name KEY_EXC
        * @memberof accedo.VKey
        */
        KEY_EXC : '!',
        /**
        * @constant
        * @name KEY_AT
        * @memberof accedo.VKey
        */
        KEY_AT : '@',
        /**
        * @constant
        * @name KEY_HASH
        * @memberof accedo.VKey
        */
        KEY_HASH : '#',
        /**
        * @constant
        * @name KEY_DOLLAR
        * @memberof accedo.VKey
        */
        KEY_DOLLAR : '$',
        /**
        * @constant
        * @name KEY_PERCENT
        * @memberof accedo.VKey
        */
        KEY_PERCENT : '%',
        /**
        * @constant
        * @name KEY_CARET
        * @memberof accedo.VKey
        */
        KEY_CARET : '^',
        /**
        * @constant
        * @name KEY_AMP
        * @memberof accedo.VKey
        */
        KEY_AMP : '&',
        /**
        * @constant
        * @name KEY_STAR
        * @memberof accedo.VKey
        */
        KEY_STAR : '*',
        /**
        * @constant
        * @name KEY_OPEN_PAREN
        * @memberof accedo.VKey
        */
        KEY_OPEN_PAREN: '(',
        /**
        * @constant
        * @name KEY_CLOSE_PAREN
        * @memberof accedo.VKey
        */
        KEY_CLOSE_PAREN : ')',
        /**
        * @constant
        * @name KEY_MINUS
        * @memberof accedo.VKey
        */
        KEY_MINUS : '-',
        /**
        * @constant
        * @name KEY_UNDERSCORE
        * @memberof accedo.VKey
        */
        KEY_UNDERSCORE: '_',
        /**
        * @constant
        * @name KEY_EQUALS
        * @memberof accedo.VKey
        */
        KEY_EQUALS : '=',
        /**
        * @constant
        * @name KEY_PLUS
        * @memberof accedo.VKey
        */
        KEY_PLUS : '+',
        /**
        * @constant
        * @name KEY_OPEN_BRACKET
        * @memberof accedo.VKey
        */
        KEY_OPEN_BRACKET: '[',
        /**
        * @constant
        * @name KEY_OPEN_BRACE
        * @memberof accedo.VKey
        */
        KEY_OPEN_BRACE : '{',
        /**
        * @constant
        * @name KEY_CLOSE_BRACKET
        * @memberof accedo.VKey
        */
        KEY_CLOSE_BRACKET : ']',
        /**
        * @constant
        * @name KEY_CLOSE_BRACE
        * @memberof accedo.VKey
        */
        KEY_CLOSE_BRACE: '}',
        /**
        * @constant
        * @name KEY_BACK_SLASH
        * @memberof accedo.VKey
        */
        KEY_BACK_SLASH : '\\',
        /**
        * @constant
        * @name KEY_PIPE
        * @memberof accedo.VKey
        */
        KEY_PIPE : '|',
        /**
        * @constant
        * @name KEY_SEMICOLON
        * @memberof accedo.VKey
        */
        KEY_SEMICOLON: ';',
        /**
        * @constant
        * @name KEY_COLON
        * @memberof accedo.VKey
        */
        KEY_COLON : ':',
        /**
        * @constant
        * @name KEY_SINGLE_QUOTE
        * @memberof accedo.VKey
        */
        KEY_SINGLE_QUOTE : "'",
        /**
        * @constant
        * @name KEY_QUOTE
        * @memberof accedo.VKey
        */
        KEY_QUOTE: '"',
        /**
        * @constant
        * @name KEY_COMMA
        * @memberof accedo.VKey
        */
        KEY_COMMA : ',',
        /**
        * @constant
        * @name KEY_LESS
        * @memberof accedo.VKey
        */
        KEY_LESS : '<',
        /**
        * @constant
        * @name KEY_PERIOD
        * @memberof accedo.VKey
        */
        KEY_PERIOD: '.',
        /**
        * @constant
        * @name KEY_GREATER
        * @memberof accedo.VKey
        */
        KEY_GREATER : '>',
        /**
        * @constant
        * @name KEY_SLASH
        * @memberof accedo.VKey
        */
        KEY_SLASH : '/',
        /**
        * @constant
        * @name KEY_QUESTION
        * @memberof accedo.VKey
        */
        KEY_QUESTION: '?',
        /**
        * @constant
        * @name KEY_EURO
        * @memberof accedo.VKey
        */
        KEY_EURO: "€",
        /**
        * @constant
        * @name KEY_DEGREE
        * @memberof accedo.VKey
        */
        KEY_DEGREE: '°',
        /**
        * @constant
        * @name KEY_KEYBOARD_BACKSPACE
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_BACKSPACE: 'backspace',
        /**
        * @constant
        * @name KEY_KEYBOARD_TAB
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_TAB: 'keyboard_tab',
        /**
        * @constant
        * @name KEY_KEYBOARD_INSERT
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_INSERT: 'keyboard_insert',
        /**
        * @constant
        * @name KEY_KEYBOARD_DELETE
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_DELETE: 'keyboard_delete',
        /**
        * @constant
        * @name KEY_KEYBOARD_HOME
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_HOME: 'keyboard_home',
        /**
        * @constant
        * @name KEY_KEYBOARD_END
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_END: 'keyboard_end',
        /**
        * @constant
        * @name KEY_KEYBOARD_PAGE_UP
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_PAGE_UP: 'keyboard_page_up',
        /**
        * @constant
        * @name KEY_KEYBOARD_PAGE_DOWN
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_PAGE_DOWN: 'keyboard_page_down',
        /**
        * @constant
        * @name KEY_KEYBOARD_UP
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_UP : 'keyboard_up',
        /**
        * @constant
        * @name KEY_KEYBOARD_DOWN
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_DOWN : 'keyboard_down',
        /**
        * @constant
        * @name KEY_KEYBOARD_LEFT
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_LEFT : 'keyboard_left',
        /**
        * @constant
        * @name KEY_KEYBOARD_RIGHT
        * @memberof accedo.VKey
        */
        KEY_KEYBOARD_RIGHT : 'keyboard_right',
        /**
        * @constant
        * @name KEY_PS3_TRIANGLE
        * @memberof accedo.VKey
        */
        KEY_PS3_TRIANGLE: 'ps3_triangle',
        /**
        * @constant
        * @name KEY_PS3_SQUARE
        * @memberof accedo.VKey
        */
        KEY_PS3_SQUARE: 'ps3_square',
        /**
        * @constant
        * @name KEY_PS3_L1
        * @memberof accedo.VKey
        */
        KEY_PS3_L1:'ps3_L1',
        /**
        * @constant
        * @name KEY_PS3_L2
        * @memberof accedo.VKey
        */
        KEY_PS3_L2: 'ps3_L2',
        /**
        * @constant
        * @name KEY_PS3_L3
        * @memberof accedo.VKey
        */
        KEY_PS3_L3: 'ps3_L3',
        /**
        * @constant
        * @name KEY_PS3_R1
        * @memberof accedo.VKey
        */
        KEY_PS3_R1: 'ps3_R1',
        /**
        * @constant
        * @name KEY_PS3_R2
        * @memberof accedo.VKey
        */
        KEY_PS3_R2: 'ps3_R2',
        /**
        * @constant
        * @name KEY_PS3_R3
        * @memberof accedo.VKey
        */
        KEY_PS3_R3: 'ps3_R3',
        /**
        * @constant
        * @name KEY_PS3_START
        * @memberof accedo.VKey
        */
        KEY_PS3_START: 'ps3_start',
        /**
        * @constant
        * @name KEY_PS3_SELECT
        * @memberof accedo.VKey
        */
        KEY_PS3_SELECT: 'ps3_select',
        /**
        * @constant
        * @name KEY_SS_HISTORY
        * @memberof accedo.VKey
        */
        KEY_SS_HISTORY: 'ss_history',
        /**
        * @constant
        * @name KEY_SS_CAMERA
        * @memberof accedo.VKey
        */
        KEY_SS_CAMERA: 'ss_camera',
        /**
        * @constant
        * @name KEY_OPTION
        * @memberof accedo.VKey
        */
        KEY_OPTION:'option',
        /**
        * @constant
        * @name KEY_OPTION
        * @memberof accedo.VKey
        */
        KEY_SEARCH:'search',
        /**
        * @constant
        * @name KEY_SETUP
        * @memberof accedo.VKey
        */
        KEY_SETUP:'setup',
        /**
        * @constant
        * @name KEY_AUDIO
        * @memberof accedo.VKey
        */
        KEY_AUDIO:'audio',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_A
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_A:'wii_classic_a',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_B
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_B:'wii_classic_b',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_X
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_X:'wii_classic_x',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_Y
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_Y:'wii_classic_y',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_L
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_L:'wii_classic_l',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_R
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_R:'wii_classic_r',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_ZL
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_ZL:'wii_classic_zl',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_ZR
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_ZR:'wii_classic_zr',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_PLUS
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_PLUS:'wii_classic_plus',
        /**
        * @constant
        * @name KEY_WII_CLASSIC_MINUS
        * @memberof accedo.VKey
        */
        KEY_WII_CLASSIC_MINUS:'wii_classic_minus',
        /**
        * @constant
        * @name KEY_WII_DRC_A
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_A:'wii_drc_a',
        /**
        * @constant
        * @name KEY_WII_DRC_B
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_B:'wii_drc_b',
        /**
        * @constant
        * @name KEY_WII_DRC_X
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_X:'wii_drc_x',
        /**
        * @constant
        * @name KEY_WII_DRC_Y
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_Y:'wii_drc_y',
        /**
        * @constant
        * @name KEY_WII_DRC_L
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_L:'wii_drc_l',
        /**
        * @constant
        * @name KEY_WII_DRC_R
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_R:'wii_drc_r',
        /**
        * @constant
        * @name KEY_WII_DRC_ZL
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_ZL:'wii_drc_zl',
        /**
        * @constant
        * @name KEY_WII_DRC_ZR
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_ZR:'wii_drc_zr',
        /**
        * @constant
        * @name KEY_WII_DRC_L_STICK
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_L_STICK:'wii_drc_l_stick',
        /**
        * @constant
        * @name KEY_WII_DRC_R_STICK
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_R_STICK:'wii_drc_r_stick',
        /**
        * @constant
        * @name KEY_WII_DRC_PLUS
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_PLUS:'wii_drc_plus',
        /**
        * @constant
        * @name KEY_WII_DRC_MINUS
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_MINUS:'wii_drc_minus',
        /**
        * @constant
        * @name KEY_WII_DRC_SYNC
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_SYNC:'wii_drc_sync',
        /**
        * @constant
        * @name KEY_WII_DRC_POWER
        * @memberof accedo.VKey
        */
        KEY_WII_DRC_POWER:'wii_drc_power',
        /**
        * @constant
        * @name KEY_WII_REMOTE_1
        * @memberof accedo.VKey
        */
        KEY_WII_REMOTE_1:'wii_remote_1',
        /**
        * @constant
        * @name KEY_WII_REMOTE_2
        * @memberof accedo.VKey
        */
        KEY_WII_REMOTE_2:'wii_remote_2',
        /**
        * @constant
        * @name KEY_WII_REMOTE_A
        * @memberof accedo.VKey
        */
        KEY_WII_REMOTE_A:'wii_remote_a',
        /**
        * @constant
        * @name KEY_WII_REMOTE_B
        * @memberof accedo.VKey
        */
        KEY_WII_REMOTE_B:'wii_remote_b',
        /**
        * @constant
        * @name KEY_WII_REMOTE_MINUS
        * @memberof accedo.VKey
        */
        KEY_WII_REMOTE_MINUS:'wii_remote_minus',
        /**
        * @constant
        * @name KEY_WII_REMOTE_PLUS
        * @memberof accedo.VKey
        */
        KEY_WII_REMOTE_PLUS:'wii_remote_plus',
        /**
        * @constant
        * @name KEY_WII_NUNCHUK_C
        * @memberof accedo.VKey
        */
        KEY_WII_NUNCHUK_C:'wii_nunchuk_c',
        /**
        * @constant
        * @name KEY_WII_NUNCHUK_Z
        * @memberof accedo.VKey
        */
        KEY_WII_NUNCHUK_Z:'wii_nunchuk_z'
});
/*jslint browser: true, devel: true */
/*global accedo: false,window:false */
/**
 * Cross-browser endpoints abstraction
 * @name Env
 * @memberof accedo
 * @module
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 */
accedo.Class.create('accedo.Env', accedo.Object.extend({
    /**
     * Event: onload
     * @constant
     * @name EVT_ONLOAD
     * @memberof accedo.module:Env
     */
    EVT_ONLOAD: "env-onload",
    /**
     * Event: onunload
     * @constant
     * @name EVT_ONUNLOAD
     * @memberof accedo.module:Env
     */
    EVT_ONUNLOAD: "env-onunload",
    /**
     * Event: onkey
     * @constant
     * @name EVT_ONKEY
     * @memberof accedo.module:Env
     */
    EVT_ONKEY: "env-onkey",
    /**
     * Event: emit event when plug in the cable/unplug in the cable and it is only available in Samsung and LG,other devices won't emit this event
     * @constant
     * @name  EVT_LAN_STATUS_CHANGED
     * @memberof accedo.module:Env
     */
    EVT_LAN_STATUS_CHANGED: "evn-lan-status-changed",
    /**
     * Event: emit event when connecting to internet/HTTP status changed. it is different from the "EVT_ONCONNECTION_CHANGED" as this one will
     * keep pooling and update the status. when there is no reponse,it will emit the status change event.
     * P.S.samsung wont use ajax and gets the variable in the samsung API directly.
     * @constant
     * @name EVT_HTTP_STATUS_CHANGED
     * @memberof accedo.module:Env
     */
    EVT_HTTP_STATUS_CHANGED: "env-http-status-changed",
    /**
     * Whether the DOM is ready.
     * @name _domready
     * @memberof accedo.module:Env
     * @static
     * @private
     */
    _domready: false,
    /**
     * Whether the deivce is ready.
     * @name _deviceReady
     * @memberof accedo.module:Env
     * @static
     * @private
     */
    _deviceReady: false,
    /**
     * The screen resolution (will be assigned upon device initialization).
     * @name resolution
     * @memberof accedo.module:Env
     * @static
     * @public
     */
    resolution: null,
    /** To save the connection status when first add eventListener of EVT_NETWORK_STATUS_CHANGED
     * @private
     * @name _LANStatus
     * @memberof accedo.module:Env
     */
    _LANStatus: null,
    /** To save the network status when first add eventListener of EVT_ONCONNECTION_CHANGED
     * @private
     * @name _HTTPStatus
     * @memberof accedo.module:Env
     */
    _HTTPStatus: null,
    /** To save the internet status when first add eventListener of EVT_ONCONNECTION_CHANGED
     * @private
     * @name _checkHTTPTimeOut
     * @memberof accedo.module:Env
     */
    _checkHTTPTimeOut: null,
    /** To save the connect status when first add eventListener of EVT_NETWORK_STATUS_CHANGED
     * @private
     * @name _checkLANInterval
     * @memberof accedo.module:Env
     */
    _checkLANInterval:null,
    /** The url to check the network status
     * @private
     * @name _url
     * @memberof accedo.module:Env
     */
    _url: "",
    /** The time to check next http network connection 
     * @private
     * @name _HTTPCheckTime
     * @memberof accedo.module:Env
     */
    _HTTPCheckTime: 30000,
    /** the timeout of each request
     * @private
     * @name _HTTPTimeout
     * @memberof accedo.module:Env
     */
    _HTTPTimeout: 20000,
    /**
     * Object for storing whitelisted keys
     * @private
     * @memberof accedo.module:Env
     */
    _whitelist: {},

    /**
     * Is currently blocking keys
     * @private
     * @memberof accedo.module:Env
     */
    _isBlocking: false,
    
    /**
     * Is currently blocking Mouse
     * @private
     * @memberof accedo.module:Env
     */
    // XDK CHANGE
    // _isBlockingMouse:false,
    /**
     * Start blocking mouse events. By default, power, exit, home, stop keys are whitelisted.
     * @name blockMouse
     * @public
     * @memberof accedo.module:Env
     * @function
     */
    // XDK CHANGE
    // blockMouse: function() {
    //     this._isBlockingMouse = true;
    // },
     /**
     * Start blocking mouse events. By default, power, exit, home, stop keys are whitelisted.
     * @name unblockMouse
     * @public
     * @memberof accedo.module:Env
     * @function
     */
    // XDK CHANGE
    // unblockMouse: function() {
    //     this._isBlockingMouse = false;
    // },
     /**
     * To get the status whether block mouse
     * @name isBlockingMouse
     * @public
     * @memberof accedo.module:Env
     * @function
     */
    // XDK CHANGE
    // isBlockingMouse:function(){
    //   return this._isBlockingMouse;  
    // },
    /**
     * Start blocking keys. By default, power, exit, home, stop keys are whitelisted.
     * @name blockKeys
     * @public
     * @memberof accedo.module:Env
     * @function
     */
    blockKeys: function() {
        this._isBlocking = true;
    },

    /**
     * Stop blocking keys
     * @name unblockKeys
     * @public
     * @memberof accedo.module:Env
     * @function
     */
    unblockKeys: function() {
        this._isBlocking = false;
    },

    /**
     * Is blocking keys
     * @name isBlockingKeys
     * @public
     * @memberof accedo.module:Env
     * @function
     * @return {Boolean}
     */
    isBlockingKeys: function() {
        return this._isBlocking;
    },

    /**
     * Adds key(s) to be whitelisted when blocking is turned on.
     * By default, power, exit, home, stop keys are whitelisted.
     * @name addWhitelistedKey
     * @param {String|Array} key keys(s) to be whitelisted
     * @function
     * @public
     * @memberof accedo.module:Env
     */
    addWhitelistedKey: function(key) {
        var i;
        if (accedo.Object.isString(key)) {
            this._whitelist[key] = true;
        } else {
            for(i in key){
                this._whitelist[key[i]] = true;
            }
        }
    },

    /**
     * Alias to addWhitelistedKey().
     * @name addWhitelistedKeys
     * @param {String|Array} key keys(s) to be whitelisted
     * @function
     * @public
     * @memberof accedo.module:Env
     */
    addWhitelistedKeys: function(key) {
        this.addWhitelistedKey(key);
    },

    /**
     * Set all the key that exists in an object's members to be whitelisted (existing settings will get over-written).
     * @name setWhitelist
     * @param {Object} whitelist keys to be whitelisted, in an object map
     * @function
     * @public
     * @memberof accedo.module:Env
     */
    setWhitelist: function(whitelist) {
        this._whitelist = whitelist;
    },

    /**
     * Removes key to be whitelisted when blocking is turned on.
     * @name removeWhitelistedKey
     * @param {String} key key to be removed from whitelist
     * @function
     * @public
     * @memberof accedo.module:Env
     */
    removeWhitelistedKey: function(key) {
        delete this._whitelist[key];
    },

    /**
     * Return false to stop event dispatch if blocking a key.
     * @name onDispatchEvent
     * @function
     * @protected
     * @memberof accedo.module:Env
     */
    onDispatchEvent: function(type, data){
        return !this._isBlocking ||
        type !== accedo.Env.EVT_ONKEY ||
        !accedo.Object.isUndefined(this._whitelist[data]);
    },
    /**
     * Returns whether the current platform supports mouse or not.
     * @name hasMouse
     * @memberof accedo.module:Env
     * @function
     * @returns {Boolean} 
     * @static
     * @public
     */
    hasMouse: function() {
        // XDK_CHANGE: No mouse code necessary
        // return accedo.device.system.hasMouse();
    },
    /**
     * Initializes this class.
     * @name main
     * @memberof accedo.module:Env
     * @function
     * @static
     * @public
     */
    main: function() {
        //initialize EventDispatcher
        this.initMixin();

        this.addWhitelistedKeys([
            accedo.VKey.KEY_POWER,
            accedo.VKey.KEY_EXIT,
            accedo.VKey.KEY_HOME,
            accedo.VKey.KEY_STOP
            ]);

        // window.onload = accedo.Fn.bind(this._onload,this);  // XDK CHANGE (we use our own delayed onload)
        window.onunload = accedo.Fn.bind(this._onunload,this);
    },
    /**
     * Default device initialize callback.
     * @name _onDeviceInited
     * @memberof accedo.module:Env
     * @function
     * @static
     * @private
     */
    _onDeviceInited: function() {
        
        accedo.console.log("accedo.Env.EVT_ONLOAD");
        
        // XDK CHANGE        
        //init resolution
        // this.resolution = accedo.device.appengine.getResolution();

        // init key handling
        accedo.device.tvkey.initKeyHandling();

        accedo.Env._deviceReady = true;
        
        //call onload when device is ready
        this.dispatchEvent(this.EVT_ONLOAD);
    },
    /**
     * Default window onload handler.
     * @name _onload
     * @memberof accedo.module:Env
     * @function
     * @static
     * @private
     */
    _onload: function() {
        accedo.console.log("dom ready");
        
        accedo.Env._domready = true;
        
        accedo.device.Manager.init(accedo.Fn.bind(this._onDeviceInited,this));
    },
    /**
     * Default window onunload handler.
     * @name _onunload
     * @memberof accedo.module:Env
     * @function
     * @static
     * @private
     */
    _onunload: function () {
        //deinit device manager
        accedo.device.Manager.deinit(accedo.Fn.bind(function() {
            this.dispatchEvent(this.EVT_ONUNLOAD);
        },this));
    },
    /**
     * return the current connection status whether connect to cable/wifi
     * @name getConnectionStatus
     * @public
     * @memberof accedo.module:Env
     * @function
     * @return {boolean} false when there is no internet status
     */
    getConnectionStatus:function(){
        return accedo.device.system.checkConnection();
    },
    /**
     * to start the interval and keep checking the connection of internet
     * P.S LG will automatically show the network disconnected and exit the apps
     * @name _checkConnection
     * @param {boolean} online true if the new status is online
     * @private
     * @memberof accedo.module:Env
     * @function
     */
    _checkConnection:function(online){
        this.dispatchEvent(this.EVT_LAN_STATUS_CHANGED, online);
    },
    /**
     * to set time interval to check the physical connection
     * @name setConnectionCheckTime
     * @public
     * @param {number} time in ms to check the physical connection
     * @memberof accedo.module:Env
     * @function
     */
    setConnectionCheckTime:function(time){
        //To give device to handle
        if(accedo.device.system.setPoolingTime){
            accedo.device.system.setPoolingTime(time);
        }
    },
    /**
     * to get time interval to check the physical connection
     * @name getConnectionCheckTime
     * @public
     * @return {number} time in ms to check the physical connection
     * @memberof accedo.module:Env
     * @function
     */
    getConnectionCheckTime:function(){
        if(accedo.device.system.getPoolingTime){
            return accedo.device.system.getPoolingTime();
        }
        return 0;
    },
    /**
     * to set the url for checking the network
     * @name setCheckHTTPUrl
     * @public
     * @param {string} url the url to ping and send request
     * @param {number} checkTime (optional) ms interval to send request each,default will be 30sec.
     * @param {number} timeOut (optional) ms timeout of each request,default will be 20sec.
     * @memberof accedo.module:Env
     * @function
     */
    setCheckHTTPUrl:function(url, checkTime, timeOut){
        this._url = url;
        this._HTTPCheckTime =  checkTime || 30000;
        this._HTTPTimeout = timeOut || 20000;
    },
    /**
     * to set the url for checking the network
     * @name getCheckHTTPUrl
     * @public
     * @return {Object} Object contain the information about the check HTTP connection
     * @return {string} Object.url the url of sending the request to check the connection
     * @return {number} Object.httpCheckTime the time to check the http connection
     * @return {number} Object.httpTimeOut the time to throw the timeout
     * @memberof accedo.module:Env
     * @function
     */
    getCheckHTTPUrl:function(){
        return {
            url:this._url,
            httpCheckTime: this._HTTPCheckTime,
            httpTimeOut: this._HTTPTimeout
        };
    },
    /**
     * to start the interval and keep checking the connection of internet and check every 30 seconds
     * @name _checkHTTP
     * @private
     * @memberof accedo.module:Env
     * @function
     */
    _checkHTTP:function(url){
        var http = null , connection = null, callback, timeOutFunc;
        
        //first time checking and assign the value directly
        accedo.device.system._checkHTTP(url, accedo.Fn.bind(function(http){
            this._HTTPStatus = http;
        },this));
        
        if(this._checkHTTPTimeOut){
            clearTimeout(this._checkHTTPTimeOut);
            this._checkHTTPTimeOut = null;
        }
        
        callback  = accedo.Fn.bind(function(http){
            if(http !== this._HTTPStatus){
                this._HTTPStatus = http;
                this.dispatchEvent(this.EVT_HTTP_STATUS_CHANGED, http);
            }
            clearTimeout(this._checkHTTPTimeOut);
            this._checkHTTPTimeOut = setTimeout(timeOutFunc,this._HTTPCheckTime); 
        },this);
        
        timeOutFunc = accedo.Fn.bind(function(){
            connection = accedo.device.system.checkConnection();
            //to throw disconnect immediate when the physical connection is false. as no physical connection = no HTTP network connection
            if(connection === false && connection !== this._HTTPStatus){
                this._HTTPStatus = connection;
                this.dispatchEvent(this.EVT_HTTP_STATUS_CHANGED, this._HTTPStatus);
                clearTimeout(this._checkHTTPTimeOut);
                this._checkHTTPTimeOut = setTimeout(timeOutFunc,this._HTTPCheckTime);
                return;
            }
            accedo.device.system._checkHTTP(url, callback);
        },this);       
        this._checkHTTPTimeOut = setTimeout(timeOutFunc,this._HTTPCheckTime);
    },
    /**
     * to capture the event listener after added
     * @name onListenerAdded
     * @protected
     * @memberof accedo.module:Env
     * @function
     */
    onListenerAdded:function(type){
        switch(type){
            case this.EVT_HTTP_STATUS_CHANGED:
                this._checkHTTP(this._url);
                break;
            case this.EVT_LAN_STATUS_CHANGED:
                if(accedo.device.system.hasNetworkDetection()){
                    accedo.device.system.addEventListener(accedo.device.system.EVT_NETWORK_STATUS_CHANGED,accedo.Fn.bind(this._checkConnection,this));
                }
                break;
        }
    },
    /**
     * to capture the event listener after removed
     * @name onListenerRemoved
     * @protected
     * @memberof accedo.module:Env
     * @function
     */
    onListenerRemoved:function(type){
        switch(type){
            case this.EVT_HTTP_STATUS_CHANGED:
                if(this._checkHTTPTimeOut){
                    clearTimeout(this._checkHTTPTimeOut);
                    this._checkHTTPTimeOut = null;
                }
                break;
            case this.EVT_LAN_STATUS_CHANGED:
                if(accedo.device.system.hasNetworkDetection()){
                    accedo.device.system.removeEventListener(accedo.device.system.EVT_NETWORK_STATUS_CHANGED,accedo.Fn.bind(this._checkConnection,this));
                }
                break;
        }
           
    }
}, accedo.mixin.EventDispatcher));
/*jslint browser: true, devel: true */
/*global accedo: false , ActiveXObject:false*/

/**
 * @name Ajax
 * @desc A set of convenience functions used for communicating with a server.
 * @memberof accedo
 * @module
 * @author <a href="mailto:alex@accedobroadband.com">Alexej Kubarev</a>
 */
accedo.Class.create('accedo.Ajax', {
    /**
     * Gets a transport by trying all of the know XHR objects
     * @return A new XHR object or false if none found
     * @memberof accedo.module:Ajax
     * @function
     * @private
     */
    _getTransport : function() { // TODO: use first call branching to improve performance
        return accedo.Try(
            function() {return new XMLHttpRequest();},
            function() {return new ActiveXObject('Msxml2.XMLHTTP');},
            function() {return new ActiveXObject('Microsoft.XMLHTTP');}
        ) || false;
//        return accedo.device.Transport.create();
    },
    /**
     * Sends an ajax request.
     * @name request
     * @param {String} url The URL to send request to.
     * @param {Function} options.onFailure (optional) the function when the request is failure (including connection timeout)
     * @param {Function} options.onSuccess (optional) the function when the request is success
     * @param {Function} options.onException (optional) the function when there is some error or exception
     * @param {Function} options.onComplete (optional) the function when the request is completed no matter it is success or failed.
     * @param {Function} options.onAbort (optional) the function when the request is aborted.
     * @param {Boolean} options.cache (optional) Set this to true if javascript caching of the response is required. Defaults to false.
     * @param {Number} options.cacheLifeTime (optional) Set the desired cache life time when caching is enabled.
     * @param {Function} options.dataExtractor <pre>
     * (optional) A function that transforms the transport into a data object for onSucess callback (saves cache memory)
     * e.g. options.dataExtractor = function(transport){
     *          return {name : transport.responseJSON.name}; // this returned object will become input for onSucess callback
     *      }
     * </pre>
     * @param {Number} options.timeOut (optional) when there is no response from server,it will be timeout and throw failure
     * @param {String} options.method (optional) the HTTP request method, i.e. "get" "post", use lower case only! Defaults to "get".
     * @param {Number} options.timeOut (optional) when there is no response from server,it will be timeout and throw failure
     * @param {String} options.async (optional) whether the request to send is asynchronous or not. Defaults to true.
     * @param {Object Map} options.requestHeaders (optional) the header settings, should be contained in an Javascript plain object.
     * @param {String} options.contentType (optional) content type setting of the request.
     * @param {String} options.encoding (optional) encoding setting of the request.
     * @param {String} options.postBody (optional) the post body for the "post" and "put" reqeust.
     * @param {Number} options.timeOut (optional) request timeout interval, defaults to 30 seconds.
     * @param {String | accedo.Hash} options.parameters (optional) the request parameters.
     * @return A new accedo.AjaxRequest object with send triggered
     * @memberof accedo.module:Ajax
     * @function
     * @example accedo.Ajax.request("http://www.example.com/ajax", options);
     * @example POST request and send as form data accedo.Ajax.request("http://analytics.appledaily.com.hk/smarttv/m.php", { method: "post",contentType: "application/x-www-form-urlencoded",parameters:new accedo.Hash({type:"pageview",pageview:"/article/2"})})
     * @static
     * 
     */
    request: function (url, options) {
        
        var ret = new accedo.AjaxRequest(url,options);
        //Manage the transport and send the request
        ret.send();
        
        return ret;
    }
});
/*jslint browser: true, devel: true, strict: false */
/*global accedo: false,window:false */
/**
 * @desc Caching module for Ajax requests.
 * @name AjaxCache
 * @memberof accedo
 * @module
 * @author <a href="mailto:daniel.deng@accdeo.tv">Daniel Deng</a>
 */
accedo.Class.create('accedo.AjaxCache',{
    /**
     * Whether is initialized
     * @private
     */
    _isInitialized: false,
    
    /**
     * Default cache life time
     * @private
     */
    _defaultLifeTime: 60, // in seconds

    /**
     * Default purge interval
     * @private
     */
    _purgeInterval: 60, // in seconds

    /**
     * Default purge interval ID
     * @private
     */
    _purgeId: null,

    /**
     * Cache data container
     * @private
     */
    _cache: {},

    /**
     * String key for "undefined" stuff
     * @private
     */
    UNDEFINED_KEY: "__undefined__",

    /**
     * Adds a ajax response into the cache. You should only cache "GET" requests, as it is done in accedo.AjaxRequest.
     * @name add
     * @memberof accedo.module:AjaxCache
     * @function
     * @param {String} url Destination URL
     * @param {Object} options The options object for the ajax request
     * @param {Object} data The data object to be stored, if null, it will be constructed from transport parameter
     * @param {Object} transport The response transport object
     * @param {Number} cacheLifeTime (optional) Life time of the cache
     * @return {Boolean} operation successful or not
     * @static
     * @public
     */
    add: function(url, options, data, transport, cacheLifeTime) {
        var cache = this._cache,
        params = options.parameters, headers = options.requestHeaders,
        paramKey, headerKey;

        if (!this._isInitialized) {
            this.initialize();
        }

        cacheLifeTime = cacheLifeTime || options.cacheLifeTime;

        data = data || {
            status: transport.status,
            statusText: transport.statusText,
            response: transport.response,
            responseText: transport.responseText,
            responseXML: transport.responseXML,
            responseJSON: transport.responseJSON
        };

        data.isCached = true;

        //Create the structure if needed: url > param > header > [cached object]
        paramKey = accedo.Object.isUndefined(params) || accedo.Object.isNull(params) ?
            this.UNDEFINED_KEY :
            accedo.Object.isString(params) ? params : this._flatten(params.toObject());
        headerKey = accedo.Object.isUndefined(headers) || accedo.Object.isNull(headers) ?
            this.UNDEFINED_KEY :
            this._flatten(headers);
        if (!cache[url]) {
            cache[url] = {};
        }
        if (!cache[url][paramKey]) {
            cache[url][paramKey] = {};
        }
        cache[url][paramKey][headerKey] = {
            _EXP_TIMESTAMP: (new Date()).getTime() + (cacheLifeTime || this._defaultLifeTime) * 1000,
            options: options,
            data: data
        };

        accedo.console.debug('accedo.AjaxCache.add(): cached response for: '+url+' > '+paramKey+' > '+headerKey);
        return true;
    },

    /**
     * Gets a ajax response from the cache
     * @name get
     * @memberof accedo.module:AjaxCache
     * @function
     * @param {String} url Destination URL
     * @param {Object} options The options object for the ajax request
     * @return {Object} The cached item or null if not found
     * @static
     * @public
     */
    get: function(url, options) {
        /**
         * Check if cache is set.
         * If so, go through the cache and find the element with same 'args' as well as _EXP_TIMESTAMP <= CURRENT_TIMESTAMP
         * If no such Item found - return null.
         * @ignore
         */
        options = options || {};
        var cache = this._cache,
        params = options.parameters, headers = options.requestHeaders,
        paramKey = accedo.Object.isUndefined(params) || accedo.Object.isNull(params) ?
            this.UNDEFINED_KEY :
            accedo.Object.isString(params) ? params : this._flatten(params.toObject()),
        headerKey = accedo.Object.isUndefined(headers) || accedo.Object.isNull(headers) ?
            this.UNDEFINED_KEY :
            this._flatten(headers), cacheItem = null;

        if (cache[url] && cache[url][paramKey] && cache[url][paramKey][headerKey]) {

            cacheItem = cache[url][paramKey][headerKey];

            //Check whether the cached data is recent enough
            if (cacheItem._EXP_TIMESTAMP > (new Date()).getTime()) {
                // everything is fine
                cacheItem = cacheItem.data;
            } else {
                //cached data has expired, delete the entry
                accedo.console.info('Cache expired for: '+url+' > '+paramKey+' > '+headerKey);
                delete cache[url][paramKey][headerKey];
                cacheItem = null;
            }
        }
        accedo.console.debug('accedo.AjaxCache.get(): cache ' + (cacheItem ? 'hit' : 'missed') + ' for: '+url+' > '+paramKey+' > '+headerKey);
        return cacheItem;
    },

    /**
     * Removes all the expired items from the cache
     * @name purge
     * @memberof accedo.module:AjaxCache
     * @function
     * @static
     * @public
     */
    purge: function(){
        var cache = this._cache,
        url, paramKey, headerKey, curItem;

        accedo.console.debug("AjaxCache: purging started...");
        for(url in cache){
            for(paramKey in cache[url]){
                if(!cache[url].hasOwnProperty(paramKey)){
                    continue;
                }
                for(headerKey in cache[url][paramKey]){
                    if(!cache[url][paramKey].hasOwnProperty(headerKey)){
                        continue;
                    }
                    curItem = cache[url][paramKey][headerKey];
                    if(curItem._EXP_TIMESTAMP <= (new Date()).getTime()){
                        accedo.console.debug('AjaxCache: Expired cache purged for: url='+url+' > '+paramKey+' > '+headerKey);
                        delete cache[url][paramKey][headerKey];
                    }
                }
            }
        }
    },

    /**
     * Flatten an object into a string, will escape the keys and values into URL encoded string
     * @name _flatten
     * @memberof accedo.module:AjaxCache
     * @function
     * @return {Object} The flatten string
     * @static
     * @private
     */
    _flatten: function(obj){
        var keys = [], ret = "?", i = 0, len,key;
        obj = obj || {};

        for (key in obj) {
            if(obj.hasOwnProperty(key)){
                keys.push(key);
            }
        }

        for (len = keys.length; i < len; i++) {
            key = keys[i];
            ret += encodeURIComponent(key) + "=" + encodeURIComponent(obj[key]) + "&";
        }
            
        return ret;
    },
    /**
     * Schedule purging for every specified interval
     * @name schedulePurge
     * @memberof accedo.module:AjaxCache
     * @param {Number} interval (optional) Purging interval
     * @function
     * @static
     * @public
     */
    schedulePurge: function(interval){
        if (this._purgeId) {
            clearInterval(this._purgeId);
        }
        this._purgeInterval = interval || this._purgeInterval;

        this._purgeId = window.setInterval(accedo.Fn.bind(this.purge, this), this._purgeInterval * 1000);
        accedo.console.debug('AjaxCache: scheduled cache purging.');
    },
    /**
     * Set the purging interval
     * @name setPurgeInterval
     * @memberof accedo.module:AjaxCache
     * @param {Number} interval Purging interval
     * @function
     * @static
     * @public
     */
    setPurgeInterval: function(interval){
        this._purgeInterval = interval;
        this.schedulePurge();
    },
    /**
     * Set the default cache life time
     * @name setDefaultCacheLifeTime
     * @memberof accedo.module:AjaxCache
     * @param {Number} sec Default cache life time to set
     * @function
     * @static
     * @public
     */
    setDefaultCacheLifeTime: function(sec){
        this._defaultLifeTime = sec;
    },
    /**
     * Init the cache
     * @name initialize
     * @memberof accedo.module:AjaxCache
     * @function
     * @static
     * @private
     */
    initialize: function(){
        if(this._isInitialized){
            return;
        }
        this._isInitialized = true;
        this.schedulePurge();
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */

/**
 * @desc <pre>
 * Please use accedo.Ajax module if you are not sure what this class is for.
 *
 * Tested platform : Samsung, LG, Sharp, huawei, opera tv store
 * LG,Samsung 2011TV,Samsung 2012 TV, opera and workstation support post, get, put and delete methods
 * Samsung 2010 TV supports post,get and put methods.But method delete will crash the apps
 * Sharp and huawei supports post and get methods. Delete and put methods will automatically change into get method
 * </pre>
 * @name AjaxRequest
 * @memberof accedo
 * @class The internal Ajax request object used by accedo.Ajax
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 */
accedo.Class.create('accedo.AjaxRequest',["accedo.AjaxCache"], {},
{               
    /**
     * URL to send the request to.
     * @memberof accedo.AjaxRequest#
     * @field {String}
     * @private
     */
    _url : "",
    /**
     * Options mapping.
     * @memberof accedo.AjaxRequest#
     * @field {Object}
     * @private
     */
    _options: null,
    /**
     * Whether this request has finished.
     * @memberof accedo.AjaxRequest#
     * @field {boolean}
     * @private
     */
    _complete : false,
    /**
     * The native ajax transport object.
     * @memberof accedo.AjaxRequest#
     * @field {XMLHttpRequest}
     * @private
     */
    _transport : null,
    /**
     * The original URL address as passed in.
     * @memberof accedo.AjaxRequest#
     * @field {String}
     * @private
     */
    _originalUrl: "",
    /**
     * The original options object as passed in.
     * @memberof accedo.AjaxRequest#
     * @field {Object}
     * @private
     */
    _originalOptions : null,
    /**
     * To store the timeout when it sends out the request
     * @memberof accedo.AjaxRequest#
     * @field {Object}
     * @private
     */
    _connectionTimeout: null,
    /**
     * @memberof accedo.AjaxRequest#
     * @field {Object}
     * @private
     */
    _aborted : false,
    /**
     * Internal State change callback function.
     * Dispatched ReadyState handler if needed.
     * @name onStateChange
     * @memberof accedo.AjaxRequest#
     * @function
     * @private
     */
    _onStateChange : function() {
        var readyState = this._transport.readyState;
        if (readyState > 1 && !((readyState === 4) && this._complete)) {
            this._respondToReadyState(readyState);
        }
    },

    /**
     * Internal ReadyState handler.
     * Parses the response and also tries to parse the response to JSON if needed.
     * Calls state handlers from options array if such are set.
     * @name respondToReadyState
     * @memberof accedo.AjaxRequest#
     * @function
     * @private
     */
    _respondToReadyState : function(readyState) {
        var data = null;
        try {
            // State == 4 means DONE
            if(readyState === 4 && !this._aborted) {
                //when able to receive any data within time, clear timeout
                if(this._connectionTimeout){
                    clearTimeout(this._connectionTimeout);
                    this._connectionTimeout = null;
                }
                this._complete = true;
                var status = this._transport.status,success = true,resp = this._transport.responseText, contentType;
                //status 0 means its state is in UNSENT / OPENED or ERROR flag is set
                if( ((status >= 200 && status < 300) || status === 304 ) ) {
                    contentType = this._transport.getResponseHeader("content-type");
                    //Parse JSON is possible
                    if(!this._options.checkContentType) {
                        //backward compatible
                        try{
                            this._transport.responseJSON = accedo.String.parseJSON(resp);
                        }catch(ex){
                            accedo.console.log("unable to parse JSON");
                        }
                    }else if(contentType.indexOf("json") > -1){
                        this._transport.responseJSON = accedo.String.parseJSON(resp);
                    }

                    if(this._options.dataExtractor) {
                        data = this._options.dataExtractor(this._transport);
                    }

                    if(this._options.method === "get" && this._options.cache) { // Cache the response
                        accedo.AjaxCache.add(this._originalUrl, this._originalOptions, data, this._transport);
                    }

                    if(accedo.Object.isFunction(this._options.onSuccess)) {
                        this._options.onSuccess(data || this._transport);
                    }
                } else {
                    //Unsuccessfull request
                    success = false;
                    if(accedo.Object.isFunction(this._options.onFailure)) {
                        contentType = this._transport.getResponseHeader("content-type");
                        if(!this._options.checkContentType) {
                            try{
                                this._transport.responseJSON = accedo.String.parseJSON(resp);
                            }catch(ex){
                                accedo.console.log("unable to parse JSON");
                            }
                        }else if(contentType.indexOf("json") > -1){
                            this._transport.responseJSON = accedo.String.parseJSON(resp);
                        }
                        this._options.onFailure(this._transport);
                    }
                }
                // onComplete is always called when request is "DONE"
                if(accedo.Object.isFunction(this._options.onComplete)) {
                    this._options.onComplete(this._transport, success);
                }
                            
                //Prevent memory leaks in Samsung devices
                if (this._transport.destroy){
                    this._transport.destroy();
                }
                delete this._transport; // this is also for Samsung, on 3.0 compatible TVs
            }
        } catch (e) {
            if(accedo.Object.isFunction(this._options.onException)) {
                this._options.onException(this._transport, e);
                //this is for Samsung only, in official doc they say if not destroy() it'll cause memory issue
                if (this._transport.destroy){
                    this._transport.destroy();
                }
                delete this._transport; // this is also for Samsung, on 3.0 compatible TVs
            }
            accedo.console.error(e.stack);
        }
    },

    /**
     * Abort request as soon as possible.
     * @memberof accedo.AjaxRequest#
     * @public
     * @void
     */
    abort : function() {
        accedo.console.log("abort the transport");
        /* The idea is to allow user to abort the request and stop execution.*/
        //Notify this request has been aborted (will be checked against in the onSuccess)
        this._aborted = true;
                    
        try {
            this._transport.abort();
        } catch (e) {
            //Abort is not supported
            delete this._aborted;
        }
        if(accedo.Object.isFunction(this._options.onAbort)) {
            this._options.onAbort(this._transport);
        }
    },
                
    /**
     * Opens the transport, initialize it correctly and send the request
     * @memberof accedo.AjaxRequest#
     * @public
     * @void
     */
    send : function() {
        //Prepare request and send it
        try {
            if (this._transport.isCached) { // this is a cached fake transport, go success directly
                accedo.console.debug("ajax: response cached, marking as successful directly...");
                if(accedo.Object.isFunction(this._options.onSuccess)) {
                    this._options.onSuccess(this._transport);
                }
                if(accedo.Object.isFunction(this._options.onComplete)) {
                    this._options.onComplete(this._transport, true/* success=true */);
                }
                return;
            }

            this._transport.open(this._options.method.toUpperCase(), this._url, this._options.async);

            //ReadyState
            this._transport.onreadystatechange = accedo.Fn.bind(this._onStateChange, this);

            //Set request headers if needed
            var headers,name,avoid = false;
            headers = this._options.requestHeaders || {};
                        
            if (this._options.method === 'post') {
                headers['Content-type'] = this._options.contentType + (this._options.encoding ? '; charset=' + this._options.encoding : '');
            }
                        
            
            for (name in headers) {
                this._transport.setRequestHeader(name, headers[name]);
            }

            //Prepare body
            this.body = (this._options.method === 'post' || this._options.method === 'put') ? (this._options.postBody || this.params) : null;
            
            //final check for different platform to avoid those unsupported request
            switch(accedo.platform){
                case "huawei":
                case "sharp":
                    if(this._options.method === "delete" || this._options.method === "put" ){
                        avoid = true;
                    }
                    break;
                case "samsung":
                    if(accedo.device.id.getFirmwareYear() === 2010 && this._options.method === "delete"){
                        avoid = true;
                    }
                    break;
            }
            if(avoid){
                this.abort();
                accedo.console.debug("unsupport method in this device"+this._options.method);
                if(accedo.Object.isFunction(this._options.onException)) {
                    this._options.onException(this._transport);
                }
                return ;
            }    
            
            //set connection timeout before send
            this._connectionTimeout = setTimeout(accedo.Fn.bind(function () {
                this.abort();
                //to throw a failure
                if(accedo.Object.isFunction(this._options.onFailure)) {
                    this._options.onFailure(this._transport);
                }
                
                // onComplete is always called when request is "DONE"
                if(accedo.Object.isFunction(this._options.onComplete)) {
                    this._options.onComplete(this._transport, false);
                }
                //Prevent memory leaks in Samsung devices
                if (this._transport.destroy){
                    this._transport.destroy();
                }
                delete this._transport; // this is also for Samsung, on 3.0 compatible TVs
                this._connectionTimeout = null;
            },this),this._options.timeOut);
            
            this._transport.send(this.body);
            
            /* Force Firefox to handle ready state 4 for synchronous requests */
            /* Seems current Firefox does not have this issue, so code block removed */
            
            accedo.console.log("AjaxRequest: onStateChange call finished.");

        } catch (e) {
            if(accedo.Object.isFunction(this._options.onException)) {
                this._options.onException(this._transport, e);
            }
            accedo.console.error(e.stack);
        }
    },
    initialize: function(url, options) {
        //Request URL
        this._url = url;
        options = options || {};

        options.timeOut = accedo.Object.isNumber(options.timeOut)?options.timeOut:30000; //default is 30s 
        
        if(accedo.Object.isFunction(options.toObject)){ // options should be a Hash object, cast to plain object
            options = options.toObject();
        }

        this._originalUrl = url;
        this._originalOptions = options;

        // Defaults
        this._options = {
            method: 'get',
            async: true,
            contentType:  'text/plain',
            checkContentType: false,
            encoding:     'UTF-8',
            parameters:   ''
        };

        accedo.Object.extend(this._options, options);
        this._options.method = this._options.method.toLowerCase();

        if (this._options.method  === "get" && this._options.cache) { // Get cache of the response
            this._transport = accedo.AjaxCache.get(this._originalUrl, this._originalOptions);
            if (this._transport) {
                return;
            }
        }

        this._transport = accedo.Ajax._getTransport();
        //Request parameters
        this.params = accedo.Object.isString(this._options.parameters) ?
        this._options.parameters : (accedo.Object.isFunction(this._options.parameters.toQueryString) ? this._options.parameters.toQueryString(): "" );

        //If method is not GET or POST or DELETE or PUT, set _method argument and append it to querystring. Change method to POST
        if(['get', 'post', 'delete', 'put'].indexOf(this._options.method) === -1) {
            this.params += (this.params ? '&' : '') + "_method=" + this._options.method;
            this._options.method = 'post';
        }

        //Prepare URL for GET request
        if (this.params && this._options.method === 'get') {
            if ( accedo.Object.isString(this.params) && this.params.length>0){
                // when GET, append parameters to URL
                this._url += ((this._url.indexOf('?')>0) ? '&' : '?') + this.params;
            }
        }
    }

});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @name AbstractInterface
 * @memberof accedo.device
 * @class abstract class for device interfaces
 */
accedo.Class.create('accedo.device.AbstractInterface', {},{
   init: accedo.Class.abstractFn,
   deinit: accedo.Class.abstractFn
});
/*jslint browser: true, devel: true,newcap:false */
/*global accedo: false */
/**
 * @name Manager
 * @memberof accedo.device
 * @module
 * @desc <pre>
 * Manage loading device-specific interfaces. Load Devicepackage for current device platform.
 * Packages files: console, id, storage, tvkey, video. Interfaces will be loaded and handled in Devicepackage 
 * </pre>
 */
accedo.Class.create('accedo.device.Manager',
{
    /**
     * Timeout for device interface init
     * @constant
     * @name DEVICE_INIT_TIMEOUT
     * @memberof accedo.device.module:Manager
     */
    DEVICE_INIT_TIMEOUT: accedo.config.get('device.initTimeout',10 * 1000),
    /**
     * accedo.Hash to save interfaces
     * @static
     * @private
     * @name _interfaceTypes
     * @memberof accedo.device.module:Manager
     * @private
     */
    _interfaceTypes: {},
    /**
    * Counter to count interfaces loaded
    * @static
    * @private
    * @name _interfaceLoadingCount
    * @memberof accedo.device.module:Manager
    */
    _interfaceLoadingCount: 0,
    /**
     * onSuccess function to be called while success in loading interfaces
     * @static
     * @private
     * @name _onSuccess
     * @memberof accedo.device.module:Manager
     */
    _onSuccess: null,
    /**
     * onFailure function to be called while failure in loading interfaces
     * @static
     * @private
     * @name _onFailure
     * @memberof accedo.device.module:Manager
     */
    _onFailure: null,
    /**
     * Flag to determine if base device interfaces are inited
     * @private
     * @name _baseInited
     * @memberof accedo.device.module:Manager
     */
    _baseInited: false,
    /**
     * Flag to determine if device interfaces are inited
     * @private
     * @name _inited
     * @memberof accedo.device.module:Manager
     */
    _inited: false,
    /**
     * The target device string.
     * @name _targetDevice
     * @memberof accedo.device.module:Manager
     * @private
     */
    _targetDevice: accedo.config.get('device.target', false)?accedo.config.get('device.target', false):accedo.config.get('device.propriatery',false),
    /**
     * The device package definiation, assigned upon initialization.
     * @name _devicePackage
     * @memberof accedo.device.module:Manager
     * @private
     */
    _devicePackage: null,
    /**
     * @name init
     * @param {Function} onSuccess Success function to be called when success in loading interfaces
     * @param {Function} onFailure Failure function to be called when fail in loading interfaces
     * @function
     * @memberof accedo.device.module:Manager
     * @public
     * @private
     */
    init: function(onSuccess, onFailure) {
        if (this._inited) {
            if (onSuccess){
                onSuccess();
            }
            return;
        }
        
        this._onSuccess = onSuccess;
        this._onFailure = onFailure;
        
        var devicePackage,platform;
            
        if (this._targetDevice){
                
            accedo.console.info("Using configured target device "+this._targetDevice);
                
            devicePackage  = this._targetDevice + ".DevicePackage";
            platform = accedo.config.get('device.target.platform',false)?accedo.config.get('device.target.platform',false):accedo.config.get('device.propriatery.platform',false);
            if (platform===false) {
                accedo.console.error("device.name (or device.propriatery.platform) not set. (name (String) to be assigned to accedo.platform).");
                return;
            }
            accedo.platform = platform;
        }
        // use detected platform.
        // deprecated, will use "device.target.platform" config in the future
        else if (accedo.platform!=="unknown") {
            devicePackage  = "accedo.device." + accedo.platform + ".DevicePackage";
        } else {
            accedo.console.warn("unknown platform, propriatery device abstraction missing.");
            if (this._onFailure) {
                this._onFailure();
            }
            return;
        }
        
        accedo.loadDefinition(devicePackage, {
            onSuccess: accedo.Fn.bind(this._onDevicePackgeReady,this),
            onFailure: function() {
                if (onFailure){
                    onFailure();
                } else {
                    accedo.console.error("Device package not found:"+devicePackage);
                }
            }
        });
    },
    /**
    * @name deinit
    * @param {Function} callback Callback function to be called when deinit
    * @function
    * @memberof accedo.device.module:Manager
    * @public
    * @protected
    */
    deinit: function(callback) {
        //call deinit(destructor) by deinit the name
        //in PS3, it will crash when call accedo.undefine
        if(accedo.platform !== "ps3"){
            accedo.undefine(this._devicePackage);
        }
        
        /*
         * deinit registered interfaces
         */
        if(callback){
            callback();
        }
    },
    /**
    * @name _onDevicePackgeReady
    * @function
    * @memberof accedo.device.module:Manager
    * @private
    * @static
    */
    _onDevicePackgeReady: function (devicePackageDef) {
        this._interfaceLoadingCount=0;
        this._packageTimeoutHandle = setTimeout(
            accedo.Fn.bind(this._onDevicePackageInitFailure,this),
            this.DEVICE_INIT_TIMEOUT );
        var DevicePackageClass,devicePackage;
        DevicePackageClass = accedo.getNamespaceReference(devicePackageDef);
        if (!DevicePackageClass) {
            if (this._onFailure) {
                this._onFailure();
            } else {
                accedo.console.error("Device package reference not found.");
            }
            return;
        }
        
        devicePackage = new DevicePackageClass(accedo.Fn.bind(this._onDevicePackageInited,this));
        if  (!(devicePackage instanceof accedo.device.AbstractDevicePackage)) {
            if (this._onFailure) {
                this._onFailure();
            } else {
                accedo.console.error("Device package not a subclass of AbstractDevicePackage after loaded!");
            }
            return;
        }
    },
    /**
     * @name _onDevicePackageInited
     * @function
     * @memberof accedo.device.module:Manager
     * @private
     * @static
     */
    _onDevicePackageInited: function(devicePackageDef) {
        this._inited = true;
        this._devicePackage =  devicePackageDef;
        clearTimeout(this._packageTimeoutHandle);
        
        if (this._onSuccess) {
            this._onSuccess();
        }
    },
    /**
     * @name _onDevicePackageInitFailure
     * @function
     * @memberof accedo.device.module:Manager
     * @private
     * @static
     */
    _onDevicePackageInitFailure: function() {
        clearTimeout(this._packageTimeoutHandle);
        if (this._onFailure) {
            this._onFailure();
        } else {
            accedo.console.error("Initialize device package timeout!");
        }
    },
    /**
     * @name addInterfaceType
     * @param {String} type
     * @param {Object} handle
     * @param {Object} interfaceObj
     * @function
     * @memberof accedo.device.module:Manager
     * @public
     */
    addInterfaceType: function (type,handle,interfaceObj) {
        if (!(interfaceObj instanceof accedo.device.AbstractInterface)) {
            accedo.console.error("accedo.device.AbstractInterface expected.");
        }
            
        this._interfaceTypes[type] = {
            handle: handle,
            interfaceObj: interfaceObj
        };
            
        accedo.device[handle] = interfaceObj;
    },
    /**
     * @name getInterface
     * @param {String} type 
     * @returns {accedo.device.AbstractInterface} returns the interface instance
     * @function
     * @memberof accedo.device.module:Manager
     * @public
     */
    getInterface: function(type) {
        if (this._interfaceTypes[type]){
            return this._interfaceTypes[type].interfaceObj;
        }
        return null;
    }
});

/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @class
 * Manage Device specific key code and return virtual keys.
 * <br/><br/>
 * Also loads key mapping configuration, and returns the configured key when consulted.
 * <br/><br/>
 * The configuration under key 'vkey.create' will be uesd to created new virtual keys into accedo.VKey. This configuration accepts an object, or a function that take one input parameter, which is the accedo.VKey object.
 * <br/>
 * The configuration under key 'device.devicename.vkey.mapping' will be used to define/override the default key mappings.
 * @example
 * // key mapping configuration examples
 *
 * 'vkey.create':{
 *    "accedo.VKey.KEY_SPECIAL" : "special_key_id"
 *  },
 * 'device.workstation.vkey.mapping':{
 *    112 : "accedo.VKey.KEY_SPECIAL", // p key, overrides default key mapping
 *    115 : "accedo.VKey.KEY_STOP" // s key, overrides default key mapping
 *  },
 *
 *  <strong>or</strong>
 *
 * 'vkey.create':function(vkey){
 *   vkey.KEY_SPECIAL = "special_key_id";
 * },
 * @name AbstractTvKey
 * @memberof accedo.device
 * @author <a href="mailto:daniel.deng@accedo.tv">Daniel Deng</a>
 */
accedo.Class.create('accedo.device.AbstractTvKey', 'accedo.device.AbstractInterface', [], {
    /**
     * The overriding key mappings
     * @name overridingKeys
     * @memberof accedo.device.AbstractTvKey#
     * @private
     * @static
     */
    _overridingKeys : {},
    main: function(){
        var i, arr, keyName,
        keyCreation = accedo.config.get('vkey.create', {}),
        overridingMapping = accedo.config.get('device.' + accedo.platform + '.vkey.mapping', {});

        if (accedo.Object.isPlainObject(keyCreation)) {
            for(i in keyCreation) {
                arr = i.split(".");
                keyName = arr[arr.length-1];
                accedo.VKey[keyName] = keyCreation[i];
            }
        } else if (accedo.Object.isFunction(keyCreation)) {
            keyCreation(accedo.VKey);
        }

        for(i in overridingMapping) {
            accedo.device.AbstractTvKey._overridingKeys[parseInt(i, 10)] = accedo.getNamespaceReference(overridingMapping[i]);
        }
    },
    /**
    * Return configured virtual key according to the key code provided
    * @name getConfiguredKey
    * @param {Number} keyCode Key code
    * @return {accedo.VKey} VKey
    * @function
    * @memberof accedo.device.AbstractTvKey#
    * @public
    * @static
    */
    getConfiguredKey: function(keyCode){
        var keyStr = accedo.device.AbstractTvKey._overridingKeys[keyCode];
        if(keyStr){
            return keyStr;
        }
        return false;
    }
},{
    /**
     * init function to be called once interface is loaded
     * @name init
     * @function
     * @memberof accedo.device.AbstractTvKey#
     * @private
     */
    init: function(){},
    /**
     * Default window onkey handler.
     * @name _onkey
     * @memberof accedo.device.AbstractTvKey#
     * @function
     * @private
     */
    _onkey: function(keyEvent) {
        //to prevent when pressing any keys that accedo.device.tvkey is not loaded yet
        if(!accedo.device.tvkey){
            return;
        }

        //convert keys using tvKey device api
        var tvKey = this.getVirtualKey(keyEvent);

        accedo.console.log("Env.onKey: "+ tvKey );
        //Dispatch the key event
        accedo.Env.dispatchEvent(accedo.Env.EVT_ONKEY, tvKey, keyEvent);
    },
    /**
     * initializes the key handling logic for the device
     * @name initKeyHandling
     * @function
     * @memberof accedo.device.AbstractTvKey#
     * @public
     */
    initKeyHandling: function(){
        document.onkeydown = accedo.Fn.bind(this._onkey, this);
    },
    /**
     * Return virtual key according to the tv key provided
     * @name getVirtualKey
     * @param {Object} evt Key event object sent from device
     * @return {accedo.VKey} VKey
     * @function
     * @memberof accedo.device.AbstractTvKey#
     * @public
     */
    getVirtualKey: function(evt){
        var code = this.getCode(evt),
        key = accedo.device.AbstractTvKey.getConfiguredKey(code.code);

        if(!key){
            key = this.getMappedKey(code.code, code.isChar);
        }
        
        return key;
    },
    /**
     * Returns a key code object, indicating the keyCode/charCode extracted
     * @name getCode
     * @param {Object} evt Key event object sent from device
     * @return {Object} keyCode The key event code object, with code number: keyCode.code, and whether it is an charCode: keyCode.isChar
     * @function
     * @memberof accedo.device.AbstractTvKey#
     * @protected
     */
    getCode: function(evt){
        if (evt.type.toLowerCase() === "keypress") {
            return {
                code: evt.charCode,
                isChar: true
            };
        }
        return {
            code: evt.keyCode,
            isChar: false
        };
    },
    /**
     * Return virtual key according to the tv key provided
     * @name getMappedKey
     * @param {Number} code Key event code from device
     * @param {Boolean} isChar Whether this code is character code
     * @return {accedo.VKey} VKey
     * @function
     * @memberof accedo.device.AbstractTvKey#
     * @public
     */
    getMappedKey: function(code, isChar){
        return accedo.Class.abstractFn();
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @name AbstractDevicePackage
 * @memberof accedo.device
 * @class Manage loading device-specific interfaces. Load Devicepackage for current device platform. 
 * Packages files: video, tvkey, id, console, storage, system, appengine. Interfaces will be loaded and handled in Devicepackage 
 * 
 */
accedo.Class.create('accedo.device.AbstractDevicePackage', 
    [
    "accedo.device.AbstractVideo",
    "accedo.device.AbstractTvKey",
    "accedo.device.AbstractId",
    "accedo.device.AbstractConsole",
    "accedo.device.AbstractStorage",
    "accedo.device.AbstractSystem",
    "accedo.device.AbstractAppEngine"
    ],
    {
        /**
     * Name for Storage interface
     * @constant
     * @name STORAGE
     * @memberof accedo.device.AbstractDevicePackage
     */
        STORAGE : 'Storage',
        STORAGE_HANDLE: 'storage',
        /**
     * Name for Video interface
     * @constant
     * @name VIDEO_PLAYER
     * @memberof accedo.device.AbstractDevicePackage
     */
        VIDEO_PLAYER : 'Video',
        VIDEO_PLAYER_HANDLE : "video",
        /**
     * Name for TV KEY interface
     * @constant
     * @name TV_KEY
     * @memberof accedo.device.AbstractDevicePackage
     */
        TV_KEY: 'TvKey',
        TV_KEY_HANDLE : "tvkey",
        /**
     * Name for ID interface
     * @constant
     * @name ID
     * @memberof accedo.device.AbstractDevicePackage
     */
        ID: 'Id',
        ID_HANDLE: "id",
        /**
     * Name for Console interface
     * @constant
     * @name CONSOLE
     * @memberof accedo.device.AbstractDevicePackage
     */
        CONSOLE: 'Console',
        CONSOLE_HANDLE: "console",
        /**
     * Name for System interface
     * @constant
     * @name SYSTEM
     * @memberof accedo.device.AbstractDevicePackage
     */
        SYSTEM: 'System',
        SYSTEM_HANDLE: "system",
        /**
     * Name for AppEngine interface
     * @constant
     * @name APP_ENGINE
     * @memberof accedo.device.AbstractDevicePackage
     */
        APP_ENGINE: 'AppEngine',
        APP_ENGINE_HANDLE: "appengine"
    },
    {   
        initialize: function(onDeviceReady) {
            this.onDeviceReady = onDeviceReady;
            this.init();
        },
        init: accedo.Class.AbstractFn,
        deinit: accedo.Class.AbstractFn,
        ready: function() {
            var i,intf, 
            basicInterfaces = [
            // XDK_CHANGE: Removed Storage abstraction
            // {
            //     abstractInterface:  accedo.device.AbstractStorage,
            //     name: accedo.device.AbstractDevicePackage.STORAGE,
            //     handle: accedo.device.AbstractDevicePackage.STORAGE_HANDLE
            // },
            // XDK_CHANGE: Removed Video abstraction
            // {
            //     abstractInterface:  accedo.device.AbstractVideo,
            //     name: accedo.device.AbstractDevicePackage.VIDEO_PLAYER,
            //     handle: accedo.device.AbstractDevicePackage.VIDEO_PLAYER_HANDLE
            // },
            {
                abstractInterface:  accedo.device.AbstractTvKey,
                name: accedo.device.AbstractDevicePackage.TV_KEY,
                handle: accedo.device.AbstractDevicePackage.TV_KEY_HANDLE
            },
            // XDK CHANGE
            // {
            //     abstractInterface:  accedo.device.AbstractId,
            //     name: accedo.device.AbstractDevicePackage.ID,
            //     handle: accedo.device.AbstractDevicePackage.ID_HANDLE
            // },
            // {
            //     abstractInterface:  accedo.device.AbstractConsole,
            //     name: accedo.device.AbstractDevicePackage.CONSOLE,
            //     handle: accedo.device.AbstractDevicePackage.CONSOLE_HANDLE
            // },
            // XDK CHANGE
            // {
            //     abstractInterface:  accedo.device.AbstractSystem,
            //     name: accedo.device.AbstractDevicePackage.SYSTEM,
            //     handle: accedo.device.AbstractDevicePackage.SYSTEM_HANDLE
            // },
            // XDK CHANGE
            // {
            //     abstractInterface:  accedo.device.AbstractAppEngine,
            //     name: accedo.device.AbstractDevicePackage.APP_ENGINE,
            //     handle: accedo.device.AbstractDevicePackage.APP_ENGINE_HANDLE
            // }
            ];
        
            for (i=0;i<basicInterfaces.length;i++) {
                intf = accedo.device.Manager.getInterface(basicInterfaces[i].name);
                if (!intf) {
                    accedo.console.error("Basic interface not implemented in device package:name=" +basicInterfaces[i].name);
                }
                if (!(intf instanceof basicInterfaces[i].abstractInterface)) {
                    accedo.console.error("Basic interface " + basicInterfaces[i].name + " not subclassing abstract interface class:class="+basicInterfaces[i].abstractInterface.__definition + "&interface="+intf.getDefinition());
                }
                if (accedo.device[basicInterfaces[i].handle]!==intf) {
                    accedo.console.error("Basic interface " + basicInterfaces[i].name + " not assigned to the correct handle: handle="+basicInterfaces[i].handle + "&accedo.device."+basicInterfaces[i].handle+"="+ (accedo.device[basicInterfaces[i].handle] && accedo.device[basicInterfaces[i].handle].getDefinition?accedo.device[basicInterfaces[i].handle].getDefinition():accedo.device[basicInterfaces[i].handle]));
                }
            }
            
            this.onDeviceReady();
        }
    });
/*jslint browser: true, devel: true */
/*global accedo: false */
accedo.Class.create('accedo.device.panasonic.TvKey', "accedo.device.AbstractTvKey", {},{
    MAGIC_KEY_COMBO_CONSOLE: ['red', '0', 'red'],
    keys: {},
    shiftedKeys: {},
    lastKeysPressed: [],
    shiftOn: false,
    capslockOn: false,

    initialize: function(){
        var i;
        this.keys[13] = accedo.VKey.KEY_ENTER;

//RCU
        this.keys[38] = accedo.VKey.KEY_UP;
        this.keys[40] = accedo.VKey.KEY_DOWN;
        this.keys[37] = accedo.VKey.KEY_LEFT;
        this.keys[39] = accedo.VKey.KEY_RIGHT;

        this.keys[403] = accedo.VKey.KEY_RED;//F2
        this.keys[404] = accedo.VKey.KEY_GREEN;//F7
        this.keys[405] = accedo.VKey.KEY_YELLOW;//F8
        this.keys[406] = accedo.VKey.KEY_BLUE;//F9

        this.keys[459] = accedo.VKey.KEY_TEXT;
        this.keys[460] = accedo.VKey.KEY_SUBTITLE;

        this.keys[461] = accedo.VKey.KEY_BACK; //the back/Return

        this.keys[145] = accedo.VKey.KEY_EXIT; //The scroll lock key

        this.keys[33] = accedo.VKey.KEY_KEYBOARD_PAGE_UP;
        this.keys[34] = accedo.VKey.KEY_KEYBOARD_PAGE_DOWN;

        this.keys[457] = accedo.VKey.KEY_INFO;

        this.keys[19] = accedo.VKey.KEY_PAUSE; //the pause key
        this.keys[415] = accedo.VKey.KEY_PLAY;
        this.keys[416] = accedo.VKey.KEY_REC;  //"Red Record Button";
        this.keys[412] = accedo.VKey.KEY_RW; // "<<"
        this.keys[417] = accedo.VKey.KEY_FF; // ">>"
        this.keys[413] = accedo.VKey.KEY_STOP;

        // TODO: Some new key binding to be check and added here
        //this.keys[???] = accedo.VKey.KEY_NEXT; //">>|";    // currently same as KEY_FF
        //this.keys[???] = accedo.VKey.KEY_PREV; //"|<<";    // currently same as KEY_RW



        this.keys[9] = accedo.VKey.KEY_KEYBOARD_TAB; //the tab key

        this.keys[35] = accedo.VKey.KEY_KEYBOARD_END;
        this.keys[36] = accedo.VKey.KEY_KEYBOARD_HOME;
        this.keys[45] = accedo.VKey.KEY_KEYBOARD_INSERT;
        this.keys[46] = accedo.VKey.KEY_KEYBOARD_DELETE;

        this.keys[115] = 'mts'; // or prech or d_audio or popup (key F4)

        for(i=65; i<=90; i++){ // A-Z
            this.keys[i] = accedo.VKey["KEY_" + String.fromCharCode(i) + "_LOWER"];
        }

        for(i=48; i<=57; i++){ // 1-0
            this.keys[i] = accedo.VKey["KEY_" + String.fromCharCode(i)];
        }

        this.keys[32] = accedo.VKey.KEY_SPACE;

        this.keys[192] = accedo.VKey.KEY_BACK_QUOTE;
        this.keys[189] = accedo.VKey.KEY_MINUS;
        this.keys[187] = accedo.VKey.KEY_EQUALS;
        this.keys[219] = accedo.VKey.KEY_OPEN_BRACKET;
        this.keys[221] = accedo.VKey.KEY_CLOSE_BRACKET;
        this.keys[220] = accedo.VKey.KEY_BACK_SLASH;
        this.keys[186] = accedo.VKey.KEY_SEMICOLON;
        this.keys[222] = accedo.VKey.KEY_SINGLE_QUOTE;
        this.keys[188] = accedo.VKey.KEY_COMMA;
        this.keys[190] = accedo.VKey.KEY_PERIOD;
        this.keys[191] = accedo.VKey.KEY_SLASH;

        // numpad keys
        this.keys[111] = accedo.VKey.KEY_SLASH;
        this.keys[106] = accedo.VKey.KEY_STAR;
        this.keys[109] = accedo.VKey.KEY_MINUS;
        this.keys[107] = accedo.VKey.KEY_PLUS;
        for(i=96; i<=105; i++){ // 1-0
            this.keys[i] = accedo.VKey["KEY_" + (i-96)];
        }


        // ==== shifted keys ====
        for(i=65; i<=90; i++){ // A-Z
            this.shiftedKeys[i] = accedo.VKey["KEY_" + String.fromCharCode(i)];
        }

        this.shiftedKeys[48] = accedo.VKey.KEY_CLOSE_PAREN;
        this.shiftedKeys[49] = accedo.VKey.KEY_EXC;  // the left-side key 1
        this.shiftedKeys[50] = accedo.VKey.KEY_AT;
        this.shiftedKeys[51] = accedo.VKey.KEY_HASH;
        this.shiftedKeys[52] = accedo.VKey.KEY_DOLLAR;
        this.shiftedKeys[53] = accedo.VKey.KEY_PERCENT;
        this.shiftedKeys[54] = accedo.VKey.KEY_CARET;
        this.shiftedKeys[55] = accedo.VKey.KEY_AMP;
        this.shiftedKeys[56] = accedo.VKey.KEY_STAR;
        this.shiftedKeys[57] = accedo.VKey.KEY_OPEN_PAREN; // the left-side key 9

        this.shiftedKeys[192] = accedo.VKey.KEY_TILDE;
        this.shiftedKeys[189] = accedo.VKey.KEY_UNDERSCORE;
        this.shiftedKeys[187] = accedo.VKey.KEY_PLUS;
        this.shiftedKeys[219] = accedo.VKey.KEY_OPEN_BRACE;
        this.shiftedKeys[221] = accedo.VKey.KEY_CLOSE_BRACE;
        this.shiftedKeys[220] = accedo.VKey.KEY_PIPE;
        this.shiftedKeys[186] = accedo.VKey.KEY_COLON;
        this.shiftedKeys[222] = accedo.VKey.KEY_QUOTE;
        this.shiftedKeys[188] = accedo.VKey.KEY_LESS;
        this.shiftedKeys[190] = accedo.VKey.KEY_GREATER;
        this.shiftedKeys[191] = accedo.VKey.KEY_QUESTION;
    },

    initKeyHandling: function(){
        document.onkeydown = accedo.Fn.bind(this.__onKeyDown,this); // overrides the default handler
        document.onkeyup = accedo.Fn.bind(this.__onKeyUp,this);
        document.onkeypress = accedo.Fn.bind(this.__onKeyPress,this);
    },

    __onKeyDown: function(evt){
        console.log('keydown event keyCode=' + evt.keyCode);
        
        if(evt.keyCode === 20){ // capslock key
            this.capslockOn = !this.capslockOn;
            return true;
        } else if(evt.keyCode === 16) {
            this.shiftOn = true;
        }

        //convert keys using tvKey device api
        var tvKey = this.getVirtualKey(evt);
        if(tvKey){
            if (this.__checkForMagicKeyCombo(tvKey)) {
                return true;
            } else {
                //Dispatch the key event
                accedo.Env.dispatchEvent(accedo.Env.EVT_ONKEY, tvKey, evt);
            }
        }

        // special handling for backspace key
        if(evt.keyCode === 8){
            evt.preventDefault();
            return false;
        }

        return true;
    },

    __onKeyUp: function(evt){
        console.log('keyup event keyCode=' + evt.keyCode);
        if (evt.keyCode === 16) { // shift
            this.shiftOn = false;
        }
    },

    __onKeyPress: function(evt){
        console.log('keypress event keyCode=' + evt.keyCode);
    },

    __checkForMagicKeyCombo: function(tvKey) {
        // disabled
        return false;

        this.lastKeysPressed.push(tvKey);
        if (this.lastKeysPressed.length > this.MAGIC_KEY_COMBO_CONSOLE.length) {
            this.lastKeysPressed.shift();
        }
        if (this.lastKeysPressed.length < this.MAGIC_KEY_COMBO_CONSOLE.length) {
            return false;
        }

        var doToggle = false;
        var consumeKey = false;

        var magicComboComplete = true;
        for (var i = 0; i < this.MAGIC_KEY_COMBO_CONSOLE.length; i++) {
            if (this.lastKeysPressed[i] != this.MAGIC_KEY_COMBO_CONSOLE[i]) {
                magicComboComplete = false;
                break;
            }
        }

        if (this.magicMode) {
            if (tvKey == 'red') {
                doToggle = true;
                consumeKey = true;
            }
            if (freesat.isConsoleShowing() && tvKey == 'blue') {
                if (window.localStorage.getItem('FORCE_DSMCC') === 'true') {
                    freesat.console.log('<b>Turning off FORCE_DSMCC</b>');
                    window.localStorage.setItem('FORCE_DSMCC', 'false');
                } else {
                    freesat.console.log('<b>Turning on FORCE_DSMCC</b>');
                    window.localStorage.setItem('FORCE_DSMCC', 'true');
                }
                consumeKey = true;
            } else if (freesat.isConsoleShowing() && tvKey == 'yellow') {
                freesat.console.log('<b>Release details</b>');
                freesat.console.log(JSON.stringify(FREETIME_VERSION, null, 4));
                freesat.console.log('<b>Auth file is:</b>');
                freesat.console.log(this.__escapeHtml(freesat.localConfig.auth.authAsString + '\n\n').replace('\n', '<br>'));
                consumeKey = true;
            }  else if (freesat.isConsoleShowing() && tvKey == 'green') {
                freesat.marketingOptIn.cancelPreference();
                freesat.console.log('<b>Cancelled marketing opt-in preference</b>');
                consumeKey = true;
            }

        } else if (magicComboComplete) {
            this.magicMode = true;
            doToggle = true;
        }

        if (doToggle) {
            if (!freesat.isConsoleShowing()) {
                freesat.showConsole();
            } else {
                freesat.hideConsole();
            }
        }

        return consumeKey;
    },

    __escapeHtml: function(unsafe) {
        return unsafe
             .replace(/&/g, "&amp;")
             .replace(/</g, "&lt;")
             .replace(/>/g, "&gt;")
             .replace(/"/g, "&quot;")
             .replace(/'/g, "&#039;");
    },

    getMappedKey: function(keyCode, isChar){
        accedo.console.debug("TV key code: " + keyCode);

        if (this.shiftOn && this.shiftedKeys[keyCode]) {
            return this.shiftedKeys[keyCode];
        } else if(this.capslockOn && keyCode >= 65 && keyCode <= 90){
            return this.shiftedKeys[keyCode];
        }

        if(this.keys[keyCode]){
            return this.keys[keyCode];
        } else {
            return 'no such key:' + keyCode; // at least let the event fire with some unknown key
        }
    }
});

/*jslint browser: true, devel: true */
/*global accedo: false */
accedo.Class.create('accedo.device.panasonic.DevicePackage',
    'accedo.device.AbstractDevicePackage',
    [
    "accedo.device.panasonic.Storage",
    'accedo.device.panasonic.Video',
    'accedo.device.panasonic.TvKey',
    'accedo.device.panasonic.Id',
    'accedo.device.panasonic.Console',
    'accedo.device.panasonic.AppEngine',
    'accedo.device.panasonic.System'
    ],
    {}, 
    {
        init: function() {
            if (accedo.platform === "unknown") {
                accedo.platform = "panasonic";
            }

            var interfaceTypes = new accedo.Hash(this.getInterfaceTypes());
        
            interfaceTypes.each(function(pair){
                accedo.device.Manager.addInterfaceType(pair.key,pair.value.handle,pair.value.interfaceObj);
            });
            
            this.ready();
        },
        getInterfaceTypes: function () {
            var ret = {};
            
            // XDK_CHANGE: Removed Storage abstraction from XDK (it's not needed)
            // ret[accedo.device.AbstractDevicePackage.STORAGE] = {
            //     handle: accedo.device.AbstractDevicePackage.STORAGE_HANDLE,
            //     interfaceObj: new accedo.device.panasonic.Storage()
            // };

            // XDK_CHANGE: Removed Video abstraction from XDK (it's not needed)
            // ret[accedo.device.AbstractDevicePackage.VIDEO_PLAYER] = {
            //     handle: accedo.device.AbstractDevicePackage.VIDEO_PLAYER_HANDLE,
            //     interfaceObj: new accedo.device.panasonic.Video()
            // };

            ret[accedo.device.AbstractDevicePackage.TV_KEY] = {
                handle: accedo.device.AbstractDevicePackage.TV_KEY_HANDLE,
                interfaceObj: new accedo.device.panasonic.TvKey()
            };
            // XDK CHANGE
            // ret[accedo.device.AbstractDevicePackage.ID] = {
            //     handle: accedo.device.AbstractDevicePackage.ID_HANDLE,
            //     interfaceObj: new accedo.device.panasonic.Id()
            // };

            // ret[accedo.device.AbstractDevicePackage.CONSOLE] = {
            //     handle: accedo.device.AbstractDevicePackage.CONSOLE_HANDLE,
            //     interfaceObj: new accedo.device.panasonic.Console()
            // };
            // XDK CHANGE
            // ret[accedo.device.AbstractDevicePackage.APP_ENGINE] = {
            //     handle: accedo.device.AbstractDevicePackage.APP_ENGINE_HANDLE,
            //     interfaceObj: new accedo.device.panasonic.AppEngine()
            // };
            // XDK CHANGE
            // ret[accedo.device.AbstractDevicePackage.SYSTEM] = {
            //     handle: accedo.device.AbstractDevicePackage.SYSTEM_HANDLE,
            //     interfaceObj: new accedo.device.panasonic.System()
            // };

            return ret;
        }
    });


/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @desc This class do not need to be instantiated. Please just use its static members.
 * @name Evt
 * @memberof accedo.ui
 * @class Centralized UI events
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 */
accedo.Class.create("accedo.ui.Evt", {
    
    /**
     * Event: show
     * @constant
     * @name SHOW
     * @memberof accedo.ui.Evt
     * @deprecated please use accedo.ui.Evt.SHOWN
     */
    SHOW : "xdk:ui:shown",
    /**
     * Event: shown
     * @constant
     * @name SHOWN
     * @memberof accedo.ui.Evt
     */
    SHOWN : "xdk:ui:shown",
    /**
     * Event: hide
     * @constant
     * @name HIDE
     * @memberof accedo.ui.Evt
     * @deprecated please use accedo.ui.Evt.HIDDEN
     */
    HIDE : "xdk:ui:hidden",
    /**
     * Event: hidden
     * @constant
     * @name HIDDEN
     * @memberof accedo.ui.Evt
     */
    HIDDEN : "xdk:ui:hidden",
    /**
     * Event: detach
     * @constant
     * @name DETACH
     * @memberof accedo.ui.Evt
     */
    DETACH : "xdk:ui:detach",
    /**
     * Event: detach
     * @constant
     * @name DETACH_TO_DOM
     * @memberof accedo.ui.Evt
     */
    DETACHED_FROM_DOM : "xdk:ui:detachDOM",
    /**
     * Event: attach
     * @constant
     * @name ATTACH
     * @memberof accedo.ui.Evt
     */
    ATTACH : "xdk:ui:attach",
    /**
     * Event: attactDom
     * @constant
     * @name ATTACHED_TO_DOM
     * @memberof accedo.ui.Evt
     */
    ATTACHED_TO_DOM :"xdk:ui:attachDOM",
    
    /**
     * Event: attachController
     * @constant
     * @name ATTACHED_TO_CONTROLLER
     * @memberof accedo.ui.Evt
     */
    ATTACHED_TO_CONTROLLER :"xdk:ui:attachController",
    
    /**
     * Event: detachController
     * @constant
     * @name DETACHED_FROM_CONTROLLER
     * @memberof accedo.ui.Evt
     */
    DETACHED_FROM_CONTROLLER :"xdk:ui:detachController",
	
    /**
     * Event: focus
     * @constant
     * @name FOCUS
     * @memberof accedo.ui.Evt
     */
    FOCUS : "xdk:ui:focus",
    
    /**
     * Event: blur
     * @constant
     * @name BLUR
     * @memberof accedo.ui.Evt
     */
    BLUR : "xdk:ui:blur",
    
    /**
     * Event: key (key:accedo.VKey, currentFocus:accedo.ui.Evt)
     * 
     * @constant
     * @name KEY
     * @memberof accedo.ui.Evt
     */
    KEY : "xdk:ui:key",
    
    /**
     * Event: open
     * @constant
     * @name OPEN
     * @memberof accedo.ui.Evt
     */
    OPEN: "xdk:ui:open",
    
    /**
     * Event: close
     * @constant
     * @name CLOSE
     * @memberof accedo.ui.Evt
     */
    CLOSE: "xdk:ui:close",
    
    /**
     * Event: click
     * @constant
     * @name CLICK
     * @memberof accedo.ui.Evt
     * @deprecated  moved to accedo.ui.Evt.CLICK
     */
    CLICK: "xdk:ui:click",
    
    /**
     * Event: cancel
     * @constant
     * @name CANCEL
     * @memberof accedo.ui.Evt
     * @deprecated  moved to accedo.ui.Evt.CANCEL
     */
    CANCEL: "xdk:ui:cancel",
    
    /**
     * Event: confirm
     * @constant
     * @name CONFIRM
     * @memberof accedo.ui.Evt
     */
    CONFIRM: "xdk:ui:confirm",
    
    /**
     * Event: child focus
     * @constant
     * @name CHILD_FOCUS
     * @memberof accedo.ui.Evt
     */
    CHILD_FOCUS: "xdk:ui:child-focus",
    /**
     * Event: child blur
     * @constant
     * @name CHILD_BLUR
     * @memberof accedo.ui.Evt
     */
    CHILD_BLUR: "xdk:ui:child-blur",
    
    /**
     * Event: scroll
     * @constant
     * @name SCROLL
     * @memberof accedo.ui.Evt
     */
    SCROLL: "xdk:ui:scroll",
    
    /**
     * Event: selection changed
     * @constant
     * @name SELECTION_CHANGED
     * @memberof accedo.ui.Evt
     */
    SELECTION_CHANGED: 'xdk:ui:selection-changed',
    
    /**
     * Event: text changed
     * @constant
     * @name TEXT_CHANGED
     * @memberof accedo.ui.Evt
     */
    TEXT_CHANGED: 'xdk:ui:text-changed',

    /**
     * Event: value changed
     * @constant
     * @name VALUE_CHANGED
     * @memberof accedo.ui.Evt
     */
    VALUE_CHANGED: 'xdk:ui:value-changed',
    
    /**
     * Event: moved
     * @constant
     * @name MOVED
     * @memberof accedo.ui.Evt
     */
    MOVED : 'xdk:ui:moved',
    /**
     * Event: when scrollable label scroll to the top ,previously was SCROLL_TO_TOP
     * @constant
     * @name SCROLLED_TO_TOP
     * @memberof accedo.ui.Evt
     */
    SCROLLED_TO_TOP:'xdk:ui:scroll-to-top',
    /**
     * Event: when scrollable label scroll to the bottom,previously was SCROLL_TO_BOTTOM
     * @constant
     * @name SCROLLED_TO_BOTTOM
     * @memberof accedo.ui.Evt
     */
    SCROLLED_TO_BOTTOM:'xdk:ui:scroll-to-bottom',
    /**
     * Event: when a component becomes active
     * @constant
     * @name ACTIVE
     * @memberof accedo.ui.Evt
     */
    ACTIVED: 'xdk:ui:actived',
    /**
     * Event: when a component becomes inactive
     * @constant
     * @name INACTIVED
     * @memberof accedo.ui.Evt
     */
    INACTIVED: 'xdk:ui:inactived'
    
});

/*jslint browser: true, devel: true*/
/*global accedo: false */
/**
 * @desc <pre>
 * This is a class that implements event delegation.
 * Components that are listening to mouse events should register to this class.
 * Now only handles "mouseover" and "click" event delegation.
 * </pre>
 * @name EvtDelegate
 * @memberof accedo.ui
 * @module
 * @author <a href="mailto:daniel.deng@accedo.tv">Daniel Deng</a>
 */
accedo.Class.create("accedo.ui.EvtDelegate", [],
{
    /**
     * Event register for storing mapings of events, targets and targets.
     * @name _eventRegister
     * @private
     * @memberof accedo.ui.module:EvtDelegate
     * @static
     */
    _eventRegister: {},
    /**
     * The last position of mouse.
     * @name _lastMousePos
     * @private
     * @memberof accedo.ui.module:EvtDelegate
     * @static
     */
    // XDK CHANGE
    // _lastMousePos:{x:9999, y:9999},
    /**
     * Registers an event for a component.
     * @name registerEvent
     * @private
     * @memberof accedo.ui.module:EvtDelegate
     * @param {String} eventType The event type to register
     * @param {accedo.ui.AbstractComponent} The component to register with
     * @static
     */
    _registerEvent: function(eventType, component){
        var mainRegister = this._eventRegister, register, id = component.getId();
        if (!mainRegister[eventType]) {
            mainRegister[eventType] = {};
        }

        register = mainRegister[eventType];
        if (!register[id]) {
            register[id] = {};
        }

        register[id] = component;

        component.getRoot().getHTMLElement()._CompId = id;
    },
    /**
     * Unregisters an event for a component.
     * @name unregisterEvent
     * @private
     * @memberof accedo.ui.module:EvtDelegate
     * @param {String} eventType The event type to unregister
     * @param {accedo.ui.AbstractComponent} component The component to unregister.
     *         Deletes all components the an event if not provided
     * @static
     */
    _unregisterEvent: function(eventType, component){
        var key, register = this._eventRegister[eventType], id;

        if (!register) { // no such event type to delete
            return;
        }

        if (!component && register) {
            for (key in register) {
                delete register[key]._CompId;
            }
            delete this._eventRegister[eventType];
            return;
        }

        id = component.getId();
        if (!id || !register[id]) { // no such component to delete
            return;
        }

        //delete register[id].getRoot().getHTMLElement()._CompId;
        delete register[id];
    },
    /**
     * Local cache of the body element.
     * @name _bodyEle
     * @private
     * @memberof accedo.ui.module:EvtDelegate
     * @static
     */
    _bodyEle: null,
    /**
     * Local cache of the app system controller.
     * @name _appSystem
     * @private
     * @memberof accedo.ui.module:EvtDelegate
     * @static
     */
    _appSystem: null,
    
    /**
     * Registers a compnent for event delegation.
     * @name register
     * @public
     * @memberof accedo.ui.module:EvtDelegate
     * @param {accedo.ui.AbstractComponent} component The component to register.
     * @static
     */
    register: function(component){
        if (!component.getId()){
            component.setId(accedo.getGuid() + "__xdk__");
        }
        // XDK CHANGE
        // if (component.getOption('focusable',false)) {
        //     this._registerEvent("mouseover", component);
        // }
        // if (component.getOption('clickable',false)) {
        //     this._registerEvent("click", component);
        // }
    },
    /**
     * Unregisters a compnent from event delegation.
     * @name unregister
     * @public
     * @memberof accedo.ui.module:EvtDelegate
     * @param {accedo.ui.AbstractComponent} component The component to unregister.
     * @static
     */
    unregister: function(component){
        if (!component.getId()) { // nothing to unregister
            return;
        }
        if (accedo.String.startsWith(component.getId(), "__xdk__")) {
            component.setId("");
        }
        // XDK CHANGE
        // if (component.getOption('focusable',false)) {
        //     this._unregisterEvent("mouseover", component);
        // }
        // if (component.getOption('clickable',false)) {
        //     this._unregisterEvent("click", component);
        // }
    },
    /**
     * Gets a component from the register.
     * @name _getComponent
     * @private
     * @memberof accedo.ui.module:EvtDelegate
     * @param {String} eventType The event type.
     * @param {String} id The component ID.
     * @static
     */
    _getComponent: function(eventType, id){
        var register = this._eventRegister[eventType];
        if (register && register[id]) {
            return register[id];
        }
        return null;
    },
    /**
     * Initialize the event delegation mechanism.
     * @name main
     * @protected
     * @memberof accedo.ui.module:EvtDelegate
     * @static
     */
    main: function(){
        var __listener = accedo.Fn.bind(function(evt){
            //IE doesn't pass in the event object
            evt = evt || window.event;
            var target = evt.target || evt.srcElement,
            id = target._CompId,
            type = evt.type,
            component = null,
            fm = accedo.ui.FocusManager.singleton(),
            returnVal = false;
            
            // XDK CHANGE
            return true;
            // if(accedo.Env.isBlockingMouse()){
            //     return true;
            // }

            // if (type === "mouseover" && accedo.Object.isNumber(evt.clientX)) {
            //     if (evt.clientX === this._lastMousePos.x && evt.clientY === this._lastMousePos.y) {
            //         return true;
            //     }
            //     this._lastMousePos.x = evt.clientX;
            //     this._lastMousePos.y = evt.clientY;
            // }

            // // no more bubbling
            // if (evt.stopPropagation) {
            //     evt.stopPropagation();
            // } else {
            //     evt.cancelBubble = true;
            // }

            // while (!component) {
            //     if (id) { // get component if ID exist
            //         component = this._getComponent(type, id);
            //         if (component) { // did we make it?
            //             break;
            //         }
            //     }

            //     // search up along the DOM tree
            //     target = target.parentNode;
            //     if (!target || target === document.body) {
            //         break;
            //     }
            //     id = target._CompId;
            // }

            // // invalid event
            // if (!component) {
            //     if (type === "mouseover") { // clear focus before quit
            //         fm.setMouseOnOff(true);
            //         fm.requestFocus(null);
            //     }
            //     return returnVal;
            // }
            
            // if(component.eventHandling){
            //     component.eventHandling(type);
            // }
            // return returnVal;
        }, this),
        // helper function to register event listeners into HTML DOM tree
        __addListenerFunc = accedo.Fn.bind(function() {
            // XDK CHANGE
            // if (!accedo.Env.hasMouse()) {
            //     return;
            // }
            // this._bodyEle = accedo.Element.create(document.body);
            // this._bodyEle.addEventListener("mouseover", __listener);
            // this._bodyEle.addEventListener("click", __listener);
        }, this);

        accedo.Env.addEventListener(accedo.Env.EVT_ONLOAD, __addListenerFunc);
    }
},
{});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Abstract component used as a base for all UI components.
 * @name AbstractComponent
 * @memberof accedo.ui
 * @class This is the abstract component function that all UI object, be it
 *        containers or components, inherrits from.
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @extends accedo.mixin.EventDispatcher
 * @param {Object} opts Options
 * @param {Document Element} opts.root The DOM root element (the container for this component)
 * @param {Document Element} opts.parent The DOM parent element
 * @param {Integer} opts.widthStyle How to layout the width of this component
 * @param {Integer} opts.heightStyle How to layout the height of this component
 * @param {Boolean} opts.clickable Whether this component is clickable
 * @param {Boolean} opts.focusable Whether this component is focusable
 * @param {Boolean} opts.mouseFocusableOnly To set this component is only focusable by mouse
 * @param {String || accedo.ui.AbstractComponent} opts.forwardOnMouseOff the focus will be redirect to.If there is no setting, it will redirect to the last non mouse focus when it is mouseFocusableOnly.
 * @param {Boolean} opts.enterAsClick Whether to treat enter keys as clicks for this component
 * @param {String} opts.id The ID of this component
 */
accedo.Class.create("accedo.ui.AbstractComponent", ["accedo.ui.Evt", "accedo.ui.EvtDelegate"], function()
{
    return {
    
        /**
         * This indicates that the component should grow to fit its parent.
         * @constant
         * @name FIT_PARENT
         * @memberof accedo.ui.AbstractComponent
         */
        FIT_PARENT : 0x01,

        /**
         * This indicates that the component should grow to fit its content.
         * @constant
         * @name WRAP_CONTENT
         * @memberof accedo.ui.AbstractComponent
         */
        WRAP_CONTENT : 0x02,

        /**
         * Alignment: left.
         * @constant
         * @name LEFT
         * @memberof accedo.ui.AbstractComponent
         */
        LEFT : 0x01,

        /**
         * Alignment: center.
         * @constant
         * @name CENTER
         * @memberof accedo.ui.AbstractComponent
         */
        CENTER : 0x02,

        /**
         * Alignment: right.
         * @constant
         * @name RIGHT
         * @memberof accedo.ui.AbstractComponent
         */
        RIGHT : 0x04,

        /**
         * Alignment: top.
         * @constant
         * @name TOP
         * @memberof accedo.ui.AbstractComponent
         */
        TOP : 0x08,

        /**
         * Alignment: middle.
         * @constant
         * @name MIDDLE
         * @memberof accedo.ui.AbstractComponent
         */
        MIDDLE : 0x10,

        /**
         * Alignment: bottom.
         * @constant
         * @name BOTTOM
         * @memberof accedo.ui.AbstractComponent
         */
        BOTTOM : 0x20,
    
        /**
         * Event: show
         * @constant
         * @name EVT_SHOW
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.SHOW
         */
        EVT_SHOW : accedo.ui.Evt.SHOW,
        /**
         * Event: hide
         * @constant
         * @name EVT_HIDE
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.HIDE
         */
        EVT_HIDE : accedo.ui.Evt.HIDE,
        /**
         * Event: detach
         * @constant
         * @name EVT_DETACH
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.DETACH
         */
        EVT_DETACH : accedo.ui.Evt.DETACH,
        /**
         * Event: detach
         * @constant
         * @name EVT_DETACHED_FROM_DOM
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.DETACHED_FROM_DOM
         */
        EVT_DETACHED_FROM_DOM : accedo.ui.Evt.DETACHED_FROM_DOM,
        /**
         * Event: attach
         * @constant
         * @name EVT_ATTACH
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.ATTACH
         */
        EVT_ATTACH : accedo.ui.Evt.ATTACH,
        /**
         * Event: attactDom
         * @constant
         * @name EVT_ATTACHED_TO_DOM
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.ATTACHED_TO_DOM
         */
        EVT_ATTACHED_TO_DOM :accedo.ui.Evt.ATTACHED_TO_DOM,
	
        /**
         * Event: focus
         * @constant
         * @name EVT_FOCUS
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.FOCUS
         */
        EVT_FOCUS : accedo.ui.Evt.FOCUS,
    
        /**
         * Event: blur
         * @constant
         * @name EVT_BLUR
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.BLUR
         */
        EVT_BLUR : accedo.ui.Evt.BLUR,
    
        /**
         * Event: key (key:accedo.VKey, currentFocus:accedo.ui.AbstractComponent)
         *
         * @constant
         * @name EVT_KEY
         * @memberof accedo.ui.AbstractComponent
         * @deprecated moved to accedo.ui.Evt.KEY
         */
        EVT_KEY : accedo.ui.Evt.KEY,    
        /**
         * Placement: before
         * @constant
         * @name PLACE_BEFORE
         * @memberof accedo.ui.AbstractComponent
         *
         */
        PLACE_BEFORE :  "before",
        /**
         * Placement: after
         * @constant
         * @name PLACE_AFTER
         * @memberof accedo.ui.AbstractComponent
         */
        PLACE_AFTER :   "after",
        /**
         * Placement: append
         * @constant
         * @name PLACE_APPEND
         * @memberof accedo.ui.AbstractComponent
         */
        PLACE_APPEND:   "append",
        /**
         * Placement: prepend
         * @constant
         * @name PLACE_PREPEND
         * @memberof accedo.ui.AbstractComponent
         */
        PLACE_PREPEND:  "prepend",
        PLACE_CHILD:    "placeChild"
    };
},
"accedo.mixin.EventDispatcher",
{
    /**
     * Currently focused.
     * @private
     * @field
     * @ignore
     */
    _focused: false,
    /**
     * Currrently disabled
     * @private
     * @field
     * @ignore
     */
    _disabled: false,
    /**
     * ancestor disable
     * @private
     * @field
     * @ignore
     */
    _ancestorDisabled:false,
    /**
     * Currrently hidden
     * @private
     * @field
     * @ignore
     */
    _hidden: false,
    /**
     * ancestor parent hidden
     * @private
     * @field
     * @ignore
     */
    _ancestorHidden:false,
    /**
     * Parent component.
     * @protected
     * @public
     * @field
     * @ignore
     */
    parent: null,
    /**
     * Root element.
     * @protected
     * @public
     * @field
     * @ignore
     */
    root: null,
    /**
     * Options object.
     * @protected
     * @public
     * @field
     * @ignore
     */
    opts: null,
    /**
     * Whether this component is attached to controller.
     * @private
     * @field
     * @ignore
     */
    _attachedToController: false,
    
    /**
     * The timestamp of this component last focused
     * @private
     * @field
     * @ignore
     */
    _lastFocused: null,
    /**
     * This function initiates a component using the options given in the constructor.
     * @name initialize
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    initialize: function(opts) {

        this.root = opts.root;
        this.parent = opts.parent || null;
        
        this.opts = accedo.Object.clone(opts,true);

        if (this.root && this.parent) {
            this.parent.attach(this);
        }

        if (opts.id) {
            this.setId(opts.id);
        }

        if(opts.focusable){
            delete this.opts.focusable; // should be set by the line below
            this.setOption("focusable", true);
        }

        if(opts.clickable){
            delete this.opts.clickable; // should be set by the line below
            this.setOption("clickable", true);
        }

        if (opts.css) {
            this.root.addClass(opts.css);
        }
        // XDK CHANGE
        // if(opts.mouseFocusableOnly){
        //     this.setOption("mouseFocusableOnly",true);
        // }
        
        // if(opts.forwardOnMouseOff){
        //     this.setOption("forwardOnMouseOff",opts.forwardOnMouseOff);
        // }
        
        this.setWidthStyle(opts.widthStyle);
        this.setHeightStyle(opts.heightStyle);
        
        // XDK_CHANGE: No need for mouse
        // if (this.root && accedo.Env.hasMouse()) {
        //     this.initMouseHandling();
        // }
        
    },
    /**
     * Initializes any mouse handling logic.
     * @name initMouseHandling
     * @function
     * @protected
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
     // XDK CHANGE
    // initMouseHandling: function() {
    //     return;
    // },
    /**
     * Returns true if component is focusable, false otherwise.
     * @name isFocusable
     * @function
     * @public
     * @return {Boolean}
     * @memberof accedo.ui.AbstractComponent#
     */
    isFocusable: function() {
        
        //not focusable if it's hidden
        if (this._hidden || this._ancestorHidden) {
            return false;
        }
        
        //not focusable if it's disabled
        if (this._disabled || this._ancestorDisabled) {
            return false;
        }
        
        //return default focusability
        return this.getOption('focusable',false);
    },
    /**
     * Returns true if component is clickable, false otherwise.
     * @name isClickable
     * @function
     * @public
     * @return {Boolean}
     * @memberof accedo.ui.AbstractComponent#
     */
    isClickable: function() {

        //not clickable if it's hidden
        if (this._hidden || this._ancestorHidden) {
            return false;
        }

        //not clickabke if it's disabled
        if (this._disabled || this._ancestorDisabled) {
            return false;
        }

        //return default clickability
        return this.getOption('clickable',false);
    },

    /**
     * Checks to see if component is focused.
     * @name isFocused
     * @function
     * @public
     * @return {Boolean} true if focused, false otherwise.
     * @memberof accedo.ui.AbstractComponent#
     */
    isFocused: function() {
        return this._focused;
    },

    /**
     * Checks to see if component is enabled.
     * @name isEnabled
     * @function
     * @public
     * @return {Boolean} true if enabled, false otherwise.
     * @memberof accedo.ui.AbstractComponent#
     */
    isEnabled: function() {
        return !this._disabled;
    },

    /**
     * Sets enabled state for component.
     * If disabled, sets CSS class "disabled"
     * @name setEnabled
     * @function
     * @public
     * @param {Boolean} toggle true or false for enabled state
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberof accedo.ui.AbstractComponent#
     */
    setEnabled: function(toggle) {
        this._disabled = !toggle;
        if (toggle) {
            this.root.removeClass("disabled");
            
        } else {
            this.root.addClass("disabled");
        }
        
        return this;
    },
    /**
     * Set the ancestor enable state
     * @name onAncestor
     * @function
     * @protected
     * @param {Boolean} toggle true or false for enabled state
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.AbstractComponent#
     */
    onAncestorSetEnabled:function(toggle){
        this._ancestorDisabled = !toggle;
        return this;
    },
    /**
     * Set the ancestor hidden state
     * @name onAncestor
     * @function
     * @protected
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.AbstractComponent#
     */
    onAncestorShow:function(){
        this._ancestorHidden = false;
        return this;
    },
    /**
     * Set the ancestor hidden state
     * @name onAncestorHide
     * @function
     * @protected
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.AbstractComponent#
     */
    onAncestorHide:function(){
        this._ancestorHidden = true;
        return this;
    },
    /**
     * Sets the className of the widget or component
     * @name setClass
     * @function
     * @public
     * @param {String} className the name of the class
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberof accedo.ui.AbstractComponent#
     */
    setClass : function(className){
        this.root.setClass(className);
        return this;
    },
    /**
     * Adds the className to the widget or component
     * @name addClass
     * @function
     * @public
     * @param {String} className the name of the class
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberof accedo.ui.AbstractComponent#
     */
    addClass : function(className){
        this.root.addClass(className);
        return this;
    },
    /**
     * Remove the className of the widget or component
     * @name removeClass
     * @function
     * @public
     * @param {String} className the name of the class
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberof accedo.ui.AbstractComponent#
     */
    removeClass : function(className){
        this.root.removeClass(className);
        return this;
    },
    /**
     * Focuses component if it is focusable and calls custom onFocus method if such is defined.
     * Focus is identified by "focused" CSS Class name.
     * @name focus
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    focus: function() {
            
        if (this.isFocusable() && !this._focused) {
            this.root.addClass("focused");
            
            this._focused = true;
            this._lastFocused = (new Date()).getTime();
                        
            return true;
        }
        
        return false;
    },
    
    /**
     * Removes focus from (blurs) component and calls custom onBlur method if such is defined.
     * @name blur
     * @function
     * @return {boolean} true if success, false if not focused in the first place
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    blur: function() {
        if (this._focused) {
            this.root.removeClass("focused");
            this._focused = false;
            return true;
        }
        return false;
    },

    /**
     * This function returns itself if the supplied id matches its own id,
     * otherwise it null is return
     * @name get
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @returns {accedo.ui.AbstractComponent|null}
     */
    get: function(id) {
        return (this.getId() === id) ? this : null;
    },
    
    /**
     * Check if the specified container is the ancestor of current component.
     * @name isAncestor
     * @function
     * @public
     * @param {accedo.ui.Container#} container The container to check.
     * @memberof accedo.ui.AbstractComponent#
     * @returns {boolean}
     */
    isAncestor: function(container) {
        if (!this.parent) {
            return false;
        }
        if (this.parent === container) {
            return true;
        }
        
        return this.parent.isAncestor(container);
    
    },
    
    /**
     * Set this component's id
     * @name setId
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    setId: function(id) {
        //When opts.id starts with '#', we also put the ID into DOM element
        if (accedo.String.startsWith(id, '#')) {
            if (id.length > 1) {
                this.root.id = id.substr(1);
                this.root.setAttributes({
                    id:this.root.id
                });
            }
        } else {
            this.root.id = id;
        }
    },
            
    /**
     * Returns this component's id
     * @name getId
     * @function
     * @public
     * @returns {string|null}
     * @memberof accedo.ui.AbstractComponent#
     */
    getId: function() {
        return this.root.id || null;
    },
    
    /**
     * This function sets the width style of this component
     * @name setWidthStyle
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @param {Integer} style The new width style
     * @see accedo.ui.AbstractComponent.FIT_PARENT
     * @see accedo.ui.AbstractComponent.WRAP_CONTENT
     * @deprecated Please avoid this method and set this in CSS sheets instead, as this method violates logic/view layer separation.
     */
    setWidthStyle: function(style) {
        if (style === accedo.ui.AbstractComponent.FIT_PARENT) {
            this.root.setStyle({
                'width': '100%'
            });
        } else {
            this.root.removeStyle('width');
        }
    },

    /**
     * This function sets the height style of this component
     * @name setHeightStyle
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @param {Integer} style The new height style
     * @see accedo.ui.AbstractComponent.FIT_PARENT
     * @see accedo.ui.AbstractComponent.WRAP_CONTENT
     * @deprecated Please avoid this method and set this in CSS sheets instead, as this method violates logic/view layer separation.
     */
    setHeightStyle: function(style) {
        if (style === accedo.ui.AbstractComponent.FIT_PARENT) {
            this.root.setStyle({
                'height': '100%'
            });
        } else {
            this.root.removeStyle('height');
        }
    },
    
    /**
     * This function returns the specified option and falls back to the supplied
     * default value it this option has not been defined for this component.
     * @name getOption
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @param {String} name The option name
     * @param {String} [defaultValue] The fallback value
     */
    getOption: function(name, defaultValue) {
        var ret = this.opts[name];
        if (accedo.Object.isUndefined(ret)) {
            ret = defaultValue;
        }
        return ret;
    },

    /**
     * Updates the event delegation registery according to current status of the component.
     * @name _updateEvtDelegateRegistry
     * @function
     * @private
     * @param updatedOption {String} (Optional) Ghe options currently being updated
     * @param domEvent {accedo.ui.Evt} (Optional) The DOM attach/detach event that causes this update
     * @memberof accedo.ui.AbstractComponent#
     */
    _updateEvtDelegateRegistry: function(updatedOption, domEvent){
        // XDK CHANGE
        return;
        // if(!accedo.Env.hasMouse()){
        //     return;
        // }

        // var inDOM;
        // switch (domEvent) {
        //     case accedo.ui.Evt.DETACHED_FROM_DOM:
        //         inDOM = false;
        //         break;
        //     case accedo.ui.Evt.ATTACHED_TO_DOM:
        //         inDOM = true;
        //         break;
        //     default:
        //         inDOM = this.getRoot().isInDOMTree();
        // }

        // if (updatedOption && !inDOM) { // updating option while not in DOM, do nothing
        //     return;
        // }

        // if (updatedOption) { // in DOM option update, need to update
        //     if (!this.opts.focusable && !this.opts.clickable) { // no event for this component
        //         accedo.ui.EvtDelegate.unregister(this);
        //     } else { // do have events
        //         accedo.ui.EvtDelegate.register(this);
        //     }
        //     return;
        // }

        // if (!inDOM) { // out of DOM tree, do unreg
        //     accedo.ui.EvtDelegate.unregister(this);
        //     return;
        // }
        // accedo.ui.EvtDelegate.register(this);
       
    },

    /**
     * Sets a new option with given name.
     * @name setOption
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @param {String} name The option name
     * @param {String} defaultValue The value of the option
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     */
    setOption: function(name, value) {
        this.opts[name] = value;
        // XDK CHANGE
        // if (!accedo.Env.hasMouse()) {
            return this;
        // }

        // if (name === "focusable" || name === "clickable") {
        //     this._updateEvtDelegateRegistry(name);
        // }
        
        // return this;
    },
    
    
    /**
     * This function returns the root element of this component.
     * @name getRoot
     * @function
     * @return {accedo.Element} The root element
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    getRoot: function() {
        return this.root;
    },

    /**
     * This function returns the parent element of this component.
     * @name getParent
     * @function
     * @return {DOM Element} The parent element
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    getParent: function() {
        return this.parent;
    },

    /**
     * Get the first element that matches the criteria function,
     * beginning at the current element and progressing up through the DOM tree.
     * Stops when until criteria is met.
     * @name closest
     * @function
     * @param {Function} criteriaFn Would the current component if this function return is true. This function gets a current component as input.
     * @param {Function} untilFn Stops if this function return is true. This function gets a current component as input.
     * @return {DOM Element} The matched parent element, null if "unitl" reached or not found
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    closest: function(criteriaFn, untilFn){
        var component = this;
        do {
            if (untilFn && untilFn(component)) { // until reached, do not go on
                return null;
            }
            if (criteriaFn(component)) {
                return component;
            }
            component = component.getParent();
        } while(component);
        return null; // not found
    },
    /**
     * Get parent controller.
     * @name getParentController
     * @function
     * @public
     * @return {accedo.ui.Controller}
     * @memberof accedo.ui.AbstractComponent#
     **/
    getParentController:function() {
        var parent = this.getParent();
        
        while (!(parent instanceof accedo.ui.Controller)) {
            if (!parent) {
                return null;
            }
            parent = parent.getParent();
        }
        
        return parent;
    },
    /**
     * Get root controller
     * @name getRootController
     * @return {accedo.ui.RootController}
     * @public
     * @function
     * @memberof accedo.ui.AbstractComponent#
     **/
    getRootController:function() {
        var parent = this.parent;
        
        while (!(parent instanceof accedo.ui.RootController)) {
            if (!parent) {
                break;
            }
            parent = parent.getParent();
        }
        
        return parent;
    },
    /**
     * Request Focus from Focus Manager
     * @name requestFocus
     * @param {Boolean} [noActive] - Optional. Do not set as active.
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @public
     * @function
     * @memberof accedo.ui.AbstractComponent#
     **/
    setFocus:function(noActive) {
        if(!this._focused){
            accedo.ui.FocusManager.singleton().requestFocus(this, {
                noActive: noActive
            });
        }
        return this;
    },
    /**
     * To handle the event passed from event delegation
     * @name requestFocus
     * @param {String} [type] - the type of the event e.g "click","mouseover"
     * @public
     * @function
     * @memberof accedo.ui.AbstractComponent#
     **/
    // XDK CHANGE
    // eventHandling:function(type){
    //     var fm = accedo.ui.FocusManager.singleton();
    //     // handles the valid event
    //     switch (type) {
    //         case "click":
    //             this.onClick();
    //             break;
    //         case "mouseover":
    //             fm.setMouseOnOff(true);
    //             //to focus the new element
    //             if (fm.getCurrentFocus() !== this) {
    //                 fm.requestFocus(this);
    //             }
    //             break;
    //         default:
    //             break;
    //     }  
    // },
    /**
     * This function shows this component. Dispatches a 'show' event.
     * @name show
     * @function
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    show: function() {
        
        this._hidden=false;
        
        this.root.show();

        //Dispatches an event
        this.dispatchEvent(accedo.ui.Evt.SHOWN);

        return this;
    },

    /**
     * This function shows this component. Dispatches a 'hide' event.
     * @name hide
     * @function
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    hide: function() {
        
        this._hidden=true;
        
        this.root.hide();

        //Dispatches an event
        this.dispatchEvent(accedo.ui.Evt.HIDDEN);

        return this;
    },
    
    /**
     * This function toggle this component's display between the none and inherit values.
     * @name toggle
     * @function
     * @return {boolean} true if toggled to shown, false if toggled to hidden
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    toggle: function() {
        if (this.root.toggle()) {
            this._hidden=false;
            this.dispatchEvent(accedo.ui.Evt.SHOWN);
            return true;
        } else {
            this._hidden=true;
            this.dispatchEvent(accedo.ui.Evt.HIDDEN);
            return false;
        }
    },
    /**
     * This function returns whether this component is hidden or not
     * @name isHidden
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    isHidden: function() {
        return this._hidden;
    },
    /**
     * Sets the components visibility by changing roots visibility.
     * @name setVisible
     * @function
     * @param {Boolean} visible
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    setVisible: function(visible) {
        this.root.setVisible(visible);
        return this;
    },

    /**
     * This function finds out whether this component is attached
     * to a parent component or not.
     * @name isAttached
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @returns {true|false}
     */
    isAttached: function() {
        return this.parent !== null;
    },
    /**
     * This function detaches this component from its current parent
     * Dispatches a 'detach' event.
     * @name detach
     * @function
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    detach: function() {
        if (!this.isAttached()) {
            return this;
        }

        if (this.parent) {
            this.parent.remove(this);
        }
        return this;
    },
    /**
     * For internal use, this function does the real work on detaching this component from its current parent
     * Dispatches a 'detach' event.
     * @name doDetach
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    doDetach: function() {
        var inDOM = this.getRoot().isInDOMTree(), parentController = this.getParentController();	
        this.root.remove();
        this.parent = null;
        this._onDetachUpdate(inDOM,parentController);
    },
    /**
     * To dispatch update event to the controller when detach
     * @name _onDetachUpdate
     * @function
     * @private
     * @memberof accedo.ui.AbstractComponent#
     */
    _onDetachUpdate:function(inDOM,parentController){
        //Dispatches an event
        this.dispatchDetached();
		
        if (parentController) {
            this.dispatchDetachedFromController(parentController);
        }        
                
        if (inDOM) {
            this.dispatchDetachFromDOMEvent();
        }
    },
    /**
     * To dispatch update event to the controller when attach
     * @name _onAttachUpdate
     * @function
     * @private
     * @memberof accedo.ui.AbstractComponent#
     */
    _onAttachUpdate:function(){
        //Dispatches an event
        this.dispatchAttached();
        
        if (this.parent._attachedToController || this.parent instanceof accedo.ui.Controller) {
            //Dispatches an event
            this.dispatchAttachedToController(this.getParentController());
        }

        if (this.getRoot().isInDOMTree()){
            this.dispatchAttachedToDOMEvent();
        }
    },
    /**
     * To dispatch the event when it is detached to the dom
     * 
     * @name dispatchAttachedToController
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @protected
     * @public
     */
    dispatchAttachedToController: function(parentController) {
        if (this._attachedToController){
            return;
        }
        
        this._attachedToController = true;
        
        this.dispatchEvent(accedo.ui.Evt.ATTACHED_TO_CONTROLLER, parentController);
    },
    /**
     * To dispatch the event when it is attached to the dom
     * @name dispatchDetachedFromController
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @protected
     * @public
     */
    dispatchDetachedFromController: function(parentController) {
        
        if (!this._attachedToController){
            return;
        }
        
        this._attachedToController = false;
        
        this.dispatchEvent(accedo.ui.Evt.DETACHED_FROM_CONTROLLER,parentController);
    },
    /**
     * To dispatch the event when it is attached to the dom
     * @name dispatchDetachFromDOMEvent
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @protected
     * @public
     */
    dispatchDetachFromDOMEvent: function() {
        this._updateEvtDelegateRegistry(null, accedo.ui.Evt.DETACHED_FROM_DOM);

        this.dispatchEvent(accedo.ui.Evt.DETACHED_FROM_DOM);
    },
    /**
     * To dispatch the event when it is detached to the dom
     * 
     * @name dispatchAttachedToDOMEvent
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @protected
     * @public
     */
    dispatchAttachedToDOMEvent: function() {
        this._updateEvtDelegateRegistry(null, accedo.ui.Evt.ATTACHED_TO_DOM);

        this.dispatchEvent(accedo.ui.Evt.ATTACHED_TO_DOM);
    },

    /**
     * To dispatch the event when it is detached
     * @name dispatchDetached
     * @memberof accedo.ui.AbstractComponent#.
     * @function
     * @protected
     * @public
     */
    dispatchDetached: function() {
        this.dispatchEvent(accedo.ui.Evt.DETACH);
    },
    /**
     * To dispatch the event when it is attached
     *
     * @name dispatchAttached
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @protected
     * @public
     */
    dispatchAttached: function() {
        this.dispatchEvent(accedo.ui.Evt.ATTACH);
    },
    /**
     * This function attaches this component to the designated parent DOM Element (and
     * detaches itself from its current parent).
     * Dispatches a 'attach' event.
     * @name attachTo
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @param {Document Element} parent The parent element to hold current component
     * @param {Document Element} insert (optional) if undfined or boolean true value, this component is appended into parent;
     *   if a {Document Element} is passed in, it is used as a reference for the placement
     * @param {string} placement where to place this component, one of the values:
     *   accedo.ui.AbstractComponent.PLACE_BEFORE - this component will be inserted before element specified by insert;
     *   accedo.ui.AbstractComponent.PLACE_AFTER - this component will be inserted after element specified by insert;
     *   accedo.ui.AbstractComponent.PLACE_PREPEND - this component will be prepended into element specified by insert;
     *   accedo.ui.AbstractComponent.PLACE_APPEND - this component will be appended into element specified by insert;
     *   accedo.ui.AbstractComponent.PLACE_CHILD - this component will be appended into the first child of the element specified by insert;
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     */
    attachTo: function(parent, insert, placement) {
        
        this.parent = parent;
        if (accedo.Object.isUndefined(insert)|| insert===true) {
            this.parent.getRoot().appendChild(this.root);
        }
        else {
            switch (placement) {
                case accedo.ui.AbstractComponent.PLACE_AFTER:
                    insert.getParent().insertAfter(this.root,insert);
                    break;

                case accedo.ui.AbstractComponent.PLACE_BEFORE:
                    insert.getParent().insertBefore(this.root,insert);
                    break;
        
                case accedo.ui.AbstractComponent.PLACE_APPEND:
                    insert.appendChild(this.root);
                    break;

                case accedo.ui.AbstractComponent.PLACE_PREPEND:
                    insert.prependChild(this.root,true);
                    break;
                
                case accedo.ui.AbstractComponent.PLACE_CHILD:
                    insert._dom.firstChild.appendChild(this.root._dom);
                    break;

                default: // default case would be accedo.ui.AbstractComponent.PLACE_APPEND case
                    insert.appendChild(this.root);
            }
        }

        this._onAttachUpdate();
        return this;
    },
    
    /**
     * This function is used to replace target with this component
     * @name replaceTarget
     * @function
     * @public
     * @memberof accedo.ui.AbstractComponent#
     * @param {accedo.ui.AbstractComponent} target component will be replaced
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     */
    replaceTarget:function(target){
        var inDOM = target.getRoot().isInDOMTree(), parentController = target.getParentController();
        this.parent = target.getParent();
        target.getRoot().getParent().replaceChild(this.root,target.getRoot());
        target.parent = null;
        target._onDetachUpdate(inDOM,parentController);
        this._onAttachUpdate();
        return this;
    },
    /**
     * Returns components horizontal alignment
     * @name getHorizontalAlignment
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     * @deprecated this method is not used and will be removed in future
     */
    getHorizontalAlignment: function(alignment, defaultAlignment) {
        var result = null;

        /*jshint bitwise: false*/
        if (alignment & accedo.ui.AbstractComponent.LEFT) {
            result = 'left';
        }
        else if (alignment & accedo.ui.AbstractComponent.RIGHT) {
            result = 'right';
        }
        else if (alignment & accedo.ui.AbstractComponent.CENTER) {
            result = 'center';
        } else if (defaultAlignment !== null) {
            result = this.getHorizontalAlignment(defaultAlignment, null);
        }
        /*jshint bitwise: true*/

        return result;
    },

    /**
     * Returns components vertical alignment
     * @name getVerticalAlignment
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     * @deprecated this method is not used and will be removed in future
     */
    getVerticalAlignment: function(alignment, defaultAlignment) {

        var result = null;

        /*jshint bitwise:false*/
        if (alignment & accedo.ui.AbstractComponent.TOP) {
            result = 'top';
        }
        else if (alignment & accedo.ui.AbstractComponent.BOTTOM) {
            result = 'bottom';
        }
        else if (alignment & accedo.ui.AbstractComponent.MIDDLE) {
            result = 'middle';
        } else if (defaultAlignment !== null) {
            result = this.getVerticalAlignment(defaultAlignment, null);
        }
        /*jshint bitwise:true*/ 

        return result;
    },
    /**
     * This function signals this component will not be used, and should collect memory to prevent any memory leaks.
     * @name deinit
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    deinit: function() {
        this.root.deinit(); // remove listeners that are attached to DOM element
        this.removeAllListeners(); // remove listeners that are attached to event dispatcher
    },
    
    /**
     * Handle how the component should handle the key event during bubbling and disptaches the EVT_KEY event.
     * Due to chain of responsibility, should only return true if the key has been handle.
     * @name onKey
     * @param {accedo.vKey} vKey
     * @param {accedo.ui.AbstractComponent} currentFocus - The current focused element
     * @return {Boolean} whether the key has been handled
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    onKey: function(vKey, currentFocus) {
        this.dispatchEvent(accedo.ui.Evt.KEY, vKey, currentFocus);
        if (vKey === accedo.VKey.KEY_ENTER && this.getOption("enterAsClick", false)) {
            return this.onClick();
        }
        return false;
    },

    /**
     * Handle how the component should handle the click event
     * @name onClick
     * @function
     * @return {Boolean} whether the click is correctly handled (e.g. accedo.ui.Evt.CLICK event is successfully fired)
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    onClick: function(){
        if (!this.getOption("clickable", false)) {
            return false;
        }
        this.dispatchEvent(accedo.ui.Evt.CLICK);
        return true;
    },

    /**
     * Handle how the component should handle the key event during capturing
     * Due to chain of responsibility, should only return true if the key has been handle.
     * Important Note: returning true will also skips the key bubbling phase!
     * @name onKeyCapture
     * @param {accedo.vKey} vKey
     * @param {accedo.ui.AbstractComponent} currentFocus - The current focused element
     * @return {Boolean} whether the key has been handled
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.AbstractComponent#
     */
    onKeyCapture: function(vKey, currentFocus) {
        return false;
    },
    
    /**
     * Handle key event by chain of responsibility
     *
     * @name handleKey
     * @param {accedo.vKey} vKey
     * @param {accedo.ui.AbstractComponent} currentFocus - Current focused component 
     * @return {Boolean} true when it is already handled by the component itself and then stop bubbling up to its parents,otherwise(false case) keys will be handle by the rootController,focusManager or further.
     * @function
     * @protected
     * @public
     * @memberof accedo.ui.AbstractComponent#
     */
    handleKey: function(vKey, currentFocus, isCapture) {
        //currentFocus  = currentFocus || this;
        
        // for capturing, parents goes first
        if (isCapture) { 
            
            if (this.onKeyCapture(vKey, currentFocus)) {
                return true;
            }
            
            if (this.isLeaf()) {
                //change to bubbling
                return this.handleKey(vKey, currentFocus);
            }
            
            return false;
        }
        
        // for bubbling, try current component first
        if (this.onKey(vKey, currentFocus)) {
            return true;
        }

        if (this.parent && this.parent.handleKey(vKey, currentFocus)){
            return true;
        }
        
        return false;
    },
    /**
     * Set as the active widget from the container
     * @name setActive
     * @param {Boolean || accedo.ui.AbstractComponent } [noTrailActive] - Optional.If it is boolean, true will only apply to the current component while false will setActive till the root.If it is abstract component, it will set active trail until the input abstractComponent(including).If the abstractComponent is not found,it will set active until the top of the tree and act like the false case. Default is False  
     * the entire trail. False to skip.
     * active.
     * @function
     * @memberof accedo.ui.AbstractComponent#
     * @public
     */
    setActive: function(noTrailActive) {
        
        var parent = this.getParent();
        
        if (!parent) {
            return;
        }
        
        parent.setActiveChild(this);
        
        if(noTrailActive === true){
            return;
        }
        
        if(parent === noTrailActive){
            return;
        }
        
        parent.setActive(noTrailActive);
    },
    /**
     * Returns true if itself is active
     * @name isActive
     * @param {Boolean} [noTrail] - Optional. Default False to activate 
     * the entire trail. False to skip.
     * @return {Boolean} True if itself is active
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @public
     */
    isActive: function(noTrail) {
        
        var parent = this.getParent();
        
        if (!parent || (!noTrail && parent && !parent.isActive())) {
            return false;
        }
        
        return parent.isChildActive(this);
    },
    /**
     * Returns true if itself is leaf component
     * @name isLeaf
     * @return {Boolean} True if itself is leaf component
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @public
     */
    isLeaf: function() {
        return true;
    },
    /**
     * Get the most active leaf component from this node
     * @name getActiveLeaf
     * @return {accedo.ui.AbstractComponent} 
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @public
     */
    getActiveLeaf: function() {
        return this;
    },
    /**
     * Get the timesptamp when this component is focused
     * @name getLastFocused
     * @return {Integer|null} The timestamp when this component is focused
     * @memberof accedo.ui.AbstractComponent#
     * @function
     * @public
     */
    getLastFocused: function() {
        return this._lastFocused;
    }
    
});

/*jslint browser: true, devel: true */
/*global accedo: false */
accedo.loadDefinition("accedo.ui.layout.Absolute");
accedo.loadDefinition("accedo.ui.layout.Normal");
accedo.loadDefinition("accedo.ui.layout.Linear");
accedo.loadDefinition("accedo.ui.layout.Relative");
accedo.loadDefinition("accedo.ui.widget.Layout");
/**
 * Basic container component to accomplish parent-child relationship for the 
 * view's tree structure.
 * @name Container
 * @memberof accedo.ui
 * @class Container is a component that can have children.
 *     This is the class that all UI containers is based upon.
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @extends accedo.ui.AbstractComponent
 * 
 */
accedo.Class.create("accedo.ui.Container", "accedo.ui.AbstractComponent",function() {
    return {
        /**
         * Event: child focus
         * @constant
         * @name EVT_CHILD_FOCUS
         * @memberof accedo.ui.Container
         * @deprecated moved to accedo.ui.Evt.CHILD_FOCUS
         */
        EVT_CHILD_FOCUS: accedo.ui.Evt.CHILD_FOCUS,
        /**
         * Event: child blur
         * @constant
         * @name EVT_CHILD_BLUR
         * @memberof accedo.ui.Container
         * @deprecated moved to accedo.ui.Evt.CHILD_BLUR
         */
        EVT_CHILD_BLUR: accedo.ui.Evt.CHILD_BLUR
    
    };

}, {
    
    /**
     * This array holds a pointer to all child objects.
     * @private
     * @memberof accedo.ui.Container#
     */
    _children: [],
    
    
    _activeness: [],
    
    /**
     * Overrides the initalize method inherits from AbstractComponent
     * @name initialize
     * @returns {accedo.ui.Container} A handle to itself
     * @memberof accedo.ui.Container#
     * @public
     * @protected
     */
    initialize: function($super, opts) {
        opts = opts || {};

        opts.root = this.createLayoutElement(opts);
        $super(opts);
    },
    /**
     * To do after the initialization of all container widgets, and it is for container to create its children and attach to the container when there are children in options
     * @name postInitialize
     * @memberof accedo.ui.Container#
     * @protected
     */
    postInitialize:function(){
        if(!this.opts.children && !accedo.Object.isArray(this.opts.children)){
            return;
        }
        //do when there are children
        var i = 0, l = this.opts.children.length, childOpts, subview;
        if (l > 0) {
            for (; i < l; i++) {
                childOpts = this.opts.children[i];
                if(!this.opts.skipRealizeChildren){
                    subview = accedo.ui.View.realize(childOpts);
                        
                    if (subview){
                        this.attach(subview);
                    }
                }
                
            }
        } 
    },
    /**
     * Root element's Factory method for a conatiner. Called during the
     * initalization of the Container. 
     * Subclasses of Container should override this method if needed.
     * 
     * @name createLayoutElement
     * @returns {accedo.Element} - Container's root element
     * @memberof accedo.ui.Container#
     * @function
     * @protected
     * @public
     */
    createLayoutElement: function() {
        return accedo.Element.create("div");
    },
    
    /**
     * This function retrieves a child object by its ID. It is worth
     * noting that this is a recursive function, so it will search
     * child containers as well for the give ID. Overrides get function 
     * from AbstractComponent
     * @name get
     * @function
     * @param {String} id The object identifier
     * @returns {accedo.ui.AbstractComponent|null} The component having the given identifier,
     *                                     or null if not found
     * @memberof accedo.ui.Container#
     * @public
     */
    get: function(id) {
        var i, len, obj;

        if (this.getId() === id) {
            return this;
        }

        i = 0;
        len = this._children.length;
        for (; i < len; i++) {
            obj = this._children[i].get(id);
            if (obj) {
                return obj;
            }
        }
        return null;
    },
    /**
     * This function retrieves a child contrller by its full classname. It is worth
     * noting that this is a recursive function, so it will search
     * child containers as well for the give controller class. 
     * @name getControllerByDef
     * @function
     * @param {String} controllerDef - The controller's full classname
     * @returns {accedo.ui.Controller|null} The component having the given identifier,
     *                                     or null if not found
     * @memberof accedo.ui.Container#
     * @public
     */
    getControllerByDef: function(controllerDef) {
        var i, len, obj;

        if (this.getDefinition() === controllerDef) {
            return this;
        }

        i = 0;
        len = this._children.length;
        for (; i < len; i++) {
            if (!this._children[i] instanceof accedo.ui.Container) {
                continue;
            }
            obj = this._children[i].getControllerByDef(controllerDef);
            if (obj) {
                return obj;
            }
        }
        return null;
    },
    /**
     * This function retrives the current set of children for this container.
     * @name getChildren
     * @function
     * @return {Array} The child array
     * @memberof accedo.ui.Container#
     * @public
     */
    getChildren: function() {
        return accedo.Array.clone(this._children);
    },
    /**
     * This function attachs a child object.
     * @name attach
     * @function
     * @param {accedo.ui.AbstractComponent} child - The child component to be
     * attached
     * @param {Boolean|accedo.AbstractComponent} [insert] - Optional. False/Default: insert as last child.
     * True: insert as first child; AbstractComponent: At the placement target.
     * @param {accedo.ui.AbstractComponent.PLACE_BEFORE|accedo.ui.AbstractComponent.PLACE_AFTER} [placement] -  
     * PLACE_BEFORE: Insert the child before the placement target; PLACE_AFTER: Insert the child after the placement target.
     * @memberof accedo.ui.Container#
     * @public
     */
    attach: function(child, insert, placement) {
        
        if (this._children.indexOf(child)!==-1) {
            accedo.console.warn("Child is already attached to container");
            return;
        }
        
        if (accedo.Object.isUndefined(placement)) {
            placement = accedo.ui.AbstractComponent.PLACE_BEFORE;
        }
        
        if (accedo.Object.isUndefined(insert) || insert===false) {
            this._children.push(child);            
        }
        else if (insert===true){
            this._children = [child].concat(this._children);
        }
        else {
            var idx = this._children.indexOf(insert);
            
            if (idx === -1) {
                accedo.console.error("Cannot find insert target component.");
            }
            if (accedo.Object.isUndefined(placement)) {
                placement = accedo.ui.AbstractComponent.PLACE_BEFORE;
            }
            
            switch (placement) {
                case accedo.ui.AbstractComponent.PLACE_BEFORE:
                    if (idx===0) {
                        this._children.unshift(child);
                    }
                    else {
                        this._children.splice(idx-1,0,child);
                    }
                    break;
                case accedo.ui.AbstractComponent.PLACE_AFTER:
                    this._children.splice(idx,0,child);
                    break;
                case accedo.ui.AbstractComponent.PLACE_REPLACE:
                    this._children.splice(idx,1,child);
                    break;
            }
        }
        
        this.doAttachElement(child, insert, placement);
        
        
        this._activeness.push(child);
        
    },
    /**
     * replace the current child
     * @name doReplace
     * @function
     * @param {accedo.ui.AbstractComponent} child - The new child component to replace
     * @param {accedo.AbstractComponent} target the child being replaced
     * @memberof accedo.ui.Container#
     * @public
     */
    doReplace:function(child, target){
        var idx = this._children.indexOf(target), activeIdx = this._activeness.indexOf(target);
        if (idx === -1) {
            accedo.console.warn("the old one is not in the container");
            return;
        }
        
        this._children.splice(idx,1,child);
        this._activeness.splice(activeIdx,1,child);
        child.replaceTarget(target);
        
    },
    /**
     * Delegated method to handle how the child should be added to the 
     * view based on attach params.
     * 
     * @name doAttachElement
     * @override
     * @param {accedo.ui.AbstractComponent} child - see accedo.ui.Container#attach
     * @param {Boolean|accedo.AbstractComponent} [insert] - see accedo.ui.Container#attach
     * @param {accedo.ui.AbstractComponent.PLACE_BEFORE|accedo.ui.AbstractComponent.PLACE_AFTER} [placement] - see accedo.ui.Container#attach
     * @memberof accedo.ui.Container#
     * @function
     * @public
     * @protected
     */
    doAttachElement: function(child, insert, placement) {
        
        //Append if not inserting
        if (insert instanceof accedo.ui.AbstractComponent) {
            insert = insert.getRoot();
        }
        else if (!insert) {
            placement = accedo.ui.AbstractComponent.PLACE_APPEND;
            insert = this.getRoot();
        }
        else {
            placement = accedo.ui.AbstractComponent.PLACE_PREPEND;
            insert = this.getRoot();
        }
        
        child.attachTo(this, insert, placement);
    },
    /**
     * To dispatch the event to the children
     * @name dispatchAttachedToDOMEvent
     * @memberof accedo.ui.Container#
     * @function
     * @protected
     * @public
     */
    dispatchAttachedToDOMEvent: function($super) {
        var i;
        for (i=0;i<this._children.length;i++){
            this._children[i].dispatchAttachedToDOMEvent();
        }
        $super();
    },
    /**
     * To dispatch the event to the children
     * @name dispatchDetachFromDOMEvent
     * @memberof accedo.ui.Container#
     * @function
     * @protected
     * @public
     */
    dispatchDetachFromDOMEvent: function($super) {
        var i;
        for (i=0;i<this._children.length;i++){
            this._children[i].dispatchDetachFromDOMEvent();
        }
        $super();
    },
    /**
     * To dispatch the event to the 
     * @name dispatchAttachedToController
     * @memberof accedo.ui.Container#
     * @function
     * @override
     * @protected
     * @public
     */
    dispatchAttachedToController: function($super,parentController) {
        
        if (this._attachedToController){
            return;
        }
        
        var i;
        for (i=0;i<this._children.length;i++){
            this._children[i].dispatchAttachedToController(parentController);
        }
        $super(parentController);
    },
    /**
     * To dispatch the event to the children
     * @name dispatchDetachedFromController
     * @memberof accedo.ui.Container#
     * @function
     * @override
     * @protected
     * @public
     */
    dispatchDetachedFromController: function($super,parentController) {
        
        if (!this._attachedToController){
            return;
        }
        
        var i;
        for (i=0;i<this._children.length;i++){
            this._children[i].dispatchDetachedFromController(parentController);
        }
        $super(parentController);
    },
    /**
     * This function replace an existing child component with a new child component.
     * @name replace
     * @function
     * @param {accedo.ui.AbstractComponent} newChild - The child component to be added
     * @param {accedo.ui.AbstractComponent} [existingChild] - Optional. Target child to replace. 
     * Attach the newChild if existingChild not provided.
     * @memberof accedo.ui.Container#
     * @public
     */
    replace: function(newChild, existingChild) {
        
        if (newChild === existingChild) {
            return;
        }
        
        
        if (!existingChild){
            this.attach(newChild);
            return;
        } 
        
        var targetIdx = this._children.indexOf(existingChild);
        
        if (targetIdx===-1){
            this.attach(newChild);
            return;
        }
        this.doReplace(newChild,existingChild);
    },

    /**
     * This function removes a child object.
     * @name remove
     * @function
     * @param {accedo.ui.AbstractComponent} child The child object to remove
     * @memberof accedo.ui.Container#
     * @public
     */
    remove: function(child) {
        
        var idx = accedo.Array.indexOf(this._children, child), activeIdx;
        
        
        if (idx > -1) {
            this._children.splice(idx, 1);
            activeIdx = accedo.Array.indexOf(this._activeness, child);
            this._activeness.splice(activeIdx, 1);
            
            child.doDetach();
        }
        
    },

    /**
     * This function removes all child objects.
     * @name removeAll
     * @function
     * @memberof accedo.ui.Container#
     * @public
     */
    removeAll: function() {
        var i = this._children.length;
        while (i--) {
            this._children[i].detach();
        }
        this._children = [];
    },
          
    /**
     * Deinit remove event listeners for itself and also ask its children to remove the event listers
     * @name deinit
     * @function
     * @memberof accedo.ui.Container#
     * @public
     */
    deinit: function($super) {
        $super();
                
        var i, len;

        i = 0;
        len = this._children.length;
        for (; i < len; i++) {
            this._children[i].deinit();
        }
                
        this.removeAll();
    },
    /**
     * @name onChildBlur
     * @function
     * @memberof accedo.ui.Container#
     * @protected
     * @public
     */
    onChildBlur: function(child) {
        this.root.removeClass("focus-trail");
        this.dispatchEvent(accedo.ui.Evt.CHILD_BLUR, child);
        if (this.parent) {
            this.parent.onChildBlur(child);
        }
    },
    /**
     * @name onChildFocus
     * @function
     * @memberof accedo.ui.Container#
     * @protected
     * @public
     */
    onChildFocus: function(child) {
        this.root.addClass("focus-trail");
        this.dispatchEvent(accedo.ui.Evt.CHILD_FOCUS, child);
        if (this.opts.useLastFocus) {
            this.lastFocusChild = child;
        }
        if (this.parent) {
            this.parent.onChildFocus(child);
        }
    },
    /**
     * Returns whether any of it's child is focused
     * @name isChildFocused
     * @function
     * @memberof accedo.ui.Container#
     * @public
     */
    isChildFocused: function() {
        return this.root.hasClass("focus-trail");
    },
    /**
     * Handle key event by chain of responsibility, override
     *
     * @name handleKey
     * @param {accedo.vKey} vKey
     * @param {accedo.ui.AbstractComponet} currentFocus - Current focused component 
     * @return {Boolean} true when it is already handled by the component itself and then stop bubbling up to its parents,otherwise(false case) keys will be handle by the rootController,focusManager or further.
     * @function
     * @protected
     * @override
     * @memberof accedo.ui.AbstractComponent#
     */
    handleKey: function($super, vKey, currentFocus, isCapture) {
        
        if ($super(vKey, currentFocus, isCapture)) {
            return true;
        }
        
        var activeChild = this.getActiveChild();
        
        if (isCapture && !this.isLeaf() && activeChild.handleKey(vKey, currentFocus, isCapture)) { // for capturing, parents goes first
            return true;
        }
        
        return false;
    },
    /**
     * Sets enabled state for component and ancestor Enabled state for its children
     * @name setEnabled
     * @function
     * @protected
     * @override
     * @param {Boolean} toggle true or false for enabled state
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.Container#
     */
    setEnabled:function($super,toggle){
        var ret = $super(toggle);
        this.onAncestorSetEnabled(toggle,true);
        return ret;
    },
    /**
     * Set the ancestor enable state
     * @name onAncestorSetEnabled
     * @function
     * @protected
     * @override
     * @param {Boolean} toggle true or false for enabled state
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.Container#
     */
    onAncestorSetEnabled:function(toggle, isRoot){
        var i = 0, obj,
        len = this._children.length;
        if(!isRoot){
            this._ancestorDisabled = !toggle;
        }
        for (; i < len; i++) {
            obj = this._children[i];
            obj.onAncestorSetEnabled(toggle);
        }
        return this;
    },
    /**
     * show and update the status of the children's ancestor Hidden state
     * @name show
     * @function
     * @override
     * @protected
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.Container#
     */
    show:function($super){
        var ret = $super();
        this.onAncestorShow(true);
        return ret;
    },
    /**
     * Set the ancestor hide state
     * @name onAncestorShow
     * @function
     * @override
     * @protected
     * @memberOf accedo.ui.Container#
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     */
    onAncestorShow:function(isRoot){
        var i = 0, obj,
        len = this._children.length;
        if(!isRoot){
            this._ancestorHidden = false;
        }
        for (; i < len; i++) {
            obj = this._children[i];
            obj.onAncestorShow();
        } 
        return this;
    },
    /**
     * hide and update the status of the children's ancestorHidden state
     * @name hide
     * @function
     * @override
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.Container#
     */
    hide:function($super){
        var ret = $super();
        this.onAncestorHide(true);
        return ret;
    },
    /**
     * update the status of the children's ancestor Hidden state
     * @name onAncestorHide
     * @function
     * @protected
     * @override
     * @return {accedo.ui.AbstractComponent} The component itself, for chaining
     * @memberOf accedo.ui.Container#
     */
    onAncestorHide:function(isRoot){
        var i = 0, obj,
        len = this._children.length;
        if(!isRoot){
            this._ancestorHidden = true;
        }
        for (; i < len; i++) {
            obj = this._children[i];
            obj.onAncestorHide();
        } 
        return this;
    },
    /**
     * Set the child as most active
     * @name setActiveChild
     * @function
     * @param {String|accedo.ui.AbstractComponent} child - id or child instance
     * @memberof accedo.ui.Container#
     * @return {Boolean} FALSE if id not found or input is not a child of this
     * container
     */
    setActiveChild: function(child) {
        
        if (accedo.Object.isString(child)) {
            child = this.get(child);
        }
        
        if (!child) {
            return false;
        }
        var idx = accedo.Array.indexOf(this._activeness, child);
        
        if (idx <= 0) {
            if (idx < 0) {
                accedo.console.warn("Container ("+ this.getId() +": Trying to set child active that is not one of it's children.");
            }
            return false;
        }
        
        this._activeness.splice(idx,1);
        
        //add to the front
        this._activeness.unshift(child);
        
        //dispatch inactive event
        this._activeness[1].dispatchEvent(accedo.ui.Evt.INACTIVED);
        
        //dispatch active event
        this._activeness[0].dispatchEvent(accedo.ui.Evt.ACTIVE);
        
        return true;
    },
    /**
     * Get the most active child
     * @name getActiveChild
     * @function
     * @param {Number} [activeIdx] - Optional. 0 by default. Activeness of the 
     * child, 0 is most active. 
     * @return {accedo.ui.AbstractComponent|null} return null if there's no 
     * child or activeness is invalid
     * @memberof accedo.ui.Container#
     */
    getActiveChild: function(activeIdx) {
        
        if (accedo.Object.isUndefined(activeIdx)) {
            activeIdx = 0;
        }
        
        //out of scope
        if (activeIdx < 0 || activeIdx > this._activeness.length-1) {
            return null;
        }
        
        return this._activeness[activeIdx];
    },
    /**
     * Get the most active leaf component from this node
     * @name getActiveLeaf
     * @return {accedo.ui.AbstractComponent} 
     * @memberof accedo.ui.Container#
     * @override
     * @function
     * @public
     */
    getActiveLeaf: function() {
        
        if (this.isLeaf()) {
            return this;
        }
        
        return this.getActiveChild().getActiveLeaf();
    },
    /**
     * Returns true if the child is active
     * @name isChildActive
     * @return {Boolean} True if the child is active
     * @memberof accedo.ui.Container#
     * @function
     * @public
     */
    isChildActive: function(child) {
        
        var idx = accedo.Array.indexOf(this._activeness, child);
        
        return (idx === 0);
    },
    
    /**
     * Returns true if itself is leaf component
     * @name isLeaf
     * @return {Boolean} True if itself is leaf component
     * @memberof accedo.ui.Container#
     * @function
     * @override
     * @public
     */
    isLeaf: function() {
        return this._activeness.length===0;
    }
    
    
});

/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @name Normal
 * @memberof accedo.ui.layout
 * @class Normal layout container. Leaving the positioning to be decided by css
 * @author <a href="mailto:cheung.chunho@accedobroadband.com">Cheung Chun Ho</a>
 * @extends accedo.ui.Container
 * @deprecated repalced with a new class accedo.ui.widget.Layout 
 */
accedo.Class.create("accedo.ui.layout.Normal","accedo.ui.Container", {},{
    /**
    * This function is called by the container function whenever
    * a child element has been appended.
    * @name doAttachElement
    * @override
    * @function
    * @param {accedo.ui.AbstractComponent} child
    * @memberof accedo.ui.layout.Normal#
    * @public
    */
    doAttachElement: function(child) {
        var newStyles, x, y, height, width;
        child.attachTo(this, this.getRoot());
        
        newStyles = {};

        //If we have a 'y' option use it
        y = child.getOption('y');
        if (y) {
            newStyles.top = y;
        }

        //If we have a 'x' option use it
        x = child.getOption('x');
        if (x) {
            newStyles.left = x;
        }

        //Height and width
        height = child.getOption('height');
        if (height) {
            newStyles.height = height;
        }

        width = child.getOption('width');
        if (width) {
            newStyles.width = width;
        }

        child.getRoot().setStyle(newStyles);
    }
});

 /*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @class Data container class that can ask for more data, should the data provider send incomplete data (e.g. paginated),
 * and append new data to what was previously loaded.
 * @author <a href="mailto:ming.hsieh@accedobroadband.com">Ming Hsieh</a>
 * @author <a href="mailto:gregory.desfour@accedobroadband.com">Gregory Desfour</a>
 * @author <a href="mailto:andy.hui@accedobroadband.com">Andy Hui</a>
 * @author <a href="mailto:curtiss.yeung@accedo.tv">Curtiss Yeung</a>
 * @constructor
 * @param {Object} [opts] Options object
 * @param {boolean} [opts.hasPagination] (false by default)
 * @extends accedo.mixin.EventDispatcher
 * @name Ds
 * @memberof accedo.data
 */
accedo.Class.create("accedo.data.Ds", {
    EVT_APPEND: 'accedo:datasource:append',
    EVT_LOAD: 'accedo:datasource:load',
    EVT_INSERT: 'accedo:datasource:insert',
    EVT_REMOVE: 'accedo:datasource:delete',
    EVT_UPDATED: 'accedo:datasource:updated',
    EVT_RESET: 'accedo:datasource:reset'
}, 
"accedo.mixin.EventDispatcher", 
{
     /**
     * Total number of items in the datasource.It will be set when appending data automatically or if it has pagination,it can be set by developers
     * @name _totalItems
     * @memberof accedo.data.Ds#
     * @private
     */
    _totalItems:             null,
     /**
     * Total number of items in the datasource
     * @name _sourceHasPagination
     * @memberof accedo.data.Ds#
     * @private
     */
    _sourceHasPagination :   false,
     /**
     * Loaded Data of the datasource
     * @name _loadedData
     * @memberof accedo.data.Ds#
     * @private
     */
    _loadedData :            [],
     /**
     * @name _lastPageLoaded
     * @memberof accedo.data.Ds#
     * @private
     */
    _lastPageLoaded :        0,
     /**
     * to determine whether has more data if it is pagination
     * @name _hasMoreData
     * @memberof accedo.data.Ds#
     * @private
     */
    _hasMoreData:            true,
    /**
     * @memberof accedo.data.Ds#
     * @function
     * @param {Object} opts An object
     * @private
     */
    initialize: function(opts) {
        if (opts) {
            this._sourceHasPagination = !!opts.hasPagination;
        }
    },
    
    /**
     * Append data from an Array, and notify 3rd-parties by the 'accedo:datasource:append'
     * event, which passes the 'new data' object Array as argument for convenience
     * two events invoke,accedo.data.Ds.EVT_APPEND is for backward compatible and accedo.data.Ds.EVT_UPDATED currently used in grid and list
     * @param {Array} newData Array of data objects to append to the previously loaded data
     * @param {Number} start the starting dsIndex of appending data
     * @memberof accedo.data.Ds#
     * @name appendData
     * @function
     * @public
     */
    appendData: function(newData, start) {
        var fromIndex, i, newDataLen, loadedDataLen, setHasMoreDataFalse, totalItem;
                
        //This DS is not paginated - Notify there is no more data to be loaded
        if (!this._sourceHasPagination) {
            this._hasMoreData = false;
        }
                
        //Ensure we have an array as argument
        if (accedo.Object.isArray(newData)) {
            if (!accedo.Object.isUndefined(start)){
                //start is defined, so we append/overwrite loadedData with index start
                newDataLen = newData.length;
                
                //@ToDo to handle append last page first and then previous page in pagination cases
                if(start <this._loadedData.length){
                    fromIndex = start;
                }else{
                    fromIndex = this._loadedData.length;
                }
                for (i=0; i < newDataLen;i++){
                    this._loadedData[fromIndex+i] = newData[i];
                }
            }else{
                //start is not defined, so we append the data to the end of the loadedData
                fromIndex = this._loadedData.length;
                this._loadedData = this._loadedData.concat(newData);
            }
            
            this._checkMoreData();
            
            // notify 3rd-parties
            //this one is old event which is for backward compatible
            this.dispatchEvent(accedo.data.Ds.EVT_APPEND, {
                newData: newData, 
                fromIndex: fromIndex
            });
            
            //this is new Event and current event used in the list/grid
            this.dispatchEvent(accedo.data.Ds.EVT_UPDATED, {
                action: "Append",
                newData: newData,
                startIndex: fromIndex
            });

        }

    },
    /**
     * Remove items from the current data,
     * notify 3rd-parties by the "accedo.data.Ds.EVT_UPDATED" with action:Remove
     * @name removeData
     * @function
     * @param {Object|Number} index An array of items are going to removed or the number of item is going to removed (0-th indexed)
     * @memberof accedo.data.Ds#
     * @return {Boolean} true if remove successfully,false if fail to remove or no item is removed
     * @public
     */
    removeData: function(index) {
        var removedIndex = [], i,length, 
        oldDataLength = this.getCurrentSize();
        
        if (!accedo.Object.isUndefined(index)) {
            //only handle number or array cases
            if(accedo.Object.isArray(index)){
                index.sort(function(a,b){return b-a;}); // Sort the input array in decending order
                 for (i = 0,length = index.length; i < length ; i++) {
                    if (this._loadedData.splice(index[i], 1).length) {
                        removedIndex.push(index[i]);
                    }else{
                        accedo.console.warn("Item is unable to remove "+index[i]);
                    }
                }
                accedo.console.log(removedIndex);
            }else if(accedo.Object.isNumber(index)){
                if (this._loadedData.splice(index, 1).length === 0) {
                    return false;
                }
                removedIndex.push(index);
            }else{
                //neither number or array
                return false;
            }
           
            this._checkMoreData();
            
            // notify 3rd-parties
            if (removedIndex.length) {
                this.dispatchEvent(accedo.data.Ds.EVT_UPDATED, {
                    action: "Remove",
                    index: removedIndex,
                    totalItemBefore: oldDataLength,
                    totalItemAfter: this.getCurrentSize()
                });
                return true;
            } else {
                return false;
            }
            
        }
        return false;
    },
    /**
     * Insert the newData array into the current data, if index is not defined, the newData will added to the end to the current data
     * notify 3rd-parties by the "accedo.data.Ds.EVT_UPDATED" with action:Insert
     * @name insertData
     * @function
     * @param {Object} newData An array of items are going to inserted
     * @param {Number} index (Optional) index A 0-th indexed number indicated where to insert the newData
     * @return {Boolean} true if insert successfully
     * @memberof accedo.data.Ds#
     * @public
     */
    insertData: function(newData, index) {
        var firstPart, lastPart,
        oldDataLength = this.getCurrentSize();
        
        switch (typeof index) {
            case "number":
                firstPart = this._loadedData.slice(0, index);
                lastPart = this._loadedData.slice(index, oldDataLength);
                index = Math.min(index, oldDataLength);
                this._loadedData = firstPart.concat(newData, lastPart);
                break;
            default:
                index = oldDataLength;
                this._loadedData = this._loadedData.concat(newData);
                break;
        }
        this._checkMoreData();
        // notify 3rd-parties
        this.dispatchEvent(accedo.data.Ds.EVT_UPDATED,{
            action: "Insert",
            index: index,
            newData: newData,
            totalItemBefore: oldDataLength,
            totalItemAfter: this.getCurrentSize()
        });
    },
    /**
     * To update whether get enough data after each amendment
     * @name checkMoreData
     * @function
     * @memberof accedo.data.Ds#
     * @private
     */
    _checkMoreData:function(){
        var totalItem = this.getTotalItems(),
        loadedDataLen = this._loadedData.length,
        setHasMoreDataFalse,i;
        if (loadedDataLen >= totalItem) {
            //a validation to ensure loadedData is not undefined and ensure there is more data to load
            setHasMoreDataFalse = true;
            for (i=0; i < loadedDataLen; i++){
                if (accedo.Object.isUndefined(this._loadedData[i])){
                    setHasMoreDataFalse = false;
                    break;
                }
            }
            if (setHasMoreDataFalse){
                this._hasMoreData = false;
            }
        }
    },
    /**
     * This function simply dispatches the 'accedo:datasource:load' event,
     * only if it has not been indicated this datasource has no more data to fetch.
     * The receiver should then call appendData() to append new data, if any
     * @name load
     * @function
     * @param {Object} loadParams contains additional information to load data, it would be dispatched
            to the eventListener loading data. Example {dsIndex: 50} - the intended behaviour
            is to load the page with dsIndex 50
     * @memberof accedo.data.Ds#
     * @public
     */
    load: function(loadParams) {
        if (this._hasMoreData) {
            if (loadParams && loadParams.hasOwnProperty('dsIndex')){
                this.dispatchEvent(accedo.data.Ds.EVT_LOAD, {
                    dsIndex: loadParams.dsIndex
                });
            }else{
                this.dispatchEvent(accedo.data.Ds.EVT_LOAD);
            }
        }
    },

    /**
     * Return the currently loaded data in an Array.
     * @name getData
     * @function
     * @return {Array} loadedData Array of data objects that have been loaded so far
     * @memberof accedo.data.Ds#
     * @public
     */
    getData: function(){
        return this._loadedData;
    },
            
    /**
     * For ease-of-use - return a particular data object (indicated by its index among the loaded data).
     * @name getDataAtIndex
     * @function
     * @param {Number} i Index of the targeted data object in the loaded data 
     * @return {Object} A particular data object or null
     * @memberof accedo.data.Ds#
     * @public
     */
    getDataAtIndex: function(i) {
        if (i >= 0 && this._loadedData.length > i) {
            return this._loadedData[i];
        }
        return null;
    },

    /**
     * Can this datasource load more data ?
     * Before the first call to load() it is impossible to tell, in that case assume it can load more
     * (Hence hasMoreData has an initial value of true)
     * @name hasMore
     * @function
     * @return {Boolean} false if there is no need to call load() any more, true otherwise.
     * @memberof accedo.data.Ds#
     * @public
     */
    hasMore: function(){
        return this._hasMoreData;
    },
            
    /**
     * Inform this datasource can fetch no more data, as all data has been fetched already.
     * Increases performance (no more load() call), but be well sure there is really no more data
     * that can be fetched as otherwise, it will never be fetched.
     * No parameter - Setting true would not make sense as it's the default value
     * @name setHasNoMore
     * @function
     * @memberof accedo.data.Ds#
     * @public
     */
    setHasNoMore: function() {
        this._hasMoreData = false;
    },
            
    /**
     * Is this DS using pagination ? (as set by option on constructor, false otherwise)
     * @name hasPagination
     * @function
     * @return {Boolean}
     * @memberof accedo.data.Ds#
     * @public
     */
    hasPagination: function() {
        return this._sourceHasPagination;
    },

    /**
     * Return the number of elements currently in the datasource
     * @name getCurrentSize
     * @function
     * @return {Number} Current size
     * @memberof accedo.data.Ds#
     * @public
     */
    getCurrentSize: function(){
        return this._loadedData.length;
    },

    /**
     * Returns the total number of items available when fully populated
     * @name getTotalItems
     * @function
     * @return {Number} Total number of items available when fully populated if known,
     *         otherwise, returns the current item count
     * @memberof accedo.data.Ds#
     * @public
     */
    getTotalItems: function() {
        if(this._totalItems !== null){
            return this._totalItems;
        }else{
            return this.getCurrentSize();
        }
    },
            
    /**
     * Sets the total number of items available when fully populated if the argument is not undefined
     * @name setTotalItems
     * @function
     * @param {Number} n Total number of items available when fully populated 
     * @memberof accedo.data.Ds#
     * @public
     */
    setTotalItems: function(n) {
        switch(typeof n){
            case "number":
                this._totalItems = n;
                break;
            case "string":
                try{
                    this._totalItems = parseInt(n, 10);
                }catch(e){
                     accedo.console.warn("unable to parse into integer" + e);
                }
                break;
            default:
                return;
        }
        
        if(this._totalItems > this.getCurrentSize()){
            this._hasMoreData = true;
        }else{
            this._hasMoreData = false;
        }
    },
            
    /**
     * Returns the number of the last page loaded
     * @name getLastPageLoaded
     * @function
     * @return {Number} Number of the last page loaded
     * @memberof accedo.data.Ds#
     * @public
     */
    getLastPageLoaded: function() {
        return this._lastPageLoaded;
    },
            
    /**
     * Sets the number of the last page loaded if the argument is not undefined
     * @name setLastPageLoaded
     * @function
     * @param {Number} n Number of the last page loaded
     * @memberof accedo.data.Ds#
     * @public
     */
    setLastPageLoaded: function(n) {
        switch(typeof n){
            case "number":
                this._lastPageLoaded = n;
                break;
            case "string":
                try{
                    this._lastPageLoaded = parseInt(n, 10);
                }catch(e){
                    accedo.console.warn("unable to parse into integer" + e);
                }
                break;
            default:
                break;
       }
    },
            
    /**
     * Among the data that was already loaded, returns the index of the 1st that passes the iterator test
     * function successfully (meaning that applying testFn to it returns true), or -1
     * @name search
     * @function
     * @memberof accedo.data.Ds#
     * @param {Function} iterator Iterator function to use for testing (returns true for pass, false for fail)
     * @returns {Number} index of the first data to pass the iterator test, or -1
     * @public
     * @example var result = ds2.search(function(data, index) {
     *              return data.movieId == 'a01z';
     *          });
     * @example var result = ds2.search(function(data, index) {
     *              return i == 2;
     *          });
     */
    search: function(iterator) {
        var result = -1;
        accedo.Array.each(this._loadedData, function(data, i) {
            if (iterator.call(this, data, i)) {
                result = i;
                throw accedo.$break;
            }
        },this);
        return result;
    },
            
    /**
     * Remove all of the previously loaded data and now it isequal to varReset and keep it for backward compatible
     * @name purgeData
     * @function
     * @memberof accedo.data.Ds#
     */
    purgeData: function() {               
        this.varReset();
    },
    
    /**
     * Save the variables init phase in a function
     * totalItems and sourceHasPagination don't have to be reset when we call purgeData so they're not in varReset
     * it will dispatch event accedo.data.Ds.EVT_RESET
     * @name varReset
     * @function
     * @memberof accedo.data.Ds#
     */
    varReset : function() {
        this._loadedData.length = 0;//Array of loaded data objects
        this._lastPageLoaded = 0; //Number of the last page loaded (when it applies)
        this._hasMoreData = true; //Indicates there is *maybe* more loadable data, or definitely not
        //to notify 3rd party the ds is emptied,
        this.dispatchEvent(accedo.data.Ds.EVT_UPDATED, {
            action: "Reset"
        });
        
        //@depreciated event and should be used the above one
        this.dispatchEvent(accedo.data.Ds.EVT_RESET, {
            action: "reset"
        });
    }
});
// emile.js (c) 2009 Thomas Fuchs
// Licensed under the terms of the MIT license.

/*jslint browser: true, devel: true, sloppy: true */
/*global accedo: false */

/**
 * @desc External animation library.
 * Original API file can be downloaded at <a href="https://github.com/madrobby/emile">Download</a><br/>
 * emile.js (c) 2009 Thomas Fuchs<br/>
 * Licensed under the terms of the MIT license.
 * @name Emile
 * @memberof accedo.anim
 * @module
 */
accedo.Class.create("accedo.anim.Emile", {
    /**
     * @desc Stops the animation pointed by the handle parameter.
     * @name stop
     * @param  {Handle} animHandle The handle for the animation to stop.
     * @memberof accedo.anim.module:Emile
     * @function
     * @public
     * @static
     */
    stop: function(animHandle) {
        clearInterval(animHandle);
    },
    
    /**
     * Mapped to emile(el, style, opts, after).
     * For API Reference of please see 
     * <a href="https://github.com/madrobby/emile">https://github.com/madrobby/emile</a>.
     *
     * @name anim
     * @return {Handle}
     * @memberof accedo.anim.module:Emile
     * @function
     * @public
     * @static
     */
    anim: (function(){
        var parseEl = document.createElement('div'),
        props = ('backgroundColor borderBottomColor borderBottomWidth borderLeftColor borderLeftWidth '+
            'borderRightColor borderRightWidth borderSpacing borderTopColor borderTopWidth bottom color fontSize '+
            'fontWeight height left letterSpacing lineHeight marginBottom marginLeft marginRight marginTop maxHeight '+
            'maxWidth minHeight minWidth opacity outlineColor outlineOffset outlineWidth paddingBottom paddingLeft '+
            'paddingRight paddingTop right textIndent top width wordSpacing zIndex').split(' ');

        function interpolate(source,target,pos){
            return (source+(target-source)*pos).toFixed(3);
        }
        function s(str, p, c){
            return str.substr(p,c||1);
        }
        function color(source,target,pos){
            var i = 2, j, c, tmp, v = [], r = [];
            while(j=3,c=arguments[i-1],i--)
                if(s(c,0)=='r') {
                    c = c.match(/\d+/g);
                    while(j--) v.push(~~c[j]);
                } else {
                    if(c.length==4) c='#'+s(c,1)+s(c,1)+s(c,2)+s(c,2)+s(c,3)+s(c,3);
                    while(j--) v.push(parseInt(s(c,1+j*2,2), 16));
                }
            while(j--) {
                tmp = ~~(v[j+3]+(v[j]-v[j+3])*pos);
                r.push(tmp<0?0:tmp>255?255:tmp);
            }
            return 'rgb('+r.join(',')+')';
        }
  
        function parse(prop){
            var p = parseFloat(prop), q = prop.replace(/^[\-\d\.]+/,'');
            return isNaN(p) ? {
                v: q, 
                f: color, 
                u: ''
            } : {
                v: p, 
                f: interpolate, 
                u: q
            };
        }
  
        function normalize(style){
            var css, rules = {}, i = props.length, v;
            parseEl.innerHTML = '<div style="'+style+'"></div>';
            css = parseEl.childNodes[0].style;
            //in huawei EC2108(ranger), if we directly check the parseElement create above, it will have an wrong value and affect the attribute detect below
            //So we have to work around on the target style in order to solve the problem. Besides, this STB will just give the value without "px"
            if(accedo.platform === "huawei" && accedo.device.id.getModel() === "ranger" ){
                var temp, attribute, prop;
                if(style){
                    attribute = style.split(";");
                    for(prop in attribute){
                        if(attribute[prop]){
                            temp = attribute[prop].split(":");
                            if(temp.length === 2){
                                rules[temp[0]] = parse(temp[1]);
                            }
                        }
                    }
                }

            }else{
                while(i--){
                    if(v = css[props[i]]){
                        rules[props[i]] = parse(v);
                    }
                }
            }
            return rules;
        }
  
        return function(el, style, opts, after){
            el = typeof el == 'string' ? document.getElementById(el) : el;
            opts = opts || {};
            var target = normalize(style), comp = el.currentStyle ? el.currentStyle : getComputedStyle(el, null),
            prop, current = {}, start = +new Date, dur = opts.duration||200, finish = start+dur, interval,
            easing = opts.easing || function(pos){
                return (-Math.cos(pos*Math.PI)/2) + 0.5;
            };
            for(prop in target) current[prop] = parse(comp[prop]);
            interval = setInterval(function(){
                var time = +new Date, pos = time>finish ? 1 : (time-start)/dur;
                for(prop in target){
                    try{
                        el.style[prop] = target[prop].f(current[prop].v,target[prop].v,easing(pos)) + target[prop].u;
                    }catch(ex){
                        accedo.console.debug("unable to perform the animation :"+ex);
                    }
                }
                if(time>finish) {
                    clearInterval(interval);
                    opts.after && opts.after();
                    after && setTimeout(after,1);
                }
            },10);
            return interval;
        };
    })()
});
/*jslint browser: true, devel: true */
/*global accedo: false */

/**
 * <pre>
* To provide a layout properties.
* After combining previous absolute, relative, linear, normal layout, now only Normal and Linear types.
* Linear layout are items that displayed horizontally or verically.
* Other layouts are called as Normal (including the absolute/relative layout).
* And previous width,height,top,left and position (absolute/relative) setting should be set through css class.
*  e.g if you want to create a vertical linear layout of 3 children button and auto setting up those nextUp and nextDown. It will align vertically and set up those nextUp and nextDown
*  {
*    type : accedo.ui.widget.Layout,
*    layout:  accedo.ui.widget.Layout.LINEAR,
*    
*    id: "#leftMenu",
*    children : [
*        {
*           type: accedo.ui.widget.Label,
*           text: "Menu"
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonA',
*            text: 'buttonA',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonB',
*            text: 'buttonB',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonC',
*            text: 'buttonC',
*        }
*  ]
*  
*  It is similar in Layout.Grid
*  it has an extra attribute called rowSize which determine how many items on each row/col
*  For example 3x3 grid VERTICAL
*  {
*    type : accedo.ui.widget.Layout,
*    layout:  accedo.ui.widget.Layout.GRID,
*    orientation : accedo.ui.widget.Layout.VERTICAL,
*    id: "#leftMenu",*
*    children : [
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonA',
*            text: 'buttonA',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonB',
*            text: 'buttonB',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonC',
*            text: 'buttonC',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonD',
*            text: 'buttonD',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonE',
*            text: 'buttonE',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonF',
*            text: 'buttonF',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonG',
*            text: 'buttonG',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonH',
*            text: 'buttonH',
*        },
*        {
*            type: accedo.ui.widget.Button,
*            id: 'buttonI',
*            text: 'buttonI',
*        }
*  ]
* </pre>
* @name Linear
* @memberof accedo.ui.widget.Layout#
* @class This container is used to align child elements, either vertically or horizontally.
* @see accedo.ui.widget.Layout.initialize#
* @constructor
* @extends accedo.ui.Container
* @param {Object} opts The options object
* @param {Integer} [opts.orientation="accedo.ui.widget.Layout.Linear.VERTICAL"] The orientation of the layout
*/
accedo.Class.create("accedo.ui.widget.Layout", "accedo.ui.Container", {
    /**
     * This is used to indicate a vertical layout in the linear.
     * @name VERTICAL
     * @memberof accedo.ui.widget.Layout#
     * @constant
     */
    VERTICAL : 0x01,
    
    /**
     * This is used to indicate a horizontal layout in the linear.
     * @name HORIZONTAL
     * @constant
     * @memberof accedo.ui.widget.Layout#
     */
    HORIZONTAL : 0x02,
    /**
     * This is used to indicate a linear layout.
     * @name LINEAR
     * @constant
     * @memberof accedo.ui.widget.Layout#
     */
    LINEAR : 0x03,
    /**
     * This is used to indicate a normal layout which is previously absolute/relative/normal
     * @name NORMAL
     * @constant
     * @memberof accedo.ui.widget.Layout#
     */
    NORMAL : 0x04,
    /**
     * This is used to indicate a simple grid layout.
     * @name GRID
     * @constant
     * @memberof accedo.ui.widget.Layout#
     */
    GRID : 0x05
},
{	 
    /**
     * To store the row in the Grid layout
     * @name _row
     * @constant
     * @memberof accedo.ui.widget.Layout#
     */
    _row:[],

    /**
     * Initiates the linear layout
     * @memberof accedo.ui.widget.Layout#
     * @param {Object} opts The options object
     * @param {accedo.ui.widget.Layout.NORMAL | accedo.ui.widget.Layout.LINEAR} opts.layout - default is normal
     * @param {boolean} opts.autoNavigation  - (Optional)only available in Linear layout and default will be false. The usage is to auto set up the nextLeft/nextRight/nextUp/nextDown of 
     *                         the focusable elements according to the order of the children setting up in the view and those focusable elements without setting up navigation.default is false
     * @param {accedo.ui.widget.Layout.VERTICAL | accedo.ui.iwdget.Layout.HORIZONTAL } opts.orientation - (Optional)the alignment of the linear layout either vertical or horizontal default is VERTICAL
     * @param {boolean} opts.connectRows - (Optional) handles LEFT/RIGHT keys to focus on next row automatically. default: false.
     * @private
     */
    initialize: function($super,opts) {
        //default is normal
        if(!opts.layout){
            opts.layout = accedo.ui.widget.Layout.NORMAL;
        }
        switch(opts.layout){
            case accedo.ui.widget.Layout.LINEAR:
                //default autoNavigation will be true
                if(!accedo.Object.isBoolean(opts.autoNavigation)){
                    opts.autoNavigation = true;
                }
            
                //Set default orientation
                if(!opts.orientation){
                    opts.orientation = accedo.ui.widget.Layout.VERTICAL;
                }
                break;
            case accedo.ui.widget.Layout.GRID:
                //default autoNavigation will be true
                if(!accedo.Object.isBoolean(opts.autoNavigation)){
                    opts.autoNavigation = true;
                }
                
                //Set default orientation
                if(!opts.orientation){
                    opts.orientation = accedo.ui.widget.Layout.VERTICAL;
                }
                
                if(accedo.Object.isUndefined(opts.rowSize)){
                    opts.rowSize = opts.children.length;
                }
                
                break;
        }
        $super(opts);

    },
    /**
     * add a class to linear object only
     * @name createLayoutElment
     * @function
     * @memberof accedo.ui.widget.Layout#
     * @param {Object} opts the information of the layout like the type and orienataion which is set from the view
     * @protected
     * @public
     * @override
     */
    createLayoutElement: function($super, opts) {
        var div = $super();
        
        if(opts.layout === accedo.ui.widget.Layout.LINEAR){
            if (opts.orientation === accedo.ui.widget.Layout.HORIZONTAL) {
                div.addClass('accedo-ui-layout-linear-h');
            } else {
                div.addClass('accedo-ui-layout-linear-v');
            }
            return div;
        }else if(opts.layout === accedo.ui.widget.Layout.GRID){
            if (opts.orientation === accedo.ui.widget.Layout.HORIZONTAL) {
                div.addClass('accedo-ui-layout-grid-linear-h');
            } else {
                div.addClass('accedo-ui-layout-grid-linear-v');
            }	
            return div;
        }
        
        
        return $super();
        
       
    },
    /**
    * This function returns the orientation of this layout.
    * @name getOrientation
    * @see accedo.ui.widget.Layout.VERTICAL
    * @see accedo.ui.widget.Layout.HORIZONTAL
    * @public
    * @memberof accedo.ui.widget.Layout#
    * @return {Integer} The orientation
    * @function
    */
    getOrientation: function() {
        return this.opts.orientation;
    },
    /**
    * This function returns the type of layout
    * @name getLayoutType
    * @see accedo.ui.widget.Layout.NORMAL
    * @see accedo.ui.widget.Layout.LINEAR
    * @public
    * @memberof accedo.ui.widget.Layout#
    * @return {Integer} The orientation
    * @function
    */
    getLayoutType: function() {
        return this.opts.layout;
    },
    /**
     * This function attachs a child object. Overriding Container's attach 
     * function. Currently this widget only support appending childs.
     * @name attach
     * @function
     * @param {accedo.ui.AbstractComponent} child - The child component to be
     * attached
     * @memberof accedo.ui.Container#
     * @public
     * @override
     */
    attach: function($super, child) {
        return $super(child);
    },
    
    /**
    * Overriding doAttachElement function from accedo.ui.Container, called
    * at the time a new child is being added. Should handle how the child is
    * added to this container's dom elements.
    * 
    * @name doAttachElement
    * @override
    * @param {accedo.ui.AbstractComponent} child
    * @memberof accedo.ui.widget.Layout#
    * @private
    * @function
    */
    doAttachElement: function(child) {
        var div, divRow;
	
        //only need to handling appending
        if(this.opts.layout === accedo.ui.widget.Layout.LINEAR){
            
            //Create a new slot for this child
            div = accedo.Dom.element('div');

            div.addClass('accedo-ui-layout-linear-element');

            this.getRoot().appendChild(div);
            
            //Attaching child to it's parent should be after parent is in DOM, or focus management will not work well.
            child.attachTo(this,div);

        }else if( this.opts.layout === accedo.ui.widget.Layout.GRID){
            //check the number of children to be added
            if( (this.getChildren().length-1) % this.opts.rowSize === 0){
                divRow = accedo.Dom.element('div').addClass('accedo-ui-layout-grid-linear-'+(this.opts.orientation === accedo.ui.widget.Layout.HORIZONTAL?'h':'v')+'-row row-'+this._row.length);
                this._row.push(divRow);
                this.getRoot().appendChild(divRow);
            }

            //Create a new slot for this child
            div = accedo.Dom.element('div');

            div.addClass('accedo-ui-layout-grid-linear-element');

            this._row.slice(-1)[0].appendChild(div);

            //Attaching child to it's parent should be after parent is in DOM, or focus amnagement will not work well.
            child.attachTo(this,div);
            
        }else{
            child.attachTo(this, this.getRoot());
        } 

    },
    remove: function($super, child) {
        var colEle = null, rowEle = null, rowIdx,
        idx = accedo.Array.indexOf(this._children, child),
        rowCount, row, justifyItem, i;
        
        if(this.opts.layout === accedo.ui.widget.Layout.LINEAR){
            
            colEle = child.getRoot().getParent();
            
            $super(child);
            
            colEle.remove();
            
        }
        else if (this.opts.layout === accedo.ui.widget.Layout.GRID) {
            
            rowIdx = Math.floor(idx / this.opts.rowSize);
            rowCount = this._row.length;
            
            colEle = child.getRoot().getParent();
            $super(child);
            colEle.remove();
            
            for (i=rowIdx+1; i<rowCount; i++) {
                
                row = this._row[i];
                rowEle = row.getHTMLElement();
                    
                //remove from front of next row
                justifyItem = rowEle.firstChild;

                row.removeChild(justifyItem);

                //append to previous row
                this._row[i-1].appendChild(justifyItem);
                
                 
            }
            
            //remove last row if empty
            if (i === rowCount && !this._row[i-1].getHTMLElement().hasChildNodes()) {
                this._row[i-1].remove();
                this._row.splice(i-1,1);
                
            }
            
        }
        else {
            $super(child);
        }
        
    },
    /**
    * To set the default navigation and only called when first initialization and only for linear layout.
    * @name setDefaultNavigation
    * @private
    * @memberof accedo.ui.widget.Layout#
    * @function
    */
    onKey:function($super,vKey,currentFocus){
        var fm, fmRet = false;
        $super(vKey, currentFocus);
        
        if(this.opts.layout !== accedo.ui.widget.Layout.NORMAL){
            fm = accedo.ui.FocusManager.singleton();
            fmRet = fm.focusDirectionChangeByKey(vKey, this);
        
            if(fmRet){
                return true; 
            }
        }
        var parent, currentIndex, nextUpLeft, nextDownRight, nextLeftUp, nextRightDown;
        if(!(this.opts.autoNavigation && currentFocus)){
            return false;
        }
            
        parent = currentFocus.getParent();
        //the parent isn't currentFocus parent
        while(parent){
            if(parent === this){
                break;
            }
            currentFocus = parent;
            parent = currentFocus.getParent();
        }
            
        if(!parent){
            return false;
        }
            
        currentIndex = parent.getChildren().indexOf(currentFocus);
			
        switch(this.opts.layout){
            case accedo.ui.widget.Layout.LINEAR:
                switch(this.opts.orientation){
                    case accedo.ui.widget.Layout.VERTICAL:
                        nextUpLeft = accedo.VKey.KEY_UP;
                        nextDownRight = accedo.VKey.KEY_DOWN;
                        break;
                    case accedo.ui.widget.Layout.HORIZONTAL:
                        nextUpLeft = accedo.VKey.KEY_LEFT;
                        nextDownRight = accedo.VKey.KEY_RIGHT;
                        break;
                }
                if(vKey === nextUpLeft ){
                    return this._prevChildNav(parent,currentIndex);
                }else if(vKey === nextDownRight ){
                    return this._nextChildNav(parent,currentIndex);
                }   
                break;
            case accedo.ui.widget.Layout.GRID:
                switch(this.opts.orientation){
                    case accedo.ui.widget.Layout.VERTICAL:
                        nextUpLeft = accedo.VKey.KEY_UP;
                        nextDownRight = accedo.VKey.KEY_DOWN;
                        nextLeftUp = accedo.VKey.KEY_LEFT;
                        nextRightDown = accedo.VKey.KEY_RIGHT;
                        break;
                    case accedo.ui.widget.Layout.HORIZONTAL:
                        nextUpLeft = accedo.VKey.KEY_LEFT;
                        nextDownRight = accedo.VKey.KEY_RIGHT;
                        nextLeftUp = accedo.VKey.KEY_UP;
                        nextRightDown = accedo.VKey.KEY_DOWN;
                        break;
                }
                switch(vKey){
                    case nextLeftUp:
                        return this._prevChildNav(parent,currentIndex);
                    case nextRightDown:
                        return this._nextChildNav(parent,currentIndex);
                    case nextUpLeft:
                        if(currentIndex < this.opts.rowSize){
                            return false;
                        }else{
                            return this._prevRowChildNav(parent,currentIndex);
                        }
				
                        break;
                    case nextDownRight:
                        if( currentIndex >= this.opts.rowSize * (this._row.length - 1) ){
                            return false;
                        }else{
                            return this._nextRowChildNav(parent,currentIndex);
                        }
                        break;
                }			
        }
        
        return false;
    },
    /**
    * Get the next Row Last Child for autoNavigation
    * @memberof accedo.ui.widget.Layout#
    * @private
    * @return {boolean} true if the that child is focusable and exist
    * @override
    * @function _nextChildNav
    */
    _nextRowLastChild:function(parent,currentIndex){
        var nextChild, i, j, row, col;
        col = currentIndex % this.opts.rowSize;
        row = (currentIndex>0)?Math.floor(currentIndex / this.opts.rowSize):0;
        switch(this.opts.orientation){
            case accedo.ui.widget.Layout.VERTICAL:
                for( i = this.opts.rowSize-1; i >=0 ; i--){
                    nextChild = parent.getChildren()[(row+1)*this.opts.rowSize+i];
                    if(nextChild && nextChild.isFocusable()){
                        nextChild.setFocus();
                        return true;
                    }
                }
                break;
            case accedo.ui.widget.Layout.HORIZONTAL:
                for( i = row-1; i >=0 ; i--){
                    nextChild = parent.getChildren()[i*this.opts.rowSize+col+1];
                    if(nextChild && nextChild.isFocusable()){
                        nextChild.setFocus();
                        return true;
                    }
                }
                break;
        }
        return false;
    },
    /**
    * Get the next Child for autoNavigation
    * @memberof accedo.ui.widget.Layout#
    * @private
    * @return {boolean} true if the that child is focusable and exist
    * @override
    * @function _nextChildNav
    */
    _nextChildNav:function(parent,currentIndex){
        var nextChild, i, j,
        isList = this.opts.layout === accedo.ui.widget.Layout.LINEAR, 
        connectRows = this.opts.connectRows;
        
        for( i = currentIndex, j = parent.getChildren().length; i < j && (isList || connectRows || i % this.opts.rowSize < this.opts.rowSize - 1) ; i++){
            nextChild = parent.getChildren()[i+1];
            if(nextChild && nextChild.isFocusable()){
                nextChild.setFocus();
                return true;
            }
        }
        
        return false;
    },
    /**
    * Get the previous Child for autoNavigation
    * @memberof accedo.ui.widget.Layout#
    * @private
    * @return {boolean} true if the that child is focusable and exist
    * @override
    * @function _prevChildNav
    */
    _prevChildNav:function(parent,currentIndex){
        var prevChild, i, 
        isList = this.opts.layout === accedo.ui.widget.Layout.LINEAR, 
        connectRows = this.opts.connectRows;
        
        for( i = currentIndex ; i > -1 && (isList || connectRows || i % this.opts.rowSize > 0); i--){
            prevChild = parent.getChildren()[i-1];
            if(prevChild && prevChild.isFocusable()){
                prevChild.setFocus();
                return true;
            }
        }
        return false;
    },
    /**
    * Get the next row Child for autoNavigation
    * @memberof accedo.ui.widget.Layout#
    * @private
    * @return {boolean} true if the that child is focusable and exist
    * @override
    * @function _nextRowChildNav
    */
    _nextRowChildNav:function(parent,currentIndex){
        var nextChild, i, j;
        for( i = 1, j = parent.getChildren().length; currentIndex + i * this.opts.rowSize < j ; i++){
            nextChild = parent.getChildren()[currentIndex + i * this.opts.rowSize];
            if(nextChild && nextChild.isFocusable()){
                nextChild.setFocus();
                return true;
            }
        }
        return this._nextRowLastChild(parent, currentIndex);
    },
    /**
    * Get the previous row Child for autoNavigation
    * @memberof accedo.ui.widget.Layout#
    * @private
    * @return {boolean} true if the that child is focusable and exist
    * @override
    * @function _prevRowChildNav
    */
    _prevRowChildNav:function(parent,currentIndex){
        var prevChild, i;
        for(i = 1; currentIndex - (i * this.opts.rowSize) >= 0 ; i++){
            prevChild = parent.getChildren()[currentIndex - (i * this.opts.rowSize)];
            if(prevChild && prevChild.isFocusable()){
                prevChild.setFocus();
                return true;
            }
        }
        return false;
    },
    /**
    * deinit function
    * @name deinit
    * @memberof accedo.ui.widget.Layout#
    * @private
    * @override
    * @function
    */
    deinit:function($super){
        $super();
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Creates a new label instance.
 * @name Label
 * @memberof accedo.ui.widget
 * @class A UI Label object, providing basic functionality.
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @extends accedo.ui.AbstractComponent
 */
accedo.Class.create("accedo.ui.widget.Label", "accedo.ui.AbstractComponent", {},
{
    /**
     * @constructor
     * @memberof accedo.ui.widget.Label#
     * @param {Object} opts The options object
     * @param {String} opts.text The text label
     * @private
     */
    initialize: function($super,opts) {
        /**
         * Internal storage for label text.
         * @field
         * @private
         */
        opts.text = opts.text || '';

        opts.root = accedo.Element.create('div');
        opts.root.addClass('accedo-ui-label');
 
        $super(opts);
        
        this.setText(opts.text);
    },
    
    /**
     * Sets the text of the label.
     * @name setText
     * @function
     * @param {String} text The text to set
     * @param {boolean} pureText true will set the innerText/ false set innerHTML (default is false)
     * @memberof accedo.ui.widget.Label#
     * @public
     */
    setText: function(text, pureText) {
        var isPureText = pureText || false;
        //Use this.getRoot() instead of opts.root as it seems there are cases that opts.root will be changed by the other elements
        this.getRoot().setText(text, isPureText);
    },

    /**
     * Gets the text of the label.
     * @name getText
     * @function
     * @memberof accedo.ui.widget.Label#
     * @public
     * @return {String} Text of the label
     */
    getText: function() {
        return this.getRoot().getText();
    }
});

/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Creates a new button instance.
 * Can dispatch "click" event
 * @name Button
 * @memberof accedo.ui.widget
 * @class A button class, providing basic functionality.
 * @author <a href="mailto:thomas.johansson@accedobroadband.com">Thomas Johansson</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create("accedo.ui.widget.Button", "accedo.ui.Container", ["accedo.ui.widget.Label"], function(){
    return {
        /**
        * Event: click
        * @constant
        * @name EVT_CLICK
        * @memberof accedo.ui.widget.Button
        * @deprecated  moved to accedo.ui.Evt.CLICK
        */
        EVT_CLICK: accedo.ui.Evt.CLICK
    };
},
{
    /**
     * @private
     * @name _label
     * @memberof accedo.ui.widget.Button#
     */
    _label: null,
    
    /**
     * @constructor
     * @memberof accedo.ui.widget.Button#
     * @param {Object} opts The options object
     * @param {String} opts.text The text label to use for the button
     * @private
     */
    initialize: function($super,opts) {
        
        opts.text = opts.text || '';
        opts.focusable = true; //Explicit definition of object being focusable
        opts.clickable = true; //Explicit definition of object being clickable
        opts.enterAsClick = true; //Treats enter key as click
        
        $super(opts);

        this.root.addClass('accedo-ui-button');
        
        this._label = new accedo.ui.widget.Label({
            parent: this
        });
        
        this.setText(opts.text);
    },
    /**
     * This function is called by the container function whenever
     * a child element has been appended.
     * @name doAttachElement
     * @override
     * @function
     * @memberof accedo.ui.widget.Button#
     * @param {accedo.ui.AbstractComponent} child
     * @public
     */
    doAttachElement: function(child){
        this.root.appendChild(child.getRoot());
    },
    /**
     * This function sets the text label of the button.
     * @name setText
     * @function
     * @param {String} text The text label
     * @memberof accedo.ui.widget.Button#
     * @public
     */
    setText: function(text) {
        this._label.setText(text || '');
    },
            
    /**
     * Return the inner label (i.e. so as to access its DOM element)
     * @name getLabel
     * @returns {accedo.ui.widget.Label}
     * @memberof accedo.ui.widget.Button#
     * @public
     */
    getLabel: function() {
        return this._label;
    },
            
    /**
     * Return the inner label (i.e. so as to access its DOM element)
     * DO NOT USE IT ANY MORE! Use 'getLabel'. THis one is just here for compatibility, should be removed later.
     * @returns {accedo.ui.widget.Label}
     * @deprecated
     * @public
     * @memberof accedo.ui.widget.Button#
     * @name getText
     */
    getText: function() {
        return this._label.getText();
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Creates a new checkbox instance.
 * Can dispatch "click" event
 * @name Checkbox
 * @memberof accedo.ui.widget
 * @class A checkbox input control.
 * @author <a href="mailto:daniel.deng@accedo.tv">Daniel Deng</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create("accedo.ui.widget.Checkbox", "accedo.ui.Container", ["accedo.ui.widget.Label"], function(){
    return {
        /**
         * Event: click
         * @constant
         * @name EVT_CLICK
         * @memberof accedo.ui.widget.Checkbox
         * @deprecated  moved to accedo.ui.Evt.CLICK
         */
        EVT_CLICK: accedo.ui.Evt.CLICK
    };
},
{
    /**
     * @private
     * @name _value
     * @memberof accedo.ui.widget.Checkbox#
     */
    _value: false,
    /**
     * @private
     * @name _image
     * @memberof accedo.ui.widget.Checkbox#
     */
    _image: null,

    /**
     * @constructor
     * @memberof accedo.ui.widget.Checkbox#
     * @param {Object} opts The options object
     * @param {String} opts.text The text label to use for the button
     * @private
     */
    initialize: function($super,opts) {
        
        opts.focusable = true; //Explicit definition of object being focusable
        opts.clickable = true; //Explicit definition of object being clickable
        opts.enterAsClick = true; //Treats enter key as click
        
        $super(opts);

        this.root.addClass('accedo-ui-checkbox');

        this._value = !!opts.checked;
        this._image = accedo.Element.create("span");
        if (!!this._value) {
            this._image.addClass('checked');
        }
        this.root.appendChild(this._image);
    },
    /**
     * Handles click event.
     * @name onClick
     * @override
     * @function
     * @memberof accedo.ui.widget.Checkbox#
     * @protected
     * @public
     */
    onClick: function($super){
        if (!$super()) { // parent class has problem handling the click, so probably "clickable" is turned off
            return false;
        }
        this.toggle();
        return true;
    },
    /**
     * Toggles the checkbox status
     * @name toggle
     * @function
     * @memberof accedo.ui.widget.Checkbox#
     * @public
     */
    toggle: function(){
        this.check(!this._value);
    },
    /**
     * Checks the checkbox if no param, or sets the checkbox to a specified status with param
     * @name check
     * @function
     * @memberof accedo.ui.widget.Checkbox#
     * @param {Boolean} isChecked (optional) set the checkbox to this status
     * @public
     */
    check: function(isChecked){

        var toCheck = accedo.Object.isUndefined(isChecked) || !!isChecked;

        if (toCheck && !this._value) {
            this._value = toCheck;
            this._image.addClass("checked");
            this.dispatchEvent(accedo.ui.Evt.VALUE_CHANGED, this._value);
        } else if (!toCheck && this._value) {
            this._value = toCheck;
            this._image.removeClass("checked");
            this.dispatchEvent(accedo.ui.Evt.VALUE_CHANGED, this._value);
        }
    },
    /**
     * Unchecks the checkbox.
     * @name uncheck
     * @function
     * @memberof accedo.ui.widget.Checkbox#
     * @public
     */
    uncheck: function(){
        this.check(false);
    },
    /**
     * Gets the checkbox checked status
     * @name getValue
     * @function
     * @returns {Boolean} whether this checkbox is checked
     * @memberof accedo.ui.widget.Checkbox#
     * @public
     */
    getValue: function(){
        return this._value;
    },
    /**
     * This function is called by the container function whenever
     * a child element has been appended.
     * @name doAttachElement
     * @override
     * @function
     * @memberof accedo.ui.widget.Checkbox#
     * @param {accedo.ui.AbstractComponent} child
     * @public
     */
    doAttachElement: function(child){
        this.root.appendChild(child.getRoot());
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Creates a new image instance.
 * divImg attribute (boolean) true will set div background, default will be false and use image tag
 * detachLoad attribute (boolean) false will stop to load the image when detach from dom, default will be false. true will keep load the image.
 * @name Image
 * @memberof accedo.ui.widget
 * @class An image widget implementation, with function for working with Image components.
 * @author <a href="mailto:joe.chang@accedobroadband.com">Joe Chang</a>
 * @extends accedo.ui.AbstractComponent
 */
accedo.Class.create("accedo.ui.widget.Image", "accedo.ui.AbstractComponent", {},
{
    /**
    * To store the url of the image
    * @name _url
    * @memberof accedo.ui.widget.Image#
    * @private
    */
    _url:null,
    /**
    * To save the error callback
    * @name _errorCallback
    * @memberof accedo.ui.widget.Image#
    * @private
    */
    _errorCallback :null,
    /**
    * To save the error callback
    * @name _loadedCallback
    * @memberof accedo.ui.widget.Image#
    * @private
    */
    _loadedCallback :null,
    /**
    * To save the number of error called after setting a new source
    * @name _counter
    * @memberof accedo.ui.widget.Image#
    * @private
    */
    _counter:0,
    /**
    * @constructor
    * @memberof accedo.ui.widget.Image#
    * @param {Object} opts The options object
    * @param {String} opts.src The source of the image
    * @param {String} opts.errorImg The error image when the image is unable to load (only applicable to image tag)
    * @param {String} opts.divImg To tell whether this image widget use image tag or div tag to display the image
    * @private
    */
    initialize: function($super,opts) {
//        if(opts.onload){this.setLoadedCallback(opts.onload);}
//        if(opts.onerror){this.setErrorCallback(opts.onerror);}
        opts.src  = opts.src || 'data:image/gif;base64,R0lGODlhAQABAID/AMDAwAAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        opts.alt = opts.alt || '';
        
        this._url = opts.src;
        opts.detachLoad = opts.detachLoad || false;
        if (opts.divImg === true) {
            opts.root = accedo.Element.create('div');
            opts.root.setStyle({
                "background":"url("+ opts.src +")"
            });
        } else {
            opts.root = accedo.Element.create('img', {
                'src': opts.src
            });
        }

        $super(opts);
		
        //to make it become default error callback
        this.resetErrorCallback();
    },
    /**
     * To dispatch the event when it is detached to the dom
     * @name dispatchAttachedToDOMEvent
     * @function
     * @memberof accedo.ui.widget.Image#
     * @protected
     * @override
     * @public
     */
    dispatchAttachedToDOMEvent: function($super) {
        this._onAttachedDOM();
        $super();
    },
    /**
     * To dispatch the event when it is attached to the dom
     * @name dispatchDetachFromDOMEvent
     * @function
     * @memberof accedo.ui.widget.Image#
     * @protected
     * @override
     * @public
     */
    dispatchDetachFromDOMEvent: function($super) {
        $super();
        this._onDetachedFromDOM();
    },
    /**
     * to add the src when the image is attached
     * @name _onAttachedDOM
     * @function
     * @memberof accedo.ui.widget.Image#
     * @private
     */
    _onAttachedDOM:function(){
        // no need to remove src when using div
        if(this.opts.divImg){
            return;
        }
        this.getRoot().setAttributes({
            'src': this._url
        });
    },
    /**
     * to remove the src and prevent the image from preload when detached
     * @name onDetachedFromDOM
     * @function
     * @memberof accedo.ui.widget.Image#
     * @private
     */
    _onDetachedFromDOM:function(){
        //only remove src when detachLoad is true
        if(this.opts.detachLoad){
            return;
        }
        // no need to remove src when using div
        if(this.opts.divImg){
            return;
        }
        this.getRoot().removeAttribute('src');
    },
    /**
     * Sets the source of an image object.
     * @name setSrc
     * @function
     * @param {String} src The new image source
     * @memberof accedo.ui.widget.Image#
     * @public
     */
    setSrc: function(src) {
        this._url = src;
        if (this.opts.divImg === true)
        {
            this.getRoot().setStyle({
                "background":"url("+ src +")"
            });
        } else {
            this.getRoot().setAttributes({
                'src': src
            });
            this._counter = 0;
        }
    },
    /**
     * gets the source of an image object.
     * @name getSrc
     * @function
     * @retrun {String}  The image src
     * @memberof accedo.ui.widget.Image#
     * @public
     */
    getSrc:function(){
        return this._url;
    },
    /**
     * Sets alternative text for an image object.
     * @name setAlt
     * @function
     * @param {String} alt New Alternative text
     * @memberof accedo.ui.widget.Image#
     * @public
     */
    setAlt: function(alt) {
        this.opts.alt = alt;
        this.getRoot().setAttributes({
            'alt': alt
        });
    },
    /**
     * gets the alt of an image object.
     * @name getAlt
     * @function
     * @retrun {String}  The image alt
     * @memberof accedo.ui.widget.Image#
     * @public
     */
    getAlt:function(){
        return this.opts.alt;
    },
    /**
     * To set the error callback when the image is unable to load
     * @name setErrorCallback
     * @function
     * @memberof accedo.ui.widget.Image#
     * @public
     */
    setErrorCallback:function(cb){
        //only apply to non div img
        if(this.opts.divImg){
            return;
        }
        
        if(cb && accedo.Object.isFunction(cb)){
            this._errorCallback = cb;
            this.getRoot().getHTMLElement().onerror = this._errorCallback;
        }
    },
    /**
     * To set the loaded callback when the image is load
     * @name setLoadedCallback
     * @function
     * @memberof accedo.ui.widget.Image#
     * @public
     */
    setLoadedCallback:function(cb){
        //only apply to non div img
        if(this.opts.divImg){
            return;
        }

        if(cb && accedo.Object.isFunction(cb)){
            this._loadedCallback = cb;
            this.getRoot().getHTMLElement().onload = this._loadedCallback;
        }
    },
    /**
     * To reset/remove the error callback 
     * @name resetErrorCallback
     * @function
     * @memberof accedo.ui.widget.Image#
     * @public
     */
    resetErrorCallback:function(){
        //only apply to non div img
        if(this.opts.divImg){
            return;
        }
        
        //to rest the counter to 0 
        this._counter = 0;
        this._errorCallback = null;
        if(this.opts.errorImg){
            this.getRoot().getHTMLElement().onerror = accedo.Fn.bind(function(){
                
                //to avoid setting errorImg and error again which cause deadlock, so just set once when failure.
                if(!this._counter){
                    this.getRoot().setAttributes({
                        'src': this.opts.errorImg
                    });
                }
                this._counter++;
            },this);
        }else{
            this.getRoot().getHTMLElement().onerror = null;
        }
    },
    /**
     * To handle the dynamic set error image
     * @name setOption
     * @function
	 * @override
     * @memberof accedo.ui.widget.Image#
     * @protected
     */
    setOption:function($super, key, value){
        var ret = $super(key, value);
        if(key === "errorImg"){
            //only set the errorImg when there are no error Callback
            if(!this._errorCallback){
                this.resetErrorCallback();
            }
        }
        return ret;
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Creates a new input field instance.
 * @name InputField
 * @memberof accedo.ui.widget
 * @class An input field, providing basic functionality. It inherits from accedo.ui.button class
 * @param {Object} opts The options object
 * @param {String} opts.text The pre-set text for the input field
 * @param {String} opts.isPassword Whether this input field holds a password
 * @param {String} opts.maxLength The maximum length for the input text
 * @param {String} opts.hideTimeout The duration to briefly show the typed character, in seconds, set to 0 if should not show the typed character
 * @param {String} opts.passwordChar The character to display, as a masked password character
 * @author <a href="mailto:ming.hsieh@accedobroadband.com">Ming Hsieh</a>
 * @example var inputField = accedo.ui.inputField({text:'plz input here', isPassword:false, maxLength:20});
 * @author <a href="mailto:curtiss.yeung@accedo.tv">Curtiss Yeung</a>
 * @extends accedo.ui.widget.Button
 */
accedo.Class.create("accedo.ui.widget.InputField", "accedo.ui.AbstractComponent", {},
{
    /**
     * @private
     * @name _inputFieldWidth
     * @memberof accedo.ui.widget.InputField#
     */
    _inputFieldWidth: null,
    /**
     * @private
     * @name _wrapper
     * @memberof accedo.ui.widget.InputField#
     */
    _wrapper: null,
    /**
     * @private
     * @name _wrapperCopy
     * @memberof accedo.ui.widget.InputField#
     */
    _wrapperCopy: null,
    /**
     * @private
     * @name _cursor
     * @memberof accedo.ui.widget.InputField#
     */
    _cursor: null,
    /**
     * @private
     * @name _cursorPosition
     * @memberof accedo.ui.widget.InputField#
     */
    _cursorPosition: null,
    /**
     * @private
     * @name _cursorLeftPosition
     * @memberof accedo.ui.widget.InputField#
     */
    _cursorLeftPosition: null,
    /**
     * @private
     * @name _labelText
     * @memberof accedo.ui.widget.InputField#
     */
    _labelText: null,
    /**
     * @private
     * @name _labelTextCopy
     * @memberof accedo.ui.widget.InputField#
     */
    _labelTextCopy: null,
    /**
     * @private
     * @name _wrapperLeft
     * @memberof accedo.ui.widget.InputField#
     */
    _wrapperLeft: null,
    /**
     * @private
     * @name _maxLength
     * @memberof accedo.ui.widget.InputField#
     */
    _maxLength: null,
    /**
     * @private
     * @name _inputState
     * @memberof accedo.ui.widget.InputField#
     */
    _inputState: false,
    /**
     * @private
     * @name _passwordTimer
     * @memberof accedo.ui.widget.InputField#
     */
    _passwordTimer: null,
    /**
     * @private
     * @name _hideTimeout
     * @memberof accedo.ui.widget.InputField#
     */
    _hideTimeout: 0,
    
    
    /**
     * @constructor
     * @memberof accedo.ui.widget.InputField#
     * @private
     */
    initialize: function($super,opts) {
        this._maxLength = opts.maxLength || 20;
        this.isPassword = opts.isPassword || false;
        this._labelText = opts.text || '';
        this.passwordChar = opts.passwordChar || '*';
        this._hideTimeout = opts.hideTimeout || 0;
        this._cursorPosition = this._labelText.length;
        
        
        opts.focusable = opts.focusable || false;
        opts.clickable = opts.clickable || true; // default clickable
        opts.css= (!opts.css)?'accedo-ui-inputfield':opts.css+' accedo-ui-inputfield';
        opts.root = accedo.Element.create("div");
        
        $super(opts);
        
        // Create div structure
        this.container = accedo.Element.create("div");
        this.container.addClass("container");
        
        this._wrapper = accedo.Element.create("pre");
        this._wrapper.addClass("wrapper");
        
        this._wrapperCopy = accedo.Element.create("pre");
        this._wrapperCopy.addClass("wrapperCopy");
        
        this._cursor = accedo.Element.create("div");
        this._cursor.addClass("cursor");
        
        this.root.appendChild(this.container);
        this.container.appendChild(this._wrapper);
        this.container.appendChild(this._wrapperCopy);
        this.container.appendChild(this._cursor);
        
        this.setText(opts.text, true);
    },
    
    /**
     * Append character to the input field
     * @name appendChar
     * @function
     * @param {String} ch The character to append to
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    appendChar: function(ch) {
        this.insertChar(ch, this._labelText.length);
    },
    /**
     * Insert character to the input field
     * @name insertChar
     * @function
     * @param {String} ch The character to insert to
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    insertChar: function(ch, insertPos) {
        if (!this._inputState) {
            return;
        }
        if (this._labelText.length >= this._maxLength) { // If length of _labelText reach maximum, do nothing, and return
            return;
        }
        
        insertPos = accedo.Object.isUndefined(insertPos) ? this._cursorPosition : insertPos;
        var t = this.getText();
        t = accedo.String.insert(t, insertPos, ch);
        this._cursorPosition = this._cursorPosition + ch.length;
        this.setText(t, undefined, true);
        this.updateLeft('add');
        this.updateCursorPosition();
    },
    /**
     * Delete the last character in the input field
     * @name backspace
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    backspace: function() {
        if (!this._inputState) {
            return;
        }
        if (this._cursorPosition <= 0) { // If cursorPosition == 0, there must be no char before cursor, do nothing, and return
            return;
        }

        var t = this.getText(), newT = "";
        if (this._cursorPosition === 1) { // Exception case when delete the first char while the string is have more than 1 char
            if(t.length > 1) {
                newT = t.slice(this._cursorPosition, t.length);
            }
        } else {
            if(t.length > 1) {
                newT = t.slice(0, this._cursorPosition - 1) + t.slice(this._cursorPosition, t.length);
            }
        }
        this._cursorPosition--;
        this.setText(newT, undefined, true);
        this.updateLeft('del');
        this.updateCursorPosition();
    },
    /**
     * Delete the last character in the input field
     * @name deleteChar
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     * @deprecated
     */
    deleteChar: function() {
        this.backspace();
    },
    /**
     * Delete the character after cursor in the input field
     * @name del
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    del: function() {
        if (!this._inputState) {
            return;
        }
        var t = this.getText(), newT = "";
        if (this._cursorPosition >= t.length) { // If _cursorPosition == t.length, there must be no char after cursor, do nothing, and return
            return;
        }
        
        if (this._cursorPosition === t.length - 1) { // Exception case when delete the last char
            newT = t.slice(0, t.length - 1);
        } else {
            newT = t.slice(0, this._cursorPosition) + t.slice(this._cursorPosition + 1, t.length);
        }
        this.setText(newT, undefined, true);
    },
    /**
     * Handles a key by checking if it is a valid text editing input.
     * Updates text and return true if true, return false otherwise.
     * @name onKey
     * @function
     * @param {accedo.VKey} vkey The key to handle
     * @return {Boolean} whether the key has been handled
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    onKey: function(vKey) {
        if (!this.isInput()) {
            return false;
        }
        if (vKey.length === 1) {
            this.insertChar(vKey);
            return true;
        }
        switch(vKey){
            case accedo.VKey.KEY_KEYBOARD_BACKSPACE:
                this.backspace();
                return true;
            case accedo.VKey.KEY_KEYBOARD_DELETE:
                this.del();
                return true;
            default:
                break;
        }
        return false;
    },
    /**
     * Set max length of input field
     * @name setMaxLength
     * @function
     * @param {Integer} l The number of characters to set
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    setMaxLength: function(l) {
        this._maxLength = l;
    },
    /**
     * Get the input field text
     * @name getText
     * @function
     * @return {String} _labelText Return the current label text
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    getText: function() {
        return this._labelText;
    },
    /**
     * Set text on the input field
     * @name setText
     * @function
     * @param {String} text
     * @param {Boolean} forceNoTimeout
     * @param {Boolean} noUpdateCursorPosition true won't update the cursor position after the setText, default is false and will update the cursor position to the end when setText
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    setText: function(text, forceNoTimeout, noUpdateCursorPosition) {

        // incorrect input
        if (accedo.Object.isUndefined(text)) {
            return;
        }
        
        if(!text) {
            text = "";
        }
        
        var textChanged = (text !== this._labelText),len,t="",t2="",i,wrapperCopyText,
        isAddition, useTimeout, hideTextFn;
        if(text.length > this._maxLength){
            this._cursorPosition = this._maxLength;
        }else if(!noUpdateCursorPosition){
            this._cursorPosition = text.length;
        }
        text = accedo.String.truncate(text, this._maxLength, "");
        if (this.isPassword) {
            isAddition = text.length > this._labelText.length; // we are having more characters or not
            useTimeout = (this._hideTimeout > 0) && isAddition && !forceNoTimeout; // do we briefly show the last character before cursor or not

            this._labelText = text;
            len = this._labelText.length;

            if (this._passwordTimer) {
                clearTimeout(this._passwordTimer);
            }

            for(i = 0; i < this._cursorPosition - 1; i++){ // leave out the last character before cursor position
                t += this.passwordChar;
                t2 += this.passwordChar;
            }

            if(!useTimeout && this._cursorPosition > 0){
                t += this.passwordChar; // directly hide that last typed character as '*''
                t2 += this.passwordChar;
            } else {
                t += text[this._cursorPosition-1]; // show the last typed character
                t2 += text[this._cursorPosition-1];
            }
            
            for(i = 0; i < len-this._cursorPosition; i++){ // fill the remaining characters
                t += this.passwordChar;
            }

            this._setElementText(this._wrapper,t);
            this._setElementText(this._wrapperCopy,t2);

            hideTextFn = accedo.Fn.bind(function(){ // hide
                t = '';
                t2 = '';
                for(i = 0; i < len; i++){
                    t += this.passwordChar;
                }
                for(i = 0; i < this._cursorPosition; i++){
                    t2 += this.passwordChar;
                }

                this._setElementText(this._wrapper,t);
                this._setElementText(this._wrapperCopy,t2);
                this.updateCursorPosition();
                this._passwordTimer = null;
            }, this);

            if(useTimeout){
                this._passwordTimer = accedo.Fn.delay(hideTextFn, this._hideTimeout);
            } else {
                hideTextFn();
            }

        } else {
            this._setElementText(this._wrapper,text);
            if (this._cursorPosition === 0) {
                this._setElementText(this._wrapperCopy,"");
            } else {
                wrapperCopyText = accedo.String.truncate(text, this._cursorPosition, "");
                this._setElementText(this._wrapperCopy,wrapperCopyText);
            }
        }
        
        this._labelText = text;
        this._labelTextCopy = accedo.String.truncate(this._labelText, this._cursorPosition, "");
        if(!noUpdateCursorPosition){
            this.updateCursorPosition();
        }
        if(textChanged){
            this.dispatchEvent(accedo.ui.Evt.TEXT_CHANGED,this._labelText);
        }
    },
    /**
     * Sets the text to one of the elements
     * @name _setElementText
     * @param {accedo.element#} element Element to set text to
     * @param {String} text Text to set
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @private
     */
    _setElementText:function(element, text){
        var tailingSpan;
        element.setText(text, true);
        //use a span to detect workaround for the first trailing space issue
        if (text.length > 0 && element === this._wrapperCopy && accedo.platform==="samsung" && accedo.device.appengine.getLayoutEngine().name==="Gecko") {
            tailingSpan = accedo.Element.create('span');
            tailingSpan.setText("t");
            element.appendChild(tailingSpan);
        }
    },
    /**
     * Gets the text from one of the elements
     * @name _getElementText
     * @param {accedo.element#} element Element to get text from
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @private
     */
    _getElementText:function(element){
        return element.getText();
    },
    /**
     * Clear input field text
     * @name clearText
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    clearText: function() {
        this.setText("");
    },
    /**
     * Set selection on the input field
     * @name setSelection
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    setSelection: function()
    {
        this.root.addClass("selected");
    },
    /**
     * Remove selection on the input field
     * @name removeSelection
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    removeSelection: function()
    {
        this.root.removeClass("selected");
    },
    /**
     * Check if input field is selected
     * @name isSelected
     * @function
     * @return {Boolean} true if selected; false otherwise
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    isSelected: function()
    {
        return this.root.hasClass("selected");
    },
    /**
     * Check if input field is focused
     * @name isFocused
     * @function
     * @return {Boolean} true if focused; false otherwise
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    isFocused: function()
    {
        return this.root.hasClass("focused");
    },
    
    /**
     * Update the left position of both this.wrapper and this.wrapperCopy
     * @name updateLeft
     * @function
     * @param {String} method 'add' to indicated this function triggered by insertChar(), and 'del' indicate this function triggered by deleteChar()
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    updateLeft: function(method)
    {   
        this.wrapperWidth = this._wrapper.getHTMLElement().clientWidth;
        this._wrapperLeft = this._wrapper.getHTMLElement().offsetLeft;
        this.wrapperCopyWidth = this._wrapperCopy.getHTMLElement().clientWidth;
        this.wrapperCopyLeft = this._wrapperCopy.getHTMLElement().offsetLeft;
        var leftPos,wrapperCopyEle,wrapperCopyEndSpan;
        if (accedo.platform==="samsung" && accedo.device.appengine.getLayoutEngine().name==="Gecko") {
            if (this._cursorPosition>0) {
                wrapperCopyEle = this._wrapperCopy.getHTMLElement();
                wrapperCopyEndSpan = wrapperCopyEle.lastChild;
                this.wrapperCopyWidth -= wrapperCopyEndSpan.clientWidth;
            }
        }
        
        // Perform checking to ensure the cursor is within the inputField visually
        if ((this.wrapperWidth >= this._inputFieldWidth) && (this._cursorPosition === this._labelText.length)) {
            // The cursor on the end, the leftPos of wrapper should be equal to ([InputField.Width] - [Wrapper.Width])
            // in order not to have empty space at the end of inputField, and leftPos must be a negative number.
            leftPos = this._inputFieldWidth - this.wrapperWidth;
        } else if (this.wrapperWidth >= this._inputFieldWidth) {
            // There must be some charcter outside the inputField visually
            if (method === 'del') {
                if ((this.wrapperWidth + this._wrapperLeft) <= this._inputFieldWidth) {
                    leftPos = this._inputFieldWidth - this.wrapperWidth;
                }
            }
        } else {
            // The text is within the width of the InputField, set leftPos to 0
            leftPos = 0;
        }
        
        if (!accedo.Object.isUndefined(leftPos)) {
            this.updateLeftByCursor(leftPos);
        }
    },
    
    /**
     * Update left position by moving cursor
     * @name updateLeftByCursor
     * @function
     * @param {Float} leftPos desired cursor leftPos
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    updateLeftByCursor: function(leftPos)
    {
        this._wrapper.setStyle({
            "left": leftPos + 'px'
        });
        this._wrapperCopy.setStyle({
            "left": leftPos + 'px'
        });
        this._wrapperLeft = leftPos;
    },
    /**
     * Update the UI of the cursor according to the _cursorPosition number
     * @name updateCursorPosition
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    updateCursorPosition: function()
    {   
        var tempA,offsetLeft,leftPos,wrapperCopyEle,wrapperCopyEndSpan;
        // Cursor position is at the beginning charcter
        // No matter labelText have content or not, the cursor left should placed at 0px and the left of the wrapper should be 0px too.
        if (this._cursorPosition === 0) { 
            this._cursor.setStyle({
                "left": '0px'
            });
            this.updateLeftByCursor(0);
            return;
        }
        
        offsetLeft = this._wrapperCopy.getHTMLElement().offsetLeft;
        leftPos = 0;
        this.wrapperCopyWidth = this._wrapperCopy.getHTMLElement().offsetWidth;
        
        
        
        if (accedo.platform==="samsung" && accedo.device.appengine.getLayoutEngine().name==="Gecko") {
            if (this._cursorPosition>0) {
                wrapperCopyEle = this._wrapperCopy.getHTMLElement();
                wrapperCopyEndSpan = wrapperCopyEle.lastChild;
                this.wrapperCopyWidth -= wrapperCopyEndSpan.clientWidth;
            }
        }
        
        tempA = (this.wrapperCopyWidth + offsetLeft);
        
        if (tempA > this._inputFieldWidth - 1) {
            // cursor will go out of inputField, force it back to the end position
            // modify the leftPos of both wrapper and wrapperCopy
            leftPos = this._inputFieldWidth - 1;
            this.updateLeftByCursor(this._inputFieldWidth - this.wrapperCopyWidth);
        } else if (tempA > 0) {
            // cursor still within the inputField area, only need to update the cursor position, the leftPos no need to change
            leftPos = tempA;
        } else {
            // cursor will go out of inputField, force it back to the beginning position
            // modify the leftPos of both wrapper and wrapperCopy
            leftPos = 0;
            this.updateLeftByCursor(-this.wrapperCopyWidth);
        }
        
        this._cursor.setStyle({
            "left": leftPos + 'px'
        });
        
        this._cursorLeftPosition = leftPos;
    },
    /**
     * Move the cursor in certain direction
     * @name moveCursor
     * @function
     * @param {String} direction either "left" or "right"
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    moveCursor: function(direction)
    {   
        if (!this._inputState) {
            return;
        }
        if (direction.toLowerCase() === "left") {
            this.setCursorPos(this._cursorPosition - 1);
        } else if (direction.toLowerCase() === "right") {
            this.setCursorPos(this._cursorPosition + 1);
        }
    },
    /**
     * Move the cursor to a specific position
     * @name setCursorPos
     * @function
     * @param {Number} pos cursor position
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    setCursorPos: function(pos)
    {
        if (pos > this.getText().length) {
            pos = this.getText().length;
        } else if (pos < 0) {
            pos = 0;
        }

        if(this._cursorPosition !== pos){
            this._cursorPosition = pos;

            if (this._cursorPosition === 0) {
                this._labelTextCopy = "";
            } else if (this.isPassword){
                this._labelTextCopy = accedo.String.truncate(this._wrapper.getText(), this._cursorPosition, "");
            } else {
                this._labelTextCopy = accedo.String.truncate(this._labelText, this._cursorPosition, "");
            }

            this._setElementText(this._wrapperCopy, this._labelTextCopy);
            this.updateCursorPosition();
        }
    },
    /**
     * Make the inputField ready for input
     * @name startInput
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    startInput: function()
    {   
        this.root.addClass("input");
        this._inputState = true;
        if (accedo.Object.isNull(this._inputFieldWidth)) {
            this._inputFieldWidth = this.container.getHTMLElement().clientWidth;
            this.updateLeft('add');
            this.updateCursorPosition();
        }
    },
    /**
     * Make the inputField inactive
     * @name stopInput
     * @function
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    stopInput: function()
    {   
        this.root.removeClass("input");
        this._inputState = false;
    },
    /**
     * Check if inputField active?
     * @name isInput
     * @function
     * @return {Boolean} true if actived; false otherwise
     * @memberof accedo.ui.widget.InputField#
     * @public
     */
    isInput: function()
    {   
        return this.root.hasClass("input");
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Creates a new Keyboard instance.
 * Can dispatch "click" event
 * @name Keyboard
 * @memberof accedo.ui.widget
 * @class A keyboard class, providing different type of keyboard function.
 * @author <a href="mailto:thomas.lee@accedo.tv">Thomas Lee</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create("accedo.ui.widget.Keyboard", "accedo.ui.Container", ["accedo.ui.widget.Button"],{},{

    /**
     * To store the keyboard button in form of [row_no][col_no]
     * @private
     * @name keyboard 
     * @memberof accedo.ui.widget.Keyboard#
     */
    _keyboard:[],
    
    /**
     * To store the keyboard button in form of [row_no][col_no]
     * @private
     * @name _keyboardRow
     * @memberof accedo.ui.widget.Keyboard#
     */
    _keyboardRow:[],
    
    /**
     * To store the map of keyboard button in form of [row_no][col_no]
     * @private
     * @name _keyboardMap
     * @memberof accedo.ui.widget.Keyboard#
     */
    _keyboardMap:[],
	
    /**
     * To link the keyboard to this textfield
     * @private
     * @name _textField
     * @memberof accedo.ui.widget.Keyboard#
     */
    _textField:null,
	
    /**
     * To indicate the current status of lowercase or uppercase keyboard
     * @private
     * @name _lowerCase
     * @memberof accedo.ui.widget.Keyboard#
     */
    _lowerCase:true,
    /**
     * To define the callback of the click event of button
     * @private
     * @name _callback
     * @memberof accedo.ui.widget.Keyboard#
     */
    _callback:null,
    /**
     * To store the temporary value of each button
     * @private
     * @name _temp
     * @memberof accedo.ui.widget.Keyboard#
     */
    _temp:null,
    /**
     * To indicate the status of the keyboard
     * @private
     * @name _symbol
     * @memberof accedo.ui.widget.Keyboard#
     */
    _symbol:false,
    /**
     * @constructor
     * @memberof accedo.ui.widget.Keyboard#
     * @param {Object} opts The options object
     * @param {String | Object} opts String or an object
     * @private
     */
    initialize: function($super,opts) {
        opts = opts || {};
        opts.focusable = false;
		
        this.id = opts.id.replace("#","");
        //first focus of the keyboard
        opts.firstSelectedIndex = opts.firstSelectedIndex || [0,0];

        opts.letter = opts.letter || ["abcdefghi","jklmnopqr","stuvwxyz "];
        //set the button css
        opts.buttonCss = opts.buttonCss || "accedo-ui-keyboard-button";
		
        //REGULAR(STOP),CYCLE,NEXT
        opts.behaviour = opts.behaviour || "REGULAR";
        
        if(opts.behaviour === "NEXT" || opts.behaviour === "CYCLE"){
            opts.loop = true;
        }
       
        $super(opts);

        this.root.addClass('accedo-ui-keyboard');

        this.createKeyboard();
        this.setNavigation();
        this.setOption("defaultFocus",this.id+"_"+opts.firstSelectedIndex[0]+"_"+opts.firstSelectedIndex[1]);
        
    },
    
	
    /**
     * Create DIV to attach a child to the DOM tree in the right place 
     * @param {accedo.ui.AbstractComponent} child
     * @private
     * @function
     * @memberof accedo.ui.widget.Keyboard#
     * @name doAttachElement
     * @override
     */
    doAttachElement: function(child) {	
        //overide the do attachment function
        if(this.currentRow !== -1){
            child.attachTo(this, this._keyboardRow[this.currentRow], accedo.ui.AbstractComponent.PLACE_APPEND);
        }else{
            child.attachTo(this, this.getRoot(), accedo.ui.AbstractComponent.PLACE_APPEND);
        }
    },
	
    /**
     * Create the keyboard 
     * @private
     * @function
     * @memberof accedo.ui.widget.Keyboard#
     * @name createKeyboard
     */
    //create the layout of the keyboard
    createKeyboard:function(){
        //to create an key and value and pass the button setting to a temp variable
        this._temp = [];
        var row,col,row_no,col_no;
        for(row = 0;row<this.opts.letter.length;row++){
            this._temp[row] = [];
            for(col = 0 ;col<this.opts.letter[row].length;col++){
                this._temp[row][col] = {};
                if(accedo.Object.isString(this.opts.letter[row][col])){
                    this._temp[row][col].key = this.opts.letter[row][col];
                    this._temp[row][col].input = this.opts.letter[row][col];
                    this.opts.letter[row][col] = {
                        key:this.opts.letter[row][col],
                        input:this.opts.letter[row][col]
                    };
                }else{
                    this._temp[row][col] = this.opts.letter[row][col];
                    this._temp[row][col].key = this.opts.letter[row][col].key;
                    if(this.opts.letter[row][col].input){
                        this._temp[row][col].input = this.opts.letter[row][col].input;
                    }else{
                        this.opts.letter[row][col].input = this._temp[row][col].key;
                        this._temp[row][col].input = this._temp[row][col].key;
                    }
                }
            }
        }
        
        //create mapping first which in form as a table and record their position for display and naviation
        for(row_no = 0 ;row_no<this._temp.length;row_no++){
            for(col_no = 0 ;col_no<this._temp[row_no].length;col_no++){
                this.createMap(row_no,col_no);
            }
        }
        
        //create the key button on the div UI
        for(row_no = 0 ;row_no<this._temp.length ;row_no++){
            this._keyboard[row_no] = [];
            this._keyboardRow[row_no] = accedo.Element.create('div');
            this._keyboardRow[row_no].addClass("accedo-ui-keyboard-row");
            this._keyboardRow[row_no].setAttributes({
                id:this.id+"-row"+row_no
            });

            this.root.appendChild(this._keyboardRow[row_no]);
            for(col_no =0 ; col_no < this._temp[row_no].length ; col_no++){
                this.createKey(row_no,col_no);
            }
        }
        this.currentRow = -1;
        this._temp.length = 0;
    },
    
    /**
     * to find the available place(position) to put the the button on the map.
     * @private
     * @function
     * @param {Number} row row number of that button
     * @param {Number} col column number of that button
     * @memberof accedo.ui.widget.Keyboard#
     * @name getFirstArea
     */
    //to find the available place(position) to put the the button on the map.
    getFirstArea:function(row,col){
        //occupied
        var len;
        if(typeof this._keyboardMap[row][col] !== "undefined"){
            if(col<this._keyboardMap[row].length){
                //the empty space after the current col
                for(len = col ; len < this._keyboardMap[row].length ;len ++ ){
                    if(typeof this._keyboardMap[row][len] === "undefined"){
                        this._keyboardMap[row][len] = this.id+"_"+row+"_"+col;
                        return[row,len];
                    }   
                }
                return[row,this._keyboardMap[row].length];
            }else{
                return[row,this._keyboardMap[row].length];
            }
        }else{
            return [row,col];
        }  
    },
    
    /**
     * to get the textfield in keyboard
     * @public
     * @function
     * @return {accedo.ui.widget.InputField} the textField
     * @memberof accedo.ui.widget.Keyboard#
     * @name getTextField
     */  
    getTextField:function(){
        return this._textField;
    },
	
    /**
     * to create the map according to the letter parameter
     * @private
     * @function
     * @param {Number} row row number of that button
     * @param {Number} col column number of that button
     * @memberof accedo.ui.widget.Keyboard#
     * @name createMap
     */
    createMap:function(row,col){
        row = parseInt(row,10);
        col = parseInt(col,10);
        
        if(typeof this._keyboardMap[row] === "undefined"){
            this._keyboardMap[row] = [];
        }
        //to find the first position to place the btn
        var cell = this.getFirstArea(row,col), rowNo,colNo,x,y;
        this._temp[row][col].position = cell;
        //size 1 x 1
        if(accedo.Object.isString(this.opts.letter[row][col])){
            this._keyboardMap[cell[0]][cell[1]] = this.id+"_"+row+"_"+col ;
        }else{

            //other sizes    
            if(typeof this.opts.letter[row][col].rowSpan === "undefined"){
                this._temp[row][col].rowSpan = 1;
            }else{
                this._temp[row][col].rowSpan = this.opts.letter[row][col].rowSpan;
            }
            
            if(typeof this.opts.letter[row][col].colSpan === "undefined"){
                this._temp[row][col].colSpan = 1;
            }else{
                this._temp[row][col].colSpan = this.opts.letter[row][col].colSpan;
            }
            
            for(y = 0;y < this._temp[row][col].rowSpan;y++){
                rowNo = cell[0]+y;
                for( x = 0;x < this._temp[row][col].colSpan;x++){
                    colNo =cell[1]+x;
                    if(typeof this._keyboardMap[rowNo] === "undefined"){
                        this._keyboardMap[rowNo] = [];
                    }
                    this._keyboardMap[rowNo][colNo] = this.id+"_"+row+"_"+col ;
                }
            }
        }
    },

    /**
     * to create the UI of the button
     * @private
     * @function
	 * @param {Number} row row number of that button
     * @param {Number} col column number of that button
     * @memberof accedo.ui.widget.Keyboard#
     * @name createKey
     */
    //to create each key button
    createKey:function(row,col){
        col = parseInt(col,10);
        row = parseInt(row,10);

        //to create a button on display
        this._keyboard[row][col] = new accedo.ui.widget.Button({
            id: "#"+this.id+"_"+row+"_"+col,
            text: this._temp[row][col].key
        });
        var map_row,key_position, nextMode = false;
        this._keyboard[row][col].setOption('input',this._temp[row][col].input);
        this._keyboard[row][col].setOption('position',this._temp[row][col].position);

        if(this._temp[row][col].disable){
            this._keyboard[row][col].setOption('disable',true);
            this._keyboard[row][col].setEnabled(false);
        }else{
            this._keyboard[row][col].setOption('disable',false);
        }
        this._keyboard[row][col].root.setClass(this.opts.buttonCss);
        this._keyboard[row][col].setOption('colSpan',this._temp[row][col].colSpan);
        this._keyboard[row][col].setOption('rowSpan',this._temp[row][col].rowSpan);  

        if(typeof this._temp[row][col].buttonCss !== "undefined"){
            this._keyboard[row][col].root.addClass(this._temp[row][col].buttonCss);
        }
				
        this.currentRow = row;
        this.attach(this._keyboard[row][col]);
        
        key_position = this._keyboard[row][col].getOption('position');
        
        //handle the rightmost button
        if(key_position[1]+this._keyboard[row][col].getOption('colSpan') === this._keyboardMap[row].length){
            this._keyboard[row][col].onRequestFocus = accedo.Fn.bind(function(){
                var btn = this._keyboard[row][col], i, firstSelectIndex, oldId;
                if((this.getOption("behaviour") === "CYCLE" && this.getOption("loop") === false) || (this.getOption("behaviour") === "NEXT" && this.getOption("loop") === false)){
                    delete this._keyboard[row][col].opts.nextRight;
                }else{
                    map_row = key_position[0];
                    switch(this.getOption("behaviour")){
                        case "CYCLE" :
                            this._keyboard[row][col].setOption("nextRight",this._keyboardMap[map_row][0]);
                            break;
                        case "NEXT":
                            if(map_row+1>= this._keyboardMap.length){
                                this._keyboard[row][col].setOption("nextRight",this._keyboardMap[0][0]);
                                nextMode = true;
                            }else{
                                this._keyboard[row][col].setOption("nextRight",this._keyboardMap[map_row+1][0]);
                                //In the next mode, no need to handle the case when irregular button get from the next row to previous row,
                                //so 3 cases are needed to be satisified the old one is in the first col and the new one is in the last col and they are not col overlapped
                                firstSelectIndex = this.opts.firstSelectedIndex;
                                oldId = this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]];
                                if(firstSelectIndex[1] === 0 && this._keyboard[row].slice(-1)[0] === btn){
                                    nextMode = true;
                                    for( i = 0; i < btn.opts.colSpan ; i++){
                                        if(this._keyboardMap[firstSelectIndex[0]][btn.opts.position[1]+i] === oldId){
                                            //if it is the col overlap then set the new nextDown later
                                            nextMode = false;
                                            break;
                                        }
                                    }
                                }
                            }
                            break;
                    }
                }
                if(!nextMode){
                    this.handleIrregular(this._keyboard[row][col], row);
                }
                nextMode = false;
            },this);
        }else if(key_position[1] === 0 ){
            //to handle the leftmost button
            this._keyboard[row][col].onRequestFocus = accedo.Fn.bind(function(){
                var btn = this._keyboard[row][col], i, firstSelectIndex, oldId;
                if((this.getOption("behaviour") === "CYCLE" && this.getOption("loop") === false) || (this.getOption("behaviour") === "NEXT" && this.getOption("loop") === false)){
                    delete this._keyboard[row][col].opts.nextLeft;
                }else{
                    map_row = key_position[0];
                    switch(this.getOption("behaviour")){
                        case "CYCLE" :
                            this._keyboard[row][col].setOption("nextLeft",this._keyboardMap[map_row].slice(-1)[0]);
                            break;
                        case "NEXT":
                            if(map_row-1>= 0){
                                this._keyboard[row][col].setOption("nextLeft",this._keyboardMap[map_row-1].slice(-1)[0]);
                                nextMode = true;
                            }else{
                                this._keyboard[row][col].setOption("nextLeft", this._keyboardMap.slice(-1)[0].slice(-1)[0]);
                                //In the next mode, no need to handle the case when irregular button get from the next row to previous row,
                                //so 3 cases are needed to be satisified the old one is in the first col and the new one is in the last col and they are not col overlapped
                                firstSelectIndex = this.opts.firstSelectedIndex;
                                oldId = this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]];
                                if(firstSelectIndex[1] === 0 && this._keyboard[row].slice(-1)[0] === btn){
                                    nextMode = true;
                                    for( i = 0; i < btn.opts.colSpan ; i++){
                                        if(this._keyboardMap[firstSelectIndex[0]][btn.opts.position[1]+i] === oldId){
                                            //if it is the col overlap then set the new nextDown later
                                            nextMode = false;
                                            break;
                                        }
                                    }
                                } 
                            }
                            break;
                    }
                }
                if(!nextMode){
                    this.handleIrregular(this._keyboard[row][col], row);
                }
                nextMode = false;
            },this);
        }else{
            this._keyboard[row][col].onRequestFocus = accedo.Fn.bind(function(){
                this.handleIrregular(this._keyboard[row][col], row);
            },this);
        }

        //default onFocus
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.FOCUS, accedo.Fn.bind(function() {
            this.opts.firstSelectedIndex = this._keyboard[row][col].opts.position;
            if(this._textField){
                this._textField.startInput();
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }
        },this));
        
        //default onClick function
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function() {
            if(this._textField){
                this._textField.insertChar(accedo.String.unescapeHTML(this._keyboard[row][col].opts.input));
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }
            
            if(this._callback){
                this._callback();
            }
        },this));
        
        //default blur function
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.BLUR, function(){
            if(this._textField){
                this._textField.stopInput();
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }
        }, this);
        
    }, 
    /**
     * to handle the navigation of the button when to the irregular button so that it can handle back to the previous button,
     * firstSelectedIndex will be the previous button
     * @private
     * @function
     * @memberof accedo.ui.widget.Keyboard#
     * @name handleIrregular
     */
    handleIrregular:function(btn,row){
        //no need to handle if this button is not irregualr
        if(btn.opts.colSpan <=1 && btn.opts.rowSpan <=1){
            return;
        }
        
        //to handle irregular button
        var firstSelectIndex = this.opts.firstSelectedIndex, btnPosition = btn.opts.position;
        
        if(btnPosition[0] < firstSelectIndex[0] && btn.opts.rowSpan+row-1!== firstSelectIndex[0]){
            btn.setOption("nextDown",this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]]);
            return;
        }else  if(btnPosition[0] > firstSelectIndex[0] && btn.opts.rowSpan+row-1 !== firstSelectIndex[0]){
            btn.setOption("nextUp",this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]]);
            return;
        }else if(btnPosition[1] < firstSelectIndex[1] && btn.opts.rowSpan !== 1 && this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]] !== this._keyboardMap[firstSelectIndex[0]].slice(-1)[0]){
            btn.setOption("nextRight",this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]]);
            return;
        }else if(btnPosition[1] > firstSelectIndex[1] && btn.opts.rowSpan !== 1 && this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]] !== this._keyboardMap[firstSelectIndex[0]][0]){
            btn.setOption("nextLeft",this._keyboardMap[firstSelectIndex[0]][firstSelectIndex[1]]);
            return;
        }
    },
    /**
     * to set the navigation of the button
     * @private
     * @function
     * @memberof accedo.ui.widget.Keyboard#
     * @name setNavigation
     */
    //set the navgiation of each button
    setNavigation:function(){
        var row_no,col_no;
        for(row_no = 0; row_no<this._keyboard.length;row_no++){
            row_no = parseInt(row_no,10);
            for(col_no=0;col_no<this._keyboard[row_no].length;col_no++){
                col_no = parseInt(col_no,10);
                this.setNextRight(row_no,col_no);
                this.setNextLeft(row_no,col_no);
                this.setNextUp(row_no,col_no);
                this.setNextDown(row_no,col_no);
            }
        }
    },
    
    /**
     * to map the navigation of nextRight
     * @private
     * @function
	 * @param {Number} row_no row number of that button
     * @param {Number} col_no column number of that button
     * @memberof accedo.ui.widget.keyboard#
     * @name setNextRight
     */
    setNextRight:function(row_no,col_no){

        var nextRight = "",
        map_row = this._keyboard[row_no][col_no].opts.position[0],
        map_col = this._keyboard[row_no][col_no].opts.position[1],
        cell;

        //Handle nextRight
        if( map_col+1 < this._keyboardMap[map_row].length && col_no < this._keyboard[row_no].length-1 ){
            //normal case

            while(map_col+1<this._keyboardMap[map_row].length){
                map_col++;
                if(this._keyboardMap[map_row][map_col] !==  this.id+"_"+row_no + "_" + col_no){
                    cell = this._keyboardMap[map_row][map_col].split("_");
                    if(!this._keyboard[cell[1]][cell[2]].opts.disable){
                        nextRight = this._keyboardMap[map_row][map_col];
                        break;
                    }
                }
            }
        }

        if(nextRight !== ""){
            this._keyboard[row_no][col_no].setOption("nextRight",nextRight);
        }
    },
    
    /**
     * to map the navigation of nextLeft
     * @private
     * @function
	 * @param {Number} row_no row number of that button
     * @param {Number} col_no column number of that button
     * @memberof accedo.ui.widget.Keyboard#
     * @name setNextLeft
     */
    setNextLeft:function(row_no,col_no){
        var nextLeft = "",
        map_row = this._keyboard[row_no][col_no].opts.position[0],
        map_col = this._keyboard[row_no][col_no].opts.position[1],
        cell;
        
        //Handle nextLeft
        if(map_col-1 >= 0){
            //normal case
            while(map_col-1>=0){
                map_col--;
                if(this._keyboardMap[map_row][map_col] !== this.id+"_"+ row_no + "_" + col_no){
                    cell = this._keyboardMap[map_row][map_col].split("_");
                    if(!this._keyboard[cell[1]][cell[2]].opts.disable){
                        nextLeft = this._keyboardMap[map_row][map_col];
                        break;
                    }
                }
            }
        }
        if(nextLeft !==""){
            this._keyboard[row_no][col_no].setOption("nextLeft",nextLeft);
        }
    },

	
    /**
     * to map the navigation of nextUp
     * @private
     * @function
	 * @param {Number} row_no row number of that button
     * @param {Number} col_no column number of that button
     * @memberof accedo.ui.widget.Keyboard#
     * @name setNextUp
     */
    setNextUp:function(row_no,col_no){
        var nextUp = "",
        map_row = this._keyboard[row_no][col_no].opts.position[0],
        map_col = this._keyboard[row_no][col_no].opts.position[1],
        cell;
        
        //Handle nextUp
        if(map_row-1 >= 0){
            //normal case
            while(map_row-1>=0){
                map_row--;
                map_col = this._keyboard[row_no][col_no].opts.position[1];
                if( (map_col >= this._keyboardMap[map_row].length) && (this._keyboardMap[map_row][this._keyboardMap[map_row].legnth-1] !==  this.id+"_"+row_no + "_" + col_no)){
                    cell = this._keyboardMap[map_row][this._keyboardMap[map_row].length-1].split("_");
                    if(!this._keyboard[cell[1]][cell[2]].opts.disable){
                        nextUp = this._keyboardMap[map_row][this._keyboardMap[map_row].length-1];
                    }
                    break;
                }else if(this._keyboardMap[map_row][map_col] !==  this.id+"_"+row_no + "_" + col_no){
                    cell = this._keyboardMap[map_row][map_col].split("_");
                    if(!this._keyboard[cell[1]][cell[2]].opts.disable){
                        nextUp = this._keyboardMap[map_row][map_col];
                    }
                    if(nextUp){
                        break;
                    }
                }
            }
        }
        if(nextUp !==""){
            this._keyboard[row_no][col_no].setOption("nextUp",nextUp);
        }
    },

    /**
     * to map the navigation of nextDown
     * @private
     * @function
	 * @param {Number} row_no row number of that button
     * @param {Number} col_no column number of that button
     * @memberof accedo.ui.widget.Keyboard#
     * @name setNextDown
     */
    setNextDown:function(row_no,col_no){
        var nextDown = "",
        map_row = this._keyboard[row_no][col_no].opts.position[0],
        map_col = this._keyboard[row_no][col_no].opts.position[1],
        cell ; 

        //Handle nextDown
        if(map_row+1 < this._keyboardMap.length){
            //normal case
            while(map_row+1< this._keyboardMap.length){
                map_row++;
                map_col = this._keyboard[row_no][col_no].opts.position[1];
                if( (map_col >= this._keyboardMap[map_row].length) && (this._keyboardMap[map_row][this._keyboardMap[map_row].legnth-1] !==  this.id+"_"+row_no + "_" + col_no) ){
                    cell = this._keyboardMap[map_row][this._keyboardMap[map_row].length-1].split("_");
                    if(!this._keyboard[cell[1]][cell[2]].opts.disable && cell[1] !== row_no){
                        nextDown = this._keyboardMap[map_row][this._keyboardMap[map_row].length-1];
                    }
                    break;
                }else if(this._keyboardMap[map_row][map_col] !==  this.id+"_"+row_no + "_" + col_no && typeof this._keyboardMap[map_row][map_col] !=="undefined"){
                    cell = this._keyboardMap[map_row][map_col].split("_");
                    if(!this._keyboard[cell[1]][cell[2]].opts.disable){
                        nextDown = this._keyboardMap[map_row][map_col];
                    }else{
                        while(map_col>0){
                            map_col--;
                            cell = this._keyboardMap[map_row][map_col].split("_");
                            if(!this._keyboard[cell[1]][cell[2]].opts.disable){
                                nextDown = this._keyboardMap[map_row][map_col];
                                break;
                            }
                        }
                    }
                    if(nextDown){
                        break;
                    }
                }
            }
        }
        if(nextDown !== ""){
            this._keyboard[row_no][col_no].setOption("nextDown",nextDown);
        }
    },

    /**
    * To link the textfield to the keyboard
    * @public
    * @param {accedo.ui.widget.InputField} tf textfield which use to show the output of the keyboard
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setTextField
    */	
    setTextField:function(tf){
        if(tf instanceof accedo.ui.widget.InputField || tf instanceof accedo.ui.widget.TextArea){
            this._textField = tf ;
        }else{
            this._textField = null;
        }
    },

    /**
    * To set the text of button. when it is in character mode, it will replace the display text (key attribute of the button) and override the default display text. When it is in symbol mode, it will replace the text of that symbol button.
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {String} text the text shown on the button
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonText
    */	
    setButtonText:function(row,col,text){
        if( row>this._keyboard.length || col>this._keyboard[row].length ){
            return;
        }
                    
        if(this._symbol){
            if(accedo.Object.isString(this.symbolLetter[row][col])){
                this.symbolLetter[row][col] = text;
            }else if(this.symbolLetter[row][col].key){
                this.symbolLetter[row][col].key = text;
            }
        }else{
            this.opts.letter[row][col].key = text;
        }
        this._keyboard[row][col].setText(text);  
    },
    
    /**
    * To set the css of button 
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {String} css of the button
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonCss
    */	
    setButtonCss:function(row,col,css){
        if( row>this._keyboard.length || col>this._keyboard[row].length ){
            return;
        }
        this._keyboard[row][col].root.addClass(css);
    },
    
	
    /**
    * To set the button with original function that input a character into the textfield with a callback after inserting the text
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setOriginalOnClick
    */	
    setOriginalOnClick:function(row,col,callback){
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function() {
            if(this._textField){
                this._textField.insertChar(accedo.String.unescapeHTML(this._keyboard[row][col].opts.input));
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }
            
            if( !accedo.Object.isFunction(callback)){
                return;
            }else{
                callback();
            }
        },this));
    },
	
    /**
    * To set the button with a complete new function and with a callback after that function
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonOnClick
    */	
    setButtonOnClick:function(row,col,callback){
        if( row > this._keyboard.length || col>this._keyboard[row].length || !accedo.Object.isFunction(callback)){
            return;
        }
        //new onClick function
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(callback,this));
    },
    
    /**
    * To set the button with a delete function and with a callback after that function
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonOnClickDel
    */	
    setButtonOnClickDel:function(row,col,callback){
        if( row>this._keyboard.length || col>this._keyboard[row].length){
            return;
        }
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function(){
            if(this._textField){
                if (this._textField.backspace) {
                    this._textField.backspace();
                } else {
                    this._textField.deleteChar(); // for backward compatibility
                }
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }

            if( !accedo.Object.isFunction(callback)){
                return;
            }else{
                callback();
            }
        },this));
    },
    
    /**
    * To set the button with a move left function of inputfield and with a callback after that function
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonOnClickMoveLeft
    */	
    setButtonOnClickMoveLeft:function(row,col,callback){
        if( row>this._keyboard.length || col>this._keyboard[row].length){
            return;
        }
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function(){
            if(this._textField){
                this._textField.moveCursor("left");
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }
            if( !accedo.Object.isFunction(callback)){
                return;
            }else{
                callback();
            }
        },this));
    },
    
    /**
    * To set the button with a move right function of inputfield and with a callback after that function
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonOnClickMoveRight
    */	
    setButtonOnClickMoveRight:function(row,col,callback){
        if( row>this._keyboard.length || col>this._keyboard[row].length){
            return;
        }
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function(){
            if(this._textField){
                this._textField.moveCursor("right");
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }
            if( !accedo.Object.isFunction(callback)){
                return;
            }else{
                callback();
            }
        },this));
    },
    
    /**
    * To set the button with a space function of inputfield and with a callback after that function
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonOnClickMoveSpace
    */	
    setButtonOnClickSpace:function(row,col,callback){
        if( row>this._keyboard.length || col>this._keyboard[row].length){
            return;
        }
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function(){
            if(this._textField){
                this._textField.startInput();
                this._textField.insertChar(" ");
            }else{
                accedo.console.debug("no textField set to the keyboard");
            }
            
            if( !accedo.Object.isFunction(callback)){
                return;
            }else{
                callback();
            }
        },this));
    },
    
    /**
    * To set the button with a capital function of button with 1 character and with a callback after that function
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonCapital
    */	
    setButtonCapital:function(row,col,callback){
        var row_no,col_no;
        if( row>this._keyboard.length || col>this._keyboard[row].length){
            return;
        }
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK,this);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function(){
            this._lowerCase = !this._lowerCase;
            for(row_no= 0 ;row_no<this._keyboard.length;row_no++){
                for(col_no = 0;col_no<this._keyboard[row_no].length;col_no++){
                    if(this._keyboard[row_no][col_no].getText().length<=1){
                        if(!this._lowerCase){
                            this._keyboard[row_no][col_no].opts.input = this._keyboard[row_no][col_no].opts.input.toUpperCase();
                            this._keyboard[row_no][col_no].setText(this._keyboard[row_no][col_no].getText().toUpperCase());
                        }else{
                            this._keyboard[row_no][col_no].opts.input = this._keyboard[row_no][col_no].opts.input.toLowerCase();
                            this._keyboard[row_no][col_no].setText(this._keyboard[row_no][col_no].getText().toLowerCase());
                        }
                    }
                }
            }
            if( !accedo.Object.isFunction(callback)){
                return;
            }else{
                callback();
            }
        },this));
        
    },

    /**
    * To set the button with a symbol letter on the current button and with a callback after that function
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @param {Array} symbol the letter of the symbol. Symbol format should be matched with the initialized one.otherwise,need to handle the navigation,create,remove the button..
    * @param {Function} callback the callback function after 
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setButtonSymbol
    */
    setButtonSymbol:function(row,col,symbol,callback){
        if(row>this._keyboard.length || col>this._keyboard[row].length){
            return;
        }
        this._keyboard[row][col].removeAllListeners(accedo.ui.Evt.CLICK,this);
        this._keyboard[row][col].addEventListener(accedo.ui.Evt.CLICK,accedo.Fn.bind(function(){
            this.symbolLetter = symbol;
            if(!this._symbol){
                this._symbol = true;
                this.changeAllButtonText(this.symbolLetter);
            }else{
                this.resetKeyboard();
            }
            if( !accedo.Object.isFunction(callback)){
                return;
            }else{
                callback();
            }
        },this));
        
    },
    /**
    * To reset the character to ABC
    * @private
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name resetKeyboard
    */
    resetKeyboard:function(){
        var col_no,row_no;
        //if it is not symbol,change back to the orignal character when setting in the view
        this._symbol = false;
        for(row_no = 0 ;row_no<this.opts.letter.length;row_no++){
            for(col_no = 0 ;col_no< this.opts.letter[row_no].length;col_no++){
                if(!this._lowerCase && this.opts.letter[row_no][col_no].key.length <= 1){
                    this.opts.letter[row_no][col_no].key = this.opts.letter[row_no][col_no].key.toUpperCase();
                    this.opts.letter[row_no][col_no].input = this.opts.letter[row_no][col_no].input.toUpperCase();
                }else if(this.opts.letter[row_no][col_no].key.length <= 1){
                    this.opts.letter[row_no][col_no].key = this.opts.letter[row_no][col_no].key.toLowerCase();
                    this.opts.letter[row_no][col_no].input = this.opts.letter[row_no][col_no].input.toLowerCase();
                }
            }
        }
        this.changeAllButtonText(this.opts.letter);
    },
    /**
    * To get the specific button where you can setOption or specific properties of that button
    * @public
	* @param {Number} row the row number of that button
	* @param {Number} col the column number of that button
	* @returns {accedo.ui.widget.Button} Button
	* @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name getButton
    */
    getButton:function(row,col){
        return this._keyboard[row][col];
    },

	
    /**
    * To all the button
    * @public
    * @param {Number} row the row number of that button
    * @param {Number} col the column number of that button
    * @returns {Array} Array of accedo.ui.widget.Button which in form of Array[row][col]
    * @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name getAllButton
    */
    getAllButton:function(){
        return this._keyboard;
    },
   
    /**
    * to set the default callback when each key is pressed.
    * @public
	* @param {cb} function
	* @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name setDefaultCallback
    */
    setDefaultCallback:function(cb){
        if(accedo.Object.isFunction(cb)){
            this._callback = cb;
        }
    },
    /**
    * To change the text of each button with a character
    * @private
	* @param {Array} letter the text of each button in form of letter[row][col]
	* @memberof accedo.ui.widget.Keyboard#
    * @function
    * @name changeAllButtonText
    */
    changeAllButtonText:function(letter){
        var row_no,col_no;
        for(row_no = 0;row_no<letter.length;row_no++){
            for(col_no=0;col_no< letter[row_no].length;col_no++){
                if(typeof letter[row_no][col_no] !== "undefined"){
                    if(accedo.Object.isString(letter[row_no][col_no])){
                        this._keyboard[row_no][col_no].opts.key = letter[row_no][col_no];
                        this._keyboard[row_no][col_no].opts.input = letter[row_no][col_no];
                    }else{
                        this._keyboard[row_no][col_no].opts.key = letter[row_no][col_no].key;
                        if(typeof letter[row_no][col_no].input !== "undefined"){
                            this._keyboard[row_no][col_no].opts.input = letter[row_no][col_no].input;
                        }else{
                            this._keyboard[row_no][col_no].opts.input = letter[row_no][col_no].key;
                        }
                    }
                    this._keyboard[row_no][col_no].setText(this._keyboard[row_no][col_no].opts.key);
                }
            }
        }
    }
});

/*jslint browser: true, devel: true, newcap:false */
/*global accedo:false */

/**
 * <pre>
 * List container.
 * - Based on the linear layout (but only extending accedo.ui.container), the biggest
 *      difference is the List is linked to a datasource.
 * - Displays part or all of its content based on a user-defined template. Properties
 *      common to all children are defined in the view (see the childTemplate option), while
 *      the particulars of each child are defined in the controller and passed as a callback.
 *      Read details about it in the setDisplayHelper function comments.
 * - Also noteworthy is the fact it uses an internal focus so as not to mess with the
 *       main one. The CSS class used is 'selected' instead of 'focused', and the selected
 *       element can be distinguished even when the focus has moved out of the list ,
 *       you can define onSelect and onUnselect callbacks on the list's children.
 *       However, it is unable to focus on the list area using mouse. Only child is focusable
 *       and it will be treat as the child focused and selected in css instead of list focused and child selected in css.
 * - Supports CAROUSEL behaviour on option. Carousel-style lists can display less children than
 *      are present in the datasource like any REGULAR list, but they will also display more, by
 *      duplicating children, when there are more visible elements than data objects in the datasource.
 *      For CAROUSELs, the selected element always appears in the same position (unless calling setSelectedIndex),
 *      and any selection move scrolls the list. Also the last element is linked to the first one.
 *      A REGULAR list scrolls only when necessary and doesn't have this 'looping' behaviour.
 * - Supports association with a accedo.ui.scrollbar via the public setScrollbar function.
 * - Dispatches 'accedo.ui.Evt.SCROLL' event with a 'reverse' indicator when the list's selection is successfully scrolled
 * - Dispatches 'accedo.ui.Evt.SELECTION_CHANGED' event with a 'reverse' indicator when the list's selection is changed
 * - Options: those supported by any container, and :
 *    - orientation   (accedo.ui.widget.List.HORIZONTAL (default) | accedo.ui.widget.List.VERTICAL)
 *    - behaviour     (accedo.ui.widget.List.REGULAR (default) |  accedo.ui.widget.List.CAROUSEL)
 *    - visibleSize   (Number (default 1)) How many children to display. 0 means all of them.
 *    - preloadSize   (Number (default 0)) How many children to preload in advance (created but not displayed).
 *    - noCache       (boolean (default false)) If true, the list's cache will not be used, and removed children will be deinited immediately.
 *    - scrollStep    (Number (default 1)) For REGULAR lists only, How many elements to scroll when scrolling the list.
 *    - loop          (boolean (default false)) For REGULAR lists only, do we link the selection of the child at the backward
 *                  border to the one at the forward border, and vice-versa. NOTE when going left or up from the first child,
 *                  you will end up selecting the last child loaded by the datasource (not necessarily the last 'loadable' one).
 *    - forwardFocus   (boolean) True to refer the focus to the child. For example, if the forwardFocus is true and childTemplate is a container 
 *                      that consists of severl buttons.Developer can refer the focus onto one of the buttons in the container as the focus will be 
 *                      forward when the child is focus.Default is false which keep the focus on the container and unable to forward to its children.
 *    - scrollBeforeBorder (Number | Array (default [0,0])) For REGULAR lists only, try to scroll when the selection is N or
 *                  less children away from the border.
 *                  If the option is a number, N is this number.
 *                  If the option is an Array, N is the first number in the Array when moving backward, and
 *                                                  the second number in the array when moving forward.
 *    - forceSelectOnFocus (boolean (default false)) If true, when the list gets focused it will launch the onSelect callback
 *                  even if the selection was already set.
 *    - firstSelectedIndex  (Number (default 0)) The child at the specified list index will be selected when the list is focused
 *                  for the 1st time. Use -1 to set it to the last child.
 *    - animation (boolean) If true,it will enable the animation effect when scroll
 *    - effect (Object) It will be object effect of the animation.
 *    - rowHeight (number) (optional) it is for animation in the list,it will scroll with the rowHeight.If it is not defined,it will automatically detected using the element offsetHeight
 *    - rowWidth (number) (optional) it is for animation in the list,it will scroll with the rowWidth.If it is not defined,it will automatically detected using the element offsetWidth
 *    - childTemplate (Object). which will be used in setDisplayHelper.
 *      For example, it can use a button widget as a template. Then you can set the properties inside the setDisplayHelper function
 *      childTemplate:{
 *          type:accedo.ui.widget.Button
 *      }
 * - Notes:
 *    - You can change the datasource dynamically like appendDate,insertData, removeData.
 *    - You can grow or shrink the list's size dynamically (the selected child will not be removed)
 *    - The nextLeft/nextRight (for a horizontal list), nextUp/nextUp (for a vertical list) options
 *      will be overwritten to manage the list's internal selection, however, the value you specified
 *      will be used to focus another element when the current selection is a REGULAR list's border.
 * </pre>
 * 
 * @name List
 * @memberof accedo.ui.widget
 * @class This container is used to align child elements, either vertically
 *        or horizontally.
 * @author <a href="mailto:gregory.desfour@accedobroadband.com">Gregory Desfour</a>
 * @extends accedo.ui.Container
 * @param {Object} opts The options object
 * @param {Integer} [opts.orientation="accedo.ui.widget.List.VERTICAL"]
 */
accedo.Class.create("accedo.ui.widget.List", "accedo.ui.Container", ["accedo.data.Ds","accedo.anim.Emile"], function() {
    return {
        /**
         * This is used to indicate a regular behaviour
         * The scrollStep option is only available in this behaviour
         * @constant
         * @name REGULAR
         * @memberof accedo.ui.widget.List
         */
        REGULAR : 0x01,
    
        /**
         * This is used to indicate a carousel behaviour
         * The scrollStep option is NOT available in this behaviour (will be set to 1)
         * @constant
         * @name CAROUSEL
         * @memberof accedo.ui.widget.List
         */
        CAROUSEL : 0x02,
    
        /**
         * This is used to indicate a vertical layout.
         * @constant
         * @memberof accedo.ui.widget.List
         */
        VERTICAL : 0x01,
    
        /**
         * This is used to indicate a horizontal layout.
         * @constant
         * @name HORIZONTAL
         * @memberof accedo.ui.widget.List
         */
        HORIZONTAL : 0x02,
        /**
         * This places a child at:
         * <pre>
         *  - - -
         * |     |
         * |x    |
         * |     |
         *  - - -
         * </pre>
         * @name GRAVITY_LEFT
         * @constant
         * @memberof accedo.ui.widget.List
         * @deprecated suggest to do it on css class
         */
        GRAVITY_LEFT : 0x01,    
        /**
         * This places a child at:
         * <pre>
         *  - - -
         * |     |
         * |    x|
         * |     |
         *  - - -
         * </pre>
         * @name GRAVITY_RIGHT
         * @constant
         * @memberof accedo.ui.widget.List
         * @deprecated suggest to do it on css class
         */
        GRAVITY_RIGHT : 0x02,

        /**
         * This places a child at:
         * <pre>
         *  - - -
         * |     |
         * |  x  |
         * |     |
         *  - - -
         * </pre>
         * @name GRAVITY_CENTER
         * @constant
         * @memberof accedo.ui.widget.List
         * @deprecated suggest to do it on css class
         */
        GRAVITY_CENTER : 0x04,
        /**
         * Event: moved. MOVED event will be dispatched when the list moved/scrolling that it will also change the selected child
         * @constant
         * @name EVT_MOVED
         * @memberof accedo.ui.widget.List
         * @deprecated moved to accedo.ui.Evt.MOVED
         */
        EVT_MOVED : accedo.ui.Evt.MOVED,
    
        /**
         * Event: selection chaged. SELECTION_CHANGED event will be dispatched when the current selected child is different from the new selected one including dynamically changing the selected child or using mouse to focus a new one
         * @constant
         * @name EVT_SELECTION_CHANGED
         * @memberof accedo.ui.widget.List
         * @deprecated moved to accedo.ui.Evt.SELECTION_CHANGED
         */
        EVT_SELECTION_CHANGED : accedo.ui.Evt.SELECTION_CHANGED,

        /**
         * Event: scroll. SCROLL event will dispatch when the list scroll
         * @constant
         * @name EVT_SCROLL
         * @memberof accedo.ui.widget.List
         * @deprecated moved to accedo.ui.Evt.SCROLL
         */
        EVT_SCROLL : accedo.ui.Evt.SCROLL,
    
        /**
         * Event: click
         * @constant
         * @name EVT_CLICK
         * @memberof accedo.ui.widget.List
         * @deprecated moved to accedo.ui.Evt.CLICK
         */
        EVT_CLICK : accedo.ui.Evt.CLICK,
    
        /**
         * Event: attach
         * @constant
         * @name EVT_ATTACH
         * @memberof accedo.ui.widget.List
         * @deprecated moved to accedo.ui.Evt.ATTACH
         */
        EVT_ATTACH : accedo.ui.Evt.ATTACH,
    
        /**
         * Event: detach
         * @constant
         * @name EVT_DETACH
         * @memberof accedo.ui.widget.List
         * @deprecated moved to accedo.ui.Evt.DETACH
         */
        EVT_DETACH : accedo.ui.Evt.DETACH,
    
        /**
         * This is used to indicate a scrollbar will update layout base on each item movement.
         * @constant
         * @name SCROLL_BASE_ON_ITEM
         * @memberof accedo.ui.widget.List
         */
        SCROLL_BASE_ON_ITEM : 0x01,
    
        /**
         * This is used to indicate a scrollbar will update layout base on each page movement.
         * @constant
         * @name SCROLL_BASE_ON_PAGE
         * @memberof accedo.ui.widget.List
         */
        SCROLL_BASE_ON_PAGE : 0x02
    };
}, {
    /**
     * @private
     * @name _isAllVisible
     * @memberof accedo.ui.widget.List#
     */
    _isAllVisible: false,
    
    //Cache array used for DOM preloading (will preload images too), empty at first
    /**
     * @private
     * @name _cache
     * @memberof accedo.ui.widget.List#
     */
    _cache : [],  
    /**
     * @private
     * @name _callbackOnSelect
     * @memberof accedo.ui.widget.List#
     */
    _callbackOnSelect : true,

    /**
     * @private
     * @name _waitForMoreData
     * @memberof accedo.ui.widget.List#
     */
    _waitForMoreData: null,
    /**
     * @private
     * @name _datasource
     * @memberof accedo.ui.widget.List#
     */
    _datasource: null,
    /**
     * @private
     * @name _selectRequest
     * @memberof accedo.ui.widget.List#
     */
    _selectRequest: null,
    //Iterator used to generate unique IDs for the created child components
    /**
     * @private
     * @name _iteratorChildId
     * @memberof accedo.ui.widget.List#
     */
    _iteratorChildId: null,
    /**
     * @private
     * @name _selectedChild
     * @memberof accedo.ui.widget.List#
     */
    _selectedChild: null,
    /**
     * @private
     * @name _scrollCondition
     * @memberof accedo.ui.widget.List#
     */
    _scrollCondition : null,
    /**
     * @private
     * @name _noCache
     * @memberof accedo.ui.widget.List#
     */
    _noCache : null,
    /** to indicate if the list is doing animation
     * @private
     * @name _animating
     * @memberof accedo.ui.widget.List#
     */
    _animating : false,
    /** to indicate if it is a carousel list
     * @private
     * @name _isCarousel
     * @memberof accedo.ui.widget.List#
     */
    _isCarousel : false,
    /** 
     * @private
     * @name _scrollbar
     * @memberof accedo.ui.widget.List#
     */
    _scrollbar : null,
    /** the height of each item which use for animation to scroll with 
     * @private
     * @name _rowHeight
     * @memberof accedo.ui.widget.List#
     */
    _rowHeight : 0,
    /** the width of each item which use for animation to scroll with 
     * @private
     * @name _rowWidth
     * @memberof accedo.ui.widget.List#
     */
    _rowWidth : 0,
    /** forward the focus to the children, otherwise it will remain the focus ont the list
     * @private
     * @name _forwardFocus
     * @memberof accedo.ui.widget.List#
     */
    _forwardFocus:false,
    /**  To store which is checked
     * @private
     * @name _checkedIndex
     * @memberOf accedo.ui.widget.List#
     */
    _checkedIndex:-1,
    /** To store the css name of checked one
     * @private
     * @name _checkedCss
     * @memberOf accedo.ui.widget.List#
     */
    _checkedCss:"checked",
    /** To store the animation
     * @private
     * @name _animation
     * @memberOf accedo.ui.widget.List#
     */
    _animation: null,
    /** To check the state whether it is scrolling
     * @private
     * @name _scrolling
     * @memberOf accedo.ui.widget.List#
     */
    _scrolling:false,
    /** To count the number of children during scrolling process
     * @private
     * @name  _newChildrenNo
     * @memberOf accedo.ui.widget.List#
     */
    _newChildrenNo:0,
    /**
     * @constructor
     * @memberof accedo.ui.widget.List#
     * @function
     * @param {String | Object} opts String or an object
     * @private
     */
    initialize: function ($super, opts) {
        var nextRightDown, nextLeftUp;
        
        opts = opts || {};
        
        //Focusable unless explicitely specified otherwise
        if (opts.focusable !== false) {
            opts.focusable = true;
        }
        
        //Set default values of properties not set or incorrect
        if (!( opts.orientation)) {
            opts.orientation = accedo.ui.widget.List.HORIZONTAL;
        }
        else if (accedo.Object.isString(opts.orientation)) {
            opts.orientation = accedo.getNamespaceReference(opts.orientation);
        }
        
        //Set default values of undefined or incorrect properties
        if (!(opts.scrollbarBehaviour)) {
            opts.scrollbarBehaviour = accedo.ui.widget.List.SCROLL_BASE_ON_ITEM;
        }
        
        //Set default selected index to 0 unless it is defined
        if (typeof opts.firstSelectedIndex !== 'number') {
            opts.firstSelectedIndex = 0;
        }
        
        if (typeof opts.visibleSize === "undefined") {
            opts.visibleSize = 1;
        } else if (opts.visibleSize === 0) {
            this._isAllVisible = true;
        }
        
        if (!(opts.preloadSize) || opts.preloadSize < 0 ) {
            opts.preloadSize = 0;
        }
        
        if (!(opts.behaviour)) {
            opts.behaviour = accedo.ui.widget.List.REGULAR;
        }
        else if (accedo.Object.isString(opts.behaviour)) {
            opts.behaviour = accedo.getNamespaceReference(opts.behaviour);
        }
        
        //Behaviour-specific properties
        switch(opts.behaviour) {
            case accedo.ui.widget.List.REGULAR:
                if (!( opts.scrollStep) || opts.scrollStep > opts.visibleSize) {
                    opts.scrollStep = 1;
                }
                
                //Turn scrollBeforeBorder into an array if necessary
                if (typeof opts.scrollBeforeBorder === 'number') {
                    opts.scrollBeforeBorder = [opts.scrollBeforeBorder, opts.scrollBeforeBorder];
                } else if (!(opts.scrollBeforeBorder) || opts.scrollBeforeBorder.length !== 2) {
                    opts.scrollBeforeBorder = [0, 0];
                }
                this._isCarousel = false;
                break;
            case accedo.ui.widget.List.CAROUSEL:
                //Set the properties that are not for the user to define in a CAROUSEL
                opts.scrollStep = 1;
                this._isCarousel = true; //for convenience in the following code
                break;
        }
        
        opts.animation = opts.animation || false;
        opts.effect = opts.effect || {
            duration:500
        };
        
        if( accedo.platform === "huawei" ){
            opts.animation = false;
        }
		
        this._noCache = opts.noCache || false;
        this._forwardFocus = opts.forwardFocus || false;
        
        //Iterator used to generate unique IDs for the created child components
        this._iteratorChildId = 0;
        
        $super(opts);
        
        if (opts.orientation === accedo.ui.widget.List.HORIZONTAL) {
            this.root.addClass('accedo-ui-list accedo-ui-layout-linear-h');
        } else {
            this.root.addClass('accedo-ui-list accedo-ui-layout-linear-v');
        }
        
        //absolute position is for animation that setting top or left        
        this.container = accedo.Dom.element("div").setStyle({
            "position":"relative"
        });
        this.container.addClass("accedo-ui-list-div");
        if(this.opts.orientation === accedo.ui.widget.List.HORIZONTAL && this.opts.animation){
            //it will append 1 more item using animation,so it need to set 200% in order to show 
            //all the item during animation.E.g 3 visible size,when animation start,it will append one more,so it needs to set 200% in order to make the extra item 
            //on the same row in the horizontal list
            this.container.setStyle({
                "width":"200%"
            });
        }
        this._animating = false;
        this.root.appendChild(this.container);
        
        this.determineScrollingBehaviour();
    },
    /**
     * To handle the nextRight or nextDown. If there is available items on the right/down side, it will move the focus to the next one,otherwise it will use the original nextDown/nextRight which is defined in the view
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name _newRightDown
     */
    _newRightDown :function() {
        if(this._animating){
            //do nothing
            return true;
        }
        //Move selection forward
        if (!this._moveSelection(false)) {
            //Moving was impossible. Do we loop?
            if (!this._isCarousel && this.opts.loop && this.getCurrentSize()) {
                if (!this._waitForMoreData ){
                    //select the 1st element for looping
                    this.select(0);
                }
                return true;
            } else {
                return false;
            }
        }else{
            //can move along within the list
            return true;
        }
    },
    /**
     * To handle the nextLeft or nextUp. If there is available items on the left/up side, it will move the focus to the next one,otherwise it will use the original nextUp/nextLeft which is defined in the view
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name _newLeftUp
     */
    _newLeftUp :function() {
        if(this._animating){
            //do nothing
            return true;
        }
        //Move selection backward
        if (!this._moveSelection(true)) {
            //Moving was impossible. Do we loop?
            if (!this._isCarousel && this.opts.loop && this.getCurrentSize()) {
                if (!this._waitForMoreData){
                    //looping to the last item in datasource
                    this.select(this._datasource.getTotalItems()-1);
                }
                return true;
            }else {
                return false;
            }
        }else{
            //can move along within the list
            return true;
        }
    },
    /**
     * Correct the index if the list is a CAROUSEL and index is out of bounds
     * @returns Number the corrected index (or the original one if no correction is needed or the list is REGULAR)
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name _getCorrectedIndex
     */
    _getCorrectedIndex : function(index) {
        var dsSize = this._datasource && this._datasource.getCurrentSize();
        if (!dsSize) {
            return index;
        }
        if (this._isCarousel) {
            while (index >= dsSize) {
                index -= dsSize;
            }
            while(index < 0) {
                index += dsSize;
            }
        }
        return index;
    },
    /**
     * get visible Size
     * @returns {Number} the visible Size of the list. If options set to show all then the visible size is the total number in the datasource
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name getVisibleSize
     */
    getVisibleSize : function() {
        return (this._isAllVisible && this._datasource)?this._datasource.getCurrentSize():this.opts.visibleSize;
    },
    /**
     * get the current number of children
     * @returns {Number} the current number of children
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name getCurrentSize
     */
    getCurrentSize : function() {
        return this.getChildren().length;
    },
        
    /**
     * Shortcut function to get the first child of the list
     * @returns {accedo.ui.AbstractComponent} the current first child, or null
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name getFirstChild
     */
    getFirstChild : function() {
        if (this.getCurrentSize()) {  
            return this.getChildren()[0];
        }
        return null;
    },
        
    /**
     * get the last child of the list
     * @returns {accedo.ui.AbstractComponent} the current last child, or null
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name getLastChild
     */
    getLastChild : function() {
        var myLength = this.getCurrentSize();
        if (myLength) {  
            return this.getChildren()[myLength-1];
        }
        return null;
    },
    
    /**
     * get the n-th child of the list. e.g get the second visibl item in the list, getChildAtIndex(1)
     * @param {Number} index Index of the child to be returned
     * @returns {accedo.ui.AbstractComponent} the current n-th child, or null
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name getChildAtIndex
     */
    getChildAtIndex: function(index) {
        var myLength = this.getCurrentSize();
        if (myLength && index >= 0 && index < myLength) {  
            return this.getChildren()[index];
        }
        return null;
    }, 
    /**
     * This function groups the necessary actions to be done when a
     * child of this list is focused (so it is only is useful for focusable children).
     * Essentially what we do is, for ease of use, select this child in the list.
     * This allows selecting a child using the mouse instead of the keyboard...
     * for non-CAROUSEL lists only
     * @param {accedo.ui.AbstractComponent} child One of the list's children
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name onChildFocus
     */
    onChildFocus : function($super,child) {
        $super(child);
        //Select this child - not for carousels ! Controlling a carousel by mouse would be a headache to implement.
        if (!this._isCarousel) {
            if(this._forwardFocus){
                //to loop until it is children of the list
                while(accedo.Array.indexOf(this.getChildren(), child) === -1){
                    //unable to find the the child in the list
                    if(child === this){
                        return;
                    }
                    child = child.getParent();
                }
            }
            this.selectChild(child);
        }
    },

    /**
     * Function creating a new child ready to be put in the list, using some data.
     * Triggered by datasource events.
     * @param {Number} dsIndex Index of the data object in the datasource, to create this child
     * @returns {accedo.ui.AbstractComponent} The new child or null
     * @private 
     * @function
     * @memberof accedo.ui.widget.List#
     * @name _createChild
     */
    _createChild : function(dsIndex) {
        var data, childOptions, newChild;
        
        //Requirements check : childTemplate is an Object and has a function as its 'type' property
        if (! accedo.Object.isPlainObject(this.opts.childTemplate) && accedo.Object.isFunction(this.opts.childTemplate.type)) {
            return null;
        }
            
        //Can we get the data ?
        data = this._datasource.getDataAtIndex(dsIndex);
        if (data === null) {
            return null;
        }
        
        //Build the new child with its options and new generated id
        childOptions = accedo.Object.clone(this.opts.childTemplate);
        childOptions.id = this._generateChildId();
        
        newChild = accedo.ui.View.realize(childOptions);
        
        //We'll need this info to duplicate a child for carousels in some cases
        newChild.dsIndex = dsIndex;

        //Use the displayHelper, if any
        if (this.displayHelperFn) {
            this.displayHelperFn(newChild, data, dsIndex);
        }
   
        //Add newChild to the list cache
        this._setCache(dsIndex, newChild);
            
        return newChild;
    },
    
    /**
     * Attach a child to the DOM tree in the right place.
     * @param {accedo.ui.AbstractComponent} child
     * @param {boolean} isFirstPosition (optional) If true, append in first position. Default: false (last position)
     * @returns {accedo.ui.AbstractComponent} the attached child (usually the 1st param, but sometimes a new copy of it)
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name _attachChild
     */
    _attachChild : function(child, isFirstPosition) {
        var myChild = child;
        
        if (!isFirstPosition){
            isFirstPosition = false;
        }

        //If already attached, duplicate it (happens on carousels with a visibleSize bigger than the DS's currentSize)
        if (child.getRoot().isAttached()) {
            myChild = this._createChild(myChild.dsIndex);
        }
            
        //Use the container inherited method
        this.attach(myChild, isFirstPosition);

        return myChild;
    },
    /**
     * update the css with the id of the item (the order number in the list) e.g the css class will be accedo-ui-layout-linear-element item-0
     * @private
     * @function
     * @memberOf accedo.ui.widget.List#
     * @name _setItemCss
     */
    _setItemCss: function(){
        var i,length = Math.min(this.getChildren().length,this.opts.visibleSize) , child, cssClass;

        for( i = 0 ; i < length ; i++){
            child = this.getChildren()[i];
            child._listIndex = i;
            cssClass = "accedo-ui-layout-linear-element item-"+i;
            cssClass += (this._selectedChild === child)?" selected":"";
            cssClass += (this._checkedIndex === child.dsIndex)?" "+this._checkedCss:"";
            child.parentDiv.getHTMLElement().className = cssClass;
        }
    },
    /**
     * set Checked css to the item
     * @public
     * @function
     * @param {number} index the dsIndex to be selected
     * @param {number} (Optional) cssName the name of that css
     * @memberOf accedo.ui.widget.List#
     * @name setChecked
     */
    setChecked:function(index, cssName){
        var oldItem, newItem,firstIndex,lastIndex;
        this._checkedCss = cssName || this._checkedCss;
        firstIndex = (this.getFirstChild()?this.getFirstChild().dsIndex:0),
        lastIndex = (this.getLastChild()?this.getLastChild().dsIndex:0);
        //it is inside the current display
        if(this._checkedIndex>= firstIndex && this._checkedIndex <= lastIndex){
            oldItem = this._getCache(this._checkedIndex).child;
            if(oldItem){
                oldItem.parentDiv.removeClass(this._checkedCss);
            }
        }
        this._checkedIndex = index;
        //it is inside the current display
        if(index>= firstIndex && index <= lastIndex){
            newItem = this._getCache(this._checkedIndex).child;
            if(newItem){
                newItem.parentDiv.addClass(this._checkedCss);
            }
        }
    },
    /**
     * return the checked index
     * @public
     * @function
     * @param {number} the dsIndex of the checked item
     * @memberOf accedo.ui.widget.List#
     * @name getChecked
     */
    getChecked:function(){
        return this._checkedIndex;
    },
    /**
     * Create DIV to attach a child to the DOM tree in the right place 
     * @param {accedo.ui.AbstractComponent} child
     * @param {boolean} insert (optional) If true, append in first position. Default: false (last position)
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name doAttachElement
     * @override
     */
    doAttachElement: function(child, insert) {
        //Create a new slot for this child
        
        var div = accedo.Element.create('div'),divStyles = {}, height,width;
                
        child.parentDiv = div;
        child.getContainer = function(){
            return child.parentDiv;
        };
        //Insert the new child's container in the correct position
        if (this.getCurrentSize() && insert) {
            this.container.prependChild(div);
        } else {
            this.container.appendChild(div);
        }
        
        div.addClass('accedo-ui-layout-linear-element');
        //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        //@deprecate: it should be set through css class
        height = child.getOption('height');
        if (height) {
            divStyles.height = height;
        }
        width = child.getOption('width');
        if (width) {
            divStyles.width = width;
        }
        
        if (accedo.Object.isString(child.getOption('gravity'))) {
            child.setOption('gravity', accedo.getNamespaceReference(child.getOption('gravity')));
        }

        switch (child.getOption('gravity')) {
            case accedo.ui.widget.List.GRAVITY_LEFT:
                divStyles.textAlign = 'left';
                break;
            case accedo.ui.widget.List.GRAVITY_RIGHT:
                divStyles.textAlign = 'right';
                break;
            case accedo.ui.widget.List.GRAVITY_CENTER:
                divStyles.textAlign = 'center';
                break;
        }
        
        /*****************************************************************/
        child.attachTo(this, div, accedo.ui.AbstractComponent.PLACE_APPEND);

    },
    
    /**
     * Returns a string to be set as a child component's ID
     * We just prefix this list's ID by a meaningless counter and increment this latter one
     * @returns {string}
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name _generateChildId
     */
    _generateChildId : function() {
        if (!this.getId()){
            this.setId(accedo.getGuid() + "__xdk__");
        }
        return this.getId() + '_' + this._iteratorChildId++;
    },
    
    /**
     * Select one of the list's children (more precisely, add a 'selected' CSS class
     * to its DOM parent). This acts as the list's internal focus.
     * Beforehand it will unselect the previously selected child if any (ONLY if the child corresponding
     * to childId can be found in the list). 
     * Also fires the new selected child's onSelect method if any, unless skipCallback is true.
     * Be careful to select a component that is indeed a list's child when using this private function.
     * @param {accedo.ui.AbstractComponent} child ID of the child to be selected
     * @param {boolean} skipCallback When true, do not trigger the child's onSelect (set to true when
     *                      the child focus triggers its selection so as to avoid an infinite loop)
     * @returns {boolean} true if the selection was effectively changed
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name selectChild
     */
    selectChild : function(child, skipCallback , _options) {
        //Don't bother doing anything if the current selection is already OK or nothing is to be selected
        if (child && (child !== this._selectedChild)) {
            
            //Remove the previous selection if any
            this.unselect();
            //Select the given child
            this._selectedChild = child;
            //forward the focus when press left/right to move selection / scroll
            if(this._forwardFocus && ((_options && _options._focus) || this.isChildFocused())){
                this._selectedChild.setFocus();
            }
            this._selectedChild.setActive(true);
            if(this._selectedChild.getRoot().getParent()){
                child.getRoot().getParent().addClass('selected');
            }
            
            //dispatch event after the scrolling cases, other cases can do directly
            this._dispatchSelectEvent = function(){
                if (_options && _options._move) {
                    this.dispatchEvent(accedo.ui.Evt.SELECTION_CHANGED, {
                        reverse: _options._move
                    });
                }
                else {
                    this.dispatchEvent(accedo.ui.Evt.SELECTION_CHANGED);
                }
            
                //Try launching its onSelect method, unless specified otherwise
                if (!skipCallback) {
                    this._triggerOnSelect();
                }
            };
            
            if(!this._scrolling){
                this._dispatchSelectEvent();
                this._dispatchSelectEvent = null;
            }
            
            return true;
        }
        return false;
    },
    /**
     * Triggers the child's onSelect function if defined, unless specified otherwise
     * @private
     * @function
     * @memberof accedo.ui.widget.List#
     * @name _triggerOnSelect
     */
    _triggerOnSelect : function() {
        if (this._selectedChild && this._callbackOnSelect && accedo.Object.isFunction(this._selectedChild.onSelect)) {
            this._selectedChild.onSelect();
        }
        //Update the scrollbar after each selection change
        this._updateScrollbar();
    },
    /**
     * Unselect the current selection and launch its onUnselect callback, if any
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name unselect
     */
    unselect : function() {
        if (this._selectedChild) {
            if (accedo.Object.isFunction(this._selectedChild.onUnselect)) {
                this._selectedChild.onUnselect.call(this._selectedChild);
            }
			
            if(this._selectedChild.getRoot().getParent()){
                this._selectedChild.getRoot().getParent().removeClass('selected');
            }

        }
        this._selectedChild = null;
    },
    /**
     * onKey function that will implement the onKey of the child
     * @private
     * @protected
     * @function
     * @memberof accedo.ui.widget.List#
     * @name onKey
     */
    onKey: function($super, vKey, currentFocus){
        var fm, fmRet = false, leftUp, rightDown;
        $super(vKey, currentFocus);
        
        fm = accedo.ui.FocusManager.singleton();
        fmRet = fm.focusDirectionChangeByKey(vKey, this);
        
        if(fmRet){
            return true; 
        }

        if(this.opts.orientation === accedo.ui.widget.List.HORIZONTAL){
            leftUp = accedo.VKey.KEY_LEFT;
            rightDown = accedo.VKey.KEY_RIGHT;
        }else{
            leftUp = accedo.VKey.KEY_UP;
            rightDown = accedo.VKey.KEY_DOWN;
        }
        //to give back the focus from child to list
        if(!this._forwardFocus && currentFocus!== this){
            this.setFocus();
        }
        switch (vKey) {
            case leftUp:
                return this._newLeftUp();
            case rightDown:
                return this._newRightDown();
        }
    },
    /**
     * Scroll to the child with respect to the current selected one. It is only available to the child in the list now
     * @param {accedo.ui.AbstractComponent} child to be scrolled
     * @returns {boolean} true if it is scrolled.
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name scrollToChild
     */
    scrollToChild:function(child){
        if(this._animating){
            accedo.console.warn("it is unable to scroll to the child when it is animating.");
            return false;
        }
        
        if(child.getParent() !== this){
            accedo.console.warn("it is not belong to the list.Only children can be scrolled to.");
            return false;
        }
        
        var listIndex = child._listIndex,
        diff = listIndex - this.getSelectedIndex();
		
        if(!diff){
            accedo.console.warn("it is unable to scroll when it is already selected.");
            return false;
        }

        if(diff > 0){
            this.scroll(false,false,diff);
        }else{
            this.scroll(true,false,-diff);
        }
        return true;
    },
    /**
     * Scroll the current display forward by given parameter or default scrollStep.
     * Then takes cares of preloading data if needed.
     * @param {boolean} reverse (optional) When true, scroll backward. Otherwise scroll forward
     * @param {boolean} skipSelection (optional) If true, do not change the selection after the scroll. 
     * However, if the item is outside the list, it will auto select the border one. 
     * If false, it will change the focus to the new item at the same position (list index) of the current focus.
     * @param {Number} scrollStep (optional) If set, override the current scrollStep option
     * @returns {boolean} true if the list was scrolled at least once
     * @public
     * @function
     * @memberof accedo.ui.widget.List#
     * @name scroll
     */
    scroll : function(reverse, skipSelection, scrollStep) {
        //to prevent extra scroll when it is animating
        if(this._animating){
            return;
        }
        //if it is carousel, scroll must not skip selection change, otherwise focus will be wrong
        if(this._isCarousel){
            skipSelection = false;
        }
        var i, index, hasScrolled, child, preloadSize,border,removedChild, currentListIndex, newSelectionPos;

        scrollStep = scrollStep || this.opts.scrollStep || 1;
        preloadSize = this.opts.preloadSize || 0;
        hasScrolled = false;
        this._newChildrenNo = 0;
        currentListIndex = this.getSelection()._listIndex;
        
        //Each scrolling step means fetching a child and attaching it
        while(scrollStep--) {
            //Get the next child from the cache or create it
            index = this.getBorderIndex(reverse) + (reverse ? -1 : 1);
            child = this._fetchChild(index);
            if (child) {
                //Attach it
                this._scrolling = true;
                child = this._attachChild(child, reverse);
                
                //if more than an item, developer need to set the more specific css name inside
                // setAnimation function themselves in order to run the css3 animation
                child.getContainer().addClass("item-"+(reverse?"left":"right"));
                this._newChildrenNo++;
            } else {
                //The child could not be found, prevent further scrolling
                break;
            }
        }
		
      
		
        if(this._scrolling){
            //change the focus to the new child
            if (!hasScrolled) {
                if(skipSelection){
                    newSelectionPos = ( reverse ?currentListIndex+this._newChildrenNo:currentListIndex);
                    //outside the screen and then pick the border one
                    if(reverse && newSelectionPos >= this.opts.visibleSize ){
                        newSelectionPos = this.opts.visibleSize - 1;
                    }else if( newSelectionPos < this._newChildrenNo){
                        newSelectionPos = this._newChildrenNo;
                    }
                }else{
                    newSelectionPos = (reverse ? currentListIndex : currentListIndex +  this._newChildrenNo);
                }
				
                if(this.getChildren()[newSelectionPos]){
                    this.selectChild(this.getChildren()[newSelectionPos]);
                }
            }
            this._scrollReverse = reverse;
			
            //it will go through this step _startAnimation -> _animation doneAnimation -> doneAnimation
            //_animation is set in the controller and then developer need to call doneAnimation to tel the list
            this._startAnimation(reverse);
			
            hasScrolled = true;
        }
        
        if (hasScrolled) { 
            //Each preloading step means fetching a child (NOT attaching it)
            i = 1;
            while (preloadSize--) {
                this._fetchChild(this.getBorderIndex(reverse) + i++ * (reverse ? -1 : 1));
            }
            
            border = this.getBorderIndex(!reverse);
            if(this.opts.animation){
                if(!reverse){
                    border++;
                }else if(reverse && border>0){
                    border--;
                }
            }
            if(this._noCache){
                removedChild = null;
                //remove cache when it is outside the preload size
                if(!reverse){
                    if( border - this.opts.preloadSize -1 >= 0 ){
                        removedChild = this._getCache(border - this.opts.preloadSize -1);
                    }
                }else{
                    if(border + this.opts.preloadSize + 1 < this._cache.length  ){
                        removedChild = this._getCache(border + this.opts.preloadSize + 1);
                    }
                }

                if(removedChild && removedChild.child && removedChild.child.deinit){
                    removedChild.child.deinit();
                    this._removeCache(removedChild.child.dsIndex);
                    removedChild = null;
                }
            }        
        }
            
        return hasScrolled;
    },
    /**
    * it will go through this step _startAnimation -> _animation doneAnimation -> doneAnimation
    * _animation is set in the controller and then developer need to call doneAnimation to tel the list
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @name _startAnimation
    */
    _startAnimation:function(reverse){
        //since no animation available, then it is done animation
        if(!this.opts.animation){
            this.doneAnimation();
            return;
        }
        this._animating = true;
        //to check if set the animation and call end themselves
        if(this._animation){
            //sometimes it may not created completely
            accedo.Fn.delay(this._animation,0.01,reverse, this);
            return;
        }
        
        this._emileAnimation(reverse);
        
    },
    /**
    * the original emile Animation for backward compatible
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @name _emileAnimation
    */
    _emileAnimation:function(reverse){
        //default to use emile
        var childToRemove, element, afterEffect;
        childToRemove = reverse ? this.getLastChild() : this.getFirstChild();
                
        afterEffect = accedo.Fn.bind(function() {
            if(this.opts.orientation === accedo.ui.widget.List.VERTICAL){
                this.container.getHTMLElement().style.top = "0px";
            }else{
                this.container.getHTMLElement().style.left = "0px";
            }
            this.doneAnimation();
        },this); 
        
        //to get rowHeight/rowWidth everytime and depends on the remove item
        element = childToRemove.getRoot().getHTMLElement();
        if(!accedo.Object.isNumber(this.opts.rowHeight)){
            this._rowHeight = element.offsetHeight;
        }else{
            this._rowHeight = this.opts.rowHeight;
        }
        if(!accedo.Object.isNumber(this.opts.rowWidth)){
            this._rowWidth = element.offsetWidth; 
        }else{
            this._rowWidth = this.opts.rowWidth;
        }

        this.opts.effect.after = afterEffect;
        if(!reverse){
            //go down
            if(this.opts.orientation === accedo.ui.widget.List.VERTICAL){
                this.container.getHTMLElement().style.top = "0px";
                accedo.anim.Emile.anim(this.container.getHTMLElement(),'top:-'+this._newChildrenNo*this._rowHeight+'px',this.opts.effect);
            }else{
                //go right
                this.container.getHTMLElement().style.left = "0px";
                accedo.anim.Emile.anim(this.container.getHTMLElement(),'left:-'+this._newChildrenNo*this._rowWidth+'px',this.opts.effect);
            }
        }else{
            //go up
            if(this.opts.orientation === accedo.ui.widget.List.VERTICAL){
                this.container.getHTMLElement().style.top = '-'+this._newChildrenNo*this._rowHeight+'px';
                accedo.anim.Emile.anim(this.container.getHTMLElement(),'top: 0px',this.opts.effect);
            }else{
                //go left
                this.container.getHTMLElement().style.left = '-'+this._newChildrenNo*this._rowWidth+'px';
                accedo.anim.Emile.anim(this.container.getHTMLElement(),'left: 0px',this.opts.effect);
            }
        }
    },
    /**
    * To be called when the animation end, so that the list will continue the animation
    * @public
    * @function
    * @memberof accedo.ui.widget.List#
    * @name doneAnimation
    */
    doneAnimation:function(){
        while(this._newChildrenNo>0){
            var childToRemove = this._scrollReverse ? this.getLastChild() : this.getFirstChild();
            this.remove(childToRemove);
            this._newChildrenNo--;
        }
        
        //after scrolling, we have to ensure the selected children in the list
        if(!this._selectedChild.isAttached()){
            if(this._scrollReverse){
                this.selectChild(this.getFirstChild());
            } else {
                this.selectChild(this.getLastChild());
            }
        }
        
        this._animating = false;
        this._updateScrollbar();
        this._setItemCss();
        
        //Notify the list did scroll
        this.dispatchEvent(accedo.ui.Evt.SCROLL, {
            reverse: this._scrollReverse
        });
        
        this._scrolling = false;
        
        //to dispatch select event after scrolling
        if(this._dispatchSelectEvent){
            this._dispatchSelectEvent();
            this._dispatchSelectEvent = null;
        }
    },
    /**
    * set animation
    * @public
    * @function
    * @memberof accedo.ui.widget.List#
    * @name setAnimation
    */
    setAnimation:function(fn){
        this._animation = fn;
    },
    /**
    * unset animation
    * @public
    * @function
    * @memberof accedo.ui.widget.List#
    * @name resetAnimation
    */
    resetAnimation:function(){
        this._animation = null;
    },
    /**
    * Return the DS index of the data corresponding to the first child (if reverse), or to the last one
    * @param {boolean} reverse
    * @returns {Number}
    * @public
    * @function
    * @memberof accedo.ui.widget.List#
    * @name getBorderIndex
    */
    getBorderIndex : function(reverse) {
        return reverse ? (this.getFirstChild()?this.getFirstChild().dsIndex:0) : (this.getLastChild()?this.getLastChild().dsIndex:0);
    },
        
    /**
    * Adds a child to the cache Array.
    * @param {Number} index Index of the data represented by this child in the datasource
    * @param {accedo.ui.AbstractComponent} child to cache
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @name _setCache
    */
    _setCache : function(index, child) {
        //Keep a reference to the parent as the link is broken when we remove a child from a container.
        this._cache[index] = {
            child: child
        };
    },
        
    /**
    * Pops an element from the cache Array
    * @param {Number} index Index of the data represented by this child in the datasource
    * @returns {accedo.ui.AbstractComponent} the cached child or null
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @name _getCache
    */
    _getCache : function(index) {
        
        if (index >= 0 && this._cache.length > index) {
            return this._cache[index];
        }
        return null;
    },
    /**
    * remove an element from the cache Array
    * @param {Number} index Index of the data represented by this child in the datasource
    * @returns {boolean} true if removed, false if it doesn't exist
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @name _removeCache
    */
    _removeCache:function(index){
        if(this._getCache(index)){
            this._cache.splice(index,1,null);
            return true;
        }
        return false;
    },    
    /**
    * Gets a child from cache if possible, otherwise create it.
    * The requested index is corrected if the list has a CAROUSEL behaviour
    * @param {Number} index
    * @returns the child ready to be appended to the list, or null
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @name _fetchChild
    */
    _fetchChild : function(index) {
        var child, myCache;
            
        if (index < 0 && !this._isCarousel) {
            return null;
        }
            
        //prevent preloading of elements which are out of bounds
        if (!this._isCarousel && this._datasource.getTotalItems() > 0 && index >= this._datasource.getTotalItems()){
            return null;
        }
            
        //Correct the out of bounds index for a carousel
        index = this._getCorrectedIndex(index);
            
        //In the cache ?
        myCache = this._getCache(index);
        if (myCache) {
            child = myCache.child;
        }
        if (!child) {
            //Can be created using the datasource?
            child = this._createChild(index);
            if (!child) {
                //Could not be created, ask the DS for more data for the particular child
                this._askForMoreData(index);
            }
        }
            
        return child;
    },
    
    /**
    * Ask the datasource for more data, unless we've already asked
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @param {Number} dsIndex (optional) specifiy the ds position your want to get more data. If not specified it'll just ask for more data appending to the datasource
    * @name _askForMoreData
    */
    _askForMoreData : function(dsIndex) {
        if (!this._waitForMoreData) {
            //Notify we shouldn't ask for more data until we first receive some
            this._waitForMoreData = true;
            if (typeof dsIndex === 'number'){
                this._datasource.load({
                    dsIndex: dsIndex
                });
            }else{
                this._datasource.load();
            }
        }
    },
      
    /**
    * Set the logic base on different scrollbarBehaviour value, prevent if case usage in updateScrollbar in order
    * to enhance the performance.
    * @public
    * @returns {Boolean} Success of the setting behaviour
    * @function
    * @memberof accedo.ui.widget.List#
    * @name determineScrollingBehaviour
    */
    determineScrollingBehaviour: function() {
        switch (this.opts.scrollbarBehaviour) {
            case accedo.ui.widget.List.SCROLL_BASE_ON_ITEM:
                this._scrollCondition = 1;
                this._customizeScrolling = function () {
                    //some datasource doesn't know it's total size, we'll just use the current size for the scrollbar
                    var total = this._datasource.getCurrentSize();
                    if(total <= 1){
                        this._scrollbar.hide();
                    }else{
                        this._scrollbar.update(this._selectedChild.dsIndex, 1, total);
                    }
                };
                return true;
            case accedo.ui.widget.List.SCROLL_BASE_ON_PAGE:
                this._scrollCondition = this.getVisibleSize();
                this._customizeScrolling = function () {
                    //some datasource doesn't know it's total size, we'll just use the current size for the scrollbar
                    var total = this._datasource.getCurrentSize(),
                    scrollablePage = Math.ceil(total - (this.getVisibleSize())) + 1;
                    if(scrollablePage <= 1){
                        this._scrollbar.hide();
                    }else{
                        this._scrollbar.update(this.getFirstChild().dsIndex, 1, scrollablePage); // According to each page
                    }
                };
                return true;
        }
        return false;
    },
    
    /**
    * A decided function that to be overrided by determineScrollingBehaviour
    * @see determineScrollingBehaviour
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @name _customizeScrolling
    */
    _customizeScrolling: function() {
    // To be overrided.
    },

    /**
    * Updates the scrollbar: hides it when the list is childless or shows all of its children,
    * refreshes and shows it otherwise.
    * Could be simply called on attachChild and onRemove - but then there may be flickering when these
    * happen successively, so instead we call it after functions that affect the display (scroll,
    * size change, new elements received from the datasource)
    * @private
    * @function
    * @memberof accedo.ui.widget.List#
    * @ignore
    * @name _updateScrollbar
    */
    _updateScrollbar : function() {
        if (!this._scrollbar || !this._datasource || !this._selectedChild) {
            return;
        }
        //In some cases a scrollbar is not needed (0 or 1 child)
        if (this._datasource.getTotalItems() <= this._scrollCondition) {
            this._scrollbar.hide();
        } else {
            this._scrollbar.show();
            this._customizeScrolling();
        }
    },
    
    /**
    * @public
    * @function
    * @memberof accedo.ui.widget.List#
    * @name focus
    * @override
    */
    focus : function($super) {
        if (accedo.ui.FocusManager.singleton().isMouseOn()) {
            return false;
        }

        var ret, selectionChanged = this.ensureSelection();
        if(!this._forwardFocus){
            ret = $super();
        }
        if (!selectionChanged && this.opts.forceSelectOnFocus) {
            this._triggerOnSelect();
        }
        //User-defined additional callback
        if (accedo.Object.isFunction(this.additionalOnFocus)) {
            this.additionalOnFocus();
        }
        if(this._forwardFocus){
            if(this._selectedChild){
                this._selectedChild.setFocus();
            }
            //since list will pass the focus to its children and then it won't be the currentFocus
            return false;
        }else{
            return ret;
        }
    },    
    /**
    * Provides the user with a way of defining his own onFocus callback,
    * as setting the list's onFocus method would overwrite the above behaviour
    * @param {Function} fn User-defined onFocus callback 
    * @public
    * @function
    * @memberof accedo.ui.widget.List#
    * @name setAdditionalOnFocus
    */
    setAdditionalOnFocus : function(fn) {
        this.additionalOnFocus = fn;
    },
            
    /**
    * Ensures something is selected (if not, select the user-defined firstSelectedIndex
    * or the last child)
    * @public
    * @memberof accedo.ui.widget.List#
    * @function
    * @name ensureSelection
    * @returns {boolean} True when a new selection was set, false when it was not needed
    */
    ensureSelection : function() {
        if (!this._selectedChild) {
            this.selectChild(this.getChildAtIndex(this.opts.firstSelectedIndex) || this.getLastChild());
            return true;
        }
        return false;
    },
            
    /**
    * Overrided remove child function.
    * @param {accedo.ui.AbstractComponent} child Child to be removed
    * @param {Number} previousIndex Index of the child before it was removed
    * @param {boolean} removeAll Are we removing all the children ? (previousIndex is meaningless then)
    * @public
    * @memberof accedo.ui.widget.List#
    * @function
    * @name remove
    * @override
    */
    remove: function($super, child, previousIndex, removeAll) {
        //First, unselect the current selection, otherwise it will automatically select
        //another one when removed, and then trigger an unwanted onSelect
        var i = 0, valid = false, len = this.getChildren().length;
        
        if (removeAll) {
            this.unselect();
        }
        //Now we can assume we're removing one element only, and the children Array has been updated
        //(Not true for removeAll, but we've treated that matter already).
                
        //If it was the selected child, change the selection to another child...
        if (this.getChildren().length) {
            if (child === this._selectedChild) {
                if (previousIndex === 0) {
                    this.selectChild(this.getFirstChild());
                } else {
                    this.selectChild(this.getLastChild());
                }
            }
        } else { //...Unless there is nothing to select!
            this.unselect();
        }
                
        //Remove parent DIV
        if (child.getRoot().getParent()) {
            child.getRoot().getParent().remove();
        }

        $super(child);
    },
    
    /**
    * Overrided remove all function.
    * @public
    * @memberof accedo.ui.widget.List#
    * @function
    * @name removeAll
    * @override
    */
    removeAll: function($super) {
        $super();
        this.container.removeAll();
        this._animating = false;
    },
            
    /**
    * Move the list selection to the next child, if possible.
    * Move forward by default, backward if reverse is set to true
    * @param {boolean} reverse When true, move backward
    * @returns {boolean} true if the selection changed
    * @private
    * @memberof accedo.ui.widget.List#
    * @function
    * @name _moveSelection
    */
    _moveSelection: function(reverse) {
        var pos, newSelectionPos, hasMoved, newSelection;
        if (!this._selectedChild) {
            return false;
        }
                
        //Find out whether the new selection is currently displayed or not
        pos = accedo.Array.indexOf(this.getChildren(), this._selectedChild);
        newSelectionPos = pos + (reverse ? -1 : 1);
                
        if (newSelectionPos < 0 || newSelectionPos === this.getChildren().length) {
            //Moving to an element that is not displayed - scroll needed
            hasMoved = this.scroll(reverse);
        } else {
            //Moving to a displayed element
            newSelection = this.getChildren()[newSelectionPos];
                    
                    
            /* Any selection change scrolls the list if it is a carousel
                    (do it before selecting the element for a better user experience as
                    onSelect callbacks can be time consuming). A regular list can also scroll
                    to another displayed child, according to opts.scrollBeforeBorder
                 */
            if (this._isCarousel ||
                (reverse && pos <= this.opts.scrollBeforeBorder[0]) ||
                (!reverse && (this.getCurrentSize() - 1 - pos) <= this.opts.scrollBeforeBorder[1])
                ) {
                this.scroll(reverse, true);
            }
                    
            this.selectChild(newSelection, undefined, {
                _move : reverse,
                _focus : true
            });
                    
            hasMoved = true;
        }
                
        if (hasMoved) {
            //Notify the list did move (some UI effects can be triggered based on that)
            this.dispatchEvent(accedo.ui.Evt.MOVED, {
                reverse: reverse
            });
        }

        return hasMoved;
    },
            
    /**
         * Added by request - sets the selection to one of the displayed children by indicating
         * its current index in the list (previously this function name was 'setSelection')
         * @param {Number} childIndex index of the displayed child to select
         *      (N will select the N-th child, use -1 for the last one)
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name setSelectedIndex
         */
    setSelectedIndex: function(childIndex) {
        var newSelection;
                
        if (childIndex === -1) {
            newSelection = this.getLastChild();
        } else if (childIndex >= 0 && childIndex < this.getCurrentSize()) {
            newSelection = this.getChildren()[childIndex];
        }
                
        //This won't bug if newSelection was not set
        this.selectChild(newSelection);
    },
            
    /**
         * Returns the list index of the selected child (relatively to displayed children only)
         * @returns {Number} the list index of the selected child, or -1
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name getSelectedIndex
         */
    getSelectedIndex: function() {
        var result = -1;
                
        if (!this._selectedChild) {
            return result;
        }
        accedo.Array.each(this.getChildren(), function(child, i) {
            if (this._selectedChild === child) {
                result = i;
                throw accedo.$break;
            }
        }, this);
                
        return result;
    },      
    /**
         * Returns the currently selected child, or null
         * @returns {accedo.ui.AbstractComponent} The currently selected child, or null
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name getSelection
         */
    getSelection: function() {
        return this._selectedChild;
    },      
    /**
         * This function returns the behaviour of this layout.
         * @see accedo.ui.widget.List.REGULAR
         * @see accedo.ui.widget.List.CAROUSEL
         * @returns {Integer} The behaviour
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name getBehaviour
         */
    getBehaviour: function() {
        return this.opts.behaviour;
    },
            
    /**
         * This function returns the orientation of this layout.
         * @see accedo.ui.widget.List.VERTICAL
         * @see accedo.ui.widget.List.HORIZONTAL
         * @returns {Integer} The orientation
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name getOrientation
         */
    getOrientation: function() {
        return this.opts.orientation;
    },
          
    /**
         * Returns the currently scrollbarBehaviour
         * @public
         * @returns {Integer} The currently scrollbarBehaviour constant
         * @function
         * @memberof accedo.ui.widget.List#
         * @name getScrollbarBehaviour
         */
    getScrollbarBehaviour: function() {
        return this.opts.scrollbarBehaviour;
    },
    
    /**
         * Sets the scrollbarBehaviour value in order to change the scrollbar scrolling behaviour.
         * @public
         * @param {Integer} behaviour Set this.opts.scrollbarBehaviour
         * @returns {Boolean} Success of the setScrollbarBehaviour
         * @function
         * @memberof accedo.ui.widget.List#
         * @name setSelectedPosition
         */
    setScrollbarBehaviour: function(behaviour){
        this.opts.scrollbarBehaviour = behaviour;
        return this.determineScrollingBehaviour();
    },
    
    /**
         * Changing the visibleSize option dynamically.
         * The currently selected element (if any) will not change and will not be removed.
         * @param {Number} size
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name setVisibleSize
         */
    setVisibleSize: function(size) {
        var child, 
        diff = this.getVisibleSize() - size;
                
        //Don't do anything if: Same size than before, illegal argument, no datasource linked or no current children 
        if (!diff || size < 0 || !this._datasource || !this.getCurrentSize()) {
            return;
        }
                
        //From now on use the effective visible size (as the list may have less children than opts.visibleSize)
        diff = this.getChildren().length - size;
                
        //If everything was visible
        if (this.getVisibleSize() === 0) {
            //Notify it's not the case any more
            this._isAllVisible = false;
        } else if (size === 0) {
            //If everything is to be visible, notify so
            this._isAllVisible = true;
            //Calculate the current size difference for easier calculations
            diff = this.getVisibleSize() - this._datasource.getCurrentSize();
        }
                
        //The list may grow...
        if (diff < 0) {
            //Add elements at the forward side
            while (diff !== 0 && (child = this._fetchChild(this.getLastChild().dsIndex + 1))) {
                diff++;
                this._attachChild(child);
            }
            //Add elements at the backward side
            while (diff !== 0 && (child = this._fetchChild(this.getFirstChild().dsIndex - 1))) {
                diff++;
                this._attachChild(child, true);
            }
        } else if (diff > 0) { //... or it may shrink
            //Remove elements at the forward side if not the selected child, backward side otherwise
            while (diff !== 0) {
                diff--;
                this.remove(this.getLastChild() !== this._selectedChild ? this.getLastChild() : this.getFirstChild());
            }
        }
                
        this.opts.visibleSize = size;
        if (this.opts.visibleSize === 0) {
            this._isAllVisible = true;
        }
    },
            
    /**
         * Changing the firstSelectedIndex option.
         * @param {Number} index (index of the child in the list's children Array)
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name setFirstSelectedIndex
         */
    setFirstSelectedIndex: function(index) {
        this.opts.firstSelectedIndex = index;
    },
       
    /**
         * Activates or deactivates the automatic triggering of the children's onSelect function
         * when it gets selected in the list.
         * Usefull if you want to scroll or set a new selection without triggering any
         * of the children onSelect callbacks.
         * Remember to re-activate it !
         * @param {boolean} isActivated True to (re)activate the children onSelect callbacks.
         *                              False to deactivate it.
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @activateCallbackOnSelect
         */
    activateCallbackOnSelect: function(isActivated) {
        this._callbackOnSelect = isActivated;
    },
            
    /**
         * Deletes the cache's contents and reset the cache to an empty Array
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name purgeCache
         */
    purgeCache : function() {
        var i = 0,
        len = this._cache.length;
                
        //deinit the children
        for (; i < len; i++) {
            if (this._cache[i] && this._cache[i].child && this._cache[i].child.deinit){
                this._cache[i].child.deinit();
            }
        }
                
        this._cache.length = 0;
    },
            
    /**
         * Selects an element by its index in the DS (usually you wil use the DS's search function to find the wanted index)
         * Sometimes it isn't possible: if no datasource has been set yet or its size is 0, the selection will apply
         * when a datasource with data is set or when it receives data for the first time.
         * Otherwise if the index is out of the DS bounds, nothing will happen.
         * Optionnally you can decide where the new selection will appear in the list, using listIndex.
         * @param {Number} dsIndex Index, in the DS, of the data to select or -1 (last item)
         * @param {Number} listIndex (optional) If the search is successful, the list will try to display
         *  its newly selected child as the one in the specified index. If this option is set to undefined or null,
         *  listIndex will be set to :
         *      -(for a carousel) the value of the 'firstSelectedIndex' option used at list creation time.
         *      -(otherwise) the highest possible value.
         * @returns {boolean} True if the selection was successful
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name select
         */
    select: function(dsIndex, listIndex) {
        var child,tempChild, elementsToAdd;

        //No datasource or the datasource doesn't contain the data we want -> apply the select later
        if (!this._datasource || (-1 !== dsIndex && this._datasource.getDataAtIndex(dsIndex) === null)) {
            //prevent double selection which would easily trigger by pressing the left/right button on looping
            if (this._selectRequest && this._selectRequest.dsIndex === dsIndex){
                return false;
            }else{
                // Keep this request in memory to launch it when a datasource is set
                this._selectRequest = {
                    dsIndex: dsIndex, 
                    listIndex: listIndex
                };
            }
            //Request for the specified data if we did not have enough
            if(this._datasource){
                this._datasource.load({
                    dsIndex: dsIndex
                });
            }
            return false;
        }
                
        //clear the select request
        this._selectRequest = null;
                
        elementsToAdd = this.getVisibleSize() || this._datasource.getCurrentSize();

        //...or if it is already selected
        if (this._selectedChild && dsIndex === this._selectedChild.dsIndex) {
            return true;
        }
                
        //Correct the listIndex parameter as explained above
        if (!listIndex && listIndex !== 0) {
            if (this._isCarousel) {
                listIndex = this.opts.firstSelectedIndex;
            } else if (this._isAllVisible) {
                listIndex = this._datasource.getCurrentSize() - 1;
            } else {
                listIndex = this.getVisibleSize() - 1;
            }
        }
                
        //Refreshing the list display
        //Step 1: Remove all children
        this.removeAll();
        //Step 2: Add and select the child that matches the search
        child = this._fetchChild(dsIndex === -1 ? this._datasource.getCurrentSize() - 1 : dsIndex);
        this._attachChild(child);
        elementsToAdd--;
        //Step 3: Add the proper number of elements at the backward side
        while (listIndex && elementsToAdd && (tempChild = this._fetchChild(this.getFirstChild().dsIndex - 1))) {
            elementsToAdd--;
            listIndex--;
            this._attachChild(tempChild, true);
        }
                
        //Step 4: Add the proper number of elements at the forward side
        while (elementsToAdd && (tempChild = this._fetchChild(this.getLastChild().dsIndex + 1))) {
            elementsToAdd--;
            this._attachChild(tempChild);
        }
                
        //Step 5: Add some again at the backward side if not enough could be added forward
        while (elementsToAdd && (tempChild = this._fetchChild(this.getFirstChild().dsIndex - 1))) {
            elementsToAdd--;
            this._attachChild(tempChild, true);
        }
        this.selectChild(child,undefined,{
            _focus:true
        });
        this._updateScrollbar();
        this._setItemCss();
        return true;
    },         
    /**
         * Associates a datasource to this list and listens to it.
         * Also does the needed operations if another datasource was previously defined.
         * @param {accedo.data.Ds} ds
         * @example var myDS = myAPI.getMovies('a1234'); // Supposing it returns a DS !
         *          list.setDataSource(myDS);
         *          myDS.load();
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name setDatasource
         */
    setDatasource: function(ds) {
        //If the given datasource is already registered for this list, do nothing
        if (this._datasource === ds) {
            return;
        }
                
        //Remove the previous datasource, if any
        if (this._datasource) {
            this._datasource.removeEventListener(accedo.data.Ds.EVT_UPDATED, this._onDatasourceHandling, this);
        }
                
        //Remove all the current children (method inherited from container)
        this.removeAll();
                
        //Clear the list cache
        this.purgeCache();
        
        //Reset the selected child
        this._selectedChild = null;
                                
        //Register the new datasource
        this._datasource = ds;
        
        if (ds) {
            if(this.opts.loop && ds.hasPagination()){
                accedo.console.warn("loop is not support in pagination datasource.");
            }
            //Notify we can ask for even more data when needed
            this._waitForMoreData = false;

            //Listen for more data to come
            ds.addEventListener(accedo.data.Ds.EVT_UPDATED, this._onDatasourceHandling, this);

            //Append the existing data (as we missed the previous events)
            this._onDatasourceAppend({
                newData: ds.getData(), 
                startIndex: 0
            });
            this._setItemCss();
        }

    },
    /**
         * This function is to determine the datasource handling and separate into different functions like append,remove or insert data
         * @param {Object} evt Object with operation done, modified index current data length and expected data length after operation
         * @private
         * @function
         * @memberof accedo.ui.widget.List#
         * @name onDatasourceUpdated
         */
    _onDatasourceHandling:function(evt){
        switch(evt.action){
            case "Insert":
                this._onDatasourceInsert(evt);
                this._setItemCss();
                break;
            case "Remove":
                this._onDatasourceRemove(evt);
                this._setItemCss();
                break;
            case "Append":
                this._onDatasourceAppend(evt);
                this._setItemCss();
                break;
            case "Reset":
                this._onDatasourceReset();
                break;
            default:
                accedo.console.log("No related datasource handling");
                break;
        }
    },
    /**
         * This function is designed to remove all the data and completely remove all the data and waiting for appendData if need further
         * @private
         * @function
         * @memberof accedo.ui.widget.List#
         * @name _onDatasourceReset
         */
    _onDatasourceReset:function(){
        this.removeAll();
        this.empty();
        this._updateScrollbar();
    },
    /**
         * This function is designed to append the data in the list by accedo.data.Ds.EVT_UPDATED and with evt.action "Append"
         * @param {Object} evt Object with operation done, modified index current data length and expected data length after operation
         * @private
         * @function
         * @memberof accedo.ui.widget.List#
         * @name _onDatasourceAppend
         */
    _onDatasourceAppend:function(evt){
        var i,child,visibleSize,borderIndex,noUpdate,lastSelectIndex,predictItemLength,borderLastIndex,itemAddBefore,newDataLen,k,appendIndex,preloadSize;
        
        newDataLen = evt.newData.length; // The length of the new insert data
        appendIndex = evt.startIndex; // The insert position
        itemAddBefore = this._datasource.getCurrentSize() - newDataLen;
        noUpdate = true; // True means we do not need visual update
        visibleSize = this.getVisibleSize() || 0; // The number of visible childs in the list
        lastSelectIndex = this.getSelectedIndex(); // The focused item index before this operation                
        borderIndex = this.getBorderIndex(true); 
        borderLastIndex = this.getBorderIndex(false);
        predictItemLength = (!visibleSize?newDataLen:visibleSize);

        //Step 1: To check whether have existing one item and need to be modified
        this._waitForMoreData = false;
        for (i = 0; i < newDataLen; i++) {
            
            if(appendIndex+i<itemAddBefore){
                this._cache.splice(appendIndex, 1,null);
                if(!this._noCache){
                    //update with new cache if it has inside the datasource
                    this._fetchChild(appendIndex+i);
                }
                //if there are less than visible Size and appendIndex is inside the currentSize to indiciate it need to redraw
                if(itemAddBefore < visibleSize && appendIndex+i< visibleSize){
                    noUpdate = false;
                //if it is inside the current show items
                }else if(appendIndex+i >= borderIndex && appendIndex+i <= borderLastIndex){
                    noUpdate = false;
                }else if(appendIndex+i === itemAddBefore){
                    break;
                }
            }else{
                break;
            }
        }

        //Step 2: To check whether have to insert within screen
        for (; i < newDataLen; i++) {
            this._cache.splice(appendIndex + i, 0, null);
            if(!this._noCache){
                this._fetchChild(appendIndex+i);
            }
            
            if(this._isCarousel){
                if((appendIndex+i >= borderIndex && appendIndex+i <= borderLastIndex) || (appendIndex+i >= borderIndex && borderLastIndex<=borderIndex)){
                    noUpdate = false;
                }else if(!visibleSize || itemAddBefore < visibleSize && appendIndex+i< visibleSize){
                    //if there are less than visible Size and appendIndex is outside the currentSize
                    noUpdate = false;
                }
            }else{
                if(!visibleSize || (itemAddBefore < visibleSize && appendIndex+i< visibleSize)){
                    noUpdate = false;
                }
            }
        }

        if(!noUpdate){
            this.removeAll();
            // Step 1b: Attach all childs back to the list
            for (i = 0; i < predictItemLength; i++) {
                k = this._getCorrectedIndex(borderIndex+i);
                child = this._fetchChild(k);
                if (child) {
                    child = this._attachChild(child, false);
                }
            }

        }

        this.unselect();
        if(lastSelectIndex === -1){
            lastSelectIndex = this.opts.firstSelectedIndex;
        }
        this.setSelectedIndex(lastSelectIndex);
        this._updateScrollbar();
        
        this._ensurePreload();
        
        //Treat the previously requested 'select' call that couldn't be treated at that time
        if (this._selectRequest) {
            this.select(this._selectRequest.dsIndex, this._selectRequest.listIndex);
        }
    },
    /**
         * To ensure the preload items are preloaded
         * @private
         * @function
         * @memberof accedo.ui.widget.List#
         * @name _ensurePreload
         */
    _ensurePreload:function(){
        var preloadSize = this.opts.preloadSize,i = 1;
        while (preloadSize--) {
            var frontIndex = this.getBorderIndex(true) + i*-1,
            endIndex = this.getBorderIndex(false) + i*1;
            i++;
            if(frontIndex > 0  && frontIndex < this._datasource.getCurrentSize()){
                this._fetchChild(frontIndex);
            }
            
            if(endIndex > 0  && endIndex < this._datasource.getCurrentSize()){
                this._fetchChild(endIndex);
            }
        }
    },
    /**
         * This function is designed to remove the data in the list by accedo.data.Ds.EVT_UPDATED and with evt.action "Remove"
         * @param {Object} evt Object with operation done, modified index current data length and expected data length after operation
         * @private
         * @function
         * @memberof accedo.ui.widget.List#
         * @name _onDatasourceRemove
         */
    _onDatasourceRemove:function(evt){
        var i,j,child,visibleSize,childToRemove,loopTime,borderIndex,startIndex,modiLen,targetIndex,noUpdate,numOfItemToAdd,modiIndex,lastSelectIndex,predictItemLength,
        borderLastIndex,itemAddBefore,loopingIndex,count,bb,k;

        i = 0;
        j = 0;// Looping parameter
        noUpdate = true;// True means we do not need visual update
        numOfItemToAdd = 0; // A counter to store how many child we need to add it back to the list
        modiIndex = evt.index; // The array of dsIndex that we are going to remove, sorted in descending order
        lastSelectIndex = this.getSelectedIndex(); // The focused item index before this operation
        predictItemLength = evt.totalItemAfter; // Expected number of child will be available in the list
        startIndex = modiIndex[modiIndex.length - 1]; // Get the smallest dsIndex that we are going to remove
        modiLen = modiIndex.length;

        visibleSize = this.getVisibleSize() || 0;
        borderIndex = this.getBorderIndex(true);
        borderLastIndex = this.getBorderIndex(false); // The dsIndex of the last child
        
        if(this._isCarousel && borderIndex>startIndex){
            loopTime = this._datasource.getCurrentSize()+1-borderIndex+borderLastIndex-startIndex;
        }else{
            loopTime = borderIndex + visibleSize - startIndex;
        }
        targetIndex = startIndex - borderIndex;


        // Step 1: Determine is visual update need to perform
        for (i = 0; i < modiLen; i++) {
            if (((borderLastIndex >= modiIndex[i]) && (modiIndex[i] >= borderIndex)) || (this._isCarousel && borderLastIndex<borderIndex && modiIndex[i]<=borderLastIndex)) {
                noUpdate = false;
                break;
            }
        }

        // Step 1a: Determine can this list show all the child, if yes, special handling
        if (predictItemLength <= visibleSize){
            // Inside this case, the list is able to display all the childs after remove operation done
            // Step 1a-1: Detaching all child
            this.removeAll();

            // Step 1a-2: Clear the list cache
            this.purgeCache();
            if(this._isCarousel){
                predictItemLength = visibleSize;
            }
            // Step 1a-3: Attach all childs back to the list
            for (i = 0; i < predictItemLength; i++) {
                k = this._getCorrectedIndex(i);
                child = this._fetchChild(k);
                if (child) {
                    child = this._attachChild(child, false);
                }
            }

            // Step 1a-4: Focus last focused items
            if (lastSelectIndex > predictItemLength - 1) {
                // The last focused index is larger than the total number of available child
                // hence we can not focus that child because it is not exist, we then modify the lastSelectIndex to the last focusable child index
                lastSelectIndex = predictItemLength - 1;
            }
            this.setSelectedIndex(lastSelectIndex); // Select the child

            // Step 1a-5: Force it to hide the scrollbar
            this._updateScrollbar();
            this._ensurePreload();
            return; // break this switch case since all the procedure done
        }


        // Step 2: Total available number of child is larger than the maximum children can be visible in this list.
        //         Then we detach all child that may modified (i.e. the dsIndex larger than the smallest modified dsIndex
        if (!noUpdate) {
            if (loopTime > visibleSize) { // In some case, we remove a lot of children, as a result that this loopTime variable will be 
                // larger than the visibleSize, bound this loopTime to visibleSize
                // e.g. We remove 7 items in a list that can only display 5 children per page
                loopTime = visibleSize;
            }
            if(borderIndex>startIndex){
                targetIndex = this._datasource.getCurrentSize()+1-borderIndex+startIndex;
            }else{
                targetIndex = startIndex - borderIndex;
            }
            if (loopTime > 0){
                while (loopTime--) {
                    childToRemove = this.getChildAtIndex(targetIndex);
                    if (childToRemove) {
                        this.remove(childToRemove);
                        numOfItemToAdd++;
                    }
                } 
            }
        }
        //if(!this._noCache){
        // Step 3: Purge the unwanted cache only
        for (i = 0; i < modiIndex.length; i++) {
            if (this._getCache[modiIndex[i]] && this._cache[modiIndex[i]].child) {
                this._cache[modiIndex[i]].child.deinit();
            }
            this._cache.splice(modiIndex[i],1);
        }

        // Step 3a: Reschedule the dsIndex
        for (j = startIndex; j < this._cache.length; j++) {
            if(this._cache[j] && this._cache[j].child){
                this._cache[j].child.dsIndex = j;
            }
        }
        //}
        // Step 4: Add the child before the unremove child, 
        //         this may occurs due to there is not enough childs available after the affected dsIndex
        //         If there is not enough item to display, special handling
        if (predictItemLength > visibleSize && !this._isCarousel) {
            // Enough childs to fill all the list
            if (numOfItemToAdd + borderIndex > predictItemLength - 1) {
                count = numOfItemToAdd + borderIndex - predictItemLength + 1;
                // Insert child at the beginning
                for (j = 0; j < count; j++) {
                    child = this._fetchChild(borderIndex - (visibleSize - numOfItemToAdd) - j);
                    if (child) {
                        child = this._attachChild(child, true);
                    }
                }
                numOfItemToAdd = numOfItemToAdd - count;
            }
        } else if(!this._isCarousel){
            // Special handling
            itemAddBefore = 0;
            if (borderIndex !== 0) {
                // First display item was not ds first item
                bb = borderIndex - 1;
                for (; bb >= 0; bb--) {
                    child = this._fetchChild(bb);
                    if (child) {
                        child = this._attachChild(child, true);
                        itemAddBefore++;
                    }
                }
            }
            numOfItemToAdd = predictItemLength - (visibleSize - numOfItemToAdd) - itemAddBefore;
        }

        // Step 5: According the number of item we detach, add new child back 
        if (!noUpdate) {
            // Determine how many child we need to add it back
            if (predictItemLength >= visibleSize) {
                loopingIndex = borderIndex + visibleSize - numOfItemToAdd;
                if(this._isCarousel && borderLastIndex<borderIndex){
                    loopingIndex--;
                }
            } else {
                loopingIndex = borderIndex + predictItemLength - numOfItemToAdd;
            }
            if (numOfItemToAdd > 0) {
                while (numOfItemToAdd--) {
                    loopingIndex = this._getCorrectedIndex(loopingIndex);
                    child = this._fetchChild(loopingIndex);
                    child = this._attachChild(child, false);
                    loopingIndex++;
                }
            }
        }

        // Step 6: Focus last focused item position
        if (lastSelectIndex > predictItemLength - 1) {
            lastSelectIndex = predictItemLength - 1;
        }
        this.setSelectedIndex(lastSelectIndex);

        // Step 7: Update the scrollbar
        this._updateScrollbar(); 
        this._ensurePreload();
    },
    /**
         * This function is designed to insert the data in the list by accedo.data.Ds.EVT_UPDATED and with evt.action "Insert"
         * @param {Object} evt Object with operation done, modified index current data length and expected data length after operation
         * @private
         * @function
         * @memberof accedo.ui.widget.List#
         * @name _onDatasourceInsert
         */
    _onDatasourceInsert: function (evt) {
        var i,j,child,visibleSize,childToRemove,loopTime,borderIndex,targetIndex,noUpdate,numOfItemToAdd,lastSelectIndex,previousNumberOfItem,predictItemLength,
        borderLastIndex,loopingIndex,insertIndex,newDataLen,k;

        newDataLen = evt.newData.length; // The length of the new insert data
        insertIndex = evt.index; // The insert position
        noUpdate = true; // True means we do not need visual update
        numOfItemToAdd = 0; // A counter to store how many child we need to add it back to the list
        visibleSize = this.getVisibleSize() || 0; // The number of visible childs in the list
        lastSelectIndex = this.getSelectedIndex(); // The focused item index before this operation
        previousNumberOfItem = evt.totalItemBefore; // The number of available childs before the operation
        predictItemLength = evt.totalItemAfter; // Expected number of child will be available in the list

        // For empty list
        // Step 1: Is the list empty before insert anything?
        if (previousNumberOfItem === 0) {
            // Empty list, call another function to handle this case
            this._onDatasourceAppend({
                newData: evt.newData,
                startIndex: 0
            });
            this._setItemCss();
            return; // Break this operation
        }

        borderIndex = this.getBorderIndex(true); 
        borderLastIndex = this.getBorderIndex(false);
        if(this._isCarousel && borderIndex>insertIndex){
            loopTime = visibleSize - (this._datasource.getCurrentSize()-1-borderIndex+insertIndex);
        }else{
            loopTime = borderIndex + visibleSize - insertIndex;
        }

        // Step 1a: If the list can show all the childs, special handling
        if (predictItemLength <= visibleSize) {
            // Step 1a-1: Detaching all child
            this.removeAll();

            // Step 1a-2: Modified the cache
            for (i = 0; i < newDataLen; i++) {
                this._cache.splice(insertIndex + i, 0, null); 
                if(!this._noCache){
                    this._fetchChild(insertIndex + i);
                }
            }

            // Step 1a-3: Reschedule the dsIndex in the cache
            for (j = 0; j < this._cache.length; j++) {
                if(this._cache[j] && this._cache[j].child){
                    this._cache[j].child.dsIndex = j;
                }
            }

            if(this._isCarousel){
                predictItemLength = visibleSize;
            }
            // Step 1a-4: Attach all childs back to the list
            for (i = 0; i < predictItemLength; i++) {
                k = this._getCorrectedIndex(i);
                child = this._fetchChild(k);
                if (child) {
                    child = this._attachChild(child, false);
                }
            }
            this.unselect();
            // Step 1a-5: Focus last focus items
            if(lastSelectIndex === -1){
                lastSelectIndex = this.opts.firstSelectedIndex;
            }
            this.setSelectedIndex(lastSelectIndex);

            // Step 1a-6: Hide the scrollbar;
            this._updateScrollbar();
            this._ensurePreload();
            return;
        }

        // Step 2: Check the insertIndex is currently in display
        if(this._isCarousel && ((insertIndex >= borderIndex && insertIndex <= borderLastIndex) || (insertIndex <= borderLastIndex && borderLastIndex<=borderIndex) || (insertIndex >= borderIndex && borderIndex <= borderLastIndex))){
            noUpdate = false;
        }else if ((borderLastIndex >= insertIndex) && (insertIndex >= borderIndex)) {
            noUpdate = false;
        }

        // Step 3: Detach all child after the insertIndex
        if (!noUpdate) {
            if(borderIndex>insertIndex){
                targetIndex = this._datasource.getCurrentSize()-1-borderIndex+insertIndex;
            }else{
                targetIndex = insertIndex - borderIndex;
            }
            if (loopTime > 0){
                while (loopTime--) {
                    childToRemove = this.getChildAtIndex(targetIndex);
                    if (childToRemove) {
                        this.remove(childToRemove);
                        numOfItemToAdd++;
                    }else if(targetIndex < visibleSize){
                        numOfItemToAdd++;
                    }
                } 
            }
        }

        // Step 4: Modify the cache
        for ( i = 0; i < newDataLen; i++) {
            this._cache.splice(insertIndex + i, 0, null);
            if(!this._noCache){
                this._fetchChild(insertIndex + i); // Insert an empty Object into the cache, preventing overwriting existing object
            }
        }

        // Step 5: Attach all child after the insertIndex
        if (!noUpdate) {
            if(this._isCarousel && borderIndex>insertIndex){
                loopingIndex = visibleSize - (this._datasource.getCurrentSize()-1-borderIndex+ numOfItemToAdd);
            }else{
                loopingIndex = borderIndex + visibleSize - numOfItemToAdd;
            }
            if (numOfItemToAdd > 0) {
                while (numOfItemToAdd--) {
                    child = this._fetchChild(loopingIndex);
                    child = this._attachChild(child, false);
                    loopingIndex++;
                }
            }
        }

        // Step 6: Reschedule the dsIndex
        for (j = 0; j < this._cache.length; j++) {
            if(this._cache[j] && this._cache[j].child){
                this._cache[j].child.dsIndex = j;
            }
        }

        this.unselect();
        // Step 7: Focus last focused item position or the first selected index when previously is empty
        if(lastSelectIndex === -1){
            lastSelectIndex = this.opts.firstSelectedIndex;
        }
        this.setSelectedIndex(lastSelectIndex);

        // Step 8: Update the scrollbar
        this._updateScrollbar();
        this._ensurePreload();
        return;
        
    },
        
    /**
         * Function to get direct access to the datasource
         * @returns {accedo.data.Ds}
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name getDatasource
         */
    getDatasource: function() {
        return this._datasource;
    },

    /**
         * Associate a scrollbar to this list, then updates it.
         * The previous one (if any) will stop being updated but won't be reset.
         * This is not available for a CAROUSEL.
         * @param {accedo.ui.scrollbar} sb 
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name setScrollbar
         */
    setScrollbar: function(sb) {
        if (!this._isCarousel) {
            this._scrollbar = sb;
            this._updateScrollbar();
        }
    },      
    /**
         * Allows setting the nextLeft or nextDown option of the List that will be used when the current selection
         * is at the list's left (for an horizontal list) or top (for a vertical one) border.
         * By default the view's nextLeft or nextDown option of the list is used.
         * @param {Object} XDK ID or callback function
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name setBorderNextLeftUp
         * @deprecated use onKey to perform the internal navigaton and the boundary case should setOption to the nextLeft/nextRight/nextUp/nextDown to the list
         */
    setBorderNextLeftUp: function(next) {
        
        if( this.opts.orientation === accedo.ui.widget.List.HORIZONTAL ){
            this.setOption("nextLeft",next);
        }else{
            this.setOption("nextUp",next);
        }
    },
            
    /**
         * Allows setting the nextRight or nextUp option of the List that will be used when the current selection
         * is at the list's right (for an horizontal list) or down (for a vertical one) border.
         * By default the view's nextRight or nextUp option of the list is used.
         * @param {Object} XDK ID or callback function
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name setBorderNextRightDown
         * @deprecated use onKey to perform the internal navigaton and the boundary case should setOption to the nextLeft/nextRight/nextUp/nextDown to the list
         */
    setBorderNextRightDown: function(next) {
        if( this.opts.orientation === accedo.ui.widget.List.HORIZONTAL ){
            this.setOption("nextRight",next);
        }else{
            this.setOption("nextDown",next);
        }
    },
    
    /**
         * Empty this list, unselect any selected child
         * @public
         * @function
         * @memberof accedo.ui.widget.List#
         * @name empty
         */
    empty: function(){
        var i, len;

        i = 0;
                
        len = this._cache.length;
        for (; i < len; i++) {
            if (this._cache[i] && this._cache[i].child && this._cache[i].child.deinit){
                this._cache[i].child.deinit();
            }
            this._cache[i] = null;
        }
        this._cache.length = 0;
    },
            
    /**
         * Associates a displayHelper to this list.
         * This helper defines what to do when an element is created for display (it is then possible
         * to add click events, callbacks on selection, etc).
         * Typically you will want to set the component's onSelect property, as it is the one that gets
         * triggered when the component is selected in the list, and a 'click' event.
         * @public
         * @param {Function} dh The displayer helper function
         * @param context (optional) The context to bind helper function to
         * @example
         *  setDisplayHelper(function(component, category) {
         *      //Just do whatever you want with the provided data
         *      component.getRoot().setAttributes({'src': category.coverimagemedium});
         *      component.onSelect = function() {
         *              accedo.console.info('You just selected ' + category.title);
         *              myController.get('summary').setText(category.shortsynopsis);
         *      };
         *      component.onUnselect = function() {
         *              accedo.console.info('You just unselected ' + category.title);
         *      };
         *      component.addEventListener('click', function() {
         *              accedo.console.info('You just clicked on ' + category.title);
         *      });
         *  });
         *  @memberof accedo.ui.widget.List#
         *  @function
         *  @name setDisplayHelper
         */
    setDisplayHelper: function(dh, context) {
        if (context) {
            dh = accedo.Fn.bind(dh, context);
        }
        this.displayHelperFn = dh;
    },
    
    /**
         * Removes the link to the datasource, which then removes all visible and preloaded children.
         * This method is here because it's not trivial to know that setDatasource(null) is the thing to do
         * to free the resources used by this list without deleting it completely.
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name purgeList
         */
    purgeList: function() {
        this.setDatasource(null);
    },
    /**
         * Remove all listeners of the children
         * @public
         * @memberof accedo.ui.widget.List#
         * @function
         * @name deinit
         */
    deinit: function(){
        this.removeAllListeners();
        this.empty();
        this.purgeList();
    }
});

accedo.Class.create("freesat.widget.InputField", "accedo.ui.widget.InputField", {},
{
	initialize: function ($super, opts) {
		$super(opts);
		this.numberInputs = [
			"0 ",
			"1",
			"ABC2",
			"DEF3",
			"GHI4",
			"JKL5",
			"MNO6",
			"PQRS7",
			"TUV8",
			"WXYZ9"
		];
		this.t9span = accedo.Element.create('span');
		this.t9span.addClass("inverse");
		this.currentT9Number = -1;
		this.currentT9Index = -1;
	},

	numberInput: function (vKey) {
		switch (vKey) {
			case accedo.VKey.KEY_0:
				this.setCurrentT9Number(0);
				this.appendT9();
				break;
			case accedo.VKey.KEY_1:
				this.setCurrentT9Number(1);
				this.appendT9();
				break;
			case accedo.VKey.KEY_2:
				this.setCurrentT9Number(2);
				this.appendT9();
				break;
			case accedo.VKey.KEY_3:
				this.setCurrentT9Number(3);
				this.appendT9();
				break;
			case accedo.VKey.KEY_4:
				this.setCurrentT9Number(4);
				this.appendT9();
				break;
			case accedo.VKey.KEY_5:
				this.setCurrentT9Number(5);
				this.appendT9();
				break;
			case accedo.VKey.KEY_6:
				this.setCurrentT9Number(6);
				this.appendT9();
				break;
			case accedo.VKey.KEY_7:
				this.setCurrentT9Number(7);
				this.appendT9();
				break;
			case accedo.VKey.KEY_8:
				this.setCurrentT9Number(8);
				this.appendT9();
				break;
			case accedo.VKey.KEY_9:
				this.setCurrentT9Number(9);
				this.appendT9();
				break;
		}
	},

	appendT9: function () {
		if (this.currentT9Index === -1) {
			this._wrapper.appendChild(this.t9span);
		}
		this.currentT9Index = (this.currentT9Index + 1) % this.numberInputs[this.currentT9Number].length;
		this.t9span.setText(this.numberInputs[this.currentT9Number].substr(this.currentT9Index, 1));
		this.createPrintInterval();
	},

	createPrintInterval: function () {
		var that = this;
		if (this.printInterval) {
			clearInterval(this.printInterval);
			this.printInterval = null;
		}
		this.printInterval = setInterval(function () {
			clearInterval(that.printInterval);
			this.printInterval = null;
			that.setCurrentT9Number(-1);
		}, 1500);
	},

	printT9: function () {
		this.insertChar(this.numberInputs[this.currentT9Number].substr(this.currentT9Index, 1));
	},

	setCurrentT9Number: function (curNum) {
		if (this.currentT9Number !== curNum && this.currentT9Index !== -1) {
			var char = this.numberInputs[this.currentT9Number].substr(this.currentT9Index, 1);
			this.currentT9Index = -1;
			this.insertChar(char);
		}
		this.currentT9Number = curNum;
	},

	// overwrite XDK InputField's function to make sure pending T9 char is written before any other input
	insertChar: function (ch, insertPos) {
        
        // print pending T9 char
        this.setCurrentT9Number(-1);

        // copied from XDK InputField widget:
        if (!this._inputState) {
            return;
        }
        if (this._labelText.length >= this._maxLength) { // If length of _labelText reach maximum, do nothing, and return
            return;
        }
        
        insertPos = accedo.Object.isUndefined(insertPos) ? this._cursorPosition : insertPos;
        var t = this.getText();
        t = accedo.String.insert(t, insertPos, ch);
        this._cursorPosition = this._cursorPosition + ch.length;
        this.setText(t, undefined, true);
        this.updateLeft('add');
        this.updateCursorPosition();
	},

	backPress: function () {
		var hadText = this.getText().length > 0 || this.printInterval;
		if (this.printInterval) {
			clearInterval(this.printInterval);
			this.printInterval = null;
			this._wrapper.removeChild(this.t9span);
			this.currentT9Index = -1;
		} else {
			this.backspace();
		}
		return hadText;
	}

});
/*global accedo: false, app: false, console:false */
// JSLint comment to prevent failure because of object definitions in other files. 

/**
 * @fileOverview throbber widget for showing a loading spinner
 * @author Niklas Bjorken <niklas.bjorken@accedo.tv>
 */
 
/**
* Throbber class
* @name Throbber
* @memberOf accedo.ui.widget
* @class Throbber class
* @extends accedo.ui.RootController
*/


accedo.Class.create("freesat.widget.LoadingIcon", "accedo.ui.Container", [], {}, {
    /**
     * The width and height of the throbber PNG image.
     * @private
     * @name spinner
     * @memberOf accedo.ui.widget.Throbber#
     */
    dimensions: {
        w: 203,
        h: 72
    },
    
    /**
     * Speed with which the animation is shown. 0..n. Higher is slower, lower is faster.
     * The value corresponds to how many miliseconds each animation frame should show.
     * @private
     * @name spinner
     * @memberOf accedo.ui.widget.Throbber#
     */
    animationSpeed: 100, // ms
    
    /**
     * Tells the animation how many frames the sprite contains. Adjust this value if the number of
     * sprites changes.
     * @private
     * @name spinner
     * @memberOf accedo.ui.widget.Throbber#
     */
    spriteFrames: 20,
    
    /**
     * Reference to the throbber object
     * @private
     * @name spinner
     * @memberOf accedo.ui.widget.Throbber#
     */
    spinnerReference: null,
    
    
    /**
     * @constructor
     * @memberOf accedo.ui.widget.Throbber#
     * @param {Object} opts The options object
     * @private
     */
    initialize: function($super, opts) {
        opts.focusable = false; //Explicit definition of object not being focusable
        $super(opts);
        this.root.addClass('accedo-ui-throbber');
    },
    

    /**
     * This function shows the throbber
     * @name show
     * @function
     * @param {accedo.ui.AbstractComponent} element The element/component in which to show the throbber
     * @param {Boolean} useDelay If true, the loader will not show until 500 ms has passed, given that hide has not been called during this time
     * @memberOf accedo.ui.widget.Throbber#
     * @public
     */
    show: function(element, useDelay, notBlocking) {
        useDelay = useDelay || false;
    
        clearTimeout(this.showTimeout);
        clearTimeout(this.hideTimeout);
        clearInterval(this.updateInterval);
        
        // If spinner already exists, remove it
        // @todo: instead of removing it, re-use it?
        if (this.spinnerReference){
            this.spinnerReference.remove();
        }
        
        // Keep track of where in the sprite loop we are
        var currentLoop = 0;
        
        var centerX = (1280 - this.dimensions.w) / 2;
        var centerY = (720 - this.dimensions.h) / 2;

        if (element){
            // Get the DOM node of the element for the spinner to append itself to
            var target = element._dom || element.getRoot()._dom;
            
            // Get the height/width/x/y of the element we want to show the throbber in.
            // No xDK support for getting position and dimension values from DOM objects, 
            // so we'll have to use direct calls to the DOM
            var clientRect = target.getBoundingClientRect();
            var parentX = clientRect.left;
            var parentY = clientRect.top;
            var parentW = target.clientWidth;
            var parentH = target.clientHeight;
            
            // Calculate the point where to place the throbber so that it is centered within the parent element
            centerX = parentX + ((parentW - this.dimensions.w) / 2);
            centerY = parentY + ((parentH - this.dimensions.h) / 2);

        }
        // Create a div with the throbber PNG sprite as background (defined in css class "pngThrobber")
        // and position it accoring to x/y/w/h of target
        this.spinnerReference = new accedo.Element('div', {
            style: {
                top: centerY + 'px',
                left: centerX + 'px',
                width: this.dimensions.w + 'px',
                height: this.dimensions.h + 'px'
            }
        });
        this.spinnerReference.setVisible(false);
        this.spinnerReference.setClass('loadingIcon');
        
        var self = this;
        
        // Start the loading animation
        this.updateInterval = setInterval(function(){
            self.spinnerReference.setAttributes({
                style: {
                    backgroundPosition: '0 ' + (currentLoop * -self.dimensions.h) + 'px'
                }
            });
            currentLoop++;
            if (currentLoop >= self.spriteFrames){
                currentLoop = 0;
            }
        }, this.animationSpeed);
        
        // Add the div to the body
        var body = new accedo.Element(document.getElementsByTagName('body')[0]); // xDK does not provide a way to get the body element?(!?)
        body.appendChild(this.spinnerReference);
        
        // Make the throbber visible
        if (useDelay){
            this.showTimeout = setTimeout(function(){
                self.spinnerReference && self.spinnerReference.setVisible(true);
            }, 500);
        } else {
            self.spinnerReference && self.spinnerReference.setVisible(true);
        }
        if (!notBlocking){
            accedo.blockNavigation = true;
        }
    },

    /**
     * This function hides the throbber
     * @name hide
     * @function
     * @param {Boolean} useDelay If true, the loader will not hide until 500 ms has passed, given that hide has not been called again during this time
     * @memberOf accedo.ui.widget.Throbber#
     * @public
     */
    hide: function(useDelay, keepNavigationBlock) {
        useDelay = useDelay || false;
        
        clearTimeout(this.showTimeout);
        clearTimeout(this.hideTimeout);
        
        if (useDelay){
            var self = this;
            this.hideTimeout = setTimeout(function(){
                clearInterval(self.updateInterval);
                self.spinnerReference && self.spinnerReference.remove();
                self.spinnerReference = null;
                if (accedo.blockNavigation && !keepNavigationBlock){
                    accedo.blockNavigation = false;
                }
            }, 500);
        } else {
            clearInterval(this.updateInterval);
            this.spinnerReference && this.spinnerReference.remove();
            this.spinnerReference = null;
            if (accedo.blockNavigation && !keepNavigationBlock){
                accedo.blockNavigation = false;
            }
        }
    },

    isShowing: function () {
        return !!this.spinnerReference;
    }
    
});
/*jslint browser: true, devel: true,newcap:false */
/*global accedo: false */
/**
 * @desc provide view related functionality
 * @name View
 * @memberof accedo.ui
 * @module
 */
accedo.Class.create("accedo.ui.View", {
    /**
     * Create a view
     *
     * @name create
     * @param {String} viewPath - Definition string for the view e.g. 'app.views.MainView'
     * @param {Array} [dependencies] - Array of definition strings that this view depends on. e.g. ['accedo.ui.Label'].
     * Dependencies will be loaded automatically if it's not loaded.
     * @param {Object|Function} view - View Definition Object or function that returns the View Definition Object. 
     * NOTE that if a function is supplied with dependencies. The function will be executed after all the dependencies 
     * is defined.
     * @memberof accedo.ui.module:View
     * @function
     * @static
     * @public
     */
    create: function() {
        var viewPath = null, dependencies = null, properties = Array.prototype.slice.call(arguments) , view=null, viewDef;
        
        viewPath = properties.shift();
        if (!accedo.Object.isString(viewPath)) {
            freesat.console.log("First parameter must be a string");
            accedo.console.error("First parameter must be a string");
        }
        viewDef = accedo.extractPath(viewPath);
        if (accedo.Object.isArray(properties[0])) {
            dependencies = properties.shift();
        }

        view = properties.shift();

        freesat.console.log('View create : ' + viewPath);
        if (accedo.Object.isFunction(view)) {
            accedo.loadDefinitions(dependencies, {
                onSuccess: function() {
                    //prepare the namespace
                    accedo.create(viewDef.namespace, viewDef.identifier, view());
                }
            });
        }else if (accedo.Object.isPlainObject(view)) {
            if (dependencies) {
                accedo.loadDefinitions(dependencies);
            }
            accedo.create(viewDef.namespace, viewDef.identifier, view);
        }
        else {
            accedo.console.error("Unknown view declaration type: "+ view);
            freesat.console.log("Unknown view declaration type: "+ view);
        }
    },
    
    /**
     * Realize a view
     * 
     * @name realize
     * @param {Object} viewDef - View Definition Object
     * @param {Object} [options] - Options 
     * @param {Function} [options.helper] - Optional. Assign helper function to handle objects. Helper function should return true if it's handled successful; false if not; accedo.ui.AbstractComponent if it can be attached.
     * that didn't define the 'type' attribute
     * @param {accedo.ui.AbstractComponet} [parent] - Private. For view node traversing.
     * @memberof accedo.ui.module:View
     * @function
     * @static
     * @public
     */
    realize: function (viewDef, options, parent) {
        
        options = options || {};
        
        if (!accedo.Object.isPlainObject(viewDef)) {
            accedo.console.error("View JSON is incorrect: " + viewDef);
        }
        
        var view;
        
        /**
         * @private
         * @ignore
         */
        function realizeChildren(component, subViewDef) {
            //instantiate children elements
            var i, l, childOpts, subview;
            if (accedo.Object.isArray(subViewDef.children) && subViewDef.children.length>0) {
                i = 0;
                l = subViewDef.children.length;
                for (; i < l; i++) {
                    childOpts = subViewDef.children[i];
                    
                    subview = accedo.ui.View.realize(childOpts, options, component);
                    if (subview){
                        component.attach(subview);
                    }
                }
            }
        }
        
        
        
        if (viewDef.type && !accedo.Object.isFunction(viewDef.type)) {
            accedo.console.warn("Trying to realize unknown subview. Probably dependencies is not ready! View def:" + accedo.Object.toJSON(viewDef));
            return null;
        }
        //defined using constructor reference
        if (viewDef.type) {
            viewDef = accedo.Object.clone(viewDef);
            viewDef.skipRealizeChildren = true;
            view = new viewDef.type(viewDef);
            realizeChildren(view,viewDef);
            return view;
        }

        if (!accedo.Object.isFunction(options.helper)) {
            accedo.console.warn("Trying to realize unknown subview. Probably dependencies is not ready! View def:" + accedo.Object.toJSON(viewDef));
            return null;
        }

        // use options.helper to init view
        view = options.helper(viewDef,parent);
        if (view instanceof accedo.ui.AbstractComponent) {
            return view;
        }
        if (view === false) {
            //failed, but carry on
            accedo.console.warn("Helper function failed to realize unknown subview");
            return null;
        }
        // helper function executed, consider as handled successfully
        return null;
    }
});
/*jslint browser: true, devel: true,newcap:false */
/*global accedo: false */
/**
 * <pre>
 * Controller Cache uses the reference counting mechanism. Therefore, once there's
 * no ref of usage, the controller instance should be released and return to 
 * cache. There are 4 levels of releasing level that can be configured to: 
 * 3. View Level - The controller instance and it's view is retained.
 * 2. Object Level - The controller instance is retained but it's view is 
 * deinitialized.
 * 1. Definition Level - The controller instance is deinitialized. Only the definition is retained.
 * 0. None - The definition is also unloaded from memory.
 *
 * <strong> This class is deprecated since version 1.0.6. </strong>
 *
 * Controller Cache. Caches controller by it's definition string, based on
 * current reference count. i.e. The reference is keep referenced in the Controller
 * Cache until it's reference count reaches 0.
 *
 * For example, please see history manager.
 * </pre>
 *
 * @name ControllerCache
 * @memberof accedo.ui
 * @class
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @extends accedo.mixin.EventDispatcher
 * @example var controllerCache = accedo.ui.ControllerCache.singleton();
 * @deprecated
 */
accedo.Class.create("accedo.ui.ControllerCache", {
    /**
     * Preload level: none
     * @constant
     * @name PRELOAD_NONE
     * @memberof accedo.ui.ControllerCache
     * @deprecated
     */
    PRELOAD_NONE    : 0,
    /**
     * Preload level: definition
     * @constant
     * @name PRELOAD_DEF
     * @memberof accedo.ui.ControllerCache
     * @deprecated
     */
    PRELOAD_DEF     : 1,
    /**
     * Preload level: object
     * @constant
     * @name PRELOAD_OBJ
     * @memberof accedo.ui.ControllerCache
     * @deprecated
     */
    PRELOAD_OBJ     : 2,
    /**
     * Preload level: view
     * @constant
     * @name PRELOAD_VIEW
     * @memberof accedo.ui.ControllerCache
     * @deprecated
     */
    PRELOAD_VIEW    : 3,
    /**
     * Preload level option
     * @private
     * @static
     * @name _optPreload
     * @memberof accedo.ui.ControllerCache
     * @deprecated
     */
    _optPreload : accedo.config.get('ui.controllerCache.lv',1),
    /**
     * Preloaded event prefix
     * @constant
     * @name EVT_PRELOADED_PREFIX
     * @memberof accedo.ui.ControllerCache
     * @deprecated
     */
    EVT_PRELOADED_PREFIX :  "preload-complete-",
    /**
     * Internal class singleton instance.
     * @name _singleton
     * @private
     * @static
     * @memberof accedo.ui.ControllerCache
     * 
     **/
    _singleton      : null,
    /*
     * @override
     * @name main
     * @static
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.ControllerCache
     * 
     */
    main: function() {
        this._singleton = new accedo.ui.ControllerCache();
    },
    /*
     * Singleton method to get the instance.
     * @name singleton
     * @static
     * @returns {accedo.ui.ControllerCache} The singleton instance
     * @function
     * @memberof accedo.ui.ControllerCache
     * @public
     * @deprecated
     */
    singleton: function() {
        return this._singleton;
    }
}, 
"accedo.mixin.EventDispatcher",
{
    /**
     * The controller definition hash map storage.
     * @name _registrar
     * @memberof accedo.ui.ControllerCache#
     * @private
     */
    _registrar  : null,
    /**
     * The controller object hash map storage.
     * @name _controllers
     * @memberof accedo.ui.ControllerCache#
     * @private
     */
    _controllers : null,
    /**
     * The controller object reference counters.
     * @name _controllersRefCount
     * @memberof accedo.ui.ControllerCache#
     * @private
     */
    _controllersRefCount: {},
    /*
     * Overrides the initialize method
     * @name initialize
     * @function
     * @public
     * @protected
     */
    initialize: function() {
        this._registrar = new accedo.Hash({});
        this._controllers = new accedo.Hash({});
    },
    /**
     * Load a controller into cache
     * @name load
     * @param {String} def - definition string of the controller
     * @memberof accedo.ui.ControllerCache
     * @function
     * @static
     * @public
     * @deprecated
     */
    load: function(def,options) {
        var reg = this._registrar.get(def), preload, ref;
        
        //registed
        if (!accedo.Object.isUndefined(reg)) {
            return reg;
        }
        
        //No preloading
        if  (accedo.ui.ControllerCache._optPreload === accedo.ui.ControllerCache.PRELOAD_NONE) {
            this._registrar.set(def, null);
            return null;
        }
        
        preload = accedo.Fn.bind(function(def) {
            
            var reg, Klass, onPreloaded = accedo.Fn.bind(function(controller){
                this._registrar.set(def,controller);
                this.dispatchEvent(accedo.ui.ControllerCache.EVT_PRELOADED_PREFIX+def, controller);
                this.removeAllListeners(accedo.ui.ControllerCache.EVT_PRELOADED_PREFIX+def);
            },this);
                
            if  (accedo.ui.ControllerCache._optPreload >= accedo.ui.ControllerCache.PRELOAD_DEF) {
                reg = accedo.getNamespaceReference(def);
            }
                
            if  (accedo.ui.ControllerCache._optPreload >= accedo.ui.ControllerCache.PRELOAD_OBJ) {
                Klass = reg; 
                reg = new Klass();
            }
                
            if  (accedo.ui.ControllerCache._optPreload >= accedo.ui.ControllerCache.PRELOAD_VIEW) {
                reg.initView(onPreloaded);
            }
            else {
                onPreloaded(reg);
            }
                
            
        },this);
       
        ref = accedo.getNamespaceReference(def);
        if (!ref) {
            accedo.loadDefinition(def,{
                onSuccess: accedo.Fn.bind(preload,this),
                onFailure: accedo.Fn.bind(function(){
                    this._registrar.unset(def);
                    if (options && options.onFailure) {
                        options.onFailure();
                    }
                    
                },this)
            });
            //Set as preloading state
            this._registrar.set(def,-1);
        
            //Return as preloading state
            return -1;
        }
        
        preload(def);
       
        return this._registrar.get(def);
    },
    /**
     * Get controller reference from the cache, while referencing the controller
     * 
     * @name ref
     * @param {String} def - controller definition string
     * @param {Object} [options] 
     * @param {Fn} [options.onSuccess] - callback on Controller loaded 
     * @memberof accedo.ui.ControllerCache
     * @function
     * @static
     * @public
     * @deprecated
     */
    ref: function (def, options) {
        
        var controller = this._controllers.get(def), doLoad, ret;
        if (controller) {
            this._controllersRefCount[def]++;
            accedo.console.info("controllerCache: " + def+" loaded, RefCount=" +this._controllersRefCount[def]);
            if (options && options.onSuccess) {
                options.onSuccess(controller);
            }
            return;
        }
        
        doLoad = accedo.Fn.bind(function(reg) {
            var Klass, onLoaded = accedo.Fn.bind(function(controller) {
                this._controllers.set(def,controller);
            
                this._controllersRefCount[def]=1;
            
                accedo.console.info("controllerCache: " + def+" loaded, RefCount=" +this._controllersRefCount[def]);
            
                if (options.onSuccess) {
                    options.onSuccess(controller);
                }
            },this);
            
            if  (accedo.ui.ControllerCache._optPreload < accedo.ui.ControllerCache.PRELOAD_DEF) {
                reg = accedo.getNamespaceReference(reg);
            }
                
            if  (accedo.ui.ControllerCache._optPreload < accedo.ui.ControllerCache.PRELOAD_OBJ) {
                Klass = reg;
                reg = new Klass();
            }
                
            if  (accedo.ui.ControllerCache._optPreload < accedo.ui.ControllerCache.PRELOAD_VIEW) {
                reg.initView(onLoaded);
            }
            else {
                onLoaded(reg);
            }
            
        },this);
        
        
        ret = this.load(def,options);

        // definition is loading, add callback
        if (ret === -1) {
            this.addEventListener(accedo.ui.ControllerCache.EVT_PRELOADED_PREFIX+def, doLoad,this);
            return;
        }
        
        // no preloading
        if (ret === null) {
            accedo.loadDefinition(def, {
                onSuccess: accedo.Fn.bind(doLoad,this)
            });
            return ;
        }
        
        //load with preloaded _registrar
        doLoad(ret);
    },
    /**
     * Returns and unreference a controller from cache
     * 
     * @name unref
     * @param {String} def - controller definition string
     * @param {Function} onSuccess - callback on Controller loaded
     * @memberof accedo.ui.ControllerCache
     * @function
     * @static
     * @public
     * @deprecated
     */
    unref: function(def) {
        
        var controller = this._controllers.get(def);
        if (!controller) {
            return;
        }
        
        this._controllersRefCount[def]--;
        
        if (this._controllersRefCount[def]===0) {
        
            if  (accedo.ui.ControllerCache._optPreload < accedo.ui.ControllerCache.PRELOAD_VIEW) {
                controller.deinitView();
            }

            if  (accedo.ui.ControllerCache._optPreload < accedo.ui.ControllerCache.PRELOAD_OBJ) {
                controller.deinit();
            }

            if  (accedo.ui.ControllerCache._optPreload < accedo.ui.ControllerCache.PRELOAD_DEF) {
                //Fully remove controller from memory
                accedo.undefine(def);
            }

            //removes it from cache
            this._controllers.unset(def);
        }
        else {
            accedo.console.info("controllerCache: " + def+" unloaded, RefCount=" +this._controllersRefCount[def]);
        }
        
    },
    /**
     * Unloads a controller from the cache(to remove the 
     * controller fully from the memory).
     * @name unload
     * @param {String} def - controller definition string
     * @memberof accedo.ui.ControllerCache
     * @function
     * @static
     * @public
     * @deprecated
     */
    unload: function (def) {
        
        var reg = this._registrar.get(def),
        doUnregister;
        
        //not registered
        if (!reg) {
            return;
        }
        
        if (this._controllers.get(def)) {
            accedo.console.error("Cannot unregister controller as it is still cached.");
        }
        
        doUnregister = accedo.Fn.bind(function(reg) {
            
            if  (accedo.ui.ControllerCache._optPreload >= accedo.ui.ControllerCache.PRELOAD_VIEW) {
                reg.deinitView();
            }
        
            if  (accedo.ui.ControllerCache._optPreload >= accedo.ui.ControllerCache.PRELOAD_OBJ) {
                reg.deinit();
            }
                
            if  (accedo.ui.ControllerCache._optPreload >= accedo.ui.ControllerCache.PRELOAD_DEF) {
                //Fully remove controller from memory
                accedo.undefine(def);
            }
            
            this._registrar.unset(def);
        },this);
        
        if (reg === -1) {
            this.addEventListener(accedo.ui.ControllerCache.EVT_PRELOADED_PREFIX+def, doUnregister,this);
            return;
        }
        
        doUnregister(reg);
    },
    
    /**
     * Get a controller from the cache.
     * @name get
     * @param {String} def - controller definition string
     * @returns {NULL|accedo.ui.Controller} returns controller if it's cached, null if controller is not in cache.
     * @memberof accedo.ui.ControllerCache
     * @function
     * @static
     * @public
     * @deprecated - use accedo.ui.ControllerManger.getController() instead if you want to ontain an active controller;
     */
    get: function(def) {
        return this._controllers.get(def);
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Controller Manager centralizes the navigation behavior of controllers: open, 
 * replace and close of controller(s).
 * @name ControllerManager
 * @memberof accedo.ui
 * @class Controller Manager Implementation
 * @example var controllerManager = accedo.ui.ControllerManager.singleton();
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 */
accedo.Class.create("accedo.ui.ControllerManager",{
    /**
     * Internal class singleton instance.
     * @name _singleton
     * @private
     * @static
     * @memberof accedo.ui.ControllerManager
     **/
    _singleton      : null,
    /*
     * @override
     * @name main
     * @static
     * @function
     * @public
     * @protected
     * @memberof accedo.ui.ControllerManager
     */
    main: function() {
        this._singleton = new accedo.ui.ControllerManager();
    },
    /**
     * Get the singleton instance of ControllerManager
     * @returns accedo.ui.ControllerManager
     * @memberof accedo.ui.ControllerManager
     * @function
     * @static
     * @public
     */
    singleton: function() {
        return this._singleton;
    }
    
}, 
"accedo.mixin.EventDispatcher", 
{
    /**
     * Get the controller by the id of controller. If no id is set for a controller,
     * controller will return it's class definition string.
     * @param {String} id - id or definition string of the controller 
     * @memberof accedo.ui.ControllerManager#
     * @function
     * @public
     */
    getController: function(id) {
        
        var ret = accedo.ui.AppSystem.singleton().get(id);
        
        if (!ret) {
            //tries to get by classname if couldn't get by id
            ret = accedo.ui.AppSystem.singleton().getControllerByDef(id);
        } 
        
        return ret;
    },
    /**
     * Create and initialize a controller with the view
     * @param {String} controllerDef - Controller's definition string
     * @param {Object} options - Optional.
     * @param {Function} [options.onSuccess] - Optional. Callback on navigate success
     * @param {Function} [options.onFailure] - Optional. Callback on navigate failure
     * @param {String} [options.id] - Optional. Id for the created controller
     * @public
     * @memberof accedo.ui.ControllerManager#
     * @ignore
     */
    createController: function(controllerDef, options) {
        var ref = accedo.getNamespaceReference(controllerDef);
        if (!ref) {
            accedo.loadDefinition(controllerDef,{
                onSuccess: accedo.Fn.bind(function(def){
                    var ref = accedo.getNamespaceReference(def);
                    this._initController(ref, options);
                },this),
                onFailure: accedo.Fn.bind(function(def){
                    if (options && options.onFailure) {
                        options.onFailure();
                    }
                },this)
            });
            
            return;
        }

        this._initController(ref, options);
    },
    /**
     * Initialize a controller
     * @param {accedo.ui.Class} ref - class reference that has extended accedo.ui.Controller
     * @param {Object} options - Optional.
     * @param {Function} [options.onSuccess] - Optional. Callback on navigate success
     * @param {Function} [options.onFailure] - Optional. Callback on navigate failure
     * @param {String} [options.id] - Optional. Id for the created controller
     * @private
     * @memberof accedo.ui.ControllerManager#
     * @ignore
     */
    _initController: function(ref, options) {
        
        var controller = new ref();
        
        if (options && accedo.Object.isString(options.id)) {
            controller.setId(options.id);
        }
        
        controller.initView(function() {
            if (options && options.onSuccess) {
                options.onSuccess(controller);
            }
        });
    },
    /**
     * Changes from current controller to a new one in the same container.
     * @param {String|accedo.ui.Controller} front - Controller's definition string 
     * or Controller instance.
     * @param {String|accedo.ui.Controller} controller - Controller's definition 
     * string or Controller instance.
     * @param {Object} [context] - Optional. Context object to be passed to the new controller
     * @param {Object} [options] - Optional. Options
     * @param {Function} [options.onSuccess] - Optional. Callback on navigate success
     * @param {Function} [options.onFailure] - Optional. Callback on navigate failure
     * @param {Boolean} [options.noHistory] - Optional. Indicate whether this navigate is 
     * saved to the History. Default: false.
     * @param {Boolean} [options.skipSameController] - Optional. Indicate whether
     * this replacement should be skipped if front and back is the same controller
     * @param {String} [options.frontFocus] - Optional. Input the component's 
     * ID to be focused when navigate back from history.
     * @param {String} [options.id] - Optional. id for created the controller. Used 
     * only when opening a controller by it's definition string
     * @memberof accedo.ui.ControllerManager#
     * @public
     */
    replaceController : function(front, controller, context, options) {
        options = options || {};
        
        //deprecated, remove in 2.0 : skip navigate to same controller 
        if (options.skipSameController && (accedo.Object.isString(controller)?controller:controller.getDefinition()) === (accedo.Object.isString(front)?front:front.getDefinition())) {
            return;
        }

        if (accedo.Object.isString(front)) {
            front = this.getController(front);
        }
        
        if (!front || !(front instanceof accedo.ui.Controller)) {
            accedo.console.error("Invalid front controller.");
        }
        function _doChangeController(controller) {
            //update the controller with context data
            // extract controller data from commit function
            if(front.getParent())front.getParent().replace(controller, front);
            
            //reset the last controller conext data
            front.resetController();
                
            //setup the controller with context data
            controller.setupController(context);
            
            if (options && accedo.Object.isFunction(options.onSuccess)) {
                options.onSuccess(controller);
            }
                        
        }
        if (accedo.Object.isString(controller)) {
            this.createController(controller,{
                onSuccess: _doChangeController,
                onFailure: function() {
                    if (options && options.onFailure) {
                        options.onFailure();
                    }
                },
                id: (options && options.id)? options.id : null
            });
        }
        else if (controller instanceof accedo.ui.Controller){
            _doChangeController(controller);
        }
        else {
            accedo.console.error("Expected controller to be definition string.");
        }
    },
    /**
     * Open a controller in a container.
     * Controller reference may be obtained via accedo.ui.ControllerManager#getController
     * @param {String|accedo.ui.Controller} controller - Controller's definition string or Controller instance
     * @param {accedo.ui.Container} container - Container where the controller should be opened at
     * @param {Object} [context] - Optional. Context object to be passed to the new controller
     * @param {Object} [options] - Optional. Options
     * @param {Function} [options.onSuccess] - Optional. Callback on navigate success
     * @param {Function} [options.onFailure] - Optional. Callback on navigate failure
     * @param {Boolean} [options.noHistory] - Optional. Indicate whether this navigate is 
     * @param {String} [options.id] - Optional. id for created the controller. Used 
     * only when opening a controller by it's definition string
     * saved to the History. Default: false.
     * @memberof accedo.ui.ControllerManager#
     * @public
     */
    openController: function(controller, container, context, options) {
        
        context = context || {};
        
        function _doOpenController(controller) {

            var navigation = {
                controller: controller.getDefinition(),
                controllerRef: controller,
                context: context,
                options: options
            };
            
            //controller is already being attached, save the current context to fallback on history back
            if (controller.parent) {
                navigation.frontContainer = controller.parent;
                navigation.front = controller.getDefinition();
                navigation.frontRef = controller;
            }
                
            //Attach to the container
            container.attach(controller);
            
            if (controller.getRootController().getHistoryManager && 
                !(options && options.noHistory)) {
                controller.getRootController().getHistoryManager().historyPush(navigation);
            }
            
            //new controller shouldn't be added to other container when created
            if (!controller.getParent()){
                accedo.console.error("Parent for open controller is not specified.");
            }
                
            //setup the controller with context data
            controller.setupController(context);
                
            if (options && accedo.Object.isFunction(options.onSuccess)) {
                options.onSuccess(controller);
            }
            
        }
        
        if (accedo.Object.isString(controller)) {
            this.createController(controller,{
                onSuccess: _doOpenController,
                onFailure: function() {
                    if (options && options.onFailure) {
                        options.onFailure();
                    }
                },
                id: (options && options.id)? options.id : undefined
            });
        }
        else if (controller instanceof accedo.ui.Controller){
            
            //do not set the id if opening a controller reference
            if (options && options.id) {
                delete options.id;
            }
            
            _doOpenController(controller);
        }
        else {
            accedo.console.error("Expected controller to be definition string or accedo.ui.Controller instance.");
        }
    },
    /**
     * Close a controller by it's definition string. 
     * Controller reference may be obtained via accedo.ui.ControllerManager#getController
     * @param {String|accedo.ui.Controller} controller - Controller's definition string or Controller instance
     * @param {Object} [options] - Optional. Options
     * @param {Function} [options.onSuccess] - Optional. Callback on navigate success
     * @param {Function} [options.onFailure] - Optional. Callback on navigate failure
     * @param {Boolean} [options.noHistory] - Optional. Indicate whether this navigate is 
     * saved to the History. Default: false.
     * @memberof accedo.ui.ControllerManager#
     * @public
     */
    closeController: function(controller, options) {
        
        if (accedo.Object.isString(controller)) {
            controller = this.getController(controller);
            if (!controller) {
                accedo.console.error("Cannot find controller to close.");
                return;
            }
        }
        
        if (!(controller instanceof accedo.ui.Controller)) {
            accedo.console.error("Invalid input for controller param");
            return;
        }
        
        var parent = controller.getParent();
        if (!parent) {
            accedo.console.error("Controller not attached to any parent");
            return;
        }
        
        
        if (controller.getRootController().getHistoryManager && 
            !(options && options.noHistory)) {
            
            controller.getRootController().getHistoryManager().historyPush({
                front: controller.getDefinition(),
                frontRef: controller,
                options: options
            });
        }
        
        //detach to close the controller
        controller.detach();
        
        //reset the controller 
        controller.reset();
        
        if (options && accedo.Object.isFunction(options.onSuccess)) {
            options.onSuccess(controller);
        }
        
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * Controller is also a node in the view tree structure. It sub classes the Container 
 * in order to inherits the parent functionalities.
 *  
 * @name Controller
 * @memberof accedo.ui
 * @class An abstract controller. It is the base logic container for it's children UI objects.
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create("accedo.ui.Controller", "accedo.ui.Container", 
[
    "accedo.ui.View",
    "accedo.ui.ControllerCache",
    "accedo.ui.ControllerManager"
],
{},
{
    /**
     * Whether the view has been initialized.
     * @name _viewInited
     * @memberof accedo.ui.Controller#
     * @private
     */
    _viewInited: false,
    /**
     * The view object reference.
     * @name _viewRef
     * @memberof accedo.ui.Controller#
     * @private
     */
    _viewRef: null,
    /**
     * The hash map object to store the sub-controllers.
     * @name _subControllers
     * @memberof accedo.ui.Controller#
     * @private
     */
    _subControllers: null,
    /**
     * Overrides accedo.ui.Container's initialize method
     * @override
     * @public
     * @protected
     * @param {Object} opts - Options for initalizing the Container
     * @memberof accedo.ui.Controller#
     * @function
     */
    initialize: function($super, opts) {
        
        opts = opts || {};
            
        opts.focusable = false;

        opts.id = opts.id || this.getDefinition(); // always give controller an ID
            
        $super(opts);
            
        this.getRoot().addClass("accedo-ui-controller");
        
        this.setEnabled(true);
            
        this._subControllers = [];
    },
    /**
     * Overriding the deinit function
     * @override
     * @protected
     * @public
     * @funciton
     * @memberof accedo.ui.Controller#
     */
    deinit: function($super){
        this.viewDef = null;
            
        this.removeAllListeners();
        
        $super();
    },
    /**
     * Set current view.
     * @param {Object} viewDef - View Definition Object
     * @protected
     * @public
     * @function
     * @memberof accedo.ui.Controller#
     */
    setView: function(viewDef) {
        this.viewDef = viewDef;
    },
     /**
     * To override as there are no option to be set in the controller and focus should be the firstChildren
     * @param {Object} viewDef - View Definition Object
     * @protected
     * @public
     * @function
     * @memberof accedo.ui.Controller#
     */
    setFocus:function($super){
        return $super();
    },
    /**
     * Overriding container's getController to optimize by using getSubController
     *
     * @name getControllerByDef
     * @function
     * @override
     * @param {String} controllerDef - The controller's full classname
     * @returns {accedo.ui.Controller|null} The component having the given identifier,
     *                                     or null if not found
     * @memberof accedo.ui.Controller#
     * @see accedo.ui.Container#getController
     * @public
     */
    getControllerByDef: function(controllerDef) {
            
        if (this.getDefinition()===controllerDef) {
            return this;
        }
            
        var ret,i=0,len=this._subControllers.length;
            
        for (;i<len;i++) {
            ret = this._subControllers[i].getControllerByDef(controllerDef);
            if (ret) {
                return ret;
            }
        }
            
        return null;
    },
    /**
     * Prepare the view in the display (video memory).
     * @protected
     * @public
     * @name initView
     * @memberof accedo.ui.Controller#
     * @function
     */
    initView: function (callback) {
        if (!this.viewDef) {
            accedo.console.error("View definition not set:controller="+this._class);
        }

        var subCtrlrCount = 0, travsing = true, ret = this;

        //Realize the view
        this.viewRef = accedo.ui.View.realize(this.viewDef, {
            //helper function to realize the sub-controllers
            helper: function(viewDef,parent) {
                var placeHolder;

                if (viewDef.controller && accedo.Object.isString(viewDef.controller)){
                    subCtrlrCount++;

                    //put the placeholder in now to preserve the DOM tree order
                    if (parent) {
                        placeHolder = new accedo.ui.AbstractComponent({
                            root : accedo.Element.create("div")
                        });
                        parent.attach(placeHolder);
                    }
                    accedo.ui.ControllerManager.singleton().createController(viewDef.controller, {
                        onSuccess:function(controller) {
                            subCtrlrCount--;

                            if (parent && placeHolder) {
                                parent.replace(controller, placeHolder);
                            }

                            if (!travsing && subCtrlrCount===0 && accedo.Object.isFunction(callback)){
                                callback(ret);
                            }
                        },
                        id: viewDef.id
                    });
                }
                else if (viewDef.controller) {
                    accedo.console.warn("sub-controllers must be specified as string in the view.");
                }
                else {
                    accedo.console.warn("Sub-view or sub-controller is not properly defined in the view! View def:" + accedo.Object.toJSON(viewDef));
                }

                return true;
            }
        });

        travsing = false;

        this.attach(this.viewRef);

        if (subCtrlrCount===0 && accedo.Object.isFunction(callback)) {
            callback(ret);
        }
    },
        
    /**
     * Remove the view from the display (video memory).
     * @name deinitView
     * @protected
     * @public
     * @memberof accedo.ui.Controller#
     * @function
     */
    deinitView: function() {
           
        //destroy the view
        if (this.viewRef) {
            this.remove(this.viewRef);
            delete this.viewRef;
        }
    },
    /**
     * Setup the subcontrollers if context data for them exists,
     * then calls this controller's setup.
     * @name setupController
     * @memberof accedo.ui.Controller#
     * @function
     * @param {Object} context - Data for controller to update
     * @param {accedo.ui.Hash} [context.subControllers] - optional. hash
     * of controller definition to context's of them to setup
     * @protected
     * @public
     */
    setupController: function(context) {

            
        var i=0,len=this._subControllers.length;

        this.setup(context);
            
        //setup sub controllers if context.subControllers[controllerDef] exists
        for (;i<len;i++) {
            var subController = this._subControllers[i], subCtrlrId = subController.getId();
            if (context && context.subControllers && context.subControllers[subCtrlrId]) {
                subController.setupController(context.subControllers[subCtrlrId]);
            }
            else {
                subController.setupController();
            }
        }
        if (this.postSetup) {
            this.postSetup(context);
        }
        
        //to handle set default Focus dynamically case
        if(!this.getOption("defaultFocus") && this.getChildren()[0]){
            this.setOption("defaultFocus",this.getChildren()[0]);
        }
    },
    /**
     * Setup the view with context data. This function will be called
     * after the view is ready and before the controller is being show.
     * Developer should hanlde controller's context data here.
     * @name setup
     * @memberof accedo.ui.Controller#
     * @function
     * @protected
     * @public
     * @param {Object} context - Data for controller to update from
     */
    setup: function(context) {
        accedo.console.warn("Setup function not overrided. Should handle setting up context data for "+this.getDefinition()+ " here.");
    },
    /**
     * Reset the subcontrollers, then calls this controller's setup.
     * @name resetController
     * @protected
     * @public
     * @memberof accedo.ui.Controller#
     * @function
     */
    resetController: function() {
            
        var i=0,len=this._subControllers.length;
            
        //setup sub controllers if context.subControllers[controllerDef] exists
        for (;i<len;i++) {
            this._subControllers[i].resetController();
        }
            
        this.reset();
    },
    /**
     * Reset controller's current data. This function will be called before
     * a controller is hide. Developer should handle resetting the
     * controllers context data here.
     * @name reset
     * @memberof accedo.ui.Controller#
     * @function
     * @protected
     * @public
     */
    reset: function() {
        accedo.console.warn("Reset function not overrided. Should handle resetting context data for "+this.getDefinition() + " here.");
    },
    /**
     * Changes from current controller to a new one in the same container.
     * Controller reference may be obtained either via direct linking or from appDef - accedo.app#getControllerById
     * @param {String} controller - Controller's definition string
     * @param {Object} [context] - Optional. Context object to be passed to the new controller
     * @param {Object} [options] - Optional. Options
     * @param {Function} [options.onSuccess] - Optional. Callback on navigate success
     * @param {Function} [options.onFailure] - Optional. Callback on navigate failure
     * @param {Boolean} [options.noHistory] - Optional. Indicate whether this navigate is
     * saved to the History. Default: true.
     * @param {String} [options.frontFocus] - Optional. Input the component's
     * ID to be focused when navigate back from history
     * @deprecated Please use accedo.ui.ControllerManager.singleton().replaceController()
     *      or accedo.ui.AppController.changeToController() for the controller just below App Controller level
     * @protected
     * @public
     * @memberof accedo.ui.Controller#
     * @function
     */
    navigate: function(controller, context, options) {
                        
        options = options || {};
            
        options.skipSameController = true;
            
        accedo.ui.ControllerManager.singleton().replaceController(this.getDefinition(),controller,context, options);
                       
    },
    /**
     * Override and not to forward event's to it's sub-components
     * @name dispatchAttachedToController
     * @memberof accedo.ui.Controller#
     * @function
     * @override
     * @public
     * @protected
     */
    dispatchAttachedToController: function($super, parentController) {
        if (this._attachedToController) {
            return;
        }
        parentController.addSubController(this);
        this._attachedToController = true;
        $super(parentController);
    },
    /**
     * Override and not to forward event's to it's sub-components
     * @name dispatchDetachedFromController
     * @memberof accedo.ui.Controller#
     * @function
     * @override
     * @public
     * @protected
     */
    dispatchDetachedFromController: function($super, parentController) {
        if (!this._attachedToController){
            return;
        }
        parentController.removeSubController(this);
        this._attachedToController = false;
        $super(parentController);
    },
    /**
     * Track sub controller added to this controller
     * @name addSubController
     * @public
     * @protected
     * @function
     * @return {Boolean} success or not
     * @param {accedo.ui.Controller} controller
     * @memberof accedo.ui.Controller#
     */
        addSubController: function(controller) {
            
        //Controller already added, ignore could be second controller
        if (this._subControllers.indexOf(controller)>0) {
            accedo.console.warn(controller.getId() + " is already a sub-controller of "+this.getId());
            return false;
        }
            
        this._subControllers.push(controller);
            
        return true;
            
    },
    /**
     * Track sub controller removed from this controller
     * @name removeSubController
     * @public
     * @protected
     * @function
     * @param {accedo.ui.Controller} controller
     * @return {Boolean} success or not
     * @memberof accedo.ui.Controller#
     */
    removeSubController: function(controller) {
            
        var idx = this._subControllers.indexOf(controller);
            
        if (idx<0) {
            return false;
        }
        this._subControllers.splice(idx,1);
            
        return true;
    },
    /**
     * Retreive the sub controllers' id attached to this controller
     * @name getSubControllers
     * @function
     * @return {Array} Controller classname strings
     * @public
     * @memberof accedo.ui.Controller#
     */
    getSubControllers: function(idx) {
        var ret=[],i=0,len=this._subControllers.length;
        if (!accedo.Object.isUndefined(idx)){
            if(len <= idx ){
                return null;
            }
            return this._subControllers[idx];
        }

        for (;i<len;i++) {
            ret.push(this._subControllers[i].getId());
        }
            
        return ret;
    },
    /**
     * Retreive the a sub controller attached to this controller by a controller definition string
     * @name getSubController
     * @param {String} id - id of controller or controller classname
     * @function
     * @return {accedo.ui.Controller|null} The specified controller or null if not found
     * @public
     * @memberof accedo.ui.Controller#
     */
    getSubController: function(id) {
            
        var subCtrlr,i=0,len=this._subControllers.length;
            
        for (;i<len;i++) {
            subCtrlr = this._subControllers[i];
            if (subCtrlr.getId()===id) {
                return subCtrlr;
            }
        }
            
        return null;
    },
    /**
     * Retreives the context object, along with the sub-controllers' contexts
     * @name getControllerContext
     * @function
     * @return {Object} The context object, also, all the sub-controller contexts will be stored under key "subControllers"
     * @public
     * @memberof accedo.ui.Controller#
     */
    getControllerContext: function() {

        var subCtrlr,context = this.getContext(), subContext = null, i=0,len=this._subControllers.length;
            
        for (;i<len;i++) {
            subCtrlr = this._subControllers[i];
            subContext = subCtrlr.getControllerContext();
            if (subContext) {
                context = context || {};
                context.subControllers = context.subControllers || {};
                context.subControllers[subCtrlr.getId()] = subContext;
            }
        }
            
        return context;
    },
    /**
     * Retreives the context object, suppose to be override by extending controller
     * @name getContext
     * @function
     * @return {Object} The context object (which would be null for this class)
     * @public
     * @memberof accedo.ui.Controller#
     */
    getContext: function() {
        return null;
    }
});
/*jslint browser: true, devel: true */
/*global accedo: false */ 
/**
 * <pre>
 * Focus manager 
 * Handling the focus automation here. Based on options defined in components:
 * - nextRight      : Target component's id / abstract component / callback function
 * - nextLeft       : Target component's id / abstract component / callback function
 * - nextUp         : Target component's id / abstract component / callback function
 * - nextDown       : Target component's id / abstract component / callback functionth
 * If the nextUp/nextDown/nextRight/nextLeft is a callback function, it will run the callback in focus direction change and check the return value.
 * When return value is false/undefined, it will keep going to find its parent navigation option until the toppest (which has no navigation Handle) or the parent navigation handling
 * When return value is true, it will stop checking the parent. So when writing callback of nextUp/nextDown/nextRight/nextTop, please make sure if it is necessary to continue the navigation
 * handling after the callback.
 *  For example, the nextUp/nextDown option in list/grid are callback functions. If you want to just do the internal nextUp/nextDown without find the parent one,you need to return true
 *  If you reach the border,it will return the original set value when initial.It may not defined and need to look up for the parent handling,so you may need to return false to keep looping the parent.
 * - defaultFocus   : Child component's id that the component shall delegate the focus to. 
 * - useLastFocus   : Indicate if the component shall delegate the focus to the last 
 *                    focused child component. 
 * </pre>
 * @name FocusManager
 * @memberof accedo.ui
 * @class Focus Manager implementation.
 * @example var focusManger = accedo.ui.FocusManager.singleton();
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 */
accedo.Class.create("accedo.ui.FocusManager", {
    /**
     * Focus direction: UP.
     * @static
     * @field
     * @name FOCUS_UP
     * @memberof accedo.ui.FocusManager
     */
    FOCUS_UP            : 0x01,
    /**
     * Focus direction: DOWN.
     * @static
     * @field
     * @name FOCUS_DOWN
     * @memberof accedo.ui.FocusManager
     */
    FOCUS_DOWN          : 0x02,
    /**
     * Focus direction: LEFT.
     * @static
     * @field
     * @name FOCUS_LEFT
     * @memberof accedo.ui.FocusManager
     */
    FOCUS_LEFT          : 0x04,
    /**
     * Focus direction: RIGHT.
     * @static
     * @field
     * @name FOCUS_RIGHT
     * @memberof accedo.ui.FocusManager
     */
    FOCUS_RIGHT         : 0x08,
    
    _instance: null,
    
    main: function() {
        // XDK CHANGE
        // if(accedo.platform === "lg"){
        //     accedo.Env.addEventListener(accedo.Env.EVT_ONLOAD,function(){
        //         accedo.device.system.addEventListener(accedo.device.lg.System.EVT_ONMOUSEON, function(){
        //             accedo.ui.FocusManager.singleton().requestFocus(null);
        //         });
        //         //to handle LG mouseoff event and use FOCUS_LEFT as a dummy
        //         accedo.device.system.addEventListener(accedo.device.lg.System.EVT_ONMOUSEOFF, function(){
        //             var fm = accedo.ui.FocusManager.singleton(), currentFocus = fm.getCurrentFocus();
        //             fm.setMouseOnOff(false);
        //             if(!currentFocus){
        //                 return fm.requestLastFocus();
        //             }
                    
        //             //to handle case when mouse is on the focus with mousefocusable only when mouseoff, it will get forward focus to the desired one or last non mouse focusable one.
        //             if(currentFocus && currentFocus.getOption("mouseFocusableOnly")){
        //                 return fm.requestFocus(currentFocus);
        //             }
        //         });
        //     });
        // }
    },
    
    singleton: function() {
        
        if (!this._instance) {
            this._instance = new accedo.ui.FocusManager();
        }
        
        return this._instance;
    }
} , {
    /**
     * To store whether using mouse
     * @private
     * @name _mouseON
     * @memberof accedo.ui.FocusManager#
     */
    _mouseON:false,
    /**
     * lastFocus 
     * @private
     * @name _lastFocus
     * @memberof accedo.ui.FocusManager#
     */
    _lastFocus: null,
    /**
     * lastNonMouseFocus 
     * @private
     * @name _lastNonMouseFocus
     * @memberof accedo.ui.FocusManager#
     */
    _lastNonMouseFocus: null,
    /**
     * current Focus
     * @private
     * @name _currentFocus
     * @memberof accedo.ui.FocusManager#
     */
    _currentFocus: null,
    /**
     * to initial the focus manager and set which root Controller that this focus Manager belongs to
     * @public
     * @protected
     * @name initialize
     * @memberof accedo.ui.FocusManager#
     */
    initialize: function() {
    },
    /**
     * to check if the mouse is on
     * @public
     * @name isMouseOn
     * @function
     * @return {boolean} the status of whether mouse is ON
     * @memberof accedo.ui.FocusManager#
     */
    isMouseOn:function(){
        return this._mouseON;
    },
    /**
     * it needs to set mouseon off dynamically before focusDirectionChange and in the keyhandler of rootController in order to have a right status
     * To indictate the current status of mouse
     * @public
     * @name setMouseOnOff
     * @param {boolean} bool the status of whether mouse is ON
     * @memberof accedo.ui.FocusManager#
     * @function
     */
    setMouseOnOff:function(bool){
        this._mouseON = bool;  
    },
    /**
     * Gets the forwarded focus of a specified component when this component is being focused on.
     * @param {accedo.ui.AbstractComponent} component Component to be focused
     * @memberof accedo.ui.FocusManager#
     * @function
     * @private
     */
    _getForwardFocus : function(component) {
        if (!component){
            return null;
        }
        var forward = null,
        mouseOn = this._mouseON,
        useLastFocus = component.getOption('useLastFocus'),
        lastFocus = component.lastFocusChild,
        defaultFocus = component.getOption('defaultFocus'),
        ret, hasOnRequestFocus = false, forwardOnMouseOff;

        //checks whether a component wanna handle the focus itself
        if (accedo.Object.isFunction(component.onRequestFocus)) {
            hasOnRequestFocus = true;
            //focus handler can return a component to forward the focus to
            ret = component.onRequestFocus();
            
            //If onRequestFocus return a string / abstract component, it will return the focus item
            //if onRequestFocus return false, it will return false which tell requestFocus no forward focus on that element
            if (accedo.Object.isString(ret)) {
                forward = component.get(ret);
            }else if (ret instanceof accedo.ui.AbstractComponent) {
                forward = ret;
            }else if ( ret === false){
                forward = false;
            }
        }

        // commit if return is valid
        if (hasOnRequestFocus && forward!==false){
            return forward;
        }
        // fallback case: a valid "lastFocus" option is specified
        else if (useLastFocus && (lastFocus instanceof accedo.ui.AbstractComponent) && lastFocus.isAncestor(component)) {
            forward = lastFocus;
        }
        // fallback case: a valid "defaultFocus" option is specified (in string)
        else if (defaultFocus && accedo.Object.isString(defaultFocus) && !mouseOn ) {
            //only have default focus when it is mouseoff
            forward = component.get(defaultFocus);
        }
        // fallback case: a valid "defaultFocus" option is specified (as a component)
        else if (defaultFocus && (defaultFocus instanceof accedo.ui.AbstractComponent)) {
            forward = defaultFocus;
        }
        //fallback case for mouse is off and isMouseFocusbale true
        else if(!mouseOn && component.getOption("mouseFocusableOnly")){
            //only have forwardOnMouseOff focus when it is mouseoff
            forwardOnMouseOff = component.getOption('forwardOnMouseOff');
            if(forwardOnMouseOff && accedo.Object.isString(forwardOnMouseOff)){
                var target = component;
                while (target) {
                    forward = target.get(forwardOnMouseOff);
                    if(forward){
                        break;
                    }
                    target = target.getParent();
                }
            }else if(forwardOnMouseOff && forwardOnMouseOff instanceof accedo.ui.AbstractComponent){
                forward = forwardOnMouseOff;
            }else if(this._lastNonMouseFocus){
                forward = this._lastNonMouseFocus;
            }
        }
        
        return forward;
    },
    /**
     * Find the latest focused component from most active trail.
     * @name _findLastActiveFocus
     * @param {accedo.ui.AbstractComponent} [from] - The component node to start 
     * at.
     * @return {accedo.ui.AbstractComponent|null} Null if entire trail has not
     * been focused before.
     * @memberof accedo.ui.FocusManager#
     * @function
     * @private
     */
    _findLastActiveFocus: function(from) {
        
        
        from = from || accedo.ui.AppSystem.singleton();
        
        var component = from.getActiveLeaf(),
        lastFocused = null, lastActiveFocus = null;
        
        while (component) {
            
            if (!component.isFocusable()) {
                component = component.getParent();
                continue;
            }
            
            if (lastFocused < component.getLastFocused()) {
                lastActiveFocus = component;
                lastFocused = component.getLastFocused();
            } 
            component = component.getParent();
        }
        
        if (lastFocused===null) {
            return null;
        }
        
        return lastActiveFocus;
    },
    /**
     * @private
     * @ignore
     */
    _handleFocusDetached: function() {
        this.requestFocus(this._findLastActiveFocus(),{
            noActive:true
        });
    },
    /**
     * Explicitly focus given component, if possible.
     * @param {accedo.ui.AbstractComponent} component Component to focus
     * @param {Object} [options] - Optional. Param holder.
     * @param {noActive} [options] - Optional. True to not set Active while 
     * requested focus.
     * @memberof accedo.ui.FocusManager#
     * @function
     * @return {boolean} true if successfully focused a component or removed focus (if so is requested), false if otherwise
     * @public
     */
    requestFocus: function(component,options) {
        var orig = component,
        target = this._getForwardFocus(component), componentParent = null;
        
        while (target) {
            component = target;
            target = null;
            target = this._getForwardFocus(component);
        }
        
        if (orig===null || (component && component.isFocusable())) {
        
            if(this._currentFocus) {
                this._currentFocus.blur();
                this._currentFocus.dispatchEvent(accedo.ui.Evt.BLUR);
                
                componentParent = this._currentFocus.getParent();
                if (componentParent) {
                    componentParent.onChildBlur(component);
                }
                
                if(!this._currentFocus.getOption("mouseFocusableOnly")){
                    this._lastNonMouseFocus = this._currentFocus;
                }
                
                this._lastFocus = this._currentFocus;
                this._lastFocus.removeEventListener(accedo.ui.Evt.DETACHED_FROM_DOM,this._handleFocusDetached,this);
            }
            
            this._currentFocus = null;
            componentParent = null;
        
            if (component && component.isFocusable() && component.focus()) {
                this._currentFocus = component;
                
                //Set focus to be active
                if (!(options && options.noActive)) {
                    this._currentFocus.setActive();
                }
                
                this._currentFocus.dispatchEvent(accedo.ui.Evt.FOCUS);
                this._currentFocus.addEventListener(accedo.ui.Evt.DETACHED_FROM_DOM,this._handleFocusDetached,this);
                
                componentParent = component.getParent();
                if (componentParent) {
                    componentParent.onChildFocus(component);
                }
            }
            return true;
        }
        else {
            accedo.console.warn("Delegated component is not Focusable!");
            return false;
        }
    },

    /**
     * return the current focus
     * @return {accedo.ui.AbstractComponent} current focused component
     * @memberof accedo.ui.FocusManager#
     * @function
     * @public
     */
    getCurrentFocus : function(){
        return this._currentFocus || null;
    },
    /**
     * return the current focus
     * @param {string} key on the direction
     * @param  {accedo.ui.AbstractComponent} container (optional) to check the direction change handling until this container.
     * @return {Boolean} true if directional navigation is successfully made
     * @memberof accedo.ui.FocusManager#
     * @function
     * @name focusDirectionChangeByKey
     * @public
     */
    focusDirectionChangeByKey:function(key, container){
        var ret = false;
        switch (key) {
            case accedo.VKey.KEY_UP:
                ret =this.focusDirectionChange(accedo.ui.FocusManager.FOCUS_UP, container);
                break;
            case accedo.VKey.KEY_DOWN:
                ret = this.focusDirectionChange(accedo.ui.FocusManager.FOCUS_DOWN,container);
                break;
            case accedo.VKey.KEY_LEFT:
                ret = this.focusDirectionChange(accedo.ui.FocusManager.FOCUS_LEFT,container);
                break;
            case accedo.VKey.KEY_RIGHT:
                ret = this.focusDirectionChange(accedo.ui.FocusManager.FOCUS_RIGHT,container);
                break;
        }
        return ret;
    },
    /**
     * Change focus in given direction.It will check the option of that direction. 
     * Firstly, it will check if the direction handling of current Element exist.If it is undefined, it will check its parent. 
     * If it is defined,it will go through the following sequence
     * null value will stop the focus direction change.
     * callback function will be run first and get back the return value. (false/undefined will continue to check its parent handling, true will stop the focus direction change)
     * string/abstract component ,it will be focused and current element will loose focus.
     * @param {accedo.ui.FocusManager.FOCUS_UP | accedo.ui.FocusManager.FOCUS_DOWN | accedo.ui.FocusManager.FOCUS_LEFT | accedo.ui.FocusManager.FOCUS_RIGHT} direction Focus direction
     * @param  {accedo.ui.AbstractComponent} container (optional) to check the direction change handling until this container. 
     * @return {Boolean} true if directional navigation is successfully made
     * @memberof accedo.ui.FocusManager#
     * @function
     * @public
     */
    focusDirectionChange: function(direction, container) {
        
        //handle mouseoff using workstation
        // if (this.isMouseOn()){
        //     this.setMouseOnOff(false);
        //     if(!this.getCurrentFocus()){
        //         return this.requestLastFocus();
        //     }
        // }
        
        if (container && !container.isChildFocused()) {
            return false;
        }
        
        container = container || accedo.ui.AppSystem.singleton();

        var focusable, opt, component;
        
        this._mouseON = false;
        
        if(this._currentFocus) {
            //Block checking predefines
            component = this._currentFocus;
            if(direction === accedo.ui.FocusManager.FOCUS_UP) {
                direction = "nextUp";
            } else if(direction === accedo.ui.FocusManager.FOCUS_DOWN) {
                direction = "nextDown";
            } else if(direction === accedo.ui.FocusManager.FOCUS_LEFT) {
                direction = "nextLeft";
            } else if(direction === accedo.ui.FocusManager.FOCUS_RIGHT) {
                direction = "nextRight";
            }
            
            
            //check for parent's direction recursively until it's defined or 
            //up to root 
            while (component && component !== container) {
                opt = component.getOption(direction);
                
                if(!accedo.Object.isUndefined(opt)){
                    if(opt === null) {
                        //Explicit empty difinition found - block navigation
                        return false;
                    }
                
                    /*additional checking to ensure the return value of the callback function
                    if return value is true, it will stop the recursion and won't do anything.
                    If it is false/ undefined, it will keep running the recursion to check the parent*/
                    if(accedo.Object.isFunction(opt)){
                        opt = opt.call(this._currentFocus);
                        accedo.console.debug("return value of callback"+ opt);
                        if(accedo.Object.isBoolean(opt) && opt){
                            return false;
                        }
                    }

                    //If opt is a component ID / a abstract component, it will stop recursion
                    if (accedo.Object.isString(opt)){
                        focusable = container.get(opt);
                        if(!focusable && container !== accedo.ui.AppSystem.singleton()){
                            focusable = accedo.ui.AppSystem.singleton().get(opt);
                        }
                        break;
                    }else if(opt instanceof accedo.ui.AbstractComponent){
                        focusable = opt;
                        break;
                    }
                }
                component = component.getParent();
            }
        }else{
            focusable = this._findLastActiveFocus();
        }
        
        //Focusable element is defined? If so, request focus for it
        if(focusable && this.requestFocus(focusable)) {
            return true;
        }else{
            return false;
        }
    },
    /**
     * request the last focus
     * @memberof accedo.ui.FocusManager#
     * @function
     * @public
     * @name requestLastFocus
     */
    requestLastFocus:function(){
        if(this._lastFocus){
            return this.requestFocus(this._lastFocus);
        }
        return false;
    }
});

/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * RootController - Apparently the controller that acts the root of all controllers.
 * At the layout point of view, developer can see it as the current focused 
 * display layer. There should be no more than one component that's focused  
 * in each layer. Therefore, the FocusManger works at the RootController level,
 * and handles also navigation events before it's passed to the children 
 * compoenents. For details on how the focus manager works, please see 
 * accedo.ui.FocusManager.
 * 
 * In the layout point of view. It's always a full screen container.
 *
 * @name RootController
 * @memberof accedo.ui
 * @class This is the root controller.
 * @extends accedo.ui.Controller
 * 
 */
accedo.Class.create("accedo.ui.RootController", "accedo.ui.Controller",
    [],{},{
        /**
         * The external key handler array.
         * @name _extKeyHandlers
         * @private
         * @static
         * @memberof accedo.ui.RootController#
         **/
        _extKeyHandlers: [],
        /**
         * Initializes the focus manager and extended controller
         * @name initialize
         * @function
         * @protected
         * @public
         * @memberof accedo.ui.RootController#
         */
        initialize: function($super, opts) {
            opts = opts || {};
            opts.css = "root-controller";
            $super(opts);
            
        },
        /**
        * Overrides the getRootController method
        * @name getRootController
        * @return {accedo.ui.RootController} 
        * @public
        * @memberof accedo.ui.RootController#
        * @function
        **/
        getRootController: function() {
            return this;
        },
        /**
        * Get the focus manager 
        * @name getFocusManager
        * @return {accedo.ui.FocusManager} the focus manager
        * @public
        * @memberof accedo.ui.RootController#
        * @function
        **/
        getFocusManager: function() {
            return accedo.ui.FocusManager.singleton();
        },
        /**
        * Addes a key handler from external source
        * @name addExtKeyHandler
        * @param {Function} callback The handler function
        * @param {Number} index (optional) The index to put the handler function to in the array storage
        * @return {Function} the callback function
        * @public
        * @memberof accedo.ui.RootController#
        * @function
        **/
        addExtKeyHandler: function(callback, index) {
            if (accedo.Object.isUndefined(index) || index<0) {
                index = this._extKeyHandlers.length;
            }
            
            this._extKeyHandlers.splice(index,0,callback);
            
            //return the callback as the handle;
            return callback;
        },
        /**
        * Removes a key handler from external source
        * @name removeExtKeyHandler
        * @param {Function} callback The handler function
        * @return {Function} the callback function
        * @public
        * @memberof accedo.ui.RootController#
        * @function
        **/
        removeExtKeyHandler: function(handle){
            var idx = this._extKeyHandlers.indexOf(handle);
            
            if (!idx) {
                return;
            }
            
            this._extKeyHandlers.splice(idx,1);
        },
        
        onKey: function($super, vKey, activeLeaf) {
            var handler, i;
            
            // if not handled, check if there are any _extKeyHandlers
            for (i=0;i< this._extKeyHandlers.length;i++) {
                handler = this._extKeyHandlers[i];
                
                if (handler(vKey)){
                    return true;
                }
            }
            
            return $super(vKey,activeLeaf);
        }
        
    });
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * @name AppController
 * @memberof accedo.ui
 * @class AppController
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @example var appController = accedo.ui.AppController.singleton();
 * @extends accedo.ui.RootController        
 */
accedo.Class.create("accedo.ui.AppController", "accedo.ui.RootController", 
    ["accedo.ui.layout.Normal"],
    {
        /**
         * Configuration value of enabling history manager, true by default
         * @name ENABLE_HISTORY 
         * @const
         * @memberof accedo.ui.AppController#
         */
        ENABLE_HISTORY: accedo.config.get("ui.appcontroller.enablehistory",true),
        /**
         * Internal class singleton instance.
         * @name _singleton
         * @static
         * @private
         * @memberof accedo.ui.AppController#
         **/
        _singleton: null,
        /**
         * @override
         * @public
         * @protected
         * @memberof accedo.ui.AppController#
         **/
        main:function() {
            accedo.Env.addEventListener(accedo.Env.EVT_ONLOAD, this._onload,this,true);
        },
        /**
         * Internal accedo.Env.EVT_ONLOAD event listener
         * @name _onload
         * @function
         * @private
         * @static
         * @memberof accedo.ui.AppController#
         **/
        _onload:function() {
            accedo.ui.AppSystem.singleton().attach(accedo.ui.AppController.singleton());
            if(accedo.ui.ctrlr && accedo.ui.ctrlr.Loading && accedo.ui.ctrlr.Loading.getAutoStart()){
                accedo.ui.ctrlr.Loading.doingAutoStart(accedo.ui.AppController.singleton());
            }else{
                accedo.ui.AppSystem.singleton().setCurrentController(accedo.ui.AppController.singleton());
            }
        },
        /**
         * Returns the singleton instance of this class.
         * @name singleton
         * @function
         * @return {accedo.ui.AppController#} The singleton instance.
         * @public
         * @static
         * @memberof accedo.ui.AppController#
         **/
        singleton: function() {
            if (this._singleton === null)
            {
                this._singleton = new accedo.ui.AppController();
            }
            return this._singleton;
        },
        /**
         * Sets the current App.
         * @name setApp
         * @function
         * @param {Object} appDef Application sub-controller definition object
         * @public
         * @static
         * @memberof accedo.ui.AppController#
         **/
        setApp: function(appDef) {
            this.singleton().setApp(appDef);
        },
        /**
         * A proxy method to the singleton's changeToController(), for easier access.
         * @name changeToController
         * @function
         * @param {Object} appDef - Application sub-controller definition object
         * @public
         * @static
         * @memberof accedo.ui.AppController
         **/
        changeToController: function(controller, context, options) {
            this.singleton().changeToController(controller, context, options);
        },
        /**
         * Bootstraps App Controller, firing up the App.
         * @name ready
         * @function
         * @public
         * @static
         * @memberof accedo.ui.AppController#
         **/
        ready: function() {
            this.singleton().onload();
        }
    },
    {
        /**
         * The application definition.
         * @name _appDef
         * @memberof accedo.ui.AppController#
         * @private
         */
        _appDef: null,
        /**
         * The storage map for controller definitions.
         * @name _controllerDefs
         * @memberof accedo.ui.AppController#
         * @private
         */
        _controllerDefs: {},
        /**
         * The history manager instance.
         * @name _historyManager
         * @memberof accedo.ui.AppController#
         * @private
         */
        _historyManager: null,
        /**
         * @override
         * @name initialize
         * @memberof accedo.ui.AppController#
         * @protected
         * @public
         */
        initialize: function($super) {
            if (accedo.ui.AppController.ENABLE_HISTORY) {
                this._historyManager = new accedo.ui.HistoryManager(this);
            }
            $super({
                id: "#app-controller"
            });
        },
        /**
         * Set Application definition.
         * @name setApp
         * @memberof accedo.ui.AppController#
         * @function
         * @public
         * @param {Object} appDef - Application Definition Object
         */
        setApp: function(appDef) {
            freesat.console.log('@X@ Freetime app start');
            this._appDef = appDef;
        },
        /**
         * Get the history manager
         * @name getHistoryManager
         * @return {accedo.ui.HistoryManager} the focus manager
         * @public
         * @memberof accedo.ui.AppController#
         **/
        getHistoryManager: function() {
            return this._historyManager;
        },
        /**
         * Changes current view to a new controller.
         * @name changeToController
         * @memberof accedo.ui.AppController#
         * @function
         * @public
         * @param {String} controller's title of the new controller
         * @param {Object} context - Parameters holder to be passed to the 
         * controller's setup function, if any (optional)
         */
        changeToController: function(controller, context, frontFocus) {
            var options = {},ctrlMgr, activeChild;
            if(!accedo.Object.isPlainObject(frontFocus)){
                //backward compatible which is frontFocus
                options.frontFocus = frontFocus;
            }else{
                options = accedo.Object.clone(frontFocus,true);
            }
			
            options.onSuccess = function(controller){
                controller.setFocus();
                if(frontFocus && accedo.Object.isFunction(frontFocus.onSuccess)){
                    frontFocus.onSuccess();
                }
            };

            options.onFailure = function(){
                accedo.console.error("Failed to change controller: "+ controller);
                if(frontFocus && accedo.Object.isFunction(frontFocus.onFailure)){
                    frontFocus.onFailure();
                }
            };		

            options.onHistoryBack = function() {
                
                if(frontFocus && accedo.Object.isFunction(frontFocus.onHistoryBack)){
                    frontFocus.onHistoryBack();
                }
            };
						
            if(accedo.Object.isString(controller)) {
                
                ctrlMgr = accedo.ui.ControllerManager.singleton();
                
                if (this._controllerDefs[controller]) {
                    //creates the controller instance
                    activeChild = this.getActiveChild();
                    if (activeChild){
                        ctrlMgr.replaceController(activeChild,this._controllerDefs[controller].controller, context, options);
                    } else {
                        options.noHistory = true;
                        ctrlMgr.openController(this._controllerDefs[controller].controller, this, context , options);
                    }
                }
                else {
                    accedo.console.error("Undefined controller: id="+controller);
                }
            }
        },
        /**
         * Bootstrap function to be called onLoad of DOM (or any time desired).
         * It will initializes main app controller defined
         * 
         * @name onload
         * @memberof accedo.ui.AppController
         * @function
         * @protected
         * @public
         */
        onload: function() {

            if (this._appDef && accedo.Object.isPlainObject(this._appDef)) {
                if (accedo.Object.isPlainObject(this._appDef.controllers)) {

                    var mainControllerId, controllerDef, firstControllerId, id,
                    controllers = this._appDef.controllers;

                    for (id in controllers) {
                        controllerDef = this._appDef.controllers[id];

                        this._controllerDefs[id] = controllerDef;

                        if (!firstControllerId) {
                            firstControllerId = id;
                        }

                        if (controllerDef.main === true) {
                            mainControllerId = id;
                        }
                    
                    }

                    if (!mainControllerId && firstControllerId) {
                        mainControllerId = firstControllerId;
                    }

                    this.changeToController(mainControllerId, {
                        id: mainControllerId
                    },true);
                }
            }

        },
        getCurrentController:function(){
            return this.getActiveChild();
        }
        
    });
/*jslint browser: true, devel: true */
/*global accedo: false */
/**
 * <pre>
 * Root of the XDK Application Framework 
 * AppSystem acts as the proxy of all input signals that will be passed to the 
 * controllers. In terms of the architecture. Only Root controllers can be 
 * attached to it. It keep tracks of the current RootController and only forward
 * the events to it.
 * </pre>
 * 
 * @class Application system controller, for global app handling
 * @author <a href="mailto:andy.hui@accedo.tv">Andy Hui</a>
 * @name AppSystem
 * @memberof accedo.ui
 * @example var appSystem = accedo.ui.AppSystem.singleton();
 * @extends accedo.ui.layout.Absolute
 */
accedo.Class.create("accedo.ui.AppSystem", "accedo.ui.Controller", 
["accedo.Env","accedo.ui.FocusManager", "accedo.ui.AppController"],
    {
        /**
         * Delegate onLoad
         * @name onLoad
         * @static
         * @memberof accedo.ui.AppSystem#
         */
        onLoad: null,
        /**
         * Internal class singleton instance.
         * @name _singleton
         * @private
         * @static
         * @memberof accedo.ui.AppSystem#
         **/
        _singleton: null,
        /**
         * Internal modules storage map object.
         * @name _modules
         * @private
         * @static
         * @memberof accedo.ui.AppSystem#
         **/
        _modules: [],
        /*
         * Singleton method to get the instance of AppSystem
         * @name singleton
         * @static
         * @returns {accedo.ui.AppSystem#} The AppSystem singleton instance
         * @function
         * @memberof accedo.ui.AppSystem#
         * @public
         */
        singleton: function() {
            if (this._singleton === null) {
                this._singleton = new accedo.ui.AppSystem();
                if (accedo.Object.isFunction(this.onLoad)) {
                   this.onLoad(this._singleton);
                }
            }
            return this._singleton;
        },  
        /*
         * @override
         * @name main
         * @static
         * @function
         * @public
         * @protected
         * @memberof accedo.ui.AppSystem#
         */
        main: function(){
            this._initModules();
        },
        /*
         * Internal method that initializes all required moduls for the App System.
         * @name _initModules
         * @static
         * @function
         * @private
         * @memberof accedo.ui.AppSystem#
         */
        _initModules: function() {
            //init loading controller if needed
            if (accedo.config.get('loading.enable', false)){
                this._modules.push("accedo.ui.ctrlr.Loading");
            }
            
            accedo.loadDefinitions(this._modules);
        }
    },
    {
        /**
         * Resolution of the current Apps
         * @name _resolution
         * @memberof accedo.ui.AppSystem#
         * @private
         */
        // XDK CHANGE
        // _resolution:null,
        /*
         * Overrides the initialize method
         * @name initialize
         * @function
         * @public
         * @protected
         */
        initialize: function($super) {
            
            $super({
                id: "#TVArea"
            });
            
            this.getRoot().addClass(accedo.platform);
            this._resolution = accedo.config.get("ui.appsystem.resolution", {
                // XDK CHANGE
                // "width":accedo.Env.resolution.width,
                // "height":accedo.Env.resolution.height
                "width": 1280,
                "height": 720
            });
            this.getRoot().addClass("_"+this._resolution.width+"x"+this._resolution.height);
    
            //Centalized keyEventHandler
            accedo.Env.addEventListener(accedo.Env.EVT_ONKEY, accedo.Fn.bind(this.keyEventHandler, this));
            
            document.body.appendChild(this.getRoot().getHTMLElement());
            
        },
        /*
         * get the current resolution size of the application
         * @name getResolution
         * @function
         * @public
         */
        // XDK CHANGE
        // getResolution:function(){
        //     return this._resolution;
        // },
        /*
         * Overrides the getParent method
         * @name getParent
         * @function
         * @public
         */
        getParent: function() {
            return null;
        },
        /**
         * Gets pointer for current RootController instance.
         * @name getCurrentController
         * @memberof accedo.ui.AppSystem#
         * @function
         * @return {accedo.ui.AbstractComponent} 
         * @public
         */
        getCurrentController: function() {
            return this.getActiveChild();
        },
        /**
         * Set current RootController instance.
         * @name setCurrentController
         * @param {accedo.ui.AbstractComponent} component - the child component
         * @memberof accedo.ui.AppSystem#
         * @return {Boolean} False if controller is not 
         * @function
         * @public
         */
        setCurrentController: function(component) {
            return this.setActiveChild(component);
        },
        /*
         * Internal key event handler.
         * @name keyEventHandler
         * @memberof accedo.ui.AppSystem#
         * @param {accedo.VKey} vKey The virtual key event.
         * @return {Boolean} Whether the key event has been handled.
         * @function
         * @public
         * @protected
         */
        keyEventHandler: function (vKey) {
            
            var activeLeaf = this.getActiveLeaf(),
            fm = accedo.ui.FocusManager.singleton(),ret = false,
            onKeyTargetAsCurrentFocus = accedo.config.get("ui.focus.onKeyTargetAsCurrentFocus",true),
            onKeyWhileTargetNotNullOnly = accedo.config.get("ui.focus.onKeyWhileTargetNotNullOnly",false),
            target = onKeyTargetAsCurrentFocus?fm.getCurrentFocus():activeLeaf;

            //Do not pass key if no current focus, for backward compatibility
            if (!(onKeyWhileTargetNotNullOnly && target===null)) {
                // call key capturing and bubbling
                if (this.handleKey(vKey, target, true)) { 
                    return true;
                }
            }
            
            //otherwise try handle by focus manager
            ret = fm.focusDirectionChangeByKey(vKey);
            
            if(ret){
                return true;
            }
            
            if (vKey === accedo.VKey.KEY_EXIT) {
                accedo.device.system.exit();
                return true;
            }
            return false;
        }
        
    });
/**
 * User: Dean Clancy
 * Date: 11/12/13
 * Time: 11:09
 */
/*global accedo: false */
/**
 * Creates a new Freesatconfig instance.
 * @name Freesatconfig
 * @memberof freesat.Freesatconfig
 * @class A Freesatconfig class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.config.Freesatconfig", [], {},
    {
        regions : [],
        ts_streams : [],
        prefix_data : [],
        eitfile_prefix : null,
        recdata_loc: null,
        auth_file_loc: null,
        mheg_loader: null,
        ui_banner_timeout: 5,

        parseXML: function (configXML) {
            try{
                var i;
                var data = configXML.getElementsByTagName(freesat.datamode)[0];
                var regions = data.getElementsByTagName('regions')[0];

                for(i=0; i< regions.childNodes.length; i++){
                    var obj = {};
                    try{
                        obj.id = regions.childNodes[i].getAttribute('id');
                        obj.pr_id = regions.childNodes[i].getAttribute('pr_id');
                        obj.sr_id = regions.childNodes[i].getAttribute('sr_id');
                        obj.pr_name = regions.childNodes[i].getAttribute('pr_name');
                        obj.sr_name = regions.childNodes[i].getAttribute('sr_name');
                        this.regions.push(obj);
                    }catch(e){
                        //ignore
                    }
                }
                var ts_streams = data.getElementsByTagName('ts_streams')[0];
                for(i=0; i< ts_streams.childNodes.length; i++){
                    try{
                        var obja = {};
                        obja.id = ts_streams.childNodes[i].getAttribute('id');
                        obja.tsid = ts_streams.childNodes[i].getAttribute('tsid');
                        obja.nid = ts_streams.childNodes[i].getAttribute('nid');
                        this.ts_streams.push(obja);
                    }catch(e){
                        //ignore
                    }

                }
                var prefix_data = data.getElementsByTagName('prefix_data')[0];
                for(i=0; i< prefix_data.childNodes.length; i++){
                    try{
                        var objb = {};
                        objb.id = prefix_data.childNodes[i].getAttribute('id');
                        objb.table = prefix_data.childNodes[i].getAttribute('table');
                        objb.url = prefix_data.childNodes[i];
                        this.prefix_data.push(objb);
                    }catch(e){
                        //ignore
                    }
                }
                try{
                    this.recdata_loc = data.getElementsByTagName('recdata_loc')[0].childNodes[0].nodeValue;
                    this.auth_file_loc = data.getElementsByTagName('auth_file_loc')[0].childNodes[0].nodeValue;
                }catch(e){
                    //ignore
                }

                try {
                    this.mheg_loader = data.getElementsByTagName('mheg_loader')[0].childNodes[0].nodeValue;
                } catch (e) {
                    // ignore
                }

                try {
                    this.eitfile_prefix = data.getElementsByTagName('eitfile_prefix')[0].childNodes[0].nodeValue;
                } catch (e) {
                    // ignore
                }

                try {
                    var hometp = data.getElementsByTagName('hometp')[0].getElementsByTagName('ts')[0];
                    this.home_transponder = {
                        onid: parseInt(hometp.getAttribute('onid')),
                        tsid: parseInt(hometp.getAttribute('tsid')),
                        svcid: parseInt(hometp.getAttribute('svcid'))
                    };
                } catch (e) {
                    // ignore
                    console.log(e);
                }

                try {
                    var oSerializer = new XMLSerializer();
                    freesat.console.log('Freesat config: ' + oSerializer.serializeToString(configXML));
                } catch (e) {
                    freesat.console.log('couldn\'t serialize ');
                }
                try{
                    var vendor = configXML.getElementsByTagName('vendor')[0];
                    this.ui_banner_timeout = vendor.getElementsByTagName('ui_banner_timeout')[0].childNodes[0].nodeValue;
                }catch(e){
                    freesat.console.log('freesatconfig ui_banner_timeout FAIL : ' + e);
                }
            }catch(e){
                freesat.console.log('freesatconfig FAIL : ' + e);
            }
        }
    }
);
/**
 * User: Dean Clancy
 * Date: 11/12/13
 * Time: 12:59
 */
/*global accedo: false */
/**
 * Creates a new Auth instance.
 * @name Auth
 * @memberof freesat.Auth
 * @class A Auth class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.config.Auth", [], {},
    {
        DEFAULT_IP_ABOUT_TIMEOUT_SECS: 10,
        DEFAULT_IP_SEARCH_TIMEOUT_SECS: 15,

        authAsString: null,

        username: null,
        password: null,
        type: null,

        receiverData: null,
        epg: null,
        images: null,
        help: null,
        ipchannelsmgr: null,
        epgAbout: null,
        onDemand: null,
        showcase: null,

        appMonitor: null,
        appMonitorCustomData: null,
        appMonitorFailureUrl: null,
        appMonitorSessionTimeoutSecs: null,
        appMonitorSessionRefreshSecs: null,

        ipservicelist : null,

        devicemanagement : null,
        analytics_storeLimit: 2500,
        analytics: null,
        dialacl: null,
        freetimeui: null,

        mhegplayers: null,
        loggedAuthBefore: false,

        parseElementTextValue: function(authXML, tagName) {
            var result = null;
            try {
                result = authXML.getElementsByTagName(tagName)[0].childNodes[0].nodeValue;
            } catch(e) {
                // fall-through
            }

            if (!this.loggedAuthBefore) {
                this.loggedAuthBefore = true;
                try {
                    var oSerializer = new XMLSerializer();
                    this.authAsString = oSerializer.serializeToString(authXML);
                    freesat.console.log('Auth: ' + this.authAsString);
                } catch (e) {
                    freesat.console.log('couldn\'t serialize ');
                }
            }

            if (!result) {
                freesat.console.log('********')
                freesat.console.log('********')
                freesat.console.log('WARNING: Failed to read tag "' + tagName + '" from auth file');
                freesat.console.log('********')
                freesat.console.log('********')
            }
            return result;
        },

        parseElementAttribute: function(authXML, tagName, attrib) {
            var result = null;
            try {
                result = authXML.getElementsByTagName(tagName)[0].getAttribute(attrib);
            } catch(e) {
                // fall-through
            }

            if (!result) {
                freesat.console.log('********')
                freesat.console.log('********')
                freesat.console.log('WARNING: Failed to read tag "' + tagName + '", attribute "' + attrib + '" from auth file');
                freesat.console.log('********')
                freesat.console.log('********')
            }
            return result;
        },

        parseXML: function (authXML) {
            this.password = this.parseElementAttribute(authXML, 'auth', 'password');
            this.username = this.parseElementAttribute(authXML, 'auth', 'username');
            this.type = this.parseElementAttribute(authXML, 'auth', 'type');
            this.receiverData = this.parseElementTextValue(authXML, 'receiverData');
            this.epg = this.parseElementTextValue(authXML, 'epg');
            this.images = this.parseElementTextValue(authXML, 'images');
            this.ipservicelist = this.parseElementTextValue(authXML, 'ipservicelist');
            this.help = this.parseElementTextValue(authXML, 'help');
            this.ipchannelsmgr = this.parseElementTextValue(authXML, 'ipchannelsmgr');
            this.epgAbout = this.parseElementTextValue(authXML, 'epgAbout');
            this.epgAboutTimeoutSecs = this.parseElementAttribute(authXML, 'epgAbout', 'timeoutSeconds');
            this.epgSearch = this.parseElementTextValue(authXML, 'epgSearch');
            this.epgSearchTimeoutSecs = this.parseElementAttribute(authXML, 'epgSearch', 'timeoutSeconds');
            this.devicemanagement = this.parseElementTextValue(authXML, 'devicemanagement');
            this.analytics = this.parseElementTextValue(authXML, 'analytics');
            this.analytics_storeLimit = this.parseElementAttribute(authXML, 'analytics', 'storeLimit');
            this.dialacl = this.parseElementTextValue(authXML, 'dialacl');
            this.freetimeui = this.parseElementTextValue(authXML, 'freetimeui');
            this.mhegplayers = this.parseElementTextValue(authXML, 'mhegplayers');
            this.onDemand = this.parseElementTextValue(authXML, 'od');
            this.showcase = this.parseElementTextValue(authXML, 'sc');
            this.appMonitor = this.parseElementTextValue(authXML, 'appmonitor');
            this.appMonitorCustomData = this.parseElementAttribute(authXML, 'appmonitor', 'customdata');
            this.appMonitorFailureUrl = this.parseElementTextValue(authXML, 'failure');
            this.appMonitorSessionTimeoutSecs = this.parseElementAttribute(authXML, 'appmonitor', 'sessionTimeoutSecs');
            this.appMonitorSessionRefreshSecs = this.parseElementAttribute(authXML, 'appmonitor', 'sessionRefreshSecs');

            if (!this.onDemand) {
                this.onDemand = this.parseElementTextValue(authXML, 'regional');
            }

            if (!this.showcase) {
                this.showcase = this.parseElementTextValue(authXML, 'regional');
            }

            if (!this.epgAboutTimeoutSecs) {
                this.epgAboutTimeoutSecs = this.DEFAULT_IP_ABOUT_TIMEOUT_SECS;
            }

            if (!this.epgSearchTimeoutSecs) {
                this.epgSearchTimeoutSecs = this.DEFAULT_IP_SEARCH_TIMEOUT_SECS;
            }

            if (window.localStorage.getItem('HACK_SEARCH_ABOUT') === 'true') {
                freesat.console.log('********')
                freesat.console.log('********')
                freesat.console.log('WARNING: IP Search and IP About hack enabled');
                freesat.console.log('********')
                freesat.console.log('********')

                this.epgAbout = 'http://fdp-sv21-ip-about-ng-v1-0.int1.freetime-platform.net/dsat/regional';
                this.epgSearch = 'http://fdp-sv22-ip-search-ng-v1-0.int1.freetime-platform.net/dsat/regional';
            }
        }
    }
);
/**
 * User: Dean Clancy
 * Date: 11/12/13
 * Time: 13:18
 */
/*global accedo: false */
/**
 * Creates a new Ipservicelist instance.
 * @name Ipservicelist
 * @memberof freesat.Ipservicelist
 * @class A Ipservicelist class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.config.Ipservicelist", [], {},
    {
        onDemandChannel: {},
        numberOfOnDemandChannels: 0,
        ondemandFSATids: [],
        parseXML: function(ipslXML){
            try{
                var i, cutv = ipslXML.getElementsByTagName('cutvServices')[0];
                for(i=0; i<cutv.childNodes.length; i++){
                    try{
                        if(cutv.childNodes[i].getAttribute('cutv')==='true'){
                            //freesatSID ===  nid - tsid - sid
                            var freesatSID = cutv.childNodes[i].getAttribute('freesatSID');
                            var obj = {};
                            obj.id = cutv.childNodes[i].getAttribute('id');
                            obj.SID = cutv.childNodes[i].getAttribute('SID');
                            obj.dvb = cutv.childNodes[i].getAttribute('dvb');
                            this.onDemandChannel[String(freesatSID)] = obj;
                            this.numberOfOnDemandChannels = this.numberOfOnDemandChannels +1;
                            this.ondemandFSATids.push(String(freesatSID));
                        }
                    }catch(e){
                        //ignore
                    }
                }
            }catch(e){
                freesat.console.log(e);
            }
        }
    }
);
/**
 * User: Dean Clancy
 * Date: 21/01/14
 * Time: 10:54
 */
/*global accedo: false */
/**
 * Creates a new Imagefactory instance.
 * @name Imagefactory
 * @memberof freesat.Imagefactory
 * @class A Imagefactory class, functionality to determine the url used to procure an image.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.Imagefactory", [], {},
    {
        /**
         *
         * @param defStr    the string given by the provider to supply the image
         * @param imgwidth  image width required
         * @param imgheight image height required
         * @param firstcall is this the first call for this image (if fs-img another location can be provided)
         * @returns {String}
         */
        getImageURL: function (defStr, imgwidth, imgheight, firstcall) {
            var retVar = '';
            try{
                if(defStr){
                    if(defStr.indexOf('http://')>-1){
                        retVar = defStr;
                    }else if(defStr.indexOf('fs-img://')>-1){
                        var imgArr = freesat.localConfig.imgmapData.imageMappings.img, i, stbImg, ipImg;
                        if(imgArr && imgArr.length > 0){
                            for(i=0;i<imgArr.length;i++){
//                                freesat.console.log('imgArr[i].id :'+imgArr[i].id);
                                if(imgArr[i].id === defStr){
                                    if(firstcall)
                                        retVar = freesat.localConfig.auth.image + imgArr[i].ip;
                                    else
                                        retVar = freesat.localConfig.config.recdata_loc + imgArr[i].stb;
                                    break;
                                }
                            }
                        }
                    } else if (defStr.indexOf('dsm://') === 0) {
                        var relativeUrl = defStr.substr('dsm://'.length);
                        retVar = freesat.localConfig.config.eitfile_prefix + '/' + relativeUrl;
                    } else{
                        retVar = freesat.localConfig.auth.images +'/'+ String(imgwidth)+ 'x' + String(imgheight) + '-0' + defStr;
                    }
                }
            }catch(e){
                freesat.console.log('Imagefactory.getImageURL [ERROR] '+ e);
            }
            return retVar;
        }
    }
);
    /**
 * User: Dean Clancy
 * Date: 14/11/13
 * Time: 12:46
 */
/*global accedo: false */
/**
 * Creates a new GetLocalFile instance.
 * @name GetLocalFile
 * @memberof freesat.GetLocalFile
 * @class A GetLocalFile class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.GetLocalFile", [], {},
    {
        /**
         * retrieveXML
         * @description retrieves fileName from local filesystem
         * @param fileName
         */
        retrieveXML: function (fileName, callback) {
            var client;
            if(!freesat.oipfEnabled){
                var arr = fileName.split('///');
                var url = location.href.split('main');
                fileName = url[0] + arr[1];
            }
            freesat.console.log(fileName);
            if (window.XMLHttpRequest){
                client=new XMLHttpRequest();
            }
            client.open("GET",fileName,false);
            client.onreadystatechange = this.readyStateChange(client, callback);
            client.send();
        },
        readyStateChange : function(client, callback) {
            return function (){
                if (client.readyState == XMLHttpRequest.DONE) {
                    //freesat.console.log('client.readyState ' + client.readyState);
                    //freesat.console.log('client.status ' + client.status);
                    callback(client.responseXML);
                }
            };
        },
        retrieveJSON: function(fileName, callback, rJSON) {
            var client;
            if(!freesat.oipfEnabled){
                var arr = fileName.split('///');
                var url = location.href.split('main');
                fileName = url[0] + arr[1];
            }
            freesat.console.log(fileName);
            if (window.XMLHttpRequest){
                client=new XMLHttpRequest();
            }
            client.open("GET",fileName,false);
            client.onreadystatechange = this.jsonReadyStateChange(client, callback, rJSON);
            client.send();
        },
        jsonReadyStateChange : function(client, callback, rJSON) {
            return function (){
                if (client.readyState == XMLHttpRequest.DONE) {
                    //freesat.console.log('client.readyState ' + client.readyState);
                    //freesat.console.log('client.status ' + client.status);
                    if(rJSON){
                        callback(JSON.parse(client.responseText));
                    }else{
                        callback(client.responseText);
                    }
                }
            };
        }

    }
);
/**
 * Gets a remote file with timeout, basic auth etc.
 */
accedo.Class.create("freesat.models.GetRemoteFile", [], {}, {
    _getInternal: function(isXmlRequest, url, successCallback, failureCallback, username, password) {
        var client = new XMLHttpRequest();

        client.open('GET', url, true, username, password);

        var that = this;
        var gotTimeout = false;
        var timeoutTimer;

        client.onreadystatechange = function() {
            if (client.readyState == XMLHttpRequest.DONE) {
                window.clearTimeout(timeoutTimer);

                var response = isXmlRequest ? client.responseXML : client.responseText;

                if ((client.status == 200) && response) {
                    freesat.console.log('Successfully got ' + url);
                    successCallback(response);
                } else {
                    freesat.console.log('Failed to get ' + url + ' (status code = ' + client.status + ')');
                    failureCallback(client.status);
                }
            }
        }

        timeoutTimer = window.setTimeout(function() {
            freesat.console.log('Got timeout waiting for ' + url);
            gotTimeout = true;
            client.abort();
        }, freesat.defaultHttpTimeoutMs);

        client.send();
    },

    getAsync: function(url, successCallback, failureCallback, username, password) {
        this._getInternal(false, url, successCallback, failureCallback, username, password);
    },

    getAsyncXml: function(url, successCallback, failureCallback, username, password) {
        this._getInternal(true, url, successCallback, failureCallback, username, password);
    }
});
/**
 * User: Dean Clancy
 * Date: 14/11/13
 * Time: 12:54
 */
/*global accedo: false */
/**
 * Creates a new LocalConfig instance.
 * @name LocalConfig
 * @memberof freesat.LocalConfig
 * @class A LocalConfig class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.LocalConfig", ['freesat.models.GetLocalFile', 'freesat.models.config.Freesatconfig', 'freesat.models.config.Auth', 'freesat.models.config.Ipservicelist'], {}, {
        config: null,
        auth: null,
        ipservicelist: null,
        baseURL : null,
        updateData : null,
        imgmapData: null,
        callback: null,
        initialize: function () {
            this.ipservicelist = new freesat.models.config.Ipservicelist();
            this.auth = new freesat.models.config.Auth();
        },
        startup: function(callback) {
            var getter = new freesat.models.GetLocalFile();
            this.callback = callback;
            this.loadConfig(getter);
        },
        loadConfig: function(getter){
            try{
                var confURL = accedo.config.get('config_url');
                freesat.console.log('freesat.testingHooks : ' + freesat.testingHooks);
                if(freesat.testingHooks){
                    this.baseURL = location.href.split('main.html')[0];
                    freesat.console.log('base : ' + this.baseURL);
                    confURL = this.baseURL + confURL.split('///')[1];
                }
                getter.retrieveXML(confURL, this.configSuccess(this, getter));
            }catch(e){
                freesat.console.log('fsat config failure : ' + e);
            }
        },
        configSuccess: function(scope, getter){
            return function(configXML){
                if(configXML){
                    scope.config = new freesat.models.config.Freesatconfig();
                    scope.config.parseXML(configXML);
                    scope.loadAuth(getter);
                }else{
                    freesat.console.log('fsat config failure : empty ');
                }
            };
        },
        loadAuth: function(getter){
            try{
                var authURL = this.config.auth_file_loc;
                if(this.baseURL){
                    authURL = this.baseURL + String(authURL).split('///')[1];
                }

                // Use local auth.xml file if specified is querystring
                if (window.useLocalAuth){
                    authURL = location.href.split('main.html')[0] + 'fakedata/auth_dtt.xml';
                }

                getter.retrieveXML(authURL, this.authSuccess(this, getter));
            }catch(e){
                freesat.console.log('fsat auth failure : ' + e);
            }
        },
        authSuccess: function(scope, getter){
            return function(authXML){
                if(authXML){
                    scope.auth.parseXML(authXML);
                }else{
                    freesat.console.log('fsat auth failure : empty ');
                }
                scope.loadIPServiceList(scope, getter);
            };
        },
        loadIPServiceList: function(scope, getter){
            try{
                var ipslURL;
                if(this.baseURL){
                    ipslURL = this.config.recdata_loc + '/ipservicelist/ipservicelist.xml';
                    ipslURL = this.baseURL + String(ipslURL).split('///')[1];
                    getter.retrieveXML(ipslURL, this.ipslSuccess(this, getter));
                }else{
                    ipslURL = this.auth.ipservicelist + '/ipservicelist/ipservices/ipservices.xml';

                    var doneCallback = scope.ipslSuccess(scope, getter);
                    var remoteGetter = new freesat.models.GetRemoteFile();
                    remoteGetter.getAsyncXml(
                        ipslURL,
                        function(result) {
                            doneCallback(result);
                        },
                        function() {
                            doneCallback(null);
                        }
                    );
                }
            }catch(e){
                freesat.console.log('fsat ipsl failure : ' + e);
            }
        },
        ipslSuccess: function(scope, getter){
            return function(ipslXML){
                if(ipslXML){
                    try{
                        scope.ipservicelist.parseXML(ipslXML);
                    }catch(e){
                        freesat.console.log('fsat ipsl FAIL : '+ e);
                        freesat.ipServicesLoadFailed = true;
                    }
                }else{
                    freesat.console.log('fsat ipsl failure : empty ');
                    freesat.ipServicesLoadFailed = true;
                }
                var updateURL = scope.config.recdata_loc + '/jsupdate/update.json';
                if(scope.baseURL){
                    updateURL = scope.baseURL + String(updateURL).split('///')[1];
                }

                getter.retrieveJSON(updateURL, scope.updateDataSuccess(scope, getter));
            };
        },
        updateDataSuccess: function(scope, getter){
            return function(updateJSON){
                if(updateJSON){
                    freesat.console.log('update data : parse ');
                    try{
                        scope.updateData = JSON.parse(updateJSON);
                    }catch(e){
                        freesat.console.log('update parse FAIL : ' + e);
                    }
                    freesat.console.log('update data : parse DONE ');
                }else{
                    freesat.console.log('update data : empty ');
                }
                var imgmapURL = scope.config.recdata_loc + '/img/imgmap.json';
                if(scope.baseURL){
                    imgmapURL = scope.baseURL + String(imgmapURL).split('///')[1];
                }
                getter.retrieveJSON(imgmapURL, scope.imgmapDataSuccess(scope, getter));
            };
        },
        imgmapDataSuccess: function(scope, getter){
            return function(imgmapJSON){
                if(imgmapJSON){
                    freesat.console.log('update data : parse ');
                    try{
                        scope.imgmapData = JSON.parse(imgmapJSON);
                        freesat.channelData.setChannelLogos(imgmapJSON);

                    }catch(e){
                        freesat.console.log('imgmap parse FAIL : ' + e);
                    }
                    freesat.console.log('imgmap data : parse DONE ');
                }else{
                    freesat.console.log('imgmap data : empty ');
                }
                scope.loadDsmcc(scope, getter);
            };
        },
        loadDsmcc: function(scope, getter) {
            freesat.console.log('in loadDsmcc');
            try {
                console.log('eitfile_prefix=' + scope.config.eitfile_prefix);
                var dsmccStatusUrl = 'file:///transient/freetime/freetimed/ram/dsm_status.json';
                console.log('Loading dsmcc status from ' + dsmccStatusUrl);
                getter.retrieveJSON(dsmccStatusUrl, scope.dsmccSuccess(scope, getter), false);
            } catch(e){
                freesat.console.log('DSM-CC status fetch failure : ' + e);
                scope.callback();
            }
        },
        dsmccSuccess: function(scope, getter) {
            freesat.console.log('in dsmccSuccess');
            return function(dsmccStatus) {
                if(dsmccStatus) {
                    console.log(dsmccStatus);
                }else {
                    freesat.console.log('DSM-CC status file is empty');
                }
                scope.callback();
            };
        }
    }
);
accedo.Class.create('freesat.models.MarketingOptIn', ['freesat.controllers.MessagePopup'], {},{
    KEY_NAME: 'marketingOptIn',
    initialize: function($super) {
    },
    showDialogIfRequired: function() {
        if (!freesat.marketingOptIn.isLocalStorageAvailable()) {
            return;
        }

        if (!freesat.marketingOptIn.hasExpressedPreference()) {
            // set pref. now as it is the default if user exits dialog
            // (and we won't get an opportunity to intercept that keypress)
            freesat.marketingOptIn.doOptIn();

            freesat.masterController.instanciateMessagePopup(
                    'Freetime update',
                    null,
                    'This latest update has the option for you to share usage information with Freesat. Understanding TV viewing habits helps us to improve the Freetime service. Rest assured that the data collection is anonymous and we would never share personal data without your permission.',
                    null,
                    'OK, I’m happy to help',
                    'No thanks',
                    function(button) {
                        switch(button) {
                            case 0: // back: nothing to do, as opted-in earlier by default
                                break;
                            case 1: // OK: nothing to do, as opted-in earlier by default
                                break;
                            case 2: // No thanks
                                freesat.masterController.instanciateMessagePopup(
                                    'your usage data',
                                    'You have chosen not to share usage data with Freesat.',
                                    'You can change this setting at any time in the \'Your usage data\' section of Help.',
                                    null,
                                    'Confirm and close');

                                freesat.marketingOptIn.doOptOut();
                                break;
                        }
                    },
                    true);
            return;
        }
    },
    hasOptedIn: function() {
        if (!this.isLocalStorageAvailable()) {
            return false;
        }

        return (window.localStorage.getItem(this.KEY_NAME) === 'true');
    },
    doOptIn: function() {
        if (!this.isLocalStorageAvailable()) {
            return;
        }

        freesat.console.log('Opting in to sharing data usage');
        window.localStorage.setItem(this.KEY_NAME, 'true');
        freesat.am.setOptIn(true);
    },
    doOptOut: function() {
        if (!this.isLocalStorageAvailable()) {
            return;
        }

        freesat.console.log('Opting out of sharing data usage');
        window.localStorage.setItem(this.KEY_NAME, 'false');
        freesat.am.setOptIn(false);
    },
    hasExpressedPreference: function() {
        if (!this.isLocalStorageAvailable()) {
            return false;
        }

        if (window.localStorage.getItem(this.KEY_NAME)) {
            return true;
        } else {
            return false;
        }
    },
    cancelPreference: function() {
        if (!this.isLocalStorageAvailable()) {
            return;
        }

        window.localStorage.removeItem(this.KEY_NAME);
    },
    isLocalStorageAvailable: function() {
        if (typeof window.localStorage == 'undefined') {
            freesat.console.log('WARNING - local storage not available, marketing opt-in will not work');
            return false;
        } else {
            return true;
        }
    }
});

/**
 * User: James Rayner
 * Date: 16/04/15
 * Time: 13:26
 */
/*global accedo: false */
/**
 * Creates a new DSM-CC instance.
 * @name Dsmcc
 * @memberof freesat.LocalConfig
 * @class A LocalConfig class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.Dsmcc", ['freesat.models.GetLocalFile'], {}, {
    DSM_STATE_UNKNOWN: 'UNKNOWN',
    DSM_STATE_STOPPED: 'STOPPED',
    DSM_STATE_UPDATING: 'UPDATING',
    DSM_STATE_COMPLETE: 'COMPLETE',
    
    initialize: function () {
    },

    readDsmccState: function(cb) {
        var scope = this;
        var filename = freesat.localConfig.config.eitfile_prefix + '/../dsm_status.json';
        new freesat.models.GetLocalFile().retrieveJSON(
            filename,
            function(result) {
                if (result) {
                    // work around invalid JSON returned by middleware
                    if (result.indexOf(scope.DSM_STATE_STOPPED) > -1) {
                        cb(scope.DSM_STATE_STOPPED);
                    } else if (result.indexOf(scope.DSM_STATE_UPDATING) > -1) {
                        cb(scope.DSM_STATE_UPDATING);
                    } else if (result.indexOf(scope.DSM_STATE_COMPLETE) > -1) {
                        cb(scope.DSM_STATE_COMPLETE);
                    } else {
                        freesat.console.log('WARNING: unexpected dsm_status.json ' + result);
                        cb(scope.DSM_STATE_UNKNOWN);
                    }
                } else {
                    freesat.console.log('WARNING: failed to retrieve ' + filename);
                    cb(scope.DSM_STATE_UNKNOWN);
                }
            },
            false);
    },

    getUrlForRegionalDsmccFile: function(filename, useFallback) {
        var FALLBACK_REGION = 65535;
        var regions = freesat.localConfig.config.regions;

        if (!Array.isArray(regions) || (regions.length != 1)) {
            freesat.console.log('Failed to get region');
            return null;
        }

        var primaryRegion = regions[0].pr_id;
        var secondaryRegion = useFallback ? FALLBACK_REGION : regions[0].sr_id;

        return freesat.localConfig.config.eitfile_prefix
            + '/regions/' + primaryRegion + '/' + secondaryRegion
            + '/' + filename;
    },

    getRegionalisedFile: function(filename, cb) {
        var scope = this,
            preferredUrl = this.getUrlForRegionalDsmccFile(filename, false),
            fallbackUrl = this.getUrlForRegionalDsmccFile(filename, true);

        // first look for primary/secondary
        new freesat.models.GetLocalFile().retrieveJSON(preferredUrl, function(data) {
            if (data) {
                freesat.console.log('Found regionalised data using primary/secondary at ' + preferredUrl);
                cb(data);
                return;
            }

            // fallback to primary/65535
            new freesat.models.GetLocalFile().retrieveJSON(fallbackUrl, function(data) {
                if (data) {
                    freesat.console.log('Found fallback regionalised data using primary/65535 at ' + fallbackUrl);
                    cb(data);
                    return;
                }
                
                // not found
                cb(null);
            });
        });
    },

    waitForCarouselState: function(requiredState, optionalAbortState, timeoutMilliseconds, cb) {
        var POLL_INTERVAL_MS = 250;
        var scope = this;
        var startTime = new Date().getTime();
        var checkFunc = function() {
            scope.readDsmccState(function(state) {
                if (scope.cancelWait) {
                    freesat.console.log('DSM-CC carousel update cancelled.');
                    cb(false);
                    return;
                }

                freesat.console.log('DSM-CC state: ' + state);

                if (state == requiredState) {
                    cb(true);
                    return;
                }

                if ((optionalAbortState && (state == optionalAbortState)) || (state == scope.DSM_STATE_UNKNOWN)) {
                    freesat.console.log('DSM-CC carousel update failure. Expecting '
                        + requiredState + ' but got ' + state + '.');
                    cb(false);
                    return;
                }

                if ((new Date().getTime()) - startTime > timeoutMilliseconds) {
                    freesat.console.log('DSM-CC carousel update timeout waiting for state ' + requiredState + '. Final state was ' + state);
                    cb(false);
                    return;
                }

                setTimeout(checkFunc, POLL_INTERVAL_MS);
            });
        }   

        checkFunc();
    }, 

    waitForCarouselUpdateOfFile: function(filename, cb) {
        var scope = this;
        freesat.console.log('tuning to home transponder to attempt retrieval of ' + filename);
        
        this.doCarouselUpdate(function(success) {
            if (scope.cancelWait) {
                return;
            }

            if (!success) {
                cb(null);
                return;
            }

            freesat.console.log('DSM-CC carousel successfully updated. Fetching file: ' + filename);
            if (filename.indexOf('file://') == 0) {
                new freesat.models.GetLocalFile().retrieveJSON(filename, cb);
            } else {
                scope.getRegionalisedFile(filename, cb);
            }
        });
    },

    doCarouselUpdate: function(cb) {
        var UPDATE_START_TIMEOUT_MS = 20000; // 20s
        var UPDATE_COMPLETE_TIMEOUT_MS = 120000; // 2mins

        var scope = this;

        // tune to home transponder, but remember original LCN
        this.cancelWait = false;
        this.downloading = true;
        this.originalLcn = freesat.oipfElement.videoBroadcast.currentChannel.ccid.substr(5);
        freesat.video.channelChanger.channelChange(scope.lookupHomeTransponderLcn());

        scope.waitForCarouselState(
            scope.DSM_STATE_UPDATING,
            null,
            UPDATE_START_TIMEOUT_MS,
            function(success) {
                if (!success) {
                    scope.downloading = false;
                    scope.tuneToOriginalChannel();
                    cb(false);
                    return;
                }

                scope.waitForCarouselState(
                    scope.DSM_STATE_COMPLETE,
                    scope.DSM_STATE_STOPPED,
                    UPDATE_COMPLETE_TIMEOUT_MS,
                    function(success) {
                        scope.downloading = false;
                        scope.tuneToOriginalChannel();

                        if (!success) {
                            cb(false);
                            return;
                        }

                        cb(true);
                    }
                );
            }
        );
    },

    cancelWaitForCarousel: function() {
        freesat.console.log('cancelling wait for DSM-CC carousel');
        this.cancelWait = true;
        this.tuneToOriginalChannel();
    },
    
    tuneToOriginalChannel: function() {
        if (this.originalLcn) {
            freesat.video.channelChanger.channelChange(this.originalLcn);
            this.originalLcn = null;
        }
    },

    lookupHomeTransponderLcn: function() {
        // TODO: hard-coding for time being because iterating channels seems to be very slow (4s approx!)
        return 900;
        /*
        var result = null;
        var homeTransponder = freesat.localConfig.config.home_transponder;
        var channelList = freesat.channelData.channelList;
        for (var i = 0; i < channelList.length; i++) {
            var channel = channelList[i];
            if ((channel.onid == homeTransponder.onid) && (channel.tsid == homeTransponder.tsid)) {
                result = channel.ccid.substr('ccid:'.length); // omit 'ccid:' prefix
                break;
            }
        }
        */
    },

    dumpTopLevelDirectory: function() {
        this.dumpDirectory(freesat.localConfig.config.eitfile_prefix);
    },

    dumpRegionalDirectory: function(optionalPrimaryRegion, optionalSecondaryRegion) {
        var url = freesat.localConfig.config.eitfile_prefix + '/regions';
        if (optionalPrimaryRegion) {
            url += '/' + optionalPrimaryRegion;
            if (optionalSecondaryRegion) {
                url += '/' + optionalSecondaryRegion;
            }
        }
        this.dumpDirectory(url);
    },

    dumpEpgChannelDirectory: function(freesatServiceId) {
        this.dumpDirectory(freesat.localConfig.config.eitfile_prefix + '/epg/' + freesatServiceId)
    },

    dumpDirectory: function(dirUrl) {
        new freesat.models.GetLocalFile().retrieveJSON(
            dirUrl,
            function(result) {
                if (result) {
                    freesat.console.log('DSM-CC directory listing of ' + dirUrl + ': ' + result);
                }else {
                    freesat.console.log('Failed to fetch DSM-CC directory listing of ' + dirUrl);
                }
            },
            false);
    }
});
accedo.Class.create("freesat.models.Recordings", [], {}, {
    RECORDING_CHANGE_OK: 0,
    RECORDING_CHANGE_NOK: 1,
    RECORDING_CLASH: 2,
    MAX_NUM_RECORDINGS: 15,
    _REINDEX_RECORDINGS_AFTER_MS: 10000, // 10s

    _scheduledRecordingsByFsidAndStartTime: {},
    _lastScheduledRecordingsIndex: 0,

    hasPVR: function() {
        // don't enable recordings if undefined or false
        return freesat.oipfElement.config.localSystem.pvrEnabled === true;
    },

    hasRecording: function (programme) {
        var result = false;
        // only check for recordings if PVR enabled
        if (this.hasPVR()) {
            result = programme.hasRecording;
        }
        return result;
    },

    isRecordingScheduled: function (fsatSvcIdOrCrid, startTime, ignorePvrEnabled) {
        if (!ignorePvrEnabled && !this.hasPVR()) {
            return false;
        }

        var freesatServiceID = null;
        if (fsatSvcIdOrCrid.indexOf('ccid:') === 0) {
            var channelForProgramme = freesat.channelData.getChannelObjectByCCID(fsatSvcIdOrCrid);
            
            if (!channelForProgramme) {
                freesat.console.log('WARNING: couldn\'t find channel for ccid ' + fsatSvcIdOrCrid);
                return false;
            }

            freesatServiceID = channelForProgramme.freesatServiceID;
        } else {
            freesatServiceID = fsatSvcIdOrCrid;
        }

        this.indexScheduledRecordings();
        var index = freesatServiceID + ';' + startTime;

        return (typeof this._scheduledRecordingsByFsidAndStartTime[index] !== 'undefined');
    },

    isRecordingClash: function (startTime, duration) {
        var startTime1 = startTime;
        var endTime1 = startTime1 + duration;

        this.indexScheduledRecordings();
        for (var key in this._scheduledRecordingsByFsidAndStartTime) {
            var recording = this._scheduledRecordingsByFsidAndStartTime[key];
            var startTime2 = recording.startTime;
            var endTime2 = startTime2 + recording.duration;

            if ((endTime1 <= startTime2) || (endTime2 <= startTime1)) {
                continue; // 1 finishes before 2, or 2 finishes before 1
            }
            return true;
        }
        return false;
    },

    checkRecordingReminderLimitReached: function() {
        var numRecordingsAndReminders = freesat.oipfElement.recordingsManager.getScheduledRecordings().length;
        if (numRecordingsAndReminders >= this.MAX_NUM_RECORDINGS) {
            var iconType = this.hasPVR() ? 'recording' : 'reminder';
            var title = 'set ' + (this.hasPVR() ? 'recording' : 'reminder');
            var cta = 'A maximum of ' + this.MAX_NUM_RECORDINGS + ' reminders and recordings may be scheduled.<br/>Press the \'Menu\' button then choose Timer/Timer Programming to manage your reminders and recordings.';
            var description = 'It has not been possible to set this '  + (this.hasPVR() ? 'recording' : 'reminder');
            setTimeout(function() {
                freesat.masterController.instanciateMessagePopup(
                    title,
                    description,
                    cta,
                    iconType,
                    null,
                    null,
                    null,
                    true);
            }, 1000);

            return true;
        }

        return false;
    },

    setRecording: function (programme) {
        freesat.console.log('got setRecording request for programme ' + programme.name);
        if (!programme || !programme.programmeID) {
            freesat.console.log('    invalid request')
            return this.RECORDING_CHANGE_NOK;
        }

        if (this.isRecordingScheduled(programme.channelID, programme.startTime)) {
            console.log('    recording already scheduled - ignoring');
            return this.RECORDING_CHANGE_NOK;
        }

        if (this.checkRecordingReminderLimitReached()) {
            console.log('    recording/reminder limit reached - ignoring');
            return this.RECORDING_CHANGE_NOK;
        }

        var isClash = this.isRecordingClash(programme.startTime, programme.duration);

        freesat.console.log('scheduling recording using record API...');

        var scheduledRecording = freesat.oipfElement.recordingsManager.record(programme);

        freesat.console.log('returned ScheduledRecording object:');
        freesat.console.log('    state=' + scheduledRecording.state);

        this._lastScheduledRecordingsIndex = 0; // expire cached recordings

        if (isClash) {
            setTimeout(function() {
                freesat.masterController.instanciateMessagePopup(
                    'recording clash',
                    'This programme overlaps with another recording',
                    'Press \'Menu\' button then choose Timer/Timer Programming for more information.',
                    'recording',
                    null,
                    null,
                    null,
                    true);
            }, 1000);

            return this.RECORDING_CLASH;
        } else {
            return this.RECORDING_CHANGE_OK;
        }
    },


    // Cancels the recording of a given item
    removeRecording: function(programme) {
        freesat.console.log('got removeRecording request for programme ' + programme.name);

        var recordingToRemove = this._findRecordingForProgramme(programme);
        if (!recordingToRemove) {
            freesat.console.log('Failed to find recording');
            return this.RECORDING_CHANGE_NOK;
        }

        freesat.console.log('removing recording using recorder.remove API...');
        freesat.oipfElement.recordingsManager.remove(recordingToRemove);

        freesat.console.log('    ...done');

        this._lastScheduledRecordingsIndex = 0; // expire cached recordings

        return this.RECORDING_CHANGE_OK;
    },

    _findRecordingForProgramme: function(programme) {
        var recordings = freesat.oipfElement.recordingsManager.getScheduledRecordings();
        var result = null;
        for (var i = 0; i < recordings.length; i++) {
            var recording = recordings[i];
            if (!recording.channel) {
                // can happen on switch between DSAT and DTT on Pro4
                continue;
            }

            if ((recording.channel.ccid == programme.channelID) && (recording.startTime == programme.startTime)) {
                result = recording;
                break;
            }
        }
        return result;
    },

    indexScheduledRecordings: function(doFlush) {
        var millisecondsSinceLastIndex = new Date().getTime() - this._lastScheduledRecordingsIndex;
        if (!doFlush && (millisecondsSinceLastIndex < this._REINDEX_RECORDINGS_AFTER_MS)) {
            return;
        }
        freesat.console.log('reindexing scheduled recordings');

        this._scheduledRecordingsByFsidAndStartTime = {};

        var recordings = freesat.oipfElement.recordingsManager.getScheduledRecordings();
        for (var i = 0; i < recordings.length; i++) {
            // we may not know the channel on switch between DSAT and DTT, but keep track for clash management
            var freesatServiceId = 'UNKNOWN';
            if (recordings[i].channel) {
                freesatServiceId = recordings[i].channel.freesatServiceID;
            }

            var index = freesatServiceId + ';' + recordings[i].startTime;
            this._scheduledRecordingsByFsidAndStartTime[index] = {
                freesatServiceId: freesatServiceId,
                startTime: recordings[i].startTime,
                duration: recordings[i].duration
            };
            console.log('    ' + index);
        }

        this._lastScheduledRecordingsIndex = new Date().getTime();
    },

    dumpScheduledRecordings: function() {
        console.log('SCHEDULED RECORDINGS');

        var recordings = freesat.oipfElement.recordingsManager.getScheduledRecordings();
        for (var i = 0; i < recordings.length; i++) {
            var index = recordings[i].channel.ccid + ' @ ' + new Date(recordings[i].startTime * 1000);
            console.log('    ' + index);
        }
    },

    showNoHddWarning: function() {
        freesat.masterController.instanciateMessagePopup(
            'set recording',
            'Connect a USB disk drive to start recording your fave shows.',
            'The first time, press \'Menu\' button then choose \'Setup/USB Device Setup\'.',
            'recording');
    }
});

accedo.Class.create("freesat.models.Reminders", [], {}, {
    REMINDER_CHANGE_OK: 0,
    REMINDER_CHANGE_NOK: 1,
    REMINDER_CLASH: 2,

    isEnabledForReminders: function() {
        // reminders are not supported if pvrEnabled is undefined or true, only when it is false.
        var pvrEnabled = freesat.oipfElement.config.localSystem.pvrEnabled;
        return (typeof pvrEnabled !== 'undefined') && !pvrEnabled;
    },

    hasReminder: function (programme) {
        if (!this.isEnabledForReminders()) {
            return false;
        }

        console.log('    hasReminder:  ' + programme.hasReminder);

        return programme.hasReminder;
    },

    isReminderScheduled: function (fsatSvcIdOrCrid, startTime) {
        if (!this.isEnabledForReminders()) {
            return false;
        }

        return freesat.masterController.recordingsHandler.isRecordingScheduled(fsatSvcIdOrCrid, startTime, true);
    },

    setReminder: function (programme) {
        console.log('got setReminder request for programme ' + programme.name);
        if (!programme || !programme.programmeID || !this.isEnabledForReminders()) {
            console.log('    invalid request')
            return this.REMINDER_CHANGE_NOK;
        }

        if (freesat.masterController.recordingsHandler.checkRecordingReminderLimitReached()) {
            console.log('    recording/reminder limit reached - ignoring');
            return this.RECORDING_CHANGE_NOK;
        }

        var isClash = freesat.masterController.recordingsHandler.isRecordingClash(programme.startTime, programme.duration);

        // note that Panasonic use regular recorder API for scheduling reminders
        var scheduledReminder = freesat.oipfElement.recordingsManager.record(programme);

        if (isClash) {
            setTimeout(function() {
                freesat.masterController.instanciateMessagePopup(
                    'reminder clash',
                    'This programme overlaps with another reminder',
                    'Press \'Menu\' button then choose Timer/Timer Programming for more information.',
                    'reminder',
                    null,
                    null,
                    null,
                    true);
            }, 1000);
            return this.REMINDER_CLASH;
        } else {
            return this.REMINDER_CHANGE_OK;
        }
    },


    // Cancels the reminder of a given item
    removeReminder: function (programme) {
        console.log('got removeReminder request for programme ' + programme.name);

        if (!programme || !programme.programmeID || !this.isEnabledForReminders()) {
            console.log('    invalid request')
            return this.REMINDER_CHANGE_NOK;
        }

        console.log('removing reminder using recorder.remove API...');

        // note that Panasonic use regular recorder API for scheduling reminders
        freesat.masterController.recordingsHandler.removeRecording(programme);
        console.log('    ...done');

        return this.REMINDER_CHANGE_OK;
    }
});

accedo.Class.create("freesat.widget.AudioOptionsItem", "accedo.ui.Container", ["accedo.ui.widget.Label"], function () {
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.ButtonWith3FieldsAndImage
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _labelLeft
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _labelLeft: null,

        /**
         * @private
         * @name _labelRight
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _labelRight: null,

        _lefttext : null,
        _righttext : null,
        /**
         * @constructor
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @param {Object} opts The options object
         * @param {String} opts.text The text label to use for the button
         * @private
         */
        initialize: function ($super, opts) {
            freesat.console.log("initing compenntn");
            this._lefttext = opts.lefttext || "";
            this._righttext = opts.righttext || "";
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
//            opts.enterAsClick = true; //Treats enter key as click
            $super(opts);

            this.root.addClass("audioOptionsItem");
            var css = "audioOptionsItem-label";
            this._labelLeft = new accedo.ui.widget.Label({
                parent: this,
                css: css
            });
            if (this._lefttext.length > 24) {
                this._lefttext = this._lefttext.substring(0, 24) + "...";
            }
            this.setLeftText(this._lefttext);
            this._labelRight = new accedo.ui.widget.Label({
                parent: this,
                css: css
            });
            if (this._righttext.length > 24) {
                this._righttext = this._righttext.substring(0, 24) + "...";
            }
            this.setRightText(this._righttext);
        },

        /**
         *
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @name onRequestFocus
         * @function
         * @desc sets the image to the focused src when item is focused
         */
        onRequestFocus: function(){
            this._labelLeft.addClass("focused");
            this._labelRight.addClass("focused");
        },

        /**
         *
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @name blur
         * @function
         * @override
         * @desc sets the image to the src and remove focused from class when item loses focus
         * @returns {boolean}
         */
        blur: function(){
            this._labelLeft.removeClass("focused");
            this._labelRight.removeClass("focused");
            if (this._focused) {
                this.root.removeClass("focused");
                this._focused = false;
                return true;
            }
            return false;
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        /**
         * This function sets the text left label of the button.
         * @name setLeftText
         * @function
         * @param {String} text The text label
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        setLeftText: function(text) {
            this._labelLeft.setText(text || "");
        },


        /**
         * This function sets the text right label of the button.
         * @name setRightText
         * @function
         * @param {String} text The text label
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        setRightText: function(text) {
            this._labelRight.setText(text || "");
        },

        /**
         * Return the inner right label (i.e. so as to access its DOM element)
         * @name getRightLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        getRightLabel: function() {
            return this._labelRight;
        },

        /**
         * Return the inner left label (i.e. so as to access its DOM element)
         * @name getLeftLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        getLeftLabel: function() {
            return this._labelLeft;
        },

        /**
         * Return the inner left label (i.e. so as to access its DOM element)
         * DO NOT USE IT ANY MORE! Use "getLabel". THis one is just here for compatibility, should be removed later.
         * @returns {accedo.ui.widget.Label}
         * @deprecated
         * @public
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @name getText
         */
        getText: function() {
            return this._labelLeft.getText();
        }
    });
/**
 * freesat keyboard
 * @name Keyboard
 * @namespace
 * @memberOf freesat.widget
 * @class Standard keyboard for Reference App.
 * @augments accedo.ui.widget.Keyboard
 * @author <a href="mailto:rhys.jeffreys@accedo.tv">Rhys Jeffreys</a>
 */
accedo.Class.create("freesat.widget.Keyboard", "accedo.ui.widget.Keyboard",{},
{
    /**
     * @constructor
     * @memberOf freesat.widget.Keyboard#
     * @function
     * @param {Object} opts The options object
     * @private
     */
    initialize: function($super,opts) {
        opts.enableReturn = opts.enableReturn || true;
        opts.enableEnter = opts.enableEnter || true;
        opts.spaceKey = opts.spaceKey || 'Space';
        opts.submitKey = opts.submitKey || 'Enter';
        opts.buttonCss = opts.buttonCss || "app-widget-keyboard-button";
        opts.enableCamelCase = opts.enableCamelCase || false;
        opts.pad = 'letters';

        opts.letter = opts.letter || [
            ["A","B","C","D","E"],
            ["F","G","H","I","J"],
            ["K","L","M","N","O"],
            ["P","Q","R","S","T"],
            ["U","V","W","X","Y"],
            ["Z","1","2","3","4"],
            ["5","6","7","8","9"],
            ["0",
                {
                    key: "SPACE",
                    colSpan: 2,
                    rowSpan: 1,
                    buttonCss: "keyboard-2colbutton"
                },
                {
                    key: "DEL",
                    colSpan: 2,
                    rowSpan: 1,
                    buttonCss: "keyboard-2colbutton"
                }
            ]
        ];

        $super(opts);

        this.setButtonOnClickDel(7,2);
        this.setButtonOnClickSpace(7,1);

    },

    getTextField: function(){
        return this._textField;
    },

    getTextFieldText:function(){
        return this._textField.getText();
    },

    reset: function(){
        this.active = false;
    },

    onChildFocus: function(child) {
        this.dispatchEvent(accedo.ui.Evt.CHILD_FOCUS, child);
    },

    onChildBlur: function(child) {
        this.dispatchEvent(accedo.ui.Evt.CHILD_BLUR, child);
    }
});
/**
 * @name ProgrammeGridItemWithIcon
 * @namespace
 * @memberOf freesat.widget
 * @class
 */
accedo.Class.create('freesat.widget.ProgrammeGridItemWithIcon','accedo.ui.widget.Button',[],{},
    {
        _data:null,
        _container:null,
        _background:null,
        _timeLabel:null,
        _programmeNameLabel: null,
        _icon:null,
        _id: null,

        iconType:{
            'reminder':'images/1280x720/icon/icon_reminder.png',
            'recording':'images/1280x720/icon/icon_rec.png',
            'seriesrecording':'images/1280x720/icon/icon_rec_series.png',
            'ondemand': 'images/1280x720/icon/icon_on_demand.png',
            'film': 'images/1280x720/icon/icon_film.png',
            'paid': 'images/1280x720/icon/icon_paid_content.png'
        },

        initialize: function($super,opts){
            $super(opts);

            this._container = new accedo.ui.Container({
                id:'#programmeGridItemContainer_' + this.getId(),
                css:'programmeGridItemContainer'
            });

            this._id = opts.id;
            this._timeLabel = this._createTimeLabel(opts.time || '');
            this._programmeNameLabel = this._createProgrammeLabel(opts.programmeName || '');
            this.attach(this._container);
        },

        onRequestFocus: function(){
            this._programmeNameLabel.addClass('focused');
            this._timeLabel.addClass('focused');
            this.notifyFocused(this);
        },

        notifyFocused: function(button){},

        blur: function(){
            this._programmeNameLabel.removeClass('focused');
            this._timeLabel.removeClass('focused');
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        dispatchAttachedToDOMEvent: function($super) {
            $super();
        },

        _createTimeLabel: function(text) {
            var timeStr = text.toString().substring(0,2).concat(':').concat(text.toString().substring(2));
            if(timeStr==':'){
                timeStr='';
            }
            var label = new accedo.ui.widget.Label({
                id:'#programmeTimeHourLabel_'+this.getId(),
                css: 'programmeTimeLabel',
                parent: this,
                text: timeStr
            });
            this._container.attach(label);
            return label;
        },

        _createProgrammeLabel : function(text){
            var label = new accedo.ui.widget.Label({
                id:'#programmeNameLabel_'+this.getId(),
                css: 'programmeNameLabel',
                parent: this,
                text: text
            });
            this._container.attach(label);
            return label;
        },

        _createIcon: function(src){
            var icon = new accedo.ui.widget.Image({
                id: '#programmeNameIcon_'+this.getId(),
                src: src,
                css: 'grid-icon'
            });
            this._container.attach(icon);
            return icon;
        },

        _getTimeLabel: function()
        {
            return this._timeLabel;
        },

        _getProgammeLabel: function()
        {
            return this._programmeNameLabel;
        },

        setProgrammeName:function(name){
            this._programmeNameLabel.setText(String(name));
        },

        setProgrammeTime:function(time){
            var str = this.formatGuideTime(time);
            if(str.length > 1){
                this._timeLabel.setText(str);
            }
        },

        setIcon:function(type){
            if (!this._icon) {
                this._icon = this._createIcon(this.iconType[type]);
            } else {
                this._icon.setSrc(this.iconType[type]);
            }
        },

        setIconType: function(progObj) {
            var isRecording = freesat.masterController.recordingsHandler.hasRecording(progObj),
                hasReminder = freesat.masterController.remindersHandler.hasReminder(progObj),
                crids = progObj.CRIDS,
                seriesCRID,
                type;
            this._programmeNameLabel.removeClass('hasIcon');

            if (crids) {

                for (var i = 0 ; i < crids.length ; i++) {

                    if (crids[i].indexOf('0x32') > -1) {
                        seriesCRID = 'crid' + crids[i].split('crid')[1];
                        break;
                    }
                }
            }

            if (isRecording) {
                type = 'recording';
            } else if (hasReminder) {
                type = 'reminder';
            } else if (freesat.channelData.getHasTSTVfromProgramme(progObj)) {
                type = 'ondemand';
            } else if (progObj.genre && progObj.genre[0] === 'Movie') {
                type = 'film';
            }

            if (progObj === "") {
                progObj = {name: 'EMPTY'};
            }

            freesat.console.log('NAME: ' + progObj.name + '; TYPE: ' + type);

            if (type !== undefined) {
                this.setIcon(type);
                this._programmeNameLabel.addClass('hasIcon');
            } else {

//                var currentIcon = document.getElementById('programmeNameIcon_'+this.getId());
//
//                if (currentIcon) {
                try {
                    freesat.console.log('programmeNameIcon_' + this.getId() + ' : HAS CURRENT ICON');
                    var currentIcon = this._container.get('programmeNameIcon_'+this.getId());
                    freesat.console.log(currentIcon);
                    currentIcon.detach();
                    this._icon = null;
                } catch (e) {
                    freesat.console.log('NO ICON - ' + e);
                }

//                }
            }
        },

        setRecording: function(inRecording) {
          if (inRecording) {
              this.setIcon('recording');
              this._programmeNameLabel.addClass('hasIcon');
          }  else {
              this.setIcon('');
              this._programmeNameLabel.removeClass('hasIcon');
          }
        },

        setReminder: function(hasReminder) {
          if (hasReminder) {
              this.setIcon('reminder');
              this._programmeNameLabel.addClass('hasIcon');
          }  else {
              this.setIcon('');
              this._programmeNameLabel.removeClass('hasIcon');
          }
        },

        setProgrammeData: function(progObj) {

            if (progObj) {
                this.setProgrammeName(progObj.name);
                this.setProgrammeTime(progObj.startTime);
                this._data = progObj;
                this.setIconType(progObj);
            } else {
                this.setProgrammeName('');
                this._timeLabel.setText('');
                this._data = null;
                this.setIconType('');
            }
        },

        formatGuideTime: function (seconds){
            var d = new Date(seconds * 1000),
                local_offset = d.getTimezoneOffset(),
                _date;

            try {
                _date = new Date((seconds  - (local_offset * 60)) * 1000);
                var dateStr = _date.toISOString().substr(0, _date.toISOString().length-5);
                var tArr = dateStr.split('T');

                return tArr[1].substr(0,5);
            } catch (Error) {
                _date = new Date(seconds);

                if (_date && seconds) {
                    var retString  = '<h1>:<m1>';
                    var tArr1 = seconds.split('T')[1].split(':');

                    return retString.replace('<h1>',tArr1[0]).replace('<m1>',tArr1[1]);
                }
            }
        }
    });
/**
 * @name EarlierProgrammeGridItem
 * @namespace
 * @memberOf freesat.widget
 * @class
 */
accedo.Class.create('freesat.widget.EarlierProgrammeGridItem','accedo.ui.widget.Button',[],{},
    {
        _container:null,
        _timeHourLabel:null,
        _programmeNameLabel:null,
        _actionLabel: null,
        _data:null,
        timeHoursLabelText:'',
        programmeText:'',
        _itemindex:-1,
        _watchable:false,

        initialize: function($super,opts){
            $super(opts);

            var radix = 10;
            this._itemindex = parseInt(opts.index, radix);

            this._container = new accedo.ui.Container({
                id:'#earlierProgrammeGridItemContainer',
                css:'earlierProgrammeGridItemContainer'
            });

            this._timeHourLabel = this._createTimeLabel(this._container, this._itemindex);
            this._programmeNameLabel = this._createProgrammeLabel(this._container, this._itemindex);
            this._actionLabel = this._createActionLabel(this._container, this._itemindex);

            if(opts.data) {
                this.setProgrammeData(opts.data);
            }
            this.attach(this._container);
        },

        formatLengthTime : function (time){
            var retString  = '<h> hours <m> minutes';
            var minretVal = ((time/60)%60).toFixed();
            var hhretval = ((time/60)/60)<1?0:((time/60)/60).toFixed();
            return retString.replace('<h>',hhretval).replace('<m>',minretVal);
        },

        dispatchAttachedToDOMEvent: function($super) {
            $super();
//            this._truncateLabelText();
        },

        _truncateLabelText: function(){
            var elementName = 'programmeNameLabel_'.concat(this._itemindex);
            var element = document.getElementById(elementName);

            if(element)
            {
                element.onfocus = this.onfocus;
                var curr_width = element.clientWidth;
                if(curr_width){
                    var fontSize = String(window.getComputedStyle(element).getPropertyValue('font-size'));
                    fontSize = fontSize.toString().substring(0,2);
                    var numchars = (curr_width/parseFloat(fontSize *0.4)).toFixed();
                    if(element.textContent.length >10 && element.textContent.length > (numchars -10)){
                        element.textContent = element.textContent.substring(0,numchars-10)+'...';
                    }
                }
                element.onfocus = this.onfocus;
            }
        },

        onRequestFocus : function (){
            freesat.console.log(this.root.id);
            this._programmeNameLabel.addClass('focused');
            this._timeHourLabel.addClass('focused');
            this._actionLabel.addClass('focused');
            this.notifyFocused(this);
        },

        notifyFocused: function(button){},

        blur: function (){
            this._programmeNameLabel.removeClass('focused');
            this._timeHourLabel.removeClass('focused');
            this._actionLabel.removeClass('focused');
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        _createTimeLabel: function(container, itemindex) {
            var label = new accedo.ui.widget.Label({
                id:'#programmeTimeLabel_'+itemindex,
                css: 'programmeTimeLabel',
                parent: this
            });
            container.attach(label);
            return label;
        },

        _createTimeMinuteLable: function(container,text,itemindex){
            var label = new accedo.ui.widget.Label({
                id:'#programmeTimeMinLabel_'+itemindex,
                css: 'programmeTimeMinLabel',
                parent: this,
                text: text.toString().substring(2)
            });
            container.attach(label);
            return label;
        },

        _createProgrammeLabel : function(container,itemindex){
            var label = new accedo.ui.widget.Label({
               id:'#programmeNameLabel_'+itemindex,
                css: 'programmeNameLabel',
                parent: this
            });
            container.attach(label);
            return label;
        },

        _createActionLabel : function(container, itemindex){
            var mungo = new accedo.ui.widget.Label({
                id:'#actionLabel_'+itemindex,
                parent: this
            });
            container.attach(mungo);
            return mungo;
        },

        extractTime: function (seconds){
            try{
                var _date = new Date(seconds*1000);
                var dateStr = _date.toISOString().substr(0, _date.toISOString().length-5);
                var tArr = dateStr.split('T');
                return tArr[1].substr(0,5);
            }catch(e){
                return '--:--';
            }
        },

        setProgrammeData: function(itemData){
            this._data = itemData;
//            this._timeHourLabel.setText(this.extractTime(this._data.startTime));
            this._timeHourLabel.setText(itemData.StartText);
            this._programmeNameLabel.setText(itemData.name);
            var action_css_tag = '';
            var lang = new freesat.language.ActionMenuEn();
//            if(itemData.action){
            if(itemData.StatusText !== 'Available soon'){
//                actionLabel = lang.WATCH_NOW;
                action_css_tag = '_watch_now';
                this._watchable = true;
            }else{
//                actionLabel = lang.LATER;
                action_css_tag = '_later';
            }
            this._actionLabel.addClass('actionLabel'.concat(action_css_tag));
            this._actionLabel.setText(itemData.StatusText);
//            this._truncateLabelText();
        }
});

/**
 * @name LaterEpgLineItem
 * @namespace
 * @memberOf freesat.widget
 * @class
 */
accedo.Class.create('freesat.widget.LaterEpgLineItem','accedo.ui.widget.Button',[],{},
    {
        _NO_ICON_HORIZ_SPACE: 30,
        _ICON_HORIZ_SPACE: 60,

        _data:null,
        _container:null,
        _background:null,
        _programmeNameLabel: null,
        _icon:null,
        _cellWidth:0,

        iconType:{
            'reminder':'images/1280x720/icon/icon_reminder.png',
            'recording':'images/1280x720/icon/icon_rec.png',
            'seriesrecording':'images/1280x720/icon/icon_rec_series.png'
        },

        initialize: function($super,opts){
            var programmeLabel;

            $super(opts);

            this._data = opts.opts;
            this._cellWidth = opts.cellWidth;

            this._background = new accedo.ui.Container({
                id:'#programme_item_background',
                css:'programme_item_background hidden'
            });
            this.attach(this._background);

            this._container = new accedo.ui.Container({
                id:'#programme_item',
                css:'programme_item'
            });

            if (opts.opts.truncate) {
                programmeLabel = '..';
            } else {
                programmeLabel = opts.opts.name;
            }

            this._programmeNameLabel = this._createProgrammeLabel(programmeLabel); //if this is a native programme object it should be .name
            
            if (opts.icon) {
                this._icon = this._createIcon(this.iconType[opts.opts.icon]);
            }

            this.attach(this._container);
        },

        onRequestFocus: function(){
            this._programmeNameLabel.addClass('focused');
            this.notifyFocused(this);
        },

        notifyFocused: function(button){},

        blur: function(){
            this._programmeNameLabel.removeClass('focused');

            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        dispatchAttachedToDOMEvent: function($super) {
            $super();
//            this._truncateLabelText();
        },

        _truncateLabelText: function(){
            var elementName = 'programmeNameLabel_'.concat(this.getId());
            var element = document.getElementById(elementName);
            if(element)
            {
                var curr_width = element.clientWidth;
                if(curr_width)
                {
                    var fontSize = String(window.getComputedStyle(element).getPropertyValue('font-size'));
                    fontSize = fontSize.toString().substring(0,2);
                    var numchars = (curr_width/parseFloat(fontSize * 0.4)).toFixed();
                    if(element.textContent.length >10 && element.textContent.length > (numchars -20))
                    {
                        element.textContent = element.textContent.substring(0,numchars-20)+'...';
                    }
                }
            }
        },

        _setIconSpace: function(leaveSpaceForIcon) {
            console.log('_setIconSpace: cellwidth=' + this._cellWidth + ', leaveSpaceForIcon=' + leaveSpaceForIcon);

            var maxWidth = this._cellWidth - (leaveSpaceForIcon ? this._ICON_HORIZ_SPACE : this._NO_ICON_HORIZ_SPACE);
            if (maxWidth < 0) {
                maxWidth = 0;
            }
            this._programmeNameLabel.root._dom.style.maxWidth = maxWidth + 'px';

            this._programmeNameLabel.show();
            if (leaveSpaceForIcon) {
                if (this._cellWidth < 75) {
                    // no room for ellipsis: just show icon
                    this._programmeNameLabel.hide();
                }
            }
        },

        _createProgrammeLabel : function(text){
            var label = new accedo.ui.widget.Label({
                id:'#programmeNameLabel_'+this.getId(),
                css: 'programmeNameLabel',
                parent: this,
                text: text
            });
            this._container.attach(label);
            return label;
        },

        _createIcon: function(src){
            var ico = new accedo.ui.widget.Image({
                id: '#programmeNameIcon_' + this.getId(),
                src: src,
                css: 'grid-icon'
            });
            this._container.attach(ico);
            return ico;
        },

        _getTimeLabel: function()
        {
            return this._timeLabel;
        },

        _getProgammeLabel: function()
        {
            return this._programmeNameLabel;
        },

        getProgrammeName:function(){
            return this._programmeNameLabel.getText();
        },

        setProgrammeName:function(name){
            this._programmeNameLabel.setText(String(name));
        },

        setProgrammeTime:function(time){
            this._timeLabel.setText(time.toString().substring(0,2).concat(':').concat(time.toString().substring(2)));
        },

        setIcon: function (type) {
            if (!this._icon) {
                this._icon = this._createIcon(this.iconType[type]);
            } else {
                this._icon.setSrc(this.iconType[type]);
            }
            this._icon.addEventListener('attachToDOM');
        },

        setIconType: function (progObj) {
            var isRecording = freesat.masterController.recordingsHandler.hasRecording(progObj),
                isReminderSet = freesat.masterController.remindersHandler.hasReminder(progObj),
                crids = progObj.CRIDS,
                seriesCRID,
                type;

            if (crids) {
                for (var i = 0 ; i < crids.length ; i++) {

                    if (crids[i].indexOf('0x32') > -1) {
                        seriesCRID = 'crid' + crids[i].split('crid')[1];
                        break;
                    }
                }
            }

            if (isRecording) {
                type = 'recording';
            } else if (isReminderSet) {
                type = 'reminder';
            }
            else if (freesat.channelData.getHasTSTVfromProgramme(progObj)) {
                type = 'ondemand';
            } else if (progObj.genre && progObj.genre[0] === 'Movie') {
                type = 'film';
            }

            if (progObj === "") {
                progObj = { name: 'EMPTY' };
            }

            freesat.console.log('NAME: ' + progObj.name + '; TYPE: ' + type);

            if (type !== undefined) {
                this.setIcon(type);
                this._setIconSpace(true);
            } else {
                this.setIcon('');
                this._setIconSpace(false);
            }
        },
        setCellWidth: function(cellWidth) {
            this._cellWidth = cellWidth;
            // now adjust to leave space for icon, as necessary
            this.setIconType(this.getProgrammeData());
        },
        setAsNowItem: function(){
            this._background.removeClass('hidden');
        },
        getProgrammeData: function (){
            return this._data;
        },
        setProgrammeData: function (progObj){
            this._data = progObj;
        },
        setRecording: function(inRecording) {
          if (inRecording) {
              this.setIcon('recording');
              this._setIconSpace(true);
          }  else {
              this.setIcon('');
              this._setIconSpace(false);
          }
        },
        setReminder: function (reminderSet) {
            if (reminderSet) {
                this.setIcon('reminder');
                this._setIconSpace(true);
            } else {
                this.setIcon('');
                this._setIconSpace(false);
            }
        },
    });
/**
 * @name ProgramObject
 * @namespace freesat.models.vo.ProgramObject
 * @memberOf freesat.models.vo
 * @class object to hold single epg listing
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create('freesat.models.vo.ProgramObject', [], {}, {
    //directly from data
    getName: function(){
        return this.programmeObject.name;},
    getShortSynopsis: function(){
        return this.programmeObject.description;},
    getLongSynopsis: function(){
        return this.programmeObject.longDescription;},
    getAvailabilityStart: function(){
        return this.programmeObject.startTime;},
    getDurationInSeconds: function(){
        return this.programmeObject.duration;},
    getchannelId: function(){
        return this.programmeObject.channelID;},// From oipf programme object (ccid)
    getChannelId: function(){
        return parseInt(this.programmeObject.channelID.split(':')[1], null);}, //LCN
    getlcn: function(){
        return parseInt(this.programmeObject.channelID.split(':')[1], null);},
    getepisode: function(){
        return this.programmeObject.episode;},//Integer
    gettotalEpisodes: function(){
        return this.programmeObject.totalEpisodes;},//Integer
    getprogrammeID: function(){
        return this.programmeObject.programmeID;},//readonly
    getprogrammeIDType: function(){
        return this.programmeObject.programmeIDType;},//Integer
    getparentalRatings: function(){
        return this.programmeObject.parentalRatings;},
    getchannel: function(){
        return this.programmeObject.channel;},
    getblocked: function(){
        return this.programmeObject.blocked;},//Boolean
    getshowType: function(){
        return this.programmeObject.showType;},//Integer

    getsubtitles: function(){
        return this.programmeObject.subtitles; // Boolean
    },

    getisHD: function(){
        return this.programmeObject.isHD; // Boolean
    },
    getaudioType: function(){
        return this.programmeObject.audioType; //Integer
    },

    gethasDolby: function(){
        return (this.programmeObject.audioType === 4); // Compare audioType is 4 and return boolean
    },
    getisMultilingual: function(){
        return this.programmeObject.isMultilingual;},//Boolean
    getgenre: function(){
        return this.programmeObject.genre;},
    gethasRecording: function(){
        return freesat.masterController.recordingsHandler.hasRecording(this.programmeObject)},//Boolean
    getaudioLanguages: function(){
        return this.programmeObject.audioLanguages;},
    getsubtitleLanguages: function(){
        return this.programmeObject.subtitleLanguages;},
    getlocked: function(){
        return this.programmeObject.locked;},//Boolean
    getrecording: function(){
        return false;},

    gethasSignLanguage: function() {
        return this.programmeObject.hasSignLanguage; // Boolean
//            return false;
    },
    gethasAudioDescription: function() {
        return this.programmeObject.audioDescription; // Boolean
    }, //false;}, //TODO fix

    getis3D: function(){
        return this.programmeObject.is3D; //Boolean
    },
    getonlineAvailability: function(){
        return this.programmeObject.onlineAvailability;},//Boolean

    gethasGuidance: function(){
        return this.programmeObject.hasGuidance; //Boolean
    },
    getHDSimulcast: function(){
        return this.programmeObject.HDSimulcast; //readonly
    },
    getonlineMediaLocation: function(){
        return this.programmeObject.onlineMediaLocation;},//readonly
    getimageMediaLocation: function(){
        return this.programmeObject.imageMediaLocation;},//readonly
    getpromotionalContentMediaLocation: function(){
        return this.programmeObject.promotionalContentMediaLocation;},//readonly
    getprogrammeCRID: function(){
        return this.programmeObject.programmeCRID;},//readonly
    getseriesCRID: function(){
        return this.programmeObject.seriesCRID;},//readonly
    getrecommendationCRID: function(){
        return this.programmeObject.recommendationCRID;}, //readonly

    getIsSeries: function() {
        return Boolean(this.seriesCRID);
    },

    getPlayFromStart: function() { // Available on IP and DSAT
        return false;
    },





    getTstv: function() {
        return this.programmeObject.HasTstv;
    },

    getShowcase: function() {
        return false; // Available from IP or DSAT
    },


    id: '',
    AvailabilityEnd: '',
    Genres: '',
    HasTstv: false,
    Tstv: null,
    top: 0,
    left: 0,
    width: 0,
    height: 0,
    image: '',
    //extrapoluted  //TODO: Populate these properties
//    isFilm: false,
    get isFilm(){

    },

    isPay: false,
    isShowcase: false,
    isRecording: false,
    hasDolby: false,
    hasPlayFromStart: false,
    //From oipf programme


    programmeObject: null,

    ChannelName: null,
    ChannelLogo: null
});

/**
 * @name MenuObject
 * @namespace freesat.models.vo.MenuObject
 * @memberOf freesat.models.vo
 * @class object to hold single epg listing plus additional data needed for selection menu
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create('freesat.models.vo.MenuObject', [], {}, {
    backFocus: null,
    programObject: null,
    context: '', // sets button to be focused eg. 'Watch now' / 'Record'
    location: '', // set in fullscreen Controller
    timeZone: '', // needs to be :- now / next / earlier / later / showcaseOD / showcase / search (showcase/next/later are all future broadcast)
    radio: false,
    onDemand: false,
    later: false,
    hasMask: true, // in behind should be true and front should be false //TODO: Set this from the fullscreen Controller
    top: 0,
    left: 0,
    width: 0,
    height: 0
});

/**
 * @name TsTvObject
 * @namespace freesat.models.vo.TsTvObject
 * @memberOf freesat.models.vo
 * @class object to hold single catchup details
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create('freesat.models.vo.TsTvObject', [], {}, {
    model: '',
    availabilityEnd: '',
    mediaAvailability: '',
    availabilityStart: '',
    url: ''
    //------------------------ data example
//<Tstv>
//<Option
// model="Catchup"
// availabilityEnd="2013-07-18T16:14:00Z" mediaAvailability="1"
// availabilityStart="2013-07-11T16:15:00Z">
// http://services.platform.freesat.tv/g2/ipchannels/ipchanman.php?channel=901&amp;programmeid=b01pfnd9&amp;ui_id=4.0.0
// </Option>
//</Tstv>
});
/**
 * @name ProgramItemWithImage
 * @namespace
 * @memberOf freesat.widget
 * @class
 */
accedo.Class.create('freesat.widget.ProgramItemWithImage', 'accedo.ui.widget.Button', [], {},{
    _container: null,
    _id: '',
    _image: null,
    _label: null,

    initialize: function($super, opts){
        $super(opts);

        this._id = 'programItemWithImage_'.concat(opts.id);

        //create container
        this._container = new accedo.ui.Container({
            id: '#programItemWithImage_'.concat(opts.id),
            css: 'programItemWithImaged'
        });

        //create image
        this._image = this._createImageTag();
        this._container.attach(this._image);

        //create label
        this._label = this._createNameTag();
        this._container.attach(this._label);
        this.attach(this._container);

        this.focusable = true;
        if(opts.data){
            this.setProgrammeData(opts.data);
        }
        if(opts.url){
            this._image.setSrc(opts.url);
        }
    },

    onRequestFocus:function (){
        this.notifyFocused(this);
    },

    notifyFocused: function(button){},

    _createImageTag: function (){
        return new accedo.ui.widget.Image({
            id: '#PGI_Image',
            css: 'PGI_Image'
        });
    },

    _createNameTag: function (){
        return new accedo.ui.widget.Label({
            id: '#PGI_Name',
            css: 'PGI_Name',
            parent: this
        });
    },

    setProgrammeData: function(itemData){
        this._data = itemData;
        if(itemData){
            this._image.setSrc(itemData.image);
            this._label.setText(itemData.name);
        }
    }
});
/**
 * @name Icon
 * @namespace
 * @memberOf freesat.widget
 * @class
 */
accedo.Class.create("freesat.widget.Icon", "accedo.ui.widget.Image", [], {},
    {
        _attachedCallback: null,

        /**
         * @constructor
         * @param  {object} opts Options
         * @memberOf freesat.widget.Icon
         */
        initialize: function($super, opts) {
            $super(opts);
            this._attachedCallback = opts.callback;
        },

        /**
         * To dispatch the event when it is detached to the dom
         * @memberOf freesat.widget.Icon
         */
        dispatchAttachedToDOMEvent: function($super) {
            $super();
            if (this._attachedCallback) {
                this._attachedCallback();
            }
        },

        /**
         * Set source
         * @param {string} src Image source URL
         * @memberOf freesat.widget.Icon
         */
        setSrc: function($super, src) {
            $super(src);
            if (this._attachedCallback) {
                this._attachedCallback();
            }
        },

        /**
         * Set callback function
         * @param {Function} callback
         * @memberOf freesat.widget.Icon
         */
        setCallback: function(callback) {
            if (this._attachedCallback) {
                this._attachedCallback = callback;
            }
        }
    }
);
/**
 * Creates a new imageButton instance.
 * Can dispatch 'click' event
 * @name ImageButton
 * @namespace
 * @memberof freesat.widget
 * @class A imageButton class, providing basic functionality.
 * @author <a href='mailto:dean.clancy@accedo.tv'>Dean Clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create('freesat.widget.ImageButton', 'accedo.ui.Container', ['accedo.ui.widget.Image'], function(){
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.ImageButton
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _Image
         * @memberof freesat.widget.ImageButton#
         */
        _image: null,

        /**
         * @private
         * @name _LabelImage
         * @memberof freesat.widget.ImageButton#
         */
        _labelImage: null,

        /**
         *
         * @private
         * @name _imageSrc
         * @memberof freesat.widget.ImageButton#
         */
        _imageSrc: null,

        /**
         *
         * @private
         * @name _focusedSrc
         * @memberof freesat.widget.ImageButton#
         */
        _focusedSrc: null,

        /**
         *
         * @private
         * @name _labelSrc
         * @memberof freesat.widget.ImageButton#
         */
        _labelSrc: null,

        /**
         *
         * @private
         * @name _labelFocusedSrc
         * @memberof freesat.widget.ImageButton#
         */
        _labelFocusedSrc: null,

        /**
         * @constructor
         * @memberof freesat.widget.ImageButton#
         * @param {Object} opts The options object
         * @param {String} opts.src The source of the image to use for the button
         * @param {String} opts.focusedsrc The source of the foocused image to use for the button
         * @param {String} opts.labelsrc The source of the image label to use for the button
         * @private
         */
        initialize: function($super, opts) {
            this._imageSrc = opts.src || '';
            this._focusedSrc = opts.focusedsrc || '';
            this._labelSrc = opts.labelsrc || '';
            this._labelFocusedSrc = opts.labelfocusedsrc || '';
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
//            opts.enterAsClick = true; //Treats enter key as click

            $super(opts);

            this.root.addClass('accedo-ui-image-button');

            this._image = new accedo.ui.widget.Image({
                parent: this,
                src: this._imageSrc,
                css: 'zeroPosition'
            });
            if(this._labelSrc.length > 0){
                this._labelImage = new accedo.ui.widget.Image({
                    parent: this,
                    id: '#' + this.getId() + '-label',
                    src: this._labelSrc,
                    css: 'zeroPosition'
                });
            }
        },

        onRequestFocus: function(){
            if(this._focusedSrc.length>0){
                this.setImage(this._focusedSrc);
            }
            if(this._labelImage){
                if(this._labelFocusedSrc.length>0){
                    this.setImageLabel(this._labelFocusedSrc);
                }
            }
        },

        activateFocus: function(){
            this.onRequestFocus();
        },

        deactivateFocus: function(){
            this.setImage(this._imageSrc);
            if(this._labelImage){
                if(this._labelSrc.length>0){
                    this.setImageLabel(this._labelSrc);
                }
            }
        },

        blur: function(){
            this.deactivateFocus();
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.ImageButton#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        /**
         * This function sets the image of the button.
         * @name setImage
         * @function
         * @param {String} src The Source label
         * @memberof freesat.widget.ImageButton#
         * @public
         */
        setImage: function(src) {
            this._image.setSrc(src || '');
        },

        /**
         * This function sets the label image of the button.
         * @name setImageLabel
         * @function
         * @param {String} src The source label
         * @memberof freesat.widget.ImageButton#
         * @public
         */
        setImageLabel: function(src) {
            var curImg = (this._labelImage.getSrc()==this._labelSrc);
            this._labelSrc = src || '';
            if (curImg) this._labelImage.setSrc(src || '');
        },

        /**
         * This function sets the label image of the button.
         * @name setFocusImageLabel
         * @function
         * @param {String} src The focused source label
         * @memberof freesat.widget.ImageButton#
         * @public
         */
        setFocusImageLabel: function(src) {
            var curImg = (this._labelImage.getSrc()==this._labelFocusedSrc);
            this._labelFocusedSrc = src || '';
            if (curImg) this._labelFocusedSrc.setSrc(src || '');
        },

        /**
         * Return the label image(i.e. so as to access its DOM element)
         * @name getImageLabel
         * @returns {accedo.ui.widget.Image}
         * @memberof freesat.widget.ImageButton#
         * @public
         */
        getImageLabel: function() {
            return this._labelImage;
        },

        /**
         * Return the image(i.e. so as to access its DOM element)
         * @name getImage
         * @returns {accedo.ui.widget.Image}
         * @memberof freesat.widget.ImageButton#
         * @public
         */
        getImage: function() {
            return this._image;
        }
    });
/**
 * Creates a new imageButton instance.
 * Can dispatch 'click' event
 * @name ButtonWithIcon
 * @namespace
 * @memberof freesat.widget
 * @class A buttonWithIcon class, providing basic functionality.
 * @author <a href='mailto:dean.clancy@accedo.tv'>Dean Clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create('freesat.widget.ButtonWithIcon', 'accedo.ui.Container', ['accedo.ui.widget.Image','accedo.ui.widget.Label'], function(){
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.ButtonWithIcon
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _image
         * @memberof freesat.widget.ButtonWithIcon#
         */
        _image: null,

        /**
         *
         * @private
         * @name _imageSrc
         * @memberof freesat.widget.ButtonWithIcon#
         */
        _imageSrc: null,

        /**
         *
         * @private
         * @name _focusedSrc
         * @memberof freesat.widget.ButtonWithIcon#
         */
        _focusedSrc: null,

        /**
         *
         * @private
         * @name _label
         * @memberof freesat.widget.ButtonWithIcon#
         */
        _label: null,

        /**
         *
         * @private
         * @name _labelText
         * @memberof freesat.widget.ButtonWithIcon#
         */
        _labelText: null,

        /**
         * @constructor
         * @memberof freesat.widget.ButtonWithIcon#
         * @param {Object} opts The options object
         * @param {String} opts.src The source of the image to use for the button
         * @param {String} opts.focusedsrc The source of the foocused image to use for the button
         * @param {String} opts.text The source of the text for the label to use for the button
         * @private
         */
        initialize: function($super,opts) {
            this._imageSrc = opts.src || '';
            this._focusedSrc = opts.focusedsrc || '';
            this._labelText = opts.text || '';
            this._align = opts.align || '';
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
//            if (typeof opts.enterAsClick === 'undefined') {
//                opts.enterAsClick = true; //Treats enter key as click
//            }

            $super(opts);

            this.root.addClass('accedo-ui-button-with-icon');

            var scss = 'bwiLabelPosition'+this._align;
            if(this._labelText.length > 0){
                this._label = new accedo.ui.widget.Label({
                    parent: this,
                    id: '#' + this.getId() + '-label',
                    text: this._labelText,
                    css: scss
                });
            }
            this.setImage(this._imageSrc);
        },

        onRequestFocus: function(){
            this.setImage(this._focusedSrc);
            this.notifyFocused(this);
        },

        blur: function(){
            this.setImage(this._imageSrc);
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.ButtonWithIcon#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        /**
         * This function sets the image of the button.
         * @name setImage
         * @function
         * @param {String} src The Source of the icon
         * @memberof freesat.widget.ButtonWithIcon#
         * @public
         */
        setImage: function(src) {
            if(this._image){
                if(src && src.length > 0){
                    this._image.setSrc(src);
                }else{
                    this._image.doDetach();
                    this._image = null;
                }
            }else{
                var scss = 'bwiIconPosition'+this._align;
                this._image = new accedo.ui.widget.Image({
                    parent: this,
                    src: src,
                    css: scss
                });
            }
        },

        /**
         * Return the image(i.e. so as to access its DOM element)
         * @name getImage
         * @returns {accedo.ui.widget.Image}
         * @memberof freesat.widget.ButtonWithIcon#
         * @public
         */
        getImage: function() {
            return this._image;
        },

        /**
         * Return the text of label
         * @name getText
         * @returns {string}
         * @memberof freesat.widget.ButtonWithIcon#
         * @public
         */
        getText: function() {
            return this._label.getText();
        },

        notifyFocused: function(button){}
    });
/**
 * Creates a new vTextImageButton instance.
 * Can dispatch 'click' event
 * @name VTextImageButton
 * @namespace
 * @memberof freesat.widget
 * @class A imageButton class, providing basic functionality.
 * @author <a href='mailto:dean.clancy@accedo.tv'>Dean Clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create('freesat.widget.VTextImageButton', 'accedo.ui.Container', ['accedo.ui.widget.Image'], function(){
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.VTextImageButton
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _Image
         * @memberof freesat.widget.VTextImageButton#
         */
        _mask: null,

        /**
         * @private
         * @name _Image
         * @memberof freesat.widget.VTextImageButton#
         */
        _image: null,

        /**
         * @private
         * @name _Label
         * @memberof freesat.widget.VTextImageButton#
         */
        _label: null,

        /**
         *
         * @private
         * @name _imageSrc
         * @memberof freesat.widget.VTextImageButton#
         */
        _imageSrc: null,

        /**
         *
         * @private
         * @name _focusedSrc
         * @memberof freesat.widget.VTextImageButton#
         */
        _focusedSrc: null,

        /**
         *
         * @private
         * @name _labelSrc
         * @memberof freesat.widget.VTextImageButton#
         */
        _labelText: null,

        /**
         * @constructor
         * @memberof freesat.widget.VTextImageButton#
         * @param {Object} opts The options object
         * @param {String} opts.src The source of the image to use for the button
         * @param {String} opts.focusedsrc The source of the foocused image to use for the button
         * @param {String} opts.labelsrc The source of the image label to use for the button
         * @private
         */
        initialize: function($super,opts) {
            this._imageSrc = opts.src || '';
            this._focusedSrc = opts.focusedsrc || '';
            this._labelText= opts.text || '';
            this._align = opts.align || 'left'; // needs to be left or right
            this._textcss = opts.textcss || ''; //
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
  //          opts.enterAsClick = true; //Treats enter key as click

            $super(opts);

            this.root.addClass('accedo-ui-image-button');

            this._mask = new accedo.ui.Container({
                parent: this,
                css: 'arrow-mask'
            });

            this._image = new accedo.ui.widget.Image({
                parent: this._mask,
                src: this._imageSrc,
                css: 'zeroPosition'
            });
            if(this._labelText.length > 0){
                this.createLabel(this._labelText);
            }
        },

        createLabel: function(text){
            this._label = new accedo.ui.widget.Label({
                parent: this,
                id: '#' + this.getId() + '-label',
                text: text,
                css: 'img-label arrow-label ' + this._textcss + ' rotate-' + this._align
            });
        },

        onRequestFocus: function(){
            if(this._focusedSrc.length>0){
                this.setImage(this._focusedSrc);
            }
            if(this._label){
                this._label.addClass('highlight');
            }
        },

        activateFocus: function(){
            this.onRequestFocus();
        },

        deactivateFocus: function(){
            this.setImage(this._imageSrc);
            if(this._label){
                if(this._label){
                    this._label.removeClass('highlight');
                }
            }
        },

        blur: function(){
            this.deactivateFocus();
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        active: function() {
            var self = this;
            this.root.addClass('active');

            setTimeout(function() {
                self.root.removeClass('active');
            }, 750);
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.VTextImageButton#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        /**
         * This function sets the image of the button.
         * @name setImage
         * @function
         * @param {String} src The Source label
         * @memberof freesat.widget.VTextImageButton#
         * @public
         */
        setImage: function(src) {
            this._image.setSrc(src || '');
        },

        /**
         * This function sets the label image of the button.
         * @name setText
         * @function
         * @param {String} src The source label
         * @memberof freesat.widget.VTextImageButton#
         * @public
         */
        setText: function(text) {
            this._labelText = text;
            if(this._label){
                this._label.setText(text);
            }else{
                this.createLabel(text);
            }
        },

        /**
         * Return the label image(i.e. so as to access its DOM element)
         * @name getLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.VTextImageButton#
         * @public
         */
        getLabel: function() {
            return this._label;
        },

        /**
         * Return the image(i.e. so as to access its DOM element)
         * @name getImage
         * @returns {accedo.ui.widget.Image}
         * @memberof freesat.widget.VTextImageButton#
         * @public
         */
        getImage: function() {
            return this._image;
        }
    });
/**
 * @name OneLineGuideButton
 * @namespace
 * @memberOf freesat.widget
 * @class
 */
accedo.Class.create(
    'freesat.widget.OneLineGuideButton',
    'accedo.ui.Container',
    ['accedo.ui.widget.Label'],
    function () {
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof accedo.ui.widget.Button
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        _timeLabel   : null,
        _programName : null,

        initialize: function ($super, opts) {
            opts || (opts = {});

            opts.focusable = true;
            opts.clickable = true;
//            opts.enterAsClick = true;

            $super(opts);

            this._timeLabel = new accedo.ui.widget.Label({
                parent : this,
                css  : 'time'
            });
            this._programName = new accedo.ui.widget.Label({
                id: 'programmeNameLabel-olg-'.concat(this.getId()),
                css    : 'program-name'
            });
            this.attach(this._programName);
        },
        dispatchAttachedToDOMEvent: function($super) {
            $super();
//            this._truncateLabelText();
        },
        _truncateLabelText: function(){
            var element = this._programName.root._dom;
            if(element){
                var curr_width = element.clientWidth;
                if(curr_width){
                    var counter = 12;
                    var fontSize = String(window.getComputedStyle(element).getPropertyValue('font-size'));
                    fontSize = fontSize.toString().substring(0,2);
                    var numchars = (curr_width/parseFloat(fontSize *0.4)).toFixed();
                    if(element.textContent.length > counter && element.textContent.length > (numchars - counter)){
                        element.textContent = element.textContent.substring(0,numchars-counter)+'...';
                    }
                }
            }
        },
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        setTime: function (text) {
            this._timeLabel.setText(this.formatGuideTime(text));
        },

        setProgram: function (text) {
            this._programName.setText(text || '');
        },

        setProgrammeData: function (data) {
            if(data)
                this._data = data;
            else
                this._data = {};

            this.setProgram(this._data.name);
            this.setTime(this._data.startTime);
//            this._truncateLabelText();
        },
        formatGuideTime: function (seconds){
            var d = new Date(seconds * 1000),
                local_offset = d.getTimezoneOffset(),
               _date;
            try {
                _date = new Date((seconds  - (local_offset * 60)) * 1000);
                var dateStr = _date.toISOString().substr(0, _date.toISOString().length-5);
                var tArr = dateStr.split('T');
                return tArr[1].substr(0,5);
            } catch (Error) {
                _date = new Date(seconds);

                if(_date && seconds) {
                    var retString  = '<h1>:<m1>';
                    var tArr1 = seconds.split('T')[1].split(':');
                    return retString.replace('<h1>',tArr1[0]).replace('<m1>',tArr1[1]);
                }
            }
        }
    }
);

/**
 * Creates a new channelLogo instance.
 * Can dispatch 'click' event
 * @name ChannelLogo
 * @namespace
 * @memberof freesat.widget
 * @class A imageButton class, providing basic functionality.
 * @author <a href='mailto:dean.clancy@accedo.tv'>Dean Clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create('freesat.widget.ChannelLogo', 'accedo.ui.Container', ['accedo.ui.widget.Image', 'accedo.ui.widget.Label'], function(){
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.ChannelLogo
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _background
         * @memberof freesat.widget.ChannelLogo#
         */
        _background: null,

        /**
         * @private
         * @name _Image
         * @memberof freesat.widget.ChannelLogo#
         */
        _image: null,

        /**
         * @private
         * @name _Label
         * @memberof freesat.widget.ChannelLogo#
         */
        _label: null,

        /**
         *
         * @private
         * @name _imageSrc
         * @memberof freesat.widget.ChannelLogo#
         */
        _imageSrc: null,

        /**
         *
         * @private
         * @name _focusedSrc
         * @memberof freesat.widget.ChannelLogo#
         */
        _focusedSrc: null,

        /**
         *
         * @private
         * @name _labelSrc
         * @memberof freesat.widget.ChannelLogo#
         */
        _labelText: null,

        /**
         * @constructor
         * @memberof freesat.widget.ChannelLogo#
         * @param {Object} opts The options object
         * @param {String} opts.src The source of the image to use for the button
         * @param {String} opts.focusedsrc The source of the foocused image to use for the button
         * @param {String} opts.labelsrc The source of the image label to use for the button
         * @private
         */
        initialize: function($super,opts) {
            this._imageSrc = opts.src || '';
            this._focusedSrc = opts.focusedsrc || '';
            this._labelText= opts.text || '';
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
//            opts.enterAsClick = true; //Treats enter key as click

            $super(opts);

            this.root.addClass('accedo-ui-channel-logo');

            this._background = new accedo.ui.Container({
                parent: this,
                css: 'channel-logo-background'
            });

            this.createLogoImage(this._imageSrc);

            if(this._labelText.length > 0){
                this.createLabel(this._labelText);
            }
        },

        createLogoImage: function(src){
            if(this._image)this._image.detach();
            this._image = new accedo.ui.widget.Image({
                parent: this,
                src: src,
                css: 'channel-logo-img hidden'
            });

            this._image.setLoadedCallback(this.onImageLoad(this));
            this._image.setErrorCallback(this.onImageFail(this));
        },

        onImageLoad: function(local){
            return function(){
                if(local._image)local._image.removeClass('hidden');

                /*
                    Bit of a hack but the onImageLoad was firing after the onImageFail callbacks for subsequent
                    launches of the OLG. Seemed to be triggered by the Base64 invisible GIF
                 */
                if(local._label && local._image._url !== 'data:image/gif;base64,R0lGODlhAQABAID/AMDAwAAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==') {
                    local._label.addClass('hidden');
                }

            };
        },

        onImageFail: function(local){
            return function(){
                freesat.console.log('onImageFail');
                if(local._image)local._image.addClass('hidden');
                if(local._label) local._label.removeClass('hidden');
            };
        },

        focused: function(){
            this._background.addClass('focused');
            if(this._label){
                if(this._label){
                    this._label.addClass('focused');
                }
            }
        },

        defocused: function(){
            this._background.removeClass('focused');
            if(this._label){
                if(this._label){
                    this._label.removeClass('focused');
                }
            }
        },

        highlight: function(){

            if (this._focusedSrc) {
                this.setSrc(this._focusedSrc);
            }

            if(this._label){
                this._label.addClass('highlight');
            }
            this._background.addClass('highlight');
        },

        dehighlight: function(){
            this.setSrc(this._imageSrc);

            if(this._label){
                this._label.removeClass('highlight');
            }
            this._background.removeClass('highlight');
        },

        createLabel: function(text){
            this._label = new accedo.ui.widget.Label({
                parent: this,
                id: '#' + this.getId() + '-label',
                text: text,
                css: 'channel-logo-label'
            });
        },

        onRequestFocus: function(){

            if(this._focusedSrc){
                this.setSrc(this._focusedSrc);
            }
            this._background.addClass('highlight');
            if(this._label){
                if(this._label){
                    this._label.addClass('highlight');
                }
            }
        },

        activateFocus: function(){
            this.onRequestFocus();
        },

        deactivateFocus: function(){
            this.setSrc(this._imageSrc);
            this._background.removeClass('highlight');
            if(this._label){
                if(this._label){
                    this._label.removeClass('highlight');
                }
            }
        },

        blur: function(){
            this.deactivateFocus();
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.ChannelLogo#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        /**
         * This function sets the image of the button.
         * @name setSrc
         * @function
         * @param {String} src The Source label
         * @memberof freesat.widget.ChannelLogo#
         * @public
         */
        setSrc: function(src) {
            freesat.console.log('logo src : ' + src);
            if(src!==undefined && src!==null){
                this.createLogoImage(src);
            }else{
                this.onImageFail(this)();
            }
            this._image.addClass('hidden');
            this.setText(this._labelText);
        },

        /**
         * This function sets the label image of the button.
         * @name setText
         * @function
         * @param {String} src The source label
         * @memberof freesat.widget.ChannelLogo#
         * @public
         */
        setText: function(text) {
            this._labelText = text;
            if(this._label){
                this._label.setText(text);
            }else{
                this.createLabel(text);
            }
        },

        /**
         * Return the label image(i.e. so as to access its DOM element)
         * @name getLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ChannelLogo#
         * @public
         */
        getLabel: function() {
            return this._label;
        },

        /**
         * Return the image(i.e. so as to access its DOM element)
         * @name getImage
         * @returns {accedo.ui.widget.Image}
         * @memberof freesat.widget.ChannelLogo#
         * @public
         */
        getImage: function() {
            return this._image;
        }
    });
    /**
 * Creates a new ShowcaseItem instance.
 * Can dispatch "click" event
 * @name ShowcaseItem
 * @namespace
 * @memberof freesat.widget
 * @class A ShowcaseItem class.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create("freesat.widget.ShowcaseItem", "accedo.ui.Container", ["accedo.ui.widget.Image"], function(){
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.ShowcaseItem
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _image
         * @memberof freesat.widget.ShowcaseItem#
         */
        _image: null,
        /**
         *
         * @private
         * @name _imageSrc
         * @memberof freesat.widget.ShowcaseItem#
         */
        _imageSrc: null,
        /**
         * @private
         * @name _logo
         * @memberof freesat.widget.ShowcaseItem#
         */
        _logo: null,
        /**
         *
         * @private
         * @name _logoSrc
         * @memberof freesat.widget.ShowcaseItem#
         */
        _logoSrc: null,
        /**
         * @private
         * @name _strap
         * @memberof freesat.widget.ShowcaseItem#
         */
        _strap: null,
        /**
         *
         * @private
         * @name _strapSrc
         * @memberof freesat.widget.ShowcaseItem#
         */
        _strapSrc: null,
        /**
         *
         * @private
         * @name _overlay
         * @memberof freesat.widget.ShowcaseItem#
         */
        _overlay: null,
        /**
         *
         * @private
         * @name _overlaySrc
         * @memberof freesat.widget.ShowcaseItem#
         */
        _overlaySrc: null,
        /**
         *
         * @private
         * @name _overlayFocusedSrc
         * @memberof freesat.widget.ShowcaseItem#
         */
        _overlayFocusedSrc: null,
        /**
         * @private
         * @name _title
         * @memberof freesat.widget.ShowcaseItem#
         */
        _title: null,
        /**
         *
         * @private
         * @name _titleSrc
         * @memberof freesat.widget.ShowcaseItem#
         */
        _titleText: null,
        /**
         * @private
         * @name _subtitle
         * @memberof freesat.widget.ShowcaseItem#
         */
        _subtitle: null,
        /**
         *
         * @private
         * @name _subtitleSrc
         * @memberof freesat.widget.ShowcaseItem#
         */
        _subtitleText: null,
        /**
         *@private
         * @name _size
         * @memberof freesat.widget.ShowcaseItem#
         */
        _size: null,
        /**
         * @private
         * @name _iconvalue
         * @memberof freesat.widget.ShowcaseItem#
         * @desc 1 = play icon : 2 = film icon : 4 = pay icon : 8 = rec icon : 16 = reminder icon ::: 0-31 gives all possible combinations
         */
        _iconvalue: null,
        _playIcon: null,
        _filmIcon: null,
        _payIcon: null,
        _recIcon: null,
        _reminderIcon: null,
        /**
         * @constructor
         * @memberof freesat.widget.ShowcaseItem#
         * @param {Object} opts The options object
         * @param {String} opts.src The source of the image to use for the button
         * @param {String} opts.focusedsrc The source of the foocused image to use for the button
         * @param {String} opts.labelsrc The source of the image label to use for the button
         * @private
         */
        initialize: function($super,opts) {
            this._overlaySrc = opts.overlaysrc || '';
            this._overlayFocusedSrc = opts.overlayfocusedsrc || '';
            this._size = opts.size || [260,165];
            this._od = (opts.od=='true') ? true : false;
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
//            opts.enterAsClick = false; //Treats enter key as click

            $super(opts);

            this.root.addClass('freesat-ui-showcase-item');

            this._image = this.createImage(this, '#sc-image', 'zero-position');
            this._overlay = this.createImage(this, '#sc-overlay', 'zero-position');
            this._overlay.setSrc(this._overlaySrc);
            this._logo = this.createImage(this, '#sc-logo', 'zero-position');
            this._strap = this.createImage(this, '#sc-strap', null, this._size[0]);
            var lWidth = this._size[0] > 260 ? this._size[0] - 30 : this._size[0] - 16;
            this._title =  this.createLabel('#sc-title', lWidth);
            var substr;
            if (this._od){
                substr = '#sc-subtitle-od';
            }else{
                substr = '#sc-subtitle';
            }
            this._subtitle = this.createLabel(substr, lWidth);
            this.createIcons();

        },
        createIcons: function(){
            var id = 'sc-icon-container',
                big = false;
            if (this._size[0]>260) {
                id = id + '-big';
                big = true;
            }
            var iconContainer = new accedo.ui.Container({
                parent: this,
                css: id
            });
            this._playIcon = this.createImage(iconContainer, '#sc-icon', 'hidden');
            this._playIcon.setSrc(big ? 'images/1280x720/icon/icon_ondemand.png' : 'images/1280x720/icon/icon_ondemand_small.png');
            this._filmIcon = this.createImage(iconContainer, '#sc-icon', 'hidden');
            this._filmIcon.setSrc(big ? 'images/1280x720/icon/icon_film.png' : 'images/1280x720/icon/icon_film_small.png');
            this._payIcon = this.createImage(iconContainer, '#sc-icon', 'hidden');
            this._payIcon.setSrc(big ? 'images/1280x720/icon/icon_paid_content.png' : 'images/1280x720/icon/icon_paid_content_small.png');
            this._recIcon = this.createImage(iconContainer, '#sc-icon', 'hidden');
            this._recIcon.setSrc(big ? 'images/1280x720/icon/icon_rec.png' : 'images/1280x720/icon/icon_rec_small.png');
            this._reminderIcon = this.createImage(iconContainer, '#sc-icon', 'hidden');
            this._reminderIcon.setSrc(big ? 'images/1280x720/icon/icon_reminder.png' : 'images/1280x720/icon/icon_reminder_small.png');
        },
        createImage: function(parent, id, css, width){
            var obj = {
                parent: parent,
                id: id
            };
            if(css) obj.css = css;
            var image = new accedo.ui.widget.Image(obj);
            if(width){
                image.root.setStyle({
                    'left':String(width-82)+'px'
                });
            }
            return image;
        } ,
        createLabel: function(id, width){
            if(this._size[0]>260) id = id + '-big';
            var label = new accedo.ui.widget.Label({
                parent: this,
                id: id
            });
            label.root.setStyle({
                'width':String(width)+'px'
            });
            return label;
        } ,
        onRequestFocus: function(){
            if(this._overlayFocusedSrc.length>0){
                this._overlay.setSrc(this._overlayFocusedSrc);
            }
            if(this._title){
                this._title.addClass('highlight');
            }
            this.notifyFocused(this);
        },
        blur: function(){
            this.deactivateFocus();
            if (this._focused) {
                this.root.removeClass("focused");
                this._focused = false;
                return true;
            }
            return false;
        },
        notifyFocused: function(button){},
        activateFocus: function(){
            this.onRequestFocus();
        },
        deactivateFocus: function(){
            if(this._overlaySrc.length>0){
                this._overlay.setSrc(this._overlaySrc);
            }
            if(this._title){
                this._title.removeClass('highlight');
            }
        },
        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.ShowcaseItem#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },
        setImageSrc: function(value){
            this._imageSrc = value;
            this._image.setSrc(value);
        },
        setLogoSrc: function(value){
            this._logoSrc = value;
            if(this._size[0]>260){
                this._logo.addClass('sc-scale-big-logo');
                this._logo.removeClass('sc-scale-logo');
            }else{
                this._logo.removeClass('sc-scale-big-logo');
                this._logo.addClass('sc-scale-logo');
            }
            this._logo.setSrc(value);
        },
        setStrapSrc: function(value){
            this._strapSrc = value;
            this._strap.setSrc(value);
        },
        setTitleText: function(value){
            this._titleText = value;
            this._title.setText(value);
        },
        setSubtitleText: function(value){
            this._subtitleText = value;
            this._subtitle.setText(value);
        },
        setIconValue: function(value){
            this._iconvalue = value;
            if(value>15){
                value = value - 16;
                this._reminderIcon.removeClass('hidden');
            } else {
                this._reminderIcon.addClass('hidden');
            }
            if(value>7){
                value = value - 8;
                this._recIcon.removeClass('hidden');
            } else {
                this._recIcon.addClass('hidden');
            }
            if(value>3){
                value = value - 4;
                this._payIcon.removeClass('hidden');
            } else {
                this._payIcon.addClass('hidden');
            }
            if(value>1){
                value = value - 2;
                this._filmIcon.removeClass('hidden');
            } else {
                this._filmIcon.addClass('hidden');
            }
            if(value>0){
                this._playIcon.removeClass('hidden');
            }else {
                this._playIcon.addClass('hidden');
            }
        },
        setProgrammeData: function(itemData){
            this._data = itemData;
            this.updateProgrammeData();
        },
        getProgrammeData: function() {
            return this._data;
        },
        updateProgrammeData: function() {
            var itemData = this._data;
            if(itemData){
                var imgFactory = new freesat.models.Imagefactory(),
                    subtitle = itemData.sc_title2,
                    logo = freesat.channelData.getChannelLogo(itemData.fsvc_id, false),
                    now, end;

                if (logo) {
                    this.setLogoSrc(logo);
                }

                if (itemData.sc_image) {
                    this.setImageSrc(imgFactory.getImageURL(itemData.sc_image, this._size[0], this._size[1], true));
                } else {
                    if(this._size[0]>260){
                        this.setImageSrc('images/placeholders/sc-not-found-big.png');
                    } else {
                        this.setImageSrc('images/placeholders/sc-not-found.png');
                    }
                }

                if (itemData.pick_flag) {
                    this.setStrapSrc(imgFactory.getImageURL(itemData.pick_flag));
                }
                this.setTitleText(itemData.sc_title);

                if (subtitle) {

                    if (!this._od) {
                        now = Math.round(new Date().getTime() / 1000);
                        end = itemData.startTime + itemData.duration;
                        subtitle = (end < now) ? 'Programme ended' : subtitle;
                    }
                    this.setSubtitleText(subtitle);
                }
                this._iconvalue = (itemData.icons.ondemand) ? 1 : 0; // 1 = play icon : 2 = film icon : 4 = pay icon ::: 0-7 gives all possible combinations
                this._iconvalue = (itemData.genre=='film') ? this._iconvalue+2 : this._iconvalue;
                this._iconvalue = (itemData.icons.pay) ? this._iconvalue+4 : this._iconvalue;

                this.setIconValue((freesat.masterController.recordingsHandler.isRecordingScheduled(itemData.fsvc_id, itemData.startTime)) ? this._iconvalue+8 : this._iconvalue);
                this.setIconValue((freesat.masterController.remindersHandler.isReminderScheduled(itemData.fsvc_id, itemData.startTime)) ? this._iconvalue+16 : this._iconvalue);
            }
        }
    }
);
/**
 * @namespace freesat.widget
 * @memberOf freesat
 */

/**
 * Creates a new ButtonWith2FieldsAndImage instance.
 * Can dispatch "click" event
 * @name ButtonWith2FieldsAndImage
 * @namespace
 * @memberof accedo.ui.widget
 * @class A button class, providing basic functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">dean clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create('freesat.widget.ButtonWith2FieldsAndImage', 'accedo.ui.Container', ['accedo.ui.widget.Label','accedo.ui.widget.Image'], function(){
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.ButtonWith2FieldsAndImage
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _labelLeft
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _labelLeft: null,

        /**
         * @private
         * @name _labelRight
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _labelRight: null,

        /**
         * @private
         * @name _image
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _image: null,

        /**
         * @private
         * @name _imageBG
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _imageBackground: null,

        /**
         * @private
         * @name _src
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _src: null,

        /**
         * @private
         * @name _focusedsrc
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _focusedsrc: null,

        /**
         * @private
         * @name _propergatedCSS
         * @desc css to be propergated to label children
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _propergatedCSS: null,

        _lefttext: null,
        _righttext: null,
        /**
         * @constructor
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @param {Object} opts The options object
         * @param {String} opts.text The text label to use for the button
         * @private
         */
        initialize: function($super,opts) {

            this._lefttext = opts.lefttext || '';
            this._righttext = opts.righttext || '';
            this._src = opts.src || null;
            this._focusedsrc = opts.focusedsrc || null;
            this._propergatedCSS = opts.pcss || null;
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
//            opts.enterAsClick = true; //Treats enter key as click

            $super(opts);

            this.root.addClass('accedo-ui-button-with-2-fields-and-image');
            if (opts.imageBackground) {
                this.setImageBackground();
            }
            this.setImage(this._src);
            var css = 'bw2fai-label-left ' + this._propergatedCSS;
            this._labelLeft = new accedo.ui.widget.Label({
                parent: this,
                css: css
            });
            this.setLeftText(this._lefttext);
            css = 'bw2fai-label-right ' + this._propergatedCSS;
            this._labelRight = new accedo.ui.widget.Label({
                parent: this,
                css: css
            });
            this.setRightText(this._righttext);
        },

        /**
         *
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @name onRequestFocus
         * @function
         * @desc sets the image to the focused src when item is focused
         */
        onRequestFocus: function(){
            if(this._focusedsrc){
                this.setImage(this._focusedsrc);
            }
            this._labelLeft.addClass('focused');
            this._labelRight.addClass('focused');
            this.notifyFocused(this);
        },

        /**
         *
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @name blur
         * @function
         * @override
         * @desc sets the image to the src and remove focused from class when item loses focus
         * @returns {boolean}
         */
        blur: function(){
            if(this._src){
                this.setImage(this._src);
            }
            this._labelLeft.removeClass('focused');
            this._labelRight.removeClass('focused');
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            return false;
        },

        /**
         * This function sets the image of the button.
         * @name setImage
         * @function
         * @param {String} src The Source of the icon
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @public
         */
        setImage: function(src) {
            if (src && src.label) {

                if (this._image) {
                    this._image.setText(src.label);
                } else {
                    this._image = new accedo.ui.widget.Label({
                        parent: (this._imageBackground) ? this._imageBackground : this,
                        text: src.label,
                        css: 'bw2fai-image-label'
                    });
                }
            } else if(this._image){
                if(src && src.length > 0){
                    this._image.setSrc(src);
                }else{
                    this._image.doDetach();
                    this._image = null;
                }
            }else{
                this._image = new accedo.ui.widget.Image({
                    parent: (this._imageBackground) ? this._imageBackground : this,
                    src: src,
                    css: 'bw2fai-image'
                });
            }
        },

        /**
         * This function sets the image of the button.
         * @name setImageBackground
         * @function Creates a container for the image
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @public
         */
        setImageBackground: function() {
            if (!this._imageBackground){
                this._imageBackground = new accedo.ui.Container({
                    parent: this,
                    css: 'bw2fai-image-background'
                });
            }
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        /**
         * This function sets the text left label of the button.
         * @name setLeftText
         * @function
         * @param {String} text The text label
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @public
         */
        setLeftText: function(text) {
            this._labelLeft.setText(text || '');
        },

        /**
         * This function sets the text right label of the button.
         * @name setRightText
         * @function
         * @param {String} text The text label
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @public
         */
        setRightText: function(text) {
            this._labelRight.setText(text || '');
        },

        /**
         * Return the inner right label (i.e. so as to access its DOM element)
         * @name getRightLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @public
         */
        getRightLabel: function() {
            return this._labelRight;
        },

        /**
         * Return the inner left label (i.e. so as to access its DOM element)
         * @name getLeftLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @public
         */
        getLeftLabel: function() {
            return this._labelLeft;
        },

        /**
         * Return the inner left label (i.e. so as to access its DOM element)
         * DO NOT USE IT ANY MORE! Use 'getLabel'. THis one is just here for compatibility, should be removed later.
         * @returns {accedo.ui.widget.Label}
         * @deprecated
         * @public
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         * @name getText
         */
        getText: function() {
            return this._labelLeft.getText();
        },

        notifyFocused: function(button){}
    });
/**
 * Creates a new ButtonWith3FieldsAndImage instance.
 * Can dispatch 'click' event
 * @name ButtonWith3FieldsAndImage
 * @namespace
 * @memberof accedo.ui.widget
 * @class A button class, providing basic functionality.
 * @author <a href='mailto:dean.clancy@accedo.tv'>dean clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create('freesat.widget.ButtonWith3FieldsAndImage', 'accedo.ui.Container', ['accedo.ui.widget.Label','accedo.ui.widget.Image'], function(){
        return {
            /**
             * Event: click
             * @constant
             * @name EVT_CLICK
             * @memberof freesat.widget.ButtonWith3FieldsAndImage
             * @deprecated  moved to accedo.ui.Evt.CLICK
             */
            EVT_CLICK: accedo.ui.Evt.CLICK
        };
    },
    {
        /**
         * @private
         * @name _labelLeft
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _labelLeft: null,

        /**
         * @private
         * @name _rightContainer
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _rightContainer: null,

        /**
         * @private
         * @name _middleContainer
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _middleContainer: null,

        /**
         * @private
         * @name _labelMiddle
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _labelMiddle: null,

        /**
         * @private
         * @name _labelRight
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _labelRight: null,

        /**
         * @private
         * @name _image
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _image: null,

        /**
         * @private
         * @name _src
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _src: null,

        /**
         * @private
         * @name _focusedsrc
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         */
        _focusedsrc: null,

        /**
         * @private
         * @name _propergatedCSS
         * @desc css to be propergated to label children
         * @memberof freesat.widget.ButtonWith2FieldsAndImage#
         */
        _propergatedCSS: null,

        _lefttext : null,
        _middletext : null,
        _righttext : null,
        /**
         * @constructor
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @param {Object} opts The options object
         * @param {String} opts.text The text label to use for the button
         * @private
         */
        initialize: function($super,opts) {
            this._lefttext = opts.lefttext || '';
            this._middletext = opts.middletext || '';
            this._middleicon = opts.middleicon || '';
            this._righttext = opts.righttext || '';
            this._src = opts.src || null;
            this._focusedsrc = opts.focusedsrc || null;
            this._propergatedCSS = opts.pcss || null;
            opts.focusable = true; //Explicit definition of object being focusable
            opts.clickable = true; //Explicit definition of object being clickable
//            opts.enterAsClick = true; //Treats enter key as click

            $super(opts);

            this.root.addClass('accedo-ui-button-with-3-fields-and-image');
            this.setImage(this._src);
            var css = 'bw2fai-label-left ' + this._propergatedCSS;
            this._labelLeft = new accedo.ui.widget.Label({
                parent: this,
                css: css
            });

            this.setLeftText(this._lefttext);
            this._rightContainer = new accedo.ui.Container(
                {
                    parent: this,
                    css: 'bw2fai-container'
                }
            );
            css = 'bw2fai-label-middle ' + this._propergatedCSS;
            this._middleContainer = new accedo.ui.Container(
                {
                    parent: this._rightContainer,
                    css: css
                }
            );
            this._labelMiddle = new accedo.ui.widget.Label({
                parent: this._middleContainer,
                css: css
            });
            this.setMiddleText(this._middletext);
            css = 'bw2fai-icon-middle';
            this._iconMiddle = new accedo.ui.widget.Image({
                parent: this._middleContainer,
                css: css
            });
            this.setMiddleIcon(this._middleicon);
            css = 'bw2fai-label-right ' + this._propergatedCSS;
            this._labelRight = new accedo.ui.widget.Label({
                parent: this._rightContainer,
                css: css
            });
            this.setRightText(this._righttext);
        },

        /**
         *
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @name onRequestFocus
         * @function
         * @desc sets the image to the focused src when item is focused
         */
        onRequestFocus: function(){

            if(this._focusedsrc){
                this.setImage(this._focusedsrc);
            }
            this._labelLeft.addClass('focused');
            this._labelMiddle.addClass('focused');
            this._labelRight.addClass('focused');
            this.notifyFocused(this);
        },

        /**
         *
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @name blur
         * @function
         * @override
         * @desc sets the image to the src and remove focused from class when item loses focus
         * @returns {boolean}
         */
        blur: function(){
            if(this._src){
                this.setImage(this._src);
            }
            this._labelLeft.removeClass('focused');
            this._labelMiddle.removeClass('focused');
            this._labelRight.removeClass('focused');
            if (this._focused) {
                this.root.removeClass('focused');
                this._focused = false;
                return true;
            }
            this.notifyBlurred(this);
            return false;
        },

        /**
         * This function sets the image of the button.
         * @name setImage
         * @function
         * @param {String} src The Source of the icon
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        setImage: function(src) {
            if (src.label) {
//                if (src.label.length > 9) {
//                    src.label = src.label.substring(0, 7) + "...";
//                }
                if (this._image) {
                    this._image.setText(src.label);
                } else {
                    this._image = new accedo.ui.widget.Label({
                        parent: this,
                        text: src.label,
                        css: 'bw2fai-image-label'
                    });
                }
            } else if (this._image){
                if(src && src.length > 0){
                    this._image.setSrc(src);
                }else{
                    this._image.doDetach();
                    this._image = null;
                }
            }else{
                this._image = new accedo.ui.widget.Image({
                    parent: this,
                    src: src,
                    css: 'bw2fai-image'
                });
            }
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        },

        /**
         * This function sets the text left label of the button.
         * @name setLeftText
         * @function
         * @param {String} text The text label
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        setLeftText: function(text) {
            this._labelLeft.setText(text || '');
        },

        /**
         * This function sets the text in the middle of the button. If non-empty
         * then this will override any middle icon.
         * @name setMiddleText
         * @function
         * @param {String} text The text label
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        setMiddleText: function(text) {
            var txt = text || '';

            this._labelMiddle.setText(txt);
            this._updateMiddleTextOrIcon();
        },

        /**
         * This function sets the icon in the middle (only visible if no middle
         * text specified).
         * @name setMiddleIcon
         * @function
         * @param {String} src The source of the image
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        setMiddleIcon: function(src) {
            this._iconMiddle.setSrc(src);
            this._updateMiddleTextOrIcon();
        },

        /**
         * This function sets the text right label of the button.
         * @name setRightText
         * @function
         * @param {String} text The text label
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        setRightText: function(text) {
            this._labelRight.setText(text || '');
        },

        /**
         * Return the inner right label (i.e. so as to access its DOM element)
         * @name getRightLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        getRightLabel: function() {
            return this._labelRight;
        },

        /**
         * Return the inner left label (i.e. so as to access its DOM element)
         * @name getMiddleLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        getMiddleLabel: function() {
            return this._labelMiddle;
        },

        /**
         * Return the inner left label (i.e. so as to access its DOM element)
         * @name getLeftLabel
         * @returns {accedo.ui.widget.Label}
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @public
         */
        getLeftLabel: function() {
            return this._labelLeft;
        },

        /**
         * Return the inner left label (i.e. so as to access its DOM element)
         * DO NOT USE IT ANY MORE! Use 'getLabel'. THis one is just here for compatibility, should be removed later.
         * @returns {accedo.ui.widget.Label}
         * @deprecated
         * @public
         * @memberof freesat.widget.ButtonWith3FieldsAndImage#
         * @name getText
         */
        getText: function() {
            return this._labelLeft.getText();
        },

        notifyFocused: function(button){},

        /**
         * Update visibility of the label and icon in the middle. If specified,
         * middle text overrides icon. Only one can be visible.
         */
        _updateMiddleTextOrIcon: function() {
            if (!this._iconMiddle || !this._labelMiddle) {
                return; // not fully constructed yet
            }
            var middleIcon = this._iconMiddle.getSrc() || '';
            var middleText = this._labelMiddle.getText() || '';
            if ((middleText === '') && (middleIcon === '')) {
                this.root.addClass('no-middle') ;
            } else if (middleText != '') {
                this.root.removeClass('no-middle') ;
                this._labelMiddle.setVisible(true);
                this._iconMiddle.setVisible(false);
            } else if (middleIcon != '') {
                this.root.removeClass('no-middle') ;
                this._labelMiddle.setVisible(false);
                this._iconMiddle.setVisible(true);
            }
        }
    });
/**
 * Creates a new imageButton instance.
 * Can dispatch 'click' event
 * @name SelectionMenuMask
 * @namespace
 * @memberof freesat.widget
 * @class A imageButton class, providing basic functionality.
 * @author <a href='mailto:dean.clancy@accedo.tv'>Dean Clancy</a>
 * @extends accedo.ui.Container
 */
accedo.Class.create('freesat.widget.SelectionMenuMask', 'accedo.ui.Container', ['accedo.ui.widget.Image'], function(){ return {};},
    {
        /**
         * @private
         * @name _imageTop
         * @memberof freesat.widget.SelectionMenuMask#
         */
        _imageTop: null,

        /**
         * @private
         * @name _imageLeft
         * @memberof freesat.widget.SelectionMenuMask#
         */
        _imageLeft: null,

        /**
         * @private
         * @name _imageBottom
         * @memberof freesat.widget.SelectionMenuMask#
         */
        _imageBottom: null,

        /**
         * @private
         * @name _imageRight
         * @memberof freesat.widget.SelectionMenuMask#
         */
        _imageRight: null,

        /**
         *
         * @private
         * @name _imageSrc
         * @memberof freesat.widget.SelectionMenuMask#
         */
        _imageSrc: null,

        /**
         * @private
         * @name _width
         * @memberof freesat.widget.SelectionMenuMask#
         * @desc the overall width
         */
        _width:null,

        /**
         * @private
         * @name _height
         * @memberof freesat.widget.SelectionMenuMask#
         * @desc the overall height
         */
        _height:null,

        /**
         * @private
         * @name _btntop
         * @memberof freesat.widget.SelectionMenuMask#
         * @desc the top position of the button that is not to be masked
         */
        _btntop: 0,

        /**
         * @private
         * @name _btnleft
         * @memberof freesat.widget.SelectionMenuMask#
         * @desc the left position of the button that is not to be masked
         */
        _btnleft:null,

        /**
         * @private
         * @name _btnwidth
         * @memberof freesat.widget.SelectionMenuMask#
         * @desc the width position of the button that is not to be masked
         */
        _btnwidth:null,

        /**
         * @private
         * @name _btnheight
         * @memberof freesat.widget.SelectionMenuMask#
         * @desc the height position of the button that is not to be masked
         */
        _btnheight:null,

        /**
         * @constructor
         * @memberof freesat.widget.SelectionMenuMask#
         * @param $super
         * @param string opts.src The source of the image to use for the button
         * @param number opts.width
         * @param number opts.height
         * @param number opts.btntop
         * @param number opts.btnleft
         * @param number opts.btnwidth
         * @param number opts.btnheight
         */
        initialize: function($super,opts) {
            this._imageSrc = opts.src || '';
            this._width = opts.width || 0;
            this._height = opts.height || 0;
            this._btntop = opts.btntop || 0;
            this._btnleft = opts.btnleft || 0;
            this._btnwidth = opts.btnwidth || this._width;
            this._btnheight = opts.btnheight || this._height;
            opts.focusable = false; //Explicit definition of object being focusable
            opts.clickable = false; //Explicit definition of object being clickable
            opts.enterAsClick = false; //Treats enter key as click

            $super(opts);

            this.root.addClass('freesat-tlg-selection-menu-mask');

            var postop, posleft;
            var zerotop = freesat.fullscreenHeight - this._height;
            this.createMaskedImageElement('topImage', 0, 0,
                this.createMaskElement('selectionTop', String(zerotop) + 'px','0px', String(this._width) + 'px', String(this._btntop-zerotop) + 'px')
            );
            postop = this._btntop + this._btnheight;
            this.createMaskedImageElement('bottomImage', String(zerotop-postop) + 'px', '0px',
                this.createMaskElement('selectionBottom', String(postop) + 'px', '0px', String(this._width) + 'px', String(freesat.fullscreenHeight - (this._btntop)) + 'px')
            );
            postop = this._btntop;
            this.createMaskedImageElement('leftImage', '-' + String(postop-zerotop) + 'px','0px',
                this.createMaskElement('selectionLeft', String(postop) + 'px', '0px', String(this._btnleft) + 'px', String(this._btnheight) + 'px')
            );
            posleft= this._btnleft + this._btnwidth;
            this.createMaskedImageElement('rightImage', '-' + String(postop-zerotop) + 'px', '-' + String(posleft) + 'px',
                this.createMaskElement('selectionRight', String(postop) + 'px', String(posleft) + 'px', String(this._width-(this._btnleft + this._btnwidth)) + 'px', String(this._btnheight) + 'px')
            );
        },

        /**
         * @desc Creates a mask container
         * @memberof freesat.widget.SelectionMenuMask#
         * @param id
         * @param top
         * @param left
         * @param width
         * @param height
         * @returns {accedo.ui.Container}
         */
        createMaskElement: function(id, top, left, width, height){
            var ele = new accedo.ui.Container({
                id:'#'+id,
                css:'selectionGradMaskContainer'
            });
            this.attach(ele);
            ele.root.setStyle({
                'position': 'absolute;',
                'top': top,
                'left': left,
                'width': width,
                'height': height,
                'overflow': 'hidden'
            });
            return ele;
        },

        /**
         * @desc Creates an image
         * @memberof freesat.widget.SelectionMenuMask#
         * @param id
         * @param top
         * @param left
         * @param parent
         * @returns {accedo.ui.widget.Image}
         */
        createMaskedImageElement: function(id, top, left, parent){
            var obj = {
                id:'#'+id,
                src: this._imageSrc
            };
            var ele = new accedo.ui.widget.Image(obj);
            parent.attach(ele);
            ele.root.setStyle({
                'top': String(top),
                'left': String(left),
                'position': 'absolute'
            });
            return ele;
        },

        /**
         * This function is called by the container function whenever
         * a child element has been appended.
         * @name doAttachElement
         * @override
         * @function
         * @memberof freesat.widget.SelectionMenuMask#
         * @param {accedo.ui.AbstractComponent} child
         * @public
         */
        doAttachElement: function(child){
            this.root.appendChild(child.getRoot());
        }
    });
/**
 * Creates a new ActionMenuEn instance.
 * @name ActionMenuEn
 * @memberof freesat.language
 * @class A ActionMenuEn class, LANGUAGE.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.language.ActionMenuEn", [], {},
    {
        LISTEN_NOW: 'Listen now',
        WATCH_NOW: 'Watch now',
        WATCH_IN_HD: 'Watch now in HD',
        LISTEN_FROM_START: 'Listen from start',
        WATCH_FROM_START: 'Watch from start',
        AUDIO_OPTIONS: 'Audio options',
        RESUME: 'Resume',
        WATCH_OD: 'Watch On Demand',
        LISTEN_OD: 'Listen On Demand',
        RECORD: 'Record',
        SERIES_RECORD: 'Series Record',
        CANCEL_RECORD: 'Cancel record',
        CANCEL_SERIES_RECORD: 'Cancel series record',
        SET_REMINDER: 'Set reminder',
        REMOVE_REMINDER: 'Remove reminder',
        KEEP: 'Keep',
        DONT_KEEP: "Don't keep",
        DELETE: 'Delete',
        ABOUT: 'About',
        PLAY_CLIP: 'Play Clip',
        EPISODES: 'Episodes',
        CLOSE_MENU: 'Close menu',
        LATER: 'Later'
    }
);
/**
 * @namespace freesat.language
 * @memberOf freesat
 */

/**
 * Creates a new AboutMenuEn instance.
 * @name AboutMenuEn
 * @memberof freesat.language
 * @class A AboutMenuEn class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.language.AboutMenuEn", [], {},
    {
        SHOWING_AGAIN: 'Showing again',
        EPISODES: 'Other Episodes',
        ABOUT: 'About',
        CAST: 'Cast &amp; Crew',
        MORE_LIKE_THIS: 'More like this',
        MORE: 'More',
        RECOMEND: 'Recommendations'
    }
);
/**
 * Creates a new CommonEn instance.
 * @name CommonEn
 * @memberof freesat.language
 * @class A CommonEn class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.language.CommonEn", [], {},
    {
        CATEGORIES: 'Categories',
        EPISODE: 'Episode',
        SERIES: 'Series',
        HR: 'hr',
        MIN: 'min',
        HRS: 'hrs',
        MINS: 'mins'
    }
);
/**
 * Created with JetBrains WebStorm.
 * User: Dean Clancy
 * Date: 07/10/13
 * Time: 07:38
 * To change this template use File | Settings | File Templates.
 */
/*global accedo: false */
/**
 * Creates a new ShowCaseEn instance.
 * @name ShowCaseEn
 * @memberof freesat.language
 * @class A ShowCaseEn class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.language.ShowCaseEn", [], {},
    {
        SHOWCASE: 'showcase',
        STRAPLINE: "bringing you the best of this week's tv",
        OD_STRAPLINE: 'the pick of the programmes to watch right now',
        SC_STRAPLINE: 'what you should watch this week',
        PLACEHOLDER_STRAPLINE: 'Top picks to watch on TV and On Demand',
        ON_DEMAND: 'on demand',
        ON_TV: 'on tv',
        MENU_CLOSE : 'Close menu',
        MENU_KEEP : 'Keep',
        MENU_WATCH : 'Watch now',
        MENU_PLAY: 'Play clip',
        MENU_RECORD: 'Record',
        MENU_CANCEL_RECORD: 'Cancel record',
        MENU_ABOUT : 'About',
        MENU_SET_REMINDER : 'Set reminder',
        MENU_CANCEL_REMINDER : 'Cancel reminder',
        ZOOM_NofNN : '{n} of {nn} picks',
        SEASON_AND_EPISODES : 'Season {s}, Episode {e}'
    }
);
/**
 * Creates a new HomeMenuEn instance.
 * @name HomeMenuEn
 * @memberof freesat.language
 * @class A HomeMenuEn class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.language.HomeMenuEn", [], {},
    {
        HOME: 'home',
        SEARCH: 'Search',
        ON_DEMAND: 'On Demand',
        TV_GUIDE: 'TV Guide',
        SHOWCASE: 'Showcase',
        HELP: 'Help'
    }
);
/**
 * Creates a new FiltersEn instance.
 * @name FiltersEn
 * @memberof freesat.language
 * @class A FiltersEn class, language file.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.language.FiltersEn", [], {},
    {
        KIDS : 'Kids',
        ALL : 'All',
        FILMS : 'Films',
        SPORT : 'Sport',
        FACTUAL : 'Factual',
        TV: 'TV',
        RADIO: 'Radio',
        ALL_RESULTS: 'All results',
        AUDIO_DESCRIPTION: 'Audio Description',
        SUBTITLES: 'Subtitles',
        SIGNING: 'Signing'
    }
);
/**
 * Creates a new HomeMenuEn instance.
 * @name SearchEn
 * @memberof freesat.language
 * @class A SearchEn class, bla functionality.
 * @author <a href="mailto:rhys.jeffreys@accedo.tv">Rhys Jeffreys</a>
 */
accedo.Class.create("freesat.language.SearchEn", [], {},
    {
        SEARCH: 'search',
        STRAPLINE_TITLE: 'find a programme',
        INSTRUCTIONS_HEADER: 'Type in a minimum of 3 characters to search for a programme.',
        INSTRUCTIONS: '<p>Enter a programme name using the arrow keys and OK to select characters on the on-screen keyboard.</p><p>Alternatively, use the number keys on your remote the same way you would on your mobile phone.</p>',
        SEARCH_RESULTS: 'search results',
        RESULT_TYPES: 'Result types:',
        SELECT_RESULT_TYPES: 'Select result type:',
        ON_NOW: "On Now",
        TODAY: "Today"
    }
);
/**
 * Created with JetBrains WebStorm.
 * User: Dean Clancy
 * Date: 07/10/13
 * Time: 07:38
 * To change this template use File | Settings | File Templates.
 */
/*global accedo: false */
/**
 * Creates a new ShowCaseEn instance.
 * @name ShowCaseEn
 * @memberof freesat.language
 * @class A ShowCaseEn class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.language.HelpEn", [], {},
    {
        HELP: 'help',
        SECTION_TITLE: "everything you need to know",
        STRAPLINE: "everything you need to know",
        // OD_STRAPLINE: 'the pick of the programmes to watch right now',
        // SC_STRAPLINE: 'what you should watch this week',
    }
);
/**
 * @namespace freesat.language
 * @memberOf freesat
 */

/**
 * Creates a new OnDemandEn instance.
 * @name OnDemandEn
 * @memberof freesat.language
 * @class A OnDemandEn class, bla functionality.
 */
accedo.Class.create("freesat.language.OnDemandEn", [], {},
    {
        LAUNCH_BUTTON: 'Launch Now',
        CLOSE_BUTTON: 'Close'
    }
);
/**
 * @name IpProgramObject
 * @namespace freesat.models.vo.IpProgramObject
 * @memberOf freesat.models.vo
 * @class object to hold single epg listing
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create('freesat.models.vo.IpProgramObject', [], {}, {
    name: '',//readonly
    longName: '',//readonly
    description: '',//readonly
    longDescription: '',//readonly
    startTime: 0,//Integer
    duration: 0,//Integer
    channelID: '',//readonly
    episode: 0,//Integer
    totalEpisodes: 0,//Integer
    programmeID: '',//readonly
    programmeIDType: 0,//Integer
    parentalRatings: null,
    blocked: false,//Boolean
    showType: 0,//Integer
    subtitles: false,//Boolean
    isHD: false,//Boolean
    audioType: 0,//Integer
    isMultilingual: false,//Boolean
    genre: null,
    hasRecording: false,//Boolean
    audioLanguages: null,
    subtitleLanguages: null,
    locked: false,//Boolean
    recording: null,
    hasSignLanguage: false,//Boolean
    hasAudioDescription: false,//Boolean
    is3D: false,//Boolean
    onlineAvailability: false,//Boolean
    hasGuidance: false,//Boolean
    HDSimulcast: '',//readonly
    onlineMediaLocation: '',//readonly
    imageMediaLocation: '',//readonly
    promotionalContentMediaLocation: '',//readonly
    programmeCRID: '',//readonly
    seriesCRID: '',//readonly
    recommendationCRID: '',//readonly
    CRIDS: []
});

/**
 * User: Dean Clancy
 * Date: 13/12/13
 * Time: 08:25
 */
/*global accedo: false */
/**
 * Creates a new EarlierContainer instance.
 * @name EarlierContainer
 * @memberof freesat.EarlierContainer
 * @class A EarlierContainer class, .
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.vo.EarlierContainer", [], {},
    {
        //Page info
        id: 1,
        date: "02-12-2013",
        heading_title: "earlier today",
        forwardtext: "now",
        backwardtext: "yesterday",
        UpdateDT: 0,
        RefreshDT:0,
        expireDT:0,


        promos:[],

        initalItem: 1,
        programmes:[]
    }
);
/**
 * User: Dean Clancy
 * Date: 13/12/13
 * Time: 11:00
 */
/*global accedo: false */
/**
 * Creates a new EarlierItem instance.
 * @name EarlierItem
 * @memberof freesat.EarlierItem
 * @class A EarlierItem class, .
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.vo.EarlierItem", [], {},
    {
        available:null,
        eventId:null,
        name:null,
        duration:null,
        AvailabilityText:null,
        StartText:null,
        StatusText:null,
        ShortSynopsis:null,
        description:null,
        image:null,
        guidance:null,
        subtitle :null,
        hasSignLanguage :null,
        hasAudioDescription :null,
        audioType :null,
        is3D :null,
        isHD :null,
        isShowcase :null,
        HDLinkage :null,
        series :null,
        linkSID :null,
        linkValue :null,
        links :null,
        channelID:null,
        dolby:null,
        threed:null,
        get isSeries(){
            return this.series;
        },
        get gethasDolby(){
            return this.dolby;
        }

    }
);
/**
 * @name Search
 * @namespace freesat.models.vo.Search
 * @memberOf freesat.models.vo
 * @class object to hold single epg listing plus additional data needed for selection menu
 * @author <a href="mailto:rhys.jeffreys@accedo.tv">Rhys Jeffreys</a>
 */
accedo.Class.create('freesat.models.vo.Search', [], {}, {
    name: '',
    highlighted: '',
    startTime: '',
    date: {},
    count: 0,
    svcId: null,
    evtId: null,
    seriesNo: null,
    episodeNo: null,
    duration: 0,
    searchPos: 0,
    channel: '',
    logo: '',
    logo_highlight: '',
    programObject: null,
    events: [],
    type: null,
    ontv: [],
    ondemand: [],
    availabilityText: ''
});
/**
 * @name Region
 * @namespace freesat.models.vo.Region
 * @memberOf freesat.models.vo
 * @class
 * @author <a href="mailto:sean.barnes@accedo.tv">Sean Barnes</a>
 */
accedo.Class.create('freesat.models.vo.Region', [], {}, {
    _regionId : 0,
    _batId : 0,
    _name : '',
    genres: [],

        /**
         * Get current region id
         * @return {number} Current region id
         * @memberOf freesat.models.vo.Region
         */
        getRegion: function() {
            return this._regionId;
        },

        /**
         * Set new region id
         * @param {number} regionId New region id
         * @memberOf freesat.models.vo.Region
         */
        setRegion: function(regionId) {
            this._regionId = regionId;
        },

        /**
         * Get current bat id
         * @return {number} Current bat id
         * @memberOf freesat.models.vo.Region
         */
        getBatId: function() {
            return this._batId;
        },

        /**
         * Set new bat id
         * @param {number} batId New bat id
         * @memberOf freesat.models.vo.Region
         */
        setBatId: function(batId) {
            this._batId = batId;
        },

        /**
         * Set new region name
         * @param {string} name New region name
         * @memberOf freesat.models.vo.Region
         */
        setName: function(name) {
            this._name = name;
        },

        /**
         * Get current region name
         * @return {string} Current region name
         * @memberOf freesat.models.vo.Region
         */
        getName: function() {
            return this._name;
        }
    }
);

/**
 * @name Genre
 * @namespace freesat.models.vo.Genre
 * @memberOf freesat.models.vo
 * @class
 * @author <a href="mailto:sean.barnes@accedo.tv">Sean Barnes</a>
 */
accedo.Class.create('freesat.models.vo.Genre', [], {}, {
    _name : '',
    _serviceGroupId : 0,
    _channels : [],

        /**
         * Set new genre name
         * @param {string} name New genre name
         * @memberOf freesat.models.vo.Genre
         */
        setName: function(name) {
            this._name = name;
        },

        /**
         * Get current genre name
         * @return {string} Current genre name
         * @memberOf freesat.models.vo.Genre
         */
        getName: function() {
            return this._name;
        },

        /**
         * Set new service group id
         * @param {number} serviceId New service group id
         * @memberOf freesat.models.vo.Genre
         */
        setServiceGroupId: function(serviceId) {
            this._serviceGroupId = serviceId;
        },

        /**
         * Get current service group id
         * @return {number} Current service group id
         * @memberOf freesat.models.vo.Genre
         */
        getServiceGroupId: function() {
            return this._serviceGroupId;
        },

        /**
         * Add new channel
         * @param {object} channel New channel
         * @memberOf freesat.models.vo.Genre
         */
        addChannel: function(channel) {
            this._channels.push(channel);
        }
    }
);
/**
 * @namespace freesat.models.vo
 * @memberOf freesat.models
 */

/**
 * @name Channel
 * @namespace freesat.models.vo.Channel
 * @memberOf freesat.models.vo
 * @class object to hold single channel details
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create('freesat.models.vo.Channel', [], {}, {


    _object: null,

    /** @public */
        /**
         * Get the current channel id
         * @return {number} Current channel id
         * @memberOf freesat.models.vo.Channel
         */
        getChannelId: function() {
            return parseInt(this._object.ccid.split(':')[1], null);
        },
        /**
         * Get the current channel name
         * @return {string} Current channel name
         * @memberOf freesat.models.vo.Channel
         */
        getchannelname: function() {
            return this._object.name;
        },
        /**
         * Get the current 'logical channel name'
         * @return {number} Current logical channel name
         * @memberOf freesat.models.vo.Channel
         */
        getlcn: function() {
            return parseInt(this._object.ccid.split(':')[1], null);
        },
        /**
         * Set the current logo URL
         * @return {string} Current logo URL
         * @memberOf freesat.models.vo.Channel
         */
        getlogoURL: function() {
            return this._object.logoURL;
        },
        _tstv : false ,
        /**
         * Get the current 'Time Shift TV' availability
         * @return {boolean} 'Time Shift TV' availability
         * @memberOf freesat.models.vo.Channel
         */
        getHasTSTV: function() {
            return this._tstv;
        },

        getchannelType: function() {
            return this._object.channelType;
        },

        getidType: function() {
            return this._object.idType;
        },

        getccid: function() {
            return this._object.ccid;
        },

        gettunerID: function() {
            return this._object.tunerID;
        },

        getonid: function() {
            return this._object.onid;
        },

        getnid: function() {
            return this._object.nid;
        },

        gettsid: function() {
            return this._object.tsid;
        },

        getsid: function() {
            return this._object.sid;
        },

        getsourceID: function() {
            return this._object.sourceID;
        },

        getfreq: function() {
            return this._object.freq;
        },

        getcni: function() {
            return this._object.cni;
        },

        getname: function() {
            return this._object.name;
        },

        getmajorChannel: function() {
            return this._object.majorChannel;
        },

        getminorChannel: function() {
            return this._object.minorChannel;
        },

        getdsd: function() {
            return this._object.dsd;
        },

        getfavourite: function() {
            return this._object.favourite;
        },

        getlocked: function() {
            return this._object._locked;
        },

        getmanualBlock: function() {
            return this._object.manualBlock;
        },

        getipBroadcastID: function() {
            return this._object.ipBroadcastID;
        },

        getchannelMaxBitRate: function() {
            return this._object.channelMaxBitRate;
        },

        getchannelTTR: function() {
            return this._object.channelTTR;
        },

        getrecordable: function() {
            return this._object.recordable;
        },

        getlongName: function() {
            return this._object.longName;
        },

        getdescription: function() {
            return this._object.description;
        },

        getauthorised: function() {
            return this._object.authorised;
        },

        getgenre: function() {
            return this._object.genre;
        },

        gethidden: function() {
            return this._object.hidden;
        }
    }
);

/**
 * @name ChannelEvent
 * @namespace freesat.models.vo.ChannelEvent
 * @memberOf freesat.models.vo
 * @class object to hold single event returned from
 * 'now and next',
 * 'now and next all' (wrapped in a channel tag),
 * 'now next next all' (wrapped in a channel tag)
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create('freesat.models.vo.ChannelEvent', [], {}, {
        name: '',
        description: '',
        startTime: '',
        duration: '',
        serviceId: 0,
        eventId: 0,
        image: '',
        genre: '',
        seriesNo: 0,
        episodeNo: 0,
        trailer: '',
        hasTstv: false,
        tstv: null
    }
);
/// ---------------------------- example data
//  <event>
//    <name>Homes Under the Hammer</name>
//    <description>Martin Roberts and Lucy Alexander visit a property in Devon, a former coach house in Kent and a terraced house in Birmingham. [S]</description>
//    <startTime>1378112400</startTime>
//    <duration>3600</duration>
//    <svcId>501</svcId>
//    <evtId>36741</evtId>
//    <image>http://images.platform.freesat.tv/g2/img/epg/rb/6321034.l.jpg</image>
//    <genre>Film</genre>
//    <seriesNo>2</seriesNo>
//    <episodeNo>3</episodeNo>
//    <trailer>http://apps.platform.freesat.tv/g2/htmlapps/helpplayer/player.php?type=2</trailer>
//    <hasTstv>false</hasTstv>
//  </event>

/**
 * @name ChannelEventContainer
 * @namespace freesat.models.vo.ChannelEventContainer
 * @memberOf freesat.models.vo
 * @class
 * @author <a href="mailto:sean.barnes@accedo.tv">Sean Barnes</a>
 */
accedo.Class.create('freesat.models.vo.ChannelEventContainer', [], {}, {
    _channelId : 0,
    _events : [],

        /**
         * Set the channel id
         * @param {number} id New channel id
         * @memberOf freesat.models.vo.ChannelEventContainer
         */
        setChannelId: function(id) {
            this._channelId = id;
        },

        /**
         * Get the channel id
         * @return {number} Current channel id
         * @memberOf freesat.models.vo.ChannelEventContainer
         */
        getChannelId: function() {
            return this._channelId;
        },

        /**
         * Add channel event
         * @param {object} event Channel event
         * @memberOf freesat.models.vo.ChannelEventContainer
         */
        addChannelEvent: function(event) {
            if (this._events === undefined) {
                this._events = [];
            }
            this._events.push(event);
        },

        /**
         * Get the first event in the stack
         * @return {object} event
         * @memberOf freesat.models.vo.ChannelEventContainer
         */
        getFirstEvent: function() {
            if (this._events.length > 0) {
                return this._events[0];
            }
            return null;
        },

        /**
         * Get all events from the stack
         * @return {object[]} events
         * @memberOf freesat.models.vo.ChannelEventContainer
         */
        getAllEvents: function() {
            return this._events;
        }
});
/**
 * @namespace freesat.models
 * @memberOf freesat
 */

/**
 * @name ChannelData
 * @namespace
 * @memberOf freesat.models
 * @class
 * @author <a href="mailto:sean.barnes@accedo.tv">Sean Barnes</a>
 */
accedo.Class.create('freesat.models.ChannelData', [], {}, {
        _callbackArray : [],
        _data : undefined,
        channelList: null,
        channelListLength: null,
        channelLogos: {},
        genreList: null,
        isReady: false,
        assignCCID: true,


        OUTPUT_FORMAT: 'json',
        URL_ROOT: 'http://servicelist.dev-platform.freetime.tv/g2ip',

        _callback : function(scope) {
            var i;
            for (i = 0; i < scope._callbackArray.length; i++){
                scope._callbackArray[i]();
            }
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.ChannelData
         */
        _onIPDataCompleted: function(scope){
            return function(data) {
                scope._data = data;
                if(data){
                    if(scope.channelList){
                        scope.mapIPtoOIPFchannels(scope);
                        scope._callback(scope);
                    }
                }else{
                    scope._callback(scope);
                }
            };
        },
        _getChannel: function(scope) {
            return function(ccid) {
                var ch = 0;
                for (var loop=0; loop<scope.channelListLength; loop++) {
                    if (scope.channelList[loop].ccid == ccid) {
                        ch = scope.channelList[loop];
                        break;
                    }
                }
                return ch;
            };
		},
        _onDataCompleted: function(scope){
            return function(chList) {
                    if(chList){
                            scope.channelList = chList;
                            scope.channelListLength = chList.length;
//TODO: JSON
//            return function(chConfig) {
//                if(chConfig){
//                        scope.channelListOIPF = chConfig.channelList;
//                        scope.channelList = JSON.parse(chConfig.channelListToJSON());
//                        scope.channelListLength = scope.channelList.length;
//                        // extend JSON parsed channel list to include getChannel()
//                        scope.channelList.getChannel = scope._getChannel(scope);
                        freesat.console.log(scope.channelListLength + ' channels in channelList');
                        freesat.dataStore = new freesat.data.dataStore();
                        freesat.dataStore.setup();
                        scope._callback(scope);
                }
            };
        },
        /**
         * mapIPtoOIPFchannels
         * @description map oipf channels to ip data
         */
        mapIPtoOIPFchannels : function(scope){
            var i, n, q;//, tArr;
            if(scope._data && scope.channelList){
                for(i=0;i<scope.channelListLength;i++){
                    var _genres = scope._data.regions.genre;
                    for (n = 0; n < _genres.length; n++) {
                        for (q = 0; q < _genres[n].channels.length; q++) {
                            //TODO this should not be channelname - it should work from some id
//                         tArr = scope._data[n].channelServiceId.split(':');
//                         if(scope.channelList[i].onid == tArr[0] && scope.channelList[i].tsid == tArr[1] && scope.channelList[i].sid == tArr[2]){
                            if (_genres[n].channels[q].channelname == scope.channelList[i].channelname) {
                                scope.channelList[i].ipobject = _genres[n].channels[q];
                                scope.channelList[i].channelid = _genres[n].channels[q].channelid;
                                break;
                            }
                        }
                    }
                }
            }
            scope.isReady = true;
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.ChannelData
         */
        getData: function() {
            return this._data;
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.ChannelData
         */
        initialiseData: function(callback) {
            this.registerCallback(callback);
            this.getChannels(this._onDataCompleted(this));
//            this.getIPChannels(this._onIPDataCompleted(this));

        },
        registerCallback: function(callback){
            this._callbackArray.push(callback);
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.ChannelData
         */
        getServiceGroupIdForChannel: function(channelLCN) {
            //if(freesat.channelData.channelList.getChannel('ccid:'+channelLCN).channelType = 1)return 200;
            return 0;
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.ChannelData
         */
        getChannelNameAndLogo: function(channelId) {
            var channel = this.channelList.getChannel('ccid:'+channelId);
            if (channel) {
                return {
                    name: channel.name,
                    logo: channel.logoURL,
                    channelnumber: parseInt(channel.ccid.split(':')[1], null)
                };
            }
            return {
                name: 'Not Found',
                logo: channelId + '/missing/from/channelList',
                channelnumber: 0
            };
        },

        /**
         *
         * @param id {string} - serviceID
         * @returns {*} channel from channelList or false
         */
        getChannelObjectByServiceID: function(sid) {
            var DSAT_ONID = 2;
            var DTT_ONID = 9018;
            var onid = (freesat.datamode === 'dsat') ? DSAT_ONID : DTT_ONID;
            var result = this.channelList.getChannelByTriplet(
                (freesat.datamode === 'dsat') ? DSAT_ONID : DTT_ONID,
                null,
                sid);

            if (!result) {
                freesat.console.log('WARNING: couldn\'t find channel with triplet ' + onid + '..' + sid);
            }

            return result;
        },

        /**
         *
         * @param ccid {string} - CCID (inc. prefix e.g. 'ccid:120')
         * @returns {*} channel from channelList or null
         */
        getChannelObjectByCCID: function(ccid) {
            var result = this.channelList.getChannel(ccid);
            if  (!result) {
                freesat.console.log('WARNING: couldn\'t find channel with ccid ' + ccid);
            }

            return result;

        },

        /**
         * Convenience function to derive channel object from either a SID or
         * CCID (depending on prefix).
         *
         * @param id {string} - CCID (inc. prefix e.g. 'ccid:120') or a SID
         * @returns {*} channel from channelList or null
         */
        getChannelObjectByCcidOrServiceID: function(id) {
            var channelForProgramme = null;
            if ((typeof id === 'string') && id.indexOf('ccid:') === 0) {
                return freesat.channelData.getChannelObjectByCCID(id);
            } else {
                return freesat.channelData.getChannelObjectByServiceID(id);
            }
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.ChannelData
         */
        getRadioChannelFSImage: function(channelId) {
            var channel = this.channelList.getChannel('ccid:'+channelId);
            if (channel.radioImage) {
                //TODO connect up to real radio image when available FREESAT-364
                return channel.radioImage;
            } else {
                return null;
            }
        },
        getRadioChannelFSImageFromLCN: function(lcn) {
            var channel = this.channelList.getChannel('ccid:'+lcn);
            if (channel.radioImage) {
                //TODO connect up to real radio image when available FREESAT-364
                return channel.radioImage;
            } else {
                return null;
            }
        },


        setChannelLogos: function(json) {
            if (json) {
                var data = JSON.parse(json),
                    images = data.imageMappings.img,
                    len, id, stb, fsatid, obj, i;

                freesat.console.log('setChannelLogos');
                freesat.console.log(data);

                try {
                    len = images.length;
                    freesat.console.log('Length: ' + len);
                } catch (e) {
                    freesat.console.log('Cannot access imageMappings.img: ' + e);
                }

                for (i = 0; i < len; i++) {
                    id = images[i].id;
                    if (id.indexOf('fs-img://fsatid/') !== -1) {
                        fsatid = id.replace('fs-img://fsatid/', '').replace('-hi', '');

                        if (!freesat.channelData.channelLogos[fsatid]) {
                            freesat.channelData.channelLogos[fsatid] = {
                                img: images[i].stb
                            };
                        } else {
                            if (id === ('fs-img://fsatid/' + fsatid + '-hi')) {
                                freesat.channelData.channelLogos[fsatid].img_hi = images[i].stb;
                            }
                            if (id === ('fs-img://fsatid/' + fsatid)) {
                                freesat.channelData.channelLogos[fsatid].img = images[i].stb;
                            }
                        }
                    }
                }
            }
        },
        ////////////////////////
        // Channel  metadata  //
        ////////////////////////

        getChannelLogo: function(fsID, highlight) {
            var channelLogos = freesat.channelData.channelLogos,
                url, img;

            try {
                url = freesat.localConfig.config.recdata_loc;
            } catch (e) {
                console.log('LOCAL CONFIG ERROR: ' + e);
            }

            if (channelLogos[fsID] && url) {
                if (highlight) {
                    img = url + channelLogos[fsID].img_hi;
                } else {
                    img = url + channelLogos[fsID].img;
                }
            }
            return img;
        },

        getHasTSTVfromChannel: function(channel){
            var hasTSTV = freesat.localConfig.ipservicelist.onDemandChannel[channel.freesatServiceID];
            freesat.console.log('hasTSTV: ' + hasTSTV + '; freesatServiceId: ' + channel.freesatServiceID);
            if (hasTSTV) {
                freesat.localConfig.ipservicelist.onDemandChannel[channel.freesatServiceID].ccid = channel.ccid;
                return true;
            }
            return false;
        },
        assignCCIDtoTSTVchannels: function(){
            if(this.assignCCID){
                var i, q, ccid, len = freesat.localConfig.ipservicelist.ondemandFSATids.length;
                var got = 0, channel, chFsatId, searchArray = [];
                for(i=0; i<len; i++){
                    ccid = freesat.localConfig.ipservicelist.onDemandChannel[freesat.localConfig.ipservicelist.ondemandFSATids[i]].ccid;
                    if(ccid === null || ccid === undefined){
                        searchArray.push(freesat.localConfig.ipservicelist.ondemandFSATids[i]);
                    }
                }
                len = searchArray.length;
                if(len>0){
                    for(i=0; i<freesat.channelData.channelListLength; i++){
                        channel = freesat.channelData.channelList[i]
                        chFsatId = String(channel.freesatServiceID);
                        for(q=0; q<len; q++){
                            if(chFsatId== String(searchArray[q])){
                                freesat.localConfig.ipservicelist.onDemandChannel[searchArray[q]].ccid = String(channel.ccid);
                                searchArray.splice(q, 1);
                                if(searchArray.length == 0) {
                                    return;
                                }
                                len = searchArray.length;
                                break;
                            }
                        }
                    }
                }
                this.assignCCID = false;
            }
        },
        returnCUTVchannellist: function(){
            var q, retArr = [],
                ondemandFSATids = freesat.localConfig.ipservicelist.ondemandFSATids,
                onDemandChannel = freesat.localConfig.ipservicelist.onDemandChannel,
                len = ondemandFSATids.length;

            if(len>0){
                for(q=0; q<len; q++){
                    if(onDemandChannel[ondemandFSATids[q]].ccid){
                        retArr.push(onDemandChannel[ondemandFSATids[q]].ccid.split(':')[1]);
                    }
                }
            }
            retArr.sort(function(a,b){return a - b});
            return retArr;
        },
        compareCCID: function(a,b) {
            if (a.ccid < b.ccid)
                return -1;
            if (a.ccid > b.ccid)
                return 1;
            return 0;
        },
        returnChannelListIndexFromCCID: function(ccid){
            var q;
            if(ccid.indexOf('ccid')<0) ccid = 'ccid'+ccid;
            for (q=0; q<freesat.channelData.channelListLength; q++) {
                if (freesat.channelData.channelList[q].ccid === ccid) {
                    return q;
                }
            }
            return null;
        },
        getShowcase: function(programme){
            try{
                if(freesat.getZone()==='earlier'){
                    return programme.isShowcase;
                }else{
                    return false;
                }
            }catch(e){
                freesat.console.log('channelData.getShowcase '+e);
                return false;
            }
        },
        hasRecording: function(programme){
            return false;
        },
        hasHDSimulcast: function(programme){
            try{
                if(freesat.getZone()==='earlier'){
                    return (programme.HDLinkage)?true:false;
                }else{
                    return programme.HDSimulcast;
                }
            }catch(e){
                freesat.console.log('channelData.hasHDSimulcast '+e);
                return false;
            }
        },
        getPlayFromStart: function(programme){
            return this.getHasTSTVfromProgramme(programme);
        },
        earlierPlayFromStart: function(programme){
            return false;
        },
        getHasTSTVfromProgramme: function(programme){
            //programme.channel triplet
            //TODO ip data / dsat data
            return false;
        },
        getIsProgrammeRecording: function(programme) {
            return freesat.masterController.recordingsHandler.hasRecording(programme);
        },
        gethasRecording: function(programme) {
            if (programme._class && programme._class == 'freesat.models.vo.IpProgramObject') {
                // no OIPF object available here - check against channel & start time
                return freesat.masterController.recordingsHandler.isRecordingScheduled(
                    programme.channelID,
                    programme.startTime);
            }
            return freesat.masterController.recordingsHandler.hasRecording(programme);
        },
        getIsSeries: function(programme){
            try{
                var crids = programme.CRIDS;
                for(var i = 0 ; i < crids.length ; i++){
                    if(crids[i].indexOf('0x32')>-1){
                        return true;
                    }
                }
            }catch(e){
                freesat.console.log('ChannelData no CRIDs');
            }
            return false;
        },
        gethasDolby: function(programme){
            return (freesat.datamode === 'dsat' && programme && programme.audioType === 4); // Compare audioType is 4 and return boolean
        },

        gethasSurroundSound: function(programme) {
            return (freesat.datamode === 'dtt' && programme && programme.audioType === 4);
        },

        getHasReminder: function(programme) {
            if (programme._class && programme._class == 'freesat.models.vo.IpProgramObject') {
                // no OIPF object available here - check against channel & start time
                return freesat.masterController.remindersHandler.isReminderScheduled(
                    programme.channelID,
                    programme.startTime);
            }
            return freesat.masterController.remindersHandler.hasReminder(programme);
        },


        ////////////////////////
        // Channels  Requests //
        ////////////////////////

        getChannels: function(completedCallback) {
            var channelList = freesat.oipfElement.searchManager.getChannelConfig().channelList;
            completedCallback(channelList);
            return null;
            
//TODO: JSON
//            var myChannelConfig = freesat.oipfElement.searchManager.getChannelConfig();
//            completedCallback(myChannelConfig);
        },

        getIPChannels: function(completedCallback) {
            var URL = this.URL_ROOT + '/channels/' + this.OUTPUT_FORMAT + '/chlist/' + freesat.dataStore.bat + '/' + freesat.dataStore.region;
            accedo.Ajax.request(URL, {
                method: 'get',
                contentType: 'application/json',
                onSuccess: function(data) {
                    completedCallback(data.responseJSON);
                },
                onFailure: function() {
                    completedCallback(null);
                }
            }, this);
        }
    }
);

/**
 * @namespace freesat.models.oipf
 * @memberOf freesat.models
 */

/**
 * Creates a new Elements instance.
 * @name Elements
 * @namespace
 * @memberOf freesat.models.oipf
 * @class A Elements class, bla functionality.
 * @author <a href='mailto:dean.clancy@accedo.tv'>Dean Clancy</a>
 */
accedo.Class.create('freesat.models.oipf.Elements', [], {},
    {
        videoBroadcast: null,
        searchManager: null,
        recordingsManager: null,
        appManager:null,
        config: null,
        mSearch: null,
        searchActive: false,
        nowNextTTL: null,
        searchType: null,
        /**
         * initialize
         * @memberOf freesat.models.oipf.Elements
         */
        initialize:function(){
            this.setup();
        },

        /**
         * Setup
         * @description create references to oipf objects and setup the oipf enviroment
         * @memberOf freesat.models.oipf.Elements
         */
        setup: function(){
            this.videoBroadcast = document.getElementById('video');
            this.searchManager = document.getElementById('srchmgr');
            this.config = document.getElementById('oipfcfg');
            this.recordingsManager = document.getElementById('oipfrecorder');

            try{
                if(window.oipfObjectFactory.isObjectSupported('application/oipfSearchManager')){
                    freesat.oipfEnabled = true;
                }else{
                    freesat.oipfEnabled = false;
                }
            }catch (e){
                freesat.oipfEnabled = false;
            }
            this.setKeyset();
            this.appManager = document.getElementById('appmgr').getOwnerApplication(document);
            try{
                this.appManager.onLowMemory = this.onLowMemory;
            }catch(e){
                freesat.console.log('onLowMenory fail : \n' + e);
            }
        },

        /**
         * isHomescreen()
         * returns boolean
         */
        isHomescreen: function(){
            try{
                this.appManager = document.getElementById('appmgr').getOwnerApplication(document);
                freesat.console.log('appManager.applicationState : '+this.appManager.applicationState);
                return (this.appManager.applicationState === 2);
            }catch(e){
                freesat.console.log('isHomescreen ' + e);
                return true;
            }
        },
        /**
         * hide
         */
        hide: function(){
            this.appManager = document.getElementById('appmgr').getOwnerApplication(document);
            this.appManager.hide();
        },
        /**
         * show
         */
        show: function(){
            this.appManager = document.getElementById('appmgr').getOwnerApplication(document);
            this.appManager.show();
        },

        /**
         * setKeyset
         * configure the keyset required
         */
        setKeyset: function() {
            var mask = 0x1+0x2+0x4+0x8+0x10+0x40+0x80+0x200+0x400;
            if (freesat.section && freesat.section.length > 0){
                if(freesat.section != 'ftvg') {
                    mask += 0x100;
                }
            }
            this.setKeys(mask);
        },
        setKeysetWithoutNumbers: function() {
            freesat.console.log('oipfElement setKeysetWithoutNumbers');
            var mask = 0x1+0x2+0x4+0x8+0x10+0x20+0x40+0x80+0x200+0x400;
            this.setKeys(mask);
        },
        setKeysetWithNumbers: function() {
            freesat.console.log('oipfElement setKeysetWithNumbers');
            var mask = 0x1+0x2+0x4+0x8+0x10+0x20+0x40+0x80+0x200+0x400;
            mask += 0x100;
            this.setKeys(mask);
        },
        setKeys: function(mask){
            try {
                this.appManager = document.getElementById('appmgr').getOwnerApplication(document);
                this.appManager.privateData.keyset.setValue(mask);
            } catch (e) {
                console.log('Exception setting keyset: ' + e);
            }
        },
        /**
         * setKeyset
         * configure the keyset required
         */
        setKeysetShow: function() {
            var red = 0x1; //red
            var green = 0x2; //green
            var yellow = 0x4; //yellow
            var blue = 0x8; //blue
            var nav = 0x10;
            var pgUpDown = 0x40;
            var info = 0x80;
            var mask = nav + pgUpDown + info+green+yellow+blue+red;
            try {
                var app = document.getElementById('appmgr').getOwnerApplication(document);
                app.privateData.keyset.setValue(mask);
                app.privateData.keyset.value = mask;
                app.show();
            } catch (e) {
                // ignore
            }
        },
        setKeysetHide: function() {
            var red = 0x1; //red
            var green = 0x2; //green
            var yellow = 0x4; //yellow
            var blue = 0x8; //blue
            var nav = 0x10;
            var pgUpDown = 0x40;
            var info = 0x80;
            var mask = nav + pgUpDown + info+green+yellow+blue;
            try {
                var app = document.getElementById('appmgr').getOwnerApplication(document);
                app.privateData.keyset.setValue(mask);
                app.privateData.keyset.value = mask;
                app.hide();
            } catch (e) {
                // ignore
            }
        },

/**
         * getFreeMem
         */
        getFreeMem: function(){
            try{
                freesat.console.log('[APP] free mem = ' + this.appManager.privateData.getFreeMem());
            }catch(e){
                freesat.console.log('[ERROR] ' + e );
            }
        },
        /**
         * onLowMemory
         */
        onLowMemory: function(){
            freesat.console.log('******------ LOW MEMORY ------******');
        },
        /**
         * garbageCollection
         */
        garbageCollection: function(){
            var app = this.appManager.getOwnerApplication(document);
            try{
                freesat.console.log('garbageCollection : ' + app.gc());
            }catch(e){
                freesat.console.log('garbageCollection fail : \n' + e);
            }
        }
    }
);
/**
 * Creates a new Clock instance.
 * @name Clock
 * @namespace
 * @memberof freesat.models
 * @class A Clock class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create('freesat.models.Clock', [], {},
    {
        /**
         * @private
         * @name _clockLabel
         * @memberof freesat.models.Clock#
         */
        _clockLabel : null,
        _clockInterval: null,
        minute: null,
        startTime: null,
        duration: null,

        /**
         * @todo - documentation
         * @memberOf freesat.models.Clock
         */
        currentTime: function(){
            var curDate = new Date(),
                hh = curDate.getHours(),
                mm = curDate.getMinutes(),
                zone = freesat.getZone();

            if (!this.minute) {
                this.minute = mm;
            } else if (mm !== this.minute) {

                this.minute = mm;

                if (this.startTime && (zone === 'now' || zone === 'later')) {
                    this.updateDuration(this.startTime, this.duration);
                }

                if (zone === 'later' && (mm === 30 || mm === 0)) {
                    freesat.behind.setNowGridStartTime();
                    freesat.behind.animationClass.hideLaterGrid();
                }
            }

            var hStr = (String(hh).length>1)? String(hh) : '0' + String(hh);
            var mStr = (String(mm).length>1)? String(mm) : '0' + String(mm);
            return hStr + ':' + mStr;
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.Clock
         */
        updateClock: function(clockLabel){
            if(clockLabel){
                this._clockLabel = clockLabel;
                if(!this._clockInterval){
                    this._clockInterval = setInterval(this.setClock(this), 1000);
                    this.setClock(this)();
                }
            }
        },

        setDuration: function(startTime, duration) {
            this.startTime = startTime;
            this.duration = duration;
        },

        updateDuration: function(startTime, duration) {
            // Empty function to be overwritten by callback
            freesat.console.log(startTime + ', ' + duration);
        },

        unsetDuration: function() {
            this.startTime = null;
            this.duration = null;
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.Clock
         */
        setClock: function(scope){
            return function(){
                if (scope._clockLabel) {
                    scope._clockLabel.setText(scope.currentTime());
                } else {
                    if(scope._clockInterval){
                        clearInterval(scope._clockInterval);
                    }
                }
            };
        },

        /**
         * @todo - documentation
         * @memberOf freesat.models.Clock
         */
        reset: function(){
            if(this._clockInterval){
                clearInterval(this._clockInterval);
            }
        }
    }
);
/**
 * User: Dean Clancy
 * Date: 13/11/13
 * Time: 07:47
 */
/*global accedo: false */
/**
 * Creates a new NetworkInfo instance.
 * @name NetworkInfo
 * @memberof freesat.NetworkInfo
 * @class A NetworkInfo class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.models.NetworkInfo", [], {},
    {
        connected: false,
        isConnected: function () {
            this.investigateNetwork();
            return this.connected;
        },
        investigateNetwork: function() {
            var i, len, net = freesat.oipfElement.config.localSystem.networkInterfaces;

            len = net.length;

            for (i=0; i<len; i++) {
                this.connected = (this.connected)?this.connected:net.item(i).connected;
            }
        }
    }
);
var Interface = function(name, methods) {
    if (arguments.length !== 2) {
        throw new Error('Interface constructor called with ' + arguments.length + 'arguments, but expected exactly 2.');
    }
    this.name = name;
    this.methods = [];
    for (var i = 0, len = methods.length; i < len; i++) {
        if (typeof methods[i] !== 'string') {
            throw new Error('Interface constructor expects method names to be ' + 'passed in as a string.');
        }
        this.methods.push(methods[i]);
    }
};

/** @static class method. */
Interface.ensureImplements = function(object) {
    if (arguments.length < 2) {
        throw new Error('Function Interface.ensureImplements called with ' + arguments.length + 'arguments, but expected at least 2.');
    }
    for (var i = 1, len = arguments.length; i < len; i++) {
        var interface = arguments[i];
        if (interface.constructor !== Interface) {
            throw new Error('Function Interface.ensureImplements expects arguments' + 'two and above to be instances of Interface.');
        }
        for (var j = 0, methodsLen = interface.methods.length; j < methodsLen; j++) {
            var method = interface.methods[j];
            if (!object[method] || typeof object[method] !== 'function') {
                throw new Error('Function Interface.ensureImplements: object ' + 'does not implement the ' + interface.name + ' interface. Method ' + method + ' was not found.');
            }
        }
    }
};


//definition of the IDataSource Interface
var IDataSource = new Interface('IpDataSource', [
]);

/**
 * @namespace freesat.data
 * @memberOf freesat
 */

/**
 * @name dataInterfaceFactory
 * @namespace
 * @memberof freesat.data
 * @class
 */
accedo.Class.create('freesat.data.dataInterfaceFactory', [], {}, {
        /**
         * @memberOf dataInterfaceFactory
         */
        getInterface : function (){
            var _dataInterface = null;
            switch (freesat.datamode){
                case 'DSAT':
                case 'dsat':
                case 'DTT':
                case 'dtt':
//                case 'DTT-ip':
                    _dataInterface = new freesat.data.Dtt_data();
                    break;
            }
            _dataInterface.setup();
            return _dataInterface;
        }
    }
);

/**
 * User: Dean Clancy
 * Date: 24/10/13
 * Time: 13:45
 */
/*global accedo: false */
/**
 * Creates a new Dtt_data instance.
 * @name Dtt_data
 * @memberof freesat.Dtt_data
 * @class A Dtt_data class, dtt data methods.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.data.Dtt_data", [], {},
    {
        cr: '\n',
        RECEIVER_TYPE: 'g2',
        OUTPUT_FORMAT: 'json',
//        URL_ROOT: 'http://servicelist.dev-platform.freetime.tv/g2ip',
//        URL_ROOT: 'http://search.dev-platform.freetime.tv:7790/' + freesat.datamode,
        URL_ROOT: freesat.models.config.Auth.epgAbout,

        username: null,
        password: null,
        bat: null,
        region: null,

        client: null,

        ready: false,
        readyCallback: null,

//        nowNextArray: [],
//        nowNextData: [],
        nowNextGenreArray: [],

        collected:0,
        collectItem: 0,
        genreCollected: 0,

        newDataCall: null,
        newDataCallArgs: [],
        dataCallback: null,

        hasMoreRecommendations: false,
        moreRecommendationsCallback: null,
        moreRecommendationsIndex: 0,
        recommendationsCRIDS: [],

        aboutTimeout: null,

        get updateFunc() {
            return;
        },

        set updateFunc(val){
            this.nownext.updateFunc = val;
        },
//        updateFunc: null,
//        updateInterval: null,
//        updateArray: [],
//
//        retry: 0,
//        retryData: [],
//        retryFunc: null,
//        retryInterval: null,

        abortiveSearch: false,

        laterTimeout: null,

        ondemand: null,
        earlier: null,
        nownext: null,
        later: null,

        setup: function(){
            freesat.console.log('DTT setup');
            this.ondemand = new freesat.data.datafunc.Ondemand();
            this.earlier = new freesat.data.datafunc.Earlier();
            this.later = new freesat.data.datafunc.Later();
            this.nownext = new freesat.data.datafunc.Nownext();
            this.nownext.owner = this;
            this.nownext.resetNowNextArray();
            this.ready = true;
            if (this.readyCallback) this.readyCallback();
            freesat.console.log('DTT setup DONE');
        },
        getStartTime: function() {
            return new Date().getHours();
        },
        ////////////////
        // USER SETUP //
        ////////////////
        setCredentials: function(username, password) {
            this.username = username;
            this.password = password;
        },

        setBatAndRegion: function() {
        },

        onPostCodeSuccess: function(scope){
            return function() {
            };
        },
        getNowAndNextAll: function(dataCallback, topPositionIndex, callbackAt){
            freesat.console.log('#### dtt_data getNowAndNextAll');
            this.nownext.filteredList = false;
            this.nownext.getNowAndNextAll(dataCallback, topPositionIndex, callbackAt);
        },

        getNowAndNextGenre: function(dataCallback, topPositionIndex, callbackAt) {
            freesat.console.log('1: getNowAndNextGenre');
            this.nownext.filteredList = true;
            this.nownext.getNowAndNextAll(dataCallback, topPositionIndex, callbackAt);
        },

        ////////////////
        // Channels   //
        ////////////////

        getChannels: function(completedCallback) {
            completedCallback(freesat.channelData.channelList);
//            TODO: JSON
//            completedCallback(freesat.channelData.channelListOIPF);
        },

        ///////////////
        //    IP     //
        ///////////////
        request: function(url, callback, scope, timeoutSecs) {
            var timeoutMillis = timeoutSecs ? (timeoutSecs * 1000) : 10000;
            freesat.console.log('request (timeout=' + timeoutSecs + 's): ' + url);
            freesat.masterController.showLoading();
            scope.client = new XMLHttpRequest();

            var timeoutTimer = window.setTimeout(function() {
                freesat.console.log('Request timed out after ' + timeoutMillis + 'ms - aborting');
                scope.client.abort();
            }, timeoutMillis);

            scope.client.onreadystatechange = function() {
                if (scope.client.readyState == XMLHttpRequest.DONE) {
                    window.clearTimeout(timeoutTimer);
                    if (scope.client.status === 200) {
                        scope.data = callback.success(callback.returnCallback);
                    } else {
                        freesat.console.log('Got error result, status code=' + scope.client.status);
                        scope.data = callback.failure(scope.client.status, callback.returnCallback);
                    }
                }
            };

            try {
                scope.client.open('get', url, true, scope.username, scope.password);
                scope.client.send();
            } catch (Error) {
                freesat.console.log(Error.message);
                window.clearTimeout(timeoutTimer);
            }
        },

        failure: function() {
            freesat.masterController.hideLoading();
            freesat.console.log('Failed to retrieve file');
        },

        /////////////////
        // Earlier     //
        /////////////////
        getEarlierData: function(channel, startday, starthour, numberofhours, dataCallback){
            this.earlier.getEarlierData(channel, startday, starthour, numberofhours, dataCallback);
        },

        /////////////////
        // Now     //
        /////////////////
        findNowAndNextForChannel: function(callback, channel){
            this.nownext.findNowAndNextForChannel(this.nownext, channel, callback);
        },

        /////////////////
        // About       //
        /////////////////

        /* IP ABOUT */
        ipAbout: function(dataCallback, freesatServiceId, eventId) {
            var progId = eventId;
            if (typeof progId === 'string'){
                progId = parseInt(progId.split(';')[1], 16);
            }

            var url = this.checkUrlForRegional(freesat.localConfig.auth.epgAbout);
            url += (url.lastIndexOf('/') === url.length - 1) ? '' : '/';
            url += 'ipabout/' + this.OUTPUT_FORMAT + '/' + freesatServiceId + '/' + progId;
            url = this.addRegionalValues(url);

            var callback = {
                success: this.onIPAboutSuccess(this, freesatServiceId, progId),
                failure: this.onIPAboutFailure(this),
                returnCallback: dataCallback
            };

            freesat.masterController.showLoading();

            this.request(url, callback, this, freesat.localConfig.auth.epgAboutTimeoutSecs);
        },

        onIPAboutSuccess: function(scope, channelID, progID){
            return function(dataCallback) {
                freesat.masterController.hideLoading();
                var data = JSON.parse(scope.client.responseText);

                // Add programme ids so correct programme can be identfied
                data.queryParams = {
                    channelID: channelID,
                    progID: progID
                }
//                var result = dataCallback(data); //scope.parseSearchResults(data);
                dataCallback(data);
            };
        },
        onIPAboutFailure: function(scope){
            return function(statusCode, dataCallback) {
                freesat.masterController.hideLoading();
                dataCallback(null);
            };
        },

        getIpShowingAgain: function(dataCallback){

        },

        /////////////////
        // ShowingAgain//
        /////////////////
        getShowingAgain: function(dataCallback, programme) {
              this.getCRIDSearch(dataCallback, '0x31', programme);
        },
        getEpisodes: function(dataCallback, programme) {
              this.getCRIDSearch(dataCallback, '0x32', programme);
        },
        getRecommend1: function(dataCallback, programme) {
            this.getCRIDSearch(dataCallback, '0x33', programme);
            this.getRecommendationsCRID(programme);
        },
        getRecommendationsCRID: function(dataCallback, moreDataCallback, programme) {
            try {
                var crids = programme.CRIDS,
                    len = crids.length,
                    count = 0,
                    recommendationsCRID = [],
                    searchType = '0x33',
                    crid,
                    args,
                    i;

                for (i = 0; i < len; i++) {
                    crid = crids[i];
                    console.log('CRID: ' + crid);

                    if (crid.indexOf('0x33') !== -1) {
                        recommendationsCRID.push('crid'+String(crid).split('crid')[1]);
                        count += 1;
                    }
                }

                if (count > 0) {

                    if (count > 1) {
                        this.recommendationsCRIDS = recommendationsCRID;
                        this.moreRecommendationsCallback = moreDataCallback;
                        this.hasMoreRecommendations = true;
                        this.moreRecommendationsIndex = 1;
                    }
                    args = [dataCallback, recommendationsCRID[0], searchType];
                    this.getAbout(args);
                } else {
                    dataCallback();
                }

            } catch (e) {
                freesat.console.log('Cannot access Recommendations CRID: ' + e);
                dataCallback();
            }

        },

        getMoreRecommendations: function() {
            var args = [this.moreRecommendationsCallback, this.recommendationsCRIDS[this.moreRecommendationsIndex], '0x33'];


            if (this.recommendationsCRIDS.length - 1 === this.moreRecommendationsIndex) {
                this.moreRecommendationsIndex = 0;
                this.hasMoreRecommendations = false;
                this.recommendationsCRIDS = [];
                this.moreRecommendationsCallback = null;
            } else {
                this.moreRecommendationsIndex += 1;
            }
            this.getAbout(args);

        },

        getCRIDSearch: function(dataCallback, searchType, programme){
            try{
                var searchCRID,
                    crids = programme.CRIDS,
                    len = crids.length,
                    args;
                
                for(var i = 0 ; i < len; i++){
                    if(String(crids[i]).indexOf(searchType)>-1){
                        searchCRID = 'crid'+String(crids[i]).split('crid')[1];
                        break;
                    }
                }

                if (searchCRID) {
                    args = [dataCallback, searchCRID, searchType];
                    this.getAbout(args);
                } else {
                    dataCallback();
                }

            }catch(e){
                dataCallback();
            }
        },
        searchTimeout: function() {

            if (this.aboutTimeout) {
                clearTimeout(this.aboutTimeout);
            }
            this.abortiveSearch = true;
            freesat.oipfElement.mSearch.result.abort();
        },
        getAbout: function(args) {
            var dataCallback = args[0], CRID = args[1], type = args[2],
                self = this;

            console.log('getAbout '+ type + ' = ' + CRID); // freesat.
            if(this.killActiveSearch('killAll', this.getAbout,  args)){
                return;
            }
            freesat.searchActive = true;
            freesat.oipfElement.searchType = 'about';
            freesat.oipfElement.mSearch = freesat.oipfElement.searchManager.createSearch( 1 );
            if (type === '0x33') { // When searching a recommendations CRID you want to constrain to the Programmes they reference
                type = '0x31';
            }
            try{
                freesat.oipfElement.mSearch.addCRIDConstraint(CRID, parseInt(type, null));
            }catch(e){
                freesat.console.log('getAbout try '+ e);
                freesat.masterController.hideLoading();
                freesat.oipfElement.searchActive = false;
                dataCallback([]);
                return;
            }
            var now = Math.round(new Date().getTime()/1000.0);
            var gridStart = ((Math.floor(now/1800)) *1800);
            var startQuery = freesat.oipfElement.mSearch.createQuery("Programme.startTime", 3, String(gridStart));
            freesat.oipfElement.mSearch.setQuery(startQuery);
            freesat.oipfElement.searchManager.onMetadataSearch = this.getAboutResults(this, dataCallback);
            freesat.oipfElement.mSearch.result.getResults(0, 50);

            this.aboutTimeout = setTimeout(self.searchTimeout, 2000);
        },
//        0x31, crid://authority/programmeCridString
//        0x32, crid://authority/seriesCridString
//        0x33, crid://authority/recommedationCridString1
//        0x33, crid://authority/recommedationCridString2
//        0x33, crid://authority/recommedationCridString3
        getAboutResults: function(scope, dataCallback){
            return function(search, state){
                var i, arr = [], searchresultlength = search.result.length;

                if (scope.aboutTimeout) {
                    clearTimeout(scope.aboutTimeout);
                }

                console.log('getAboutResults: ' + state);
                freesat.oipfElement.mSearch = null;
                freesat.oipfElement.searchManager.onMetadataSearch = null;
                freesat.console.log('CRID search results '+searchresultlength);
                var myState = state;
                if(scope.abortiveSearch)myState=3;
                switch(myState){
                    case 0:      //complete
                        for(i = 0; i < searchresultlength; i++){
                            arr.push(search.result[i]);
                        }
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        dataCallback(arr);
                        break;
                    case 3:     //aborted
                        scope.abortiveSearch = false;
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        dataCallback([]); // Return an empty array
//                        if(scope.newDataCall){
//                            console.log('');
//                            scope.newDataCall(scope.newDataCallArgs);
//                        }
                        break;
                    case 4:     //insuficient resources
                        //todo retry ?
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        break;

                }
            };
        },

        searchProgrammes: function(svcIdOrCcid, startTime, endTime, maxResults, cb) {
            var channelForProgramme = freesat.channelData.getChannelObjectByCcidOrServiceID(svcIdOrCcid);
            if (!channelForProgramme) {
                cb(null);
                return;
            }

            freesat.oipfElement.searchActive = true;
            freesat.oipfElement.searchType = 'later';
            freesat.oipfElement.mSearch = freesat.oipfElement.searchManager.createSearch( 1 );
            freesat.oipfElement.mSearch.addChannelConstraint(channelForProgramme);

            var searchStart = startTime;
            if (!searchStart) {
                var searchdate = new Date().getTime();
                searchStart = Math.round(searchdate / 1000);
            }

            var searchEnd = searchStart + 3600; // add an hour by default
            if (endTime) {
                searchEnd = searchStart + endTime;
            }

            var startQuery = freesat.oipfElement.mSearch.createQuery("Programme.startTime", 3, String(searchStart));
            var endQuery = freesat.oipfElement.mSearch.createQuery("Programme.startTime", 4, String(searchEnd));
            var query = startQuery.and(endQuery);

            freesat.oipfElement.mSearch.setQuery(query);
        
            freesat.oipfElement.searchManager.onMetadataSearch = function(search, state) {
                freesat.oipfElement.mSearch = null;
                cb(search.result);
            }
            freesat.oipfElement.mSearch.result.getResults(0, maxResults ? maxResults : 1);
        },

        getLaterData: function(numberOfChannels, channelIndex, startday, starthour, numberofhours, dataCallback) {
            this.later.getLaterData(numberOfChannels, channelIndex, startday, starthour, numberofhours, dataCallback);
        },
        getLater3rdDimension: function(starthour, numberofhours){
            return Math.floor(starthour / 2);
        },

        killActiveSearch: function(searchType, newSearch, args){
            freesat.console.log('killActiveSearch '+ 'freesat.oipfElement.searchManager.onMetadataSearch');
            if(freesat.oipfElement.searchManager.onMetadataSearch && freesat.oipfElement.searchActive && freesat.oipfElement.mSearch && freesat.oipfElement.mSearch.result){
                if(freesat.oipfElement.searchType !== searchType){
                    if(freesat.oipfElement.searchType === 'nownext'){
                        freesat.oipfElement.nowNextTTL = 0;
                    }
                    freesat.oipfElement.mSearch.result.abort();
                    this.newDataCall = newSearch;
                    this.newDataCallArgs = args;
                    this.abortiveSearch = true;
                    freesat.console.log('killActiveSearch true');
                    return true;
                }
            }
            return false;
        },
        voidAllSearchs: function(){
            this.abortiveSearch = true;
            this.newDataCall = null;
            this.newDataCallArgs = null;
            freesat.masterController.hideLoading();
            freesat.console.log('VOID SEARCHs');
            this.killActiveSearch('everySearch', null, null);
        },

        ////////////////
        // SEARCH     //
        ////////////////
        search: function(keywords, direction, dataCallback) {
            var arr = [keywords, direction, dataCallback];

            var isIpAvailable = freesat.ipavailable;
            if (isIpAvailable && (window.localStorage.getItem('FORCE_DSMCC') === 'true')) {
                freesat.console.log('WARNING - setting IP to unavailable because FORCE_DSMCC is true');
                isIpAvailable = false;
            }

            if(isIpAvailable){
                this.ipsearch(keywords, direction, dataCallback);
            }else{
                this.oipfsearch(arr);
            }
        },
        oipfsearch: function(args) {
            var keywords = args[0], direction = args[1], dataCallback = args[2];
            if(this.killActiveSearch('killall', this.oipfsearch, args)){
                return;
            }
            freesat.oipfElement.searchType = 'search';
            freesat.oipfElement.searchActive = true;
            freesat.masterController.showLoading();
            freesat.oipfElement.mSearch = freesat.oipfElement.searchManager.createSearch( 1 );
            var wordQuery = freesat.oipfElement.mSearch.createQuery('Programme.name', 6, keywords);
            var startQuery = freesat.oipfElement.mSearch.createQuery("Programme.startTime", 3, String(new Date().getTime() / 1000));
            var query = startQuery.and(wordQuery);
            freesat.oipfElement.mSearch.setQuery(query);

//            freesat.oipfElement.mSearch.setQuery(wordQuery);
            freesat.oipfElement.searchManager.onMetadataSearch = this.getSearchResults(this, dataCallback);
            freesat.oipfElement.mSearch.result.getResults(0, 50);
            freesat.console.log('[dtt_data] oipfSearch keywords:'+keywords);
        },
        getSearchResults: function(scope, dataCallback){
                return function(search, state){
                    freesat.console.log('[dtt_data] getSearchResults state : '+state);

                    freesat.oipfElement.mSearch = null;
                    freesat.oipfElement.searchManager.onMetadataSearch = null;
                    var myState = state;
                    if(scope.abortiveSearch)myState=3;
                    switch(myState){
                        case 0:      //complete
//                            TODO: JSON
//                            var results = JSON.parse(search.result.resultsToJSON());
                            freesat.masterController.hideLoading();
                            freesat.oipfElement.searchActive = false;
//                            freesat.console.log('[dtt_data] getSearchResults:'+results.length);
                            dataCallback(search.result);
//                            TODO: JSON
//                            dataCallback(results);
                            break;
                        case 3:     //aborted
                            freesat.console.log("search aborted - state = 3");
                            freesat.masterController.hideLoading();
                            freesat.oipfElement.searchActive = false;
                            scope.abortiveSearch = false;
                            if(scope.newDataCall){
                                scope.newDataCall(scope.newDataCallArgs);
                            }
                            break;
                        case 4:     //insuficient resources
                            freesat.console.log("insufficient resources - state = 4");
                            //todo retry ?
                            freesat.masterController.hideLoading();
                            freesat.oipfElement.searchActive = false;
                            break;

                    }
                };
        },
        ipsearch: function(keywords, direction, dataCallback) {
            var url = this.checkUrlForRegional(freesat.localConfig.auth.epgSearch) + '/search/' + this.OUTPUT_FORMAT + '/all/' + encodeURIComponent(keywords),
                callback;
            url = this.addRegionalValues(url);

            callback = {
                success: this.onIPSearchSuccess(this),
                failure: this.onIPSearchFailure(this),
                returnCallback: dataCallback
            };
            this.request(url, callback, this, freesat.localConfig.auth.epgSearchTimeoutSecs);
        },
        onIPSearchSuccess: function(scope){
            return function(dataCallback) {
                freesat.masterController.hideLoading();
                var data = JSON.parse(scope.client.responseText);
//                var result = dataCallback(data); //scope.parseSearchResults(data);
                dataCallback(data);
            };
        },
        onIPSearchFailure: function(scope){
            return function(statusCode, dataCallback) {
                freesat.masterController.hideLoading();
                dataCallback(null);
            };
        },


        ////////////////
        // Showcase   //
        ////////////////TODO Where's the real feed ? ? ?
        getShowcaseData : function (callback){
            this.search('eas','forward',callback);
        },

        ////////////////
        // Parsing    //
        ////////////////

        parseSearchResults: function(data) {
            var events = data.event, arr = [], len;

            len = events.length;
            freesat.console.log('ev len : ' + len);

            for (var i=0; i < len; i++){
                arr.push(this.parseIPprogramme(events[i]));
            }
            return arr;
        },
        parseIPprogramme: function(prog){
            var resObject = new freesat.models.vo.IpProgramObject();
            resObject.name = prog.name;
            resObject.channelID = prog.evtId;
            resObject.startTime = prog.startTime;
            resObject.duration = prog.duration;
            resObject.description = prog.description;
            resObject.image = prog.image;
            resObject.genre = prog.genre;
            resObject.episodes = prog.episodeNo;
            return resObject;
        },
        returnNoDataProgramme: function(ccid, startday, starthour, numberofhours){
            var resObject = new freesat.models.vo.IpProgramObject();
            resObject.name = 'No data available';
            resObject.channelID = ccid;
            freesat.console.log('_____ ');
            if(typeof startday !== "undefined"){
                var date;

                try {
                    if (freesat.behind.gridStartTime) {
                        resObject.startTime = freesat.behind.gridStartTime;
                    }
                } catch (e) {
                    freesat.console.log('NO BEHIND gridStartTime ' + e);
                    date = new Date();
                    resObject.startTime = Math.round(new Date(date.getTime() + (startday * 86400) + (starthour * 3600)).getTime() / 1000);

                }

                resObject.duration = 9000; // 2 and half hours // numberofhours * 3600; //
                freesat.console.log('_____ new mbj '+ resObject.startTime +' @@@ '+ resObject.duration);
            }
            return resObject;
        },

        parseTSTV: function(tstv) {
            var resObject = new freesat.models.vo.TsTvObject();

            resObject.availabilityStart = tstv.availabilityStart;
            resObject.availabilityEnd = tstv.availabilityEnd;
            resObject.mediaAvailabilty = tstv.mediaAvailable;
            resObject.url = tstv.mediaLocation;
            //@todo - remove?
            //update this to allow for more than one TStv object per event.
            //resObject.model = tstv.model;

            return resObject;
        },

        checkUrlForRegional: function(url) {

            if (url.indexOf('regional') < 0) {
                url += '/regional';
            }
            return url;
        },

        addRegionalValues: function(url) {
            var regions = freesat.localConfig.config.regions,
                len, i, pr_id, sr_name;

            if (Array.isArray(regions)) {
                len = regions.length;

                for (i=0; i<len; i++) {
                    pr_id = regions[i].pr_id;

                    if (pr_id) { // DSAT : /<bat_id>/<region_id>
                        url += '/' + pr_id + '/' + regions[i].sr_id;
                    } else { // DTT : /<Region1_Primary_Region]/[Region1_Secondary]/{optional <Region2_Primary_Region]/[Region2_Secondary]....}
                        sr_name = regions[i].sr_name || 'null';
                        url += '/' + regions[i].pr_name + '/' + sr_name;
                    }
                }
            }

            console.log('addRegionalValues: ' + url);

            return url;

        }
    }
);

/* global IDataSource:true, datainterface:true, Interface:true */

/**
 * @name dataStore
 * @namespace
 * @memberof freesat.data
 * @class
 */
accedo.Class.create('freesat.data.dataStore', [], {}, {
    /**
     * credentials
     * @private
     */
    credentials : [
        {
            id: 'ip',
            username: 'accedo',
            password: 'acc3d0!27'
        },
        {
            id: 'fake',
            username: 'accedo',
            password: 'acc3d0!27'
        },
        {
            id: 'dtt',
            username: 'accedo',
            password: 'acc3d0!27'
        },
        {
            id: 'dsat',
            username: 'accedo',
            password: 'acc3d0!27'
        }
    ],
    postcode : null,
    bat : null,

    /**
     * allData
     * @public
     * @structure nowNextAll / laterAll / earlierAll / earlierImageAll /
     */
    allData: {earlierAll: {}, nowNextAll: [], laterAll: [], laterData: [], nowNextGenre: []},
    OnDemand:null,

    setup: function(){
        freesat.console.log('[dataStore] setup');
        for(var ch=0; ch < freesat.channelData.channelListLength; ch++){
            this.allData.nowNextAll.push([]);
//            this.allData.nowNextGenre.push([]);
            this.allData.laterAll.push([]);      // array of channels with 8 days each having 12 cols/pages
            this.allData.laterData.push([]);      // array of channels with 8 days each having 12 cols/pages
            for (var day = 0; day < 12; day++){
                this.allData.laterAll[ch].push([]);
                this.allData.laterData[ch].push([]);
                for (var col = 0; col < 12; col++){
                    this.allData.laterAll[ch][day].push([]);
                    this.allData.laterData[ch][day].push(0);
                }
            }
        }

    },
    resetNowNextAll: function(){
        this.allData.nowNextAll = [];
        for(var ch=0; ch < freesat.channelData.channelListLength; ch++){
            this.allData.nowNextAll.push([]);
        }
    },
    setLaterGenre: function() {
        var len = freesat.channelData.genreList.length,
            i, j, k;

        this.allData.laterGenre = [];
        this.allData.laterGenreData = [];

        for (i = 0; i < len; i++) {
            this.allData.laterGenre.push([]);
            this.allData.laterGenreData.push([]);

            for (j = 0; j < 8; j++) {
                this.allData.laterGenre[i].push([]);
                this.allData.laterGenreData[i].push([]);

                for (k = 0; k < 12; k++) {
                    this.allData.laterGenre[i][j].push([]);
                    this.allData.laterGenreData[i][j].push(0);
                }
            }
        }
    },
    /**
     * getCredentials
     * @returns {object}
     */
    getCredentials: function () {
        for (var _i = 0; _i < this.credentials.length; _i++) {
            if (this.credentials[_i].id == freesat.datamode) {
                return this.credentials[_i];
            }
        }
        return null;
    },
    /**
     * getInterface
     * @returns {object} data interface | accedo.device.AbstractInterface
     */
    getInterface: function() {
        if(freesat.dataStore.dataLayer) return freesat.dataStore.dataLayer;
        var datainterface = freesat.dataInterfaceFactory.getInterface();
//        Interface.ensureImplements(datainterface, IDataSource);
        freesat.dataStore.dataLayer = datainterface;
        return datainterface;
    },
    /**
     * setCredentials
     * @param datatype
     * @param username
     * @param password
     */
    setCredentials: function(datatype, username, password) {
        if (this.getCredentials(datatype) === null || this.getCredentials(datatype) === undefined) {
            this.credentials.push({
                id: datatype,
                username: username,
                password: password
            });
        }
    }
});

/**
 * User: Dean Clancy
 * Date: 22/01/14
 * Time: 09:52
 */
/*global accedo: false */
/**
 * Creates a new Ondemand instance.
 * @name Ondemand
 * @memberof freesat.Ondemand
 * @class A Ondemand class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.data.datafunc.Ondemand", ["freesat.models.GetLocalFile"], {},
    {
        isIpAvailable: function() {
freesat.console.log('FORCE_DSMCC=' + window.localStorage.getItem('FORCE_DSMCC'));
            if (freesat.ipavailable && (window.localStorage.getItem('FORCE_DSMCC') === 'true')) {
                freesat.console.log('WARNING - forcing fetch using DSM-CC');
                return false;
            }
freesat.console.log('isIpAvailable returning: ' + freesat.ipavailable);
            return freesat.ipavailable;
        },
        getOnDemand: function (dataCallback) {
            try{
                var url = freesat.localConfig.auth.onDemand + '/regional/od/json',
                    regArr = freesat.localConfig.config.regions,
                    i, len = regArr.length,
                    prId, sr_name;

                for(i=0; i<len; i++){
                    prId = regArr[i].pr_id;
                    if (prId) {
                        url += '/' + prId + '/' + regArr[i].sr_id;
                    } else {
                        sr_name = regArr[i].sr_name || 'null';
                        url += '/' + regArr[i].pr_name + '/' + sr_name;
                    }
                }

                freesat.console.log('ondemand' + url);

                this.request(url, dataCallback, this);
            }catch(e){
                freesat.am.techEvent('/onDemand', {
                    category: 'exception',
                    action: 'applicationException',
                    description: 'Got following exception for request ' + url + ': ' + e
                });

                dataCallback();
            }
        },
        request: function(url, callback, scope) {
            freesat.console.log('request' + url);
            freesat.masterController.showLoading();

            var remoteGetter = new freesat.models.GetRemoteFile();
            remoteGetter.getAsync(
                url,
                function(result) {
                    freesat.masterController.hideLoading();
                    scope.data = JSON.parse(result);
                    console.log(JSON.stringify(scope.data, null, 4));
                    callback(scope.data);
                },
                function(statusCode) {
                    freesat.am.techEvent('/onDemand', {
                        category: 'exception',
                        action: 'applicationException',
                        description: 'Got status code ' + statusCode + ' for request ' + url
                    });

                    freesat.masterController.hideLoading();
                    freesat.console.log('Failed to retrieve file');
                    freesat.behind.noip_onload = true;
                    freesat.behind.showError();

                    callback(null);
                },
                scope.username,
                scope.password
            );
        },

        getOnDemandUsingDSMCC: function (dataCallback) {
            freesat.console.log('[DSMCC] Fetching On Demand data');
            var scope = this;

            freesat.dsmcc.getRegionalisedFile('od.json', function(result) {
                var data = scope.parseDsmccData(result);
                if (data) {
                    freesat.console.log('Got fresh On Demand data from DSM-CC carousel - no need to tune to home transponder');
                    dataCallback(data);
                    return;
                }

                // try tuning to home transport stream to acquire carousel
                freesat.behind.setFetchingDsmcc();
                freesat.video.hide();

                freesat.console.log('No fresh On Demand data available from DSM-CC - tuning to home transponder in order to attempt to acquire it...');
                freesat.dsmcc.waitForCarouselUpdateOfFile(
                    'od.json',
                    function(result) {
                        freesat.video.show();
                        if (result) {
                            jsonResult = JSON.parse(result);
                            freesat.console.log('Successfully got On Demand data from DSM-CC carousel');
                            freesat.behind.hideError();
                            dataCallback(jsonResult);
                        } else {
                            freesat.console.log('Failed to get On Demand data from DSM-CC carousel');
                            freesat.behind.setError();
                        }
                    });
            });
        },
        setDsmccFetchCallback: function(callback) {
            this.dsmccFetchCallback = callback;
        },
        parseDsmccData: function(data) {
            try {
                var parsedData = JSON.parse(data);
                return parsedData;
            }
            catch (e) {
                // ignore
            }
            return null;
        }
    }
);
/**
 * User: Dean Clancy
 * Date: 28/01/14
 * Time: 11:21
 */
/*global accedo: false */
/**
 * Creates a new Earlier instance.
 * @name Earlier
 * @memberof freesat.Earlier
 * @class A Earlier class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.data.datafunc.Earlier", [], {},
    {
        client:null,
        data: null,
        fsatId: null,
        dataCallback: null,
        channel: null,

        isIpAvailable: function() {
            return freesat.ipavailable;
        },

        resetChannel: function() {
            this.fsatId = null;
            this.dataCallback = null;
            this.channel = null;
        },

        getEarlierData: function(channel, startday, starthour, numberofhours, dataCallback){
            var fsatId = channel.freesatServiceID;
            freesat.behind.earlierChannel = channel;

            this.resetChannel();

            if(this.checkEarlierData(startday, fsatId, dataCallback, channel)) {
                return;
            }

            this.fsatId = fsatId;
            this.dataCallback = dataCallback;
            this.channel = channel;
            window.earlierScope = this;

            freesat.masterController.showLoading();
            if (this.isIpAvailable()) {
                var callback = {
                    success: this.earlierSuccess(this, this.getEarlierCUTV, channel),
                    failure: this.getEarlierCUTV
                };

                this.request(
                    freesat.localConfig.auth.epg + '/epg/' + fsatId + '/cutv_now.json',
                    callback,
                    this);
            } else {
                freesat.masterController.hideLoading();
                dataCallback(false);
            }
        },

        /**
         * Get the cutv.json
         */
        getEarlierCUTV: function() {
            var scope = window.earlierScope,
                url = freesat.localConfig.auth.epg + '/epg/' + scope.fsatId + '/cutv.json';

            var failure = function () {
                freesat.masterController.hideLoading();

                if (freesat.dataStore.allData.earlierAll[scope.fsatId].length > 0) {
                    scope.dataCallback(scope.channel);
                } else {
                    scope.dataCallback(false);
                }
            };
            var callback = {
                success: scope.earlierSuccess(scope, scope.dataCallback, scope.channel),
                failure: failure
            };
            freesat.masterController.showLoading();

            if (scope.isIpAvailable()) {
                scope.request(url, callback, scope);
            } else {
                failure();
            }
        },

        checkEarlierData: function(startDay, fsatSID, dataCallback, channel){
            var nowDate = new Date().getTime() / 1000;
            if(startDay < 0) startDay *= -1;
            try{
                var eData = freesat.dataStore.allData.earlierAll[fsatSID][startDay];
                if(eData){
                    if(eData.RefreshDT > nowDate){
                        dataCallback(channel);
                        return true;
                    }else if(eData.RefreshDT < nowDate && eData.expiryDT > nowDate){
                        dataCallback(channel);
                        return false;
                    }else if(eData.expiryDT < nowDate){
                        freesat.dataStore.allData.earlierAll[fsatSID][startDay] = null;
                        return false;
                    }
                }else{
                    freesat.dataStore.allData.earlierAll[fsatSID] = [null]; // , null
                    return false;
                }
            }catch(e){
                freesat.dataStore.allData.earlierAll[fsatSID] = [null]; // , null
                return false;
            }


        },
        earlierSuccess: function(scope, dataCallback, channel){
            return function(){
                console.log('earlierSuccess');
                freesat.masterController.hideLoading();
                var data = JSON.parse(scope.responseText);
                if(scope.parseEarlierObject(data, channel, dataCallback))
                    dataCallback(channel);
            };
        },
        request: function(url, callback, scope) {
            freesat.console.log('requesting ' + url);
            freesat.masterController.showLoading();
            scope.responseText = '';
            scope.client = new XMLHttpRequest();
            scope.client.onreadystatechange = function() {
                if (scope.client.readyState == XMLHttpRequest.DONE) {
                    if (scope.client.status === 200) {
                        scope.responseText = scope.client.responseText;
                        scope.data = callback.success(callback.returnCallback);
                    } else {
                        scope.data = callback.failure(scope.client.status);
                    }
                }
            };

            try {
                scope.client.open('get', url, true, scope.username, scope.password);
                scope.client.send();
            } catch (Error) {
                freesat.console.log(Error.message);
            }
        },
        failure: function() {
            freesat.masterController.hideLoading();
            freesat.console.log('Failed to retrieve earlier file');
        },
        parseEarlierObject: function(earlierObj, channel, dataCallback) {
            var page, i, fsatSID = channel.freesatServiceID,
                epgPages = earlierObj.epgPages,
                version = epgPages.version,
                UpdateDT = version.updateDT,
                RefreshDT = version.refreshDT,
                expiryDT = version.expiryDT,
                pages = epgPages.pages.page,
                num = epgPages.pages.num,
                len,
                obj,
                index,
                earlier_object;

            if (num > 0) { // Check it's not an incomplete JSON file
                len = pages.length;

                if (len) {

                    for (i = 0; i < len; i++) {
                        freesat.console.log('i : ' +i);
                        page = pages[i];
                        obj = this.parsePage(page, channel, UpdateDT, RefreshDT, expiryDT);
                        if(obj){
                            freesat.console.log('@@@@@@@@@ : ' + fsatSID + ' : ' + (obj.id-1) + ' : ' +  obj.programmes.length);

                            index = obj.id-1;

                            earlier_object = freesat.dataStore.allData.earlierAll[fsatSID][index];

                            if (earlier_object !== null && earlier_object !== undefined) { // Because the id is not continuing from cutv_now

                                if (earlier_object['date'] !== obj.date) { // If it's not null or undefined, check the dates don't match
                                    index = obj.id;
                                }
                            }
                            freesat.dataStore.allData.earlierAll[fsatSID][index] = obj;

                        }else{
                            freesat.masterController.hideLoading();
                            dataCallback(false);
                            return false;
                        }
                    }
                    return true;
                } else {
                    freesat.console.log('parsePage');
                    obj = this.parsePage(pages, channel, UpdateDT, RefreshDT, expiryDT);
                    if(obj){
                        freesat.dataStore.allData.earlierAll[fsatSID][obj.id-1] = obj;
                        return true;
                    }else{
                        freesat.masterController.hideLoading();
                        dataCallback(false);
                        return false;
                    }
                }
            } else {
                freesat.masterController.hideLoading();

                // Check for valid cutv_now data
                if (freesat.dataStore.allData.earlierAll[channel.freesatServiceID][0]) {
                    dataCallback(channel);
                } else {
                    dataCallback(false);
                }
                return false;
            }
        },

        parsePage: function(page, channel, UpdateDT, RefreshDT, expiryDT){
            try{
                var resObject = new freesat.models.vo.EarlierContainer();
                resObject.id = parseInt(page.id, null);
                resObject.date= page.date;
                resObject.heading_title= page.heading_title;
                resObject.forwardtext= page.forwardtext;
                resObject.backwardtext= page.backwardtext;
                resObject.initialItem = page.itemlist.initalItem;
                resObject.UpdateDT = UpdateDT;
                resObject.RefreshDT = RefreshDT;
                resObject.expiryDT = expiryDT;
                var nowTime = new Date().getTime() / 1000;
                if(expiryDT > nowTime){
                    this.parsePromo(resObject, page.promo, channel.ccid);
                    this.parseProgramme(resObject, page.itemlist, channel.ccid);
                    return resObject;
                }else{
                    return false;
                }
            }catch(e){
                freesat.console.log('parsePageError : '+ e);
                return false;
            }
        },
        parseProgramme: function(resObject, itemlist, channelccid){
            var i, n, arr, links, item = itemlist.item;
            for (i=0; i<item.length; i++){
                var pItem = new freesat.models.vo.EarlierItem();
                pItem.channelID = channelccid;
                pItem.id = parseInt(item[i].id, null);
                pItem.available = item[i].available;
                pItem.lock = item[i].lock;
                pItem.eventId = item[i].eventId;
                pItem.name = item[i].Name;
                pItem.duration = parseInt(item[i].DurationInSeconds, null);
                pItem.AvailabilityText= item[i].AvailabilityText;//: ;//Available until 18 Dec;//,
                pItem.StartText = item[i].StartText;//: ;//16:30;//,
                pItem.StatusText = item[i].StatusText;//: ;//Available soon;//,
                pItem.ShortSynopsis = item[i].ShortSynopsis;
                pItem.description = item[i].ShortSynopsis;
                pItem.image = item[i].image;
                pItem.icons = item[i].icons;

                links = item[i].links;
                if (links && links.content_link && links.content_link.length) {
                    for(n = 0; n < links.content_link.length; n++){
                        if (links.content_link[n].type === "CTV" ||
                            links.content_link[n].type === "HTML" ||
                            links.content_link[n].type === "MHEG" ||
                            links.content_link[n].type === "HBBTV") {
                            pItem.linkType = links.content_link[n].type;
                            pItem.linkText = links.content_link[n].text;
                            pItem.linkServiceId = links.content_link[n].serviceid;
                            break;
                        }
                    }
                }
                if (pItem.linkType && pItem.linkText) {
                    pItem.deepLink = "native://freetime/" + pItem.linkType.toLowerCase() + "/[%]" + freesat.localConfig.auth.ipchannelsmgr + pItem.linkText + "[%]";
                    pItem.deepLinkType = pItem.linkType;
                    pItem.deepLinkServiceId = pItem.linkServiceId;
                }
                resObject.programmes.push(pItem);
            }
            return resObject;
        },
        parsePromo: function(resObject, promo, channelccid){
            var i, n, arr, links, items, promoItem = promo.promoitem;

            items = parseInt(promo.promocount, 10);

            if (items > 0) {

                if (items === 1) {
                    promoItem = [promoItem];
                }

                for (i=0; i<promoItem.length; i++){
                    var pItem = new freesat.models.vo.EarlierItem();
                    pItem.channelID = channelccid;
                    pItem.id = parseInt(promoItem[i].id, null);
                    pItem.available = promoItem[i].available;
                    pItem.lock = promoItem[i].lock;
                    pItem.eventId = promoItem[i].eventId;
                    pItem.name = promoItem[i].Name;
                    pItem.duration = parseInt(promoItem[i].DurationInSeconds, null);
                    pItem.AvailabilityText= promoItem[i].AvailabilityText;//: ;//Available until 18 Dec;//,
                    pItem.StartText = promoItem[i].StartText;//: ;//16:30;//,
                    pItem.StatusText = promoItem[i].StatusText;//: ;//Available soon;//,
                    pItem.ShortSynopsis = promoItem[i].ShortSynopsis;
                    pItem.description = promoItem[i].ShortSynopsis;
                    pItem.image = promoItem[i].image;
                    pItem.icons = promoItem[i].icons;
                    pItem.isPromo = true;
                    links = promoItem[i].links;
                    if (links && links.content_link && links.content_link.length) {
                        for(n = 0; n < links.content_link.length; n++){
                            if (links.content_link[n].type === "CTV" ||
                                links.content_link[n].type === "HTML" ||
                                links.content_link[n].type === "MHEG" ||
                                links.content_link[n].type === "HBBTV") {
                                pItem.linkType = links.content_link[n].type;
                                pItem.linkText = links.content_link[n].text;
                                break;
                            }
                        }
                    }
                    if (pItem.linkType && pItem.linkText) {
                        pItem.deepLink = "native://freetime/" + pItem.linkType.toLowerCase() + "/[%]" + freesat.localConfig.auth.ipchannelsmgr + pItem.linkText + "[%]";
                    }
                    resObject.promos.push(pItem);
                }
                return resObject;
            }
        }
    }
);
/**
 * User: Dean Clancy
 * Date: 28/02/14
 * Time: 10:28
 */
/*global accedo: false */
/**
 * Creates a new Later instance.
 * @name Later
 * @memberof freesat.Later
 * @class A Later class, functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.data.datafunc.Later", [], {},
    {
        laterTimeout: null,
        callback: null,
        getLaterData: function(numberOfChannels, channelIndex, startday, starthour, numberofhours, dataCallback) {
            freesat.behind.dataInterface.newDataCallArgs = [numberOfChannels, channelIndex, startday, starthour, numberofhours, this] ;
            freesat.behind.dataInterface.dataCallback = dataCallback;
            freesat.masterController.showLoading();
            freesat.oipfElement.searchActive = true;
            this.getLaterPage(freesat.behind.dataInterface.newDataCallArgs);
        },
        getLaterPage: function(args) {
            var numberOfChannels = args[0], channelIndex = args[1], startday=args[2], starthour=args[3], numberofhours=args[4], scope=args[5];
            if(scope.needLaterData(numberOfChannels, channelIndex, startday, starthour, numberofhours, scope)){
                freesat.console.log('getLaterPage queryLater');
                scope.queryLater(freesat.behind.dataInterface.newDataCallArgs);
            }else{
                freesat.masterController.hideLoading();
                if(freesat.behind.dataInterface.dataCallback) freesat.behind.dataInterface.dataCallback(channelIndex, startday, starthour, numberofhours);
                freesat.behind.dataInterface.dataCallback = null;
                freesat.oipfElement.searchManager.onMetadataUpdate = null;
                this.callback = null;
            }
        },
        requeryLater: function(args){
            return function(){
                if(freesat.overlay && freesat.overlay.buttons){
                    return;
                }

                var scope = args[5];

                if (freesat.getZone() === 'later') {
                    freesat.console.log('onMetaDataUpdate requeryLater - channelIndex: ' + args[1] + ', channels: ' + args[0] + ', day: ' + args[2]);
                    freesat.behind.dataInterface.dataCallback = scope.callback;
                    freesat.behind.laterUpdate = true;

                    try {
                        scope.queryLater(args);
                    } catch (e) {
                        freesat.console.log('CANNOT REQUERY LATER: ' + e);
                        freesat.behind.dataInterface.dataCallback = null;
                        freesat.behind.laterUpdate = false;
                    }
                } else {
                    console.log('requeryLater not in Later');
                }
            };
        },
        queryLater: function(args){
            var numberOfChannels = args[0], channelIndex = args[1], startday=args[2], starthour=args[3], numberofhours=args[4], scope=args[5], channelList = freesat.channelData.channelList,
                channelListLength = freesat.channelData.channelListLength, date, searchdate, endDate, channelIDIndex, i, n, startQuery, endQuery, query;

            freesat.console.log('queryLater');
            if (freesat.behind.dataInterface.killActiveSearch('killAll', this.queryLater, args)) {
                return;
            }
            freesat.oipfElement.searchActive = true;
            freesat.oipfElement.searchType = 'later';
            try {
                searchdate = freesat.behind.gridStartTime;
            } catch(e) {
                freesat.console.log('CATCH ERROR: ' + e);
                date = new Date();
                date.setHours(starthour);
                searchdate = date.getTime() + (startday * 86400000);
                searchdate = Math.round(searchdate / 1000);
            }
            endDate = searchdate + (numberofhours * 3600) + 3600; // Add an extra hour
            channelIDIndex = [];
            freesat.oipfElement.mSearch = freesat.oipfElement.searchManager.createSearch( 1 );
            freesat.console.log('-*- queryLater adding channelConstraint for ' + numberOfChannels + ' channels');
            try {
                if (freesat.behind.isGenreList) {
                    channelList = freesat.channelData.genreList;
                    channelListLength = channelList.length;
                }
            } catch (e) {
                freesat.console.log('No freesat.behind:' + e);
            }

            for (i=0; i<numberOfChannels; i++) {
                n = (channelIndex + i < channelListLength) ? channelIndex + i : (channelIndex + i) - channelListLength;
                freesat.oipfElement.mSearch.addChannelConstraint(channelList[n]);
                channelIDIndex.push(channelList[n].ccid.split(':')[1]);
                freesat.console.log('-*- queryLater added channelConstraint for '+channelIDIndex[channelIDIndex.length-1]);
            }
            startQuery = freesat.oipfElement.mSearch.createQuery("Programme.startTime", 3, String(searchdate));
            endQuery = freesat.oipfElement.mSearch.createQuery("Programme.startTime", 4, String(endDate));

            freesat.console.log('=||= queryLater: ' + freesat.getEasyTime(searchdate) + ' : ' + freesat.getEasyTime(endDate) + ' =||=');

            query = startQuery.and(endQuery);

            freesat.oipfElement.mSearch.setQuery(query);

            this.callback = freesat.behind.dataInterface.dataCallback;
            freesat.oipfElement.searchManager.onMetadataUpdate = this.requeryLater(args);

            freesat.oipfElement.searchManager.onMetadataSearch = this.queryLaterResults(this, numberOfChannels, channelIDIndex, channelIndex, startday, starthour, numberofhours);
            freesat.oipfElement.mSearch.result.getResults(0, 100);
            freesat.console.log('-*- queryLater search called get results -*- startTime'+searchdate+ '  endTime '+endDate);
            scope.laterTimeout = setInterval(this.queryLaterResults(scope, numberOfChannels, channelIDIndex, channelIndex, startday, starthour, numberofhours), 3000);
        },
        queryLaterResults: function(scope, numberOfChannels, channelIDIndex, channelIndex, startday, starthour, numberofhours){
            return function(search, state){
                freesat.console.log('queryLaterResults');
                var channelListLength = freesat.channelData.channelListLength,
                    laterArray = freesat.dataStore.allData.laterAll,
                    laterData = freesat.dataStore.allData.laterData,
                    i, arr = [], lcn, searchresultlength = 0, myState, chIndex,
                    indexOfLCN, len;

                clearInterval(scope.laterTimeout);

                if (state === undefined) {
                    state = 3;
                }

                try {
                    if (freesat.behind.isGenreList) {
                        laterArray = freesat.dataStore.allData.laterGenre;
                        laterData = freesat.dataStore.allData.laterGenreData;
                    }
                } catch (e) {
                    freesat.console.log('No freesat.behind:' + e);
                }

                try {
                    searchresultlength = search.result.length;
                    freesat.oipfElement.mSearch = null;
                    freesat.oipfElement.searchManager.onMetadataSearch = null;
                    for (i=0; i < numberOfChannels; i++) {
                        arr.push([]);
                    }
                } catch (e) {
                    freesat.console.log('[dtt_data] queryLaterResults '+e);
                }
                myState = state;
                if (freesat.behind.dataInterface.abortiveSearch) {
                    myState=3;
                }
                switch (myState) {
                    case 0:      //complete
                        searchresultlength = search.result.length;
                        for (i = 0; i < numberOfChannels; i++) {
                            try {
                                laterArray[channelIndex + i][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)]=[];
                                laterData[channelIndex + i][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)] = 0;
                            } catch (e) {
                                freesat.console.log('Empty channel response: ' + e);
                            }
                        }
                        for (i = 0; i < searchresultlength; i++) {
                            lcn = String(search.result[i].channelID).split(':')[1];
                            indexOfLCN = channelIndex+channelIDIndex.indexOf(lcn);
                            if (channelIDIndex.indexOf(lcn) !== -1) {
                                chIndex = (indexOfLCN >= channelListLength) ? indexOfLCN - channelListLength : indexOfLCN;
                                try{
                                    var dt = new Date(search.result[i].startTime*1000);
                                    laterArray[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)].push(search.result[i]);
                                    laterData[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)] = new Date().getTime() + 300000;
                                }catch(e){
                                    freesat.console.log('[dtt_data] queryLaterResults indexing fail '+ e);
                                }
                            }
                        }
                        len = channelIDIndex.length;
                        for(i=0; i < len; i++){
                            lcn = channelIDIndex[i];
                            indexOfLCN = channelIndex+channelIDIndex.indexOf(lcn);
                            freesat.console.log('no data for '+ lcn);
                            chIndex = (indexOfLCN >= channelListLength) ? indexOfLCN - channelListLength : indexOfLCN;
                            if(laterArray[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)]){
                                if(laterArray[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)].length <= 0){
                                    laterArray[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)].push(freesat.behind.dataInterface.returnNoDataProgramme(('ccid:'+lcn), startday, starthour, numberofhours));
                                    laterData[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)] = new Date().getTime() + 300000;
                                }
                            }else{
                                laterArray[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)].push(freesat.behind.dataInterface.returnNoDataProgramme(('ccid:'+lcn), startday, starthour, numberofhours));
                                laterData[chIndex][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)] = new Date().getTime() + 300000;
                            }
                        }
                        freesat.oipfElement.searchActive = false;
                        freesat.masterController.hideLoading();
                        if(freesat.behind.dataInterface.dataCallback) freesat.behind.dataInterface.dataCallback(channelIndex, startday, starthour, numberofhours);
                        freesat.behind.dataInterface.dataCallback = null;
                        break;
                    case 3:     //aborted
                        freesat.behind.dataInterface.abortiveSearch = false;
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        if(freesat.behind.dataInterface.newDataCall){
                            freesat.behind.dataInterface.newDataCall(freesat.behind.dataInterface.newDataCallArgs);
                        }
                        break;
                    case 4:     //insuficient resources
                        //todo retry ?
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        break;

                }
            };
        },
        //////////////////////
        //////////////////////
        // later time check //
        //////////////////////
        //////////////////////
        needLaterData: function(numberOfChannels, channelIndex, startday, starthour, numberofhours, scope) {
            var channelListLength = freesat.channelData.channelListLength, laterData = freesat.dataStore.allData.laterData, i, n;

            try {
                if (freesat.behind.isGenreList) {
                    channelListLength = freesat.channelData.genreList.length;
                    laterData = freesat.dataStore.allData.laterGenreData;
                }
            } catch (e) {
                freesat.console.log('No freesat.behind:' + e);
            }

            for (i=0; i<numberOfChannels; i++) {
                n = (channelIndex + i < channelListLength) ? channelIndex + i : (channelIndex + i)-channelListLength;
                freesat.console.log('[dtt_data] needLaterData '+ n+' F '+startday+' F '+freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours));
                if(laterData[n][startday][freesat.behind.dataInterface.getLater3rdDimension(starthour,numberofhours)] < new Date().getTime()){
                    return true;
                }
            }
            return false;

        }
    }
);
/**
 * User: Dean Clancy
 * Date: 29/01/14
 * Time: 09:11
 */
/*global accedo: false */
/**
 * Creates a new Nownnext instance.
 * @name Nownnext
 * @memberof freesat.Nownnext
 * @class A Nownnext class, bla functionality.
 * @author <a href="mailto:dean.clancy@accedo.tv">Dean Clancy</a>
 */
accedo.Class.create("freesat.data.datafunc.Nownext", [], {},
    {
        owner:null,

        collected:0,
        collectItem: 0,

        updateFunc: null,
        updateInterval: null,
        updateArray: [],

        retry: 0,
        retryData: [],
        retryFunc: null,
        retryInterval: null,

        nowNextData: [],
        nowNextFilteredData: [],

        dataCallback: null,

        filteredList: false,


        resetNowNextArray: function(){
            var i, len = freesat.channelData.channelListLength;
            this.collected = 0;

            for(i=0; i < len; i++){
                this.nowNextData.push(0);
            }
        },
        getStartTime: function() {
            return new Date().getHours();
        },
        getNowAndNextAll: function(dataCallback, topPositionIndex, callbackAt) {
            var channelListLength = (this.filteredList)?freesat.channelData.genreList.length:freesat.channelData.channelListLength;
            this.retryData = [];
            this.updateData = [];
            this.retry = 0;
            this.collected = 0;
            freesat.masterController.showLoading();
            this.dataCallback=dataCallback;
            var args = [this, topPositionIndex, callbackAt];
            if(callbackAt>=channelListLength) callbackAt = channelListLength;
            if(this.owner.killActiveSearch('killAll', this.findNowAndNextForPageChannels, args)){
                return;
            }
            this.findNowAndNextForPageChannels(args);
        },
        findNowAndNextForPageChannels: function(args){
            var scope = args[0], topPositionIndex = args[1], callbackAt = args[2];
            scope.newDataCallArgs = args;
            if(scope.getItemToCollect(topPositionIndex, callbackAt)){
                scope.findNowAndNextForChannel(scope);
            }
        },
        setNNData: function(index, arr){
            if(!this.filteredList){
                freesat.dataStore.allData.nowNextAll[index] = arr;
            }else{
                freesat.dataStore.allData.nowNextGenre[index] = arr;
            }
        },
        setNNDataItem: function(ind1, ind2, item){
            if(!this.filteredList){
                freesat.dataStore.allData.nowNextAll[ind1][ind2] = item;
            }else{
                freesat.dataStore.allData.nowNextGenre[ind1][ind2] = item;
            }
        },
        getNNDataItem: function(ind1, ind2){
            if(!this.filteredList){
                return freesat.dataStore.allData.nowNextAll[ind1][ind2];
            }else{
                return freesat.dataStore.allData.nowNextGenre[ind1][ind2];
            }
        },
        getItemToCollect: function(topPositionIndex, callbackAt){
            var startPosition = topPositionIndex, currentTime = new Date().getTime();
            this.collectItem = topPositionIndex + this.collected;
            if(this.collected===callbackAt) {
                this.doDataCallback();
            }
            if(this.collected>=(callbackAt*2)){
                this.collectItem = (topPositionIndex - (callbackAt*3))+this.collected;
            }
            var len = (this.filteredList)?freesat.channelData.genreList.length:freesat.channelData.channelListLength;
            if(this.collectItem < 0) this.collectItem += len;
            if(this.collectItem >= len){
                this.doDataCallback(topPositionIndex, callbackAt);
                return this.finishSearch(topPositionIndex, callbackAt);
            }
            var data = (this.filteredList)?this.nowNextFilteredData:this.nowNextData;
            while(data[this.collectItem] > currentTime){
                ++this.collected;
                if(this.collected===callbackAt) {
                    this.doDataCallback();
                    return this.finishSearch(topPositionIndex, callbackAt);
                }
                if(this.collected>=(callbackAt*2))startPosition = topPositionIndex - (callbackAt*3);
                if(startPosition<0)startPosition+=len;
                this.collectItem = startPosition + this.collected;
                if(this.collected === (callbackAt*3)){
                    return this.finishSearch(topPositionIndex, callbackAt);
                }
                if(this.collectItem >= len){
                    this.doDataCallback();
                    return this.finishSearch(topPositionIndex, callbackAt);
                }
            }
            return true;
        },
        doDataCallback: function(){
            if(this.dataCallback){
                this.dataCallback();
                this.dataCallback = null;
            }
        },
        finishSearch: function(topPositionIndex, callbackAt){
            freesat.oipfElement.searchActive = false;
            freesat.masterController.hideLoading();
            return false;
        },

        findNowAndNextForChannel: function(scope, channel, callback){
            try{
                freesat.oipfElement.searchActive = true;
                freesat.oipfElement.searchType = 'nownext';
                if (!channel){
                    channel = (this.filteredList)?freesat.channelData.genreList[this.collectItem]:freesat.channelData.channelList[this.collectItem];
                }

                freesat.oipfElement.mSearch = freesat.oipfElement.searchManager.createSearch(1);
                freesat.oipfElement.mSearch.findProgrammesFromStream(channel, null);
                freesat.oipfElement.searchManager.onMetadataSearch = scope.findNowAndNextResults(scope, channel.ccid, callback);
                trace('XDK.getResults'+channel.ccid, true);
                freesat.oipfElement.mSearch.result.getResults(0, 2);
            }catch(e){
                freesat.masterController.hideLoading();
                freesat.oipfElement.searchActive = false;
            }
        },
        findNowAndNextResults: function(scope, ccid, callback){
            return function(search, state){
                trace('XDK.getResults'+ccid);
                freesat.oipfElement.searchManager.onMetadataSearch = null;
                var i, arr = [], searchresultlength, myState = state;
                if(scope.owner.abortiveSearch)myState=3;

                switch(myState){
                    case 0:      //complete
//                        TODO: JSON
//                        var results = JSON.parse(search.result.resultsToJSON());
//                        searchresultlength = results.length;

                        searchresultlength = search.result.length;
                        if(searchresultlength < 2){
                            arr = scope.insertNNnoDataProg(searchresultlength, search.result, ccid);
//                            TODO JSON
//                            arr = scope.insertNNnoDataProg(searchresultlength, results, ccid);
                        }else{
                            var firstProg = search.result[0];
                            var secondProg = search.result[1];
                            var timeNow = (new Date().getTime()) / 1000;

                            try {
                                if ((firstProg.startTime > timeNow) || ((firstProg.startTime + firstProg.duration) < timeNow)) {
                                    // first prog not showing now - check second
                                    if ((secondProg.startTime > timeNow) || ((secondProg.startTime + secondProg.duration) < timeNow)) {
                                        // second prog not showing now either - put in 'no data' placeholders
                                        arr.push(scope.owner.returnNoDataProgramme(ccid));
                                        arr.push(scope.owner.returnNoDataProgramme(ccid));
                                    } else {
                                        // second prog is showing now - add it as the now item but use 'no data' placeholder for next
                                        arr.push(secondProg);
                                        arr.push(scope.owner.returnNoDataProgramme(ccid));
                                    }
                                } else {
                                    arr.push(firstProg);

                                    if (secondProg.startTime === (firstProg.startTime + firstProg.duration)) {
                                        arr.push(secondProg);
                                    } else {
                                        arr.push(scope.owner.returnNoDataProgramme(ccid));
                                    }
                                }
                            } catch (e) {
                                freesat.console.log('ERROR findNowAndNextAllResults: ' + e);
                            }
                        }

                        var uTime = new Date().getTime() + 300000;
                        uTime = uTime - (uTime %  300000) + 60000; //update time one min after % 5 min
                        if(scope.updateInterval)clearInterval(scope.updateInterval);
                        scope.updateInterval = setInterval(scope.updateNNData(scope, true), (uTime - (new Date().getTime())) );
                        if(scope.updateData && scope.updateData.length>0){
                            if(scope.retryInterval)clearInterval(scope.retryInterval);
                            scope.retryInterval = setInterval(scope.updateNNData(scope, false), 10000);
                        }
                        var data = (scope.filteredList)?scope.nowNextFilteredData:scope.nowNextData;
                        data[scope.collectItem] = uTime;
                        scope.setNNData(scope.collectItem, arr);
                        if (scope.newDataCallArgs && scope.newDataCallArgs.length){
                            scope.findNowAndNextForPageChannels(scope.newDataCallArgs);
                        }
                        if (typeof callback === 'function'){
                            callback(arr);
                        }
                        break;
                    case 3:     //aborted
                        scope.owner.abortiveSearch = false;
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        if(scope.newDataCall){
                            scope.newDataCall(scope.newDataCallArgs);
                        }
                        break;
                    case 4:     //insuficient resources
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        break;

                }
            };
        },
        insertNNnoDataProg: function(searchresultlength, result, ccid){
            var arr = [], date = new Date().getTime() / 1000;
            if(searchresultlength==1){
                if(result[0].startTime <= date){
                    arr.push(result[0]);
                    arr.push(this.owner.returnNoDataProgramme(ccid));
                }else{
                    arr.push(this.owner.returnNoDataProgramme(ccid));
                    arr.push(this.owner.returnNoDataProgramme(ccid));
                }
            }else{
                arr.push(this.owner.returnNoDataProgramme(ccid));
                arr.push(this.owner.returnNoDataProgramme(ccid));
            }
            return arr;
        },
        updateNNData: function(scope, update){
            return function(){
                if(update){
                    clearInterval(scope.updateInterval);
                }
                if(freesat.overlay && freesat.overlay.buttons){
                    return;
                }
                if(scope.retryInterval)clearInterval(scope.retryInterval);
                if(freesat.getZone() !== 'now' || freesat.oipfElement.searchActive){
                    scope.updateFunc = null;
                    return;
                }
                if(update){
                    scope.updateSetCheck();
                }else{
                    scope.retrySetCheck();
                }
            };
        },
        retrySetCheck: function(){
            freesat.console.log('retrySetCheck');
            if(this.updateFunc && this.retryData.length > 0){
                this.retryNNdata(this.retryData);
                setInterval(this.updateNNData(this, false), 10*1000);
            }
        },
        retryNNdata: function(ccidList){
            if(this.retry === this.retryData.length){
                this.retry = 0;
            }else{
                try{
                    freesat.oipfElement.searchActive = true;
                    freesat.oipfElement.searchType = 'nownextretry';
                    var list = (this.filteredList)?freesat.channelData.genreList:freesat.channelData.channelList;
//                    TODO: JSON
//                    var list = (this.filteredList)?freesat.channelData.genreList:freesat.channelData.channelListOIPF;
                    var channel = list.getChannel(this.retryData[this.retry]);
                    freesat.oipfElement.mSearch = freesat.oipfElement.searchManager.createSearch(1);
                    freesat.oipfElement.mSearch.findProgrammesFromStream(channel, null);
                    freesat.oipfElement.searchManager.onMetadataSearch = this.retryNowAndNextResults(this, this.retryData[this.retry]);
                    ++this.retry;
                    freesat.oipfElement.mSearch.result.getResults(0, 2);
                }catch(e){
                    freesat.oipfElement.searchActive = false;
                }
            }
        },
        retryNowAndNextResults: function(scope, ccid){
            return function(search, state){
                freesat.console.log('retryNowAndNextResults');
                freesat.oipfElement.searchManager.onMetadataSearch = null;
                var i, arr = [], searchresultlength, myState = state;
                if(scope.owner.abortiveSearch)myState=3;
                switch(myState){
                    case 0:      //complete
//                        TODO: JSON
//                        var results = JSON.parse(search.result.resultsToJSON());
//                        searchresultlength = results.length;
                        searchresultlength = search.result.length;
                        if(searchresultlength < 2){
                            arr = scope.insertNNnoDataProg(searchresultlength, search.result, ccid);
//                            TODO: JSON
//                            arr = scope.insertNNnoDataProg(searchresultlength, results, ccid);
                        }else{
                            try {
                                for(i = 0; i < searchresultlength; i++){
                                    arr.push(search.result[i]);
//                                    TODO JSON
//                                    arr.push(results[i]);
                                }
                                scope.retryData.splice(scope.retryData.indexOf(ccid),1);
                            } catch (e) {
                                freesat.console.log('ERROR findNowAndNextAllResults: ' + e);
                            }
                        }
//                        if(scope.diffData(arr, ccid, scope.collectItem)){
                            scope.setNNData(scope.collectItem, arr);
//                        }
                        scope.retryNNdata(scope.newDataCallArgs);
                        break;
                    case 3:     //aborted
                        scope.owner.abortiveSearch = false;
                        freesat.oipfElement.searchActive = false;
                        if(scope.newDataCall){
                            scope.newDataCall(scope.newDataCallArgs);
                        }
                        break;
                    case 4:     //insuficient resources
                        freesat.oipfElement.searchActive = false;
                        break;

                }
            };
        },
//        diffData: function(arr, ccid){
//
//        },
        updateSetCheck: function(){
            var lcnList,topPos;
            freesat.console.log('updateSetCheck');
            try{
                lcnList = freesat.behind.channelLCNlist;
                topPos = freesat.behind.topPositionIndex;
            }catch(e){
                try{
                    lcnList = freesat.front.channelLCNlist;
                    topPos = freesat.front.topPositionIndex;
                }catch(err){
                    this.updateFunc = null;
                    freesat.console.log('updateSetCheck cancelled');
                    return;
                }
            }
            if(this.updateFunc){
                this.updateArray = [];
                this.checkNNdata(topPos, lcnList, this);
                setInterval(this.updateNNData(this, true), 5*60*1000);
            }
        },
        checkNNdata : function(topPos, lcnList, scope){
            var i;
            freesat.console.log('checkNNdata');
            if(scope.updateArray.length < lcnList.length){
                try{
                    freesat.oipfElement.searchActive = true;
                    freesat.oipfElement.searchType = 'nownext';
                    var list = (this.filteredList)?freesat.channelData.genreList:freesat.channelData.channelList;
//                    TODO: JSON
//                    var list = (this.filteredList)?freesat.channelData.genreList:freesat.channelData.channelListOIPF;
                    var channel = list.getChannel('ccid:'+lcnList[scope.updateArray.length]);
                    freesat.oipfElement.mSearch = freesat.oipfElement.searchManager.createSearch(1);
                    freesat.oipfElement.mSearch.findProgrammesFromStream(channel, null);
                    freesat.oipfElement.searchManager.onMetadataSearch = scope.updateNNResults(topPos, lcnList, scope);
                    freesat.oipfElement.mSearch.result.getResults(0, 2);
                }catch(e){
                    clearInterval(scope.updateInterval);
                    freesat.oipfElement.searchActive = false;
                }
            }else{
                var callUpdate = false;
                for(i=0; i< lcnList.length; i++){
                    freesat.console.log('TIMES === '+scope.updateArray[i][0].startTime+' != '+scope.getNNDataItem(topPos + i,0).startTime);
                    if(scope.updateArray[i][0].startTime != scope.getNNDataItem(topPos + i,0).startTime){
                        callUpdate = true;
                        break;
                    }
                    if(scope.updateArray[i][1].startTime != scope.getNNDataItem(topPos + i,1).startTime){
                        callUpdate = true;
                        break;
                    }
                }
                freesat.console.log('NN Update required = '+ callUpdate);
                if(callUpdate){
                    for(i=0; i< lcnList.length; i++){
                        scope.setNNDataItem(topPos + i,0, scope.updateArray[i][0]);
                        scope.setNNDataItem(topPos + i,1, scope.updateArray[i][1]);
                    }
                    scope.updateFunc();
                }
            }
        },
        updateNNResults: function(topPos, lcnList, scope){
            return function(search, state){
                freesat.oipfElement.searchManager.onMetadataSearch = null;
                var i, arr = [], searchresultlength;
                freesat.console.log('updateNNResults');
                switch(state){
                    case 0:      //complete
//                        TODO: JSON
//                        var results = JSON.parse(search.result.resultsToJSON());
//                        searchresultlength = results.length;
                        searchresultlength = search.result.length;
                        try {
                            for(i = 0; i < searchresultlength; i++){
                                arr.push(search.result[i]);
//                                TODO: JSON
//                                arr.push(results[i]);
                            }
                        } catch (e) {
                            freesat.console.log('ERROR updateNNResults: ' + e);
                        }
                        var uTime = new Date().getTime() + 300000;
                        if((uTime %  300000)>200){
                            uTime = uTime - (uTime %  300000);
                        }
                        freesat.console.log('nn update - '+lcnList[scope.updateArray.length]);
                        scope.updateArray.push(arr);
                        freesat.oipfElement.searchActive = false;
                        scope.checkNNdata(topPos, lcnList, scope);
                        break;
                    case 3:     //aborted
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        if(scope.newDataCall){
                            scope.newDataCall(scope.newDataCallArgs);
                        }
                        break;
                    case 4:     //insuficient resources
                        freesat.masterController.hideLoading();
                        freesat.oipfElement.searchActive = false;
                        break;

                }
            };
        }

    }
);
accedo.ui.View.create('freesat.view.subabout.ProgramList', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#programList',
    children: [
        {
            type: accedo.ui.Container,
            id: '#subcontainer',
            css: 'container',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#programListTitle'
                },
                {
                    type: accedo.ui.Container,
                    id: '#programListItemContainer',
                    children: []
                },
                {
                    type: accedo.ui.Container,
                    id: '#iconContainer',
                    children: [
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageDown'
                        },
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageUp'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#pageNumber'
                        }
                    ]
                }
            ]
        }
    ]
});
accedo.ui.View.create('freesat.view.subabout.About', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#aboutright',
    children: [
        {
            type: accedo.ui.Container,
            id: '#subcontainer',
            css: 'container',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#aboutTitle'
                },
                {
                    type: accedo.ui.Container,
                    id: '#aboutTextBackground'
                },
                {
                    type: accedo.ui.Container,
                    id: '#aboutTextContainer',
                    children: [
                        {
                            type: accedo.ui.Container,
                            id: '#aboutTextPage',
                            children: [
                                {
                                    type: accedo.ui.widget.Label,
                                    id: '#aboutSeriesInfo'
                                },
                                {
                                    type: accedo.ui.widget.Label,
                                    id: '#aboutCategory'
                                },
                                {
                                    type: accedo.ui.widget.Image,
                                    id: '#aboutGuidanceIcon'
                                },
                                {
                                    type: accedo.ui.widget.Label,
                                    id: '#aboutGuidance'
                                },
                                {
                                    type: accedo.ui.widget.Label,
                                    id: '#aboutParagraph'
                                }
                            ]
                        }
                    ]
                },
                {
                    type: accedo.ui.Container,
                    id: '#iconContainer',
                    children: [
                        {
                            type: accedo.ui.widget.Image,
                            id: '#aboutPageDown',
                            src: 'images/1280x720/button/nav_pagedown.png'
                        },
                        {
                            type: accedo.ui.widget.Image,
                            id: '#aboutPageUp',
                            src: 'images/1280x720/button/nav_pageup.png'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#pageNumber'
                        }
                    ]
                }
            ]
        }
    ]
});
accedo.ui.View.create('freesat.view.subabout.Cast', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#cast',
    children: [
        {
            type: accedo.ui.Container,
            id: '#subcontainer',
            css: 'container',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#castTitle'
                },
                {
                    type: accedo.ui.Container,
                    id: '#castItemContainer',
                    children: []
                },
                {
                    type: accedo.ui.Container,
                    id: '#iconContainer',
                    children: [
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageDown'
                        },
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageUp'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#pageNumber'
                        }
                    ]
                }
            ]
        }
    ]
});
accedo.ui.View.create('freesat.view.subabout.More', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#more',
    children: [
        {
            type: accedo.ui.Container,
            id: '#subcontainer',
            css: 'container',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#moreTitle'
                },
                {
                    type: accedo.ui.Container,
                    id: '#moreItemContainer',
                    children: []
                },
                {
                    type: accedo.ui.Container,
                    id: '#iconContainer',
                    children: [
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageDown'
                        },
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageUp'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#pageNumber'
                        }
                    ]
                }
            ]
        }
    ]
});
accedo.ui.View.create('freesat.view.subabout.Recommendations', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#recommendations',
    children: [
        {
            type: accedo.ui.Container,
            id: '#subcontainer',
            css: 'container',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#recommendationsTitle'
                },
                {
                    type: accedo.ui.Container,
                    id: '#recommendationsItemContainer',
                    children: []
                },
                {
                    type: accedo.ui.Container,
                    id: '#iconContainer',
                    children: [
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageDown'
                        },
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageUp'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#pageNumber'
                        }
                    ]
                }
            ]
        }
    ]
});
accedo.ui.View.create('freesat.view.subabout.Episodes', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#episodes',
    children: [
        {
            type: accedo.ui.Container,
            id: '#subcontainer',
            css: 'container',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#episodesTitle'
                },
                {
                    type: accedo.ui.Container,
                    id: '#episodesItemContainer',
                    children: []
                },
                {
                    type: accedo.ui.Container,
                    id: '#iconContainer',
                    children: [
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageDown'
                        },
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageUp'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#pageNumber'
                        }
                    ]
                }
            ]
        }
    ]
});
accedo.ui.View.create('freesat.view.subabout.Showing', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#showing',
    children: [
        {
            type: accedo.ui.Container,
            id: '#subcontainer',
            css: 'container',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#showingTitle'
                },
                {
                    type: accedo.ui.Container,
                    id: '#showingItemContainer',
                    children: []
                },
                {
                    type: accedo.ui.Container,
                    id: '#iconContainer',
                    children: [
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageDown'
                        },
                        {
                            type: accedo.ui.widget.Image,
                            id: '#pageUp'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#pageNumber'
                        }
                    ]
                }
            ]
        }
    ]
});
/**
 * Created with JetBrains WebStorm.
 * User: dean clancy
 * Date: 10/09/13
 * Time: 06:54
 * To change this template use File | Settings | File Templates.
 */
accedo.ui.View.create('freesat.view.subviews.Footer', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#footer',
    css: 'footer show-red',
    children: [
        {
            type: accedo.ui.Container,
            id: '#redButton',
            css: 'red btn colour',
            children: [
                {
                    type: accedo.ui.widget.Image,
                    id: '#redButtonImage',
                    css: 'button',
                    src: 'images/1280x720/button/footer_red.png'

                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#redButtonLabel',
                    css: 'label',
                    text: 'Home Menu'
                }
            ]
        },
        {
            type: accedo.ui.Container,
            id: '#greenButton',
            css: 'green btn colour',
            children: [
                {
                    type: accedo.ui.widget.Image,
                    id: '#greenButtonImage',
                    css: 'button',
                    src: 'images/1280x720/button/footer_green.png'

                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#greenButtonLabel',
                    css: 'label',
                    text: 'Filter'
                }
            ]
        },
        {
            type: accedo.ui.Container,
            id: '#yellowButton',
            css: 'yellow btn colour',
            children: [
                {
                    type: accedo.ui.widget.Image,
                    id: '#yellowButtonImage',
                    css: 'button',
                    src: 'images/1280x720/button/footer_yellow.png'

                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#yellowButtonLabel',
                    css: 'label',
                    text: ''
                }
            ]
        },
        {
            type: accedo.ui.Container,
            id: '#blueButton',
            css: 'blue btn colour',
            children: [
                {
                    type: accedo.ui.widget.Image,
                    id: '#blueButtonImage',
                    css: 'button',
                    src: 'images/1280x720/button/footer_blue.png'

                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#blueButtonLabel',
                    css: 'label',
                    text: ''
                }
            ]
        },
        {
            type: accedo.ui.Container,
            id: '#pagingButton',
            css: 'paging btn',
            children: [
                {
                    type: accedo.ui.widget.Image,
                    id: '#pagingButtonImage',
                    css: 'button',
                    src: 'images/1280x720/button/footer_paging.png'

                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#pagingButtonLabel',
                    css: 'label',
                    text: 'Page Up / Down'
                }
            ]
        }
    ]
});
/**
 * Created with JetBrains WebStorm.
 * User: dean clancy
 * Date: 10/09/13
 * Time: 06:54
 * To change this template use File | Settings | File Templates.
 */
accedo.ui.View.create('freesat.view.subviews.Header', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#header',
    children: [
        {
            type: accedo.ui.widget.Image,
            id: '#headerbackgrounnd'
        },
        {
            type: accedo.ui.Container,
            id: '#pageHeader',
            css: 'pageHeader',
            children: [
                {
                    type: accedo.ui.Container,
                    css: 'logoContainer',
                    children:[
                        {
                            type: accedo.ui.widget.Image,
                            id: 'freesatLogo',
                            css: 'freesatLogo'
                        }
                    ]
                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#pageTitle'
                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#pageClock'
                },
                {
                    type: accedo.ui.Container,
                    id: '#programmeInformation',
                    css: 'programmeInformation',
                    children: [
                        {
                            type: accedo.ui.widget.Label,
                            id: '#headerMsg',
                            css: 'headerMsg'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#channelName',
                            css: 'channelName'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#programmeTitle',
                            css: 'programmeTitle'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#programmeDescription',
                            css: 'programmeDescription'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#seasonandEpisode',
                            css: 'seasonandEpisode'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#programmeLength',
                            css: 'programmeLength'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#programmeDuration',
                            css: 'programmeDuration'
                        }
                    ]
                },
                {
                    type: accedo.ui.Container,
                    id: '#searchResultsHeader',
                    css: 'search-results-hdr',
                    children: [
                        {
                            type: accedo.ui.widget.Label,
                            id: '#searchOnTV',
                            css: 'search-label'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#searchOnOD',
                            css: 'search-label'
                        }
                    ]
                },
                {
                    type: accedo.ui.widget.Label,
                    id: '#programmeDate',
                    css: 'programmeDate'
                }
            ]
        },
        {
            type: accedo.ui.Container,
            id: '#programmeMetadataIconsContainer',
            css: 'dataDuration',
            children: [
                {
                    type: accedo.ui.widget.Label,
                    id: '#guideProgrammeTime',
                    css: 'inline text-label'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon6',
                    css: 'inline progMetadataIcon icon-ondemand',
                    src: 'images/1280x720/icon/icon_ondemand.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon8',
                    css: 'inline progMetadataIcon icon-restart',
                    src: 'images/1280x720/icon/icon_restart.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon8',
                    css: 'inline progMetadataIcon icon-pay',
                    src: 'images/1280x720/icon/icon_paid_content.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon0',
                    css: 'inline progMetadataIcon icon-guidance',
                    src: 'images/1280x720/icon/icon_guidance.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon5',
                    css: 'inline progMetadataIcon icon-sub',
                    src: 'images/1280x720/icon/icon_sub.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon4',
                    css: 'inline progMetadataIcon icon-sl',
                    src: 'images/1280x720/icon/icon_sl.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon3',
                    css: 'inline progMetadataIcon icon-ad',
                    src: 'images/1280x720/icon/icon_ad.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon2',
                    css: 'inline progMetadataIcon icon-series',
                    src: 'images/1280x720/icon/icon_series.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon1',
                    css: 'inline progMetadataIcon icon-3d',
                    src: 'images/1280x720/icon/icon_3d.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon1',
                    css: 'inline progMetadataIcon icon-simulcast',
                    src: 'images/1280x720/icon/icon_SD_HD.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon1',
                    css: 'inline progMetadataIcon icon-hd',
                    src: 'images/1280x720/icon/icon_HD.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon7',
                    css: 'inline progMetadataIcon icon-dolby',
                    src: 'images/1280x720/icon/icon_5.1audio.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon11',
                    css: 'inline progMetadataIcon icon-surround',
                    src: 'images/1280x720/icon/icon_surround.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon9',
                    css: 'inline progMetadataIcon icon-showcase',
                    src: 'images/1280x720/icon/icon_showcase.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon12',
                    css: 'inline progMetadataIcon icon-reminder',
                    src: 'images/1280x720/icon/icon_reminder.png'
                },
                {
                    type: accedo.ui.widget.Image,
                    id: '#metaIcon13',
                    css: 'inline progMetadataIcon icon-recording',
                    src: 'images/1280x720/icon/icon_rec.png'
                }
            ]
        }
    ]
});

accedo.ui.View.create('freesat.view.subviews.Empty', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: 'empty'
});
accedo.ui.View.create('freesat.view.About', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#about',
    children: [
//        {
//            id: '#header-controller',
//            controller: 'freesat.controllers.subcontrollers.Header'
//        },
        {
            type: accedo.ui.Container,
            id: '#container',
            css: 'container',
            children: [
                {
                    type: accedo.ui.Container,
                    id: '#leftcontainer',
                    css: 'leftcontainer'
                },
                {
                    type: accedo.ui.Container,
                    id: '#rightcontainer',
                    css: 'rightcontainer',
                    children: [
                        {
                            controller: 'freesat.controllers.subabout.About'
                        }
                    ]
                }
            ]
        }
    ]
});
accedo.ui.View.create('freesat.view.Behind', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#behind'
});
accedo.ui.View.create('freesat.view.Front', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#front'
});
accedo.ui.View.create('freesat.view.Fullscreen', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#child-container',
    children: [
        {
            id: '#behind-container',
            controller: 'freesat.controllers.Behind'
        },
        {
            id: '#video-container',
            controller: 'freesat.controllers.Video'
        },
        {
            id: '#front-container',
            controller: 'freesat.controllers.Front'
        },
        {
            id: '#overlay-container',
            controller: 'freesat.controllers.Overlay'
        },
        {
            id: '#popup-container',
            controller: 'freesat.controllers.PopupBlank'
        }
    ]
});

/**
 * Created with JetBrains WebStorm.
 * User: Dean Clancy
 * Date: 14/10/13
 * Time: 19:58
 */
accedo.ui.View.create('freesat.view.HomeMenu', {
    type: accedo.ui.layout.Normal,
    defaultFocus: '',
    id: '#HomeMenu',
    children: [
        {
            type: accedo.ui.Container,
            id: '#home-container',
            css: 'hidden',
            children:[

                {
                    type: accedo.ui.Container,
                    id: '#background-right'
                },
                {
                    type: accedo.ui.Container,
                    id: '#background-right-top'
                },
                {
                    type: accedo.ui.Container,
                    id: '#background-right-top-top'
                },
                {
                    type: accedo.ui.Container,
                    id: '#background-right-top-right'
                },
                {
                    type: accedo.ui.Container,
                    id: '#background-left',
                    children:[
                        {
                            type: accedo.ui.Container,
                            css: 'logoContainer',
                            children:[
                                {
                                    type: accedo.ui.widget.Image,
                                    id: 'freesatLogo',
                                    css: 'freesatLogo',
                                    src: 'images/1280x720/logo/freetime-125x24.png'
                                }
                            ]
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#home-clock'
                        },
                        {
                            type: accedo.ui.widget.Label,
                            id: '#home-title'
                        },
                        {
                            type: accedo.ui.Container,
                            id: '#mask',
                            children:[
                                {
                                    type : accedo.ui.Container,
                                    id : '#highlight',
                                    children : [
                                        {
                                            type : accedo.ui.widget.Image,
                                            id : '#home-highlight-image',
                                            src : 'images/1280x720/icon/ico_arrow.