/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

/* Copyright (c) 2014 - 2015 Panasonic Corporation */

/* This Source Code Form is "Incompatible With Secondary Licenses",
 * as defined by the Mozilla Public License, v. 2.0. */

"use strict";

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;
const Cr = Components.results;
const CC = Components.Constructor;

Cu.import("resource://gre/modules/XPCOMUtils.jsm");

const UnixServerSocket = CC(
        "@mozilla.org/network/server-socket;1", "nsIServerSocket", "initWithFilename"),
      PanaUnixSocketInternal = Cc[
        '@mozilla.org/pana-unix-socket;1'].createInstance(Ci.nsIPanaUnixSocketInternal);

XPCOMUtils.defineLazyModuleGetter(this, 'setTimeout', // jshint ignore:line
  'resource://gre/modules/Timer.jsm');

/*
 * Debug logging function
 */

let debug = false;
function LOG(msg) {
  if (debug) {
    dump("PanaUnixServerSocket: " + msg + "\n");
  }
}

let do_make_sockfile = function(sunPath) {
  let file = Components.classes["@mozilla.org/file/local;1"]
                       .createInstance(Components.interfaces.nsILocalFile);
  file.initWithPath(sunPath);
  LOG('do_make_sockfile: ' + file.exists());
  if (file.exists()) {
    file.remove(false);
  }
  return file;
};

/*
 * nsIDOMPanaUnixServerSocket object
 */

function PanaUnixServerSocket() {
  this._onconnect = null;

  this._neckoPanaUnixServerSocket = null;
}

PanaUnixServerSocket.prototype = {
  __exposedProps__: {
    onconnect: 'rw',
  },
  get onconnect() {
    return this._onconnect;
  },
  set onconnect(f) {
    this._onconnect = f;
  },

  forceReleaseFD: function puss_forceReleaseFD() {
    // Force GC to release socket fd.
    setTimeout(function puss_gc_timer() {
      LOG('forceGC');
      Cu.forceGC();
    }, 5000);
  },

  _callListenerAcceptCommon: function puss_callListenerAcceptCommon(socket) {
    LOG('_callListenerAcceptCommon');
    if (this._onconnect) {
      try {
        this["onconnect"].call(null, socket);
      } catch (e) {
        debug("Onconnect error");
      }      
    }
    else {
      debug("Received unexpected connection!");
    }
  },
  init: function puss_init(aWindowObj) {
    LOG('init');
  },

  /* nsIPanaUnixServerSocketInternal method */
  listen: function puss_listen(sunPath) {
    LOG('listen sunPath=' + sunPath);
    if (this._neckoPanaUnixServerSocket == null) {
      let socketName = do_make_sockfile(sunPath);
      const allPermissions = parseInt("777", 8);
      this._neckoPanaUnixServerSocket = new UnixServerSocket(socketName, allPermissions, -1);
      this._sunPath = sunPath;
      this._neckoPanaUnixServerSocket.asyncListen(this);
    }
    else {
      throw new Error("Parent PanaUnixServerSocket has already listening. \n");
    }
  },
  /* end nsIPanaUnixServerSocketInternal method */

  close: function puss_close() {
    LOG('close');

    /*　Close ServerSocket　*/
    if (this._neckoPanaUnixServerSocket) {
      this._neckoPanaUnixServerSocket.close();
      this.forceReleaseFD();
    }
  },

  // nsIServerSocketListener (Triggered by _neckoPanaUnixServerSocket.asyncListen)
  onSocketAccepted: function puss_onSocketAccepted(server, trans) {
    LOG('onSocketAccepted');
    try {
      let that = PanaUnixSocketInternal.createAcceptedParent(trans);
      this._callListenerAcceptCommon(that);
    }
    catch(e) {
      LOG('onSocketAccepted error');
      trans.close(Cr.NS_BINDING_ABORTED);
      this.forceReleaseFD();
    }
  },

  // nsIServerSocketListener (Triggered by _neckoPanaUnixServerSocket.asyncListen)
  onStopListening: function puss_onStopListening(server, status) {
    LOG('onStopListening');
    if (status != Cr.NS_BINDING_ABORTED) {
      throw new Error("Server socket was closed by unexpected reason.");
    }
    this._neckoPanaUnixServerSocket = null;
  },

  classID: Components.ID("{f2b2b40c-125a-11e4-b4fc-ac220bcc9310}"),

  classInfo: XPCOMUtils.generateCI({
    classID: Components.ID("{f2b2b40c-125a-11e4-b4fc-ac220bcc9310}"),
    classDescription: "Server PanaUnix Socket",
    interfaces: [
      Ci.nsIDOMPanaUnixServerSocket,
      Ci.nsISupportsWeakReference
    ],
    flags: Ci.nsIClassInfo.DOM_OBJECT,
  }),

  QueryInterface: XPCOMUtils.generateQI([
    Ci.nsIDOMPanaUnixServerSocket,
    Ci.nsIPanaUnixServerSocketInternal,
    Ci.nsISupportsWeakReference
  ])
}

this.NSGetFactory = XPCOMUtils.generateNSGetFactory([PanaUnixServerSocket]);
